/*
  q3.cpp
		
	Q3Plug 1.1b5
		
  Quake 3 server query

  Author:   Markus Baumgartner
  Compiler: Microsoft Visual C++ 6.0
  Last Mod: 31/12/2000
	Tab size: 2
*/


#include <assert.h>
#include <string.h>
#include <windows.h>
#include <stdio.h>

#include "npapi.h"
#include "commctrl.h"

#ifndef _Q3_H_
#include "q3.h"
#endif

#ifndef _Q3PLUG_H_
#include "q3plug.h"
#endif


COLORREF Q3_colors[10] = {
  RGB(255,255,255),       /* BLACK  (white here) */
  RGB(255,  0,  0),       /* RED    */
  RGB(  0,220,  0),       /* GREEN  */
  RGB(255,255, 20),       /* YELLOW */
	RGB( 90,150,255),       /* BLUE   */
	RGB(  0,160,160),       /* CYAN   */
	RGB(195, 50,190),       /* MAGENTA*/
	RGB(255,255,255),       /* WHITE  */
	RGB(255,  0,  0),       /* RED    */
  RGB(  0,220,  0)        /* GREEN  */
};

// game-specific information
void Q3_getInfo(char* text[]) {
  text[0] = "Quake 3 Arena";
  text[1] = "Quake 3 Arena server query code written by Markus Baumgartner\nmarkus.baumgartner@liwest.at";
  text[2] = "quake3.exe";
  text[3] = "+connect %s:%s";                     // connect string used in command line (hostname:port)
}

// column names
void Q3_getColumns(char* cols[]) {
  cols[0] = "Ping";
  cols[1] = "Score";
  cols[2] = "Player";
}

/* Strips q3 color codes from player names (^0, ^1, .. , ^9) */
/* also removes any non-alphanumerical characters */
char* Q3_stripColors(char* name) {
  assert(name);
  
  if (strlen(name) > 255)
    return name;
  
	char tmp2[256];
	char *tmp = tmp2;
  char *savename = name;

  while (*name) {
   	if ((*name >= 'a' && *name <= 'z') || (*name >= 'A' && *name <= 'Z')  || 
			  (*name >= '0' && *name <= '9')) 
      *tmp++ = *name;
    else if ((*name=='^') && (*(name+1) >= '0') && (*(name+1) <= '9')) 
      name++;
		name++;
  }
	*tmp='\0';
  return strcpy(savename, tmp2);
	
}

// custom compare function to compare columns correctly
// as we do not have information about team score availabe during sorting,
// this info is encoded in the player name to allow for correct sorting
int Q3_compareFunc(char *val1, char *val2, int sortCol, int sortDir) {
  // don't sort empty items
  if (strlen(val1)==0 && strlen(val2)==0)
    return 0;
  if (strlen(val1)==0)
    return 1;
  if (strlen(val2)==0)
    return -1;
  
  switch (sortCol) {
	  case 0: return sortDir*(atoi(val1) - atoi(val2));
		case 1: return sortDir*(atoi(val1) - atoi(val2));
  	case 2: if (*val1=='\\' || *val2=='\\') // no reverse sorting for team display mode
              return -stricmp(Q3_stripColors(val1),Q3_stripColors(val2));
            return sortDir*stricmp(Q3_stripColors(val1),Q3_stripColors(val2));
    default: return 0;
  }
}

// in case of an OSP server, this function resturns the gametype
// otherwise it returns -1
int Q3_OSP_gametype(PluginInstance *This) {
  
  assert(This);
  assert(This->buffer);
  char *type;

  if ( NULL!=strstr(This->buffer, "gameversion\\OSP ") || NULL!=strstr(This->buffer, "gamename\\osp")) {
    type = strstr(This->buffer, "g_gametype\\");
    if (type) {
      type+=11;
      return (*type - '0');
    }
  }  
  return -1;
}

// extract sematics from some rules
void Q3_checkRule(PluginInstance *This, structRule r, MODData *m) {
  assert(This);
	assert(m);

	if (0==strcmp(r.name, "sv_hostname"))
    strncpy(This->hostname, r.value, 127);

	if (0==strcmp(r.name, "sv_maxclients")) 
    strncpy(This->maxclients, r.value, 7);

  if (0==strcmp(r.name, "Players_Red")) 
    m->OSP_players_red = r.value;

  if (0==strcmp(r.name, "Players_Blue")) 
    m->OSP_players_blue = r.value;

  if (0==strcmp(r.name, "Players_Active")) 
    m->OSP_players_active = r.value;

  if (0==strcmp(r.name, "Score_Red")) 
    if ((*r.value - '0') < 0 || (*r.value - '0') > 9)
      m->OSP_score_red = " ";
    else
      m->OSP_score_red = r.value;

  if (0==strcmp(r.name, "Score_Blue")) 
    if ((*r.value - '0') < 0 || (*r.value - '0') > 9)
      m->OSP_score_blue = " ";
    else
      m->OSP_score_blue = r.value;
}

// returns TRUE if the given player number is a member of the given OSP player string
// returns number of players in given team in num  
BOOL Q3_OSP_isInTeam(int number, char *playerstring, int* num) {
  char *token;
  char tmp[128];
  int x, count=0;
  BOOL result = FALSE;

  if (!playerstring)
    return FALSE;
  
  strncpy(tmp, playerstring, sizeof(tmp)-1);

  token = strtok(tmp, " ");
  while (token) {
    x = atoi(token);
    if (x > 0) {
      count++;
      result = result || number == x; 
    }
    token = strtok(NULL, " ");
  }
  if (num)
    *num = count;
  return result;
}


// this function provides us with a 'score' for the given player
// which is then used for proper player sorting (in OSP display mode)
int Q3_OSP_getScore(PluginInstance *This, MODData m, int number, int dummy) {
  
  int score_red, score_blue;
  
  int win_base =  20000;
  int lose_base = 15000;
  int play_base = 10000;
  int spec_base = 5000;
  int dummybonus = 2000;
  
  switch (dummy) {
    case OSP_SPEC:    return spec_base+dummybonus;
    case OSP_PLAYER:  return play_base+dummybonus;
  } 

  switch (m.OSP_gametype) {
    case Q3_GAMETYPE_CTF: // capture the flag
    case Q3_GAMETYPE_CA:  // clan arena
    case Q3_GAMETYPE_TDM: // team deathmatch
      assert(m.OSP_score_red && m.OSP_score_blue);
      score_red = atoi(m.OSP_score_red);
      score_blue = atoi(m.OSP_score_blue);
      
      if (Q3_OSP_isInTeam(number, m.OSP_players_red, NULL)) // score for red team members 
        if (score_red > score_blue)
          return win_base;
        else
          return lose_base;

      if (Q3_OSP_isInTeam(number, m.OSP_players_blue, NULL)) // score for blue team members
        if (score_red <= score_blue)            
          return win_base;
        else
          return lose_base;

      if (dummy==OSP_TEAM_RED)                          // score for dummy players
        if (score_red > score_blue)
          return win_base + dummybonus;
        else
          return lose_base + dummybonus;

      if (dummy==OSP_TEAM_BLUE)                        // score for dummy players
        if (score_red <= score_blue)
          return win_base + dummybonus;
        else
          return lose_base + dummybonus;

      return spec_base;
      
    case Q3_GAMETYPE_FFA:   // Free For All
    case Q3_GAMETYPE_DUEL:  // 1on1
  
      if (Q3_OSP_isInTeam(number, m.OSP_players_active, NULL))
        return play_base;
      else
        return spec_base;
  }
  return 0;
}

// extract next token	and update position
char* Q3_nextToken(char **pos, char limit) {
	char *cur;
	char *tmp;

	assert(pos);
 	cur = *pos;
	
	if (!cur || !*cur)
	  return NULL;
	
	tmp = cur;

	while (*cur!='\0' && *cur!=limit)
		cur++;

	if (*cur==limit) {
		*pos=cur+1;
		*cur='\0';
	}
	else
		*pos=NULL;
		
	return tmp;
}


// do some initialization with Q3 data
BOOL Q3_initBuffer(PluginInstance *This) {

  assert(This);
	assert(This->buffer);

	// check if packet is valid
	if (!strstr(This->buffer, "statusResponse")) {
    return FALSE;
  }

	// init pointers to data
	This->rulePos = strchr(This->buffer, '\n');
	if (! This->rulePos)
		return FALSE;
	This->rulePos+=2;
	This->playerPos = strchr(This->rulePos, '\n');
	if (!This->playerPos)
		return FALSE;
	*This->playerPos = '\0';
	This->playerPos++;
  	
	return TRUE;	 	
}

// add dummy player items for OSP server
void Q3_handleOSP_Rules(PluginInstance *This, MODData m, int numplayers) {
  char tmp3[128];
  structPlayer player;
  int score=0;

  assert(This);
               
  if (m.count_specs > 0) {
    score += Q3_OSP_getScore(This,m,0,OSP_SPEC);
    player.frags = "";
    player.ping = "";
    sprintf(tmp3, "\\%05dD %s",score,"^6<<Spectators>>");
    player.name = tmp3;
    UI_insertPlayer(This, player);
  }

  switch (m.OSP_gametype) {
    case Q3_GAMETYPE_CTF:
    case Q3_GAMETYPE_CA:
    case Q3_GAMETYPE_TDM:  // Team DM (insert dummy players for both teams)
      assert(m.OSP_score_red && m.OSP_score_blue);
      player.ping = "";
      if ( m.count_red > 0) {
        score = Q3_OSP_getScore(This, m,0,OSP_TEAM_RED) + atoi(m.OSP_score_red);
        sprintf(tmp3, "\\%05dD %s",score,"^1<<Team RED>>");
        player.frags = m.OSP_score_red;
        player.name = tmp3;  
			  UI_insertPlayer(This, player);
      }
      if ( m.count_blue > 0) {
        score = Q3_OSP_getScore(This, m,0,OSP_TEAM_BLUE) + atoi(m.OSP_score_blue);
        sprintf(tmp3, "\\%05dD %s",score,"^4<<Team BLUE>>");
        player.frags = m.OSP_score_blue;
        player.name = tmp3;  
			  UI_insertPlayer(This, player);
      } 
      break;
    case Q3_GAMETYPE_FFA:   // FFA
    case Q3_GAMETYPE_DUEL:  // 1on1
      if ( m.count_active > 0) {
        player.ping = "";
        score = Q3_OSP_getScore(This, m,0,OSP_PLAYER);
        player.frags = "";
        sprintf(tmp3, "\\%05dD %s",score,"^4<<Players>>");
        player.name = tmp3;  
			  UI_insertPlayer(This, player);
      }
  }
}

// encode sorting information in player name (in OSP display mode)
void Q3_handleOSP_Player(PluginInstance *This, char *buf, int bufsize, structPlayer player, int count, MODData m) {

  char tmp[128];
  int score;

  assert(buf);
  assert(player.frags && player.name && player.ping);
  
  score = Q3_OSP_getScore(This, m, count, 0) + atoi(player.frags);
      
  if (Q3_OSP_isInTeam(count, m.OSP_players_red,NULL)) 
 	  sprintf(tmp, "\\%05dR %s",score,player.name);
  else if (Q3_OSP_isInTeam(count, m.OSP_players_blue,NULL)) 
   	sprintf(tmp, "\\%05dB %s",score,player.name);
  else if (Q3_OSP_isInTeam(count, m.OSP_players_active,NULL))
    sprintf(tmp, "\\%05dB %s",score,player.name);
  else
    sprintf(tmp, "\\%05dS %s",score,player.name);
	  
  strncpy(buf, tmp, bufsize-1);
}

// this function is called when the plugin window is notified of the arrival
// of a packet on its socket
BOOL Q3_receivePacket(PluginInstance *p) {
 	char *cur;
  char *pos;
	MODData m;
  LARGE_INTEGER counter, freq;
  
  char tmp[128];
  int count=0;

 	structRule rule;
	structPlayer player;

  assert(p);
  assert(p->text);
 
  m.OSP_players_red = "";
  m.OSP_players_blue = "";
  m.OSP_score_red = "0";
  m.OSP_score_blue = "0";
  m.OSP_players_active = "";

 	// get packet time
	if (QueryPerformanceFrequency(&freq) && QueryPerformanceCounter(&counter))
		p->ticks = 1000*(counter.QuadPart - p->ticks) / freq.QuadPart;
	else
    p->ticks = GetTickCount() - p->ticks; 
  
  p->buffer = (char*) malloc(BUFSIZE);
	assert(p->buffer);

	// clear buffer (don't remove this!)
	memset(p->buffer, 0, BUFSIZE);

	/* receive packet */
  if (recv(p->sock, p->buffer, BUFSIZE, 0) == SOCKET_ERROR) {
    wsprintf(p->text, "Receive error (%d)", WSAGetLastError());
    free(p->buffer);
		return FALSE;
  }

	if (!Q3_initBuffer(p)) {
		strcpy(p->text, "Invalid packet");
		free(p->buffer);
		return FALSE;
	 }

  strcpy(p->text, "");
  itoa((int) p->ticks, p->ping, 10);
  
  m.OSP_gametype = Q3_OSP_gametype(p);    

  // parse rules
	cur = Q3_nextToken(&p->rulePos, '\\');
	while ( cur ) {
		rule.name = cur;
		cur = Q3_nextToken(&p->rulePos, '\\');
		rule.value = cur;
		Q3_checkRule(p, rule, &m);
		UI_insertRule(p, rule);
		cur = Q3_nextToken(&p->rulePos, '\\');
	}
	
	//parse players
  count=0;
  cur = Q3_nextToken(&p->playerPos, '\n' );
	while ( cur ) {
	 	count++;
    player.frags = cur;
		pos = strchr(cur, ' ');
		assert(pos);
		*pos = '\0';
		player.ping = ++pos;
		pos = strchr(pos, ' ');
		assert(pos);
		*pos = '\0';
		player.name = ++pos;
		
    if (m.OSP_gametype >= 0 && isChecked(p->subMenu, 3)) { // encode info in player name
      Q3_handleOSP_Player(p, tmp, 128, player, count, m);
      player.name = tmp;      
    }
    UI_insertPlayer(p, player);
		cur = Q3_nextToken(&p->playerPos, '\n' );
	}		
  
  sprintf(p->numplayers, "%d", count);

  if (m.OSP_gametype >= 0) {
    setGrayed(p->subMenu, 3, FALSE);
    if (isChecked(p->subMenu, 3)) {
      Q3_OSP_isInTeam(0, m.OSP_players_blue,&m.count_blue);
      Q3_OSP_isInTeam(0, m.OSP_players_red,&m.count_red);
      Q3_OSP_isInTeam(0, m.OSP_players_active,&m.count_active);
      m.count_specs = count - (m.count_blue + m.count_red + m.count_active);
      Q3_handleOSP_Rules(p, m, count);
      if (p->sortCol2 == 1 && p->sortDir2 == -1) // override default sorting (frags) for OSP
			  p->sortCol2 = 2;
    }
	}

	// now we can free the buffer and close socket
  free(p->buffer);
  closesocket(p->sock);

	p->playerPos = NULL;
	p->rulePos = NULL;

  return TRUE;
}

// Sends a status-request packet to destination server */
BOOL Q3_sendPacket(PluginInstance *p) {
  struct sockaddr_in sa;
  LARGE_INTEGER t;
  HOSTENT *host;
  int err;

  assert(p);

  /* create new socket */
  p->sock = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
  if (p->sock == INVALID_SOCKET) {
    wsprintf(p->text, "Socket error (%d)", WSAGetLastError());
    return FALSE;
  }
    
  /* set parameters */
  sa.sin_family = AF_INET;
  sa.sin_port = htons(atoi(p->port));
  sa.sin_addr.S_un.S_addr = inet_addr(p->server);

  if (sa.sin_addr.S_un.S_addr == -1) {
    host = gethostbyname(p->server);
    if (!host) {
      strcpy(p->text, "Invalid server name");
      return FALSE;
    }
    memcpy(&(sa.sin_addr.s_addr), host->h_addr, sizeof(int));
  }
    
  /* send it (finally) */
  err= sendto(p->sock, Q3_getStatus, sizeof(Q3_getStatus), 0, (sockaddr*) &sa, sizeof(sa));
  if (err == SOCKET_ERROR) {
    wsprintf(p->text, "Send error (%d)", WSAGetLastError());
    return FALSE;
  }

  if (QueryPerformanceCounter(&t))
		p->ticks = t.QuadPart;
	else
		p->ticks = GetTickCount();
 
  /* set socket to non-blocking mode */
  err=WSAAsyncSelect(p->sock, p->fhWnd, WM_SERVER_RESPONSE, FD_READ);
  if (err == SOCKET_ERROR) {
    wsprintf(p->text, "Select error (%d)", WSAGetLastError());
    return FALSE;
  }
  
  return TRUE;
}


// custom draw procedure for Q3 colored names and display of encoded names
// in case of OSP
BOOL Q3_drawName(PluginInstance *This, HDC hdc, RECT win, char *name) {
	HGDIOBJ object;
	int iWidth,diameter=0,offset;
  COLORREF col;
  HBRUSH brush;
  
  object = (HGDIOBJ) SendMessage(This->listbox2, WM_GETFONT, 0,0);
  
  if (!object) 
    object = GetStockObject(ANSI_VAR_FONT);

  if (!object)
    return FALSE;

  COLORREF save = Q3_colors[0];
  Q3_colors[0] = This->color;

  SelectObject(hdc, object);
	SetTextAlign(hdc, TA_LEFT|TA_TOP);
	SetTextColor(hdc,Q3_colors[0]);        // black is default

	while (*name) {
		if (*name == '^' && (*(name+1) >= '0') && (*(name+1) <= '9')) {  // change text color?
			SetTextColor(hdc, Q3_colors[*(name+1)-'0']);
		  name++;
    } else if (*name == '\\') {
      name+=6; // skip score                                         // do we have to draw a team symbol?
      switch (*name) {
        case 'R': col = Q3_colors[1]; break;
        case 'B': col = Q3_colors[4]; break;
        case 'S': col = Q3_colors[6]; break;
      }
      diameter = (win.bottom - win.top)/2;
      if (*name == 'R' || *name == 'B' || *name == 'S') {
        brush = CreateSolidBrush(col);
        SelectObject(hdc, brush);
        offset = ((win.bottom - win.top) - diameter)/2;
        Ellipse(hdc, win.left, win.top+offset, win.left+diameter, win.top+diameter+offset);
        SelectObject(hdc, GetStockObject(BLACK_BRUSH));
        DeleteObject(brush);
      }
      if (*name == 'D')
        diameter = 0;
      else
        diameter += 3;
      // name-=6;
    } else if (*name != '"') {					// skip "                    // it's just a plain character
		  TextOut(hdc, win.left+diameter,win.top,name,1); 
		  GetCharWidth(hdc, *name, *name, &iWidth);
		  win.left+= iWidth; 
		}
		name++;
	}
  Q3_colors[0] = save;
  return TRUE;   // we return TRUE in order to signal that we have drawn the name
}
