/*
    Copyright 2005-2013 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "md5.h"

#ifdef WIN32
    #include <winsock.h>
    #include <malloc.h>
    #include "winerr.h"

    #define close   closesocket
    #define sleep   Sleep
#else
    #include <unistd.h>
    #include <sys/socket.h>
    #include <sys/types.h>
    #include <arpa/inet.h>
    #include <netinet/in.h>
    #include <netdb.h>

    #define stristr strcasestr
#endif

typedef unsigned int    u32;
typedef unsigned short  u16;
typedef unsigned char   u8;



#define VER         "0.1.1"
#define MS          "master.qtracker.com"
#define MSPORT      47569
#define BUFFSZ      8192
#define LINESZ       128
#define CFG         "qtracklist.cfg"
#define C1          "\xfe\xfe\xfe\xff"
#define C2          "\xfe\xfd\xfe\xff"
#define S1          "\xfe\xfc\xfe\xff"
#define S2          "\xfe\xfb\xfe\xff"
#define REGIONS     "/region/1/region/2/region/3/region/4/region/5/region/6/region/8/region/0/"

#define SEND(x,y)   if(send(sd, x, y, 0) \
                      < 0) std_err(); \
                    fputc('.', stdout);
#define RECV(x)     for(len = 0; len < x; len += t) { \
                        t = recv(sd, buff + len, x - len, 0); \
                        if(t <= 0) goto disconnect; \
                        fputc('.', stdout); \
                    }
#define CHKREPLY(x) if(memcmp(buff, x, sizeof(x) - 1)) { \
                        printf("\n" \
                            "Error: unknown reply from server:\n" \
                            "       Expected   0x%08x\n" \
                            "       Received   0x%08x\n" \
                            "\n", *(u32 *)x, *(u32 *)buff); \
                        exit(1); \
                    }



char *stristr(const char *String, const char *Pattern);
void show_hex(char *title, u8 *data, int len);
void qtracker_handshake(u8 *ch);
void show_list(u8 *name);
u32 resolv(char *host);
void std_err(void);



int main(int argc, char *argv[]) {
    struct  sockaddr_in peer;
    u32  servers,
            ip;
    int     sd,
            t,
            len,
            i,
            dynsz,
            execlen          = 0,
            gamenum          = 0,
            quiet            = 0;
    u16     msport           = MSPORT,
            port;
    u8      tmp[4],
            *buff            = NULL,
            *ms              = MS,
            *tmpexec         = NULL,
            *execstring      = NULL,
            *execstring_ip   = NULL,
            *execstring_port = NULL,
            *execptr         = NULL,
            *p,
            *p1,
            *p2;

#ifdef WIN32
    WSADATA wsadata;
    WSAStartup(MAKEWORD(1,0), &wsadata);
#endif

    setbuf(stdout, NULL);
    setbuf(stderr, NULL);

    fputs("\n"
        "Qtracklist " VER "\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 2) {
        printf("\n"
            "Usage: %s <options>\n"
            "\n"
            "Options:\n"
            "-n NUM        to know the servers list of the game specified by NUM, this\n"
            "              number is visible when using the -l or -s option\n"
            "-l            shows the complete list of supported games and their numbers\n"
            "-s PATTERN    searchs the game name or parts of it available in the list of\n"
            "              supported games ("CFG").\n"
            "              It is case insensitve, so -s UNREAL and -s unReaL are the same\n"
            "-r \"prog...\"  lets you to execute a specific program for each IP found (as the\n"
            "              ping command for example).\n"
            "              There are 2 available parameters and are #IP and #PORT that will\n"
            "              be substituited with the IP and port of the current online game\n"
            "              server found in the list. Example: -r \"echo #IP and #PORT yeah\"\n"
            "-q            quieter output, are showed only IP and ports without additional\n"
            "              informations using the normal format IP:PORT\n"
            "-x S[:P]      lets you to specify a different master server (S) and port (P,\n"
            "              optional) instead of "MS":%d\n"
            "\n"
            " Note: the updated configuration file (if there is a new one) is available on\n"
            "       my website: http://aluigi.org/papers.htm#qtracklist\n"
            "\n", argv[0], msport);
        exit(1);
    }

    for(i = 1; i < argc; i++) {
        switch(argv[i][1]) {
            case 'n': {
                i++;
                if(!argv[i]) {
                    fputs("\n"
                        "Error: you MUST specify a game number\n"
                        "\n", stderr);
                    exit(1);
                }
                gamenum = atoi(argv[i]);
                break;
            }
            case 'l': {
                show_list(NULL);
                return(0);
                break;
            }
            case 's': {
                i++;
                if(!argv[i]) {
                    fputs("\n"
                        "Error: you MUST specify a text pattern to search in the game database\n"
                        "\n", stderr);
                    exit(1);
                }
                show_list(argv[i]);
                return(0);
                break;
            }
            case 'r': {
                i++;
                execstring = argv[i];
                execlen = strlen(execstring) + 23;

                tmpexec = malloc(execlen);
                if(!tmpexec) std_err();

                execstring_ip = strstr(execstring, "#IP");
                execstring_port = strstr(execstring, "#PORT");
                if(execstring_ip) *execstring_ip = 0;
                if(execstring_port) *execstring_port = 0;

                execlen = strlen(execstring);
                memcpy(tmpexec, execstring, execlen);
                break;
            }
            case 'q': {
                quiet = 1;
                break;
            }
            case 'x': {
                i++;
                if(!argv[i]) {
                    fputs("\n"
                        "Error: some parameters are missing\n"
                        "\n", stderr);
                    exit(1);
                }
                ms = strchr(argv[i], ':');
                if(ms) {
                    msport = atoi(ms + 1);
                    *ms = 0;
                }
                ms = argv[i];
                break;
            }
            default: {
                fprintf(stderr, "\n"
                    "Error: wrong argument (%s)\n"
                    "\n", argv[i]);
                exit(1);
                break;
            }
        }
    }

    if(!gamenum) {
        fputs("\nError: no game number specified, use the -n option to specify it or -l and -s to find it\n\n", stderr);
        exit(1);
    }

    printf("\n"
        "  Game number: %d\n"
        "  Resolving    %s\n",
        gamenum,
        ms);

    peer.sin_addr.s_addr = resolv(ms);
    peer.sin_port        = htons(msport);
    peer.sin_family      = AF_INET;

    buff = malloc(BUFFSZ);
    if(!buff) std_err();
    dynsz = BUFFSZ;

    printf(
        "  Server:      %s:%hu\n",
        inet_ntoa(peer.sin_addr), msport);

    sd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
    if(sd < 0) std_err();

    fputs("- connect: ", stdout);
    if(connect(sd, (struct sockaddr *)&peer, sizeof(peer))
      < 0) std_err();
    fputs("ok\n", stdout);

    SEND(C1, sizeof(C1) - 1);
    RECV(4);

    CHKREPLY(S1);
    RECV(16);

    show_hex("\n- challenge: ", buff, 16);
    qtracker_handshake(buff);
    show_hex("- response:  ", buff, 16);

    SEND(C2, sizeof(C2) - 1);
    SEND(buff, 16);
    tmp[0] = 0;
    tmp[1] = 0;     // 0x80 for regions
    tmp[2] = gamenum;
    tmp[3] = gamenum >> 8;
    SEND(tmp, 4);

    //len = sizeof(REGIONS);
    //SEND((void *)&len, 4);
    //SEND(REGIONS, len);

    RECV(4);
    CHKREPLY(S2);

    RECV(4);
    servers = buff[0] | (buff[1] << 8) | (buff[1] << 16) | (buff[1] << 24);

    fputs("\n- receive data:   ", stdout);
    len = 0;
    for(;;) {
        t = recv(sd, buff + len, dynsz - len, 0);
        if(t <= 0) break;
        len += t;
        if(len >= dynsz) {
            dynsz += BUFFSZ;
            buff = realloc(buff, dynsz + 1);
            if(!buff) std_err();
        }
        fputc('.', stdout);
    }
    buff[len] = 0;

    fputs("\n\n", stdout);
    p = buff;
    while((p1 = strchr(p, '\n'))) {
        *p1 = 0;

        ip = inet_addr(p);
        p2 = strchr(p, '\t');
        if(p2) {
            port = atoi(p2 + 1);
        } else {
            port = 0;
        }

        if(quiet) {
            if(p2) {
                *p2 = ':';
                p2 = strchr(p2 + 1, '\t');
                if(p2) *p2 = 0;
            }
        }

        printf("%s\n", p);

        if(execstring) {
            execptr = tmpexec + execlen;
            if(execstring_ip && !execstring_port) {
                execptr += sprintf(execptr, "%s", inet_ntoa(*(struct in_addr *)&ip));
                strcpy(execptr, execstring_ip + 3);

            } else if(execstring_port && !execstring_ip) {
                execptr += sprintf(execptr, "%d", port);
                strcpy(execptr, execstring_port + 5);

            } else if(execstring_ip < execstring_port) {
                execptr += sprintf(execptr, "%s", inet_ntoa(*(struct in_addr *)&ip));
                execptr += sprintf(execptr, "%s", execstring_ip + 3);
                execptr += sprintf(execptr, "%d", port);
                strcpy(execptr, execstring_port + 5);

            } else if(execstring_port < execstring_ip) {
                execptr += sprintf(execptr, "%d", port);
                execptr += sprintf(execptr, "%s", execstring_port + 5);
                execptr += sprintf(execptr, "%s", inet_ntoa(*(struct in_addr *)&ip));
                strcpy(execptr, execstring_ip + 3);
            }

            printf("   Execute program: \"%s\"\n", tmpexec);
            system(tmpexec);
        }

        p = p1 + 1;
    }

    close(sd);

    printf("\n\nOnline there are %u servers\n\n", servers);

    free(buff);
    return(0);

disconnect:
    fputs("\nError: you have been disconnected for unknown reasons, probably the protocol is changed\n\n", stderr);
    exit(1);
    return(-1);
}



char *stristr(const char *String, const char *Pattern)
{
      char *pptr, *sptr, *start;

      for (start = (char *)String; *start; start++)
      {
            /* find start of pattern in string */
            for ( ; (*start && (toupper(*start) != toupper(*Pattern))); start++)
                  ;
            if (!*start)
                  return 0;

            pptr = (char *)Pattern;
            sptr = (char *)start;

            while (toupper(*sptr) == toupper(*pptr))
            {
                  sptr++;
                  pptr++;

                  /* if end of pattern then pattern was found */

                  if (!*pptr)
                        return (start);
            }
      }
      return 0;
}



void show_hex(char *title, u8 *data, int len) {
    const char  hex[] = "0123456789abcdef";

    fputs(title, stdout);
    while(len--) {
        fputc(hex[*data >> 4], stdout);
        fputc(hex[*data & 15], stdout);
        data++;
    }
    fputc('\n', stdout);
}



void qtracker_handshake(u8 *ch) {
    md5_context md5t;
    int     len;
    u8      tmp[128];

    len = sprintf(
        tmp,
        "R%xE%xM%xT%xT%xE%xR%xM%xA%xJ%xM%xC%xB%x%xR%xR%xM",
        ch[0],  ch[1],  ch[2],  ch[3],  ch[4],  ch[5],  ch[6],  ch[7],
        ch[8],  ch[9],  ch[10], ch[11], ch[12], ch[13], ch[14], ch[15]);

    md5_starts(&md5t);
    md5_update(&md5t, tmp, len);
    md5_finish(&md5t, ch);
}



void show_list(u8 *name) {
    FILE    *fd;
    u8      buff[LINESZ + 1];

    fputs("- open file "CFG"\n", stdout);
    fd = fopen(CFG, "rt");
    if(!fd) {
        fputs("\nError: the configuration file has not been found in the current directory", stderr);
        std_err();
    }

    fputs("\n"
        "  NUM\tGAMENAME\n"
        "-------------------------------------------------------------------------------"
        "\n", stdout);

    if(name) {
        while(fgets(buff, LINESZ, fd)) {
            if(stristr(buff, name)) fputs(buff, stdout);
        }
    } else {
        while(fgets(buff, LINESZ, fd)) {
            fputs(buff, stdout);
        }
    }
    fputc('\n', stdout);
    fclose(fd);
}



u32 resolv(char *host) {
    struct  hostent *hp;
    u32     host_ip;

    host_ip = inet_addr(host);
    if(host_ip == INADDR_NONE) {
        hp = gethostbyname(host);
        if(!hp) {
            fprintf(stderr, "\nError: Unable to resolv hostname (%s)\n", host);
            exit(1);
        } else host_ip = *(u32 *)(hp->h_addr);
    }
    return(host_ip);
}



#ifndef WIN32
    void std_err(void) {
        perror("\nError");
        exit(1);
    }
#endif


