/*
* Copyright (C) 2002-2003 r1ch.net
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
* 
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
*
* See the GNU General Public License for more details.
*  
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*	
*/

//Originally gloommaster 0.0.3 (c) 2002-2003 r1ch.net
//quake 2 compatible master server for gloom

// QwazyWabbit mods begin
// 11-FEB-2006
// Changes for Linux and Windows portability by Geoff Joy (QwazyWabbit)
// Simply build for Linux with "gcc -o q2master master.c"
//
// 26-JAN-2007
// Made it a general q2 server (q2master v1.0)
//
// 26-JUL-2007
// Added registry keys to tell service what IP to use.
// Added key and handling for non-standard port.
// Server could be modified for use as Q3 or Q4 master server.
//
// 18-AUG-2007
// Dynamic allocation of buffer in SendServerListToClient function.
// 
// 01-SEP-2007
// General release. This project builds in VC++ 6.0 and GCC 4.x
// Complete project easily ported to VC 2005 or higher.
//

// General command line switches:

// -debug	Asserts debug mode. The program prints status messages to console
//			while running. Shows pings, heartbeats, number of servers listed.

// -ip xxx.xxx.xxx.xxx causes server to bind to a particular IP address when
//	used with multi-homed hosts. Default is 0.0.0.0 (any).

// -port xxxxx causes server to bind to a particular port. Default is 27900.
// Default port for Quake 2 master servers is 27900. If you depart from this
// you need to communicate this to your users somehow. This feature is included
// since this code could be modified to provide master services for other games.

// *** Windows ***

// Usage:
// Place executable in %systemroot%\system32 or other known location.
// To debug it as a console program, command: "master -debug" and it outputs
// status messages to the console. Ctrl-C shuts it down.
//
// From a cmd prompt type: q2master -install to install the service with defaults.
// The service will be installed as "Q2MasterServer" in the Windows service list.
//

// -install	Installs the service on Windows.
// -remove	Stops and removes the service.
// When the service is installed it is installed with "Automatic" startup type
// to allow it to start up when Windows starts. Use SCM to change this.

// Other commands:
// net start q2masterserver to start the service.
// net stop q2masterserver to stop the service.
//
// Use the Services control panel applet to start/stop, change startup modes etc.
// To uninstall the server type "master -remove" to stop and remove the active service.

// *** Linux ***

// Usage:
// -debug Sets process to debug mode and it remains attached to console.
// If debug is not specified on command line the process forks a daemon and
// detaches from terminal.
//
// Send the process a SIGTERM to stop the daemon. "kill -n SIGTERM <pid>"
// Use "netstat -anup" to see the processes/pids listening on UDP ports.
// Use "ps -ux" to list detached processes, this will show the command line that
// invoked the q2master process.
// 
// *** Mac/iMac OS/X ***
// Usage:
// Same as Linux
// Compiles on OS/X same as Linux & BSD "gcc -o q2master master.c".
//

#ifdef WIN32
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <stdio.h>
#include <stdlib.h>
#include <winsock.h>
#include <winerror.h>
#include <time.h>
#include <winwrap.h>
#include <process.h>
#include "service.h"
#include "performance.h"
#include <assert.h>

// Windows Service structs
SERVICE_STATUS          MyServiceStatus; 
SERVICE_STATUS_HANDLE   MyServiceStatusHandle;

void SetQ2MasterRegKey(char* name, char *value);
void GetQ2MasterRegKey(char* name, char *value);

#else

// Linux and Mac versions
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <errno.h>
#include <unistd.h>
#include <sys/signal.h>
#include <assert.h>

enum {FALSE, TRUE};

// stuff not defined in sys/socket.h
#define SOCKET unsigned int
#define SOCKET_ERROR -1
#define TIMEVAL struct timeval

// portability, rename or delete functions
#define WSAGetLastError() errno
#define _strnicmp strncasecmp
#define My_Main main
#define SetQ2MasterRegKey
#define	GetQ2MasterRegKey
#define WSACleanup()
#define Sleep(x)	sleep((x/1000))
void signal_handler(int sig);

#endif

// for debugging as a console application in Windows or in Linux
int Debug;
unsigned long numservers;	// global count of the currently listed servers

int runmode;	// server loop control
#define SRV_RUN		1
#define SRV_STOP	0
#define SRV_STOPPED	-1

typedef struct server_s server_t;

struct server_s {
	server_t		*prev;
	server_t		*next;
	struct sockaddr_in	ip;
	unsigned short	port;
	unsigned int	queued_pings;
	unsigned int	heartbeats;
	unsigned long	last_heartbeat;
	unsigned long	last_ping;
	unsigned char	shutdown_issued;
	unsigned char	validated;
};

server_t servers;

struct sockaddr_in listenaddress;
SOCKET out;
SOCKET listener;
TIMEVAL delay;

#ifdef WIN32
WSADATA ws;
#endif

fd_set set;
char incoming[150000];
int retval;

#define KEY_LEN 32	// give us some space
char bind_ip[KEY_LEN] = "0.0.0.0"; // default IP to bind
char bind_port[KEY_LEN] = "27900";	// default port to bind

//
// These are Windows specific but need to be defined here so GCC won't barf
//
#define REGKEY_Q2MASTERSERVER "SOFTWARE\\Q2MasterServer" // Our config data goes here
#define REGKEY_BIND_IP "Bind_IP"
#define REGKEY_BIND_PORT "Bind_Port"

// Out of band data preamble
#define OOB_SEQ "\xff\xff\xff\xff" //32 bit integer (-1) as string sequence for out of band data
#define serverstring OOB_SEQ"servers " // 12 bytes for the serverstring header

void RunFrame (void);
void ExitNicely (void);
void DropServer (server_t *server);
int  AddServer (struct sockaddr_in *from, int normal);
void QueueShutdown (struct sockaddr_in *from, server_t *myserver);
void SendServerListToClient (struct sockaddr_in *from);
void Ack (struct sockaddr_in *from);
int  HeartBeat (struct sockaddr_in *from, char *data);
void ParseCommandLine(int argc, char *argv[]);
int ParseResponse (struct sockaddr_in *from, char *data, int dglen);

//
// This becomes main for Linux
// In Windows, main is in service.c and it decides if we're going to see a console or not
// this function gets called when we have decided if we are a server or a console app.
//
int My_Main (int argc, char *argv[])
{
	int len, err;
	unsigned fromlen;
	struct sockaddr_in from;
	
	printf ("Q2-Master 1.1 originally GloomMaster\n(c) 2002-2003 r1ch.net, modifications by QwazyWabbit 2007\n");
	numservers = 0;
	
#ifdef WIN32
	// overhead to tell Windows we're using TCP/IP.
	err = WSAStartup ((WORD)MAKEWORD (1,1), &ws);
	if (err) {
		printf("Error loading Windows Sockets! Error: %i\n",err);
		return (err);
	}
#endif
	
#ifndef WIN32	// Already done in ServiceStart() if Windows
	ParseCommandLine(argc, argv);
#endif
	
	printf("Debugging mode: %i\n", Debug);
	listener = socket (AF_INET, SOCK_DGRAM, IPPROTO_UDP);
	out = socket (AF_INET, SOCK_DGRAM, IPPROTO_UDP);
	memset (&listenaddress, 0, sizeof(listenaddress));
	
	// only in Windows, null def in Linux
	GetQ2MasterRegKey(REGKEY_BIND_IP, bind_ip);
	GetQ2MasterRegKey(REGKEY_BIND_PORT, bind_port);
	
	listenaddress.sin_addr.s_addr = inet_addr(bind_ip); 
	listenaddress.sin_family = AF_INET;
	listenaddress.sin_port = htons((unsigned short)atoi(bind_port));
	
	if ((bind (listener, (struct sockaddr *)&listenaddress, sizeof(listenaddress))) == SOCKET_ERROR)
	{
		printf ("[E] Couldn't bind to port %s UDP (something is probably using it)\n", bind_port);
		return 1;
	}
	
	delay.tv_sec = 1;
	delay.tv_usec = 0;
	
	FD_SET (listener, &set);
	
	fromlen = (unsigned)sizeof(from);
	memset (&servers, 0, sizeof(servers));
	printf ("listening on %s:%s (UDP)\n", bind_ip, bind_port);
	runmode = SRV_RUN; // set loop control
	
#ifndef WIN32
	// in Linux or BSD we fork a daemon
	if (!Debug) {			// ...but not if debug mode
		if (daemon(0,0) < 0) {	
			printf("Forking error, running as console, error number was: %i\n", errno);
			Debug = 1;
		}
	}
	
	if (!Debug) 
	{
		signal(SIGCHLD, SIG_IGN); /* ignore child */
		signal(SIGTSTP, SIG_IGN); /* ignore tty signals */
		signal(SIGTTOU, SIG_IGN);
		signal(SIGTTIN, SIG_IGN);
		signal(SIGHUP, signal_handler); /* catch hangup signal */
		signal(SIGTERM, signal_handler); /* catch terminate signal */
	}
#endif
	
	while (runmode == SRV_RUN) // 1 = running, 0 = stop, -1 = stopped.
	{
		FD_SET (listener, &set);
		delay.tv_sec = 1;
		delay.tv_usec = 0;
		
		retval = select(listener + 1, &set, NULL, NULL, &delay);
		if (retval == 1)
		{
			len = recvfrom (listener, incoming, sizeof(incoming), 0, (struct sockaddr *)&from, &fromlen);
			if (len != SOCKET_ERROR)
			{
				if (len > 4)
				{
					//parse this packet
					if (!ParseResponse (&from, incoming, len)) {
						err = 50;
						runmode = SRV_STOP;	// something went wrong, AddServer failed?
					}
				}
				else
				{
					if(Debug)
						printf ("[W] runt packet from %s:%d\n", inet_ntoa (from.sin_addr), ntohs(from.sin_port));
				}
				
				//reset for next packet
				memset (incoming, 0, sizeof(incoming));
			} 
			else 
			{
				if(Debug)
					printf ("[E] socket error during select from %s:%d (%d)\n", 
					inet_ntoa (from.sin_addr), 
					ntohs(from.sin_port), 
					WSAGetLastError());
			}
		}
		
		// destroy old servers, etc
		RunFrame();
	}
	
	WSACleanup();	// Windows Sockets cleanup
	runmode = SRV_STOPPED;
	return(err);
}

//
// Called by ServiceCtrlHandler after the server loop is dead
// this frees the server memory allocations.
//
void ExitNicely (void)
{
	server_t	*server = &servers;
	server_t	*old = NULL;
	
	printf ("[I] shutting down.\n");
	
	while (server->next)
	{
		if (old)
			free (old);
		server = server->next;
		old = server;
	}
	
	if (old)
		free (old);
}

void DropServer (server_t *server)
{
	//unlink
	if (server->next)
		server->next->prev = server->prev;
	
	if (server->prev)
		server->prev->next = server->next;
	
	if (numservers != 0) 
		numservers--;
	
	free (server);
}

//
// returns TRUE if successfully added to list
//
int AddServer (struct sockaddr_in *from, int normal)
{
	server_t	*server = &servers;
	int			preserved_heartbeats = 0;
	
	while (server->next)
	{
		server = server->next;
		if (*(int *)&from->sin_addr == *(int *)&server->ip.sin_addr && from->sin_port == server->port)
		{
			//already exists - could be a pending shutdown (ie killserver, change of map, etc)
			if (server->shutdown_issued)
			{
				if(Debug)
					printf ("[I] scheduled shutdown server %s sent another ping!\n",
					inet_ntoa(from->sin_addr));
				DropServer (server);
				server = &servers;
				while (server->next)
					server = server->next;
				break;
			}
			else
			{
				if(Debug)
					printf ("[W] dupe ping from %s:%u!! ignored.\n",
					inet_ntoa (server->ip.sin_addr),
					htons(server->port));
				return TRUE;
			}
		}
	}
	
	server->next = (server_t *)malloc(sizeof(server_t));
	assert(server->next != NULL); // malloc failed if server->next == NULL
	// Note: printf implicitly calls malloc so this is likely to fail if we're really out of memory
	if (server->next == NULL) {
		printf("Fatal Error: memory allocation failed in AddServer\n");
		return FALSE;
	}
	
	server->next->prev = server;
	server = server->next;
	server->heartbeats = preserved_heartbeats;
	memcpy (&server->ip, from, sizeof(server->ip));
	server->last_heartbeat = time(NULL);
	server->next = NULL;
	server->port = from->sin_port;
	server->shutdown_issued = 0;
	server->queued_pings = 0;
	server->last_ping = 0;
	server->validated = 0;
	
	numservers++;
	
	if(Debug)
		printf ("[I] server %s:%u added to queue! (%d) number: %u\n",
		inet_ntoa (from->sin_addr),
		htons(server->port),
		normal,
		numservers);
	
	if (normal) 
	{
		struct sockaddr_in addr;
		memcpy (&addr.sin_addr, &server->ip.sin_addr, sizeof(addr.sin_addr));
		addr.sin_family = AF_INET;
		addr.sin_port = server->port;
		memset (&addr.sin_zero, 0, sizeof(addr.sin_zero));
		sendto (listener, OOB_SEQ"ack", 7, 0, (struct sockaddr *)&addr, sizeof(addr));
	}
	return TRUE;
}

//
// We received a shutdown frame from a server, set the shutdown flag
// for it and send it a ping to ack the shutdown frame.
//
void QueueShutdown (struct sockaddr_in *from, server_t *myserver)
{
	server_t	*server = &servers;
	
	if (!myserver)
	{
		while (server->next)
		{
			server = server->next;
			if (*(int *)&from->sin_addr == *(int *)&server->ip.sin_addr && from->sin_port == server->port)
			{
				myserver = server;
				break;
			}
		}
	}
	
	if (myserver)
	{
		struct sockaddr_in addr;
		memcpy (&addr.sin_addr, &myserver->ip.sin_addr, sizeof(addr.sin_addr));
		addr.sin_family = AF_INET;
		addr.sin_port = server->port;
		memset (&addr.sin_zero, 0, sizeof(addr.sin_zero));
		
		//hack, server will be dropped in next minute IF it doesn't respond to our ping
		myserver->shutdown_issued = 1;
		
		if(Debug)
			printf ("[I] shutdown queued %s:%u \n", inet_ntoa (myserver->ip.sin_addr), htons(server->port));
		
		sendto (listener, OOB_SEQ"ping", 8, 0, (struct sockaddr *)&addr, sizeof(addr));
		return;
	}
	
	else
	{
		if(Debug)
			printf ("[W] shutdown issued from unregistered server %s!\n", inet_ntoa (from->sin_addr));
	}
}

//
// Walk the server list and ping them as needed, age the ones
// we have not heard from in a while and when they get too
// old, remove them from the list.
//
void RunFrame (void)
{
	server_t		*server = &servers;
	unsigned int	curtime = time(NULL);
	
	while (server->next)
	{
		server = server->next;
		if (curtime - server->last_heartbeat > 60)
		{
			server_t *old = server;
			
			server = old->prev;
			
			if (old->shutdown_issued || old->queued_pings > 6)
			{
				if(Debug)
					printf ("[I] %s:%u shut down.\n", inet_ntoa (old->ip.sin_addr), htons(server->port));
				DropServer (old);
				continue;
			}
			
			server = old;
			
			if (curtime - server->last_ping >= 10)
			{
				struct sockaddr_in addr;
				memcpy (&addr.sin_addr, &server->ip.sin_addr, sizeof(addr.sin_addr));
				addr.sin_family = AF_INET;
				addr.sin_port = server->port;
				memset (&addr.sin_zero, 0, sizeof(addr.sin_zero));
				server->queued_pings++;
				server->last_ping = curtime;
				if(Debug)
					printf ("[I] ping %s:%u\n", inet_ntoa (server->ip.sin_addr), htons(server->port));
				sendto (listener, OOB_SEQ"ping", 8, 0, (struct sockaddr *)&addr, sizeof(addr));
			}
		}
	}
}

//
// This function assembles the serverstring preamble and 6 bytes for each
// listed server into a buffer for transmission to the client in response
// to a query frame.
//
void SendServerListToClient (struct sockaddr_in *from)
{
	int				buflen;
	char			*buff;
	server_t		*server = &servers;
	unsigned long	servercount;
	unsigned long	bufsize;
	
	// assume buffer size needed is for all current servers (numservers)
	// and eligible servers in list will always be less or equal to numservers
	
	bufsize = 12 + 6 * (numservers + 1); // 12 bytes for serverstring, 6 bytes for game server ip and port
	buflen = 0;
	buff = malloc (bufsize);
	assert(buff != NULL);	// catch it in debug
	if (buff == NULL) {
		printf("Fatal Error: memory allocation failed in SendServerListToClient\n");
		return;
	}
	
	memset (buff, 0, bufsize);
	memcpy (buff, serverstring, 12);	// 12 = length of serverstring
	buflen += 12;
	servercount = 0;
	
	while (server->next)
	{
		server = server->next;
		if (server->heartbeats >= 2 && !server->shutdown_issued && server->validated)
		{
			memcpy (buff + buflen, &server->ip.sin_addr, 4);
			buflen += 4;
			
			memcpy (buff + buflen, &server->port, 2);
			buflen += 2;
			
			servercount++;
		}
	}
	
	if(Debug)
		printf ("[I] query response (%d bytes) sent to %s:%d\n", buflen, inet_ntoa (from->sin_addr), ntohs (from->sin_port));
	
	if ((sendto (listener, buff, buflen, 0, (struct sockaddr *)from, sizeof(*from))) == SOCKET_ERROR)
	{
		if(Debug)
			printf ("[E] socket error on send! code %d.\n", WSAGetLastError());
	}
	
	if(Debug)
		printf ("[I] sent server list to client %s, servers: %u of %u\n", 
				inet_ntoa (from->sin_addr), 
				servercount, /* sent */
				numservers); /* on record */
	
	free(buff);
}

void Ack (struct sockaddr_in *from)
{
	server_t	*server = &servers;
	
	//iterate through known servers
	while (server->next)
	{
		server = server->next;
		//a match!
		if (*(int *)&from->sin_addr == *(int *)&server->ip.sin_addr && from->sin_port == server->port)
		{
			if(Debug)
				printf ("[I] ack from %s:%u (%d).\n",
				inet_ntoa (server->ip.sin_addr),
				htons(server->port),
				server->queued_pings);
			
			server->last_heartbeat = time(NULL);
			server->queued_pings = 0;
			server->heartbeats++;
			return;
		}
	}
}

int HeartBeat (struct sockaddr_in *from, char *data)
{
	server_t	*server = &servers;
	int status;
	
	status = TRUE;
	
	//walk through known servers
	while (server->next)
	{
		server = server->next;
		//a match!
		if (*(int *)&from->sin_addr == *(int *)&server->ip.sin_addr && from->sin_port == server->port)
		{
			struct sockaddr_in addr;
			
			memcpy (&addr.sin_addr, &server->ip.sin_addr, sizeof(addr.sin_addr));
			addr.sin_family = AF_INET;
			addr.sin_port = server->port;
			memset (&addr.sin_zero, 0, sizeof(addr.sin_zero));
			
			server->validated = 1;
			server->last_heartbeat = time(NULL);
			if(Debug)
				printf ("[I] heartbeat from %s:%u.\n",
				inet_ntoa (server->ip.sin_addr),
				htons(server->port));
			
			sendto (listener, OOB_SEQ"ack", 7, 0, (struct sockaddr *)&addr, sizeof(addr));
			return status;
		}
	}
	//we didn't find server in our list
	status = AddServer (from, 0);
	return status; // false if AddServer failed
}

int ParseResponse (struct sockaddr_in *from, char *data, int dglen)
{
	char *cmd = data;
	char *line = data;
	int	status = TRUE;
	
	if (_strnicmp (data, "query", 5) == 0 || _strnicmp (data, OOB_SEQ"getservers", 14) == 0)
	{
		if(Debug)
			printf ("[I] %s:%d : query (%d bytes)\n",
			inet_ntoa(from->sin_addr),
			htons(from->sin_port),
			dglen);
		
		SendServerListToClient (from);
	}
	else
	{
		while (*line && *line != '\n')
			line++;
		
		*(line++) = '\0';
		cmd += 4;
		
		if(Debug)
			printf ("[I] %s: %s (%d bytes)\n",
			cmd,
			inet_ntoa(from->sin_addr),
			dglen);
		
		if (_strnicmp (cmd, "ping", 4) == 0)
		{
			status = AddServer (from, 1);
		}
		else if (_strnicmp (cmd, "heartbeat", 9) == 0 || _strnicmp (cmd, "print", 5) == 0)
		{
			status = HeartBeat (from, line);
		}
		else if (strncmp (cmd, "ack", 3) == 0)
		{
			Ack (from);
		}
		else if (_strnicmp (cmd, "shutdown", 8) == 0)
		{
			QueueShutdown (from, NULL);
		}
		else
		{
			if(Debug) printf ("[W] Unknown command from %s!\n", inet_ntoa (from->sin_addr));
		}
	}
	return status;
}

void ParseCommandLine(int argc, char *argv[])
{
	int i = 0;
	
	if (argc >= 2)
		Debug = 3; //initializing
	for (i = 1; i < argc; i++) 
	{
		if (Debug == 3) 
		{
			if(_strnicmp(argv[i] + 1,"debug", 5) == 0)
				Debug = TRUE;	//service debugged as console
			else
				Debug = FALSE;
		}
		
		if(_strnicmp((char*)argv[i] + 1,"ip", 2) == 0)
		{
			//bind_ip, a specific host ip if desired
			strncpy(bind_ip, (char*)argv[i] + 4, sizeof(bind_ip) - 1);
			SetQ2MasterRegKey(REGKEY_BIND_IP, bind_ip);
		}
		
		if(_strnicmp((char*)argv[i] + 1,"port", 4) == 0)
		{
			//bind_port, if other than default port
			strncpy(bind_port, (char*)argv[i] + 6, sizeof(bind_port) - 1);
			SetQ2MasterRegKey(REGKEY_BIND_PORT, bind_port);
		}
	}
}

//
// This stuff plus a modified service.c and service.h
// from the Microsoft examples allows this application to be
// installed as a Windows service.
//

#ifdef WIN32

void ServiceCtrlHandler (DWORD Opcode) 
{
    switch(Opcode) 
    { 
	case SERVICE_CONTROL_STOP: 
		// Kill the server loop. 
		runmode = SRV_STOP; // zero the loop control
		while(runmode == SRV_STOP)	//give loop time to die
		{
			int i = 0;
			
			Sleep(500);	// SCM times out in 3 secs.
			i++;		// we check twice per sec.
			if(i >=	6)	// hopefully we beat the SCM timer
				break;	// still no return? rats, terminate anyway
		}
		
		ExitNicely();
		
		MyServiceStatus.dwWin32ExitCode = 0; 
		MyServiceStatus.dwCurrentState  = SERVICE_STOPPED; 
		MyServiceStatus.dwCheckPoint    = 0; 
		MyServiceStatus.dwWaitHint      = 0; 
		
		if(MyServiceStatusHandle)
			SetServiceStatus (MyServiceStatusHandle, &MyServiceStatus);
		return; 
    } 
    // Send current status. 
    SetServiceStatus (MyServiceStatusHandle,  &MyServiceStatus);
} 

void ServiceStart (DWORD argc, LPTSTR *argv) 
{ 
	ParseCommandLine(argc, argv); // we call it here and in My_Main
	
	MyServiceStatus.dwServiceType        = SERVICE_WIN32; 
	MyServiceStatus.dwCurrentState       = SERVICE_START_PENDING; 
	MyServiceStatus.dwControlsAccepted   = SERVICE_ACCEPT_STOP; 
	MyServiceStatus.dwWin32ExitCode      = 0; 
	MyServiceStatus.dwServiceSpecificExitCode = 0; 
	MyServiceStatus.dwCheckPoint         = 0; 
	MyServiceStatus.dwWaitHint           = 0; 
	
	MyServiceStatusHandle = RegisterServiceCtrlHandler(argv[0], 
		(LPHANDLER_FUNCTION)ServiceCtrlHandler); 
	
	if (!Debug && MyServiceStatusHandle == (SERVICE_STATUS_HANDLE)0) {
		printf("%s not started.\n", SZSERVICEDISPLAYNAME);
		return;
	}
	
	// Initialization complete - report running status. 
	MyServiceStatus.dwCurrentState       = SERVICE_RUNNING; 
	MyServiceStatus.dwCheckPoint         = 0; 
	MyServiceStatus.dwWaitHint           = 0; 
	
	if (!Debug)
		SetServiceStatus (MyServiceStatusHandle, &MyServiceStatus);
	
	My_Main(argc, &argv[0]);
} 

void ServiceStop(void)
{
	ServiceCtrlHandler (SERVICE_CONTROL_STOP);
}

/* 
* This sets the registry keys in "HKLM/Software/Q2MasterServer" so we can tell
* the service what IP address or port to bind to when starting up. If it's not preset
* the service will bind to 0.0.0.0:27900. Not critical on most Windows boxes
* but it can be a pain if you want to use multiple IP's on a NIC and don't want the
* service listening on all of them. Not as painful on Linux, we do the -ip switch
* in the command line.
*/
void SetQ2MasterRegKey(char* name, char *value)
{
	HKEY	hKey;
	DWORD	Disposition;
	LONG	status;
	
	status = RegCreateKeyEx(HKEY_LOCAL_MACHINE, 
		REGKEY_Q2MASTERSERVER,
		0, //always 0
		NULL,
		REG_OPTION_NON_VOLATILE,
		KEY_ALL_ACCESS,
		NULL,
		&hKey,
		&Disposition);
	
	if(Debug && status)
		printf("Error creating registry key for %s\n", SZSERVICEDISPLAYNAME);
	
	status = RegSetValueEx(hKey, name, 0, REG_SZ, value, strlen(value));
	
	if(Debug && status)
		printf("Registry key not set for IP: %s\n", bind_ip);
	
	RegCloseKey(hKey);
}

//
// As as Service, get the key and use the IP address stored there.
// If the key doesn't exist, it will be created.
// The user can add the Bind_IP or Bind_Port value 
// by hand or use the -ip x.x.x.x command line switch.
//
void GetQ2MasterRegKey(char* name, char *value)
{
	HKEY	hKey;
	DWORD	Disposition;
	LONG	status;
	DWORD	size = KEY_LEN;	// expected max size of the bind_ip or bind_port array
	
	// check value, create it if it doesn't exist
	status = RegCreateKeyEx(HKEY_LOCAL_MACHINE, 
		REGKEY_Q2MASTERSERVER,
		0, //always 0
		NULL,
		REG_OPTION_NON_VOLATILE,
		KEY_READ,
		NULL,
		&hKey,
		&Disposition);
	
	if(Debug && status)
		printf("Registry key not found\n");
	
	status = RegQueryValueEx(hKey, name, NULL, NULL, value, &size);

	if(Debug && status)
		printf("Registry value not found %s\\%s\n", REGKEY_Q2MASTERSERVER, name);
	
	RegCloseKey(hKey);
}

#else	// not doing windows

//
// handle Linux and BSD signals
//
void signal_handler(int sig)
{
	switch(sig) {
	case SIGHUP:
		break;
	case SIGTERM:
		runmode = SRV_STOP;
		while(runmode == SRV_STOP)	//give loop time to die
		{
			int i = 0;
			
			Sleep(500);	// 500 ms
			i++;		// we check twice per sec.
			if(i >=	6)
				break;	// still no return? rats, terminate anyway
		}
		
		ExitNicely();
		break;
	}
}

#endif
// end of Q2master
