/*******************************************************************

	These functions are part of the pathing system -- they are
	used to link nodes together automatically.

*******************************************************************/

#include "g_local.h"
#if compileJACKBOT
	

	/************************************************************

		Provide a list of unconnected nodes and networks (when
		everything is done right, there's only ONE network and no
		disconnected node)

	************************************************************/
	typedef struct nodenetwork_s
		{
		int	index;
		int	displayed;
		int count;
		vec3_t min;
		vec3_t max;
		} nodenetwork_t;

	// This code will cause issues in some instances
	// It also seem to eat up all the memory segments it can?!
	// Maybe there's an access violation going on somewhere
	// in there.
	void path_LinkCheckUp()
		{
		#if 0
		int i, j;
		int	numTotal = 0, numNetworks = 0, numIsolated = 0;
		unsigned short	*networkID;
		nodenetwork_t		*network, *nt;

		// Reserve memory
		networkID = (unsigned short *)malloc(jb_NumNodes * sizeof(unsigned short));
		if (!networkID)
			gi.dprintf("path_LinkCheckUp: Failed to allocate %i bytes of memory!\n", (int)(jb_NumNodes * sizeof(unsigned short)));
		memset(networkID, 0x00, jb_NumNodes * sizeof(unsigned short));

		network = (nodenetwork_t *)malloc(jb_NumNodes * sizeof(nodenetwork_t));
		if (!network)
			gi.dprintf("path_LinkCheckUp: Failed to allocate %i bytes of memory!\n", (int)(jb_NumNodes * sizeof(nodenetwork_t)));
		memset(network, 0x00, jb_NumNodes * sizeof(nodenetwork_t));

		gi.dprintf("-------- ROADMAP REPORT --------\n\n");

		// Group nodes in networks
		for (i = 0; i < jb_NumNodes; i++)
			{
			if (networkID[i])
				continue;
			// Link node to current network
			numTotal ++;
			networkID[i] = numTotal;
			// Setup network
			nt = &network[networkID[i]];
			nt->count			= 1;
			nt->displayed	= 0;
			VectorCopy(jb_Node[i].origin, nt->min);
			VectorCopy(jb_Node[i].origin, nt->max);
			// Count members
			for (j = 0; j < jb_NumNodes; j++)
				{
				if (i == j)
					continue;
				if (jb_PathTable[i][j] == BOTNODE_INVALID)
					{
					gi.dprintf("Failed to link Node[%i] to Node[%i], check Node[%i].\n", i, j, j);
					continue;
					}
				// Setup min
				if (jb_Node[j].origin[0] < nt->min[0])
					nt->min[0] = jb_Node[j].origin[0];
				if (jb_Node[j].origin[1] < nt->min[1])
					nt->min[1] = jb_Node[j].origin[1];
				if (jb_Node[j].origin[2] < nt->min[2])
					nt->min[2] = jb_Node[j].origin[2];
				// Setup max
				if (jb_Node[j].origin[0] > nt->max[0])
					nt->max[0] = jb_Node[j].origin[0];
				if (jb_Node[j].origin[1] > nt->max[1])
					nt->max[1] = jb_Node[j].origin[1];
				if (jb_Node[j].origin[2] > nt->max[2])
					nt->max[2] = jb_Node[j].origin[2];
				// Count and node network ID
				networkID[j] = networkID[i];
				nt->count ++;
				}
			// Network or isolated?
			if (nt->count == 1)
				{
				nt->index = -1;
				numIsolated ++;
				}
			else
				{
				nt->index = numNetworks;
				numNetworks ++;
				}
			}

		// REPORT //
		if ((numIsolated == 0) && (numNetworks == 1))
			gi.dprintf("All nodes are connected.\n");
		else
			{
			if (numIsolated)
				{
				for (i = 0; i < jb_NumNodes; i++)
					{
					if (network[networkID[i]].index == -1)
						gi.dprintf("Node[%i] at %s is completely isolated\n", i, vtos(jb_Node[i].origin));
					}
				}
			if (numTotal > 0)
				{
				for (i = 1; i <= numTotal; i++)
					{
					if ((network[i].index == -1) || (network[i].displayed))
						continue;
					network[i].displayed = 1;
					gi.dprintf("Network %i has %i nodes from %s to %s\n", network[i].index, network[i].count, vtos(network[i].min), vtos(network[i].max));
					}
				}
			gi.dprintf("** WARNING ** not all nodes are connected!\n");
			}
		gi.dprintf("\n--------------------------------\n");
		
		// Free memory
		free(networkID);
		free(network);
		#endif
		}
	


	/*****************************************************************

		Test for floor gaps; trace a vector between source and target.
		Starting at target, we move toward source, forcing a downward
		movement. Each fall is measured and the highest fall is stored
		for later use. If at some point during the travel we're stuck
		in a wall, go back up till we're out of the block and proceed
		forward. The the highest fall is bigger than the jumpable
		height, it means that if we were to miss our jump from source
		to target, we'd be stuck. So you'll have to jump.

	*****************************************************************/
	int trace_FloorGap(edict_t *ent, vec3_t source, vec3_t target)	// erhm, maybe you should copy source & target before messing with them (they act like pointers)
		{
		vec3_t	vN, v, vC, vX;
		trace_t	tr;
		float		dist, stepLength = 0, fall, maxFall = 0;
		vec3_t	boxMin  = {-8, -8, -(BOTMOVE_STEPSIZE / 2)};
		vec3_t	boxMax  = { 8,  8,  (BOTMOVE_STEPSIZE / 2)};

		source[2] = getFloor(source, boxMin, boxMax, ent);
		target[2] = getFloor(target, boxMin, boxMax, ent);

		// Get direction
		v[0] = target[0] - source[0];
		v[1] = target[1] - source[1];
		dist = sqrt((v[0] * v[0]) + (v[1] * v[1]));
		vN[0] = v[0] / dist;
		vN[1] = v[1] / dist;
		vN[2]	= 0;

		VectorCopy(target, vC); // target is on ground.

		while (stepLength < dist)
			{
			// Moving toward SOURCE
			stepLength += 8;
			vC[0] = target[0] - (vN[0] * stepLength);
			vC[1] = target[1] - (vN[1] * stepLength);
			// vC[2] is carried over from last move

			// If we hit something, go back up (no fall) -- beware with this code, could cause issues (endless loop).
			// REPLACE WITH float getSolidTop(vec3_t origin, vec3_t min, vec3_t max, edict_t *ignore, int maxHeight) ??
			tr = gi.trace(vC, boxMin, boxMax, vC, ent, MASK_SOLID);
			if ((tr.contents & (CONTENTS_SOLID | CONTENTS_WINDOW | CONTENTS_PLAYERCLIP)) || (tr.startsolid))
				{
				VectorCopy(vC, vX);
				while (true)
					{
					vX[2] += 8;
					tr = gi.trace(vX, boxMin, boxMax, vC, ent, MASK_SOLID);
					if (!tr.startsolid)
						{
						vC[2] = tr.endpos[2];
						break;
						}
					}
				continue;
				}

			// Hit nothing, try falling (keep track of height)
			VectorCopy(vC, v);
			vC[2] = getFloor(vC, boxMin, boxMax, ent);
			fall = v[2] - vC[2];
			if (fall > maxFall)
				maxFall = fall;
			}

		// If we were to walk from SOURCE to TARGET without jumping, we'd get stuck
		if (maxFall > BOTMOVE_STEPSIZE)
			return VEHICLE_JUMP;
		return VEHICLE_WALK;
		}


	/*****************************************************************

		Can we climb this thing? Trace forward from nodeSrc to nodeDst.
		If we hit a wall, go up (to the maximum jumpable height). If
		we're stuck in a wall, we can't jump over the obstacle. If we
		are not stuck, trace forward again, etc. The process is
		completed when either an unjumpable obstacle is found, or the
		whole distance has been covered. If the end of the trace is
		withing range of the destination point, we don't need to jump.
		Also checks for gaps in the floor via trace_FloorGap();

	*****************************************************************/
	int trace_Climb(edict_t *ent, vec3_t nodeSrc, vec3_t nodeDst, qboolean display)
		{
		vec3_t	boxMin  = {-16, -16, -(BOTMOVE_STEPSIZE / 2)};
		vec3_t	boxMax  = { 16,  16,  (BOTMOVE_STEPSIZE / 2)};
		vec3_t	box2Min = {-24, -24, -(BOTMOVE_STEPSIZE / 2)}; // close enough.
		vec3_t	box2Max = { 24,  24,  (BOTMOVE_STEPSIZE / 2)};
		vec3_t	dir;
		vec3_t	traceStart, traceForward, traceTop, targetFloor;
		trace_t tr;
		float		dist, travel = 0, stepSize = 1024, dist2;
		int			travelMode = VEHICLE_WALK;

		// Get source floor
		traceStart[0] = nodeSrc[0];
		traceStart[1] = nodeSrc[1];
		traceStart[2] = getFloor(nodeSrc, boxMin, boxMax, ent);

		// Quick test for slopes
		targetFloor[0] = nodeDst[0];
		targetFloor[1] = nodeDst[1];
		targetFloor[2] = getFloor(nodeDst, boxMin, boxMax, ent);
		if ((nodeDst[2] - targetFloor[2]) < BOTMOVE_STEPSIZE)
			{
			tr = gi.trace(traceStart, boxMin, boxMax, targetFloor, ent, CONTENTS_SOLID | CONTENTS_WINDOW | CONTENTS_PLAYERCLIP);
			if (tr.fraction == 1.00)
				{
				if (trace_FloorGap(ent, traceStart, targetFloor) == VEHICLE_JUMP)
					travelMode = VEHICLE_JUMP;
				if (display)
					showTrace(nodeSrc, nodeDst);	// DEBUG //
				return travelMode;
				}
			}

		// Get direction
		dir[0] = (nodeDst[0] - traceStart[0]);
		dir[1] = (nodeDst[1] - traceStart[1]);
		dist	 = sqrt(dir[0] * dir[0] + dir[1] * dir[1]);
		dir[0] = dir[0] / dist;
		dir[1] = dir[1] / dist;
		dir[2] = 0;

		// Climb stairs, crates and whatnots
		while (travel < dist)
			{
			// Go forward till you hit a step
			if ((travel + stepSize) > dist)
				stepSize = dist - travel;
			traceForward[0] = traceStart[0] + (dir[0] * (stepSize + 1));
			traceForward[1] = traceStart[1] + (dir[1] * (stepSize + 1));
			traceForward[2] = traceStart[2];

			//VectorAdd(traceStart, dir, traceForward);
			tr = gi.trace(traceStart, boxMin, boxMax, traceForward, ent, CONTENTS_SOLID | CONTENTS_WINDOW | CONTENTS_PLAYERCLIP);
			VectorCopy(tr.endpos, traceForward);
			if (display)
				showTrace(traceStart, traceForward);	// DEBUG //
			dist2 = VectorDistance(traceStart, traceForward);
			travel += dist2;

			// The step is long enough to contain a gap
			if (dist2 >= 32)
				{
				if (trace_FloorGap(ent, traceStart, traceForward) == VEHICLE_JUMP)
					travelMode = VEHICLE_JUMP;
				}

			// We got a step, how high is it?
			if (tr.fraction != 1.00)
				{
				traceTop[0] = traceForward[0];
				traceTop[1] = traceForward[1];
				traceTop[2] = traceForward[2] + BOTMOVE_JUMPSIZE / 2;
				tr = gi.trace(traceTop, box2Min, box2Max, traceForward, ent, CONTENTS_SOLID | CONTENTS_WINDOW | CONTENTS_PLAYERCLIP); // sneaky player clip
				if (tr.startsolid)
					{
					traceTop[2] = traceForward[2] + BOTMOVE_JUMPSIZE;
					tr = gi.trace(traceTop, box2Min, box2Max, traceForward, ent, CONTENTS_SOLID | CONTENTS_WINDOW | CONTENTS_PLAYERCLIP); // sneaky player clip
					if (tr.startsolid)
						return VEHICLE_NONE;
					}
				if (VectorDistance(traceForward, tr.endpos) > BOTMOVE_STEPSIZE) // Need to jump
					travelMode = VEHICLE_JUMP;
				VectorCopy(tr.endpos, traceStart);
				if (display)
					showTrace(traceForward, tr.endpos);
				}
			// Are we there yet?
			else
				VectorCopy(traceForward, traceStart);
			}


		if (abs(traceStart[2] - nodeDst[2]) > BOTMOVE_JUMPSIZE)
			{
			if (traceStart[2] < nodeDst[2])
				travelMode = VEHICLE_NONE;
			}

		return travelMode;
		}



	/*******************************************************

		Trace between nodes within water (BOTH nodeSrc and
		nodeDst ARE ASSUMED to be underwater). If, somewhere
		along the way, there's no water, the two nodes cannot
		be connected.

	*******************************************************/
	int trace_Swim(edict_t *ent, vec3_t nodeSrc, vec3_t nodeDst, qboolean display)
		{
		vec3_t	boxMin  = {-16, -16, -(BOTMOVE_STEPSIZE / 2)};
		vec3_t	boxMax  = { 16,  16,  (BOTMOVE_STEPSIZE / 2)};
		vec3_t	v, vN;
		int			i, steps;
		float		dist, stepLength = 0;

		VectorSet(v, nodeDst[0] - nodeSrc[0], nodeDst[1] - nodeSrc[1], nodeDst[2] - nodeSrc[2]);
		dist = VectorLength(v);
		vN[0] = v[0] / dist;
		vN[1] = v[1] / dist;
		vN[2] = v[2] / dist;
		steps = (int)(dist / (float)32) + 1;
	
		for (i = 0; i < steps; i ++)
			{
			v[0] = nodeSrc[0] + (vN[0] * stepLength);
			v[1] = nodeSrc[1] + (vN[1] * stepLength);
			v[2] = nodeSrc[2] + (vN[2] * stepLength);
			stepLength += 32;
			if (gi.pointcontents(v) & CONTENTS_WATER)
				continue;
			return VEHICLE_NONE;
			}
		return VEHICLE_WALK;
		}


		
	/*******************************************************

		Linking teleporters

	*******************************************************/
	int trace_Teleporter(edict_t *ent, botnode_t *nodeSrc, botnode_t *nodeDst)
		{
		int			j;
		edict_t	*spot1 = NULL;
		edict_t	*spot2 = NULL;

		// Get entity
		for (j = 0; j < jb_NumItems; j++)
			{
			if (&jb_Node[jb_ItemTable[j].node] == nodeSrc)
				spot1 = jb_ItemTable[j].ent;
			else if (&jb_Node[jb_ItemTable[j].node] == nodeDst)
				spot2 = jb_ItemTable[j].ent;
			}
		if ((!spot1) || (!spot2))
			return VEHICLE_NONE;

		// See if target matches
		if (!Q_stricmp(spot1->classname, "misc_teleporter"))
			if (!Q_stricmp(spot2->classname, "misc_teleporter_dest"))
				if (spot1->target)
					if (spot2->targetname)
						if (!Q_stricmp(spot1->target, spot2->targetname))
							return VEHICLE_WALK;

		return VEHICLE_NONE;
		}



	/*******************************************************

		Link elevator to something else.

	*******************************************************/
	int trace_Elevator(edict_t *ent, botnode_t *nodeSrc, botnode_t *nodeDst)
		{
		int				j;
		short int node1 = BOTNODE_INVALID, node2 = BOTNODE_INVALID;
		edict_t		*link1 = NULL, *link2 = NULL;
		vec3_t		min = { -8, -8, -8 };
		vec3_t		max = {  8,  8,  8 };
		trace_t		tr;

		// ELEVATOR TO ELEVATOR //
		if ((nodeSrc->type->value == nElevator) && (nodeDst->type->value == nElevator))
			{
			// Get entity
			for (j = 0; j < jb_NumItems; j++)
				{
				if (&jb_Node[jb_ItemTable[j].node] == nodeSrc)
					{
					link1 = jb_ItemTable[j].ent;
					node1 = jb_ItemTable[j].node;
					}
				else if (&jb_Node[jb_ItemTable[j].node] == nodeDst)
					{
					link2 = jb_ItemTable[j].ent;
					node2 = jb_ItemTable[j].node;
					}
				}
			// Both nodes are linking to the same entity and are aligned, link regardless of the distance
			if ((link1 != NULL) && (link1 == link2) && (jb_Node[node1].origin[0] == jb_Node[node2].origin[0]) && (jb_Node[node1].origin[1] == jb_Node[node2].origin[1]))
				return VEHICLE_WALK;
			return VEHICLE_NONE;
			}

		// ONE OF THE TWO NODES IS NOT AN ELEVATOR //
		if ((VectorDistance(nodeSrc->origin, nodeDst->origin) > BOTNODE_DIST_EXT) || (abs(nodeSrc->origin[2] - nodeDst->origin[2]) > BOTMOVE_STEPSIZE))
			return VEHICLE_NONE;

		// ROUGH TRACE //
		tr = gi.trace(nodeSrc->origin, min, max, nodeDst->origin, ent, MASK_BOTBLOCK);
		if (!(tr.fraction == 1.00 || tr.allsolid))
			return VEHICLE_NONE;

		return VEHICLE_WALK;
		}



	/*******************************************************

		Trace between a ladder node and another type of node.
		Only points located directly in the alignment of the
		ladder can be linked (no side jump).

	*******************************************************/
	int trace_Ladder(edict_t *ent, botnode_t *nodeSrc, botnode_t *nodeDst, qboolean display)
		{
		const int	ladderSide  = 16;
		const int	ladderRange = 48;
		float			x, y;
		vec3_t		normal, v1, v2;
		vec3_t		right, forward, angles;
		
		// Ladder to ladder always passes
		if (nodeDst->type->value == nLadder)
			return VEHICLE_WALK;

		// Only accept nodes within this range
		x = nodeDst->origin[0] - nodeSrc->origin[0];
		y = nodeDst->origin[1] - nodeSrc->origin[1];
		if (sqrt((x * x) + (y * y)) > ladderRange)
			return VEHICLE_NONE;

		// Get ladder angle
		if (!getLadderNormal(ent, nodeSrc->origin, normal))
			{
			gi.dprintf("Ladder node at %s not in front of ladder.\n", vtos(nodeSrc->origin));
			return VEHICLE_WALK;
			}
		vectoangles(normal, angles);
		AngleVectors(angles, forward, right, NULL);
		
		// LEFT SIDE //
		VectorCopy(nodeSrc->origin, v1);
		VectorMA(nodeSrc->origin, ladderRange, forward, v2);
		VectorMA(v2, ladderSide, right, v2);
		VectorMA(v1, ladderSide, right, v1);
		showTrace(v1, v2);
		if (VectorSide2D(nodeDst->origin, v1, v2) >= 0) // Right side or aligned
			{
			// RIGHT SIDE //
			VectorCopy(nodeSrc->origin, v1);
			VectorMA(nodeSrc->origin, ladderRange, forward, v2);
			VectorMA(v2, -ladderSide, right, v2);
			VectorMA(v1, -ladderSide, right, v1);
			showTrace(v1, v2);
			if (VectorSide2D(nodeDst->origin, v1, v2) <= 0) // Left side or aligned
				return VEHICLE_WALK;
			}

		return VEHICLE_NONE;
		}



	/*******************************************************

		Get max headroom between two nodes

	*******************************************************/
	float trace_MaxHeadroom(edict_t *ent, vec3_t nodeSrc, vec3_t nodeDst, qboolean display)
		{
		int			i, steps;
		float		MattFrewer = BOTNODE_DIST_MAX, dist, stepLength = 0;
		vec3_t	boxMin = {-16, -16, 0};
		vec3_t	boxMax = { 16,  16, 0};
		vec3_t	v, vN;
	
		VectorSet(v, nodeDst[0] - nodeSrc[0], nodeDst[1] - nodeSrc[1], nodeDst[2] - nodeSrc[2]);
		dist = VectorLength(v);
		vN[0] = v[0] / dist;
		vN[1] = v[1] / dist;
		vN[2] = v[2] / dist;
		steps = (int)(dist / (float)32) + 1;

		for (i = 0; i < steps; i ++)
			{
			v[0] = nodeSrc[0] + (vN[0] * stepLength);
			v[1] = nodeSrc[1] + (vN[1] * stepLength);
			v[2] = nodeSrc[2] + (vN[2] * stepLength);
			stepLength += 32;
			dist = (getCeiling(v, boxMin, boxMax, jb_Player[0]) - getFloor(v, boxMin, boxMax, jb_Player[0]));
			if (MattFrewer < dist)
				continue;
			MattFrewer = dist;
			if (MattFrewer < BOTMOVE_CROUCHSIZE)
				break;
			}

		return MattFrewer;
		}



	/*******************************************************

		There's only one major rule for airbound nodes: they
		always must be located below the previous node, and
		above the next node (it means that the source MUST
		always be HIGHER than the target).

	*******************************************************/
	int trace_Airbound(edict_t *ent, botnode_t *nodeSrc, botnode_t *nodeDst)
		{
		// The next node is higher or ladder
		if ((nodeDst->origin[2] > nodeSrc->origin[2]) || (nodeDst->type->value == nLadder))
			return VEHICLE_NONE;

		// Connecting airjump/airfall to something else
		if ((nodeSrc->type->value == nAirJump) || (nodeSrc->type->value == nAirFall))
			return VEHICLE_WALK;

		// Connecting something to airjump/airfall
		if (nodeDst->type->value == nAirJump)
			return VEHICLE_JUMP;
		return VEHICLE_WALK;
		}



	/*******************************************************

		Can we connect to that node?

	*******************************************************/
	qboolean trace_EntityToNode(edict_t *ent, botnode_t *nodeDst)
		{
		int				travel;
		botnode_t	dummy;
	
		// Create a dummy node representing the entity
		VectorCopy(ent->s.origin, dummy.origin);
		dummy.type = edit_BestNodeType(ent, true);
		if (!dummy.type)
			dummy.type = getNodeTypeByValue(nMove);
		travel = trace_HowToReach(ent, &dummy, nodeDst);

		if ((travel == VEHICLE_NONE) || (travel == VEHICLE_CROUCH) || (travel == VEHICLE_JUMP))
			return false;
		return true;
		}



	/*******************************************************

		How to connect two nodes

	*******************************************************/
	int trace_HowToReach(edict_t *ent, botnode_t *nodeSrc, botnode_t *nodeDst)
		{
		int				i;
		float			headroom;
		trace_t		tr;

		// TELEPORTERS //
		if (nodeSrc->type->value == nTeleporter)
			{
			if (nodeDst->type->value == nTeleporter)
				return trace_Teleporter(ent, nodeSrc, nodeDst);
			else
				{
				int j;
				edict_t *spot;
				for (j = 0; j < jb_NumItems; j++)
					{
					if (&jb_Node[jb_ItemTable[j].node] == nodeSrc)
						spot = jb_ItemTable[j].ent;
					}
				if (spot)
					if (!Q_stricmp(spot->classname, "misc_teleporter")) // teleport pads can only be linked to teleport targets
						return VEHICLE_NONE;
				}
			}

		/**** Everything else MUST be visible ****/
		tr = gi.trace(nodeSrc->origin, NULL, NULL, nodeDst->origin, jb_Player[0], MASK_BOTBLOCK);
		if (!(tr.fraction == 1.00 || tr.allsolid))
			return VEHICLE_NONE;

		// ELEVATORS //
		if ((nodeSrc->type->value == nElevator) || (nodeDst->type->value == nElevator))
			return trace_Elevator(ent, nodeSrc, nodeDst);
		
		/**** Everything else MUST be in range ****/
		if (VectorDistance(nodeSrc->origin, nodeDst->origin) > BOTNODE_DIST_EXT)
			return VEHICLE_NONE;

		// AIRBOUND //
		if ((nodeDst->type->value == nAirFall) || (nodeDst->type->value == nAirJump) || (nodeSrc->type->value == nAirFall) || (nodeSrc->type->value == nAirJump))
			return trace_Airbound(ent, nodeSrc, nodeDst);

		// LADDERS //
		if (nodeSrc->type->value == nLadder)
			return trace_Ladder(ent, nodeSrc, nodeDst, false);

		// WATER //
		if (nodeSrc->type->value == nWater)
			{
			if (nodeDst->type->value != nWater)
				return VEHICLE_WALK;
			return trace_Swim(ent, nodeSrc->origin, nodeDst->origin, false);
			}

		/**** Any other type of connection ****/
		headroom = trace_MaxHeadroom(ent, nodeSrc->origin, nodeDst->origin, false);
		if (headroom < BOTMOVE_CROUCHSIZE)
			return VEHICLE_NONE;
		i = trace_Climb(ent, nodeSrc->origin, nodeDst->origin, false);
		if (i == VEHICLE_NONE)
			return VEHICLE_NONE;
		else if (i == VEHICLE_JUMP)	// Need to jump
			{
			if (headroom <= 74)				// - Not enough room, can't link
				return VEHICLE_NONE;
			return VEHICLE_JUMP;			// - Enough room, link
			}
		else												// No need to jump
			{
			if (headroom <= 74)				// - Crouch recommended, link
				return VEHICLE_CROUCH;
			return VEHICLE_WALK;			// - Can walk, link
			}
		}



	/*********************************************************************

		Link nodes together (only direct neighbors)

	*********************************************************************/
	void path_LinkDirect()
		{
		int			source;
		int			target;
		int			vehicle;
		int			link;
		int			userLink = 0;

		// Reset travel ways
		jb_NumJumps = 0;
		jb_NumDucks = 0;

		// No node to link
		if (!jb_NumNodes)
			return;

		// Reset all nodes
		for (source = 0; source < jb_NumNodes; source++)
			{
			for (target = 0; target < jb_NumNodes; target++)
				{
				// RESET LINK //
				jb_PathTable[source][target] = -1;

				// NOT LINKING TO SELF //
				if (target == source)
					continue;

				// CHECK IF USER WANTS THOSE NODES LINKED TOGETHER REGARDLESS //
				if (jb_NumLinks)
					{
					userLink = 0;
					for (link = 0; link < jb_NumLinks; link++)
						{
						if ((jb_LinkTable[link].from != source) || (jb_LinkTable[link].to != target))
							continue;
						vehicle = jb_LinkTable[link].vehicle;
						userLink = 1;
						gi.dprintf("Forced link between node[%i] and node[%i]\n", source, target);
						break;
						}
					}
				
				// AUTO LINKING //
				if (userLink == 0)
					{
					// IN SAME PVS //
					if (!gi.inPVS(jb_Node[source].origin, jb_Node[target].origin))
						continue;

					// REACHABLE? //
					vehicle = trace_HowToReach(jb_Player[0], &jb_Node[source], &jb_Node[target]);
					if (vehicle == VEHICLE_NONE)
						continue;
					}

				// FIRST DEGREE LINK (DIRECT NEIGHBOR)
				jb_PathTable[source][target] = target;
				if (vehicle == VEHICLE_JUMP)
					{
					jb_JumpTable[jb_NumJumps].from = source;
					jb_JumpTable[jb_NumJumps].to   = target;
					jb_NumJumps ++;
					}
				else if (vehicle == VEHICLE_CROUCH)
					{
					jb_DuckTable[jb_NumDucks].from = source;
					jb_DuckTable[jb_NumDucks].to   = target;
					jb_NumDucks ++;
					}
				}
			}
		}



	/*********************************************************************

		Remove connections if there's a node in-between that is aligned.

	*********************************************************************/
	typedef struct
		{
		short	int	yaw;
		short int	pitch;
		short int	node;
		float			dist;
		} breaker_t;

	void path_UnlinkBreaks()
		{
		const	int	maxCount = 16;
		const int maxDiff = 10;
		double		frac, inte;
		int				i, j, k, count, z, y;
		vec3_t		delta, angle;
		breaker_t	*breaker;
		
		breaker = (breaker_t *)malloc(maxCount * sizeof(breaker_t));
		if (!breaker)
			gi.dprintf("path_UnlinkBreaks: Failed to allocate %i bytes of memory!\n", (int)(maxCount * sizeof(breaker_t)));

		for (i = 0; i < jb_NumNodes; i ++)
			{
			count = 0;
			memset(breaker, 0x00, maxCount * sizeof(breaker_t));
			// Get all links
			for (j = 0; j < jb_NumNodes; j ++)
				{
				if (jb_PathTable[i][j] == BOTNODE_INVALID)
					continue;
				if (count == maxCount)
					{
					gi.dprintf("path_UnlinkBreaks: maximum number of neighbors reached for Node[%i]\n", j);
					break;
					}
				// Get NODE
				breaker[count].node = j;
				VectorSubtract(jb_Node[j].origin, jb_Node[i].origin, delta);
				// Get DISTANCE
				breaker[count].dist = VectorLength(delta);
				VectorNormalize(delta);
				vectoangles(delta, angle);
				// Get YAW
				frac = modf((double)angle[0], &inte);
				breaker[count].yaw = (int)anglemod((float)inte);
				if (abs(frac) >= 0.5)
					breaker[count].yaw += ((inte > 0) - (inte < 0));
				// Get PITCH
				frac = modf((double)angle[1], &inte);
				breaker[count].pitch = (int)anglemod((float)inte);
				if (abs(frac) >= 0.5)
					breaker[count].pitch += ((inte > 0) - (inte < 0));
				count ++;
				}

			// Find and use link breakers
			if (count)
				{
				for (j = 0; j < count; j++)
					{
					if (breaker[j].node == BOTNODE_INVALID)
						continue;
					for (k = 0; k < count; k++)
						{
						if ((j == k) || (breaker[k].node == BOTNODE_INVALID))
							continue;
						if (breaker[k].dist > breaker[j].dist)
							{
							y = abs(breaker[k].pitch - breaker[j].pitch);
							z = abs(breaker[k].yaw - breaker[j].yaw);
							if (((y <= maxDiff) || (y >= (360 - maxDiff))) && ((z <= maxDiff) || (z >= (360 - maxDiff))))
								{
								jb_PathTable[i][breaker[k].node] = BOTNODE_INVALID;
								breaker[k].node = BOTNODE_INVALID;
								}
							}
						}
					}
				}
			}
		free(breaker);
		}

		

	/*********************************************************************

		Link nodes together. This routine tries to establish links using
		proxy nodes (that is, if A can go to B, then A can use B to reach
		all neighbors of B), it needs the first degree neighbors to be
		initialized -- that's what botNode_linkDirect() does.

	*********************************************************************/
	void path_LinkProxy()
		{
		qboolean			doAgain = true;
		int						i, j, k, cnt = 0;
		short int			proxy;
		short int			*tmpPath = 0;
		unsigned int	tmpOfs;

		// No node to link
		if (!jb_NumNodes)
			return;

		// Initiliaze temporary path table
		tmpPath = (short int *)malloc(sizeof(short int) * jb_NumNodes * jb_NumNodes);
		if (!tmpPath)
			gi.dprintf("path_LinkProxy: Failed to allocate %i bytes of memory!\n", (int)(sizeof(INT16) * jb_NumNodes * jb_NumNodes));

		for (i = 0; i < (jb_NumNodes * jb_NumNodes); i++)
			tmpPath[i] = BOTNODE_INVALID;

		// Get proxy links
		do
			{
			doAgain = false;

			// Get X degree link
			for (i = 0; i < jb_NumNodes; i++)
				{
				tmpOfs = i * jb_NumNodes;
				for (j = 0; j < jb_NumNodes; j++)
					{
					if ((jb_PathTable[i][j] == BOTNODE_INVALID) || (i == j))
						continue;
					proxy = jb_PathTable[i][j];

					for (k = 0; k < jb_NumNodes; k++)
						{
						if (jb_PathTable[j][k] == BOTNODE_INVALID)
							continue;
						tmpPath[tmpOfs + k] = proxy;
						}
					}
				}

			// Merge temporary path table with actual paths
			for (i = 0; i < jb_NumNodes; i++)
				{
				tmpOfs = i * jb_NumNodes;
				for (j = 0; j < jb_NumNodes; j++)
					{
					if ((jb_PathTable[i][j] == BOTNODE_INVALID) && (tmpPath[tmpOfs + j] != BOTNODE_INVALID) && (j != i))
						{
						jb_PathTable[i][j] = tmpPath[tmpOfs + j];
						doAgain = true;
						}
					tmpPath[tmpOfs + j] = BOTNODE_INVALID;
					}
				}
			} while (doAgain);

		// Free memory
		free(tmpPath);

		// Quick summary
		path_LinkCheckUp();
		}




#endif