/*
    msPlugInImpl.cpp - Microsoft DirectX export plug-in for MilkShape.

    Copyright (c) John Thompson, 2001-2004.  All rights reserved.
*/

#include "stdafx.h"
#include "msPlugInImpl.h"
#include "msLib.h"

    // Options file name.
#define kOptionsFileName "msXExporterOptions.opt"

// MFC application class implementation.

    // Constructor.
CMSPlugInApp::CMSPlugInApp()
{
}

	// Initialize instance.
BOOL CMSPlugInApp::InitInstance()
{
    BOOL blRet = CWinApp::InitInstance();

    return(blRet);
}

    // Message map.
BEGIN_MESSAGE_MAP(CMSPlugInApp, CWinApp)
END_MESSAGE_MAP()

    // DLL app.
CMSPlugInApp oPlugInDLLApp;

// MilkShape interface functions.

    // Required by MilkShape to return the plug-in object.
cMsPlugIn *CreatePlugIn()
{
    return new cPlugIn();
}


// Plug-in class implementation.

    // Constructor.
cPlugIn::cPlugIn()
{
    strcpy(szTitle, "DirectX (JT) ...");
    m_caFileName[0] = '\0';
    m_caBaseName[0] = '\0';
    m_caRootFrameName[0] = '\0';
    m_caAnimationSetName[0] = '\0';
    m_pFilePointer = NULL;
    m_opRootFrame = NULL;
    m_opBodyFrame = NULL;
    m_opBodyMesh = NULL;
    m_oEulerFactor.x = 1.0f;
    m_oEulerFactor.y = 1.0f;
    m_oEulerFactor.z = 1.0f;
    m_fPositionFactor = 0.10f;
    m_opAnimationSet = NULL;
    m_eFormat = kFormatDirectXRetainedMode;
    m_blExcludeGeometry = false;
    m_blAddRootFrame = false;
    m_blMeshOnly = false;
    m_blRightHandedCoordinateSystem = false;
	m_blBinary = false;
    m_fAnimationTimeFactor = 210.0f;
    m_oPositionOffset.x = 0.0f;
    m_oPositionOffset.y = 0.0f;
    m_oPositionOffset.z = 0.0f;
    m_oRotation.x = 0.0f;
    m_oRotation.y = 0.0f;
    m_oRotation.z = 0.0f;
    m_blFlipTextureVertical = false;
	m_blFlipTextureHorizontal = false;
    m_blExcludeAnimation = false;
    m_blExcludeAnimationOptions = false;
    m_blLoopAnimations = true;
    m_blSplinePositions = true;
    m_blSplineRotations = true;
    m_blMatrixKeys = false;
    m_blCenterAtOffset = false;
    m_blStandAtOffset = false;
    m_blSortOriginalVertices = (m_eFormat == kFormatDirectXSkinAndBones);
	m_blAddInterpolations = false;
	m_iInterpolationCount = 10;
    m_fAnimationRangeLow = 1.0f;
    m_fAnimationRangeHigh = 1.0f;
    m_blAddTemplates = false;
    m_blAddSkinAndBonesTemplates = false;
    m_blCloseHoles = false;
    m_blExcludeNormals = false;
}

    // Destructor.
cPlugIn::~cPlugIn()
{
}

    // Return plug-in type.
int cPlugIn::GetType()
{
    return(cMsPlugIn::eTypeExport);
}

    // Return plug-in title.
const char *cPlugIn::GetTitle()
{
    return(szTitle);
}

    // Do main plug-in action.
int cPlugIn::Execute(msModel *pMSModel)
{
        // Load options.
    LoadOptions();

        // Prompt for and open the file.
    if (PromptFile() && OpenFile())
    {
            // Check for DirectX format.
        CheckForDirectXFormat(pMSModel);

            // Prompt for options.
        if (PromptOptions())
        {
                // Save options.
            SaveOptions();

                // Set static node format member.
            Node::m_eFormat = m_eFormat;

                // Process the meshes based on format.
            switch (m_eFormat)
            {
                case kFormatDirectXRetainedMode:
                        // Do DirectX retained mode format.
                    DoRetainedMode(pMSModel);
                    break;
                        // Do DirectX 8 skin and bones format.
                case kFormatDirectXSkinAndBones:
                    DoSkinAndBones(pMSModel);
                    break;
                case kFormatJTGameSkinAndBones:
                        // Do JTGame skin and bones format.
                    DoJTGameSkinAndBones(pMSModel);
                    break;
                case kFormatJTGameRenderGroups:
                        // Do JTGame render groups format.
                    DoJTGameRenderGroups(pMSModel);
                    break;
                default:
                    break;
            }
        }

            // Clean up.
        CleanUp();

            // Close the file.
        CloseFile();
    }

    return(0);
}

static char caRegistryBuffer[256];

    // Load exporter options.
bool cPlugIn::LoadOptions()
{
#if 1
    HKEY hKey = NULL;
    int iLength = &m_cEndSave - &m_cStartSave;
    LONG lResult;
    bool blRet = false;

    lResult = RegOpenKeyEx(
            HKEY_CURRENT_USER,
            kRegistrySubKeyString,
            0,
            KEY_READ,
            (HKEY *)&hKey);

    if (lResult == ERROR_SUCCESS)
    {
        DWORD dwType = 0;
        DWORD dwCount = (DWORD)iLength;

	    lResult = RegQueryValueEx(
            hKey,
            kRegistryValueNameString,
            NULL,
            &dwType,
            (LPBYTE)caRegistryBuffer,
            &dwCount);

        if ((lResult == ERROR_SUCCESS) && (dwType == REG_BINARY))
        {
            memcpy(&m_cStartSave, caRegistryBuffer, iLength);
	        blRet = true;
        }

        RegCloseKey(hKey);

        return(blRet);
    }
#else
    FILE *fp = fopen(kOptionsFileName, "rb");

    if (fp)
    {
        fread(&m_cStartSave, 1, &m_cEndSave - &m_cStartSave, fp);
        fclose(fp);
        return(true);
    }
#endif

    return(false);
}

    // Save exporter options.
bool cPlugIn::SaveOptions()
{
#if 1
    int iLength = &m_cEndSave - &m_cStartSave;
    DWORD dwDisposition;
    HKEY hKey;
    LONG lResult;
    bool blRet = false;

    if (memcmp(&m_cStartSave, caRegistryBuffer, iLength) == 0)
        return(true);

    lResult = RegCreateKeyEx(
            HKEY_CURRENT_USER,
            kRegistrySubKeyString,
            0,
            REG_NONE,
			REG_OPTION_NON_VOLATILE, 
            KEY_WRITE,
            NULL,
            &hKey,
            &dwDisposition);

    if (lResult == ERROR_SUCCESS)
    {
        lResult = RegSetValueEx(
            hKey,
            kRegistryValueNameString,
            NULL,
            REG_BINARY,
            (LPBYTE)&m_cStartSave,
            (DWORD)iLength);

        if (lResult == ERROR_SUCCESS)
            blRet = true;

        RegCloseKey(hKey);

        return(blRet);
    }
#else
    FILE *fp = fopen(kOptionsFileName, "wb");

    if (fp)
    {
        fwrite(&m_cStartSave, 1, &m_cEndSave - &m_cStartSave, fp);
        fclose(fp);
        return(true);
    }
#endif

    return(false);
}

    // Prompt for the file.
bool cPlugIn::PromptFile()
{
    OPENFILENAME ofn;
    char szFileTitle[MS_MAX_PATH];
    char *szDefExt;
    char *szFilter;

    if ((m_eFormat == kFormatJTGameSkinAndBones) || (m_eFormat == kFormatJTGameRenderGroups))
    {
        szDefExt = "jtm";
        szFilter = 
            "JTGame Model Files (*.jtm)\0*.jtm\0"
            "DirectX Files (*.x)\0*.x\0"
            "All Files (*.*)\0*.*\0\0" ;
    }
    else
    {
        szDefExt = "x";
        szFilter =
            "DirectX Files (*.x)\0*.x\0"
            "JTGame Model Files (*.jtm)\0*.jtm\0"
            "All Files (*.*)\0*.*\0\0";
    }

    memset(&ofn, 0, sizeof(OPENFILENAME));
    
    m_caFileName[0] = '\0';
    szFileTitle[0] = '\0';

    ofn.lStructSize = sizeof(OPENFILENAME);
    ofn.lpstrDefExt = szDefExt;
    ofn.lpstrFilter = szFilter;
    ofn.lpstrFile = m_caFileName;
    ofn.nMaxFile = sizeof(m_caFileName) - 1;
    ofn.lpstrFileTitle = szFileTitle;
    ofn.nMaxFileTitle = MS_MAX_PATH;
    ofn.Flags = OFN_HIDEREADONLY | OFN_OVERWRITEPROMPT | OFN_PATHMUSTEXIST;
    ofn.lpstrTitle = "Export DirectX (JT)";

    if (!::GetSaveFileName (&ofn))
        return(false);

    m_caBaseName[0] = '\0';

    char *cpBase = strrchr(m_caFileName, '\\');

    if (!cpBase)
        cpBase = m_caFileName;
    else
        cpBase++;

    strncpy(m_caBaseName, cpBase, sizeof(m_caBaseName));
    char *cpExt = strchr(m_caBaseName, '.');

    if (cpExt)
    {
        if (_stricmp(cpExt, ".jtm") == 0)
            m_eFormat = kFormatJTGameSkinAndBones;
        else if (_stricmp(cpExt, ".x") == 0)
        {
            if (m_eFormat == kFormatJTGameSkinAndBones)
                m_eFormat = kFormatDirectXSkinAndBones;
        }

        *cpExt = '\0';
    }

        // Format top level frame name.
    sprintf(m_caRootFrameName, "%sFrame", m_caBaseName);

        // Format animation set name.
    sprintf(m_caAnimationSetName, "AnimationSet0");

    return(true);
}

    // Open the file.
bool cPlugIn::OpenFile()
{
    m_pFilePointer = fopen(m_caFileName, "wb");

    if (!m_pFilePointer)
        return(false);

    Node::m_pFilePointer = m_pFilePointer;

    return(true);
}

    // Close the file.
bool cPlugIn::CloseFile()
{
    if (m_pFilePointer)
        fclose(m_pFilePointer);

    m_pFilePointer = NULL;
    Node::m_pFilePointer = m_pFilePointer;

    return(true);
}

    // Check for DirectX format, setting a flag.
void cPlugIn::CheckForDirectXFormat(msModel *pMSModel)
{
    int iModelCount = msModel_GetMeshCount(pMSModel);
    int iModelIndex;
    int iVertexCount;
    int iVertexIndex;
    int iBoneIndex;
    int iFirstBoneIndex;
    msMesh *pMSMesh;
    msVertex *pMSVertex;

    m_fAnimationRangeHigh = (float)pMSModel->nTotalFrames;

    for (iModelIndex = 0; iModelIndex < iModelCount; iModelIndex++)
    {
            // Get MilkShape mesh.
        pMSMesh = msModel_GetMeshAt(pMSModel, iModelIndex);

            // Get vertex count.
        iVertexCount = msMesh_GetVertexCount(pMSMesh);

            // Initialize first bone index.
        iFirstBoneIndex = -1;

            // Collect vertex positions.
        for (iVertexIndex = 0; iVertexIndex < iVertexCount; iVertexIndex++)
        {
                // Get MilkShape vertex.
            pMSVertex = msMesh_GetVertexAt(pMSMesh, iVertexIndex);

                // Get the vertex bone index.
            iBoneIndex = msVertex_GetBoneIndex(pMSVertex);

                // Update first bone index.
            if (iFirstBoneIndex == -1)
                iFirstBoneIndex = iBoneIndex;

                // -1 means no bone associated, so reference the default bone.
            if (iBoneIndex != -1)
            {
                if (iFirstBoneIndex != iBoneIndex)
                {
                        // We should be using the skn and bones formats.
                    m_blSkinAndBones = true;
                    return;
                }
            }
        }
    }

        // Okay to use retained mode format.
    m_blSkinAndBones = false;
}

    // Prompt for the options.
bool cPlugIn::PromptOptions()
{
    bool blRet = false;

    AFX_MANAGE_STATE(AfxGetStaticModuleState());

    CPlugInOptions oOptions;

        // Initialize default options.
    oOptions.m_iFormat = m_eFormat;
    oOptions.m_blExcludeGeometry = m_blExcludeGeometry;
    oOptions.m_blAddRootFrame = m_blAddRootFrame;
    oOptions.m_blMeshOnly = m_blMeshOnly;
    oOptions.m_sRootFrameName = m_caRootFrameName;
    oOptions.m_fAnimationRate = m_fAnimationTimeFactor;
    oOptions.m_fPositionScale = m_fPositionFactor;
    oOptions.m_fPositionOffsetX = m_oPositionOffset.x;
    oOptions.m_fPositionOffsetY = m_oPositionOffset.y;
    oOptions.m_fPositionOffsetZ = m_oPositionOffset.z;
    oOptions.m_fRotationX = m_oRotation.x;
    oOptions.m_fRotationY = m_oRotation.y;
    oOptions.m_fRotationZ = m_oRotation.z;
    oOptions.m_blExcludeAnimation = m_blExcludeAnimation;
    oOptions.m_blExcludeAnimationOptions = m_blExcludeAnimationOptions;
    oOptions.m_blLoopAnimations = m_blLoopAnimations;
    oOptions.m_blSplinePositions = m_blSplinePositions;
    oOptions.m_blSplineRotations = m_blSplineRotations;
    oOptions.m_sAnimationSetName = m_caAnimationSetName;
    oOptions.m_blFlipTextureVertical = m_blFlipTextureVertical;
    oOptions.m_blFlipTextureHorizontal = m_blFlipTextureHorizontal;
    oOptions.m_blRightHanded = m_blRightHandedCoordinateSystem;
    oOptions.m_blCloseHoles = m_blCloseHoles;
    oOptions.m_blExcludeNormals = m_blExcludeNormals;
    oOptions.m_blBinary = m_blBinary;
    oOptions.m_blMatrixKeys = m_blMatrixKeys;
    oOptions.m_blCenterAtOffset = m_blCenterAtOffset;
    oOptions.m_blStandAtOffset = m_blStandAtOffset;
    oOptions.m_blAddInterpolations = m_blAddInterpolations;
    oOptions.m_iInterpolationCount = m_iInterpolationCount;
    oOptions.m_fAnimationRangeLow = 1.0f;
    oOptions.m_fAnimationRangeHigh = m_fAnimationRangeHigh;
    oOptions.m_blAddTemplates = m_blAddTemplates;
    oOptions.m_blAddSkinAndBonesTemplates = m_blAddSkinAndBonesTemplates;

    if (oOptions.DoModal())
    {
            // Retrieve options.
        m_eFormat = (EFormat)oOptions.m_iFormat;
        m_blExcludeGeometry = (oOptions.m_blExcludeGeometry ? true : false);
        m_blAddRootFrame = (oOptions.m_blAddRootFrame ? true : false);
        m_blMeshOnly = (oOptions.m_blMeshOnly ? true : false);
        strncpy(m_caRootFrameName, oOptions.m_sRootFrameName, sizeof(m_caRootFrameName));
        m_fAnimationTimeFactor = oOptions.m_fAnimationRate;
        m_fPositionFactor = oOptions.m_fPositionScale;
        m_oPositionOffset.x = oOptions.m_fPositionOffsetX;
        m_oPositionOffset.y = oOptions.m_fPositionOffsetY;
        m_oPositionOffset.z = oOptions.m_fPositionOffsetZ;
        m_oRotation.x = oOptions.m_fRotationX;
        m_oRotation.y = oOptions.m_fRotationY;
        m_oRotation.z = oOptions.m_fRotationZ;
        m_blExcludeAnimation = (oOptions.m_blExcludeAnimation ? true : false);
        m_blExcludeAnimationOptions = (oOptions.m_blExcludeAnimationOptions ? true : false);
        m_blLoopAnimations = (oOptions.m_blLoopAnimations ? true : false);
        m_blSplinePositions = (oOptions.m_blSplinePositions ? true : false);
        m_blSplineRotations = (oOptions.m_blSplineRotations ? true : false);
        strncpy(m_caAnimationSetName, oOptions.m_sAnimationSetName, sizeof(m_caAnimationSetName));
        m_blFlipTextureVertical = (oOptions.m_blFlipTextureVertical ? true : false);
        m_blFlipTextureHorizontal = (oOptions.m_blFlipTextureHorizontal ? true : false);
        m_blRightHandedCoordinateSystem = (oOptions.m_blRightHanded ? true : false);
        m_blCloseHoles = (oOptions.m_blCloseHoles ? true : false);
        m_blExcludeNormals = (oOptions.m_blExcludeNormals ? true : false);
        m_blBinary = (oOptions.m_blBinary ? true : false);
        m_blMatrixKeys = (oOptions.m_blMatrixKeys ? true : false);
        m_blCenterAtOffset = (oOptions.m_blCenterAtOffset ? true : false);
        m_blStandAtOffset = (oOptions.m_blStandAtOffset ? true : false);
        m_blSortOriginalVertices = (m_eFormat == kFormatDirectXSkinAndBones);
        //m_blSortOriginalVertices = false;
        m_blAddInterpolations = (oOptions.m_blAddInterpolations ? true : false);
        m_iInterpolationCount = oOptions.m_iInterpolationCount;
        m_fAnimationRangeLow = oOptions.m_fAnimationRangeLow;
        m_fAnimationRangeHigh = oOptions.m_fAnimationRangeHigh;
        m_blAddTemplates = (oOptions.m_blAddTemplates ? true : false);
        m_blAddSkinAndBonesTemplates = (oOptions.m_blAddSkinAndBonesTemplates ? true : false);

        blRet = true;
    }

    Node::m_blBinary = m_blBinary;

    return(blRet);
}

    // Do DirectX retained mode format.
bool cPlugIn::DoRetainedMode(msModel *pMSModel)
{
    bool blRet = false;

        // Output the header.
    if (!OutputFileHeader())
        goto SkipError;

        // Output templates.
    if (!OutputTemplates())
        goto SkipError;

        // Collect all the geometry into one mesh.
    CreateBodyMesh(pMSModel);

        // Create the top-level frame object.
    CreateMainFrame();

        // Create the child frames from bones.
    if (!CreateFrames(pMSModel))
        goto SkipError;

        // Create frame meshes from body mesh.
    CreateRetainedModeMeshes();

        // If not excluding the animation...
    if (!m_blExcludeAnimation && !m_blMeshOnly)
    {
            // Create the animation set.
        if (!CreateAnimations(pMSModel))
            goto SkipError;
    }

        // Output the geometry.
    if (!m_blExcludeGeometry)
        OutputGeometry();

    if (!m_blExcludeAnimation)
    {
            // Output the animation set.
        if (m_opAnimationSet)
        {
            m_opAnimationSet->Output();

                // Blank line before animation set.
            m_opAnimationSet->OutputNewLine();
        }
    }

    blRet = true;

SkipError:
    return(blRet);
}

    // Do DirectX 8 skin and bones format.
bool cPlugIn::DoSkinAndBones(msModel *pMSModel)
{
    bool blRet = false;

        // Output the header.
    if (!OutputFileHeader())
        goto SkipError;

        // Output templates.
    if (!OutputTemplates())
        goto SkipError;

        // Collect all the geometry into one mesh.
    CreateBodyMesh(pMSModel);

        // Create the top-level frame object.
    CreateMainFrame();

        // Create the child frames from bones.
    if (!CreateFrames(pMSModel))
        goto SkipError;

        // Process body mesh for skin and bones format.
    CreateSkinAndBonesMesh();

        // If not excluding the animation...
    if (!m_blExcludeAnimation && !m_blMeshOnly)
    {
            // Create the animation set.
        if (!CreateAnimations(pMSModel))
            goto SkipError;
    }

        // Output the geometry.
    if (!m_blExcludeGeometry)
        OutputGeometry();

    if (!m_blExcludeAnimation && m_opAnimationSet)
    {
            // Blank line before animation set.
        m_opAnimationSet->OutputNewLine();

            // Output the animation set.
        if (m_opAnimationSet)
            m_opAnimationSet->Output();
    }

    blRet = true;

SkipError:
    return(blRet);
}

    // Do JTGame skin and bones format.
bool cPlugIn::DoJTGameSkinAndBones(msModel *pMSModel)
{
    bool blRet = false;

        // Output the header.
    if (!OutputFileHeader())
        goto SkipError;

        // Output templates.
    if (!OutputTemplates())
        goto SkipError;

        // Collect all the geometry into one mesh.
    CreateBodyMesh(pMSModel);

        // Create the top-level frame object.
    CreateMainFrame();

        // Create bone matrix transformations.
    if (!CreateTransformations(pMSModel))
        goto SkipError;

        // Link body mesh and parent frame.
    m_opRootFrame->AdoptMesh(m_opBodyMesh);

        // If not excluding the animation...
    if (!m_blExcludeAnimation && !m_blMeshOnly)
    {
            // Create the animation set.
        if (!CreateAnimations(pMSModel))
            goto SkipError;
    }

        // Output the geometry.
    if (!m_blExcludeGeometry)
        OutputGeometry();

    if (!m_blExcludeAnimation && m_opAnimationSet)
    {
            // Blank line before animation set.
        m_opAnimationSet->OutputNewLine();

            // Output the animation set.
        if (m_opAnimationSet)
            m_opAnimationSet->Output();
    }

    blRet = true;

SkipError:
    return(blRet);
}

    // Do JTGame render group format.
bool cPlugIn::DoJTGameRenderGroups(msModel *pMSModel)
{
    bool blRet = false;

        // Output the header.
    if (!OutputFileHeader())
        goto SkipError;

        // Output templates.
    if (!OutputTemplates())
        goto SkipError;

        // Collect all the geometry into one mesh, mainly just so we get the
        // offset matrices.
    CreateBodyMesh(pMSModel);

        // Create the top-level frame object.
    CreateMainFrame();

        // Create bone matrix transformations.
    if (!CreateTransformations(pMSModel))
        goto SkipError;

        // Create render groups from models.
    if (!CreateRenderGroups(pMSModel))
        goto SkipError;

        // If not excluding the animation...
    if (!m_blExcludeAnimation && !m_blMeshOnly)
    {
            // Create the animation set.
        if (!CreateAnimations(pMSModel))
            goto SkipError;
    }

        // Output the geometry.
    if (!m_blExcludeGeometry)
        OutputGeometry();

    if (!m_blExcludeAnimation && m_opAnimationSet)
    {
            // Blank line before animation set.
        m_opAnimationSet->OutputNewLine();

            // Output the animation set.
        if (m_opAnimationSet)
            m_opAnimationSet->Output();
    }

    blRet = true;

SkipError:
    return(blRet);
}

    // Create body mesh.
void cPlugIn::CreateBodyMesh(msModel *pMSModel)
{
    int iModelCount = msModel_GetMeshCount(pMSModel);
    int iModelIndex;
    int iTotalVertexCount = 0;;
    int iVertexCount = 0;;
    int iVertexIndex;
    int iVertexBase = 0;
    int iNormalBase = 0;
    int iNormalIndex;
    int iNormalCount;
    int iBoneIndex;
    int iTotalFaceCount = 0;
    int iFaceCount = 0;
    int iFaceIndex;
    int iMaterialCount = 0;
    int iMaterialIndex;
    char caName[kNameLen + 1];
    char caTextureFileName[kNameLen + 1];
    char *cpBase;
    msMesh *pMSMesh;
    msVertex *pMSVertex;
    msTriangle *pMSTriangle;
    msMaterial *pMSMaterial;
    msVec3 oMSPosition;
    msVec3 oMSNormal;
    word nMSIndices[3];
    word nMSNormalIndices[3];
    msVec2 oMSTextureVector;
    msVec4 oMSColor;
    Material *opMaterial;
    D3DXVECTOR3 oVector;
    D3DXVECTOR4 oVector4;
    D3DXVECTOR3 oNormal;
    D3DXVECTOR2 oTextureVector;
    float fZFactor = (m_blRightHandedCoordinateSystem ? 1.0f : -1.0f);
    bool blRet = true;

        // Format mesh name.
    sprintf(caName, "%sMesh", m_caBaseName);

        // Create body mesh.
    m_opBodyMesh = new Mesh(caName);

    if (m_blSkinAndBones)
        m_opBodyMesh->m_iFVFFlags = kFVFIndexedNoWeights;
    else
        m_opBodyMesh->m_iFVFFlags = kFVFNoWeights;

        // Get total vertex and face count.
    for (iModelIndex = 0; iModelIndex < iModelCount; iModelIndex++)
    {
            // Get MilkShape mesh.
        pMSMesh = msModel_GetMeshAt(pMSModel, iModelIndex);

            // Add face count;
        iTotalFaceCount += msMesh_GetTriangleCount(pMSMesh);

            // Add vertex count.
        iTotalVertexCount += msMesh_GetVertexCount(pMSMesh);
    }

        // Get material count.
    iMaterialCount = msModel_GetMaterialCount(pMSModel);

        // Pre-size arrays.
    m_opBodyMesh->m_oPositions.SetSize(iTotalVertexCount);
    m_opBodyMesh->m_oBoneIndices.SetSize(iTotalVertexCount);
    m_opBodyMesh->m_oFaceData.SetSize(iTotalFaceCount * 3);
    m_opBodyMesh->m_oNormals.SetSize(iTotalVertexCount);
    m_opBodyMesh->m_oNormalFaceData.SetSize(iTotalFaceCount * 3);
    m_opBodyMesh->m_oTextureCoordinates.SetSize(iTotalVertexCount);
    m_opBodyMesh->m_oMaterialFaceIndices.SetSize(iTotalFaceCount);
    m_opBodyMesh->m_oMaterials.SetSize(iMaterialCount);
    m_opBodyMesh->m_oMaterials.SetCount(iMaterialCount);

        // Collect materials.
    for (iMaterialIndex = 0; iMaterialIndex < iMaterialCount; iMaterialIndex++)
    {
            // Get MilkShape material.
        pMSMaterial = msModel_GetMaterialAt(pMSModel, iMaterialIndex);

        if (!pMSMaterial)
            continue;

            // Get material name.
        msMaterial_GetName(pMSMaterial, caName, MS_MAX_NAME);

            // Get texture name.
        msMaterial_GetDiffuseTexture(pMSMaterial, caTextureFileName, MS_MAX_PATH);

            // Get transparency.
        float fTransparency = msMaterial_GetTransparency(pMSMaterial);

            // Convert to valid .x names, if necessary.
		MakeNameLegal(caName, false);
		MakeNameLegal(caTextureFileName, true);

            // Remove leading path component from texture file name.
        if ((cpBase = strrchr(caTextureFileName, '\\')) != NULL)
            strcpy(caTextureFileName, cpBase + 1);

            // Create material object.
        opMaterial = new Material(caName);

            // Get diffuse color.
        msMaterial_GetDiffuse(pMSMaterial, oMSColor);
        opMaterial->m_oDiffuse.r = oMSColor[0];
        opMaterial->m_oDiffuse.g = oMSColor[1];
        opMaterial->m_oDiffuse.b = oMSColor[2];
        opMaterial->m_oDiffuse.a = fTransparency;

            // Get power.
        opMaterial->m_fPower = msMaterial_GetShininess(pMSMaterial);

            // Get specular color.
        msMaterial_GetSpecular(pMSMaterial, oMSColor);
        opMaterial->m_oSpecular.r = oMSColor[0];
        opMaterial->m_oSpecular.g = oMSColor[1];
        opMaterial->m_oSpecular.b = oMSColor[2];
        opMaterial->m_oSpecular.a = 1.0f;

            // Get specular color.
        msMaterial_GetEmissive(pMSMaterial, oMSColor);
        opMaterial->m_oEmissive.r = oMSColor[0];
        opMaterial->m_oEmissive.g = oMSColor[1];
        opMaterial->m_oEmissive.b = oMSColor[2];
        opMaterial->m_oEmissive.a = 1.0f;

            // Get texture file name.
        if (caTextureFileName[0])
            strncpy(opMaterial->m_oTextureFileName.m_caTextureFileName, caTextureFileName, kNameLen);

            // Save material.
        m_opBodyMesh->m_oMaterials.Set(iMaterialIndex, opMaterial, false);
    }

        // Collect the geometry data.
    for (iModelIndex = 0; iModelIndex < iModelCount; iModelIndex++)
    {
            // Get MilkShape mesh.
        pMSMesh = msModel_GetMeshAt(pMSModel, iModelIndex);

            // Get face count;
        iFaceCount = msMesh_GetTriangleCount(pMSMesh);

        // Extract vertices.

            // Get vertex count.
        iVertexCount = msMesh_GetVertexCount(pMSMesh);

            // Collect vertex positions.
        for (iVertexIndex = 0; iVertexIndex < iVertexCount; iVertexIndex++)
        {
                // Get MilkShape vertex.
            pMSVertex = msMesh_GetVertexAt(pMSMesh, iVertexIndex);

                // Get MilkShape vertex position.
            msVertex_GetVertex(pMSVertex, oMSPosition);

                // Get position.
            oVector.x = oMSPosition[0];
            oVector.y = oMSPosition[1];
            oVector.z = oMSPosition[2] * fZFactor;

                // Get the vertex bone index.
            iBoneIndex = msVertex_GetBoneIndex(pMSVertex);

                // -1 means no bone associated, so reference the default bone.
            if (iBoneIndex < 0)
                iBoneIndex = 0;

                // Append position.
            m_opBodyMesh->m_oPositions.Append(oVector);

                // Append bone index.
            m_opBodyMesh->m_oBoneIndices.Append(iBoneIndex);
        }

        // Extract mesh face data.

        for (iFaceIndex = 0; iFaceIndex < iFaceCount; iFaceIndex++)
        {
                // Get MilkShape triangle.
            msTriangle *pMSTriangle = msMesh_GetTriangleAt(pMSMesh, iFaceIndex);

                // Get face vertex indices.
            msTriangle_GetVertexIndices(pMSTriangle, nMSIndices);

                // Save vertex index 0.
            m_opBodyMesh->m_oFaceData.Append(nMSIndices[0] + iVertexBase);

                // Save vertex indices 1 and 2 in right or left order.
            if (m_blRightHandedCoordinateSystem)
            {
                m_opBodyMesh->m_oFaceData.Append(nMSIndices[1] + iVertexBase);
                m_opBodyMesh->m_oFaceData.Append(nMSIndices[2] + iVertexBase);
            }
            else
            {
                m_opBodyMesh->m_oFaceData.Append(nMSIndices[2] + iVertexBase);
                m_opBodyMesh->m_oFaceData.Append(nMSIndices[1] + iVertexBase);
            }
        }

        // Extract vertex normals.

            // Get normal count.
        iNormalCount = msMesh_GetVertexNormalCount(pMSMesh);

            // Collect normals.
        for (iNormalIndex = 0; iNormalIndex < iNormalCount; iNormalIndex++)
        {
                // Get normal.
            msMesh_GetVertexNormalAt(
                pMSMesh,
                iNormalIndex,
                oMSNormal);

                // Convert to left-handed.
            oVector.x = oMSNormal[0];
            oVector.y = oMSNormal[1];
            oVector.z = oMSNormal[2] * fZFactor;

                // Set normal in normal array at vertex position.
            m_opBodyMesh->m_oNormals.Append(oVector);
        }

        // Extract mesh normal face data.

            // Collect normal face data.
        for (iFaceIndex = 0; iFaceIndex < iFaceCount; iFaceIndex++)
        {
                // Get MilkShape triangle.
            pMSTriangle = msMesh_GetTriangleAt(pMSMesh, iFaceIndex);

                // Get normal indices.
            msTriangle_GetNormalIndices(pMSTriangle, nMSNormalIndices);

                // Save normal index 0.
            m_opBodyMesh->m_oNormalFaceData.Append(nMSNormalIndices[0] + iNormalBase);

                // Save normal indices 1 and 2 in right or left order.
            if (m_blRightHandedCoordinateSystem)
            {
                m_opBodyMesh->m_oNormalFaceData.Append(nMSNormalIndices[1] + iNormalBase);
                m_opBodyMesh->m_oNormalFaceData.Append(nMSNormalIndices[2] + iNormalBase);
            }
            else
            {
                m_opBodyMesh->m_oNormalFaceData.Append(nMSNormalIndices[2] + iNormalBase);
                m_opBodyMesh->m_oNormalFaceData.Append(nMSNormalIndices[1] + iNormalBase);
            }
        }

        // Extract texture coordinates.

            // Collect texture vectors.
        for (iVertexIndex = 0; iVertexIndex < iVertexCount; iVertexIndex++)
        {
            pMSVertex = msMesh_GetVertexAt(pMSMesh, iVertexIndex);

                // Get MilkShape texture vector.
            msVertex_GetTexCoords(pMSVertex, oMSTextureVector);

                // Get our texture vector.
            oTextureVector.x = oMSTextureVector[0];
            oTextureVector.y = oMSTextureVector[1];

                // Flip textures vertically if requested.
            if (m_blFlipTextureVertical)
                oTextureVector.y = 1.0f - oTextureVector.y;

                // Flip textures horizontally if requested.
            if (m_blFlipTextureHorizontal)
                oTextureVector.x = 1.0f - oTextureVector.x;

                // Save it.
            m_opBodyMesh->m_oTextureCoordinates.Append(oTextureVector);
        }

        // Extract material indices.

            // Get material index.
        iMaterialIndex = msMesh_GetMaterialIndex(pMSMesh);

            // Load face material indices.
        for (iFaceIndex = 0; iFaceIndex < iFaceCount; iFaceIndex++)
            m_opBodyMesh->m_oMaterialFaceIndices.Append(iMaterialIndex);

            // Increment base indices.
        iVertexBase += iVertexCount;
        iNormalBase += iNormalCount;
    }

        // Close holes.
    if (m_blCloseHoles)
        m_opBodyMesh->CloseHoles();

        // Get offset matrices.
    GetOffsetMatrices();

        // Offset the mesh positions.
    m_opBodyMesh->TransformPositions(m_oOffsetTranslationMatrix);
}

    // Get offset matrices.
void cPlugIn::GetOffsetMatrices()
{
    int iVertexIndex;
    int iVertexCount = m_opBodyMesh->m_oPositions.Count();
    D3DXVECTOR3 oVector;
    float BIG = 1000000.0f;
    D3DXVECTOR3 oDiffVector;
    D3DXVECTOR3 oCenterVector;
    D3DXVECTOR3 oOffsetVector(0.0f, 0.0f, 0.0f);

    D3DXMatrixIdentity(&m_oOffsetMatrix);
    D3DXMatrixIdentity(&m_oOffsetRotationMatrix);

    m_oMinVector = D3DXVECTOR3(BIG, BIG, BIG);
    m_oMaxVector = D3DXVECTOR3(-BIG, -BIG, -BIG);

        // Collect vertex positions.
    for (iVertexIndex = 0; iVertexIndex < iVertexCount; iVertexIndex++)
    {
            // Get position vector.
        oVector = m_opBodyMesh->m_oPositions.Indexed(iVertexIndex);

            // Apply scaling.
        oVector.x *= m_fPositionFactor;
        oVector.y *= m_fPositionFactor;
        oVector.z *= m_fPositionFactor;

        // Update min/max.

        if (oVector.x < m_oMinVector.x)
            m_oMinVector.x = oVector.x;

        if (oVector.y < m_oMinVector.y)
            m_oMinVector.y = oVector.y;

        if (oVector.z < m_oMinVector.z)
            m_oMinVector.z = oVector.z;

        if (oVector.x > m_oMaxVector.x)
            m_oMaxVector.x = oVector.x;

        if (oVector.y > m_oMaxVector.y)
            m_oMaxVector.y = oVector.y;

        if (oVector.z > m_oMaxVector.z)
            m_oMaxVector.z = oVector.z;

            // Reset position.
        m_opBodyMesh->m_oPositions.Set(iVertexIndex, oVector, false);
    }

        // Get min/max difference.
    oDiffVector.x = m_oMaxVector.x - m_oMinVector.x;
    oDiffVector.y = m_oMaxVector.y - m_oMinVector.y;
    oDiffVector.z = m_oMaxVector.z - m_oMinVector.z;

        // Get center.
    oCenterVector.x = m_oMinVector.x + oDiffVector.x/2.0f;
    oCenterVector.y = m_oMinVector.y + oDiffVector.y/2.0f;
    oCenterVector.z = m_oMinVector.z + oDiffVector.z/2.0f;

        // Handle centering.
    if (m_blCenterAtOffset)
    {
        oOffsetVector.x = m_oPositionOffset.x - oCenterVector.x;
        oOffsetVector.y = m_oPositionOffset.y - oCenterVector.y;
        oOffsetVector.z = m_oPositionOffset.z - oCenterVector.z;
    }
    else if (m_blStandAtOffset)
    {
        oOffsetVector.x = m_oPositionOffset.x - oCenterVector.x;
        oOffsetVector.y = m_oPositionOffset.y - m_oMinVector.y;
        oOffsetVector.z = m_oPositionOffset.z - oCenterVector.z;
    }

    // Set up transformation matrix.

        // Convert the bone position to a translation matrix.
    D3DXMatrixTranslation(
        &m_oOffsetTranslationMatrix,
        oOffsetVector.x,
        oOffsetVector.y,
        oOffsetVector.z);

        // Convert the bone rotation to a rotation matrix.
    MatrixRotationYawPitchRoll(
        &m_oOffsetRotationMatrix,
        (m_oRotation.y * PI)/180.0f,
        (m_oRotation.x * PI)/180.0f,
        (m_oRotation.z * PI)/180.0f);

        // Combine the translation and rotation matrix to create the main matrix.
    D3DXMatrixMultiply(
        &m_oOffsetMatrix,
        &m_oOffsetRotationMatrix,
        &m_oOffsetTranslationMatrix);

#if MSPLUGIN_TRACE
    DisplayMatrix("m_oOffsetMatrix", 0, m_oOffsetMatrix);
    DisplayMatrix("m_oOffsetRotationMatrix", 0, m_oOffsetRotationMatrix);
#endif
}

    // Create the main frame.
void cPlugIn::CreateMainFrame()
{
        // Create the main frame object.
    m_opRootFrame = new Frame(m_caRootFrameName);
}

    // Create child frames.
bool cPlugIn::CreateFrames(msModel *pMSModel)
{
    int iBoneCount = msModel_GetBoneCount(pMSModel);
    int iBoneIndex;
    char caName[kNameLen + 1];
    char caParentName[kNameLen + 1];
    msBone *pMSBone;
    Frame *opFrame;
    Frame *opParent;
    msVec3 oMSPosition;
    msVec3 oMSRotation;
    D3DXMATRIX oMatrix;
    D3DXMATRIX oTranslationMatrix;
    D3DXMATRIX oRotationMatrix;
    float fZFactor = (m_blRightHandedCoordinateSystem ? 1.0f : -1.0f);

    if (!iBoneCount)
    {
        m_oBoneFrames.Append(m_opRootFrame);
        return(true);
    }

        // Create a frame for each bone.
    for (iBoneIndex = 0; iBoneIndex < iBoneCount; iBoneIndex++)
    {
            // Get bone.
        pMSBone = msModel_GetBoneAt(pMSModel, iBoneIndex);

            // Get bone name.
        msBone_GetName(pMSBone, caName, kNameLen);

            // Get bone parent name.
        msBone_GetParentName(pMSBone, caParentName, kNameLen);

            // Convert to legal .x names if necessary.
		MakeNameLegal(caName, false);
		MakeNameLegal(caParentName, false);

            // Create a new frame for the bone.
        opFrame = new Frame(caName);

            // Get the bone position and rotation.
        msBone_GetPosition(pMSBone, oMSPosition);
        msBone_GetRotation(pMSBone, oMSRotation);

            // Initialize translation and rotation matrices to identities.
        D3DXMatrixIdentity(&oTranslationMatrix);
        D3DXMatrixIdentity(&oRotationMatrix);

            // Convert the bone position to a translation matrix.
        D3DXMatrixTranslation(
            &oMatrix,
            oMSPosition[0] * m_fPositionFactor,
            oMSPosition[1] * m_fPositionFactor,
            oMSPosition[2] * m_fPositionFactor * fZFactor);

            // Only offset root bone.
        if (iBoneIndex == 0)
        {
                // Add in offset.
            D3DXMatrixMultiply(
                &oTranslationMatrix,
                &m_oOffsetTranslationMatrix,
                &oMatrix);
        }
        else
            oTranslationMatrix = oMatrix;

            // Convert the bone rotation to a rotation matrix.
        MatrixRotationYawPitchRoll(
            &oRotationMatrix,
            oMSRotation[1] * m_oEulerFactor.y,
            oMSRotation[0] * m_oEulerFactor.x,
            oMSRotation[2] * m_oEulerFactor.z * fZFactor);

            // Save rotation matrix for animations.
        opFrame->m_oRotationMatrix.m_oMatrix = oRotationMatrix;

            // Combine the translation and rotation matrix.
        D3DXMatrixMultiply(
            &opFrame->m_oMatrix.m_oMatrix,
            &oRotationMatrix,
            &oTranslationMatrix);

            // Only offset root bone.
        if (iBoneIndex == 0)
        {
                // Add in rotation offset.
            D3DXMatrixMultiply(
                &oMatrix,
                &m_oOffsetRotationMatrix,
                &oRotationMatrix);

                // Add in rotation offset.
            D3DXMatrixMultiply(
                &opFrame->m_oOutputMatrix.m_oMatrix,
                &oMatrix,
                &oTranslationMatrix);
        }
        else
            opFrame->m_oOutputMatrix.m_oMatrix = opFrame->m_oMatrix.m_oMatrix;

            // Get the inverse matrix.
        D3DXMatrixInverse(
            &opFrame->m_oInverseMatrix.m_oMatrix,
            NULL,
            &opFrame->m_oMatrix.m_oMatrix);

            // Find the parent frame.
        if ((opParent = m_opRootFrame->FindDescendent(caParentName)) == NULL)
            opParent = m_opRootFrame;

            // Get the world matrix.
        D3DXMatrixMultiply(
            &opFrame->m_oWorldMatrix.m_oMatrix,
            &opFrame->m_oMatrix.m_oMatrix,
            &opParent->m_oWorldMatrix.m_oMatrix);

            // Get the world rotation matrix.
        D3DXMatrixMultiply(
            &opFrame->m_oWorldRotationMatrix.m_oMatrix,
            &oRotationMatrix,
            &opParent->m_oWorldRotationMatrix.m_oMatrix);

            // Get the inverse world matrix.
        D3DXMatrixInverse(
            &opFrame->m_oInverseWorldMatrix.m_oMatrix,
            NULL,
            &opFrame->m_oWorldMatrix.m_oMatrix);

            // Get the inverse rotation matrix.
        D3DXMatrixInverse(
            &opFrame->m_oInverseRotationMatrix.m_oMatrix,
            NULL,
            &oRotationMatrix);

            // Get the inverse world rotation matrix.
        D3DXMatrixMultiply(
            &opFrame->m_oInverseWorldRotationMatrix.m_oMatrix,
            &opFrame->m_oInverseRotationMatrix.m_oMatrix,
            &opParent->m_oInverseWorldRotationMatrix.m_oMatrix);

#if MSPLUGIN_TRACE
        // Debug info.
        //D3DXVECTOR3 oVector(6.0f, 2.0f, -2.0f);
        D3DXVECTOR3 oVector(5.0f, 5.0f, 5.0f);
        Debug("Vector(%f, %f, %f)\n", oVector.x, oVector.y, oVector.z);
        DisplayMatrix("Translation", iBoneIndex, oTranslationMatrix);
        DisplayTransformation("Translation", oVector, oTranslationMatrix);
        DisplayMatrix("Rotation", iBoneIndex, oRotationMatrix);
        DisplayTransformation("Rotation", oVector, oRotationMatrix);
        DisplayMatrix("Matrix", iBoneIndex, opFrame->m_oMatrix.m_oMatrix);
        DisplayTransformation("Matrix", oVector, opFrame->m_oMatrix.m_oMatrix);
        DisplayMatrix("WorldMatrix", iBoneIndex, opFrame->m_oWorldMatrix.m_oMatrix);
        DisplayTransformation("WorldMatrix", oVector, opFrame->m_oWorldMatrix.m_oMatrix);
        DisplayMatrix("InverseMatrix", iBoneIndex, opFrame->m_oInverseMatrix.m_oMatrix);
        DisplayTransformation("InverseMatrix", oVector, opFrame->m_oInverseMatrix.m_oMatrix);
        DisplayMatrix("InverseWorldMatrix", iBoneIndex, opFrame->m_oInverseWorldMatrix.m_oMatrix);
        DisplayTransformation("InverseWorldMatrix", oVector, opFrame->m_oInverseWorldMatrix.m_oMatrix);
        DisplayMatrix("InverseWorldRotationMatrix", iBoneIndex, opFrame->m_oInverseWorldRotationMatrix.m_oMatrix);
        DisplayTransformation("InverseWorldRotationMatrix", oVector, opFrame->m_oInverseWorldRotationMatrix.m_oMatrix);
#endif

            // Adopt the frame as a child of the parent.
        opParent->AdoptChild(opFrame);

            // Also save the frame in our frame list.
        opFrame->m_iBoneIndex = m_oBoneFrames.Count();
        m_oBoneFrames.Append(opFrame);
    }

        // If extended DirectX format.
    if (m_eFormat == kFormatDirectXSkinAndBones)
    {
            // Get main bone frame, to be parent of body frame.
        opFrame = m_oBoneFrames.Indexed(0);

            // Create body frame.
        m_opBodyFrame = new Frame("Body");

            // Adopt in main bone frame.
        opFrame->m_oChildren.Append(m_opBodyFrame);
    }

    return(true);
}

    // Create retained mode meshes.
bool cPlugIn::CreateRetainedModeMeshes()
{
        // Collect faces not assigned to any of the bones because
        // they straddle bones not imediately connected.
    CollectUnassignedFaces();

    int iTotalVertexCount = m_opBodyMesh->m_oPositions.Count();
    int iVertexCount;
    int iVertexIndex;
    int iNewVertexIndex;
    int iTotalNormalCount = m_opBodyMesh->m_oNormals.Count();
    int iNormalCount;
    int iNormalIndex;
    int iNewNormalIndex;
    int iBoneIndex;
    int iBoneCount = m_oBoneFrames.Count();
    int iVertexBoneIndex;
    int iTotalFaceCount = m_opBodyMesh->m_oFaceData.Count()/3;
    int iFaceCount;
    int iFaceIndex;
    int iFaceVertexCount = 3;
    int iFaceVertexIndex;
    int iTotalMaterialCount = m_opBodyMesh->m_oMaterials.Count();
    int iMaterialCount;
    int iMaterialIndex;
    int iNewMaterialIndex;
    char caName[kNameLen + 1];
    Frame *opBoneFrame;
    Frame *opFrame;
    Mesh *opMesh;
    Material *opMaterial;
    D3DXVECTOR3 oVector;
    D3DXVECTOR4 oVector4;
    D3DXVECTOR3 oNormal;
    D3DXVECTOR2 oTextureVector;
    IntegerArray oVertexMap(iTotalVertexCount);
    IntegerArray oNormalMap(iTotalNormalCount);
    IntegerArray oMaterialMap(iTotalMaterialCount);
    int iThisBoneCount;
    int iAdoptedBoneCount;
    bool blRet = true;

    for (iBoneIndex = 0; iBoneIndex < iBoneCount; iBoneIndex++)
    {
            // Get bone frame.
        opBoneFrame = m_oBoneFrames.Indexed(iBoneIndex);

            // Format mesh name.
        sprintf(
            caName,
            "%sMesh",
            (opBoneFrame == m_opRootFrame ? m_caBaseName : opBoneFrame->Name()));

            // Create mesh object.
        opMesh = new Mesh(caName);

            // Pre-size arrays.
        opMesh->m_oPositions.SetSize(iTotalVertexCount);
        opMesh->m_oBoneIndices.SetSize(iTotalVertexCount);
        opMesh->m_oFaceData.SetSize(iTotalFaceCount * 3);
        opMesh->m_oNormals.SetSize(iTotalVertexCount);
        opMesh->m_oTextureCoordinates.SetSize(iTotalVertexCount);
        opMesh->m_oMaterialFaceIndices.SetSize(iTotalFaceCount);
        opMesh->m_oMaterials.SetSize(iTotalMaterialCount);

            // Initialize index maps.
        oVertexMap.SetCount(0);
        oNormalMap.SetCount(0);
        oMaterialMap.SetCount(0);

        for (iVertexIndex = 0; iVertexIndex < iTotalVertexCount; iVertexIndex++)
            oVertexMap.Append(-1);

        for (iNormalIndex = 0; iNormalIndex < iTotalNormalCount; iNormalIndex++)
            oNormalMap.Append(-1);

        for (iMaterialIndex = 0; iMaterialIndex < iTotalMaterialCount; iMaterialIndex++)
            oMaterialMap.Append(-1);

        // Extract vertices.

            // Initialize counts.
        iVertexCount = 0;
        iFaceCount = 0;
        iMaterialCount = 0;

            // Initialize new indices.
        iNewVertexIndex = 0;
        iNewMaterialIndex = 0;

            // Scan face data for faces belonging to the current bone.
        for (iFaceIndex = 0; iFaceIndex < iTotalFaceCount; iFaceIndex++)
        {
                // Initialize belongs-to-bone flag.
            iThisBoneCount = 0;
            iAdoptedBoneCount = 0;

                // See if the face belongs to the current bone.
            for (iFaceVertexIndex = 0; iFaceVertexIndex < iFaceVertexCount; iFaceVertexIndex++)
            {
                    // Get vertex index.
                iVertexIndex = m_opBodyMesh->m_oFaceData.Indexed(
                    (iFaceIndex * iFaceVertexCount) + iFaceVertexIndex);

                    // Get vertex bone index.
                iVertexBoneIndex = m_opBodyMesh->m_oBoneIndices.Indexed(iVertexIndex);

                    // Check if vertex belongs to the current bone.
                if (iVertexBoneIndex == iBoneIndex)
                    iThisBoneCount++;
                else
                {
                    opFrame = m_oBoneFrames.Indexed(iVertexBoneIndex);

                    if (opFrame->m_iLevel > opBoneFrame->m_iLevel)
                        iAdoptedBoneCount++;
                }
            }

                // If the face doesn't belong to the current bone, continue.
            if (!iThisBoneCount || (iThisBoneCount + iAdoptedBoneCount != 3))
                continue;

                // Collect the vertex indices into the map.
            for (iFaceVertexIndex = 0; iFaceVertexIndex < iFaceVertexCount; iFaceVertexIndex++)
            {
                    // Get vertex index.
                iVertexIndex = m_opBodyMesh->m_oFaceData.Indexed(
                    (iFaceIndex * iFaceVertexCount) + iFaceVertexIndex);

                    // See map entry.
                oVertexMap.Set(iVertexIndex, iVertexIndex, false);

                    // Get normal index.
                iNormalIndex = m_opBodyMesh->m_oNormalFaceData.Indexed(
                    (iFaceIndex * iFaceVertexCount) + iFaceVertexIndex);

                    // See map entry.
                oNormalMap.Set(iNormalIndex, iNormalIndex, false);
            }
        }

        iNewVertexIndex = 0;

            // Fix up the vertex map.
        for (iVertexIndex = 0; iVertexIndex < iTotalVertexCount; iVertexIndex++)
        {
            if (oVertexMap.Indexed(iVertexIndex) != -1)
            {
                oVertexMap.Set(iVertexIndex, iNewVertexIndex, false);
                iNewVertexIndex++;
            }
        }

        iNewNormalIndex = 0;

            // Fix up the normal map.
        for (iNormalIndex = 0; iNormalIndex < iTotalNormalCount; iNormalIndex++)
        {
            if (oNormalMap.Indexed(iNormalIndex) != -1)
            {
                oNormalMap.Set(iNormalIndex, iNewNormalIndex, false);
                iNewNormalIndex++;
            }
        }

            // Collect the face data and material data for the current bone.
        for (iFaceIndex = 0; iFaceIndex < iTotalFaceCount; iFaceIndex++)
        {
                // Initialize belongs-to-bone flag.
            iThisBoneCount = 0;
            iAdoptedBoneCount = 0;

                // See if the face belongs to the current bone.
            for (iFaceVertexIndex = 0; iFaceVertexIndex < iFaceVertexCount; iFaceVertexIndex++)
            {
                    // Get vertex index.
                iVertexIndex = m_opBodyMesh->m_oFaceData.Indexed(
                    (iFaceIndex * iFaceVertexCount) + iFaceVertexIndex);

                    // Get vertex bone index.
                iVertexBoneIndex = m_opBodyMesh->m_oBoneIndices.Indexed(iVertexIndex);

                    // Check if vertex belongs to the current bone.
                if (iVertexBoneIndex == iBoneIndex)
                    iThisBoneCount++;
                else
                {
                    opFrame = m_oBoneFrames.Indexed(iVertexBoneIndex);

                    if (opFrame->m_iLevel > opBoneFrame->m_iLevel)
                        iAdoptedBoneCount++;
                }
            }

                // If the face doesn't belong to the current bone, continue.
            if (!iThisBoneCount || (iThisBoneCount + iAdoptedBoneCount != 3))
                continue;

                // Collect the vertex indices and face data.
            for (iFaceVertexIndex = 0; iFaceVertexIndex < iFaceVertexCount; iFaceVertexIndex++)
            {
                    // Get vertex index.
                iVertexIndex = m_opBodyMesh->m_oFaceData.Indexed(
                    (iFaceIndex * iFaceVertexCount) + iFaceVertexIndex);

                    // Get map entry.
                iNewVertexIndex = oVertexMap.Indexed(iVertexIndex);

                    // Save the vertex index in the new face data.
                opMesh->m_oFaceData.Append(iNewVertexIndex);

                    // Get normal index.
                iNormalIndex = m_opBodyMesh->m_oNormalFaceData.Indexed(
                    (iFaceIndex * iFaceVertexCount) + iFaceVertexIndex);

                    // See map entry.
                iNewNormalIndex = oNormalMap.Indexed(iNormalIndex);

                    // Save the normal index in the new normal face data.
                opMesh->m_oNormalFaceData.Append(iNewNormalIndex);
            }

                // Get face material index.
            iMaterialIndex = m_opBodyMesh->m_oMaterialFaceIndices.Indexed(iFaceIndex);

            if (iMaterialIndex != -1)
            {
                    // Get new mapped material index.
                iNewMaterialIndex = oMaterialMap.Indexed(iMaterialIndex);

                if (iNewMaterialIndex == -1)
                {
                        // Allocate a new vertex index.
                    iNewMaterialIndex = iMaterialCount++;

                        // Update the map.
                    oMaterialMap.Set(iMaterialIndex, iNewMaterialIndex, false);

                        // Get the material.
                    opMaterial = m_opBodyMesh->m_oMaterials.Indexed(iMaterialIndex);

                        // Save the material.
                    opMesh->m_oMaterials.Append(new Material(*opMaterial));
                }

                    // Save the vertex index in the new face data.
                opMesh->m_oMaterialFaceIndices.Append(iNewMaterialIndex);
            }
        }

        iVertexCount = iNewVertexIndex;

            // Collect the vertices.
        for (iVertexIndex = 0; iVertexIndex < iTotalVertexCount; iVertexIndex++)
        {
                // Get the new vertex index.
            iNewVertexIndex = oVertexMap.Indexed(iVertexIndex);

                // If vertex not part of this bone, continue.
            if (iNewVertexIndex == -1)
                continue;

                // Get the vertex.
            oVector = m_opBodyMesh->m_oPositions.Indexed(iVertexIndex);

                // Transform vertex position to the local bone coordinate space.
            D3DXVec3Transform(
                &oVector4,
                &oVector,
                &opBoneFrame->m_oInverseWorldMatrix.m_oMatrix);

                // Save the vertex.
            opMesh->m_oPositions.Append(*(D3DXVECTOR3 *)&oVector4);

                // Get the texture coordinates.
            oTextureVector = m_opBodyMesh->m_oTextureCoordinates.Indexed(iVertexIndex);

                // Save the texture coordinates.
            opMesh->m_oTextureCoordinates.Append(oTextureVector);
        }

        iNormalCount = iNewNormalIndex;

            // Collect the normals.
        if (!m_blExcludeNormals)
        {
            for (iNormalIndex = 0; iNormalIndex < iTotalNormalCount; iNormalIndex++)
            {
                    // Get the new vertex index.
                iNewNormalIndex = oNormalMap.Indexed(iNormalIndex);

                    // If vertex not part of this bone, continue.
                if (iNewNormalIndex == -1)
                    continue;

                    // Get the normal.
                oVector = m_opBodyMesh->m_oNormals.Indexed(iNormalIndex);

                    // Transform normal to bone space.
                D3DXVec3TransformNormal(
                    &oNormal,
                    &oVector,
                    &opBoneFrame->m_oInverseWorldRotationMatrix.m_oMatrix);

                    // Save the normal.
                opMesh->m_oNormals.Append(oNormal);
            }
        }

            // Fill holes.
        if (m_blCloseHoles)
            opMesh->CloseHoles();

            // Compact the normals.
        opMesh->CompactNormals();

        if (opMesh->m_oPositions.Count())
                // Save mesh in owning frame.
            opBoneFrame->m_oMeshes.Append(opMesh);
        else
                // Delete empty mesh.
            delete opMesh;
    }

    return(blRet);
}

    // Collect unassigned faces.
bool cPlugIn::CollectUnassignedFaces()
{
    int iTargetBoneIndex;
    int iVertexIndex;
    int iVertexIndex0;
    int iVertexIndex1;
    int iVertexIndex2;
    int iVertexCount;
    int iNormalIndex;
    int iBoneIndex0;
    int iBoneIndex1;
    int iBoneIndex2;
    int iBoneCount = m_oBoneFrames.Count();
    int iTotalFaceCount = m_opBodyMesh->m_oFaceData.Count()/3;
    int iFaceIndex;
    int iFaceVertexCount = 3;
    int iFaceVertexIndex;
    int iTotalMaterialCount = m_opBodyMesh->m_oMaterials.Count();
    Frame *opFrame0;
    Frame *opFrame1;
    Frame *opFrame2;
    Frame *opParentA;
    Frame *opParent;
    int iLevel0;
    int iLevel1;
    int iLevel2;
    D3DXVECTOR3 oVector;
    D3DXVECTOR2 oTextureVector;
    int iUnassignedCount;
    bool blRet = true;

        // Scan face data for faces belonging to the current bone.
    for (iFaceIndex = 0; iFaceIndex < iTotalFaceCount; iFaceIndex++)
    {
        iVertexIndex0 = m_opBodyMesh->m_oFaceData.Indexed(
            (iFaceIndex * iFaceVertexCount));
        iVertexIndex1 = m_opBodyMesh->m_oFaceData.Indexed(
            (iFaceIndex * iFaceVertexCount) + 1);
        iVertexIndex2 = m_opBodyMesh->m_oFaceData.Indexed(
            (iFaceIndex * iFaceVertexCount) + 2);

        iBoneIndex0 = m_opBodyMesh->m_oBoneIndices.Indexed(iVertexIndex0);
        iBoneIndex1 = m_opBodyMesh->m_oBoneIndices.Indexed(iVertexIndex1);
        iBoneIndex2 = m_opBodyMesh->m_oBoneIndices.Indexed(iVertexIndex2);

        opFrame0 = m_oBoneFrames.Indexed(iBoneIndex0);
        opFrame1 = m_oBoneFrames.Indexed(iBoneIndex1);
        opFrame2 = m_oBoneFrames.Indexed(iBoneIndex2);

        iLevel0 = opFrame0->m_iLevel;
        iLevel1 = opFrame1->m_iLevel;
        iLevel2 = opFrame2->m_iLevel;

        iUnassignedCount = 0;
        iTargetBoneIndex = 0;

        if ((iBoneIndex0 != iBoneIndex1) && (iLevel0 == iLevel1))
        {
            iUnassignedCount++;
        }
        else if ((iBoneIndex0 != iBoneIndex2) && (iLevel0 == iLevel2))
        {
            iUnassignedCount++;
        }
        else if ((iBoneIndex1 != iBoneIndex2) && (iLevel1 == iLevel2))
        {
            iUnassignedCount++;
        }

        if (!iUnassignedCount)
            continue;

            // Find common parent of the vertices.
        opParentA = opFrame0->FindCommonParent(opFrame1);

        if (opParentA)
            opParent = opParentA->FindCommonParent(opFrame2);
        else
            opParent = NULL;

            // Get the target bone index.
        if (opParent)
            iTargetBoneIndex = opParent->m_iBoneIndex;

            // Collect the vertex indices into the map.
        for (iFaceVertexIndex = 0; iFaceVertexIndex < iFaceVertexCount; iFaceVertexIndex++)
        {
                // Get vertex index.
            iVertexIndex = m_opBodyMesh->m_oFaceData.Indexed(
                (iFaceIndex * iFaceVertexCount) + iFaceVertexIndex);

                // Get vertex index.
            iNormalIndex = m_opBodyMesh->m_oNormalFaceData.Indexed(
                (iFaceIndex * iFaceVertexCount) + iFaceVertexIndex);

                // Get new vertex index.
            iVertexCount = m_opBodyMesh->m_oPositions.Count();

                // Copy face data.
            m_opBodyMesh->m_oFaceData.Append(iVertexCount);
            m_opBodyMesh->m_oNormalFaceData.Append(iNormalIndex);

                // Add new vertex position.
            oVector = m_opBodyMesh->m_oPositions.Indexed(iVertexIndex);
            m_opBodyMesh->m_oPositions.Append(oVector);

                // Add new texture vector.
            oTextureVector = m_opBodyMesh->m_oTextureCoordinates.Indexed(iVertexIndex);
            m_opBodyMesh->m_oTextureCoordinates.Append(oTextureVector);

                // Add bone index.
            m_opBodyMesh->m_oBoneIndices.Append(iTargetBoneIndex);
        }
    }

    return(blRet);
}

    // Process skin and bones mesh.
bool cPlugIn::CreateSkinAndBonesMesh()
{
    Frame *opBoneFrame;
    Mesh *opMesh = m_opBodyMesh;
    SkinWeightData *opSkinWeightData;
    int iVertexCount = opMesh->m_oPositions.Count();
    int iVertexIndex;
    int iOriginalVertexCount;
    int iNewVertexIndex;
    int iNormalIndex;
    int iBoneIndex;
    int iBoneCount;
    int iFaceCount = opMesh->m_oFaceData.Count()/3;
    int iFaceIndex;
    int iFaceVertexBase;
    int iFaceVertexCount = 3;
    int iFaceVertexIndex;
    int iFaceVertexTotal = opMesh->m_oFaceData.Count();
    int iBoneVertexCount;
    int iIndex;
    D3DXVECTOR3 oVector;
    D3DXVECTOR4 oVector4;
    D3DXVECTOR3 oNormal;
    D3DXVECTOR2 oTextureVector;
    IntegerArray oVertexNonOriginalIndices;
    VectorArray oSortedPositions;
    IntegerArray oSortedBoneIndices;
    TextureVectorArray oSortedTextureVectors;
    IntegerArray oVertexIndexMapToSorted;
    IntegerArray oVertexIndexMapToUnsorted;
    IntegerArray oVertexNormalMap;
    float fZFactor = (m_blRightHandedCoordinateSystem ? 1.0f : -1.0f);
    bool blRet = true;

    // Sort vertices to put unique vertices first.

        // Pre-size arrays.
    opMesh->m_oVertexOriginalIndices.SetSize(iVertexCount);
    oVertexNonOriginalIndices.SetSize(iFaceVertexTotal);

        // Collect vertex indices.
    for (iVertexIndex = 0; iVertexIndex < iVertexCount; iVertexIndex++)
    {
            // Get vertex position.
        oVector = opMesh->m_oPositions.Indexed(iVertexIndex);

            // Look for a duplicate.
        for (iIndex = 0; iIndex < iVertexIndex; iIndex++)
        {
            if (oVector == opMesh->m_oPositions.Indexed(iIndex))
                break;
        }

            // Check for duplicate.
        if (iIndex == iVertexIndex)
            opMesh->m_oVertexOriginalIndices.Append(iIndex);
        else
            oVertexNonOriginalIndices.Append(iVertexIndex);
    }

        // Presize sorted vertex position and index map arrays.
    oSortedPositions.SetSize(iVertexCount);
    oSortedTextureVectors.SetSize(iVertexCount);
    oSortedBoneIndices.SetSize(iVertexCount);
    oVertexIndexMapToSorted.SetSize(iFaceVertexTotal);
    oVertexIndexMapToSorted.SetCount(iFaceVertexTotal);
    oVertexIndexMapToUnsorted.SetSize(iFaceVertexTotal);
    oVertexIndexMapToUnsorted.SetCount(iFaceVertexTotal);

        // Load sorted vertex array - orginals first.
    for (iVertexIndex = 0; iVertexIndex < opMesh->m_oVertexOriginalIndices.Count(); iVertexIndex++)
    {
        iIndex = opMesh->m_oVertexOriginalIndices.Indexed(iVertexIndex);
        oSortedPositions.Append(opMesh->m_oPositions.Indexed(iIndex));
        oSortedTextureVectors.Append(opMesh->m_oTextureCoordinates.Indexed(iIndex));
        oSortedBoneIndices.Append(opMesh->m_oBoneIndices.Indexed(iIndex));
        oVertexIndexMapToSorted.Set(iIndex, iVertexIndex, false);
        oVertexIndexMapToUnsorted.Set(iVertexIndex, iIndex, false);
    }

        // Load sorted vertex array - duplicates next.
    for (iVertexIndex = 0; iVertexIndex < oVertexNonOriginalIndices.Count(); iVertexIndex++)
    {
        iIndex = oVertexNonOriginalIndices.Indexed(iVertexIndex);
        oSortedPositions.Append(opMesh->m_oPositions.Indexed(iIndex));
        oSortedTextureVectors.Append(opMesh->m_oTextureCoordinates.Indexed(iIndex));
        oSortedBoneIndices.Append(opMesh->m_oBoneIndices.Indexed(iIndex));
        oVertexIndexMapToSorted.Set(
            iIndex,
            iVertexIndex + opMesh->m_oVertexOriginalIndices.Count(),
            false);
        oVertexIndexMapToUnsorted.Set(
            iVertexIndex + opMesh->m_oVertexOriginalIndices.Count(),
            iIndex,
            false);
    }

        // Now copy over the vertices sorted by originality.
    opMesh->m_oPositions = oSortedPositions;
    opMesh->m_oTextureCoordinates = oSortedTextureVectors;
    opMesh->m_oBoneIndices = oSortedBoneIndices;

    // Fix up mesh face data.

    for (iFaceIndex = 0; iFaceIndex < iFaceCount; iFaceIndex++)
    {
        iFaceVertexBase = iFaceIndex * 3;

        for (iFaceVertexIndex = 0; iFaceVertexIndex < iFaceVertexCount; iFaceVertexIndex++)
        {
                // Fix up face data.
            opMesh->m_oFaceData.Set(
                iFaceVertexBase + iFaceVertexIndex,
                oVertexIndexMapToSorted.Indexed(opMesh->m_oFaceData.Indexed(iFaceVertexBase + iFaceVertexIndex)));
        }
    }

        // Get bone count.
    iBoneCount = m_oBoneFrames.Count();

        // Pre-size skin weight data array.
    opMesh->m_oSkinWeightData.SetSize(iBoneCount);

        // Process all the bones.
    for (iBoneIndex = 0; iBoneIndex < iBoneCount; iBoneIndex++)
    {
            // Get bone frame.
        opBoneFrame = m_oBoneFrames.Indexed(iBoneIndex);

            // Clear bone vertex count;
        iBoneVertexCount = 0;

            // Look for vertices with this bone index.
        for (iVertexIndex = 0; iVertexIndex < iVertexCount; iVertexIndex++)
        {
                // If vertex belongs to current bone.
            if (iBoneIndex == opMesh->m_oBoneIndices.Indexed(iVertexIndex))
                iBoneVertexCount++;
        }

            // If no vertices belong to this bone.
        if (!iBoneVertexCount)
            continue;

            // Create skin weight data object.
        opSkinWeightData = new SkinWeightData(opBoneFrame);

            // Set bone number.
        opSkinWeightData->m_iBoneNumber = iBoneIndex;

            // Pre-size vertex index and weight array.
        opSkinWeightData->m_oIndices.SetSize(iBoneVertexCount);
        opSkinWeightData->m_oWeights.SetSize(iBoneVertexCount);

            // Collect vertices with this bone index.
        for (iVertexIndex = 0; iVertexIndex < iVertexCount; iVertexIndex++)
        {
                // If vertex doesn't belong to current bone.
            if (iBoneIndex != opMesh->m_oBoneIndices.Indexed(iVertexIndex))
                continue;

                // Append vertex index and weight.
            opSkinWeightData->m_oIndices.Append(iVertexIndex);
            opSkinWeightData->m_oWeights.Append(1.0f);
        }

            // Set offset matrix.
        opSkinWeightData->m_oMatrix = opBoneFrame->m_oInverseWorldMatrix;

            // Adopt skin weight data.
        opMesh->m_oSkinWeightData.Append(opSkinWeightData);
    }

    // In addition to sorting out original vertices, we need to
    // add extra vertices for vertices sharing normals.

    iOriginalVertexCount = iVertexCount;

    oVertexNormalMap.SetSize(iFaceVertexTotal);
    oVertexNormalMap.SetCount(iFaceVertexTotal);

    opMesh->m_oVertexDuplicateIndices.SetSize(iFaceVertexTotal);
    opMesh->m_oVertexDuplicateIndices.SetCount(iFaceVertexTotal);

        // Initialize map to flag not used.
    for (iFaceVertexIndex = 0; iFaceVertexIndex < iFaceVertexTotal; iFaceVertexIndex++)
        oVertexNormalMap.Set(iFaceVertexIndex, -1, FALSE);

        // Find vertices with more than one normal, and add a new vertex
        // for them.
    for (iFaceVertexIndex = 0; iFaceVertexIndex < iFaceVertexTotal; iFaceVertexIndex++)
    {
        iVertexIndex = opMesh->m_oFaceData.Indexed(iFaceVertexIndex);
        iNormalIndex = opMesh->m_oNormalFaceData.Indexed(iFaceVertexIndex);

            // If no normal seen before for the vertex.
        if (oVertexNormalMap.Indexed(iVertexIndex) == -1)
                // Make an entry.
            oVertexNormalMap.Set(iVertexIndex, iNormalIndex);
            // Else if not the same normal, see if we need to add a vertex.
        else if (oVertexNormalMap.Indexed(iVertexIndex) != iNormalIndex)
        {
                // See if we already added a new vertex.
            for (iIndex = iOriginalVertexCount; iIndex < iVertexCount; iIndex++)
            {
                if ((opMesh->m_oVertexDuplicateIndices.Indexed(iIndex) == iVertexIndex) &&
                    (oVertexNormalMap.Indexed(iIndex) == iNormalIndex))
                {
                    iNewVertexIndex = iIndex;
                    break;
                }
            }

                // If no new entry already, make one.
            if (iIndex == iVertexCount)
            {
                    // Get the new vertex index.
                iNewVertexIndex = iVertexCount++;

                    // Append the duplicate vertex.
                opMesh->m_oPositions.Append(
                    opMesh->m_oPositions.Indexed(iVertexIndex));

                    // Append the duplicate texture coordinates.
                opMesh->m_oTextureCoordinates.Append(
                    opMesh->m_oTextureCoordinates.Indexed(iVertexIndex));

                    // Get bone index.
                iBoneIndex = opMesh->m_oBoneIndices.Indexed(iVertexIndex);

                    // Find matching skin weight data.
                for (iIndex = 0; iIndex < opMesh->m_oSkinWeightData.Count(); iIndex++)
                {
                        // Get skin weight data
                    opSkinWeightData = opMesh->m_oSkinWeightData.Indexed(iIndex);

                    if (opSkinWeightData &&
                        (opSkinWeightData->m_iBoneNumber == iBoneIndex))
                    {
                            // Append vertex index and weight.
                        opSkinWeightData->m_oIndices.Append(iNewVertexIndex);
                        opSkinWeightData->m_oWeights.Append(1.0f);
                        break;
                    }
                }

                    // Make an entry in normal map.
                oVertexNormalMap.Set(iNewVertexIndex, iNormalIndex);

                    // Add to the duplication list.
                opMesh->m_oVertexDuplicateIndices.Set(iNewVertexIndex, iVertexIndex, FALSE);
            }

                // Fix the face data.
            opMesh->m_oFaceData.Set(iFaceVertexIndex, iNewVertexIndex, FALSE);
        }
    }

    // Do vertex duplication indices.

        // Pre-size arrays.
    opMesh->m_oVertexOriginalIndices.SetCount(0);
    opMesh->m_oVertexOriginalIndices.SetSize(iVertexCount);
    opMesh->m_oVertexDuplicateIndices.SetCount(0);
    opMesh->m_oVertexDuplicateIndices.SetSize(iVertexCount);

        // Collect vertex indices.
    for (iVertexIndex = 0; iVertexIndex < iVertexCount; iVertexIndex++)
    {
            // Get vertex position.
        oVector = opMesh->m_oPositions.Indexed(iVertexIndex);

            // Look for a duplicate.
        for (iIndex = 0; iIndex < iVertexIndex; iIndex++)
        {
            if (oVector == opMesh->m_oPositions.Indexed(iIndex))
                break;
        }

            // Check for duplicate.
        if (iIndex == iVertexIndex)
            opMesh->m_oVertexOriginalIndices.Append(iIndex);
        else
            opMesh->m_oVertexDuplicateIndices.Append(iIndex);
    }

        // Save mesh in body frame.
    m_opBodyFrame->m_oMeshes.Append(opMesh);

        // Disconnect body mesh so we don't double-delete it.
    m_opBodyMesh = NULL;

    return(blRet);
}

    // Create bone matrices.
bool cPlugIn::CreateTransformations(msModel *pMSModel)
{
    int iBoneCount = msModel_GetBoneCount(pMSModel);
    int iBoneIndex;
    char caName[kNameLen + 1];
    char caParentName[kNameLen + 1];
    msBone *pMSBone;
    BoneMatrix *opMatrix;
    BoneMatrix *opParent;
    BoneMatrix oIdentityParent;
    msVec3 oMSPosition;
    msVec3 oMSRotation;
    D3DXMATRIX oTranslationMatrix;
    D3DXMATRIX oRotationMatrix;
    float fZFactor = (m_blRightHandedCoordinateSystem ? 1.0f : -1.0f);

    if (!iBoneCount)
        return(true);

        // Create a matrix for for each bone.
    for (iBoneIndex = 0; iBoneIndex < iBoneCount; iBoneIndex++)
    {
            // Get bone.
        pMSBone = msModel_GetBoneAt(pMSModel, iBoneIndex);

            // Get bone name.
        msBone_GetName(pMSBone, caName, kNameLen);

            // Get bone parent name.
        msBone_GetParentName(pMSBone, caParentName, kNameLen);

            // Convert to legal .x names if necessary.
		MakeNameLegal(caName, false);
		MakeNameLegal(caParentName, false);

            // Create a new matrix.
        opMatrix = new BoneMatrix(caName);

            // Get the bone position and rotation.
        msBone_GetPosition(pMSBone, oMSPosition);
        msBone_GetRotation(pMSBone, oMSRotation);

            // Initialize translation and rotation matrices to identities.
        D3DXMatrixIdentity(&oTranslationMatrix);
        D3DXMatrixIdentity(&oRotationMatrix);

            // Convert the bone position to a translation matrix.
        D3DXMatrixTranslation(
            &oTranslationMatrix,
            oMSPosition[0] * m_fPositionFactor,
            oMSPosition[1] * m_fPositionFactor,
            oMSPosition[2] * m_fPositionFactor * fZFactor);

            // Convert the bone rotation to a rotation matrix.
        MatrixRotationYawPitchRoll(
            &oRotationMatrix,
            oMSRotation[1] * m_oEulerFactor.y,
            oMSRotation[0] * m_oEulerFactor.x,
            oMSRotation[2] * m_oEulerFactor.z);

            // Save rotation matrix for animation.
        opMatrix->m_oRotationMatrix = oRotationMatrix;

            // Combine the translation and rotation matrix.
        D3DXMatrixMultiply(
            &opMatrix->m_oMatrix,
            &oRotationMatrix,
            &oTranslationMatrix);

            // If root bone.
        if (iBoneIndex == 0)
        {
#if 1
                // Offset matrix.
            D3DXMatrixMultiply(
                &opMatrix->m_oOutputMatrix,
                &opMatrix->m_oMatrix,
                &m_oOffsetMatrix);
#else
                // Offset matrix.
            D3DXMatrixMultiply(
                &opMatrix->m_oOutputMatrix,
                &m_oOffsetMatrix,
                &opMatrix->m_oMatrix);
#endif
        }
        else
                // Save as output matrix.
            opMatrix->m_oOutputMatrix = opMatrix->m_oMatrix;

            // Get the inverse matrix.
        D3DXMatrixInverse(
            &opMatrix->m_oInverseMatrix,
            NULL,
            &opMatrix->m_oMatrix);

            // Get the inverse rotation matrix.
        D3DXMatrixInverse(
            &opMatrix->m_oInverseRotationMatrix,
            NULL,
            &oRotationMatrix);

            // Find the parent frame.
        if ((opParent = m_opRootFrame->FindBoneMatrix(caParentName)) != NULL)
        {
                // Get the world matrix.
            D3DXMatrixMultiply(
                &opMatrix->m_oWorldMatrix,
                &opMatrix->m_oMatrix,
                &opParent->m_oWorldMatrix);

                // Get the world rotation matrix.
            D3DXMatrixMultiply(
                &opMatrix->m_oWorldRotationMatrix,
                &oRotationMatrix,
                &opParent->m_oWorldRotationMatrix);

                // Get the inverse world matrix.
            D3DXMatrixInverse(
                &opMatrix->m_oInverseWorldMatrix,
                NULL,
                &opMatrix->m_oWorldMatrix);

                // Get the inverse world rotation matrix.
            D3DXMatrixMultiply(
                &opMatrix->m_oInverseWorldRotationMatrix,
                &opMatrix->m_oInverseRotationMatrix,
                &opParent->m_oInverseWorldRotationMatrix);
        }
        else
        {
            opParent = NULL;

            opMatrix->m_oWorldMatrix = opMatrix->m_oMatrix;
            opMatrix->m_oInverseWorldMatrix = opMatrix->m_oInverseMatrix;
            opMatrix->m_oWorldRotationMatrix = opMatrix->m_oRotationMatrix;
            opMatrix->m_oInverseWorldRotationMatrix = opMatrix->m_oInverseRotationMatrix;
        }

#if MSPLUGIN_TRACE
        // Debug info.
        D3DXVECTOR3 oVector(0.0f, 0.0f, 0.0f);
        Debug("Vector(%f, %f, %f)\n", oVector.x, oVector.y, oVector.z);
        DisplayMatrix("Translation", iBoneIndex, oTranslationMatrix);
        DisplayTransformation("Translation", oVector, oTranslationMatrix);
        DisplayMatrix("Rotation", iBoneIndex, oRotationMatrix);
        DisplayTransformation("Rotation", oVector, oRotationMatrix);
        DisplayMatrix("Matrix", iBoneIndex, opMatrix->m_oMatrix);
        DisplayTransformation("Matrix", oVector, opMatrix->m_oMatrix);
        DisplayMatrix("WorldMatrix", iBoneIndex, opMatrix->m_oWorldMatrix);
        DisplayTransformation("WorldMatrix", oVector, opMatrix->m_oWorldMatrix);
        DisplayMatrix("InverseMatrix", iBoneIndex, opMatrix->m_oInverseMatrix);
        DisplayTransformation("InverseMatrix", oVector, opMatrix->m_oInverseMatrix);
        DisplayMatrix("InverseWorldMatrix", iBoneIndex, opMatrix->m_oInverseWorldMatrix);
        DisplayTransformation("InverseWorldMatrix", oVector, opMatrix->m_oInverseWorldMatrix);
        DisplayMatrix("InverseWorldRotationMatrix", iBoneIndex, opMatrix->m_oInverseWorldRotationMatrix);
        DisplayTransformation("InverseWorldRotationMatrix", oVector, opMatrix->m_oInverseWorldRotationMatrix);
#endif

            // Save parent matrix.
        opMatrix->m_opParent = opParent;

            // Adopt the matrix.
        m_opRootFrame->AdoptBoneMatrix(opMatrix);
    }

    return(true);
}

    // Create render groups.
bool cPlugIn::CreateRenderGroups(msModel *pMSModel)
{
    int iModelCount = msModel_GetMeshCount(pMSModel);
    int iModelIndex;
    int iVertexCount;
    int iVertexIndex;
    int iNormalCount;
    int iBoneIndex;
    int iFaceCount;
    int iFaceIndex;
    int iFaceVertexCount = 3;
    int iFaceVertexIndex;
    int iMaterialIndex;
    char caName[kNameLen + 1];
    char caTextureFileName[kNameLen + 1];
    char *cpBase;
    msMesh *pMSMesh;
    msVertex *pMSVertex;
    msTriangle *pMSTriangle;
    msMaterial *pMSMaterial;
    msVec3 oMSPosition;
    msVec3 oMSNormal;
    word nMSIndices[3];
    word nMSNormalIndices[3];
    msVec2 oMSTextureVector;
    msVec4 oMSColor;
    BoneMatrix *opMatrix;
    RenderGroup *opMesh;
    Material *opMaterial;
    D3DXVECTOR3 oVector;
    D3DXVECTOR4 oVector4;
    D3DXVECTOR3 oNormal;
    D3DXVECTOR2 oTextureVector;
    bool blRet = true;
    float fZFactor = (m_blRightHandedCoordinateSystem ? 1.0f : -1.0f);

    for (iModelIndex = 0; iModelIndex < iModelCount; iModelIndex++)
    {
            // Get MilkShape mesh.
        pMSMesh = msModel_GetMeshAt(pMSModel, iModelIndex);

            // Get mesh name.
        msMesh_GetName(pMSMesh, caName, kNameLen);

            // Convert to valid .x name if necessary.
		MakeNameLegal(caName, false);

            // Create mesh object.
        opMesh = new RenderGroup(caName);

        // Extract vertices.

            // Get vertex count.
        iVertexCount = msMesh_GetVertexCount(pMSMesh);

            // Presize vertex position array.
        opMesh->m_oPositions.SetSize(iVertexCount);

            // Collect vertex positions.
        for (iVertexIndex = 0; iVertexIndex < iVertexCount; iVertexIndex++)
        {
                // Get MilkShape vertex.
            pMSVertex = msMesh_GetVertexAt(pMSMesh, iVertexIndex);

                // Get MilkShape vertex position.
            msVertex_GetVertex(pMSVertex, oMSPosition);

                // Apply our position factor.
            oVector.x = oMSPosition[0] * m_fPositionFactor;
            oVector.y = oMSPosition[1] * m_fPositionFactor;
            oVector.z = fZFactor * oMSPosition[2] * m_fPositionFactor;

                // Get the vertex bone index.
            iBoneIndex = msVertex_GetBoneIndex(pMSVertex);

                // -1 means no bone associated, so reference the default bone.
            if (iBoneIndex < 0)
                iBoneIndex = 0;

                // Get matrix.
            if ((opMatrix = m_opRootFrame->GetBoneMatrix(iBoneIndex)) != NULL)
            {
                    // Transform vertex position to the local bone coordinate space.
                D3DXVec3Transform(
                    &oVector4,
                    &oVector,
                    &opMatrix->m_oInverseWorldMatrix);

                    // Save the vertex position.
                opMesh->m_oPositions.Append(*(D3DXVECTOR3 *)&oVector4);
            }
            else
                    // Save the vertex position.
                opMesh->m_oPositions.Append(oVector);

                // Save matrix index.
            opMesh->m_oMatrixIndices.Append(iBoneIndex);
        }

        // Extract mesh face data.

            // Get face count;
        iFaceCount = msMesh_GetTriangleCount(pMSMesh);

            // Presize face data array.
        opMesh->m_oFaceData.SetSize(iFaceCount * 3);

        for (iFaceIndex = 0; iFaceIndex < iFaceCount; iFaceIndex++)
        {
                // Get MilkShape triangle.
            msTriangle *pMSTriangle = msMesh_GetTriangleAt(pMSMesh, iFaceIndex);

                // Get face vertex indices.
            msTriangle_GetVertexIndices(pMSTriangle, nMSIndices);

                // Save vertex indices.
            opMesh->m_oFaceData.Append(nMSIndices[0]);

            if (m_blRightHandedCoordinateSystem)
            {
                opMesh->m_oFaceData.Append(nMSIndices[1]);
                opMesh->m_oFaceData.Append(nMSIndices[2]);
            }
            else
            {
                opMesh->m_oFaceData.Append(nMSIndices[2]);
                opMesh->m_oFaceData.Append(nMSIndices[1]);
            }
        }

        // Extract vertex normals.

            // Get normal count.
        iNormalCount = msMesh_GetVertexNormalCount(pMSMesh);

        if (!m_blExcludeNormals)
        {
                // Presize normal array.
            opMesh->m_oNormals.SetSize(iVertexCount);
            opMesh->m_oNormals.SetCount(iVertexCount);

                // Collect normals.
            for (iFaceIndex = 0; iFaceIndex < iFaceCount; iFaceIndex++)
            {
                    // Get MilkShape triangle.
                pMSTriangle = msMesh_GetTriangleAt(pMSMesh, iFaceIndex);

                    // Get normal indices.
                msTriangle_GetNormalIndices(pMSTriangle, nMSNormalIndices);

                    // Get face vertex indices.
                msTriangle_GetVertexIndices(pMSTriangle, nMSIndices);

                    // Get triangle normals and save them in the normal list.
                for (iFaceVertexIndex = 0; iFaceVertexIndex < iFaceVertexCount; iFaceVertexIndex++)
                {
                        // Get normal.
                    msMesh_GetVertexNormalAt(pMSMesh, nMSNormalIndices[iFaceVertexIndex], oMSNormal);

                        // Convert to left-handed.
                    oVector.x = oMSNormal[0];
                    oVector.y = oMSNormal[1];
                    oVector.z = fZFactor * oMSNormal[2];

                        // Get vertex.
                    pMSVertex = msMesh_GetVertexAt(pMSMesh, nMSIndices[iFaceVertexIndex]);

                        // Get bone index.
                    iBoneIndex = msVertex_GetBoneIndex(pMSVertex);

                        // If no bone, use default.
                    if (iBoneIndex < 0)
                        iBoneIndex = 0;

                        // Get matrix.
                    if ((opMatrix = m_opRootFrame->GetBoneMatrix(iBoneIndex)) != NULL)
                    {
                            // Transform normal to bone space.
                        D3DXVec3TransformNormal(
                            &oNormal,
                            &oVector,
                            &opMatrix->m_oInverseWorldRotationMatrix);

                            // Set normal in normal list.
                        opMesh->m_oNormals.Set(nMSIndices[iFaceVertexIndex], oNormal, false);
                    }
                    else
                        opMesh->m_oNormals.Set(nMSIndices[iFaceVertexIndex], oVector, false);
                }
            }
        }

        // Extract mesh normal face data (which same as the vertex face data
        // because of the need to transform the normals per each vertex/bone.

            // Get face count;
        iFaceCount = msMesh_GetTriangleCount(pMSMesh);

            // Presize face data array.
        opMesh->m_oFaceData.SetSize(iFaceCount * 3);

        for (iFaceIndex = 0; iFaceIndex < iFaceCount; iFaceIndex++)
        {
                // Get MilkShape triangle.
            msTriangle *pMSTriangle = msMesh_GetTriangleAt(pMSMesh, iFaceIndex);

                // Get face vertex indices.
            msTriangle_GetVertexIndices(pMSTriangle, nMSIndices);

                // Save vertex indices.
            opMesh->m_oNormalFaceData.Append(nMSIndices[0]);

            if (m_blRightHandedCoordinateSystem)
            {
                opMesh->m_oNormalFaceData.Append(nMSIndices[1]);
                opMesh->m_oNormalFaceData.Append(nMSIndices[2]);
            }
            else
            {
                opMesh->m_oNormalFaceData.Append(nMSIndices[2]);
                opMesh->m_oNormalFaceData.Append(nMSIndices[1]);
            }
        }

        // Extract texture coordinates.

            // Pre-size text vector array.
        opMesh->m_oTextureCoordinates.SetSize(iVertexCount);

            // Collect texture vectors.
        for (iVertexIndex = 0; iVertexIndex < iVertexCount; iVertexIndex++)
        {
                // Get MilkShape vertex.
            pMSVertex = msMesh_GetVertexAt(pMSMesh, iVertexIndex);

                // Get MilkShape texture vector.
            msVertex_GetTexCoords(pMSVertex, oMSTextureVector);

                // Get our texture vector.
            oTextureVector.x = oMSTextureVector[0];
            oTextureVector.y = oMSTextureVector[1];

                // Flip textures vertically if requested.
            if (m_blFlipTextureVertical)
                oTextureVector.y = 1.0f - oTextureVector.y;

                // Flip textures horizontally if requested.
            if (m_blFlipTextureHorizontal)
                oTextureVector.x = 1.0f - oTextureVector.x;

                // Save it.
            opMesh->m_oTextureCoordinates.Append(oTextureVector);
        }

        // Extract materials.

            // Get material index.
        iMaterialIndex = msMesh_GetMaterialIndex(pMSMesh);

            // If material present.
        if (iMaterialIndex >= 0)
        {
                // Get MilkShape material.
            pMSMaterial = msModel_GetMaterialAt(pMSModel, iMaterialIndex);

            if (pMSMaterial)
            {
                    // Get material name.
                msMaterial_GetName(pMSMaterial, caName, MS_MAX_NAME);

                    // Get texture name.
                msMaterial_GetDiffuseTexture(pMSMaterial, caTextureFileName, MS_MAX_PATH);

                    // Get transparency.
                float fTransparency = msMaterial_GetTransparency(pMSMaterial);

                    // Convert to valid .x names, if necessary.
				MakeNameLegal(caName, false);
				MakeNameLegal(caTextureFileName, true);

                    // Remove leading path component from texture file name.
                if ((cpBase = strrchr(caTextureFileName, '\\')) != NULL)
                    strcpy(caTextureFileName, cpBase + 1);

                    // Get material object.
                opMaterial = &opMesh->m_oMaterial;

                    // Set its name.
                opMaterial->SetName(caName);

                    // Get diffuse color.
                msMaterial_GetDiffuse(pMSMaterial, oMSColor);
                opMaterial->m_oDiffuse.r = oMSColor[0];
                opMaterial->m_oDiffuse.g = oMSColor[1];
                opMaterial->m_oDiffuse.b = oMSColor[2];
                opMaterial->m_oDiffuse.a = fTransparency;

                    // Get power.
                opMaterial->m_fPower = msMaterial_GetShininess(pMSMaterial);

                    // Get specular color.
                msMaterial_GetSpecular(pMSMaterial, oMSColor);
                opMaterial->m_oSpecular.r = oMSColor[0];
                opMaterial->m_oSpecular.g = oMSColor[1];
                opMaterial->m_oSpecular.b = oMSColor[2];
                opMaterial->m_oSpecular.a = 1.0f;

                    // Get specular color.
                msMaterial_GetEmissive(pMSMaterial, oMSColor);
                opMaterial->m_oEmissive.r = oMSColor[0];
                opMaterial->m_oEmissive.g = oMSColor[1];
                opMaterial->m_oEmissive.b = oMSColor[2];
                opMaterial->m_oEmissive.a = 1.0f;

                    // Get texture file name.
                if (caTextureFileName[0])
                    strncpy(opMaterial->m_oTextureFileName.m_caTextureFileName, caTextureFileName, kNameLen);
            }
        }

            // Save mesh in frame.
        m_opRootFrame->m_oRenderGroups.Append(opMesh);
    }

    return(blRet);
}

    // Create animations.
bool cPlugIn::CreateAnimations(msModel *pMSModel)
{
    int iBoneIndex;
    int iBoneCount;
    int iKeyIndex;
    int iKeyCount;
    int iPositionKeyCount;
    int iRotationKeyCount;
    msBone *pMSBone;
    msPositionKey *pMSPositionKey;
    msRotationKey *pMSRotationKey;
    msVec3 oMSPosition;
    char caName[kNameLen + 1];
    char caBoneName[kNameLen + 1];
    Frame *opFrame;
    BoneMatrix *opMatrix = NULL;
    Animation *opAnimation;
    AnimationKey *opPositionKey;
    AnimationKey *opRotationKey;
    AnimationKey *opMatrixKey;
    AnimationOptions *opOptions;
    D3DXVECTOR3 oVector;
    D3DXVECTOR4 oVector4;
    D3DXQUATERNION oQuaternion;
    D3DXMATRIX oKeyMatrix;
    D3DXMATRIX oTranslationMatrix;
    D3DXMATRIX oRotationMatrix;
    D3DXMATRIX oMatrix;
    int iAnimationCount = 0;
    int i, j;
    float fZFactor = (m_blRightHandedCoordinateSystem ? 1.0f : -1.0f);
    bool blRet = true;

    iBoneCount = msModel_GetBoneCount(pMSModel);

    if (iBoneCount)
    {
        for (iBoneIndex = 0; iBoneIndex < iBoneCount; iBoneIndex++)
        {
                // Get a bone.
            pMSBone = msModel_GetBoneAt(pMSModel, iBoneIndex);

                // Get position key count.
            iPositionKeyCount = CountKeys(pMSBone, false);

                // Get rotation key count.
            iRotationKeyCount = CountKeys(pMSBone, true);

                // Skip if no animation data.
            if ((iPositionKeyCount > 1) || (iRotationKeyCount > 1))
                iAnimationCount++;
        }

        if (!iAnimationCount)
            return(true);

            // Create the animation set object.
        m_opAnimationSet = new AnimationSet(m_caAnimationSetName);

            // Collection bone animations.
        for (iBoneIndex = 0; iBoneIndex < iBoneCount; iBoneIndex++)
        {
                // Get a bone.
            pMSBone = msModel_GetBoneAt(pMSModel, iBoneIndex);

                // Get position key count.
            iPositionKeyCount = CountKeys(pMSBone, false);

                // Get rotation key count.
            iRotationKeyCount = CountKeys(pMSBone, true);

                // Skip if no animation data.
            if ((iPositionKeyCount < 1) && (iRotationKeyCount < 1))
                continue;

                // Get the bone name.
            msBone_GetName(pMSBone, caBoneName, kNameLen);

                // Convert to legal .x name if necessary.
			MakeNameLegal(caBoneName, false);

                // Format animation name.
            sprintf(caName, "%sAnimation", caBoneName);

                // Create bone animation object.
            opAnimation = new Animation(caName);

            if (m_opRootFrame->BoneMatrixCount())
            {
                    // Get frame.
                opFrame = m_opRootFrame;

                    // Set frame.
                opAnimation->m_opFrame = opFrame;

                    // Get bone matrix.
                opMatrix = m_opRootFrame->GetBoneMatrix(iBoneIndex);

                    // Set bone transformation.
                opAnimation->m_opMatrix = opMatrix;
            }
            else
            {
                    // Get frame.
                opFrame = m_oBoneFrames.Indexed(iBoneIndex);

                    // Set frame.
                opAnimation->m_opFrame = opFrame;
            }

                // Append animation to animation set.
            m_opAnimationSet->m_oAnimations.Append(opAnimation);

            if (m_blMatrixKeys)
            {
                // Do matrix key animation.

                    // If any position keys.
                if (iPositionKeyCount && iRotationKeyCount)
                {
                        // Create animation key object.
                    opMatrixKey = new AnimationKey;

                        // Set key type (matrix).
                    opMatrixKey->m_iType = 4;

                        // Set value count;
                    opMatrixKey->m_iValueCount = 16;

                        // Pre-size arrays.
                    opMatrixKey->m_oTimes.SetSize(iPositionKeyCount);
                    opMatrixKey->m_oValues.SetSize(iPositionKeyCount * opMatrixKey->m_iValueCount);

                        // From old code.
                    if (!iBoneIndex)
                        memset(oMSPosition, 0, sizeof(oMSPosition));

                        // Get the MS key count;
                    iKeyCount = msBone_GetPositionKeyCount(pMSBone);

                        // Collect the matrix keys.
                    for (iKeyIndex = 0; iKeyIndex < iKeyCount; iKeyIndex++)
                    {
                            // Initialize translation and rotation matrices to identities.
                        D3DXMatrixIdentity(&oTranslationMatrix);
                        D3DXMatrixIdentity(&oRotationMatrix);

                            // Get a MilkShape position key.
                        pMSPositionKey = msBone_GetPositionKeyAt(pMSBone, iKeyIndex);

                            // Check for range.
                        if ((pMSPositionKey->fTime < m_fAnimationRangeLow) ||
                                (pMSPositionKey->fTime > m_fAnimationRangeHigh))
                            continue;

                            // Copy the position value.
                        memcpy(oMSPosition, pMSPositionKey->Position, sizeof(oMSPosition));

                            // Append time key value.
                        opMatrixKey->m_oTimes.Append(
                            (int)((pMSPositionKey->fTime - m_fAnimationRangeLow)
                                * (m_fAnimationTimeFactor ? m_fAnimationTimeFactor : 1.0f)));

                            // Get adjusted position key.
                        oVector.x = oMSPosition[0] * m_fPositionFactor;
                        oVector.y = oMSPosition[1] * m_fPositionFactor;
                        oVector.z = oMSPosition[2] * m_fPositionFactor * fZFactor;

                            // Convert the bone position to a translation matrix.
                        D3DXMatrixTranslation(
                            &oTranslationMatrix,
                            oVector.x,
                            oVector.y,
                            oVector.y);

                            // Get the MilkShape rotation key.
                        pMSRotationKey = msBone_GetRotationKeyAt (pMSBone, iKeyIndex);

                            // Convert the key rotation to a rotation matrix.
                        MatrixRotationYawPitchRoll(
                            &oRotationMatrix,
                            pMSRotationKey->Rotation[1] * m_oEulerFactor.y,
                            pMSRotationKey->Rotation[0] * m_oEulerFactor.x,
                            pMSRotationKey->Rotation[2] * m_oEulerFactor.z * fZFactor);

                            // Combine the translation and rotation matrix.
                        D3DXMatrixMultiply(
                            &oMatrix,
                            &oRotationMatrix,
                            &oTranslationMatrix);

                            // Combine with bone matrix.
                        if (opMatrix)
                            D3DXMatrixMultiply(
                                &oKeyMatrix,
                                &oMatrix,
                                &opMatrix->m_oMatrix);
                        else
                            D3DXMatrixMultiply(
                                &oKeyMatrix,
                                &oMatrix,
                                &opFrame->m_oMatrix.m_oMatrix);

                            // Append matrix key value.
                        for (j = 0; j < 4; j++)
                        {
                            for (i = 0; i < 4; i++)
                                opMatrixKey->m_oValues.Append(oKeyMatrix(j, i));
                        }
                    }

                        // Add interpolations if requested.
                    if (m_blAddInterpolations)
                        opMatrixKey->AddInterpolations(m_iInterpolationCount);

                        // Append to animation.
                    opAnimation->m_oKeys.Append(opMatrixKey);

                    if (!m_blExcludeAnimationOptions)
                    {
                            // Create animation options object.
                        opOptions = new AnimationOptions;

                            // Set option.  0 == repeat, 1 == 1-shot.
                        opOptions->m_iOpenClosed = (m_blLoopAnimations ? 0 : 1);

                            // Set option.  0 == spline, 1 == linear.
                        opOptions->m_iPositionQuality = (m_blSplinePositions ? 0 : 1);

                            // Append animation options object.
                        opAnimation->m_oOptions.Append(opOptions);
                    }
                }
            }
            else
            {
                // Do position animation.

                    // If any position keys.
                if (iPositionKeyCount)
                {
                        // Create animation key object.
                    opPositionKey = new AnimationKey;

                        // Set key type (position).
                    opPositionKey->m_iType = 2;

                        // Set value count;
                    opPositionKey->m_iValueCount = 3;

                        // Pre-size arrays.
                    opPositionKey->m_oTimes.SetSize(iPositionKeyCount);
                    opPositionKey->m_oValues.SetSize(iPositionKeyCount * opPositionKey->m_iValueCount);

                        // From old code.
                    if (!iBoneIndex)
                        memset(oMSPosition, 0, sizeof(oMSPosition));

                        // Get the MS key count;
                    iKeyCount = msBone_GetPositionKeyCount(pMSBone);

                        // Collect the matrix keys.
                    for (iKeyIndex = 0; iKeyIndex < iKeyCount; iKeyIndex++)
                    {
                            // Get a MilkShape position key.
                        pMSPositionKey = msBone_GetPositionKeyAt(pMSBone, iKeyIndex);

                            // Check for range.
                        if ((pMSPositionKey->fTime < m_fAnimationRangeLow) ||
                                (pMSPositionKey->fTime > m_fAnimationRangeHigh))
                            continue;

                            // Copy the position value.
                        memcpy(oMSPosition, pMSPositionKey->Position, sizeof(oMSPosition));

                            // Append time key value.
                        opPositionKey->m_oTimes.Append(
                            (int)((pMSPositionKey->fTime - m_fAnimationRangeLow)
                                * (m_fAnimationTimeFactor ? m_fAnimationTimeFactor : 1.0f)));

                            // Get adjusted position key.
                        oVector.x = oMSPosition[0] * m_fPositionFactor;
                        oVector.y = oMSPosition[1] * m_fPositionFactor;
                        oVector.z = oMSPosition[2] * m_fPositionFactor * fZFactor;

                            // Transform position key by frame matrix.
                        if (opMatrix)
                            D3DXVec3Transform(
                                &oVector4,
                                &oVector,
                                &opMatrix->m_oMatrix);
                        else
                            D3DXVec3Transform(
                                &oVector4,
                                &oVector,
                                &opFrame->m_oMatrix.m_oMatrix);

                            // Append position key value.
                        opPositionKey->m_oValues.Append(oVector4.x);
                        opPositionKey->m_oValues.Append(oVector4.y);
                        opPositionKey->m_oValues.Append(oVector4.z);
                    }

                        // Add interpolations if requested.
                    if (m_blAddInterpolations)
                        opPositionKey->AddInterpolations(m_iInterpolationCount);

                        // Append to animation.
                    opAnimation->m_oKeys.Append(opPositionKey);

                    if (!m_blExcludeAnimationOptions)
                    {
                            // Create animation options object.
                        opOptions = new AnimationOptions;

                            // Set option.  0 == repeat, 1 == 1-shot.
                        opOptions->m_iOpenClosed = (m_blLoopAnimations ? 0 : 1);

                            // Set option.  0 == spline, 1 == linear.
                        opOptions->m_iPositionQuality = (m_blSplinePositions ? 0 : 1);

                            // Append animation options object.
                        opAnimation->m_oOptions.Append(opOptions);
                    }
                }

                // Do rotation animation.

                    // If any keys.
                if (iRotationKeyCount)
                {
                        // Create animation key object.
                    opRotationKey = new AnimationKey;

                        // Set key type (quaternion).
                    opRotationKey->m_iType = 0;

                        // Set value count;
                    opRotationKey->m_iValueCount = 4;

                        // Pre-size arrays.
                    opRotationKey->m_oTimes.SetSize(iRotationKeyCount);
                    opRotationKey->m_oValues.SetSize(iRotationKeyCount * opRotationKey->m_iValueCount);

                        // Get the MS key count;
                    iKeyCount = msBone_GetRotationKeyCount(pMSBone);

                        // Collect the matrix keys.
                    for (iKeyIndex = 0; iKeyIndex < iKeyCount; iKeyIndex++)
                    {
                            // Get the MilkShape rotation key.
                        pMSRotationKey = msBone_GetRotationKeyAt (pMSBone, iKeyIndex);

                            // Check for range.
                        if ((pMSRotationKey->fTime < m_fAnimationRangeLow) ||
                                (pMSRotationKey->fTime > m_fAnimationRangeHigh))
                            continue;

                            // Convert the key rotation to a rotation matrix.
                        MatrixRotationYawPitchRoll(
                            &oRotationMatrix,
                            pMSRotationKey->Rotation[1] * m_oEulerFactor.y,
                            pMSRotationKey->Rotation[0] * m_oEulerFactor.x,
                            pMSRotationKey->Rotation[2] * m_oEulerFactor.z * fZFactor);

                            // Combine with bone matrix.
                        if (opMatrix)
                            D3DXMatrixMultiply(
                                &oKeyMatrix,
                                &oRotationMatrix,
                                &opMatrix->m_oRotationMatrix);
                        else
                            D3DXMatrixMultiply(
                                &oKeyMatrix,
                                &oRotationMatrix,
                                &opFrame->m_oRotationMatrix.m_oMatrix);

                            // Convert it to a quaternion.
                        D3DXQuaternionRotationMatrix(
                            &oQuaternion,
                            &oKeyMatrix);

                            // Append time key value.
                        opRotationKey->m_oTimes.Append(
                            (int)((pMSRotationKey->fTime - m_fAnimationRangeLow)
                                * (m_fAnimationTimeFactor ? m_fAnimationTimeFactor : 1.0f)));

                            // Adjust quaternion.
                        oQuaternion.x = -oQuaternion.x;
                        oQuaternion.y = -oQuaternion.y;
                        oQuaternion.z = -oQuaternion.z;

                            // Append rotation key value.
                        opRotationKey->m_oValues.Append(oQuaternion.w);
                        opRotationKey->m_oValues.Append(oQuaternion.x);
                        opRotationKey->m_oValues.Append(oQuaternion.y);
                        opRotationKey->m_oValues.Append(oQuaternion.z);

#if MSPLUGIN_TRACE
                        DisplayQuaternion("Quat", iBoneIndex, oQuaternion);

                        oVector.x = 5.0f;
                        oVector.y = 5.0f;
                        oVector.z = -5.0f;
                        DisplayTransformation(
                            "xform",
                            oVector,
                            oKeyMatrix);
#endif
                    }

                        // Add interpolations if requested.
                    if (m_blAddInterpolations)
                        opRotationKey->AddInterpolations(m_iInterpolationCount);

                        // Append to animation.
                    opAnimation->m_oKeys.Append(opRotationKey);

                    if (!m_blExcludeAnimationOptions)
                    {
                            // Create animation options object.
                        opOptions = new AnimationOptions;

                            // Set option.  0 == repeat, 1 == 1-shot.
                        opOptions->m_iOpenClosed = (m_blLoopAnimations ? 0 : 1);

                            // Set option.  0 == spline, 1 == linear.
                        opOptions->m_iPositionQuality = (m_blSplineRotations ? 0 : 1);

                            // Append animation options object.
                        opAnimation->m_oOptions.Append(opOptions);
                    }
                }
            }
        }
    }

    return(blRet);
}

    // Count the animation keys in a bone that fall within the range.
int cPlugIn::CountKeys(msBone *pMSBone, bool blRotation)
{
    int iIndex;
    int iCount;
    int iKeyCount = 0;
    msPositionKey *pMSPositionKey;
    msRotationKey *pMSRotationKey;

    if (blRotation)
    {
            // Get rotation key count.
        iCount = msBone_GetRotationKeyCount(pMSBone);

            // Loop through the keys.
        for (iIndex = 0; iIndex < iCount; iIndex++)
        {
                // Get the MilkShape rotation key.
            pMSRotationKey = msBone_GetRotationKeyAt (pMSBone, iIndex);

                // Check for a key within the range.
            if ((pMSRotationKey->fTime >= m_fAnimationRangeLow) &&
                    (pMSRotationKey->fTime <= m_fAnimationRangeHigh))
                iKeyCount++;
        }
    }
    else
    {
            // Get position key count.
        iCount = msBone_GetPositionKeyCount(pMSBone);

            // Loop through the keys.
        for (iIndex = 0; iIndex < iCount; iIndex++)
        {
                // Get the MilkShape position key.
            pMSPositionKey = msBone_GetPositionKeyAt (pMSBone, iIndex);

                // Check for a key within the range.
            if ((pMSPositionKey->fTime >= m_fAnimationRangeLow) &&
                    (pMSPositionKey->fTime <= m_fAnimationRangeHigh))
                iKeyCount++;
        }
    }

    return(iKeyCount);
}

    // Output the header.
bool cPlugIn::OutputFileHeader()
{
    bool blRet = true;

    if (m_blBinary)
    {
        if ((m_eFormat == kFormatJTGameSkinAndBones) ||
                (m_eFormat == kFormatJTGameRenderGroups))
            blRet = blRet && OutputString("xof 0400bin 0032");
        else
            blRet = blRet && OutputString("xof 0303bin 0032");
    }
    else
    {
        if ((m_eFormat == kFormatJTGameSkinAndBones) ||
                (m_eFormat == kFormatJTGameRenderGroups))
        {
            blRet = blRet && OutputString("xof 0400txt 0032");
            blRet = blRet && OutputString("\n\n");
            blRet = blRet && OutputString("// JTGame 3D Model - from MilkShape3D\n");
            blRet = blRet && OutputString("\n");
        }
        else
        {
            blRet = blRet && OutputString("xof 0303txt 0032");
            blRet = blRet && OutputString("\n\n");
            blRet = blRet && OutputString("// DirectX - from MilkShape3D\n");
            blRet = blRet && OutputString("\n");
        }
    }

    return(blRet);
}


    // Output the geometry.
bool cPlugIn::OutputGeometry()
{
    int iIndex;
    int iCount;
    Frame *opFrame;
    bool blRet;

    //m_opBodyMesh->Output();

    if (m_blMeshOnly)
        blRet = OutputMeshes(m_opRootFrame);
    else if (!m_blAddRootFrame && m_opRootFrame->ChildCount())
    {
        iCount = m_opRootFrame->ChildCount();

        for (iIndex = 0; iIndex < iCount; iIndex++)
        {
            opFrame = m_opRootFrame->Child(iIndex);
            blRet = opFrame->Output();

            if (!blRet)
                break;
        }
    }
    else
        blRet = m_opRootFrame->Output();

    return(blRet);
}

    // Output meshes
bool cPlugIn::OutputMeshes(Frame *opFrame)
{
    int iIndex;
    int iCount;
    RenderGroup *opRenderGroup;
    Mesh *opMesh;
    Frame *opChildFrame;
    bool blRet;

    // Output render groups for this frame.

    iCount = opFrame->m_oRenderGroups.Count();

    for (iIndex = 0; iIndex < iCount; iIndex++)
    {
        opRenderGroup = opFrame->m_oRenderGroups.Indexed(iIndex);
        blRet = opRenderGroup->Output();

        if (!blRet)
            break;
    }

    // Output meshes for this frame.

    iCount = opFrame->m_oMeshes.Count();

    for (iIndex = 0; iIndex < iCount; iIndex++)
    {
        opMesh = opFrame->m_oMeshes.Indexed(iIndex);
        blRet = opMesh->Output();

        if (!blRet)
            break;
    }

    // Descend into child frames.

    iCount = opFrame->ChildCount();

    for (iIndex = 0; iIndex < iCount; iIndex++)
    {
        opChildFrame = opFrame->Child(iIndex);
        blRet = OutputMeshes(opChildFrame);

        if (!blRet)
            break;
    }

    return(blRet);
}

    // Clean up after exporting.
void cPlugIn::CleanUp()
{
        // Delete body mesh.
    delete m_opBodyMesh;
    m_opBodyMesh = NULL;

        // Avoid double-deleting frames.
    m_oBoneFrames.OrphanAll();

        // Delete frames.
    delete m_opRootFrame;
    m_opRootFrame = NULL;
    m_opBodyFrame = NULL;

        // Clean danlging pointers.
    m_opAnimationSet = NULL;
}

// Utility/debug functions.

    // Display a debug message.
void cPlugIn::Debug(const char *cpMessage, ...)
{
    char caBuf[1024];

    va_list args;
    va_start(args, cpMessage);

        // Format message.
    vsprintf(caBuf, cpMessage, args);

        // Display message.
    OutputDebugString(caBuf);

    va_end(args);
}

    // Display a matrix debug message.
void cPlugIn::DisplayMatrix(const char *cpName, int iIndex, D3DXMATRIX matrix)
{
    Debug("    %s[%d]\n", cpName, iIndex);
    Debug("    {\n");
    Debug("        %f, %f, %f, %f,\n",
        matrix(0, 0), matrix(0, 1), matrix(0, 2), matrix(0, 3));
    Debug("        %f, %f, %f, %f,\n",
        matrix(1, 0), matrix(1, 1), matrix(1, 2), matrix(1, 3));
    Debug("        %f, %f, %f, %f,\n",
        matrix(2, 0), matrix(2, 1), matrix(2, 2), matrix(2, 3));
    Debug("        %f, %f, %f, %f;;\n",
        matrix(3, 0), matrix(3, 1), matrix(3, 2), matrix(3, 3));
    Debug("    }\n");
    Debug("\n");
}


    // Transform and display a vector debug message.
void cPlugIn::DisplayTransformation(const char *cpName, D3DXVECTOR3 oVector, D3DXMATRIX oMatrix)
{
    D3DXVECTOR4 oVector4;
        // Transform vector.
    D3DXVec3Transform(
        &oVector4,
        &oVector,
        &oMatrix);
    Debug("    %s:  (%f, %f, %f, %f)\n", cpName, oVector4.x, oVector4.y, oVector4.z, oVector4.w);
}

    // Display a quaternion debug message.
void cPlugIn::DisplayQuaternion(const char *cpName, int iIndex, D3DXQUATERNION quat)
{
    D3DXVECTOR3 oVector;
    float angle;
    D3DXQuaternionToAxisAngle(
        &quat,
        &oVector,
        &angle);
    D3DXVECTOR3 axis;
    D3DXVec3Normalize(
        &axis,
        &oVector);
    Debug("%s %d quat(%f,%f,%f,%f) = axis(%f,%f,%f) angle(%f)\n",
        cpName,
        iIndex,
        quat.w,
        quat.x,
        quat.y,
        quat.z,
        axis.x,
        axis.y,
        axis.z,
        angle);
}

    // Convert a name to be a legal .x file name.
void cPlugIn::MakeNameLegal(char *name, bool isFileName)
{
    if (!name || !*name)
        return;

    if ((*name != '_') && !isalpha(*name))
    {
        memmove(name + 1, name, strlen(name) + 1);
        *name = '_';
    }

	while (*name)
	{
		if (!isalnum(*name)
            && (!isFileName
                || ((*name != '.')
                    && (*name != '\\')
                    && (*name != '/')
                    && (*name != '-'))))
			*name = '_';

		name++;
	}
}

    // Convert a MilkShape euler rotation to a rotation matrix.
    // We do this to enforce MilkShape's euler combination order.
void cPlugIn::MatrixRotationYawPitchRoll(
    D3DXMATRIX *opRotationMatrix,
    float fYawY,
    float fPitchX,
    float fRollZ)
{
    D3DXMATRIX oMatrixX;
    D3DXMATRIX oMatrixY;
    D3DXMATRIX oMatrixZ;
    D3DXMATRIX oMatrix0;
    float fFactor = (m_blRightHandedCoordinateSystem ? 1.0f : -1.0f);

    D3DXMatrixRotationX(
        &oMatrixX,
        fFactor * fPitchX);

    D3DXMatrixRotationY(
        &oMatrixY,
        fFactor * fYawY);

    D3DXMatrixRotationZ(
        &oMatrixZ,
        fFactor * fRollZ);

    D3DXMatrixMultiply(
        &oMatrix0,
        &oMatrixY,
        &oMatrixZ);

    D3DXMatrixMultiply(
        opRotationMatrix,
        &oMatrixX,
        &oMatrix0);
}
