/*
    msPlugInData.h - Common model data for MilkShape DirectX plug-ins.

    Copyright (c) John Thompson, 2001.
    Permission hereby granted to you to use non-exclusively or modify as you wish,
    provided you preserve this copyright message.
*/

#ifndef __MS_PLUGIN_DATA_H__
#define __MS_PLUGIN_DATA_H__

#include "msPlugIn.h"
#include <d3dx8.h>                                  // Direct3Dx definitions

// Constants.

    // Format variations.
enum EFormat
{
    kFormatDirectXRetainedMode,
    kFormatDirectXSkinAndBones,
    kFormatJTGameSkinAndBones,
    kFormatJTGameRenderGroups
};

    // Flexible vertex format flags.
#define kFVFIndexedNoWeights 4374
#define kFVFNoWeights 274
#define kFVFOneWeight 278
#define kFVFTwoWeights 280

// Utility classes.

    // Array class (adpated from CGameClassTypeArray from JTGame).
template<class TYPE>
class Array
{
protected:
        // The array size.
    int m_iSize;
        // The current count of used objects in the array.
    int m_iCount;
        // The array.
    TYPE *m_opArray;
public:
        // Constructor for preallocating the array.
    Array(int iSize) :
        m_iSize(iSize),
        m_iCount(0),
        m_opArray(new TYPE[iSize])
        { }
        // Copy constructor.
    Array(const Array<TYPE>& oOther) :
        m_iSize(oOther.m_iSize),
        m_iCount(oOther.m_iCount),
        m_opArray(new TYPE[oOther.m_iSize])
        {
            for (int iIndex = 0; iIndex < m_iCount; iIndex++)
                m_opArray[iIndex] = oOther.m_opArray[iIndex];
        }
        // Default constructor.
    Array() :
        m_iSize(0),
        m_iCount(0),
        m_opArray(NULL)
        { }
        // Destructor.
    ~Array()
        { delete [] m_opArray;
          m_opArray = NULL; }
		// Assignment operator.
	Array<TYPE>& operator=(const Array<TYPE>& oOther)
		{
            delete [] m_opArray;
            m_iSize = oOther.m_iSize;
            m_iCount = oOther.m_iCount;
            if (oOther.m_opArray)
            {
                m_opArray = new TYPE[m_iSize];
                for (int iIndex = 0; iIndex < m_iCount; iIndex++)
                    m_opArray[iIndex] = oOther.m_opArray[iIndex];
            }
            else
                m_opArray = NULL;
		    return(*this);
        }
        // Cast operator.
    operator TYPE*()
        { return(m_opArray); }
        // Array operator.
    TYPE& operator[](int iIndex)
        { ASSERT(iIndex < m_iSize); return(m_opArray[iIndex]); }
        // Return the number of items in the array.
    int Count() const
        { return(m_iCount); }
        // Set array item count.  This does not modify any
        // array items.
    void SetCount(int iCount)
        {
            if (iCount > m_iSize)
                SetSize(iCount);
            m_iCount = iCount;
        }
        // Return array size.  This may or may not have meaning to
        // the derived class.  Use Count to get the number of items in
        // the array.
    int Size() const
        { return(m_iSize); }
        // Set array size.  This function will grow or shrink the
        // array if necessary, and this has meaning to the derived
        // class.
    void SetSize(int iSize)
        {
            if (iSize)
            {
                TYPE *opNewArray = new TYPE[iSize];
                if (m_iSize)
                {
                    int iCopyCount = iSize;
                    if (iCopyCount > m_iSize)
                        iCopyCount = m_iSize;
                    for (int iIndex = 0; iIndex < iCopyCount; iIndex++)
                        opNewArray[iIndex] = m_opArray[iIndex];
                }
                m_iSize = iSize;
                if (m_iCount > iSize)
                    m_iCount = iSize;
                delete [] m_opArray;
                m_opArray = opNewArray;
            }
            else
            {
                delete [] m_opArray;
                m_opArray = NULL;
                m_iSize = m_iCount = 0;
            }
        }
        // Get first item.  Returns NULL if no item.
    TYPE First()
        { return(m_opArray[0]); }
        // Get last item.  Returns NULL if no item.
    TYPE Last()
        { return(m_iCount ? m_opArray[m_iCount - 1] : m_opArray[0]); }
        // Get indexed item.  Returns NULL if invalid index or no item.
    TYPE& Indexed(int iIndex)
        { return(iIndex < m_iSize ? m_opArray[iIndex] : m_opArray[0]); }
        // Set indexed item. If blExpand is TRUE, move items up, otherwise prior item is
        // replaced.
    void Set(int iIndex, const TYPE oItem, BOOL blExpand = FALSE)
        {
            int i;
            if ((iIndex >= m_iSize) || (blExpand && (m_iCount == m_iSize)))
            {
                int iNewSize = m_iSize * 2;
                if (iNewSize <= iIndex)
                    iNewSize = iIndex + 1;
                TYPE *opNewArray = new TYPE[iNewSize];
                opNewArray[iIndex] = oItem;
                if (m_opArray)
                {
                    if (iIndex)
                    {
                        for (i = 0; (i < iIndex) && (i < m_iCount); i++)
                            opNewArray[i] = m_opArray[i];
                    }
                    if (iIndex <= m_iCount - 1)
                    {
                        int iSourceIndex = iIndex;
                        int iRemCount = m_iSize - iIndex;
                        if (!blExpand)
                        {
                            iSourceIndex++;
                            iRemCount--;
                        }
                        for (i = iIndex; iSourceIndex < m_iSize; i++, iSourceIndex++)
                            opNewArray[i] = m_opArray[iSourceIndex];
                    }
                }
                if (blExpand || (iIndex == m_iCount))
                    m_iCount++;
                delete [] m_opArray;
                m_opArray = opNewArray;
                m_iSize = iNewSize;
            }
            else if (blExpand)
            {
                if (iIndex < m_iCount - 1)
                {
                    for (i = m_iSize; i > iIndex; i--)
                        m_opArray[i] = m_opArray[i - 1];
                }
                m_opArray[iIndex] = oItem;
                m_iCount++;
            }
            else
            {
                m_opArray[iIndex] = oItem;
                if (iIndex == m_iCount)
                    m_iCount++;
            }
        }
        // Prepend to array, moving items up. 
    void Prepend(const TYPE oItem)
        { Set(0, oItem, TRUE); }
        // Append to array using count as the index. 
    void Append(const TYPE oItem)
        { Set(m_iCount, oItem, TRUE); }
        // Remove indexed item.  Moves higher items down.
    void Delete(int iIndex)
        {
            if (iIndex < m_iCount - 1)
            {
                m_iCount--;
                for (; iIndex < m_iCount; iIndex++)
                    m_opArray[iIndex] = m_opArray[iIndex + 1];
            }
            else if (iIndex == m_iCount - 1)
                m_iCount--;
        }
        // Delete all array items.
    void DeleteAll()
        {
            delete [] m_opArray;
            m_opArray = NULL;
            m_iSize = m_iCount = 0;
        }
};

    // Array class (adpated from CGameClassTypeArray from JTGame).
template<class TYPE>
class PointerArray
{
protected:
        // The array size.
    int m_iSize;
        // The current count of used objects in the array.
    int m_iCount;
        // The array.
    TYPE **m_opArray;
public:
        // Constructor for preallocating the array.
    PointerArray(int iSize) :
        m_iSize(iSize),
        m_iCount(0),
        m_opArray(new TYPE *[iSize])
        { }
        // Copy constructor.
    PointerArray(const PointerArray<TYPE>& oOther) :
        m_iSize(oOther.m_iSize),
        m_iCount(oOther.m_iCount),
        m_opArray(new TYPE *[oOther.m_iSize])
        {
            for (int iIndex = 0; iIndex < m_iCount; iIndex++)
                m_opArray[iIndex] = oOther.m_opArray[iIndex];
        }
        // Default constructor.
    PointerArray() :
        m_iSize(0),
        m_iCount(0),
        m_opArray(NULL)
        { }
        // Destructor.
    ~PointerArray()
        { DeleteAll(); }
		// Assignment operator.
	PointerArray<TYPE>& operator=(const PointerArray<TYPE>& oOther)
		{
            DeleteAll();
            m_iSize = oOther.m_iSize;
            m_iCount = oOther.m_iCount;
            if (oOther.m_opArray)
            {
                m_opArray = new TYPE *[m_iSize];
                for (int iIndex = 0; iIndex < m_iCount; iIndex++)
                    m_opArray[iIndex] = oOther.m_opArray[iIndex];
            }
            else
                m_opArray = NULL;
		    return(*this);
        }
        // Array operator.
    TYPE *operator[](int iIndex)
        { ASSERT(iIndex < m_iSize); return(m_opArray[iIndex]); }
        // Return the number of items in the array.
    int Count() const
        { return(m_iCount); }
        // Set array item count.  This does not modify any
        // array items.
    void SetCount(int iCount)
        {
            if (iCount > m_iSize)
                SetSize(iCount);
            m_iCount = iCount;
        }
        // Return array size.  This may or may not have meaning to
        // the derived class.  Use Count to get the number of items in
        // the array.
    int Size() const
        { return(m_iSize); }
        // Set array size.  This function will grow or shrink the
        // array if necessary, and this has meaning to the derived
        // class.
    void SetSize(int iSize)
        {
            if (iSize)
            {
                TYPE **opNewArray = new TYPE *[iSize];
                int iIndex;
                if (m_iSize)
                {
                    int iCopyCount = iSize;
                    if (iCopyCount > m_iSize)
                        iCopyCount = m_iSize;
                    for (iIndex = 0; iIndex < iCopyCount; iIndex++)
                        opNewArray[iIndex] = m_opArray[iIndex];
                }
                for (iIndex = m_iCount; iIndex < m_iSize; iIndex++)
                    delete m_opArray[iIndex];
                m_iSize = iSize;
                if (m_iCount > iSize)
                {
                    m_iCount = iSize;
                }
                delete [] m_opArray;
                m_opArray = opNewArray;
                for (iIndex = m_iCount; iIndex < m_iSize; iIndex++)
                    m_opArray[iIndex] = NULL;
            }
            else
                DeleteAll();
        }
        // Get first item.  Returns NULL if no item.
    TYPE *First()
        { return(m_opArray[0]); }
        // Get last item.  Returns NULL if no item.
    TYPE *Last()
        { return(m_iCount ? m_opArray[m_iCount - 1] : NULL); }
        // Get indexed item.  Returns NULL if invalid index or no item.
    TYPE *Indexed(int iIndex)
        { return(iIndex < m_iSize ? m_opArray[iIndex] : NULL); }
        // Set indexed item. If blExpand is TRUE, move items up, otherwise prior item is
        // replaced.
    void Set(int iIndex, TYPE *opItem, BOOL blExpand = FALSE)
        {
            int i;

            if (iIndex < m_iCount)
            {
                delete m_opArray[iIndex];
                m_opArray[iIndex] = NULL;
            }

            if ((iIndex >= m_iSize) || (blExpand && (m_iCount == m_iSize)))
            {
                int iNewSize = m_iSize * 2;
                if (iNewSize <= iIndex)
                    iNewSize = iIndex + 1;
                TYPE **opNewArray = new TYPE *[iNewSize];
                opNewArray[iIndex] = opItem;
                if (m_opArray)
                {
                    if (iIndex)
                    {
                        for (i = 0; (i < iIndex) && (i < m_iCount); i++)
                            opNewArray[i] = m_opArray[i];
                    }
                    if (iIndex <= m_iCount - 1)
                    {
                        int iSourceIndex = iIndex;
                        int iRemCount = m_iSize - iIndex;
                        if (!blExpand)
                        {
                            iSourceIndex++;
                            iRemCount--;
                        }
                        for (i = iIndex; iSourceIndex < m_iSize; i++, iSourceIndex++)
                            opNewArray[i] = m_opArray[iSourceIndex];
                    }
                }
                if (blExpand || (iIndex == m_iCount))
                    m_iCount++;
                delete [] m_opArray;
                m_opArray = opNewArray;
                m_iSize = iNewSize;
            }
            else if (blExpand)
            {
                if (iIndex < m_iCount - 1)
                {
                    for (i = m_iSize; i > iIndex; i--)
                        m_opArray[i] = m_opArray[i - 1];
                }
                m_opArray[iIndex] = opItem;
                m_iCount++;
            }
            else
            {
                m_opArray[iIndex] = opItem;
                if (iIndex == m_iCount)
                    m_iCount++;
            }
        }
        // Prepend to array, moving items up. 
    void Prepend(TYPE *opItem)
        { Set(0, opItem, TRUE); }
        // Append to array using count as the index. 
    void Append(TYPE *opItem)
        { Set(m_iCount, opItem, TRUE); }
        // Remove indexed item.  Moves higher items down.
    void Delete(int iIndex)
        {
            if (iIndex < m_iCount)
            {
                delete m_opArray[iIndex];
                m_opArray[iIndex] = NULL;
            }
            if (iIndex < m_iCount - 1)
            {
                m_iCount--;
                for (; iIndex < m_iCount; iIndex++)
                    m_opArray[iIndex] = m_opArray[iIndex + 1];
            }
            else if (iIndex == m_iCount - 1)
                m_iCount--;
        }
        // Orphan indexed item.  Moves higher items down.
    TYPE *Orphan(int iIndex)
        {
            TYPE *opItem = NULL;
            if (iIndex < m_iCount)
            {
                opItem = m_opArray[iIndex];
                m_opArray[iIndex] = NULL;
            }
            if (iIndex < m_iCount - 1)
            {
                m_iCount--;
                for (; iIndex < m_iCount; iIndex++)
                    m_opArray[iIndex] = m_opArray[iIndex + 1];
            }
            else if (iIndex == m_iCount - 1)
                m_iCount--;
            return(opItem);
        }
        // Delete all array items.
    void DeleteAll()
        {
            int iIndex;
            for (iIndex = 0; iIndex < m_iCount; iIndex++)
                delete m_opArray[iIndex];
            delete [] m_opArray;
            m_opArray = NULL;
            m_iSize = m_iCount = 0;
        }
        // Orphan all array items.
    void OrphanAll()
        {
            delete [] m_opArray;
            m_opArray = NULL;
            m_iSize = m_iCount = 0;
        }
        // Find an item  (assuming they derive from Node).
    TYPE *Find(const char *cpName)
        {
            int iIndex;
            TYPE *opItem;
            for (iIndex = 0; iIndex < m_iCount; iIndex++)
            {
                opItem = m_opArray[iIndex];

                if (opItem->Match(cpName))
                    return(opItem);
            }
            return(NULL);
        }
        // Output the items in the array (assuming they derive from Node).
    bool Output()
        {
            int iIndex;
            bool blRet = true;
            for (iIndex = 0; iIndex < m_iCount; iIndex++)
            {
                if (iIndex)
                    blRet = blRet && m_opArray[iIndex]->OutputNewLine();

                blRet = blRet && m_opArray[iIndex]->Output();
            }
            return(blRet);
        }
};

    // Integer array.
typedef Array<int> IntegerArray;

    // Float array.
typedef Array<float> FloatArray;

    // Vector array.
typedef Array<D3DXVECTOR3> VectorArray;

    // Vector4 array.
typedef Array<D3DXVECTOR4> Vector4Array;

    // Quaternion array.
typedef Array<D3DXQUATERNION> QuaternionArray;

    // Texture vector array.
typedef Array<D3DXVECTOR2> TextureVectorArray;

    // Arbitrary name size limit.
#define kNameLen 256

class ClassID;

    // .x file binary token values.
#define TOKEN_NAME 1
#define TOKEN_STRING 2
#define TOKEN_INTEGER 3
#define TOKEN_GUID 5
#define TOKEN_INTEGER_LIST 6
#define TOKEN_FLOAT_LIST 7
#define TOKEN_OBRACE 10
#define TOKEN_CBRACE 11
#define TOKEN_OPAREN 12
#define TOKEN_CPAREN 13
#define TOKEN_OBRACKET 14
#define TOKEN_CBRACKET 15
#define TOKEN_OANGLE 16
#define TOKEN_CANGLE 17
#define TOKEN_DOT 18
#define TOKEN_COMMA 19
#define TOKEN_SEMICOLON 20
#define TOKEN_TEMPLATE 31
#define TOKEN_WORD 40
#define TOKEN_DWORD 41
#define TOKEN_FLOAT 42
#define TOKEN_DOUBLE 43
#define TOKEN_CHAR 44
#define TOKEN_UCHAR 45
#define TOKEN_SWORD 46
#define TOKEN_SDWORD 47
#define TOKEN_VOID 48
#define TOKEN_LPSTR 49
#define TOKEN_UNICODE 50
#define TOKEN_CSTRING 51
#define TOKEN_ARRAY 52

    // Common base class.
class Node
{
public:
    char m_caName[kNameLen];
    char m_caType[kNameLen];
    static EFormat m_eFormat;
    static bool m_blBinary;
    static int m_iIndent;
    static FILE *m_pFilePointer;
    static int m_iCurrentToken;
    static IntegerArray m_oIntegerList;
    static FloatArray m_oFloatList;
    Node(const char *cpName, const char *cpType)
        {
            SetName(cpName);
            SetType(cpType);
        }
    Node()
        {
            SetName("");
            SetType("");
        }
    Node(const Node& oOther)
        {
            SetName(oOther.m_caName);
            SetType(oOther.m_caType);
        }
    const char *Name()
        { return(m_caName); }
    void SetName(const char *cpName)
        { strncpy(m_caName, cpName, sizeof(m_caName) - 1); }
    bool Match(const char *cpName)
        { return(strcmp(cpName, m_caName) == 0 ? true : false); }
    const char *Type()
        { return(m_caType); }
    void SetType(const char *cpType)
        { strncpy(m_caType, cpType, sizeof(m_caType) - 1); }
    bool IsDirectXRetainedMode()
        { return(m_eFormat == kFormatDirectXRetainedMode ? true : false); }
    bool IsDirectXSkinAndBones()
        { return(m_eFormat == kFormatDirectXSkinAndBones ? true : false); }
    bool IsJTGameSkinAndBones()
        { return(m_eFormat == kFormatJTGameSkinAndBones ? true : false); }
    bool IsJTGameRenderGroups()
        { return(m_eFormat == kFormatJTGameRenderGroups ? true : false); }
    const char *Indent();
    virtual bool Output();
    virtual bool OutputHeader(const char *cpType, const char *cpName = NULL);
    virtual bool OutputBody();
    virtual bool OutputTail();
    virtual bool OutputTemplateHeader(const char *cpType, ClassID& oClassID);
    virtual bool OutputTemplateTail();
    virtual bool OutputByte(int iValue);
    virtual bool OutputWord(int iValue);
    virtual bool OutputDWord(int iValue);
    virtual bool OutputFloat(float fValue);
    virtual bool OutputString(const char *cpFormat, ...);
    virtual bool OutputSeparator(const char *cpSeparator);
    virtual bool OutputNewLine();
    virtual bool OutputIndent();
    virtual bool OutputObjectOpenBrace();
    virtual bool OutputObjectCloseBrace();
    virtual bool OutputToken(int iToken);
    virtual bool OutputSize(int iSize);
    virtual bool OutputNameToken(const char *cpName, const char *cpSeparator = "");
    virtual bool OutputStringToken(const char *cpString, const char *cpSeparator = "");
    virtual bool OutputIntegerToken(int iValue, const char *cpSeparator = "");
    virtual bool OutputEmptyIntegerListToken();
    virtual bool OutputIntegerListToken(int iValue, const char *cpSeparator = "");
    virtual bool OutputFloatListToken(float fValue, const char *cpSeparator = "");
    virtual bool OutputEmptyFloatListToken();
    virtual bool ListEndCheck(int iToken);
    virtual bool OutputIntegerMember(int iValue);
    virtual bool OutputFloatMember(float fValue);
    virtual bool OutputStringMember(const char *cpString);
    virtual bool OutputVector2(D3DXVECTOR2& oVector, const char *cpSeparator);
    virtual bool OutputVector3(D3DXVECTOR3& oVector, const char *cpSeparator);
    virtual bool OutputFloatList(
        float *fpList,
        int iCount,
        const char *cpSeparator,
        const char *cpEndSeparator);
    virtual bool OutputMatrix(D3DXMATRIX *opMatrix, const char *cpSeparator = "");
    virtual bool OutputReference(const char *cpName);
    virtual bool OutputTemplateMember(const char *cpName, int iTypeToken);
    virtual bool OutputTemplateMember(const char *cpName, const char *cpType);
    virtual bool OutputTemplateArrayMember(
        const char *cpName, int iTypeToken, const char *cpSubscript, int iSubscript);
    virtual bool OutputTemplateArrayMember(
        const char *cpName, const char *cpType, const char *cpSubscript, int iSubscript);
    virtual bool OutputTemplateRestriction(const char *cpType);
    void Debug(const char *cpMessage, ...);
};

    // Matrix.
class Matrix : public Node
{
public:
    D3DXMATRIX m_oMatrix;
    Matrix() : Node("", "FrameTransformMatrix")
        { D3DXMatrixIdentity(&m_oMatrix); }
    ~Matrix()
        {}
    virtual bool OutputBody();
};

    // Matrix array.
typedef PointerArray<Matrix> MatrixArray;

    // Bone matrix.
class BoneMatrix : public Node
{
public:
    D3DXMATRIX m_oMatrix;
    D3DXMATRIX m_oInverseMatrix;
    D3DXMATRIX m_oRotationMatrix;
    D3DXMATRIX m_oWorldMatrix;
    D3DXMATRIX m_oWorldRotationMatrix;
    D3DXMATRIX m_oInverseWorldMatrix;
    D3DXMATRIX m_oInverseRotationMatrix;
    D3DXMATRIX m_oInverseWorldRotationMatrix;
    D3DXMATRIX m_oOutputMatrix;
    BoneMatrix *m_opParent;
    BoneMatrix(const char *cpName) :
        Node(
            cpName,
            "BoneMatrix"),
        m_opParent(NULL)
        {
            D3DXMatrixIdentity(&m_oMatrix);
            D3DXMatrixIdentity(&m_oInverseMatrix);
            D3DXMatrixIdentity(&m_oRotationMatrix);
            D3DXMatrixIdentity(&m_oWorldMatrix);
            D3DXMatrixIdentity(&m_oWorldRotationMatrix);
            D3DXMatrixIdentity(&m_oInverseWorldMatrix);
            D3DXMatrixIdentity(&m_oInverseRotationMatrix);
            D3DXMatrixIdentity(&m_oInverseWorldRotationMatrix);
        }
    BoneMatrix() :
        Node(
            "",
            "BoneMatrix"),
        m_opParent(NULL)
        {
            D3DXMatrixIdentity(&m_oMatrix);
            D3DXMatrixIdentity(&m_oInverseMatrix);
            D3DXMatrixIdentity(&m_oRotationMatrix);
            D3DXMatrixIdentity(&m_oWorldMatrix);
            D3DXMatrixIdentity(&m_oWorldRotationMatrix);
            D3DXMatrixIdentity(&m_oInverseWorldMatrix);
            D3DXMatrixIdentity(&m_oInverseRotationMatrix);
            D3DXMatrixIdentity(&m_oInverseWorldRotationMatrix);
        }
    ~BoneMatrix()
        {}
    virtual bool OutputBody();
};

    // BoneMatrix array.
typedef PointerArray<BoneMatrix> BoneMatrixArray;

class Frame;

    // Skin weight data.
class SkinWeightData : public Node
{
public:
        // Bone number.
    int m_iBoneNumber;
        // Frame.
    Frame *m_opFrame;
        // Vertex indices.
    IntegerArray m_oIndices;
        // Vertex weights.
    FloatArray m_oWeights;
        // Matrix to localize vertices to bone.
    Matrix m_oMatrix;
    SkinWeightData(Frame *opFrame) :
        Node("", "SkinWeights"),
        m_opFrame(opFrame)
        {}
    SkinWeightData() :
        Node("", "SkinWeights"),
        m_opFrame(NULL)
        {}
    ~SkinWeightData()
        {}
    virtual bool OutputBody();
};

    // Skin weight data array.
typedef PointerArray<SkinWeightData> SkinWeightDataArray;

    // TextureFileName.
class TextureFileName : public Node
{
public:
    char m_caTextureFileName[kNameLen];
    TextureFileName() : Node("", "TextureFileName")
        { m_caTextureFileName[0] = '\0'; }
    TextureFileName(const TextureFileName& oOther) : Node(oOther)
    {
        strcpy(m_caTextureFileName, oOther.m_caTextureFileName);
    }
    ~TextureFileName()
        {}
    virtual bool OutputBody();
};

    // Material.
class Material : public Node
{
public:
    D3DXCOLOR m_oDiffuse;
    D3DXCOLOR m_oSpecular;
    D3DXCOLOR m_oEmissive;
    float m_fPower;
    TextureFileName m_oTextureFileName;
    Material(const char *cpName) :
        Node(cpName, "Material"),
        m_fPower(0.0f)
        {}
    Material() :
        Node("", "Material"),
        m_fPower(0.0f)
        {}
    Material(const Material& oOther) :
        Node(oOther),
        m_oDiffuse(oOther.m_oDiffuse),
        m_oSpecular(oOther.m_oSpecular),
        m_oEmissive(oOther.m_oEmissive),
        m_fPower(oOther.m_fPower),
        m_oTextureFileName(oOther.m_oTextureFileName)
        {}
    ~Material()
        {}
    bool IsNull() const
    {
        return(
            ((m_oDiffuse.r + m_oDiffuse.g + m_oDiffuse.b + m_oDiffuse.a +
                m_oSpecular.r + m_oSpecular.g + m_oSpecular.b +
                m_oEmissive.r + m_oEmissive.g + m_oEmissive.b +
                m_fPower) == 0.0f) && !m_oTextureFileName.m_caTextureFileName[0]);
    }
    virtual bool OutputBody();
};

    // Material array.
typedef PointerArray<Material> MaterialArray;

    // RenderGroup.
class RenderGroup : public Node
{
public:
    int m_iFVFFlags;
    VectorArray m_oPositions;
    IntegerArray m_oFaceData;
    IntegerArray m_oMatrixIndices;
    VectorArray m_oNormals;
    IntegerArray m_oNormalFaceData;
    TextureVectorArray m_oTextureCoordinates;
    Material m_oMaterial;
    RenderGroup(const char *cpName) :
        Node(cpName, "RenderGroup"),
        m_iFVFFlags(kFVFIndexedNoWeights)
        {}
    RenderGroup() :
        Node("", "RenderGroup"),
        m_iFVFFlags(kFVFIndexedNoWeights)
        {}
    ~RenderGroup()
        {}
    virtual bool OutputBody();
};

    // RenderGroup array.
typedef PointerArray<RenderGroup> RenderGroupArray;

    // Mesh.
class Mesh : public Node
{
public:
    int m_iMeshFlags;
    int m_iFVFFlags;
    Frame *m_opFrame;
    VectorArray m_oPositions;
    IntegerArray m_oFaceData;
    VectorArray m_oNormals;
    IntegerArray m_oNormalFaceData;
    TextureVectorArray m_oTextureCoordinates;
    IntegerArray m_oMaterialFaceIndices;
    MaterialArray m_oMaterials;
    IntegerArray m_oBoneIndices;
    SkinWeightDataArray m_oSkinWeightData;
    IntegerArray m_oVertexOriginalIndices;
    IntegerArray m_oVertexDuplicateIndices;
    Mesh(const char *cpName) :
        Node(cpName, "Mesh"),
        m_iMeshFlags(0),
        m_iFVFFlags(0),
        m_opFrame(NULL)
        {}
    Mesh() :
        Node("", "Mesh"),
        m_iMeshFlags(0),
        m_iFVFFlags(0),
        m_opFrame(NULL)
        {}
    ~Mesh()
        {}
    virtual bool OutputBody();
    void TransformPositions(D3DXMATRIX& oMatrix);
    void TransformNormals(D3DXMATRIX& oMatrix);
    void CompactNormals();
    void CloseHoles();
    int Triangulate(
        IntegerArray& oDestFaceData,        // Output triangle index tuples.
        int *ipSourceFaceData,
        int iFaceVertexCount,
        D3DXVECTOR3 *opaVertices);
};

    // Mesh array.
typedef PointerArray<Mesh> MeshArray;

    // Edge structure.
struct Edge
{
    int m_iFaceIndex;
    int m_iVertexIndex0;
    int m_iVertexIndex1;
    D3DXVECTOR3 m_oVertexPosition0;
    D3DXVECTOR3 m_oVertexPosition1;
    int m_iNormalIndex0;
    int m_iNormalIndex1;
    int m_iMaterialIndex;
    int m_iTouchedFlags;
};

    // Edge array.
typedef Array<Edge> EdgeArray;

    // Frame array.
typedef PointerArray<Frame> FrameArray;

    // Frame.
class Frame : public Node
{
public:
    Matrix m_oMatrix;
    Matrix m_oInverseMatrix;
    Matrix m_oWorldMatrix;
    Matrix m_oRotationMatrix;
    Matrix m_oWorldRotationMatrix;
    Matrix m_oInverseWorldMatrix;
    Matrix m_oInverseRotationMatrix;
    Matrix m_oInverseWorldRotationMatrix;
    Matrix m_oOutputMatrix;
    BoneMatrixArray m_oBoneMatrices;
    FrameArray m_oChildren;
    RenderGroupArray m_oRenderGroups;
    MeshArray m_oMeshes;
    Frame *m_opParent;
    int m_iLevel;
    int m_iBoneIndex;
    Frame(const char *cpName) :
        Node(cpName, "Frame"),
        m_opParent(NULL),
        m_iLevel(0),
        m_iBoneIndex(0)
        {}
    Frame() :
        Node("", "Frame"),
        m_opParent(NULL),
        m_iLevel(0),
        m_iBoneIndex(0)
        {}
    ~Frame()
        {}
    int BoneMatrixCount()
        { return(m_oBoneMatrices.Count()); }
    BoneMatrix *GetBoneMatrix(int iIndex)
        { return(m_oBoneMatrices.Indexed(iIndex)); }
    void AdoptBoneMatrix(BoneMatrix *opMatrix)
        { m_oBoneMatrices.Append(opMatrix); }
    BoneMatrix *FindBoneMatrix(const char *cpName)
        { return(m_oBoneMatrices.Find(cpName)); }
    int ChildCount()
        { return(m_oChildren.Count()); }
    Frame *Child(int iIndex)
        { return(m_oChildren.Indexed(iIndex)); }
    void AdoptChild(Frame *opFrame)
        {
            m_oChildren.Append(opFrame);
            opFrame->m_opParent = this;
            opFrame->m_iLevel = m_iLevel + 1;
        }
    Frame *FindChild(const char *cpName)
        { return(m_oChildren.Find(cpName)); }
    Frame *FindDescendent(const char *cpName);
    Frame *FindCommonParent(Frame *opFrame);
    int MeshCount()
        { return(m_oMeshes.Count()); }
    Mesh *GetMesh(int iIndex)
        { return(m_oMeshes.Indexed(iIndex)); }
    void AdoptMesh(Mesh *opMesh)
        {
            m_oMeshes.Append(opMesh);
            opMesh->m_opFrame = this;
        }
    virtual bool OutputBody();
};

    // AnimationKey.
class AnimationKey : public Node
{
public:
    int m_iType;
    int m_iValueCount;
    IntegerArray m_oTimes;
    FloatArray m_oValues;
    AnimationKey() :
        Node("", "AnimationKey")
        {}
    ~AnimationKey()
        {}
    void AddInterpolations(int iCount);
    void MoveKey(int iDstIndex, int iSrcIndex);
    void InterpolateKey(int iBaseIndex0, int iIndex, int iBaseIndex1);
    virtual bool OutputBody();
};

    // AnimationKey array.
typedef PointerArray<AnimationKey> AnimationKeyArray;

    // AnimationOptions.
class AnimationOptions : public Node
{
public:
    int m_iOpenClosed;
    int m_iPositionQuality;
    AnimationOptions() :
        Node("", "AnimationOptions")
        {}
    ~AnimationOptions()
        {}
    virtual bool OutputBody();
};

    // AnimationOptions array.
typedef PointerArray<AnimationOptions> AnimationOptionsArray;

    // Animation.
class Animation : public Node
{
public:
    Frame *m_opFrame;
    BoneMatrix *m_opMatrix;
    AnimationKeyArray m_oKeys;
    AnimationOptionsArray m_oOptions;
    Animation(const char *cpName) :
        Node(cpName, "Animation"),
        m_opFrame(NULL),
        m_opMatrix(NULL)
        {}
    Animation() :
        Node("", "Animation"),
        m_opFrame(NULL),
        m_opMatrix(NULL)
        {}
    ~Animation()
        {}
    virtual bool OutputBody();
};

    // Animation array.
typedef PointerArray<Animation> AnimationArray;

    // AnimationSet.
class AnimationSet : public Node
{
public:
    AnimationArray m_oAnimations;
    AnimationSet(const char *cpName) :
        Node(cpName, "AnimationSet")
        {}
    AnimationSet() :
        Node("", "AnimationSet")
        {}
    ~AnimationSet()
        {}
    virtual bool OutputBody();
};

    // AnimationSet array.
typedef PointerArray<AnimationSet> AnimationSetArray;

    // The size of a GUID string in repository form, without the terminating '\0', i.e.
    // "{90D7FAE0-C724-11d3-8638-0000E856BADA}".
#define kClassIDStringLength           38

    // GUID class.
class ClassID : public Node
{
protected:
    GUID m_oGUID;
    char m_caGUID[kClassIDStringLength+1];
public:

        // Constructor for a GUID as a string,
        // i.e. "{90D7FAE0-C724-11d3-8638-0000E856BADA}".
    ClassID(const char *cpGUIDString);
        // Constructor for a raw GUID structure.
    ClassID(REFIID oGUID);
        // Constructor for a raw GUID members.
    ClassID(
        DWORD dwData1,
        WORD wData2, WORD wData3,
        BYTE bData40, BYTE bData41, BYTE bData42, BYTE bData43,
        BYTE bData44, BYTE bData45, BYTE bData46, BYTE bData47);
        // Copy constructor.
    ClassID(const ClassID& oOther);
        // Default constructor.
    ClassID();
        // Destructor.
    virtual ~ClassID();
        // Assignment operator.
    virtual ClassID& operator=(const ClassID& oOther);
        // Assignment operator.
    virtual ClassID& operator=(const GUID& oGUID);
        // Assignment operator.
    virtual ClassID& operator=(const char *cpGUIDString);
        // Equality operator.
    BOOL operator==(const ClassID& oOther) const
        { return(memcmp(&m_oGUID, &oOther.m_oGUID, sizeof(m_oGUID)) ? FALSE : TRUE); }
        // Equality operator.
    BOOL operator==(const GUID& oGUID) const
        { return(memcmp(&m_oGUID, &oGUID, sizeof(m_oGUID)) ? FALSE : TRUE); }
        // Inequality operator.
    BOOL operator!=(const ClassID& oOther) const
        { return(memcmp(&m_oGUID, &oOther.m_oGUID, sizeof(m_oGUID)) ? TRUE : FALSE); }
        // Inequality operator.
    BOOL operator!=(const GUID& oGUID) const
        { return(memcmp(&m_oGUID, &oGUID, sizeof(m_oGUID)) ? TRUE : FALSE); }
        // Cast operator.
    operator GUID()
        { return(m_oGUID); }
        // Cast operator.
    operator const GUID() const
        { return(m_oGUID); }
        // Cast operator.
    operator GUID*()
        { return(&m_oGUID); }
        // Cast operator.
    operator const GUID*() const
        { return(&m_oGUID); }
        // Cast operator.
    operator char*()
        { return(m_caGUID); }
        // Cast operator.
    operator const char*() const
        { return(m_caGUID); }

    // Name functions

        // This function returns the GUID as a string.
    virtual const char *Name() const;
        // This function sets the GUID from a string.
    virtual void SetName(const char *cpName);

    // GUID functions

        // This function returns the GUID as a non-const reference.
    GUID& ID()
        { return(m_oGUID); }
        // This function returns the GUID as a const reference.
    const GUID& ID() const
        { return(m_oGUID); }
        // This function returns the GUID as a non-const pointer.
    GUID *IDPtr()
        { return(&m_oGUID); }
        // This function returns the GUID as a const pointer.
    const GUID *IDPtr() const
        { return(&m_oGUID); }
        // This function sets the GUID from a reference.
    void SetID(const GUID& oGUID);
        // This function sets the GUID from a pointer.
    void SetID(const GUID *pGUID);
        // This function sets the GUID from the raw members.
    void SetID(
        DWORD dwData1,
        WORD wData2, WORD wData3,
        BYTE bData40, BYTE bData41, BYTE bData42, BYTE bData43,
        BYTE bData44, BYTE bData45, BYTE bData46, BYTE bData47);
        // This function returns TRUE if the GUID is NULL.
    BOOL IsNull() const
        { return(!m_oGUID.Data1 && !m_oGUID.Data2 && !m_oGUID.Data3
            && !*(DWORD *)&m_oGUID.Data4[0] && !*(DWORD *)&m_oGUID.Data4[4]); }

    // Object event functions.

        // Clear the GUID.
    virtual void OnClear();
        // Output the GUID.
    virtual bool Output();
};

#endif // __MS_PLUGIN_DATA_H__