/*
    msPlugInData.cpp - Common model data for MilkShape DirectX plug-ins.

    Copyright (c) John Thompson, 2001-2004.  All rights reserved.
*/

#include "stdafx.h"
#include "msPlugInImpl.h"

    // Format.
EFormat Node::m_eFormat;

    // Binary flag.
bool Node::m_blBinary;

    // Indentation count.
int Node::m_iIndent;

    // File pointer.
FILE *Node::m_pFilePointer;

    // Current record token.
int Node::m_iCurrentToken;

    // Integer list collector.
IntegerArray Node::m_oIntegerList;

    // Float list collector.
FloatArray Node::m_oFloatList;

    // Return indentation string.
const char *Node::Indent()
{
    static char caIndent[256];
    int i;
    char *cp = caIndent;

    for (i = 0; i < m_iIndent; i++)
    {
        *cp++ = ' ';
        *cp++ = ' ';
        *cp++ = ' ';
        *cp++ = ' ';
    }

    *cp = '\0';

    return(caIndent);
}

    // Output object.
bool Node::Output()
{
    bool blRet = true;

    blRet = blRet && OutputHeader(m_caType, m_caName);
    blRet = blRet && OutputBody();
    blRet = blRet && OutputTail();

    return(blRet);
}

    // Output object header.
bool Node::OutputHeader(const char *cpType, const char *cpName)
{
    bool blRet = true;

    blRet = blRet && OutputIndent();

    blRet = blRet && OutputNameToken(cpType);

    if (cpName && *cpName)
    {
        blRet = blRet && OutputSeparator(" ");
        blRet = blRet && OutputNameToken(cpName);
    }

    blRet = blRet && OutputNewLine();

    blRet = blRet && OutputIndent();

    blRet = blRet && OutputObjectOpenBrace();

    blRet = blRet && OutputNewLine();

    m_iIndent++;

    return(blRet);
}

    // Output object body.
bool Node::OutputBody()
{
    return(true);
}

    // Output object tail.
bool Node::OutputTail()
{
    bool blRet = true;

    m_iIndent--;

    blRet = blRet && OutputIndent();

    blRet = blRet && OutputObjectCloseBrace();

    blRet = blRet && OutputNewLine();

    return(blRet);
}

    // Output template header.
bool Node::OutputTemplateHeader(const char *cpType, ClassID& oClassID)
{
    bool blRet = true;

    blRet = blRet && OutputToken(TOKEN_TEMPLATE);
    blRet = blRet && OutputSeparator(" ");
    blRet = blRet && OutputNameToken(cpType);
    blRet = blRet && OutputNewLine();

    blRet = blRet && OutputToken(TOKEN_OBRACE);
    blRet = blRet && OutputNewLine();

    m_iIndent++;

    blRet = blRet && oClassID.Output();

    return(blRet);
}

    // Output template tail.
bool Node::OutputTemplateTail()
{
    bool blRet = true;

    m_iIndent--;

    blRet = blRet && OutputToken(TOKEN_CBRACE);
    blRet = blRet && OutputNewLine();
    blRet = blRet && OutputNewLine();

    return(blRet);
}

    // Output byte value.
bool Node::OutputByte(int iValue)
{
    unsigned char bValue = (unsigned char)iValue;

    int iRet = fwrite(&bValue, 1, 1, m_pFilePointer);

    return(iRet <= 0 ? false : true);
}

    // Output word value.
bool Node::OutputWord(int iValue)
{
    return(OutputByte(iValue) && OutputByte(iValue >> 8));
}

    // Output dword value.
bool Node::OutputDWord(int iValue)
{
    return(OutputByte(iValue) &&
        OutputByte(iValue >> 8) &&
        OutputByte(iValue >> 16) &&
        OutputByte(iValue >> 24));
}

    // Output float value.
bool Node::OutputFloat(float fValue)
{
    return(OutputDWord(*((int *)&fValue)));
}

    // Format and output a string.
bool Node::OutputString(const char *cpFormat, ...)
{
    char caBuf[1024];

    va_list args;
    va_start(args, cpFormat);

        // Format message.
    vsprintf(caBuf, cpFormat, args);

    int iRet = fprintf(m_pFilePointer, caBuf);

    va_end(args);

    return(iRet < 0 ? false : true);
}

    // Output separator.
bool Node::OutputSeparator(const char *cpSeparator)
{
    if (!m_blBinary)
    {
        if (!OutputString(cpSeparator))
            return(false);
    }

    return(true);
}

    // Output new line.
bool Node::OutputNewLine()
{
    if (!m_blBinary)
    {
        if (!OutputString("\n"))
            return(false);
    }

    return(true);
}

    // Output indentation.
bool Node::OutputIndent()
{
    if (!m_blBinary)
    {
        if (!OutputString(Indent()))
            return(false);
    }

    return(true);
}

    // Output object open brace.
bool Node::OutputObjectOpenBrace()
{
    bool blRet = true;

    if (m_blBinary)
        blRet = OutputToken(TOKEN_OBRACE);
    else
        blRet = OutputString("{");

    return(blRet);
}

    // Output object close brace.
bool Node::OutputObjectCloseBrace()
{
    bool blRet = true;

    if (m_blBinary)
        blRet = OutputToken(TOKEN_CBRACE);
    else
        blRet = OutputString("}");

    return(blRet);
}

    // Output record token.
bool Node::OutputToken(int iToken)
{
    bool blRet = true;

    if (m_blBinary)
    {
            // Check for need to output cached list.
        blRet = blRet && ListEndCheck(iToken);

            // Save token.
        m_iCurrentToken = iToken;

            // Output the token.
        blRet = blRet && OutputWord(iToken);
    }
    else
    {
        const char *cpToken;

        switch (iToken)
        {
            case TOKEN_NAME:
                cpToken = "NAME";
                break;
            case TOKEN_STRING:
                cpToken = "STRING";
                break;
            case TOKEN_INTEGER:
                cpToken = "INT";
                break;
            case TOKEN_GUID:
                cpToken = "GUID";
                break;
            case TOKEN_INTEGER_LIST:
                cpToken = "INTEGER_LIST";
                break;
            case TOKEN_FLOAT_LIST:
                cpToken = "FLOAT_LIST";
                break;
            case TOKEN_OBRACE:
                cpToken = "{";
                break;
            case TOKEN_CBRACE:
                cpToken = "}";
                break;
            case TOKEN_OPAREN:
                cpToken = "(";
                break;
            case TOKEN_CPAREN:
                cpToken = ")";
                break;
            case TOKEN_OBRACKET:
                cpToken = "[";
                break;
            case TOKEN_CBRACKET:
                cpToken = "]";
                break;
            case TOKEN_OANGLE:
                cpToken = "<";
                break;
            case TOKEN_CANGLE:
                cpToken = ">";
                break;
            case TOKEN_DOT:
                cpToken = ".";
                break;
            case TOKEN_COMMA:
                cpToken = ",";
                break;
            case TOKEN_SEMICOLON:
                cpToken = ";";
                break;
            case TOKEN_TEMPLATE:
                cpToken = "template";
                break;
            case TOKEN_WORD:
                cpToken = "WORD";
                break;
            case TOKEN_DWORD:
                cpToken = "DWORD";
                break;
            case TOKEN_FLOAT:
                cpToken = "FLOAT";
                break;
            case TOKEN_DOUBLE:
                cpToken = "DOUBLE";
                break;
            case TOKEN_CHAR:
                cpToken = "CHAR";
                break;
            case TOKEN_UCHAR:
                cpToken = "UCHAR";
                break;
            case TOKEN_SWORD:
                cpToken = "SWORD";
                break;
            case TOKEN_SDWORD:
                cpToken = "SDWORD";
                break;
            case TOKEN_VOID:
                cpToken = "VOID";
                break;
            case TOKEN_LPSTR:
                cpToken = "LPSTR";
                break;
            case TOKEN_UNICODE:
                cpToken = "UNICODE";
                break;
            case TOKEN_CSTRING:
                cpToken = "CSTRING";
                break;
            case TOKEN_ARRAY:
                cpToken = "array";
                break;
            default:
                cpToken = "(unknown)";
                break;
        }

        blRet = OutputString(cpToken);
    }

    return(blRet);
}

    // Output record size.
bool Node::OutputSize(int iSize)
{
    return(OutputDWord(iSize));
}

    // Output name toke record.
bool Node::OutputNameToken(const char *cpName, const char *cpSeparator)
{
    bool blRet;

    if (m_blBinary)
    {
        blRet = OutputToken(TOKEN_NAME);
        blRet = blRet && OutputSize(strlen(cpName));
        blRet = blRet && OutputString(cpName);
    }
    else
    {
        blRet = OutputString(cpName);
        blRet = blRet && OutputString(cpSeparator);
    }

    return(blRet);
}

    // Output string token record.
bool Node::OutputStringToken(const char *cpString, const char *cpSeparator)
{
    bool blRet = true;

    if (m_blBinary)
    {
        blRet = blRet && OutputToken(TOKEN_STRING);
        blRet = blRet && OutputSize(strlen(cpString));
        blRet = blRet && OutputString(cpString);

        if (*cpSeparator == ',')
            blRet = blRet && OutputWord(TOKEN_COMMA);
        else
            blRet = blRet && OutputWord(TOKEN_SEMICOLON);
    }
    else
        blRet = OutputString("\"%s\"%s", cpString, cpSeparator);

    return(blRet);
}

    // Output integer token record.
bool Node::OutputIntegerToken(int iValue, const char *cpSeparator)
{
    bool blRet = true;

    if (m_blBinary)
    {
        blRet = blRet && OutputToken(TOKEN_INTEGER);
        blRet = blRet && OutputDWord(iValue);
    }
    else
        blRet = OutputString("%d%s", iValue, cpSeparator);

    return(blRet);
}

    // Output integer list token record (accumulative).
bool Node::OutputIntegerListToken(int iValue, const char *cpSeparator)
{
    bool blRet = true;

    if (m_blBinary)
    {
        if (m_iCurrentToken != TOKEN_INTEGER_LIST)
        {
            blRet = ListEndCheck(TOKEN_INTEGER_LIST);
            m_iCurrentToken = TOKEN_INTEGER_LIST;
        }

        m_oIntegerList.Append(iValue);
    }
    else
        blRet = blRet && OutputString("%d%s", iValue, cpSeparator);

    return(blRet);
}

    // Output empty integer list token record.
bool Node::OutputEmptyIntegerListToken()
{
    bool blRet = true;

    if (m_blBinary)
    {
        if (m_iCurrentToken != TOKEN_INTEGER_LIST)
        {
            blRet = ListEndCheck(TOKEN_INTEGER_LIST);
            m_iCurrentToken = TOKEN_INTEGER_LIST;
        }
    }
    else
    {
        blRet = blRet && OutputIndent();
        blRet = blRet && OutputString(";");
        blRet = blRet && OutputNewLine();
    }

    return(blRet);
}

    // Output float list token record (accumulative).
bool Node::OutputFloatListToken(float fValue, const char *cpSeparator)
{
    bool blRet = true;

    if (m_blBinary)
    {
        if (m_iCurrentToken != TOKEN_FLOAT_LIST)
        {
            blRet = ListEndCheck(TOKEN_FLOAT_LIST);
            m_iCurrentToken = TOKEN_FLOAT_LIST;
        }

        m_oFloatList.Append(fValue);
    }
    else
        blRet = blRet && OutputString("%f%s", fValue, cpSeparator);

    return(blRet);
}

    // Output empty float list token record.
bool Node::OutputEmptyFloatListToken()
{
    bool blRet = true;

    if (m_blBinary)
    {
        if (m_iCurrentToken != TOKEN_FLOAT_LIST)
        {
            blRet = ListEndCheck(TOKEN_FLOAT_LIST);
            m_iCurrentToken = TOKEN_FLOAT_LIST;
        }
    }
    else
    {
        blRet = blRet && OutputIndent();
        blRet = blRet && OutputString(";");
        blRet = blRet && OutputNewLine();
    }

    return(blRet);
}

    // Check for need to terminate a list record.
bool Node::ListEndCheck(int iToken)
{
    int iCount;
    int iIndex;
    bool blRet = true;

    if (m_iCurrentToken == TOKEN_INTEGER_LIST)
    {
        if (iToken != m_iCurrentToken)
        {
            blRet = blRet && OutputWord(TOKEN_INTEGER_LIST);

            iCount = m_oIntegerList.Count();

            blRet = blRet && OutputSize(iCount);

            for (iIndex = 0; blRet && (iIndex < iCount); iIndex++)
                blRet = blRet && OutputDWord(m_oIntegerList.Indexed(iIndex));

            m_oIntegerList.DeleteAll();

            m_iCurrentToken = iToken;
        }
    }
    else if (m_iCurrentToken == TOKEN_FLOAT_LIST)
    {
        if (iToken != m_iCurrentToken)
        {
            blRet = blRet && OutputWord(TOKEN_FLOAT_LIST);

            iCount = m_oFloatList.Count();

            blRet = blRet && OutputSize(iCount);

            for (iIndex = 0; blRet && (iIndex < iCount); iIndex++)
                blRet = blRet && OutputFloat(m_oFloatList.Indexed(iIndex));

            m_oFloatList.DeleteAll();

            m_iCurrentToken = iToken;
        }
    }

    return(blRet);
}

    // Output integer member.
bool Node::OutputIntegerMember(int iValue)
{
    bool blRet;

    blRet = OutputIndent();
    blRet = blRet && OutputIntegerListToken(iValue, ";");
    blRet = blRet && OutputNewLine();

    return(blRet);
}

    // Output float member.
bool Node::OutputFloatMember(float fValue)
{
    bool blRet;

    blRet = OutputIndent();
    blRet = blRet && OutputFloatListToken(fValue, ";");
    blRet = blRet && OutputNewLine();

    return(blRet);
}

    // Output string member.
bool Node::OutputStringMember(const char *cpString)
{
    bool blRet;

    blRet = OutputIndent();
    blRet = blRet && OutputStringToken(cpString, ";");
    blRet = blRet && OutputNewLine();

    return(blRet);
}

    // Output vector 2.
bool Node::OutputVector2(D3DXVECTOR2& oVector, const char *cpSeparator)
{
    return(OutputFloatList((float *)&oVector, 2, ";", cpSeparator));
}

    // Output vector 3.
bool Node::OutputVector3(D3DXVECTOR3& oVector, const char *cpSeparator)
{
    return(OutputFloatList((float *)&oVector, 3, ";", cpSeparator));
}

    // Output float list.
bool Node::OutputFloatList(
    float *fpList,
    int iCount,
    const char *cpSeparator,
    const char *cpEndSeparator)
{
    int i;
    bool blRet = true;

    for (i = 0; i < iCount; i++)
    {
        if ((i % 4) == 0)
            blRet = blRet && OutputIndent();

        blRet = blRet && OutputFloatListToken(fpList[i], cpSeparator);

        if (((i % 4) == ((iCount % 4) - 1)) || (i == iCount - 1))
        {
            blRet = blRet && OutputSeparator(cpEndSeparator);

            if (i == iCount - 1)
                blRet = blRet && OutputNewLine();
        }
    }


    return(blRet);
}

    // Output matrix.
bool Node::OutputMatrix(D3DXMATRIX *opMatrix, const char *cpSeparator)
{
    int i;
    char caSep[5];
    bool blRet = true;

    for (i = 0; i < 16; i++)
    {
        if ((i % 4) == 0)
            blRet = blRet && OutputIndent();

        sprintf(
            caSep,
            "%s%s",
            (i == 15 ? ";;" : ","),
            (i == 15 ? cpSeparator : ""));

        blRet = blRet && OutputFloatListToken(
            (*opMatrix)(i / 4, i % 4),
            caSep);

        if ((i % 4) == 3)
            blRet = blRet && OutputNewLine();
    }

    return(blRet);
}

    // Output object reference.
bool Node::OutputReference(const char *cpName)
{
    bool blRet;

    blRet = OutputIndent();

    if (m_blBinary)
        blRet = blRet && OutputToken(TOKEN_OBRACE);
    else
        blRet = blRet && OutputString("{");

    blRet = blRet && OutputNameToken(cpName, "");

    if (m_blBinary)
        blRet = blRet && OutputToken(TOKEN_CBRACE);
    else
        blRet = blRet && OutputString("}");

    blRet = blRet && OutputNewLine();

    return(blRet);
}

    // Output template member.
bool Node::OutputTemplateMember(const char *cpName, int iTypeToken)
{
    bool blRet;

    blRet = OutputIndent();
    blRet = blRet && OutputToken(iTypeToken);
    blRet = blRet && OutputSeparator(" ");
    blRet = blRet && OutputNameToken(cpName);
    blRet = blRet && OutputToken(TOKEN_SEMICOLON);
    blRet = blRet && OutputNewLine();

    return(blRet);
}

    // Output template member.
bool Node::OutputTemplateMember(const char *cpName, const char *cpType)
{
    bool blRet;

    blRet = OutputIndent();
    blRet = blRet && OutputNameToken(cpType);
    blRet = blRet && OutputSeparator(" ");
    blRet = blRet && OutputNameToken(cpName);
    blRet = blRet && OutputToken(TOKEN_SEMICOLON);
    blRet = blRet && OutputNewLine();

    return(blRet);
}

    // Output template array member.
bool Node::OutputTemplateArrayMember(
    const char *cpName, int iTypeToken, const char *cpSubscript, int iSubscript)
{
    bool blRet;

    blRet = OutputIndent();
    blRet = blRet && OutputToken(TOKEN_ARRAY);
    blRet = blRet && OutputSeparator(" ");
    blRet = blRet && OutputToken(iTypeToken);
    blRet = blRet && OutputSeparator(" ");
    blRet = blRet && OutputNameToken(cpName);
    blRet = blRet && OutputToken(TOKEN_OBRACKET);

    if (cpSubscript && *cpSubscript)
        blRet = blRet && OutputNameToken(cpSubscript);
    else
        blRet = blRet && OutputIntegerToken(iSubscript);

    blRet = blRet && OutputToken(TOKEN_CBRACKET);
    blRet = blRet && OutputToken(TOKEN_SEMICOLON);
    blRet = blRet && OutputNewLine();

    return(blRet);
}

    // Output template array member.
bool Node::OutputTemplateArrayMember(
    const char *cpName, const char *cpType, const char *cpSubscript, int iSubscript)
{
    bool blRet;

    blRet = OutputIndent();
    blRet = blRet && OutputToken(TOKEN_ARRAY);
    blRet = blRet && OutputSeparator(" ");
    blRet = blRet && OutputNameToken(cpType);
    blRet = blRet && OutputSeparator(" ");
    blRet = blRet && OutputNameToken(cpName);
    blRet = blRet && OutputToken(TOKEN_OBRACKET);

    if (cpSubscript && *cpSubscript)
        blRet = blRet && OutputNameToken(cpSubscript);
    else
        blRet = blRet && OutputIntegerToken(iSubscript);

    blRet = blRet && OutputToken(TOKEN_CBRACKET);
    blRet = blRet && OutputToken(TOKEN_SEMICOLON);
    blRet = blRet && OutputNewLine();

    return(blRet);
}

    // Output template restriction.
bool Node::OutputTemplateRestriction(const char *cpType)
{
    bool blRet;

    blRet = OutputIndent();
    blRet = blRet && OutputToken(TOKEN_OBRACKET);

    if (*cpType == '.')
    {
        blRet = blRet && OutputToken(TOKEN_DOT);
        blRet = blRet && OutputToken(TOKEN_DOT);
        blRet = blRet && OutputToken(TOKEN_DOT);
    }
    else
        blRet = blRet && OutputNameToken(cpType);

    blRet = blRet && OutputToken(TOKEN_CBRACKET);
    blRet = blRet && OutputNewLine();

    return(blRet);
}

    // Display a debug message.
void Node::Debug(const char *cpMessage, ...)
{
    char caBuf[1024];

    va_list args;
    va_start(args, cpMessage);

        // Format message.
    vsprintf(caBuf, cpMessage, args);

        // Display message.
    OutputDebugString(caBuf);

    va_end(args);
}

    // Output object body.
bool Matrix::OutputBody()
{
    return(OutputMatrix(&m_oMatrix));
}

    // Output object body.
bool BoneMatrix::OutputBody()
{
    BOOL blRet;

    blRet = OutputMatrix(&m_oOutputMatrix);

        // Output fixup matrix.
    blRet = blRet && OutputMatrix(&m_oInverseWorldMatrix);

    if (m_opParent)
        blRet = blRet && OutputReference(m_opParent->Name());

    return(true);
}

    // Output object body.
bool SkinWeightData::OutputBody()
{
    int iIndex;
    int iCount = m_oIndices.Count();
    D3DXMATRIX *opMatrix = &m_oMatrix.m_oMatrix;
    bool blRet = true;

        // Output bone frame name.
    blRet = blRet && OutputStringMember((m_opFrame ? m_opFrame->Name() : ""));

        // Output weight count.
    blRet = blRet && OutputIntegerMember(iCount);

        // Output vertex indices.
    for (iIndex = 0; iIndex < iCount; iIndex++)
    {
        blRet = blRet && OutputIndent();

        blRet = blRet && OutputIntegerListToken(
            m_oIndices.Indexed(iIndex),
            (iIndex == iCount - 1 ? ";" : ","));

        blRet = blRet && OutputNewLine();
    }

        // Output weights.
    for (iIndex = 0; iIndex < iCount; iIndex++)
    {
        blRet = blRet && OutputIndent();

        blRet = blRet && OutputFloatListToken(
            m_oWeights.Indexed(iIndex),
            (iIndex == iCount - 1 ? ";" : ","));

        blRet = blRet && OutputNewLine();
    }

        // Output bone offset matrix.
    OutputMatrix(opMatrix);

    return(blRet);
}

    // Output object body.
bool TextureFileName::OutputBody()
{
    bool blRet;

    blRet = OutputStringMember(m_caTextureFileName);

    return(blRet);
}

    // Output object body.
bool Material::OutputBody()
{
    bool blRet = true;

        // Output diffuse color.
    blRet = blRet && OutputFloatList((float *)&m_oDiffuse, 4, ";", ";");

        // Output power.
    blRet = blRet && OutputFloatMember(m_fPower);

        // Output specular color.
    blRet = blRet && OutputFloatList((float *)&m_oSpecular, 3, ";", ";");

        // Output emissive color.
    blRet = blRet && OutputFloatList((float *)&m_oEmissive, 3, ";", ";");

        // Output texture file name.
    if (blRet && m_oTextureFileName.m_caTextureFileName[0])
    {
        blRet = blRet && OutputNewLine();
        blRet = blRet && m_oTextureFileName.Output();
    }

    return(blRet);
}

    // Output object body.
bool RenderGroup::OutputBody()
{
    int iIndex;
    int iCount;
    D3DXVECTOR3 *pPos;
    D3DXVECTOR3 *pNormal;
    D3DXVECTOR2 *pTextureVector;
    int *ipFaceData;
    bool blRet = true;

        // Output FVF flags.
    blRet = blRet && OutputIntegerMember(m_iFVFFlags);

        // Output vertex position count.
    iCount = m_oPositions.Count();
    pPos = m_oPositions;
    blRet = blRet && OutputIntegerMember(iCount);

        // Output vertex positions.
    for (iIndex = 0; iIndex < iCount; iIndex++, pPos++)
        blRet = blRet && OutputVector3(*pPos, (iIndex == iCount - 1 ? ";" : ","));

        // Output face count.
    iCount = m_oFaceData.Count()/3;
    ipFaceData = m_oFaceData;
    blRet = blRet && OutputIntegerMember(iCount);

        // Output face vertex indices.
    for (iIndex = 0; iIndex < iCount; iIndex++, ipFaceData += 3)
    {
        blRet = blRet && OutputIndent();
        blRet = blRet && OutputIntegerListToken(3, ";");
        blRet = blRet && OutputIntegerListToken(ipFaceData[0], ",");
        blRet = blRet && OutputIntegerListToken(ipFaceData[1], ",");
        blRet = blRet && OutputIntegerListToken(
            ipFaceData[2], (iIndex == iCount - 1 ? ";;" : ";,"));
        blRet = blRet && OutputNewLine();
    }

    if (m_oMatrixIndices.Count())
    {
        blRet = blRet && OutputNewLine();

            // Output object header.
        blRet = blRet && OutputHeader("RenderGroupVertexMatrixIndices");

            // Output indices per vertex count.
        blRet = blRet && OutputIntegerMember(1);

            // Output number of indices.
        iCount = m_oMatrixIndices.Count();
        blRet = blRet && OutputIntegerMember(iCount);

            // Output matrix indices.
        for (iIndex = 0; iIndex < iCount; iIndex++, pPos++)
        {
            blRet = blRet && OutputIndent();

            blRet = blRet && OutputIntegerListToken(
                m_oMatrixIndices.Indexed(iIndex),
                (iIndex == iCount - 1 ? ";" : ","));

            blRet = blRet && OutputNewLine();
        }

            // Output weights per vertex count.
        blRet = blRet && OutputIntegerMember(0);

            // Output number of weights.
        blRet = blRet && OutputIntegerMember(0);

            // Output empty weight list.
        blRet = blRet && OutputIndent();
        blRet = blRet && OutputSeparator(";");
        blRet = blRet && OutputNewLine();

            // Output object tail.
        blRet = blRet && OutputTail();
    }

        // Output normals.
    if (m_oNormals.Count())
    {
        blRet = blRet && OutputNewLine();

            // Output object header.
        blRet = blRet && OutputHeader("RenderGroupNormals");

            // Output normal count.
        iCount = m_oNormals.Count();
        pNormal = m_oNormals;
        blRet = blRet && OutputIntegerMember(iCount);

            // Output normals.
        for (iIndex = 0; iIndex < iCount; iIndex++, pNormal++)
            blRet = blRet && OutputVector3(
                *pNormal,
                (iIndex == iCount - 1 ? ";" : ","));

            // Output face count.
        iCount = m_oNormalFaceData.Count()/3;
        ipFaceData = m_oNormalFaceData;
        blRet = blRet && OutputIntegerMember(iCount);

            // Output face normal indices.
        for (iIndex = 0; iIndex < iCount; iIndex++, ipFaceData += 3)
        {
            blRet = blRet && OutputIndent();
            blRet = blRet && OutputIntegerListToken(3, ";");
            blRet = blRet && OutputIntegerListToken(ipFaceData[0], ",");
            blRet = blRet && OutputIntegerListToken(ipFaceData[1], ",");
            blRet = blRet && OutputIntegerListToken(
                ipFaceData[2], (iIndex == iCount - 1 ? ";;" : ";,"));
            blRet = blRet && OutputNewLine();
        }

            // Output object tail.
        blRet = blRet && OutputTail();
    }

        // Output texture coordinates.
    if (m_oTextureCoordinates.Count())
    {
        blRet = blRet && OutputNewLine();

            // Output object header.
        blRet = blRet && OutputHeader("RenderGroupTextureCoords");

            // Output vertex position count.
        iCount = m_oTextureCoordinates.Count();
        pTextureVector = m_oTextureCoordinates;
        blRet = blRet && OutputIntegerMember(iCount);

            // Output vertex positions.
        for (iIndex = 0; iIndex < iCount; iIndex++, pTextureVector++)
            blRet = blRet && OutputVector2(
                *pTextureVector,
                (iIndex == iCount - 1 ? ";" : ","));

            // Output object tail.
        blRet = blRet && OutputTail();
    }

        // Output material.
    if (!m_oMaterial.IsNull())
    {
        blRet = blRet && OutputNewLine();
        blRet = blRet && m_oMaterial.Output();
    }

    return(blRet);
}

    // Output object body.
bool Mesh::OutputBody()
{
    int iIndex;
    int iCount;
    D3DXVECTOR3 *pPos;
    D3DXVECTOR3 *pNormal;
    D3DXVECTOR2 *pTextureVector;
    SkinWeightData *opSkinWeightData;
    int *ipFaceData;
    bool blRet = true;

        // Output vertex position count.
    iCount = m_oPositions.Count();
    pPos = m_oPositions;
    blRet = blRet && OutputIntegerMember(iCount);

        // Output vertex positions.
    for (iIndex = 0; iIndex < iCount; iIndex++, pPos++)
        blRet = blRet && OutputVector3(*pPos, (iIndex == iCount - 1 ? ";" : ","));

        // Output face count.
    iCount = m_oFaceData.Count()/3;
    ipFaceData = m_oFaceData;
    blRet = blRet && OutputIntegerMember(iCount);

        // Output face vertex indices.
    for (iIndex = 0; iIndex < iCount; iIndex++, ipFaceData += 3)
    {
        blRet = blRet && OutputIndent();
        blRet = blRet && OutputIntegerListToken(3, ";");
        blRet = blRet && OutputIntegerListToken(ipFaceData[0], ",");
        blRet = blRet && OutputIntegerListToken(ipFaceData[1], ",");
        blRet = blRet && OutputIntegerListToken(
            ipFaceData[2], (iIndex == iCount - 1 ? ";;" : ";,"));
        blRet = blRet && OutputNewLine();
    }

        // Output JTGame skin and bones extensions.
    if (IsJTGameSkinAndBones())
    {
        // Output mesh options.

        blRet = blRet && OutputNewLine();

            // Output object header.
        blRet = blRet && OutputHeader("MeshOptions");

            // Output mesh flags.
        blRet = blRet && OutputIntegerMember(m_iMeshFlags);

            // Output flexible vertex format flags.
        blRet = blRet && OutputIntegerMember(m_iFVFFlags);

            // Output object tail.
        blRet = blRet && OutputTail();

        // Output matrix indices and vertex weights.

        blRet = blRet && OutputNewLine();

            // Output object header.
        blRet = blRet && OutputHeader("MeshBoneData");

            // Output vertex indices per vertex.
        blRet = blRet && OutputIntegerMember(1);

            // Output vertex index count.
        iCount = m_oPositions.Count();
        blRet = blRet && OutputIntegerMember(iCount);

            // Output vertex indices.
        for (iIndex = 0; iIndex < iCount; iIndex++)
        {
            blRet = blRet && OutputIndent();
            blRet = blRet && OutputIntegerListToken(
                m_oBoneIndices.Indexed(iIndex),
                (iIndex == iCount - 1 ? ";" : ","));
            blRet = blRet && OutputNewLine();
        }

            // Output vertex weights per vertex.
        blRet = blRet && OutputIntegerMember(0);

            // Output vertex weight count.
        blRet = blRet && OutputIntegerMember(0);

            // Output empty vertex weight list.
        blRet = blRet && OutputEmptyFloatListToken();

            // Output object tail.
        blRet = blRet && OutputTail();
    }

        // Output normals.
    if (m_oNormals.Count())
    {
        blRet = blRet && OutputNewLine();

            // Output object header.
        blRet = blRet && OutputHeader("MeshNormals");

            // Output normal count.
        iCount = m_oNormals.Count();
        pNormal = m_oNormals;
        blRet = blRet && OutputIntegerMember(iCount);

            // Output normals.
        for (iIndex = 0; iIndex < iCount; iIndex++, pNormal++)
            blRet = blRet && OutputVector3(
                *pNormal,
                (iIndex == iCount - 1 ? ";" : ","));

            // Output face count.
        iCount = m_oNormalFaceData.Count()/3;
        ipFaceData = m_oNormalFaceData;
        blRet = blRet && OutputIntegerMember(iCount);

            // Output face normal indices.
        for (iIndex = 0; iIndex < iCount; iIndex++, ipFaceData += 3)
        {
            blRet = blRet && OutputIndent();
            blRet = blRet && OutputIntegerListToken(3, ";");
            blRet = blRet && OutputIntegerListToken(ipFaceData[0], ",");
            blRet = blRet && OutputIntegerListToken(ipFaceData[1], ",");
            blRet = blRet && OutputIntegerListToken(
                ipFaceData[2], (iIndex == iCount - 1 ? ";;" : ";,"));
            blRet = blRet && OutputNewLine();
        }

            // Output object tail.
        blRet = blRet && OutputTail();
    }


        // Output texture coordinates.
    if (m_oTextureCoordinates.Count())
    {
        blRet = blRet && OutputNewLine();

            // Output object header.
        blRet = blRet && OutputHeader("MeshTextureCoords");

            // Output vertex position count.
        iCount = m_oTextureCoordinates.Count();
        pTextureVector = m_oTextureCoordinates;
        blRet = blRet && OutputIntegerMember(iCount);

            // Output vertex positions.
        for (iIndex = 0; iIndex < iCount; iIndex++, pTextureVector++)
            blRet = blRet && OutputVector2(
                *pTextureVector,
                (iIndex == iCount - 1 ? ";" : ","));

            // Output object tail.
        blRet = blRet && OutputTail();
    }

        // Output VertexDuplicationIndices.
    if (m_oVertexOriginalIndices.Count() || m_oVertexDuplicateIndices.Count())
    {
        blRet = blRet && OutputNewLine();

            // Output header
        blRet = blRet && OutputHeader("VertexDuplicationIndices");

            // Output total vertex index count.
        blRet = blRet && OutputIntegerMember(
            m_oVertexOriginalIndices.Count() + m_oVertexDuplicateIndices.Count());

            // Output original vertex index count.
        blRet = blRet && OutputIntegerMember(m_oVertexOriginalIndices.Count());

            // Output original vertex indices.
        iCount = m_oVertexOriginalIndices.Count();

        for (iIndex = 0; iIndex < iCount; iIndex++)
        {
            blRet = blRet && OutputIndent();

            blRet = blRet && OutputIntegerListToken(
                m_oVertexOriginalIndices.Indexed(iIndex),
                (iIndex == iCount - 1 ? ";" : ","));

            blRet = blRet && OutputNewLine();
        }

            // Output duplicate vertex indices.
        iCount = m_oVertexDuplicateIndices.Count();

        for (iIndex = 0; iIndex < iCount; iIndex++)
        {
            blRet = blRet && OutputIndent();

            blRet = blRet && OutputIntegerListToken(
                m_oVertexDuplicateIndices.Indexed(iIndex),
                (iIndex == iCount - 1 ? ";" : ","));

            blRet = blRet && OutputNewLine();
        }

            // Output object tail.
        blRet = blRet && OutputTail();
    }

        // Output material list.
    if (m_oMaterials.Count())
    {
        blRet = blRet && OutputNewLine();

        iCount = m_oMaterialFaceIndices.Count();

            // Output material list data header.
        blRet = blRet && OutputHeader("MeshMaterialList");

            // Output material count.
        blRet = blRet && OutputIntegerMember(m_oMaterials.Count());

            // Output face index count.
        blRet = blRet && OutputIntegerMember(iCount);

            // Output face indices.
        for (iIndex = 0; blRet && (iIndex < iCount); iIndex++)
        {
            blRet = blRet && OutputIndent();

            blRet = blRet && OutputIntegerListToken(
                m_oMaterialFaceIndices.Indexed(iIndex),
                (iIndex == iCount - 1 ? ";" : ","));

            blRet = blRet && OutputNewLine();
        }

            // Output materials.
        iCount = m_oMaterials.Count();

        for (iIndex = 0; blRet && (iIndex < iCount); iIndex++)
        {
            blRet = blRet && OutputNewLine();
            blRet = blRet && m_oMaterials.Indexed(iIndex)->Output();
        }

            // Output object tail.
        blRet = blRet && OutputTail();
    }

        // Output skin weight data.
    if (m_oSkinWeightData.Count())
    {
        blRet = blRet && OutputNewLine();

            // Output XSkinMeshHeader header.
        blRet = blRet && OutputHeader("XSkinMeshHeader");

            // Output weights per vertex count.
        blRet = blRet && OutputIntegerMember(1);

            // Output weights per face count.
        blRet = blRet && OutputIntegerMember(3);

            // Output SkinWeight count.
        iCount = m_oSkinWeightData.Count();
        blRet = blRet && OutputIntegerMember(iCount);

            // Output object tail.
        blRet = blRet && OutputTail();

            // Output SkinWeigth objects.
        for (iIndex = 0; iIndex < iCount; iIndex++)
        {
            opSkinWeightData = m_oSkinWeightData.Indexed(iIndex);
            blRet = blRet && OutputNewLine();
            blRet = blRet && opSkinWeightData->Output();
        }
    }

    return(blRet);
}

    // Transform mesh positions.
void Mesh::TransformPositions(D3DXMATRIX& oMatrix)
{
    int iIndex;
    int iCount = m_oPositions.Count();
    D3DXVECTOR3 oVector;
    D3DXVECTOR4 oVector4;

    for (iIndex = 0; iIndex < iCount; iIndex++)
    {
            // Get the vertex.
        oVector = m_oPositions.Indexed(iIndex);

            // Transform vertex position to the local bone coordinate space.
        D3DXVec3Transform(
            &oVector4,
            &oVector,
            &oMatrix);

            // Save the vertex.
        m_oPositions.Set(iIndex, *(D3DXVECTOR3 *)&oVector4, false);
    }
}

    // Transform mesh normals.
void Mesh::TransformNormals(D3DXMATRIX& oMatrix)
{
    int iIndex;
    int iCount = m_oPositions.Count();
    D3DXVECTOR3 oVector;
    D3DXVECTOR3 oNormal;

    for (iIndex = 0; iIndex < iCount; iIndex++)
    {
            // Get the normal.
        oVector = m_oNormals.Indexed(iIndex);

            // Transform normal to bone space.
        D3DXVec3TransformNormal(
            &oNormal,
            &oVector,
            &oMatrix);

            // Save the vertex.
        m_oNormals.Set(iIndex, oNormal, false);
    }
}

    // Compact normals.
void Mesh::CompactNormals()
{
    int i, j;
    int iCount = m_oNormals.Count();
    VectorArray oCompacted(iCount);
    IntegerArray oMap(iCount);
    D3DXVECTOR3 oNormal;

    oMap.SetCount(iCount);

        // Collect unique normals.
    for (i = 0; i < iCount; i++)
    {
        oNormal = m_oNormals.Indexed(i);

        for (j = 0; j < oCompacted.Count(); j++)
        {
            if (oNormal == oCompacted.Indexed(j))
                break;
        }

        if (j == oCompacted.Count())
            oCompacted.Append(oNormal);

        oMap.Set(i, j, false);
    }

        // Save new normals.
    m_oNormals = oCompacted;

        // Fixup normal face data.
    iCount = m_oNormalFaceData.Count();

    for (i = 0; i < iCount; i++)
    {
        m_oNormalFaceData.Set(
            i,
            oMap.Indexed(m_oNormalFaceData.Indexed(i)),
            false);
    }
}

#if 1
#define MatchVector(oVector0, oVector1) (oVector0 == oVector1)
#else
#define kCoordDiff 0.005f

    // Compare vectors (in case error accumulation is an issue).
inline bool MatchVector(const D3DXVECTOR3& oVector0, const D3DXVECTOR3& oVector1)
{
    D3DXVECTOR3 oDiff(
        oVector1.x - oVector0.x,
        oVector1.y - oVector0.y,
        oVector1.z - oVector0.z);

    if (oDiff.x < 0.0f)
        oDiff.x = -oDiff.x;

    if (oDiff.y < 0.0f)
        oDiff.y = -oDiff.y;

    if (oDiff.z < 0.0f)
        oDiff.z = -oDiff.z;

    if ((oDiff.x > kCoordDiff) || (oDiff.y > kCoordDiff) || (oDiff.z > kCoordDiff))
        return(false);

    return(true);
}
#endif

    // Close holes in the mesh.
void Mesh::CloseHoles()
{
    int iVertexIndex;
    int iFaceIndex;
    int iFaceCount = m_oFaceData.Count()/3;
    int iNewFaceCount;
    int iFaceVertexIndex;
    int iFaceVertexCount = 3;
    int iEdgeIndex;
    int iEdgeCount = m_oFaceData.Count();
    int iSubEdgeIndex;
    int iBaseEdgeIndex;
    int iLastEdgeIndex;
    int iHoleEdgeIndex;
    int iHoleEdgeCount;
    int iHoleCount = 0;
    IntegerArray oHoleIndices;
    VectorArray oVectors;
    EdgeArray oAllEdges(iEdgeCount);
    EdgeArray oAllHoleEdges(iEdgeCount);
    EdgeArray oHoleEdges(iEdgeCount);
    Edge *opEdge;
    Edge *opEdge0;
    Edge *opEdge1;
    Edge *opEdgeBase;
    int *ipFaceData = &m_oFaceData.Indexed(0);
    int *ipNormalFaceData = &m_oNormalFaceData.Indexed(0);

    oAllEdges.SetCount(iEdgeCount);
    opEdge = &oAllEdges.Indexed(0);
    memset(opEdge, 0, iEdgeCount * sizeof(Edge));

        // Collect all edges.
    for (iFaceIndex = 0; iFaceIndex < iFaceCount; iFaceIndex++, ipFaceData += iFaceVertexCount)
    {
        for (iFaceVertexIndex = 0; iFaceVertexIndex < iFaceVertexCount; iFaceVertexIndex++, opEdge++)
        {
            opEdge->m_iFaceIndex = iFaceIndex;
            opEdge->m_iVertexIndex0 = ipFaceData[iFaceVertexIndex];
            opEdge->m_iVertexIndex1 = ipFaceData[(iFaceVertexIndex + 1) % 3];
            opEdge->m_oVertexPosition0 = m_oPositions.Indexed(opEdge->m_iVertexIndex0);
            opEdge->m_oVertexPosition1 = m_oPositions.Indexed(opEdge->m_iVertexIndex1);
            opEdge->m_iNormalIndex0 = ipNormalFaceData[iFaceVertexIndex];
            opEdge->m_iNormalIndex1 = ipNormalFaceData[(iFaceVertexIndex + 1) % 3];
            opEdge->m_iMaterialIndex = m_oMaterialFaceIndices.Indexed(iFaceIndex);
        }
    }

    opEdge0 = &oAllEdges.Indexed(0);

        // Find non-shared edges.  These are the edges around the hole.
    for (iEdgeIndex = 0; iEdgeIndex < iEdgeCount; iEdgeIndex++, opEdge0++)
    {
        opEdge1 = &oAllEdges.Indexed(0);

        for (iSubEdgeIndex = 0; iSubEdgeIndex < iEdgeCount; iSubEdgeIndex++, opEdge1++)
        {
            if (iSubEdgeIndex == iEdgeIndex)
                continue;

            if ((MatchVector(opEdge0->m_oVertexPosition0, opEdge1->m_oVertexPosition0) &&
                    MatchVector(opEdge0->m_oVertexPosition1, opEdge1->m_oVertexPosition1)) ||
                (MatchVector(opEdge0->m_oVertexPosition0, opEdge1->m_oVertexPosition1) &&
                    MatchVector(opEdge0->m_oVertexPosition1, opEdge1->m_oVertexPosition0)))
                break;
        }

            // If we didn't find a shared edge, save it as hole edge.
        if (iSubEdgeIndex == iEdgeCount)
            oAllHoleEdges.Append(*opEdge0);
    }

        // We are now only interested in the hole edges.
    iEdgeCount = oAllHoleEdges.Count();

        // Pre-size hole edge array.
    oHoleEdges.SetSize(iEdgeCount);
    oHoleEdges.SetCount(iEdgeCount);
    memset((Edge *)oHoleEdges, 0, sizeof(Edge) * iEdgeCount);

        // Find individual holes and fill them.
    while (1)
    {
            // Find an edge not yet used.
        for (iBaseEdgeIndex = 0; iBaseEdgeIndex < iEdgeCount; iBaseEdgeIndex++)
        {
            if (oAllHoleEdges.Indexed(iBaseEdgeIndex).m_iTouchedFlags == 0)
            {
                oAllHoleEdges.Indexed(iBaseEdgeIndex).m_iTouchedFlags |= (1 << iHoleCount);
                break;
            }
        }

            // If all edges used, we are done.
        if (iBaseEdgeIndex == iEdgeCount)
            break;

            // Initialize hole edge data.
        iHoleEdgeIndex = iHoleEdgeCount = 1;

            // Initialize first hole edge.
        opEdge0 = &oAllHoleEdges.Indexed(iBaseEdgeIndex);
        opEdgeBase = opEdge0;
        oHoleEdges.Indexed(0) = *opEdge0;

        iLastEdgeIndex = -1;

            // Find linked edges.
        while (1)
        {
                // Find one linked edge.
            for (iEdgeIndex = 0; iEdgeIndex < iEdgeCount; iEdgeIndex++)
            {
                opEdge1 = &oAllHoleEdges.Indexed(iEdgeIndex);

                if (opEdge1->m_iTouchedFlags & (1 << iHoleCount))
                    continue;

                if (MatchVector(opEdge0->m_oVertexPosition1, opEdge1->m_oVertexPosition0))
                        // Got one.
                    break;
                else if (MatchVector(opEdge0->m_oVertexPosition1, opEdge1->m_oVertexPosition1))
                {
                        // Swap vertex order.
                    int iTmp = opEdge1->m_iVertexIndex0;
                    opEdge1->m_iVertexIndex0 = opEdge1->m_iVertexIndex1;
                    opEdge1->m_iVertexIndex1 = iTmp;
                    iTmp = opEdge1->m_iNormalIndex0;
                    opEdge1->m_iNormalIndex0 = opEdge1->m_iNormalIndex1;
                    opEdge1->m_iNormalIndex1 = iTmp;
                    D3DXVECTOR3 oTmp(opEdge1->m_oVertexPosition0);
                    opEdge1->m_oVertexPosition0 = opEdge1->m_oVertexPosition1;
                    opEdge1->m_oVertexPosition1 = oTmp;

                        // Got one.
                    break;
                }
            }

                // If no linked edge found our hole is complete.
            if (iEdgeIndex == iEdgeCount)
                break;

                // Mark edge as touched.
            opEdge1->m_iTouchedFlags |= (1 << iHoleCount);

                // Save edge as part of hole..
            oHoleEdges.Indexed(iHoleEdgeCount++) = *opEdge1;

                // Save last edge.
            iLastEdgeIndex = iEdgeIndex;

                // Set up to find the next linked edge.
            opEdge0 = opEdge1;
        }

            // If enough edges to make a triangle.
        if (iHoleEdgeCount > 1)
        {
                // Presize index array.
            if (oHoleIndices.Size() < iEdgeCount)
                oHoleIndices.SetSize(iEdgeCount);

            oHoleIndices.SetCount(0);

                // Presize vector array.
            if (oVectors.Size() < iEdgeCount)
                oVectors.SetSize(iEdgeCount);

            oVectors.SetCount(0);

                // Collect vertices.
            for (iHoleEdgeIndex = 0; iHoleEdgeIndex < iHoleEdgeCount; iHoleEdgeIndex++)
            {
                iVertexIndex = oHoleEdges.Indexed(iHoleEdgeIndex).m_iVertexIndex0;
                oHoleIndices.Append(iVertexIndex);
                oVectors.Append(m_oPositions.Indexed(iVertexIndex));
            }

                // Get starting face index.
            iFaceIndex = m_oFaceData.Count()/iFaceVertexCount;

                // Triangulate edges into new face data.
            iNewFaceCount = Triangulate(
                m_oFaceData,
                oHoleIndices,
                iHoleEdgeCount,
                oVectors);

                // Get starting face index and count.
            iFaceCount = iFaceIndex + iNewFaceCount;

                // Add normal and material data.
            for (; iFaceIndex < iFaceCount; iFaceIndex++)
            {
                for (iFaceVertexIndex = 0; iFaceVertexIndex < iFaceVertexCount; iFaceVertexIndex++)
                {
                        // Get vertex index.
                    iVertexIndex = m_oFaceData.Indexed(
                        (iFaceIndex * 3) + iFaceVertexIndex);

                        // Find edge so we can get the normal material index.
                    for (iHoleEdgeIndex = 0; iHoleEdgeIndex < iHoleEdgeCount; iHoleEdgeIndex++)
                    {
                        if (oHoleEdges.Indexed(iHoleEdgeIndex).m_iVertexIndex0 == iVertexIndex)
                        {
                                // Reuse the same normal index.
                            m_oNormalFaceData.Append(
                                oHoleEdges.Indexed(iHoleEdgeIndex).m_iNormalIndex0);

                            if (iFaceVertexIndex == 0)
                                m_oMaterialFaceIndices.Append(
                                    oHoleEdges.Indexed(iHoleEdgeIndex).m_iMaterialIndex);

                            break;
                        }
                    }
                }
            }
        }
    }
}



    // Triangulate an array of vertex indices.  Returns triangle count.
int Mesh::Triangulate(
    IntegerArray& oDestFaceData,        // Output triangle index tuples.
    int *ipSourceFaceData,              // Pointer to input primitive vertex indices.
    int iFaceVertexCount,               // Number of primitive vertices.
    D3DXVECTOR3 *opaVertices)           // Vertex table.
{
    int iTriangleCount = iFaceVertexCount - 2;
    int iTriangleIndex = 0;
    int iVertexCount = iFaceVertexCount;
    int iCount;
    int iVertex0 = 0;
    int iVertex1 = 0;
    int iVertex2 = 0;
    int iVertexTerminal;
    BYTE baUsedArray[256];
    BYTE *bpUsedArray;
    D3DXVECTOR3 *opVertex0;
    D3DXVECTOR3 *opVertex1;
    D3DXVECTOR3 *opVertex2;

    if (iFaceVertexCount > 256)
        bpUsedArray = new BYTE[iVertexCount];
    else
        bpUsedArray = &baUsedArray[0];

    memset(bpUsedArray, 0, iVertexCount);

    while (iTriangleIndex < iTriangleCount)
    {
        iVertexTerminal = iVertex0;

        while (1)
        {
                // Get next vertex 1.
            iVertex1 = iVertex0 + 1;

            for (iCount = iVertexCount; iCount > 0; iCount--)
            {
                if (iVertex1 == iVertexCount)
                    iVertex1 = 0;

                if (bpUsedArray[iVertex1])
                    iVertex1++;
                else
                    break;
            }

                // Get next vertex 2.
            iVertex2 = iVertex1 + 1;

            for (iCount = iVertexCount; iCount > 0; iCount--)
            {
                if (iVertex2 == iVertexCount)
                    iVertex2 = 0;

                if (bpUsedArray[iVertex2])
                    iVertex2++;
                else
                    break;
            }

            if (iVertex2 == iVertexTerminal)
            {
                Debug("Triangulate:  Bad primitive.");
                return(0);
            }

            opVertex0 = &opaVertices[iVertex0];
            opVertex1 = &opaVertices[iVertex1];
            opVertex2 = &opaVertices[iVertex2];

            D3DXVECTOR3 pEdge10(*opVertex0 - *opVertex1);
            D3DXVECTOR3 pEdge12(*opVertex2 - *opVertex1);

            D3DXVec3Normalize(&pEdge10, &pEdge10);
            D3DXVec3Normalize(&pEdge12, &pEdge12);

            float coCos = D3DXVec3Dot(&pEdge10, &pEdge12);

            if ((coCos == 1.0f) || (coCos == -1.0f))
            {
                if (iTriangleIndex == iTriangleCount - 1)
                {
                    break;
                }

                iVertex0 = iVertex1;
            }
            else
                    // It's an ear, so break it off.
                break;
        }

            // Set vertex indices.
        oDestFaceData.Append(ipSourceFaceData[iVertex0]);
//        oDestFaceData.Append(ipSourceFaceData[iVertex1]);
        oDestFaceData.Append(ipSourceFaceData[iVertex2]);
        oDestFaceData.Append(ipSourceFaceData[iVertex1]);

            // Mark corner vertex as used to get it out of the pool.
        bpUsedArray[iVertex1] = 1;

            // Start next ear search with vertex 2.
        iVertex0 = iVertex2;

            // Increment triangle counter.
        iTriangleIndex++;
    }

    if (bpUsedArray != &baUsedArray[0])
        delete [] bpUsedArray;

    return((int)iFaceVertexCount - 2);
}

    // Find descendent frame.
Frame *Frame::FindDescendent(const char *cpName)
{
    int iIndex;
    int iCount = m_oChildren.Count();
    Frame *opChild;

    for (iIndex = 0; iIndex < iCount; iIndex++)
    {
        opChild = m_oChildren.Indexed(iIndex);

        if (opChild->Match(cpName))
            return(opChild);

        if ((opChild = opChild->FindDescendent(cpName)) != NULL)
            return(opChild);
    }

    return(NULL);
}

    // Find common parent.
Frame *Frame::FindCommonParent(Frame *opFrame)
{
    Frame *opParentA;
    Frame *opParentB;

    for (opParentA = this; opParentA != NULL; opParentA = opParentA->m_opParent)
    {
        for (opParentB = this; opParentB != NULL; opParentB = opParentB->m_opParent)
        {
            if (opParentA == opParentB)
                break;
        }
    }

    return(opParentA);
}

    // Output object body.
bool Frame::OutputBody()
{
    bool blRet = true;

    if (m_oBoneMatrices.Count())
    {
            // Output bone matrices.
        blRet = blRet && m_oBoneMatrices.Output();

        if (m_oBoneMatrices.Count() &&
                (m_oRenderGroups.Count() || m_oMeshes.Count()))
            blRet = blRet && OutputNewLine();

            // Output meshes.
        if (m_oMeshes.Count())
            blRet = blRet && m_oMeshes.Output();
        else if (m_oRenderGroups.Count())
            blRet = blRet && m_oRenderGroups.Output();
    }
    else
    {
            // Output FrameTransformMatrix.
        blRet = blRet && m_oOutputMatrix.Output();

        if (m_oChildren.Count())
            blRet = blRet && OutputNewLine();

            // Output child frames.
        blRet = blRet && m_oChildren.Output();

        if (m_oMeshes.Count())
            blRet = blRet && OutputNewLine();

            // Output meshes.
        blRet = blRet && m_oMeshes.Output();
    }

    return(blRet);
}

    // Add key interpolations between keys.
void AnimationKey::AddInterpolations(int iCount)
{
    int iOldKeyCount = m_oTimes.Count();
    int iNewKeyCount = ((iOldKeyCount - 1) * iCount) + 1;
    int iOldIndex = iOldKeyCount - 1;
    int iNewIndex = iNewKeyCount - 1;
    int iLoopCount;
    int iIndex;

        // Grow key arrays.
    m_oTimes.SetSize(iNewKeyCount);
    m_oTimes.SetCount(iNewKeyCount);
    m_oValues.SetSize(iNewKeyCount * m_iValueCount);
    m_oValues.SetCount(iNewKeyCount * m_iValueCount);

        // Move last key.
    MoveKey(iNewIndex, iOldIndex);

        // Loop original keys.
    for (iOldIndex--, iNewIndex -= iCount; iOldIndex >= 0; iOldIndex--, iNewIndex -= iCount)
    {
            // Move base key.
        MoveKey(iNewIndex, iOldIndex);

            // Loop interpolated keys.
        for (iIndex = iNewIndex + 1, iLoopCount = iCount - 1;
                iLoopCount > 0;
                iLoopCount--, iIndex++)
                // Do interpolation.
            InterpolateKey(iNewIndex, iIndex, iNewIndex + iCount);
    }
}

    // Move a key.
void AnimationKey::MoveKey(int iDstIndex, int iSrcIndex)
{
    int iCount = m_iValueCount;
    int iValueSrcIndex = iSrcIndex * m_iValueCount;
    int iValueDstIndex = iDstIndex * m_iValueCount;

    m_oTimes.Set(iDstIndex, m_oTimes.Indexed(iSrcIndex), FALSE);

    while (iCount--)
    {
        m_oValues.Set(iValueDstIndex, m_oValues.Indexed(iValueSrcIndex), FALSE);
        iValueDstIndex++;
        iValueSrcIndex++;
    }
}

    // Interpolate a key given the indices.
void AnimationKey::InterpolateKey(int iBaseIndex0, int iIndex, int iBaseIndex1)
{
    int iValueBaseIndex0 = iBaseIndex0 * m_iValueCount;
    int iValueIndex = iIndex * m_iValueCount;
    int iValueBaseIndex1 = iBaseIndex1 * m_iValueCount;
    int iRange = iBaseIndex1 - iBaseIndex0;
    int iSelector = iIndex - iBaseIndex0;
    float fSelector = (float)iSelector/iRange;

        // Interpolate key time.
    int iBaseTime0 = m_oTimes.Indexed(iBaseIndex0);
    int iDiffTime = m_oTimes.Indexed(iBaseIndex1) - iBaseTime0;
    int iTime = iBaseTime0 + (iDiffTime * iSelector)/iRange;
    m_oTimes.Set(iIndex, iTime);

        // Interpolate key based on value.
    switch (m_iType)
    {
            // Rotation.
        case 0:
            {
                D3DXQUATERNION oBaseQuat0(
                    m_oValues.Indexed(iValueBaseIndex0 + 1),
                    m_oValues.Indexed(iValueBaseIndex0 + 2),
                    m_oValues.Indexed(iValueBaseIndex0 + 3),
                    m_oValues.Indexed(iValueBaseIndex0));
                D3DXQUATERNION oQuat;
                D3DXQUATERNION oBaseQuat1(
                    m_oValues.Indexed(iValueBaseIndex1 + 1),
                    m_oValues.Indexed(iValueBaseIndex1 + 2),
                    m_oValues.Indexed(iValueBaseIndex1 + 3),
                    m_oValues.Indexed(iValueBaseIndex1));
                D3DXQuaternionSlerp(
                    &oQuat,
                    &oBaseQuat0,
                    &oBaseQuat1,
                    fSelector);
                m_oValues.Set(iValueIndex, oQuat.w, FALSE);
                m_oValues.Set(iValueIndex + 1, oQuat.x, FALSE);
                m_oValues.Set(iValueIndex + 2, oQuat.y, FALSE);
                m_oValues.Set(iValueIndex + 3, oQuat.z, FALSE);
            }
            break;
            // Position.
        case 2:
            {
                D3DXVECTOR3 oBaseVector0(
                    m_oValues.Indexed(iValueBaseIndex0 + 0),
                    m_oValues.Indexed(iValueBaseIndex0 + 1),
                    m_oValues.Indexed(iValueBaseIndex0 + 2));
                D3DXVECTOR3 oVector;
                D3DXVECTOR3 oBaseVector1(
                    m_oValues.Indexed(iValueBaseIndex1 + 0),
                    m_oValues.Indexed(iValueBaseIndex1 + 1),
                    m_oValues.Indexed(iValueBaseIndex1 + 2));
                D3DXVec3Lerp(
                    &oVector,
                    &oBaseVector0,
                    &oBaseVector1,
                    fSelector);
                m_oValues.Set(iValueIndex, oVector.x, FALSE);
                m_oValues.Set(iValueIndex + 1, oVector.y, FALSE);
                m_oValues.Set(iValueIndex + 2, oVector.z, FALSE);
            }
            break;
            // Matrix.
        case 4:
            {
                D3DXMATRIX oBaseMatrix0(
                    m_oValues.Indexed(iValueBaseIndex0),
                    m_oValues.Indexed(iValueBaseIndex0 + 1),
                    m_oValues.Indexed(iValueBaseIndex0 + 2),
                    m_oValues.Indexed(iValueBaseIndex0 + 3),
                    m_oValues.Indexed(iValueBaseIndex0 + 4),
                    m_oValues.Indexed(iValueBaseIndex0 + 5),
                    m_oValues.Indexed(iValueBaseIndex0 + 6),
                    m_oValues.Indexed(iValueBaseIndex0 + 7),
                    m_oValues.Indexed(iValueBaseIndex0 + 8),
                    m_oValues.Indexed(iValueBaseIndex0 + 9),
                    m_oValues.Indexed(iValueBaseIndex0 + 10),
                    m_oValues.Indexed(iValueBaseIndex0 + 11),
                    m_oValues.Indexed(iValueBaseIndex0 + 12),
                    m_oValues.Indexed(iValueBaseIndex0 + 13),
                    m_oValues.Indexed(iValueBaseIndex0 + 14),
                    m_oValues.Indexed(iValueBaseIndex0 + 15));
                D3DXMATRIX oMatrix;
                D3DXMATRIX oBaseMatrix1(
                    m_oValues.Indexed(iValueBaseIndex1),
                    m_oValues.Indexed(iValueBaseIndex1 + 1),
                    m_oValues.Indexed(iValueBaseIndex1 + 2),
                    m_oValues.Indexed(iValueBaseIndex1 + 3),
                    m_oValues.Indexed(iValueBaseIndex1 + 4),
                    m_oValues.Indexed(iValueBaseIndex1 + 5),
                    m_oValues.Indexed(iValueBaseIndex1 + 6),
                    m_oValues.Indexed(iValueBaseIndex1 + 7),
                    m_oValues.Indexed(iValueBaseIndex1 + 8),
                    m_oValues.Indexed(iValueBaseIndex1 + 9),
                    m_oValues.Indexed(iValueBaseIndex1 + 10),
                    m_oValues.Indexed(iValueBaseIndex1 + 11),
                    m_oValues.Indexed(iValueBaseIndex1 + 12),
                    m_oValues.Indexed(iValueBaseIndex1 + 13),
                    m_oValues.Indexed(iValueBaseIndex1 + 14),
                    m_oValues.Indexed(iValueBaseIndex1 + 15));
                    // Extract translation.
                D3DXVECTOR3 oBaseVector0(
                    oBaseMatrix0(3, 0),
                    oBaseMatrix0(3, 1),
                    oBaseMatrix0(3, 2));
                D3DXVECTOR3 oVector;
                D3DXVECTOR3 oBaseVector1(
                    oBaseMatrix1(3, 0),
                    oBaseMatrix1(3, 1),
                    oBaseMatrix1(3, 2));
                    // Interpolate translation.
                D3DXVec3Lerp(
                    &oVector,
                    &oBaseVector0,
                    &oBaseVector1,
                    fSelector);
                    // Extract rotation quaternions.
                D3DXQUATERNION oBaseQuat0;
                D3DXQUATERNION oQuat;
                D3DXQUATERNION oBaseQuat1;
                D3DXQuaternionRotationMatrix(
                    &oBaseQuat0,
                    &oBaseMatrix0);
                D3DXQuaternionRotationMatrix(
                    &oBaseQuat1,
                    &oBaseMatrix1);
                    // Interpolate quaternions.
                D3DXQuaternionSlerp(
                    &oQuat,
                    &oBaseQuat0,
                    &oBaseQuat1,
                    fSelector);
                    // Build interpolated matrix from rotation.
                D3DXMatrixRotationQuaternion(
                    &oMatrix,
                    &oQuat);
                    // Add in translation.
                oMatrix(3, 0) = oVector.x;
                oMatrix(3, 1) = oVector.y;
                oMatrix(3, 2) = oVector.z;
                    // Set interpolated key value.
                m_oValues.Set(iValueIndex + 0, oMatrix(0, 0), FALSE);
                m_oValues.Set(iValueIndex + 1, oMatrix(0, 1), FALSE);
                m_oValues.Set(iValueIndex + 2, oMatrix(0, 2), FALSE);
                m_oValues.Set(iValueIndex + 3, oMatrix(0, 3), FALSE);
                m_oValues.Set(iValueIndex + 4, oMatrix(1, 0), FALSE);
                m_oValues.Set(iValueIndex + 5, oMatrix(1, 1), FALSE);
                m_oValues.Set(iValueIndex + 6, oMatrix(1, 2), FALSE);
                m_oValues.Set(iValueIndex + 7, oMatrix(1, 3), FALSE);
                m_oValues.Set(iValueIndex + 8, oMatrix(2, 0), FALSE);
                m_oValues.Set(iValueIndex + 9, oMatrix(2, 1), FALSE);
                m_oValues.Set(iValueIndex + 10, oMatrix(2, 2), FALSE);
                m_oValues.Set(iValueIndex + 11, oMatrix(2, 3), FALSE);
                m_oValues.Set(iValueIndex + 12, oMatrix(3, 0) , FALSE);
                m_oValues.Set(iValueIndex + 13, oMatrix(3, 1), FALSE);
                m_oValues.Set(iValueIndex + 14, oMatrix(3, 2), FALSE);
                m_oValues.Set(iValueIndex + 15, oMatrix(3, 3), FALSE);
            }
            break;
        default:
            break;
    }
}

    // Output object body.
bool AnimationKey::OutputBody()
{
    int iIndex;
    int iCount = m_oTimes.Count();
    int iTime;
    D3DXVECTOR4 oValue;
    float *pValue;
    int iValueIndex;
    bool blRet = true;

        // Output animation type.
    blRet = blRet && OutputIntegerMember(m_iType);

        // Output animation key count.
    blRet = blRet && OutputIntegerMember(iCount);

        // Output times and values.
    for (iIndex = 0, pValue = m_oValues; iIndex < iCount; iIndex++)
    {
        blRet = blRet && OutputIndent();

            // Get time.
        iTime = m_oTimes.Indexed(iIndex);

            // Output key time.
        blRet = blRet && OutputIntegerListToken(iTime, ";");

            // Output key time and value count.
        blRet = blRet && OutputIntegerListToken(m_iValueCount, ";");

            // Output value.
        for (iValueIndex = 0; iValueIndex < m_iValueCount; iValueIndex++, pValue++)
        {
            if ((m_iValueCount > 4) && !(iValueIndex & 3))
            {
                blRet = blRet && OutputNewLine();
                blRet = blRet && OutputIndent();
            }

            blRet = blRet && OutputFloatListToken(
                *pValue,
                (iValueIndex == m_iValueCount - 1 ? ";;" : ","));
        }

        if (iIndex == iCount - 1)
            blRet = blRet && OutputSeparator(";\n");
        else
            blRet = blRet && OutputSeparator(",\n");
    }

    return(blRet);
}

    // Output object body.
bool AnimationOptions::OutputBody()
{
    bool blRet = true;

        // Output openclosed member.
    blRet = blRet && OutputIntegerMember(m_iOpenClosed);

        // Output positionquality member.
    blRet = blRet && OutputIntegerMember(m_iPositionQuality);

    return(blRet);
}

    // Output object body.
bool Animation::OutputBody()
{
    int iIndex;
    int iCount = m_oKeys.Count();
    bool blRet = true;

        // Output frame reference.
    if (m_opFrame)
        blRet = blRet && OutputReference(m_opFrame->Name());

        // Output bone matrix reference.
    if (m_opMatrix)
        blRet = blRet && OutputReference(m_opMatrix->Name());

    for (iIndex = 0; iIndex < iCount; iIndex++)
    {
        blRet = blRet && OutputNewLine();

            // Output animation.
        blRet = blRet && m_oKeys.Indexed(iIndex)->Output();

        if (m_oOptions.Count())
        {
            blRet = blRet && OutputNewLine();

                // Output options.
            blRet = blRet && m_oOptions.Indexed(iIndex)->Output();
        }
    }

    return(blRet);
}

    // Output object body.
bool AnimationSet::OutputBody()
{
    bool blRet = true;

        // Output animations.
    blRet = blRet && m_oAnimations.Output();

    return(blRet);
}

    // Constructor for a GUID as a string,
    // i.e. "{90D7FAE0-C724-11d3-8638-0000E856BADA}".

ClassID::ClassID(const char *cpGUIDString)
{
    SetName(cpGUIDString);
}


    // Constructor for a raw GUID.

ClassID::ClassID(const GUID& oGUID)
{
    SetID(oGUID);
}


    // Constructor for a raw GUID members.

ClassID::ClassID(
    DWORD dwData1,
    WORD wData2, WORD wData3,
    BYTE bData40, BYTE bData41, BYTE bData42, BYTE bData43,
    BYTE bData44, BYTE bData45, BYTE bData46, BYTE bData47)
{
    SetID(
        dwData1,
        wData2, wData3,
        bData40, bData41, bData42, bData43,
        bData44, bData45, bData46, bData47);
}


    // Copy constructor.

ClassID::ClassID(const ClassID& oOther) :
    Node(oOther),
    m_oGUID(oOther.m_oGUID)
{
    memcpy(m_caGUID, oOther.m_caGUID, sizeof(m_caGUID));
}


    // Default constructor.

ClassID::ClassID()
{
    OnClear();
}


    // Destructor.

ClassID::~ClassID()
{
}


    // Assignment operator.

ClassID& ClassID::operator=(const ClassID& oOther)
{
    Node::operator=(oOther);
    m_oGUID = oOther.m_oGUID;
    memcpy(m_caGUID, oOther.m_caGUID, sizeof(m_caGUID));
    return(*this);
}


    // Assignment operator.

ClassID& ClassID::operator=(const GUID& oGUID)
{
    SetID(oGUID);
    return(*this);
}


    // Assignment operator.

ClassID& ClassID::operator=(const char *cpGUIDString)
{
    SetName(cpGUIDString);
    return(*this);
}


    // This function returns the GUID as a string.

const char *ClassID::Name() const
{
    return(m_caGUID);
}


    // This function sets the GUID from a string.

void ClassID::SetName(const char *cpName)
{
    if (cpName)
    {
        strncpy(m_caGUID, cpName, kClassIDStringLength);
        m_caGUID[kClassIDStringLength] = '\0';

        DWORD dwData1 = 0;
        DWORD dwData2 = 0;
        DWORD dwData3 = 0;
        DWORD dwData4[8];
        int iStart = 0, iEnd = 0;

        sscanf(
            cpName,
            "%c%08X-%04X-%04X-%02X%02X-%02X%02X%02X%02X%02X%02X%c",
            &iStart,
            &dwData1,
            &dwData2,
            &dwData3,
            &dwData4[0],
            &dwData4[1],
            &dwData4[2],
            &dwData4[3],
            &dwData4[4],
            &dwData4[5],
            &dwData4[6],
            &dwData4[7],
            &iEnd);
        m_oGUID.Data1 = dwData1;
        m_oGUID.Data2 = (WORD)dwData2;
        m_oGUID.Data3 = (WORD)dwData3;
        m_oGUID.Data4[0] = (BYTE)dwData4[0];
        m_oGUID.Data4[1] = (BYTE)dwData4[1];
        m_oGUID.Data4[2] = (BYTE)dwData4[2];
        m_oGUID.Data4[3] = (BYTE)dwData4[3];
        m_oGUID.Data4[4] = (BYTE)dwData4[4];
        m_oGUID.Data4[5] = (BYTE)dwData4[5];
        m_oGUID.Data4[6] = (BYTE)dwData4[6];
        m_oGUID.Data4[7] = (BYTE)dwData4[7];
    }
    else
        OnClear();
}


    // This function sets the GUID from a reference.

void ClassID::SetID(const GUID& oGUID)
{
    m_oGUID = oGUID;
    sprintf(
        m_caGUID,
        "<%08X-%04X-%04X-%02X%02X-%02X%02X%02X%02X%02X%02X>",
        oGUID.Data1,
        oGUID.Data2,
        oGUID.Data3,
        oGUID.Data4[0],
        oGUID.Data4[1],
        oGUID.Data4[2],
        oGUID.Data4[3],
        oGUID.Data4[4],
        oGUID.Data4[5],
        oGUID.Data4[6],
        oGUID.Data4[7]);
}


    // This function sets the GUID from a pointer.

void ClassID::SetID(const GUID *pGUID)
{
    if (pGUID)
        SetID(*pGUID);
    else
        OnClear();
}


    // This function sets the GUID from the raw members.

void ClassID::SetID(
    DWORD dwData1,
    WORD wData2, WORD wData3,
    BYTE bData40, BYTE bData41, BYTE bData42, BYTE bData43,
    BYTE bData44, BYTE bData45, BYTE bData46, BYTE bData47)
{
    m_oGUID.Data1 = dwData1;
    m_oGUID.Data2 = wData2;
    m_oGUID.Data3 = wData3;
    m_oGUID.Data4[0] = bData40;
    m_oGUID.Data4[1] = bData41;
    m_oGUID.Data4[2] = bData42;
    m_oGUID.Data4[3] = bData43;
    m_oGUID.Data4[4] = bData44;
    m_oGUID.Data4[5] = bData45;
    m_oGUID.Data4[6] = bData46;
    m_oGUID.Data4[7] = bData47;
    sprintf(
        m_caGUID,
        "<%08X-%04X-%04X-%02X%02X-%02X%02X%02X%02X%02X%02X>",
        m_oGUID.Data1,
        m_oGUID.Data2,
        m_oGUID.Data3,
        m_oGUID.Data4[0],
        m_oGUID.Data4[1],
        m_oGUID.Data4[2],
        m_oGUID.Data4[3],
        m_oGUID.Data4[4],
        m_oGUID.Data4[5],
        m_oGUID.Data4[6],
        m_oGUID.Data4[7]);
}


    // Clear the GUID.

void ClassID::OnClear()
{
    memset(&m_oGUID, 0, sizeof(m_oGUID));
    strcpy(m_caGUID, "{00000000-0000-0000-0000-000000000000}");
}


    // Output the GUID.

bool ClassID::Output()
{
    bool blRet = true;

    if (m_blBinary)
    {
        blRet = blRet && OutputToken(TOKEN_GUID);

        blRet = blRet && OutputDWord(m_oGUID.Data1);
        blRet = blRet && OutputWord(m_oGUID.Data2);
        blRet = blRet && OutputWord(m_oGUID.Data3);
        blRet = blRet && OutputByte(m_oGUID.Data4[0]);
        blRet = blRet && OutputByte(m_oGUID.Data4[1]);
        blRet = blRet && OutputByte(m_oGUID.Data4[2]);
        blRet = blRet && OutputByte(m_oGUID.Data4[3]);
        blRet = blRet && OutputByte(m_oGUID.Data4[4]);
        blRet = blRet && OutputByte(m_oGUID.Data4[5]);
        blRet = blRet && OutputByte(m_oGUID.Data4[6]);
        blRet = blRet && OutputByte(m_oGUID.Data4[7]);
    }
    else
    {
        blRet = blRet && OutputIndent();
        blRet = blRet && OutputString(Name());
        blRet = blRet && OutputNewLine();
    }

    return(blRet);
}
