/*  Misfit Model 3D
 * 
 *  Copyright (c) 2004-2005 Kevin Worcester
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, 
 *  USA.
 *
 *  See the COPYING file for full license text.
 */


#include "ms3dfilter.h"

#include "weld.h"
#include "misc.h"
#include "log.h"
#include "endianconfig.h"

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <stdint.h>

#include <string>

using std::list;
using std::string;

char const Ms3dFilter::MAGIC_NUMBER[] = "MS3D000000";

/* 
	MS3D STRUCTURES 
*/

// byte-align structures
#ifdef _MSC_VER
#	pragma pack( push, packing )
#	pragma pack( 1 )
#	define PACK_STRUCT
#elif defined( __GNUC__ )
#	define PACK_STRUCT	__attribute__((packed))
#else
#	error you must byte-align these structures with the appropriate compiler directives
#endif

typedef uint8_t byte;
typedef uint16_t word;

// File header
struct MS3DHeader
{
	char m_ID[10];
	int32_t m_version;
} PACK_STRUCT;

// Vertex information
struct MS3DVertex
{
	byte m_flags;
	float m_vertex[3];
	char m_boneId;
	byte m_refCount;
} PACK_STRUCT;

// Triangle information
struct MS3DTriangle
{
	word m_flags;
	word m_vertexIndices[3];
	float m_vertexNormals[3][3];
	float m_s[3];
   float m_t[3];
	byte m_smoothingGroup;
	byte m_groupIndex;
} PACK_STRUCT;

// Material information
struct MS3DMaterial
{
    char m_name[32];
    float m_ambient[4];
    float m_diffuse[4];
    float m_specular[4];
    float m_emissive[4];
    float m_shininess;	// 0.0f - 128.0f
    float m_transparency;	// 0.0f - 1.0f
    byte m_mode;	// 0, 1, 2 is unused now
    char m_texture[128];
    char m_alphamap[128];
} PACK_STRUCT;

//	Joint information
struct MS3DJoint
{
	byte m_flags;
	char m_name[32];
	char m_parentName[32];
	float m_rotation[3];
	float m_translation[3];
	word m_numRotationKeyframes;
	word m_numTranslationKeyframes;
} PACK_STRUCT;

// Keyframe data
struct MS3DKeyframe
{
	float m_time;
	float m_parameter[3];
} PACK_STRUCT;

// Default alignment
#ifdef _MSC_VER
#	pragma pack( pop, packing )
#endif

#undef PACK_STRUCT

// flags 
//    1 = selected
//    2 = hidden
//

Model::ModelError Ms3dFilter::readFile( Model * model, const char * const filename )
{
   LOG_PROFILE();

   if ( model && filename )
   {
      FILE * fp = fopen( filename, "rb" );

      if ( fp == NULL )
      {
         if ( errno == ENOENT )
         {
            log_error( "%s: file does not exist\n", filename );
            return Model::ERROR_NO_FILE;
         }
         if ( errno == EPERM )
         {
            log_error( "%s: access denied\n", filename );
            return Model::ERROR_NO_ACCESS;
         }

         log_error( "%s: could not open file\n", filename );
         return Model::ERROR_FILE_OPEN;
      }

      string modelPath = "";
      string modelBaseName = "";
      string modelFullName = "";

      normalizePath( filename, modelFullName, modelPath, modelBaseName );
      
      model->setFilename( modelFullName.c_str() );

      vector<Model::Vertex *>   & modelVertices  = getVertexList( model );
      vector<Model::Triangle *> & modelTriangles = getTriangleList( model );
      vector<Model::Group *>    & modelGroups    = getGroupList( model );
      vector<Model::Material *> & modelMaterials = getMaterialList( model );
      vector<Model::Joint *>    & modelJoints    = getJointList( model );

      fseek( fp, 0, SEEK_END );
      unsigned fileLength = ftell( fp );
      fseek( fp, 0, SEEK_SET );

      // read whole file into memory
      uint8_t *buffer = new uint8_t[fileLength];

      if ( fread( buffer, fileLength, 1, fp ) != 1 )
      {
         delete[] buffer;
         log_error( "%s: could not read file\n", filename );
         fclose( fp );
         return Model::ERROR_FILE_READ;
      }

      fclose( fp );

      uint8_t *bufPtr = buffer;

      if ( fileLength < (sizeof( MS3DHeader ) + sizeof(word) ) )
      {
         delete[] buffer;
         return Model::ERROR_UNEXPECTED_EOF;
      }

      // Check header
      MS3DHeader * header = (MS3DHeader *) bufPtr;

      bufPtr += sizeof( MS3DHeader );

      if ( strncmp( header->m_ID, MAGIC_NUMBER, 10 ) != 0 )
      {
         delete[] buffer;
         log_error( "bad magic number\n" );
         return Model::ERROR_BAD_MAGIC;
      }

                int version = ltoh_32(header->m_version);
      if ( version < 3 || version > 4 )
      {
         delete[] buffer;
         log_error( "unsupported version\n" );
         return Model::ERROR_UNSUPPORTED_VERSION;
      }

      int numVertices = ltoh_u16(*((word*) bufPtr));
      bufPtr += sizeof( word );

      int t; // MS Visual C++ is lame

      for ( t = 0; t < numVertices; t++ )
      {
         MS3DVertex * vertex = ( MS3DVertex * ) bufPtr;
         bufPtr += sizeof( MS3DVertex );

         Model::Vertex * vert = Model::Vertex::get();
         vert->m_boneId = vertex->m_boneId;
         for ( int v = 0; v < 3; v++ )
         {
            vert->m_coord[v] = ltoh_float(vertex->m_vertex[ v ]);
         }
         modelVertices.push_back( vert );
      }

      int numTriangles = ltoh_u16(*((word*) bufPtr));
      bufPtr += sizeof( word );

      if ( (fileLength - ( bufPtr - buffer )) < (sizeof ( MS3DTriangle ) * numTriangles + sizeof( word )) )
      {
         delete[] buffer;
         return Model::ERROR_UNEXPECTED_EOF;
      }

      for ( t = 0; t < numTriangles; t++ )
      {
         Model::Triangle * curTriangle = Model::Triangle::get();
         MS3DTriangle * triangle = ( MS3DTriangle * ) bufPtr;
         bufPtr += sizeof( MS3DTriangle );

         curTriangle->m_vertexIndices[0] = ltoh_u16(triangle->m_vertexIndices[0]);
         curTriangle->m_vertexIndices[1] = ltoh_u16(triangle->m_vertexIndices[1]);
         curTriangle->m_vertexIndices[2] = ltoh_u16(triangle->m_vertexIndices[2]);

         for ( int i = 0; i < 3; i++ )
         {
            // Need to invert the T coord, since milkshape seems to store it
            // upside-down.
            curTriangle->m_s[i] = ltoh_float(triangle->m_s[i]);
            curTriangle->m_t[i] = 1.0 - ltoh_float(triangle->m_t[i]);
         }

         for ( int y = 0; y < 3; y++ )
         {
            for( int x = 0; x < 3; x++ )
            {
               curTriangle->m_vertexNormals[y][x] = ltoh_float(triangle->m_vertexNormals[y][x]);
            }
         }

         modelTriangles.push_back( curTriangle );
      }

      int numGroups = ltoh_u16(*((word*) bufPtr));
      bufPtr += sizeof( word );

      for ( t = 0; t < numGroups; t++ )
      {
         Model::Group * group = Model::Group::get();
         modelGroups.push_back( group );

         bufPtr += sizeof( byte ); // flags

         char tempstr[32];
         strncpy( tempstr, (char *) bufPtr, 32 );
         tempstr[ 32 - 1 ] = '\0';
         group->m_name = tempstr;

         log_debug( "group name: %s\n", modelGroups[t]->m_name.c_str() );

         bufPtr += sizeof( byte ) * 32;

         int numTriangles = ltoh_u16(*((word*) bufPtr));
         bufPtr += sizeof( word );

         for ( int n = 0; n < numTriangles; n++ )
         {
            group->m_triangleIndices.push_back( ltoh_u16(*((word*) bufPtr)) );
            bufPtr += sizeof( word );
         }

         group->m_materialIndex = (int8_t) *((byte*) bufPtr);
         bufPtr += sizeof( byte );

         // Already added group to m_groups
      }

      int numMaterials = ltoh_u16(*((word*) bufPtr));
      bufPtr += sizeof( word );
      log_debug( "model says %d materials\n", numMaterials );

      list<int> badList;

      for ( t = 0; t < numMaterials; t++ )
      {
         Model::Material * mat = Model::Material::get();
         MS3DMaterial * material = (MS3DMaterial *) bufPtr;
         bufPtr += sizeof( MS3DMaterial );

         mat->m_name = material->m_name;

                        mat->m_ambient[0] = ltoh_float(material->m_ambient[0]); mat->m_ambient[1] = ltoh_float(material->m_ambient[1]);
                        mat->m_ambient[2] = ltoh_float(material->m_ambient[2]); mat->m_ambient[3] = ltoh_float(material->m_ambient[3]);
                        mat->m_diffuse[0] = ltoh_float(material->m_diffuse[0]); mat->m_diffuse[1] = ltoh_float(material->m_diffuse[1]);
                        mat->m_diffuse[2] = ltoh_float(material->m_diffuse[2]); mat->m_diffuse[3] = ltoh_float(material->m_diffuse[3]);
                        mat->m_specular[0] = ltoh_float(material->m_specular[0]); mat->m_specular[1] = ltoh_float(material->m_specular[1]);
                        mat->m_specular[2] = ltoh_float(material->m_specular[2]); mat->m_specular[3] = ltoh_float(material->m_specular[3]);
                        mat->m_emissive[0] = ltoh_float(material->m_emissive[0]); mat->m_emissive[1] = ltoh_float(material->m_emissive[1]);
                        mat->m_emissive[2] = ltoh_float(material->m_emissive[2]); mat->m_emissive[3] = ltoh_float(material->m_emissive[3]);
         mat->m_shininess = ltoh_float(material->m_shininess);

         if ( material->m_texture[0] == '\0' )
         {
            log_warning( "**** MODEL FILE IS BROKEN ****\n" );
            badList.push_back(t);
         }

         replaceBackslash( material->m_texture );
         replaceBackslash( material->m_alphamap );

         // Get absolute path for texture
         string texturePath = material->m_texture;

         texturePath = fixAbsolutePath( modelPath.c_str(), texturePath.c_str() );
         texturePath = getAbsolutePath( modelPath.c_str(), texturePath.c_str() );

         mat->m_filename  = texturePath;

         // Get absolute path for alpha map
         texturePath = material->m_alphamap;

         if ( texturePath.length() > 0 )
         {
            texturePath = fixAbsolutePath( modelPath.c_str(), texturePath.c_str() );
            texturePath = getAbsolutePath( modelPath.c_str(), texturePath.c_str() );
         }

         mat->m_alphaFilename = texturePath;

         modelMaterials.push_back( mat );
      }

      for ( list<int>::iterator it = badList.begin(); it != badList.end(); it++ )
      {
         model->deleteTexture( *it );
      }

      //model->loadTextures();

      setModelInitialized( model, true );
      //model->m_initialized = true;

      float fps = ltoh_float(*((float*) bufPtr));
      bufPtr += sizeof(float);

      // current time, don't need
      //float currentTime = *((float*) bufPtr);
      bufPtr += sizeof( float );

      int numFrames = ltoh_32(*((int*) bufPtr));
      bufPtr += sizeof( int );

      setModelNumFrames( model, numFrames );

      if ( numFrames > 0 )
      {
         model->addAnimation( Model::ANIMMODE_SKELETAL, "Keyframe" );
         model->setAnimFPS( Model::ANIMMODE_SKELETAL, 0, fps );
         model->setAnimFrameCount( Model::ANIMMODE_SKELETAL, 0, numFrames );
      }

      word numJoints = ltoh_u16(*((word *) bufPtr));
      bufPtr += sizeof( word );

      struct JointNameListRec_t
      {
         int m_jointIndex;
         const char * m_name;
      };

      typedef struct JointNameListRec_t JointNameListRec;

      const uint8_t * tempPtr = bufPtr;
      JointNameListRec * nameList = new JointNameListRec[ numJoints ];
      for ( t = 0; t < numJoints; t++ )
      {
         MS3DJoint * joint = (MS3DJoint *) tempPtr;
         tempPtr += sizeof( MS3DJoint );
         tempPtr += ( sizeof( MS3DKeyframe ) * (ltoh_u16(joint->m_numRotationKeyframes) + ltoh_u16(joint->m_numTranslationKeyframes)) );

         nameList[t].m_jointIndex = t;
         nameList[t].m_name = joint->m_name;
      }

      for ( t = 0; t < numJoints; t++ )
      {
         MS3DJoint * joint = (MS3DJoint *) bufPtr;
         bufPtr += sizeof( MS3DJoint );

         int parentIndex = -1;
         if ( strlen( joint->m_parentName ) > 0 )
         {
            for ( int j = 0; j < numJoints; j++ )
            {
               if ( strcasecmp( nameList[j].m_name, joint->m_parentName ) == 0 )
               {
                  parentIndex = nameList[j].m_jointIndex;
                  break;
               }
            }

            if ( parentIndex == -1 )
            {
               delete[] buffer;
               log_error( "No parent\n" );
               return Model::ERROR_BAD_DATA; // no parent!
            }
         }

         modelJoints.push_back( Model::Joint::get() );

         for ( int i = 0; i < 3; i++ )
         {
            modelJoints[t]->m_localRotation[i]    = ltoh_float(joint->m_rotation[i]);
            modelJoints[t]->m_localTranslation[i] = ltoh_float(joint->m_translation[i]);
         }
         modelJoints[t]->m_parent = parentIndex;
         modelJoints[t]->m_name = joint->m_name;

         int numRotationKeyframes = ltoh_u16(joint->m_numRotationKeyframes);
         int numTranslationKeyframes = ltoh_u16(joint->m_numTranslationKeyframes);

         int j; // MS Visual C++ is lame
         for ( j = 0; j < numRotationKeyframes; j++ )
         {
            Model::Keyframe * mkeyframe = Model::Keyframe::get();
            MS3DKeyframe * keyframe = ( MS3DKeyframe * ) bufPtr;
            bufPtr += sizeof( MS3DKeyframe );

            mkeyframe->m_jointIndex = t;
            mkeyframe->m_time = ltoh_float(keyframe->m_time);
            for ( int i = 0; i < 3; i++ )
            {
               mkeyframe->m_parameter[i] = ltoh_float(keyframe->m_parameter[i]);
            }

            unsigned frame = (unsigned) (ltoh_float(keyframe->m_time) / (1.0 / fps)) - 1;

            model->setSkelAnimKeyframe( 0, frame, t, true,
                  mkeyframe->m_parameter[0], mkeyframe->m_parameter[1], mkeyframe->m_parameter[2] );

            mkeyframe->release();
         }
         for ( j = 0; j < numTranslationKeyframes; j++ )
         {
            Model::Keyframe * mkeyframe = Model::Keyframe::get();
            MS3DKeyframe * keyframe = ( MS3DKeyframe * ) bufPtr;
            bufPtr += sizeof( MS3DKeyframe );

            //model->setJointKeyframe( t, j, keyframe->m_time*1000.0f, keyframe->m_parameter, false );

            mkeyframe->m_jointIndex = t;
            mkeyframe->m_time = ltoh_float(keyframe->m_time);
            for ( int i = 0; i < 3; i++ )
            {
               mkeyframe->m_parameter[i] = ltoh_float(keyframe->m_parameter[i]);
            }

            unsigned frame = (unsigned) (ltoh_float(keyframe->m_time) / (1.0 / fps)) - 1;

            model->setSkelAnimKeyframe( 0, frame, t, false,
                  mkeyframe->m_parameter[0], mkeyframe->m_parameter[1], mkeyframe->m_parameter[2] );

            mkeyframe->release();
         }
      }

      log_debug( "model loaded\n" );
      log_debug( "  vertices:  %d\n", numVertices );
      log_debug( "  triangles: %d\n", numTriangles );
      log_debug( "  groups:    %d\n", numGroups );
      log_debug( "  materials: %d\n", numMaterials );
      log_debug( "  joints:    %d\n", numJoints );
      log_debug( "\n" );

      delete[] nameList;

      model->setupJoints();

      //writeFile( model, "/home/kevin/src/opengl/3dmodel/test.ms3d" );

      delete[] buffer;
      return Model::ERROR_NONE;
   }
   else
   {
      return Model::ERROR_BAD_ARGUMENT;
   }
}

Model::ModelError Ms3dFilter::writeFile( Model * model, const char * const filename, ModelFilter::Options * o )
{
   LOG_PROFILE();

   if ( model == NULL || filename == NULL || filename[0] == '\0' )
   {
      return Model::ERROR_BAD_ARGUMENT;
   }

   // Check for identical bone joint names
   {
      unsigned c = model->getBoneJointCount();
      for ( unsigned i = 0; i < c; i++ )
      {
         for ( unsigned j = i+1; j < c; j++ )
         {
            if ( strcmp( model->getBoneJointName( i ), model->getBoneJointName( j ) ) == 0 )
            {
               model->setFilterSpecificError( "Bone joints must have unique names." );
               return Model::ERROR_FILTER_SPECIFIC;
            }
         }
      }
   }

   FILE * fp = fopen( filename, "wb" );

   if ( fp )
   {
      string modelPath = "";
      string modelBaseName = "";
      string modelFullName = "";

      normalizePath( filename, modelFullName, modelPath, modelBaseName );

      // Groups don't share vertices with Milk Shape 3D
      {
         unsigned t = 0;
         unsigned vcount = model->getVertexCount();

         for ( t = 0; t < vcount; t++ )
         {
            model->selectVertex( t );
         }

         unweldSelectedVertices( model );

         unsigned tcount = model->getTriangleCount();
         for ( t = 0; t < tcount; t++ )
         {
            model->selectTriangle( t );
         }

         unsigned gcount = model->getGroupCount();
         for ( t = 0; t < gcount; t++ )
         {
            model->unselectGroup( t );
         }

         list<int> tris = model->getSelectedTriangles();
         if ( !tris.empty() )
         {
            weldSelectedVertices( model );
            int ungrouped = model->addGroup( "Ungrouped" );
            model->addSelectedToGroup( ungrouped );
         }

         // gcount doesn't include the group we just created,
         // so this operation does not re-weld vertices we just 
         // operated on.
         for ( t = 0; t < gcount; t++ )
         {
            model->unselectAll();
            model->selectGroup( t );
            weldSelectedVertices( model );
         }

         model->unselectAll();
         model->operationComplete( "Changes for save" );
      }
      
      vector<Model::Vertex *>   & modelVertices  = getVertexList( model );
      vector<Model::Triangle *> & modelTriangles = getTriangleList( model );
      vector<Model::Group *>    & modelGroups    = getGroupList( model );
      vector<Model::Material *> & modelMaterials = getMaterialList( model );
      vector<Model::Joint *>    & modelJoints    = getJointList( model );

      int32_t version = htol_32(4);
      fwrite( MAGIC_NUMBER, sizeof(char), strlen(MAGIC_NUMBER), fp );
      fwrite( &version, sizeof(int), 1, fp );

      int t;

      // write vertices
      word numVertices = (int) modelVertices.size();
                uint16_t temp16 = htol_u16(numVertices);
      fwrite( &temp16, sizeof(numVertices), 1, fp );
      for ( t = 0; t < numVertices; t++ )
      {
         Model::Vertex * mvert = modelVertices[t];
         MS3DVertex vert;
         byte refcount = 0;

         vert.m_flags = 1;
         for ( int n = 0; n < 3; n++ )
         {
            vert.m_vertex[n] = htol_float(mvert->m_coord[n]);
         }
         vert.m_boneId = mvert->m_boneId;

         for ( unsigned tri = 0; tri < modelTriangles.size(); tri++ )
         {
            for ( unsigned v = 0; v < 3; v++ )
            {
               if ( modelTriangles[tri]->m_vertexIndices[v] == (unsigned) t )
               {
                  refcount++;
               }
            }
         }

         vert.m_refCount = refcount;

         fwrite( &vert, sizeof(vert), 1, fp );
      }

      // write triangles
      word numTriangles = (int) modelTriangles.size();
                temp16 = htol_u16(numTriangles);
      fwrite( &temp16, sizeof(numTriangles), 1, fp );
      for ( t = 0; t < numTriangles; t++ )
      {
         Model::Triangle * mtri = modelTriangles[t];
         MS3DTriangle tri;

         tri.m_flags = htol_u16(1);
         for ( int v = 0; v < 3; v++ )
         {
            tri.m_vertexIndices[v] = htol_u16(mtri->m_vertexIndices[v]);
            tri.m_s[v] = htol_float(mtri->m_s[v]);
            tri.m_t[v] = htol_float(1.0 - mtri->m_t[v]);

            for ( int n = 0; n < 3; n++ )
            {
               tri.m_vertexNormals[v][n] = htol_float(mtri->m_vertexNormals[v][n]);
            }
         }

         for ( unsigned g = 0; g < modelGroups.size(); g++ )
         {
            for ( unsigned n = 0; n < modelGroups[g]->m_triangleIndices.size(); n++ )
            {
               if ( modelGroups[g]->m_triangleIndices[n] == t )
               {
                  tri.m_groupIndex = g;
                  break;
               }
            }
         }
         tri.m_smoothingGroup = 0;

         fwrite( &tri, sizeof(tri), 1, fp );
      }

      word numGroups = modelGroups.size();
                temp16 = htol_u16(numGroups);
      fwrite( &temp16, sizeof(numGroups), 1, fp );
      for ( t = 0; t < numGroups; t++ )
      {
         Model::Group * grp = modelGroups[t];
         byte flags = 0;
         fwrite( &flags, sizeof(flags), 1, fp );

         char groupname[32];
         strncpy( groupname, grp->m_name.c_str(), sizeof(groupname) );
         fwrite( groupname, sizeof(groupname), 1, fp );

         word numTriangles = grp->m_triangleIndices.size();
                        temp16 = htol_u16(numTriangles);
         fwrite( &temp16, sizeof(numTriangles), 1, fp );
         for ( int n = 0; n < numTriangles; n++ )
         {
            word index = grp->m_triangleIndices[n];
                                temp16 = htol_u16(index);
            fwrite( &temp16, sizeof(index), 1, fp );
         }

         byte material = grp->m_materialIndex;
         fwrite( &material, sizeof(material), 1, fp );
      }

      word numMaterials = modelMaterials.size();
                temp16 = htol_u16(numMaterials);
      fwrite( &temp16, sizeof(numMaterials), 1, fp );
      for ( t = 0; t < numMaterials; t++ )
      {
         Model::Material * mmat = modelMaterials[t];
         MS3DMaterial mat;

         strncpy( mat.m_name, mmat->m_name.c_str(), sizeof( mat.m_name ) );
         mat.m_name[ sizeof(mat.m_name) - 1 ]  = '\0';

         string texturePath;
         texturePath = getRelativePath( modelPath.c_str(), mmat->m_filename.c_str() );

         strncpy( mat.m_texture, texturePath.c_str(), sizeof( mat.m_texture ) );
         mat.m_texture[ sizeof(mat.m_texture) - 1 ]  = '\0';

         texturePath = getRelativePath( modelPath.c_str(), mmat->m_alphaFilename.c_str() );

         strncpy( mat.m_alphamap, texturePath.c_str(), sizeof( mat.m_alphamap ) );
         mat.m_alphamap[ sizeof(mat.m_alphamap) - 1 ]  = '\0';

         replaceSlash( mat.m_texture );
         replaceSlash( mat.m_alphamap );

         for ( int n = 0; n < 4; n++ )
         {
            mat.m_ambient[n]  = htol_float(mmat->m_ambient[n]);
            mat.m_diffuse[n]  = htol_float(mmat->m_diffuse[n]);
            mat.m_specular[n] = htol_float(mmat->m_specular[n]);
            mat.m_emissive[n] = htol_float(mmat->m_emissive[n]);
         }

         mat.m_shininess    = htol_float(mmat->m_shininess);
         mat.m_transparency = htol_float(1.0);
         mat.m_mode = 0;

         fwrite( &mat, sizeof(mat), 1, fp );
      }

      float fps = 30.0f;
      if ( model->getAnimCount( Model::ANIMMODE_SKELETAL ) > 0 )
      {
         fps = model->getAnimFPS( Model::ANIMMODE_SKELETAL, 0 );
      }
      if ( fps <= 0.0 )
      {
         fps = 30.0f;
      }

      float currentTime = 1.0;
      int32_t numFrames = model->getNumFrames();

      double spf = 1.0 / fps;

      float tempfloat = htol_float(fps);
                fwrite( &tempfloat, sizeof(fps), 1, fp );
      tempfloat = htol_float(currentTime);
      fwrite( &tempfloat, sizeof(currentTime), 1, fp );
      int32_t temp32 = htol_32(numFrames);
      fwrite( &temp32, sizeof(numFrames), 1, fp );

      word numJoints = modelJoints.size();
                temp16 = htol_16(numJoints);
      fwrite( &temp16, sizeof(numJoints), 1, fp );
      for ( t = 0; t < numJoints; t++ )
      {
         Model::Joint * mjoint = modelJoints[t];
         MS3DJoint joint;

         strncpy( joint.m_name, mjoint->m_name.c_str(), sizeof(joint.m_name) );
         joint.m_name[ sizeof( joint.m_name ) - 1 ]= '\0';
         if ( mjoint->m_parent >= 0 )
         {
            strncpy( joint.m_parentName, modelJoints[ mjoint->m_parent ]->m_name.c_str(), sizeof(joint.m_parentName) );
            joint.m_parentName[ sizeof( joint.m_parentName ) - 1 ]= '\0';
         }
         else
         {
            joint.m_parentName[0] = '\0';
         }

         joint.m_flags = 8;

         for ( int i = 0; i < 3; i++ )
         {
            joint.m_rotation[i] = htol_float(mjoint->m_localRotation[i]);
            joint.m_translation[i] = htol_float(mjoint->m_localTranslation[i]);
         }

         unsigned animcount = model->getAnimCount( Model::ANIMMODE_SKELETAL );
         unsigned framecount = 0;
         unsigned prevcount = 0;

         unsigned a = 0;
         unsigned f = 0;

         unsigned rotcount   = 0;
         unsigned transcount = 0;

         double x = 0;
         double y = 0;
         double z = 0;

         for ( a = 0; a < animcount; a++ )
         {
            framecount = model->getAnimFrameCount( Model::ANIMMODE_SKELETAL, a );

            for ( f = 0; f < framecount; f++ )
            {
               if ( model->getSkelAnimKeyframe( a, f, t, true, x, y, z ) )
               {
                  rotcount++;
               }
               if ( model->getSkelAnimKeyframe( a, f, t, false, x, y, z ) )
               {
                  transcount++;
               }
            }
         }

         joint.m_numRotationKeyframes = htol_u16(rotcount);
         joint.m_numTranslationKeyframes = htol_u16(transcount);

         log_debug( "rotation: %d\n", rotcount );
         log_debug( "translation: %d\n", transcount );

         fwrite( &joint, sizeof(joint), 1, fp );

         // Rotation keyframes
         prevcount = 0;
         for ( a = 0; a < animcount; a++ )
         {
            framecount = model->getAnimFrameCount( Model::ANIMMODE_SKELETAL, a );

            for ( f = 0; f < framecount; f++ )
            {
               if ( model->getSkelAnimKeyframe( a, f, t, true, x, y, z ) )
               {
                  MS3DKeyframe keyframe;
                  keyframe.m_time = ((double) (prevcount + f + 1)) * spf;
                  log_debug( "keyframe time: %f\n", keyframe.m_time );

                                                keyframe.m_time = htol_float(keyframe.m_time);
                  keyframe.m_parameter[0] = htol_float(x);
                  keyframe.m_parameter[1] = htol_float(y);
                  keyframe.m_parameter[2] = htol_float(z);

                  fwrite( &keyframe, sizeof(keyframe), 1, fp );
               }
            }

            prevcount += framecount;
         }

         // Translation keyframes
         prevcount = 0;
         for ( a = 0; a < animcount; a++ )
         {
            framecount = model->getAnimFrameCount( Model::ANIMMODE_SKELETAL, a );

            for ( f = 0; f < framecount; f++ )
            {
               if ( model->getSkelAnimKeyframe( a, f, t, false, x, y, z ) )
               {
                  MS3DKeyframe keyframe;
                  keyframe.m_time = ((double) (prevcount + f + 1)) * spf;
                  log_debug( "keyframe time: %f\n", keyframe.m_time );

                                                keyframe.m_time = htol_float(keyframe.m_time);
                  keyframe.m_parameter[0] = htol_float(x);
                  keyframe.m_parameter[1] = htol_float(y);
                  keyframe.m_parameter[2] = htol_float(z);

                  fwrite( &keyframe, sizeof(keyframe), 1, fp );
               }
            }
         }

         prevcount += framecount;
      }

      fclose( fp );
   }
   else
   {
      if ( errno == ENOENT )
      {
         log_error( "%s: file does not exist\n", filename );
         return Model::ERROR_NO_FILE;
      }
      if ( errno == EPERM )
      {
         log_error( "%s: access denied\n", filename );
         return Model::ERROR_NO_ACCESS;
      }

      log_error( "%s: could not open file\n", filename );
      return Model::ERROR_FILE_OPEN;
   }

   return Model::Model::ERROR_NONE;
}

bool Ms3dFilter::isSupported( const char * filename )
{
   if ( filename )
   {
      unsigned len = strlen( filename );

      if ( len >= 5 && strcasecmp( &filename[len-5], ".ms3d" ) == 0 )
      {
         return true;
      }
   }

   return false;
}

list<string> Ms3dFilter::getReadTypes()
{
   list<string> rval;

   rval.push_back( "*.ms3d" );

   return rval;
}

list<string> Ms3dFilter::getWriteTypes()
{
   list<string> rval;

   rval.push_back( "*.ms3d" );

   return rval;
}

