/*  Misfit Model 3D
 * 
 *  Copyright (c) 2004-2005 Kevin Worcester
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, 
 *  USA.
 *
 *  See the COPYING file for full license text.
 */


#include "model.h"

#ifdef MM3D_EDIT
#include "modelundo.h"
#endif // MM3D_EDIT

#include "log.h"
#include "glmath.h"
#include <math.h>

#ifdef MM3D_EDIT

int Model::addAnimation( const AnimationMode & m, const char * name )
{
   LOG_PROFILE();

   int num = -1;
   if ( name )
   {
      switch ( m )
      {
         case ANIMMODE_SKELETAL:
            {
               num = m_skelAnims.size();

               SkelAnim * anim = SkelAnim::get();
               anim->m_name = name;
               anim->m_fps  = 30.0;
               anim->m_spf  = (1.0 / anim->m_fps);
               anim->m_frameCount = 1;
               anim->m_validNormals = false;

               for ( unsigned j = 0; j < m_joints.size(); j++ )
               {
                  anim->m_jointKeyframes.push_back( KeyframeList() );
               }

               MU_AddAnimation * undo = new MU_AddAnimation();
               undo->addAnimation( num, anim );
               sendUndo( undo );

               insertSkelAnim( num, anim );
            }

            break;
         case ANIMMODE_FRAME:
            {
               num = m_frameAnims.size();

               FrameAnim * anim = FrameAnim::get();
               anim->m_name = name;
               anim->m_fps = 10.0;
               anim->m_validNormals = false;

               MU_AddAnimation * undo = new MU_AddAnimation();
               undo->addAnimation( num, anim );
               sendUndo( undo );

               insertFrameAnim( num, anim );
            }

            break;
         default:
            break;
      }
   }
   return num;
}

void Model::deleteAnimation( const AnimationMode & m, const unsigned & index )
{
   LOG_PROFILE();

   switch ( m )
   {
      case ANIMMODE_SKELETAL:
         if ( index < m_skelAnims.size() )
         {
            MU_DeleteAnimation * undo = new MU_DeleteAnimation;
            undo->deleteAnimation( index, m_skelAnims[index] );
            sendUndo( undo );

            removeSkelAnim( index );
         }
         break;
      case ANIMMODE_FRAME:
         if ( index < m_frameAnims.size() )
         {
            MU_DeleteAnimation * undo = new MU_DeleteAnimation;
            undo->deleteAnimation( index, m_frameAnims[index] );
            sendUndo( undo );

            removeFrameAnim( index );
         }
         break;
      default:
         break;
   }
}

bool Model::setAnimName( const AnimationMode & m, const unsigned & anim, const char * name )
{
   switch ( m )
   {
      case ANIMMODE_SKELETAL:
         if ( anim < m_skelAnims.size() )
         {
            MU_SetAnimName * undo = new MU_SetAnimName();
            undo->setName( m_animationMode, anim, name, m_skelAnims[anim]->m_name.c_str() );
            sendUndo( undo );

            m_skelAnims[ anim ]->m_name = name;
            return true;
         }
         break;
      case ANIMMODE_FRAME:
         if ( anim < m_frameAnims.size() )
         {
            MU_SetAnimName * undo = new MU_SetAnimName();
            undo->setName( m_animationMode, anim, name, m_frameAnims[anim]->m_name.c_str() );
            sendUndo( undo );

            m_frameAnims[ anim ]->m_name = name;
            return true;
         }
         break;
      default:
         break;
   }
   return false;
}

bool Model::setAnimFrameCount( const AnimationMode & m, const unsigned & anim, const unsigned & count )
{
   switch ( m )
   {
      case ANIMMODE_SKELETAL:
         if ( anim < m_skelAnims.size() )
         {
            SkelAnim * sa = m_skelAnims[anim];

            for ( unsigned j = 0; j < m_joints.size(); j++ )
            {
               KeyframeList & list = sa->m_jointKeyframes[j];
               unsigned k = 0;
               while ( k < list.size() )
               {
                  if ( list[k]->m_frame >= count )
                  {
                     deleteSkelAnimKeyframe( anim, list[k]->m_frame, list[k]->m_jointIndex, list[k]->m_isRotation );
                  }
                  else
                  {
                     k++;
                  }
               }
            }

            MU_SetAnimFrameCount * undo = new MU_SetAnimFrameCount();
            undo->setAnimFrameCount( m, anim, count, m_skelAnims[anim]->m_frameCount );
            sendUndo( undo );

            m_skelAnims[anim]->m_frameCount = count;

            if ( m_currentAnim == anim && m_currentFrame >= count )
            {
               setCurrentAnimationFrame( 0 );
            }
            return true;
         }
         break;
      case ANIMMODE_FRAME:
         if ( anim < m_frameAnims.size() )
         {
            FrameAnim * fa = m_frameAnims[anim];

            if ( count == fa->m_frameVertices.size() )
            {
               return true;
            }

            if ( count < fa->m_frameVertices.size() )
            {
               MU_DeleteFrameAnimFrame * undo = new MU_DeleteFrameAnimFrame;
               undo->setAnimationData( anim );

               while ( count < fa->m_frameVertices.size() )
               {
                  unsigned off = fa->m_frameVertices.size() - 1;
                  removeFrameAnimFrame( anim, off );
                  undo->deleteFrame( off, fa->m_frameVertices[off] );
               }

               sendUndo( undo );
            }
            if ( count > fa->m_frameVertices.size() )
            {
               MU_AddFrameAnimFrame * undo = new MU_AddFrameAnimFrame;
               undo->setAnimationData( anim );

               while ( count > fa->m_frameVertices.size() )
               {
                  fa->m_validNormals = false;
                  FrameAnimVertexList * list = new FrameAnimVertexList;

                  undo->addFrame( fa->m_frameVertices.size(), list );
                  insertFrameAnimFrame( anim, fa->m_frameVertices.size(), list );

                  for ( unsigned t = 0; t < m_vertices.size(); t++ )
                  {
                     FrameAnimVertex * fav = FrameAnimVertex::get();
                     for ( unsigned v = 0; v < 3; v++ )
                     {
                        fav->m_coord[v] = m_vertices[t]->m_coord[v];
                     }
                     list->push_back( fav );
                  }
               }

               sendUndo( undo );
            }

            return true;
         }
         break;
      default:
         break;
   }

   return false;
}

bool Model::setAnimFPS( const AnimationMode & m, const unsigned & anim, const double & fps )
{
   switch ( m )
   {
      case ANIMMODE_SKELETAL:
         if ( anim < m_skelAnims.size() )
         {
            MU_SetAnimFPS * undo = new MU_SetAnimFPS();
            undo->setFPS( m, anim, fps, m_skelAnims[anim]->m_fps );
            sendUndo( undo );

            m_skelAnims[anim]->m_fps = fps;
            m_skelAnims[anim]->m_spf = (1.0 / fps);

            Keyframe * kf;
            for ( unsigned j = 0; j < m_skelAnims[anim]->m_jointKeyframes.size(); j++ )
            {
               for ( unsigned f = 0; f < m_skelAnims[anim]->m_jointKeyframes[j].size(); f++ )
               {
                  kf = m_skelAnims[anim]->m_jointKeyframes[j][f];
                  kf->m_time = m_skelAnims[anim]->m_spf * kf->m_frame;
               }
            }
            return true;
         }
         break;
      case ANIMMODE_FRAME:
         if ( anim < m_frameAnims.size() )
         {
            MU_SetAnimFPS * undo = new MU_SetAnimFPS();
            undo->setFPS( m, anim, fps, m_frameAnims[anim]->m_fps );
            sendUndo( undo );

            m_frameAnims[anim]->m_fps = fps;
            return true;
         }
         break;
      default:
         break;
   }

   return false;
}

void Model::setFrameAnimVertexCount( const unsigned & vertexCount )
{
   unsigned anim = 0;
   unsigned frame = 0;

   unsigned oldCount = 0;

   unsigned acount = m_frameAnims.size();
   for ( anim = 0; anim < acount; anim++ )
   {
      unsigned fcount = m_frameAnims[ anim ]->m_frameVertices.size();
      oldCount = m_frameAnims[anim]->m_frameVertices[0]->size();
      for ( frame = 0; frame < fcount; frame++ )
      {
         FrameAnimVertex * fav = NULL;
         unsigned vcount = m_frameAnims[anim]->m_frameVertices[frame]->size();
         while ( vertexCount > vcount )
         {
            fav = FrameAnimVertex::get();
            m_frameAnims[anim]->m_frameVertices[frame]->push_back( fav );
            vcount++;
         }
         while ( vertexCount < vcount )
         {
            m_frameAnims[anim]->m_frameVertices[frame]->back()->release();
            m_frameAnims[anim]->m_frameVertices[frame]->pop_back();
            vcount--;
         }
      }
   }

   MU_SetFrameAnimVertexCount * undo = new MU_SetFrameAnimVertexCount();
   undo->setCount( vertexCount, oldCount );
   sendUndo( undo );
}

bool Model::setFrameAnimVertexCoords( const unsigned & anim, const unsigned & frame, const unsigned & vertex,
      const double & x, const double & y, const double & z )
{
   if ( anim < m_frameAnims.size() 
         && frame < m_frameAnims[anim]->m_frameVertices.size()
         && vertex < m_vertices.size() )
   {
      FrameAnimVertex * fav = NULL;

      while ( vertex >= m_frameAnims[anim]->m_frameVertices[frame]->size() )
      {
         fav = FrameAnimVertex::get();
         m_frameAnims[anim]->m_frameVertices[frame]->push_back( fav );
      }

      fav = (*m_frameAnims[anim]->m_frameVertices[frame])[vertex];

      MU_MoveFrameVertex * undo = new MU_MoveFrameVertex;
      undo->setAnimationData( anim, frame );
      undo->addVertex( vertex, x, y, z, fav->m_coord[0], fav->m_coord[1], fav->m_coord[2] );
      sendUndo( undo );

      fav->m_coord[0] = x;
      fav->m_coord[1] = y;
      fav->m_coord[2] = z;

      m_frameAnims[anim]->m_validNormals = false;
      return true;
   }
   else
   {
      return false;
   }
}

int Model::copyAnimation( const AnimationMode & mode, const unsigned & anim, const char * newName )
{
   int num = -1;
   switch ( mode )
   {
      case ANIMMODE_SKELETAL:
         if ( anim < m_skelAnims.size() )
         {
            num = addAnimation( mode, newName );
            if ( num >= 0 )
            {
               setAnimFrameCount( mode, num, getAnimFrameCount( mode, anim ) );
               setAnimFPS( mode, num, getAnimFPS( mode, anim ) );

               SkelAnim * sa = m_skelAnims[anim];

               for ( unsigned j = 0; j < sa->m_jointKeyframes.size(); j++ )
               {
                  for ( unsigned k = 0; k < sa->m_jointKeyframes[j].size(); k++ )
                  {
                     Keyframe * kf = sa->m_jointKeyframes[j][k];

                     setSkelAnimKeyframe( num, kf->m_frame, j, kf->m_isRotation,
                           kf->m_parameter[0], kf->m_parameter[1], kf->m_parameter[2] );
                  }
               }
            }
         }
         break;
      case ANIMMODE_FRAME:
         if ( anim < m_frameAnims.size() )
         {
            num = addAnimation( mode, newName );
            if ( num >= 0 )
            {
               setAnimFrameCount( mode, num, getAnimFrameCount( mode, anim ) );
               setAnimFPS( mode, num, getAnimFPS( mode, anim ) );

               FrameAnim * fa = m_frameAnims[anim];

               for ( unsigned f = 0; f < fa->m_frameVertices.size(); f++ )
               {
                  for ( unsigned v = 0; v < fa->m_frameVertices[f]->size(); v++ )
                  {
                     FrameAnimVertex * fav = (*fa->m_frameVertices[f])[v];
                     setFrameAnimVertexCoords( num, f, v,
                           fav->m_coord[0], fav->m_coord[1], fav->m_coord[2] );
                  }
               }
            }
         }
         break;
      case ANIMMODE_FRAMERELATIVE:
         break;
      default:
         break;
   }

   return num;
}

int Model::splitAnimation( const AnimationMode & mode, const unsigned & anim, const char * newName, const unsigned & frame )
{
   int num = -1;
   switch ( mode )
   {
      case ANIMMODE_SKELETAL:
         if ( anim < m_skelAnims.size() )
         {
            num = addAnimation( mode, newName );
            if ( num >= 0 )
            {
               setAnimFrameCount( mode, num, getAnimFrameCount( mode, anim ) - frame );
               setAnimFPS( mode, num, getAnimFPS( mode, anim ) );

               SkelAnim * sa = m_skelAnims[anim];

               for ( unsigned j = 0; j < sa->m_jointKeyframes.size(); j++ )
               {
                  for ( unsigned k = 0; k < sa->m_jointKeyframes[j].size(); k++ )
                  {
                     Keyframe * kf = sa->m_jointKeyframes[j][k];

                     if ( kf->m_frame >= frame )
                     {
                        setSkelAnimKeyframe( num, kf->m_frame - frame, j, kf->m_isRotation,
                              kf->m_parameter[0], kf->m_parameter[1], kf->m_parameter[2] );
                     }
                  }
               }

               setAnimFrameCount( mode, anim, frame );
               moveAnimation( mode, num, anim + 1 );
            }
         }
         break;
      case ANIMMODE_FRAME:
         if ( anim < m_frameAnims.size() )
         {
            num = addAnimation( mode, newName );
            if ( num >= 0 )
            {
               setAnimFrameCount( mode, num, getAnimFrameCount( mode, anim ) - frame );
               setAnimFPS( mode, num, getAnimFPS( mode, anim ) );

               FrameAnim * fa = m_frameAnims[anim];

               for ( unsigned f = frame; f < fa->m_frameVertices.size(); f++ )
               {
                  for ( unsigned v = 0; v < fa->m_frameVertices[f]->size(); v++ )
                  {
                     FrameAnimVertex * fav = (*fa->m_frameVertices[f])[v];
                     setFrameAnimVertexCoords( num, f - frame, v,
                           fav->m_coord[0], fav->m_coord[1], fav->m_coord[2] );
                  }
               }

               setAnimFrameCount( mode, anim, frame );
               moveAnimation( mode, num, anim + 1 );
            }
         }
         break;
      case ANIMMODE_FRAMERELATIVE:
         break;
      default:
         break;
   }

   return num;
}

bool Model::joinAnimations( const AnimationMode & mode, const unsigned & anim1, const unsigned & anim2 )
{
   log_debug( "join %d anim %d + %d\n", mode, anim1, anim2 );
   if ( anim1 == anim2 )
   {
      return true;
   }

   switch ( mode )
   {
      case ANIMMODE_SKELETAL:
         if ( anim1 < m_skelAnims.size() && anim2 < m_skelAnims.size() )
         {
            unsigned fc1 = getAnimFrameCount( mode, anim1 );
            unsigned fc2 = getAnimFrameCount( mode, anim2 );

            setAnimFrameCount( mode, anim1, fc1 + fc2 );

            SkelAnim * sa2 = m_skelAnims[anim2];

            for ( unsigned j = 0; j < sa2->m_jointKeyframes.size(); j++ )
            {
               for ( unsigned k = 0; k < sa2->m_jointKeyframes[j].size(); k++ )
               {
                  Keyframe * kf = sa2->m_jointKeyframes[j][k];

                  setSkelAnimKeyframe( anim1, kf->m_frame + fc1, j, kf->m_isRotation,
                        kf->m_parameter[0], kf->m_parameter[1], kf->m_parameter[2] );
               }
            }

            deleteAnimation( mode, anim2 );
         }
         break;
      case ANIMMODE_FRAME:
         if ( anim1 < m_frameAnims.size() && anim2 < m_frameAnims.size() )
         {
            unsigned fc1 = getAnimFrameCount( mode, anim1 );
            unsigned fc2 = getAnimFrameCount( mode, anim2 );

            setAnimFrameCount( mode, anim1, fc1 + fc2 );

            FrameAnim * fa2 = m_frameAnims[anim2];

            for ( unsigned f = 0; f < fc2; f++ )
            {
               for ( unsigned v = 0; v < fa2->m_frameVertices[f]->size(); v++ )
               {
                  FrameAnimVertex * fav = (*fa2->m_frameVertices[f])[v];
                  setFrameAnimVertexCoords( anim1, f + fc1, v,
                        fav->m_coord[0], fav->m_coord[1], fav->m_coord[2] );
               }
            }

            deleteAnimation( mode, anim2 );

            return true;
         }
         break;
      case ANIMMODE_FRAMERELATIVE:
         break;
      default:
         break;
   }


   return false;
}

bool Model::mergeAnimations( const AnimationMode & mode, const unsigned & anim1, const unsigned & anim2 )
{
   log_debug( "merge %d anim %d + %d\n", mode, anim1, anim2 );
   if ( anim1 == anim2 )
   {
      return true;
   }

   switch ( mode )
   {
      case ANIMMODE_SKELETAL:
         if ( anim1 < m_skelAnims.size() && anim2 < m_skelAnims.size() )
         {
            unsigned fc1 = getAnimFrameCount( mode, anim1 );
            unsigned fc2 = getAnimFrameCount( mode, anim2 );

            if ( fc1 == fc2 )
            {
               SkelAnim * sa = m_skelAnims[anim2];

               for ( unsigned j = 0; j < sa->m_jointKeyframes.size(); j++ )
               {
                  for ( unsigned k = 0; k < sa->m_jointKeyframes[j].size(); k++ )
                  {
                     Keyframe * kf = sa->m_jointKeyframes[j][k];

                     setSkelAnimKeyframe( anim1, kf->m_frame, j, kf->m_isRotation,
                           kf->m_parameter[0], kf->m_parameter[1], kf->m_parameter[2] );
                  }
               }

               deleteAnimation( mode, anim2 );

               return true;
            }
         }
         break;
      default:
         break;
   }

   return false;
}

bool Model::moveAnimation( const AnimationMode & mode, const unsigned & oldIndex, const unsigned & newIndex )
{
   if ( oldIndex == newIndex )
   {
      return true;
   }

   switch ( mode )
   {
      case ANIMMODE_SKELETAL:
         if ( oldIndex < m_skelAnims.size() && newIndex < m_skelAnims.size() )
         {
            vector<SkelAnim *>::iterator it = m_skelAnims.begin();
            unsigned t = 0;
            while ( t < oldIndex && it != m_skelAnims.end() )
            {
               t++;
               it++;
            }
            SkelAnim * sa = m_skelAnims[t];
            m_skelAnims.erase( it );
            t = 0;
            it = m_skelAnims.begin();
            while ( t < newIndex && it != m_skelAnims.end() )
            {
               t++;
               it++;
            }
            m_skelAnims.insert( it, sa );

            MU_MoveAnimation * undo = new MU_MoveAnimation();
            undo->moveAnimation( mode, oldIndex, newIndex );
            sendUndo( undo );
            return true;
         }
         break;
      case ANIMMODE_FRAME:
         if ( oldIndex < m_frameAnims.size() && newIndex < m_frameAnims.size() )
         {
            vector<FrameAnim *>::iterator it = m_frameAnims.begin();
            unsigned t = 0;
            while ( t < oldIndex && it != m_frameAnims.end() )
            {
               t++;
               it++;
            }
            FrameAnim * fa = m_frameAnims[t];
            m_frameAnims.erase( it );
            t = 0;
            it = m_frameAnims.begin();
            while ( t < newIndex && it != m_frameAnims.end() )
            {
               t++;
               it++;
            }
            m_frameAnims.insert( it, fa );

            MU_MoveAnimation * undo = new MU_MoveAnimation();
            undo->moveAnimation( mode, oldIndex, newIndex );
            sendUndo( undo );
            return true;
         }
         break;
      default:
         break;
   }

   return false;
}

int Model::convertAnimToFrame( const AnimationMode & mode, const unsigned & anim, const char * newName, const unsigned & frameCount )
{
   int num = -1;
   
   switch ( mode )
   {
      case ANIMMODE_SKELETAL:
         if ( newName && anim < m_skelAnims.size() )
         {
            num = addAnimation( ANIMMODE_FRAME, newName );
            if ( num >= 0 )
            {
               setAnimFrameCount( ANIMMODE_FRAME, num, frameCount );

               if ( frameCount > 0 )
               {
                  double time = (m_skelAnims[anim]->m_frameCount * (1.0 / m_skelAnims[anim]->m_fps));
                  double fps  = (double) frameCount / time;
                  double spf  = time / (double) frameCount;

                  log_debug( "resampling %d frames at %.2f fps to %d frames at %.2f fps\n",
                        m_skelAnims[anim]->m_frameCount, m_skelAnims[anim]->m_fps, frameCount, fps );

                  setAnimFPS( ANIMMODE_FRAME, num, fps );

                  setCurrentAnimation( ANIMMODE_SKELETAL, anim );

                  for ( unsigned f = 0; f < frameCount; f++ )
                  {
                     double currentTime = spf * (double) f;
                     setCurrentAnimationTime( currentTime );

                     unsigned vcount = m_vertices.size();
                     for ( unsigned v = 0; v < vcount; v++ )
                     {
                        double coord[3];
                        getVertexCoords( v, coord );

                        setFrameAnimVertexCoords( num, f, v,
                              coord[0], coord[1], coord[2] );
                     }
                  }

                  setNoAnimation();
               }
               else
               {
                  setAnimFPS( ANIMMODE_FRAME, num, 10.0 );
               }
            }
         }
         break;
      default:
         break;
   }

   return num;
}

bool Model::clearAnimFrame( const AnimationMode & m, const unsigned & anim, const unsigned & frame )
{
   switch ( m )
   {
      case ANIMMODE_SKELETAL:
         if ( anim < m_skelAnims.size() && frame < m_skelAnims[anim]->m_frameCount )
         {
            for ( unsigned j = 0; j < m_skelAnims[anim]->m_jointKeyframes.size(); j++ )
            {
               KeyframeList & list = m_skelAnims[anim]->m_jointKeyframes[j];
               KeyframeList::iterator it = list.begin();
               while ( it != list.end() )
               {
                  if ( (*it)->m_frame == frame )
                  {
                     deleteSkelAnimKeyframe( anim, frame, j, (*it)->m_isRotation );
                     it = list.begin();
                  }
                  else
                  {
                     it++;
                  }
               }
            }

            if ( anim == m_currentAnim && frame == m_currentFrame )
            {
               setCurrentAnimationFrame( m_currentFrame );
            }

            return true;
         }
         break;
      case ANIMMODE_FRAME:
         if ( anim < m_frameAnims.size() && frame < m_frameAnims[anim]->m_frameVertices.size() )
         {
            FrameAnim * fa = m_frameAnims[anim];
            for ( unsigned t = 0; t < m_vertices.size(); t++ )
            {
               setFrameAnimVertexCoords( anim, frame, t, 
                     m_vertices[t]->m_coord[0], m_vertices[t]->m_coord[1], m_vertices[t]->m_coord[2] );
            }
            fa->m_validNormals = false;
            return true;
         }
         break;
      default:
         break;
   }

   return false;
}

int Model::setSkelAnimKeyframe( const unsigned & anim, const unsigned & frame, const unsigned & joint, const bool & isRotation,
      const double & x, const double & y, const double & z )
{
   if ( anim < m_skelAnims.size() 
         && frame < m_skelAnims[anim]->m_frameCount
         && joint < m_joints.size() )
   {
      log_debug( "set %s of %d (%f,%f,%f) at frame %d\n", 
            (isRotation ? "rotation" : "translation"), joint, x, y, z, frame );

      while ( joint >= m_skelAnims[anim]->m_jointKeyframes.size() )
      {
         KeyframeList kl;
         m_skelAnims[anim]->m_jointKeyframes.push_back( kl );
      }

      //int num = m_skelAnims[anim]->m_jointKeyframes[joint].size();
      Keyframe * kf = Keyframe::get();

      kf->m_frame        = frame;
      kf->m_isRotation   = isRotation;

      bool   isNew = false;
      double oldx  = 0.0;
      double oldy  = 0.0;
      double oldz  = 0.0;

      unsigned index = 0;
      if ( m_skelAnims[anim]->m_jointKeyframes[joint].find_sorted( kf, index ) )
      {
         isNew = false;

         kf->release();

         kf = m_skelAnims[anim]->m_jointKeyframes[joint][index];

         oldx = kf->m_parameter[0];
         oldy = kf->m_parameter[1];
         oldz = kf->m_parameter[2];

         kf->m_parameter[0] = x;
         kf->m_parameter[1] = y;
         kf->m_parameter[2] = z;
         kf->m_jointIndex   = joint;
         kf->m_time         = m_skelAnims[anim]->m_spf * frame;
      }
      else
      {
         isNew = true;

         kf->m_parameter[0] = x;
         kf->m_parameter[1] = y;
         kf->m_parameter[2] = z;
         kf->m_jointIndex   = joint;
         kf->m_time         = m_skelAnims[anim]->m_spf * frame;

         m_skelAnims[anim]->m_jointKeyframes[joint].insert_sorted( kf );

         // Do lookup to return proper index
         m_skelAnims[anim]->m_jointKeyframes[joint].find_sorted( kf, index );
      }

      MU_SetAnimKeyframe * undo = new MU_SetAnimKeyframe();
      undo->setAnimationData( anim, frame, isRotation );
      undo->addBoneJoint( joint, isNew, x, y, z, oldx, oldy, oldz );
      sendUndo( undo );

      return index;
   }
   else
   {
      log_error( "anim keyframe out of range: anim %d, frame %d, joint %d\n", anim, frame, joint );
      if ( anim < m_skelAnims.size() )
      {
         log_error( "max frame is %d, max joint is %d\n", m_skelAnims[anim]->m_frameCount, m_joints.size() );
      }
      return -1;
   }
}

bool Model::deleteSkelAnimKeyframe( const unsigned & anim, const unsigned & frame, const unsigned & joint, const bool & isRotation )
{
   if ( anim < m_skelAnims.size() && frame < m_skelAnims[anim]->m_frameCount )
   {
      KeyframeList & list = m_skelAnims[anim]->m_jointKeyframes[joint];
      KeyframeList::iterator it = list.begin();
      while ( it != list.end() )
      {
         if ( (*it)->m_frame == frame && (*it)->m_isRotation == isRotation )
         {
            log_debug( "deleting keyframe for anim %d frame %d joint %d\n", anim, frame, joint, isRotation ? "rotation" : "translation" );
            MU_DeleteKeyframe * undo = new MU_DeleteKeyframe;
            undo->setAnimationData( anim );
            undo->deleteKeyframe( *it );
            sendUndo( undo );
            break;
         }
         it++;
      }
   }

   bool rval = removeSkelAnimKeyframe( anim, frame, joint, isRotation );

   if ( anim == m_currentAnim && frame == m_currentFrame )
   {
      setCurrentAnimationFrame( m_currentFrame );
   }

   return rval;
}

bool Model::insertSkelAnimKeyframe( const unsigned & anim, Keyframe * keyframe )
{
   if ( anim < m_skelAnims.size() 
         && keyframe->m_frame < m_skelAnims[anim]->m_frameCount 
         && keyframe->m_jointIndex < (signed int) m_joints.size() )
   {
      log_debug( "inserted keyframe for anim %d frame %d joint %d\n", anim, keyframe->m_frame, keyframe->m_jointIndex );
      m_skelAnims[anim]->m_jointKeyframes[keyframe->m_jointIndex].insert_sorted( keyframe );
      return true;
   }
   else
   {
      return false;
   }
}

bool Model::removeSkelAnimKeyframe( const unsigned & anim, const unsigned & frame, const unsigned & joint, const bool & isRotation, bool release )
{
   if ( anim < m_skelAnims.size() && frame < m_skelAnims[anim]->m_frameCount )
   {
      KeyframeList & list = m_skelAnims[anim]->m_jointKeyframes[joint];
      KeyframeList::iterator it = list.begin();
      while ( it != list.end() )
      {
         if ( (*it)->m_frame == frame && (*it)->m_isRotation == isRotation )
         {
            Keyframe * kf = *it;
            list.erase( it );
            if ( release )
            {
               kf->release();
            }
            break;
         }
         it++;
      }

      return true;
   }
   else
   {
      return false;
   }
}

void Model::insertFrameAnim( const unsigned & index, FrameAnim * anim )
{
   LOG_PROFILE();

   if ( index == m_frameAnims.size() )
   {
      m_frameAnims.push_back( anim );
   }
   else if ( index < m_frameAnims.size() )
   {
      unsigned count = 0;
      vector<FrameAnim *>::iterator it;
      for ( it = m_frameAnims.begin(); it != m_frameAnims.end(); it++ )
      {
         if ( count == index )
         {
            m_frameAnims.insert( it, anim );
            break;
         }
         count++;
      }
   }
   else
   {
      log_error( "index %d/%d out of range in insertFrameAnim\n", index, m_frameAnims.size() );
   }
}

void Model::removeFrameAnim( const unsigned & index )
{
   LOG_PROFILE();

   if ( index < m_frameAnims.size() )
   {
      unsigned num = 0;
      vector<FrameAnim *>::iterator it = m_frameAnims.begin();
      while ( num < index )
      {
         num++;
         it++;
      }

      m_frameAnims.erase( it );

      if ( m_frameAnims.size() > 0 )
      {
         while ( m_animationMode == ANIMMODE_FRAME && m_currentAnim >= m_frameAnims.size() )
         {
            m_currentAnim--;
         }
      }
   }
}

void Model::insertSkelAnim( const unsigned & index, SkelAnim * anim )
{
   LOG_PROFILE();

   if ( index == m_skelAnims.size() )
   {
      m_skelAnims.push_back( anim );
   }
   else if ( index < m_skelAnims.size() )
   {
      unsigned count = 0;
      vector<SkelAnim *>::iterator it;
      for ( it = m_skelAnims.begin(); it != m_skelAnims.end(); it++ )
      {
         if ( count == index )
         {
            m_skelAnims.insert( it, anim );
            break;
         }
         count++;
      }
   }
   else
   {
      log_error( "index %d/%d out of range in insertSkelAnim\n", index, m_skelAnims.size() );
   }
}

void Model::removeSkelAnim( const unsigned & index )
{
   LOG_PROFILE();

   if ( index < m_skelAnims.size() )
   {
      unsigned num = 0;
      vector<SkelAnim *>::iterator it = m_skelAnims.begin();
      while ( num < index )
      {
         num++;
         it++;
      }

      m_skelAnims.erase( it );

      if ( m_skelAnims.size() > 0 )
      {
         while ( m_animationMode == ANIMMODE_SKELETAL && m_currentAnim >= m_skelAnims.size() )
         {
            m_currentAnim--;
         }
      }
   }
}

void Model::insertFrameAnimFrame( const unsigned & anim, const unsigned & frame, 
      FrameAnimVertexList * list )
{
   if ( anim < m_frameAnims.size() )
   {
      FrameAnim * fa = m_frameAnims[anim];
      if ( frame == fa->m_frameVertices.size() )
      {
         fa->m_frameVertices.push_back( list );
      }
      else if ( frame == fa->m_frameVertices.size() )
      {
         unsigned count = 0;
         vector<FrameAnimVertexList *>::iterator it;
         for ( it = fa->m_frameVertices.begin(); it != fa->m_frameVertices.end(); it++ )
         {
            if ( count == frame )
            {
               fa->m_frameVertices.insert( it, list );
               break;
            }
            count++;
         }
      }
      else
      {
         log_error( "frame %d/%d out of range in insertFrameAnimFrame for frame %d\n", frame, fa->m_frameVertices.size(), anim );
      }
   }
}

void Model::removeFrameAnimFrame( const unsigned & anim, const unsigned & frame )
{
   if ( anim < m_frameAnims.size() && m_frameAnims[anim]->m_frameVertices.size() )
   {
      FrameAnim * fa = m_frameAnims[anim];
      unsigned count = 0;
      vector<FrameAnimVertexList *>::iterator it;
      for ( it = fa->m_frameVertices.begin(); it != fa->m_frameVertices.end(); it++ )
      {
         if ( count == frame )
         {
            fa->m_frameVertices.erase( it );
            break;
         }
         count++;
      }
   }
}

#endif // MM3D_EDIT

bool Model::setCurrentAnimation( const AnimationMode & m, const char * name )
{
#ifdef MM3D_EDIT
   bool needUndo = m_animationMode ? false : true;
#endif // MM3D_EDIT

   AnimationMode oldMode = m_animationMode;
   m_animationMode = ANIMMODE_NONE;

   log_debug( "Changing animation from %d to %d\n", oldMode, m );

   unsigned t;
   switch ( m )
   {
      case ANIMMODE_SKELETAL:
         for ( t = 0; t < m_skelAnims.size(); t++ )
         {
            if ( m_skelAnims[t]->m_name == name )
            {
               if ( ! m_skelAnims[t]->m_validNormals )
               {
                  calculateSkelNormals();
               }

               m_currentAnim = t;
               m_animationMode = m;
               m_currentFrame = 0;
               m_currentTime = 0;

#ifdef MM3D_EDIT
               if ( needUndo )
               {
                  MU_ChangeAnimState * undo = new MU_ChangeAnimState();
                  undo->setState( m_animationMode, ANIMMODE_NONE, m_currentAnim, m_currentFrame );
                  sendUndo( undo );
               }
#endif // MM3D_EDIT

               log_debug( "current animation: skeletal '%s'\n", name );

               return true;
            }
         }
         break;
      case ANIMMODE_FRAME:
         for ( t = 0; t < m_frameAnims.size(); t++ )
         {
            if ( m_frameAnims[t]->m_name == name )
            {
               if ( ! m_frameAnims[t]->m_validNormals )
               {
                  calculateFrameNormals( t );
               }

               m_currentAnim = t;
               m_animationMode = m;
               m_currentFrame = 0;
               m_currentTime = 0;

#ifdef MM3D_EDIT
               if ( needUndo )
               {
                  MU_ChangeAnimState * undo = new MU_ChangeAnimState();
                  undo->setState( m_animationMode, ANIMMODE_NONE, m_currentAnim, m_currentFrame );
                  sendUndo( undo );
               }
#endif // MM3D_EDIT

               log_debug( "current animation: frame '%s'\n", name );

               return true;
            }
         }
         break;
      default:
         break;
   }

   m_animationMode = m;
   m_currentAnim   = 0;
   m_currentFrame  = 0;
#ifdef MM3D_EDIT
   if ( needUndo )
   {
      log_debug( "sending anim state undo\n" );
      MU_ChangeAnimState * undo = new MU_ChangeAnimState();
      undo->setState( m_animationMode, ANIMMODE_NONE, m_currentAnim, m_currentFrame );
      sendUndo( undo );
   }
#endif // MM3D_EDIT

   return false;
}

bool Model::setCurrentAnimation( const AnimationMode & m, unsigned index )
{
#ifdef MM3D_EDIT
   bool needUndo = (m_animationMode) ? false : true;
#endif // MM3D_EDIT

   AnimationMode oldMode = m_animationMode;
   m_animationMode = ANIMMODE_NONE;

   log_debug( "Changing animation from %d to %d\n", oldMode, m );

   switch ( m )
   {
      case ANIMMODE_SKELETAL:
         if ( index < m_skelAnims.size() )
         {
            if ( ! m_skelAnims[index]->m_validNormals )
            {
               calculateSkelNormals();
            }

            m_currentAnim = index;
            m_animationMode = m;
            m_currentFrame = 0;
            m_currentTime = 0;

#ifdef MM3D_EDIT
            if ( needUndo )
            {
               MU_ChangeAnimState * undo = new MU_ChangeAnimState();
               undo->setState( m_animationMode, ANIMMODE_NONE, m_currentAnim, m_currentFrame );
               sendUndo( undo );
            }
#endif // MM3D_EDIT

            log_debug( "current animation: skeletal '%s'\n", m_skelAnims[index]->m_name.c_str() );

            return true;
         }
         break;
      case ANIMMODE_FRAME:
         if ( index < m_frameAnims.size() )
         {
            if ( ! m_frameAnims[index]->m_validNormals )
            {
               calculateFrameNormals( index );
            }

            m_currentAnim = index;
            m_animationMode = m;
            m_currentFrame = 0;
            m_currentTime = 0;

#ifdef MM3D_EDIT
            if ( needUndo )
            {
               MU_ChangeAnimState * undo = new MU_ChangeAnimState();
               undo->setState( m_animationMode, ANIMMODE_NONE, m_currentAnim, m_currentFrame );
               sendUndo( undo );
            }
#endif // MM3D_EDIT

            log_debug( "current animation: frame '%s'\n", m_frameAnims[index]->m_name.c_str() );

            return true;
         }
         break;
      default:
         break;
   }

   m_animationMode = m;
   m_currentAnim   = 0;
   m_currentFrame  = 0;
#ifdef MM3D_EDIT
   if ( needUndo )
   {
      log_debug( "sending anim state undo\n" );
      MU_ChangeAnimState * undo = new MU_ChangeAnimState();
      undo->setState( m_animationMode, ANIMMODE_NONE, m_currentAnim, m_currentFrame );
      sendUndo( undo );
   }
#endif // MM3D_EDIT

   return false;
}

bool Model::setCurrentAnimationFrame( unsigned frame )
{
   if ( m_animationMode == ANIMMODE_FRAME 
         && m_currentAnim < m_frameAnims.size() 
         && frame < m_frameAnims[m_currentAnim]->m_frameVertices.size() )
   {
      if ( !m_frameAnims[m_currentAnim]->m_validNormals )
      {
         calculateFrameNormals( m_currentAnim );
      }

      m_currentFrame = frame;
      return true;
   }
   else if ( m_animationMode == ANIMMODE_SKELETAL
         && m_currentAnim < m_skelAnims.size()
         && frame < m_skelAnims[m_currentAnim]->m_frameCount )
   {
      m_currentFrame = frame;
      setCurrentAnimationTime( frame * m_skelAnims[m_currentAnim]->m_spf );
      return true;
   }
   else
   {
      return false;
   }
}

bool Model::setCurrentAnimationTime( double frameTime )
{
   m_currentTime = 0;

   if ( m_animationMode == ANIMMODE_FRAME && m_currentAnim < m_frameAnims.size() )
   {
      if ( !m_frameAnims[m_currentAnim]->m_validNormals )
      {
         calculateFrameNormals( m_currentAnim );
      }

      double spf = 1.0 / m_frameAnims[m_currentAnim]->m_fps;
      double totalTime = spf * m_frameAnims[m_currentAnim]->m_frameVertices.size();
      while ( frameTime >= totalTime )
      {
         if ( !m_animationLoop )
         {
            return false;
         }
         frameTime -= totalTime;
      }

      m_currentTime = frameTime;
      m_currentFrame = (unsigned) (frameTime / spf);

      return true;
   }
   else if ( m_animationMode == ANIMMODE_SKELETAL && m_currentAnim < m_skelAnims.size() && m_skelAnims[m_currentAnim]->m_frameCount > 0 )
   {
      LOG_PROFILE();

      SkelAnim * sa = m_skelAnims[m_currentAnim];
      double totalTime = sa->m_spf * sa->m_frameCount;
      while ( frameTime > totalTime )
      {
         if ( !m_animationLoop )
         {
            return false;
         }
         frameTime -= totalTime;
      }

      m_currentTime = frameTime;

      for ( unsigned j = 0; j < sa->m_jointKeyframes.size(); j++ )
      {
         Matrix transform;

         if ( !sa->m_jointKeyframes[j].empty() )
         {
            int firstRot  = -1;
            int firstTran = -1;
            int lastRot  = -1;
            int lastTran = -1;
            int stopRot  = -1;
            int stopTran = -1;
            int rot  = -1;
            int tran = -1;
            for ( unsigned k = 0; k < sa->m_jointKeyframes[j].size(); k++ )
            {
               if ( sa->m_jointKeyframes[j][k]->m_isRotation )
               {
                  if ( firstRot == -1 )
                  {
                     firstRot = k;
                  }
                  lastRot = k;
               }
               else
               {
                  if ( firstTran == -1 )
                  {
                     firstTran = k;
                  }
                  lastTran = k;
               }

               if ( sa->m_jointKeyframes[j][k]->m_time <= frameTime )
               {
                  // Less than current time
                  // get latest keyframe for rotation and translation
                  if ( sa->m_jointKeyframes[j][k]->m_isRotation )
                  {
                     rot = k;
                  }
                  else
                  {
                     tran = k;
                  }
               }
               else
               {
                  // Greater than current time
                  // get earliest keyframe for rotation and translation
                  if ( sa->m_jointKeyframes[j][k]->m_isRotation )
                  {
                     if ( stopRot == -1 )
                     {
                        stopRot = k;
                     }
                  }
                  else
                  {
                     if ( stopTran == -1 )
                     {
                        stopTran = k;
                     }
                  }
               }
            }

            if ( m_animationLoop )
            {
               if ( rot == -1 )
               {
                  rot = lastRot;
               }
               if ( tran == -1 )
               {
                  tran = lastTran;
               }

               if ( stopRot == -1 )
               {
                  stopRot = firstRot;
               }
               if ( stopTran == -1 )
               {
                  stopTran = firstTran;
               }
            }

            stopRot  = (stopRot  == -1) ? rot  : stopRot;
            stopTran = (stopTran == -1) ? tran : stopTran;

            if ( rot >= 0 )
            {
               double temp[3];
               double diff = sa->m_jointKeyframes[j][stopRot]->m_time - sa->m_jointKeyframes[j][rot]->m_time;

               double tempTime = frameTime;

               if ( tempTime < sa->m_jointKeyframes[j][rot]->m_time )
               {
                  tempTime += (sa->m_spf * sa->m_frameCount);
               }

               if ( diff < 0.0 )
               {
                  diff += (sa->m_spf * sa->m_frameCount);
               }

               Vector va( sa->m_jointKeyframes[j][rot]->m_parameter );

               // TODO right now I'm only normalizing 
               // if X and Z are greater than 90 because those
               // are the only angles I'm having trouble with.
               // This may need to be modified to normalize
               // the other angles as well
               if ( fabs( va.get(0) ) > 90.0 * PIOVER180 
                     && fabs( va.get(2) ) > 90.0 * PIOVER180 )
               {
                  if ( va.get(0) > 0.0 )
                     va.set( 0, va.get(0) - 180.0 * PIOVER180 );
                  else
                     va.set( 0, va.get(0) + 180.0 * PIOVER180 );
                  if ( va.get(1) > 0.0 )
                     va.set( 1, 180 * PIOVER180 - va.get(1) );
                  else
                     va.set( 1, -180 * PIOVER180 - va.get(1) );
                  if ( va.get(2) > 0.0 )
                     va.set( 2, va.get(2) - 180.0 * PIOVER180 );
                  else
                     va.set( 2, va.get(2) + 180.0 * PIOVER180 );
               }

               if ( diff > 0 )
               {
                  Vector vb( sa->m_jointKeyframes[j][stopRot]->m_parameter );

                  // TODO see above
                  if ( fabs( vb.get(0) ) > 90.0 * PIOVER180 
                        && fabs( vb.get(2) ) > 90.0 * PIOVER180 )
                  {
                     if ( vb.get(0) > 0.0 )
                        vb.set( 0, vb.get(0) - 180.0 * PIOVER180 );
                     else
                        vb.set( 0, vb.get(0) + 180.0 * PIOVER180 );
                     if ( vb.get(1) > 0.0 )
                        vb.set( 1, 180 * PIOVER180 - vb.get(1) );
                     else
                        vb.set( 1, -180 * PIOVER180 - vb.get(1) );
                     if ( vb.get(2) > 0.0 )
                        vb.set( 2, vb.get(2) - 180.0 * PIOVER180 );
                     else
                        vb.set( 2, vb.get(2) + 180.0 * PIOVER180 );
                  }

                  // Wrap rotation so that we can rotate 360 degrees
                  for ( unsigned i = 0; i < 3; i++ )
                  {
                     if ( fabs(vb.get(i) - va.get(i)) >= 180.0 * PIOVER180 ) 
                     {
                        if ( va.get(i) > 0.0 )
                           vb.set( i, vb.get(i) + 360.0 * PIOVER180 );
                        else
                           vb.set( i, vb.get(i) - 360.0 * PIOVER180 );
                     }
                  }

                  double tm = (tempTime - sa->m_jointKeyframes[j][rot]->m_time) / diff;
                  va = va + (vb - va) * tm;
               }
               else
               {
                  temp[0] = sa->m_jointKeyframes[j][rot]->m_parameter[0];
                  temp[1] = sa->m_jointKeyframes[j][rot]->m_parameter[1];
                  temp[2] = sa->m_jointKeyframes[j][rot]->m_parameter[2];
               }

               transform.setRotation( va.getVector() );
            }

            if ( tran >= 0 )
            {
               double temp[3];
               double diff = sa->m_jointKeyframes[j][stopTran]->m_time - sa->m_jointKeyframes[j][tran]->m_time;

               if ( diff < 0.0 )
               {
                  diff += (sa->m_spf * sa->m_frameCount);
               }

               Vector va( sa->m_jointKeyframes[j][tran]->m_parameter );
               if ( diff > 0 )
               {
                  Vector vb( sa->m_jointKeyframes[j][stopTran]->m_parameter );
                  double tm = (frameTime - sa->m_jointKeyframes[j][tran]->m_time) / diff;
                  va = va + (vb - va) * tm;
               }
               else
               {
                  temp[0] = sa->m_jointKeyframes[j][tran]->m_parameter[0];
                  temp[1] = sa->m_jointKeyframes[j][tran]->m_parameter[1];
                  temp[2] = sa->m_jointKeyframes[j][tran]->m_parameter[2];
               }

               transform.setTranslation( va.getVector() );
            }
         }

         Matrix relativeFinal( m_joints[j]->m_relative );
         relativeFinal = transform * relativeFinal;

         if ( m_joints[j]->m_parent == -1 )
         {
            relativeFinal = relativeFinal * m_localMatrix;
            m_joints[j]->m_final = relativeFinal;
         }
         else
         {
            m_joints[j]->m_final = m_joints[ m_joints[j]->m_parent ]->m_final;
            m_joints[j]->m_final = relativeFinal * m_joints[j]->m_final;
         }
      }

      for ( unsigned v = 0; v < m_vertices.size(); v++ )
      {
         if ( m_vertices[v]->m_boneId >= 0 )
         {
            Vector norm( m_keyframeVertices[ v ].m_normal );
            norm.transform3( m_joints[ m_vertices[v]->m_boneId ]->m_final );
            norm.normalize3();
            m_keyframeVertices[v].m_rotatedNormal[0] = norm.get(0);
            m_keyframeVertices[v].m_rotatedNormal[1] = norm.get(1);
            m_keyframeVertices[v].m_rotatedNormal[2] = norm.get(2);
         }
      }

      return true;
   }
   else
   {
      return false;
   }
}

unsigned Model::getCurrentAnimation()
{
   return m_currentAnim;
}

unsigned Model::getCurrentAnimationFrame()
{
   return m_currentFrame;
}

double Model::getCurrentAnimationTime()
{
   return m_currentTime;
}

void Model::setAnimationLooping( bool o )
{
   m_animationLoop = o;
}

bool Model::isAnimationLooping()
{
   return m_animationLoop;
}

void Model::setNoAnimation()
{
#ifdef MM3D_EDIT
   if ( m_animationMode )
   {
      MU_ChangeAnimState * undo = new MU_ChangeAnimState();
      undo->setState( ANIMMODE_NONE, m_animationMode, m_currentAnim, m_currentFrame );
      sendUndo( undo );
   }
#endif // MM3D_EDIT

   m_animationMode = ANIMMODE_NONE;
   setupJoints();
}

void Model::setupJoints()
{
   LOG_PROFILE();
   
   m_validJoints = true;

   if ( m_animationMode )
   {
      return;
   }

   log_debug( "setupJoints()\n" );

   for ( unsigned j = 0; j < m_joints.size(); j++ )
   {
      Joint * joint = m_joints[ j ];
      joint->m_relative.loadIdentity();
      joint->m_relative.setRotation( joint->m_localRotation );
      joint->m_relative.setTranslation( joint->m_localTranslation );

      if ( joint->m_parent == -1 ) // parent
      {
         joint->m_absolute = joint->m_relative;
      }
      else
      {
         joint->m_absolute = m_joints[ joint->m_parent ]->m_absolute;
         joint->m_absolute = joint->m_relative * joint->m_absolute;
      }

      joint->m_final = joint->m_absolute;
//      log_debug( "\n" );
//      log_debug( "Joint %d:\n", j );
//      joint->m_final.show();
//      log_debug( "local rotation: %.2f %.2f %.2f\n", 
//            joint->m_localRotation[0], joint->m_localRotation[1], joint->m_localRotation[2] );
//      log_debug( "\n" );
   }

   for ( unsigned p = 0; p < m_points.size(); p++ )
   {
      Point * point = m_points[p];
      double trans[3] = { point->m_trans[0], point->m_trans[1], point->m_trans[2] };
      double rot[3]   = { point->m_rot[0], point->m_rot[1], point->m_rot[2] };

      if ( point->m_boneId >= 0 )
      {
         m_joints[ point->m_boneId ]->m_absolute.inverseTranslateVector( trans );
         m_joints[ point->m_boneId ]->m_absolute.inverseRotateVector( trans );
         Matrix minv = m_joints[ point->m_boneId ]->m_absolute.getInverse();
         Matrix mr;
         mr.setRotation( rot );
         mr = mr * minv;
         mr.getRotation( rot );
      }

      for ( int t = 0; t < 3; t++ )
      {
         point->m_localTranslation[t] = trans[t];
         point->m_localRotation[t]    = rot[t];
      }
   }
}

unsigned Model::getAnimCount( const AnimationMode & m ) const
{
   switch ( m )
   {
      case ANIMMODE_SKELETAL:
         return m_skelAnims.size();
         break;
      case ANIMMODE_FRAME:
         return m_frameAnims.size();
         break;
      default:
         break;
   }

   return 0;
}
const char * Model::getAnimName( const AnimationMode & m, const unsigned & anim )
{
   switch ( m )
   {
      case ANIMMODE_SKELETAL:
         if ( anim < m_skelAnims.size() )
         {
            return m_skelAnims[anim]->m_name.c_str();
         }
         break;
      case ANIMMODE_FRAME:
         if ( anim < m_frameAnims.size() )
         {
            return m_frameAnims[anim]->m_name.c_str();
         }
         break;
      default:
         break;
   }

   return NULL;
}

unsigned Model::getAnimFrameCount( const AnimationMode & m, const unsigned & anim )
{
   switch ( m )
   {
      case ANIMMODE_SKELETAL:
         if ( anim < m_skelAnims.size() )
         {
            return m_skelAnims[anim]->m_frameCount;
         }
         break;
      case ANIMMODE_FRAME:
         if ( anim < m_frameAnims.size() )
         {
            return m_frameAnims[anim]->m_frameVertices.size();
         }
         break;
      default:
         break;
   }

   return 0;
}

double Model::getAnimFPS( const AnimationMode & m, const unsigned & anim )
{
   switch ( m )
   {
      case ANIMMODE_SKELETAL:
         if ( anim < m_skelAnims.size() )
         {
            return m_skelAnims[anim]->m_fps;
         }
         break;
      case ANIMMODE_FRAME:
         if ( anim < m_frameAnims.size() )
         {
            return m_frameAnims[anim]->m_fps;
         }
         break;
      default:
         break;
   }

   return 0;
}

bool Model::getFrameAnimVertexCoords( const unsigned & anim, const unsigned & frame, const unsigned & vertex,
      double & x, double & y, double & z )
{
   if ( anim < m_frameAnims.size() 
         && frame < m_frameAnims[anim]->m_frameVertices.size()
         && vertex < m_frameAnims[anim]->m_frameVertices[frame]->size() )
   {
      FrameAnimVertex * fav = (*m_frameAnims[anim]->m_frameVertices[frame])[vertex];
      x = fav->m_coord[0];
      y = fav->m_coord[1];
      z = fav->m_coord[2];
      return true;
   }
   else if ( vertex < m_vertices.size() )
   {
      x = m_vertices[vertex]->m_coord[0];
      y = m_vertices[vertex]->m_coord[1];
      z = m_vertices[vertex]->m_coord[2];
      return true;
   }
   else
   {
      return false;
   }
}

bool Model::getFrameAnimVertexNormal( const unsigned & anim, const unsigned & frame, const unsigned & vertex,
      double & x, double & y, double & z )
{
   if ( anim < m_frameAnims.size() 
         && frame < m_frameAnims[anim]->m_frameVertices.size()
         && vertex < m_frameAnims[anim]->m_frameVertices[frame]->size() )
   {
      FrameAnimVertex * fav = (*m_frameAnims[anim]->m_frameVertices[frame])[vertex];
      x = fav->m_normal[0];
      y = fav->m_normal[1];
      z = fav->m_normal[2];
      return true;
   }
   else
   {
      return false;
   }
}

bool Model::hasSkelAnimKeyframe( const unsigned & anim, const unsigned & frame, const unsigned & joint, const bool & isRotation )
{
   if ( anim < m_skelAnims.size() 
         && frame < m_skelAnims[anim]->m_frameCount
         && joint < m_joints.size() )
   {
      while ( joint >= m_skelAnims[anim]->m_jointKeyframes.size() )
      {
         KeyframeList kl;
         m_skelAnims[anim]->m_jointKeyframes.push_back( kl );
      }

      Keyframe * kf = Keyframe::get();

      kf->m_frame        = frame;
      kf->m_isRotation   = isRotation;

      unsigned index;
      bool found = false;
      if ( m_skelAnims[anim]->m_jointKeyframes[joint].find_sorted( kf, index ) )
      {
         found = true;
      }
      kf->release();

      return found;
   }
   else
   {
      log_error( "anim keyframe out of range: anim %d, frame %d, joint %d\n", anim, frame, joint );
      return false;
   }
}

bool Model::getSkelAnimKeyframe( const unsigned & anim, const unsigned & frame, const unsigned & joint, const bool & isRotation,
      double & x, double & y, double & z )
{
   if ( anim < m_skelAnims.size() 
         && frame < m_skelAnims[anim]->m_frameCount
         && joint < m_joints.size() )
   {
      while ( joint >= m_skelAnims[anim]->m_jointKeyframes.size() )
      {
         KeyframeList kl;
         m_skelAnims[anim]->m_jointKeyframes.push_back( kl );
      }

      Keyframe * kf = Keyframe::get();

      kf->m_frame        = frame;
      kf->m_isRotation   = isRotation;

      unsigned index;
      bool found = false;
      if ( m_skelAnims[anim]->m_jointKeyframes[joint].find_sorted( kf, index ) )
      {
         log_debug( "found keyframe anim %d, frame %d, joint %d, %s\n",
               anim, frame, joint, isRotation ? "rotation" : "translation" );
         x = m_skelAnims[anim]->m_jointKeyframes[joint][index]->m_parameter[0];
         y = m_skelAnims[anim]->m_jointKeyframes[joint][index]->m_parameter[1];
         z = m_skelAnims[anim]->m_jointKeyframes[joint][index]->m_parameter[2];
         found = true;
      }
      else
      {
         //log_debug( "could not find keyframe anim %d, frame %d, joint %d, %s\n",
         //      anim, frame, joint, isRotation ? "rotation" : "translation" );
      }
      kf->release();

      return found;
   }
   else
   {
      log_error( "anim keyframe out of range: anim %d, frame %d, joint %d\n", anim, frame, joint );
      return false;
   }
}

