/*  Misfit Model 3D
 * 
 *  Copyright (c) 2004-2005 Kevin Worcester
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, 
 *  USA.
 *
 *  See the COPYING file for full license text.
 */


#include "jointwin.h"
#include "decalmgr.h"
#include "helpwin.h"

#include "mq3compat.h"

#include <qinputdialog.h>
#include <qpushbutton.h>
#include <qcombobox.h>
#include <list>

using std::list;

#include "model.h"
#include "decalmgr.h"
#include "log.h"
#include "msg.h"
#include "modelstatus.h"


JointWin::JointWin( Model * model, QWidget * parent, const char * name )
   : JointWinBase( parent, name, true, Qt::WDestructiveClose ),
     m_accel( new QAccel(this) ),
     m_model( model )
{
   m_accel->insertItem( Qt::Key_F1, 0 );
   connect( m_accel, SIGNAL(activated(int)), this, SLOT(helpNowEvent(int)) );

   for ( int t = 0; t < m_model->getBoneJointCount(); t++ )
   {
      m_jointName->insertItem( m_model->getBoneJointName(t), t );
   }

   list<int> joints = m_model->getSelectedBoneJoints();
   if ( ! joints.empty() )
   {
      m_jointName->setCurrentItem( joints.front() );
      jointNameSelected( joints.front() );
   }
   else
   {
      m_jointName->setCurrentItem( 0 );
      jointNameSelected( 0 );
   }
}

JointWin::~JointWin()
{
}

void JointWin::helpNowEvent( int id )
{
   HelpWin * win = new HelpWin( "olh_jointwin.html", true );
   win->show();
}

void JointWin::jointNameSelected( int index )
{
   if ( index < m_jointName->count() )
   {
      m_model->unselectAllBoneJoints();
      m_model->selectBoneJoint( index );

      m_deleteButton->setEnabled( true );
      m_renameButton->setEnabled( true );
      m_selectVerticesButton->setEnabled( true );
      m_assignVerticesButton->setEnabled( true );
      DecalManager::getInstance()->modelUpdated( m_model );
   }
   else
   {
      m_deleteButton->setEnabled( false );
      m_renameButton->setEnabled( false );
      m_selectVerticesButton->setEnabled( false );
      m_assignVerticesButton->setEnabled( false );
   }
}

void JointWin::deleteClicked()
{
   if ( m_jointName->count() )
   {
      m_model->deleteBoneJoint( m_jointName->currentItem() );
   }
}

void JointWin::renameClicked()
{
   if ( m_jointName->count() )
   {
      bool ok = false;
      int jointNum = m_jointName->currentItem();
      QString jointName = QInputDialog::getText("Rename joint", "Enter new joint name:", QLineEdit::Normal, m_model->getBoneJointName( jointNum ), &ok );
      if ( ok )
      {
         m_model->setBoneJointName( jointNum, jointName.latin1() );
         m_jointName->changeItem( jointName, jointNum );
      }
   }
}

void JointWin::selectVerticesClicked()
{
   if ( m_jointName->count() )
   {
      m_model->unselectAllVertices();
      list<int> vertlist = m_model->getBoneJointVertices( m_jointName->currentItem() );
      list<int>::iterator it;
      for ( it = vertlist.begin(); it != vertlist.end(); it++ )
      {
         m_model->selectVertex( *it );
      }
      DecalManager::getInstance()->modelUpdated( m_model );
   }
}

void JointWin::selectUnassignedClicked()
{
   int joint = -1;
   m_model->unselectAllVertices();
   list<int> vertlist = m_model->getBoneJointVertices( joint );
   list<int>::iterator it;
   for ( it = vertlist.begin(); it != vertlist.end(); it++ )
   {
      m_model->selectVertex( *it );
   }
   DecalManager::getInstance()->modelUpdated( m_model );
}

void JointWin::assignVerticesClicked()
{
   log_debug( "assignVerticesClicked()\n" );
   if ( m_jointName->count() )
   {
      unsigned joint = m_jointName->currentItem();
      list<int> vertlist = m_model->getSelectedVertices();
      list<int>::iterator it;
      log_debug( "assigning %d vertices to joint %d\n", vertlist.size(), joint );
      for ( it = vertlist.begin(); it != vertlist.end(); it++ )
      {
         m_model->setVertexBoneJoint( *it, joint );
      }
   }
}

void JointWin::accept()
{
   log_debug( "Joint changes complete" );
   m_model->operationComplete( "Joint changes" );
   JointWinBase::accept();
}

void JointWin::reject()
{
   log_debug( "Joint changes canceled" );
   m_model->undoCurrent();
   DecalManager::getInstance()->modelUpdated( m_model );
   JointWinBase::reject();
}

