/*  Misfit Model 3D
 * 
 *  Copyright (c) 2004-2005 Kevin Worcester
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, 
 *  USA.
 *
 *  See the COPYING file for full license text.
 */

#include "model.h"
#include "log.h"

#ifdef MM3D_EDIT

#include "modelundo.h"

void Model::setSelectionMode( Model::SelectionMode m )
{
   if ( m != m_selectionMode )
   {
      MU_SelectionMode * undo = new MU_SelectionMode;
      undo->setSelectionMode( m, m_selectionMode );
      sendUndo( undo );

      /*
      switch ( m )
      {
         case SelectVertices:
            unselectAllTriangles();
            unselectAllGroups();
            break;
         case SelectTriangles:
            if ( m_selectionMode == SelectVertices )
            {
               selectTrianglesFromVertices();
            }
            unselectAllGroups();
            break;
         case SelectGroups:
            if ( m_selectionMode == SelectVertices )
            {
               selectTrianglesFromVertices();
            }
            if ( m_selectionMode == SelectVertices || m_selectionMode == SelectTriangles )
            {
               selectGroupsFromTriangles();
            }
            break;
         case SelectJoints:
            unselectAllVertices();
            unselectAllTriangles();
            unselectAllGroups();
            break;
         default:
            break;
      }
      */
      m_selectionMode = m;
   }
}

bool Model::selectVertex( unsigned v )
{
   //printf( "select( %d )\n", v );
   if ( v < m_vertices.size() )
   {
      bool old = m_vertices[v]->m_selected;
      m_vertices[ v ]->m_selected = true;

      MU_Select * undo = new MU_Select( SelectVertices );
      undo->setSelectionDifference( v, true, old );
      sendUndo( undo );

      return true;
   }
   else
   {
      return false;
   }
}

bool Model::selectTriangle( unsigned t )
{
   if ( t < m_triangles.size() )
   {
      bool old = m_triangles[t]->m_selected;
      m_triangles[ t ]->m_selected = true;
      bool o = setUndoEnabled( false );
      //selectVerticesFromTriangles();
      selectVertex( m_triangles[t]->m_vertexIndices[0] );
      selectVertex( m_triangles[t]->m_vertexIndices[1] );
      selectVertex( m_triangles[t]->m_vertexIndices[2] );
      setUndoEnabled( o );

      MU_Select * undo = new MU_Select( SelectTriangles );
      undo->setSelectionDifference( t, true, old );
      sendUndo( undo );

      return true;
   }
   else
   {
      return false;
   }
}

bool Model::selectGroup( unsigned m )
{
   if ( m >= 0 && m < m_groups.size() )
   {
      bool old = m_groups[m]->m_selected;
      m_groups[ m ]->m_selected = true;
      bool o = setUndoEnabled( false );
      selectTrianglesFromGroups();
      setUndoEnabled( o );

      MU_Select * undo = new MU_Select( SelectGroups );
      undo->setSelectionDifference( m, true, old );
      sendUndo( undo );

      return true;
   }
   else
   {
      return false;
   }
}

bool Model::selectBoneJoint( unsigned j )
{
   if ( j >= 0 && j < m_joints.size() )
   {
      bool old = m_joints[j]->m_selected;
      m_joints[ j ]->m_selected = true;

      MU_Select * undo = new MU_Select( SelectJoints );
      undo->setSelectionDifference( j, true, old );
      sendUndo( undo );

      return true;
   }
   else
   {
      return false;
   }
}

bool Model::unselectVertex( unsigned v )
{
   //printf( "unselect( %d )\n", v );
   if ( v < m_vertices.size() )
   {
      bool old = m_vertices[v]->m_selected;
      m_vertices[ v ]->m_selected = false;

      MU_Select * undo = new MU_Select( SelectVertices );
      undo->setSelectionDifference( v, false, old );
      sendUndo( undo );

      return true;
   }
   else
   {
      return false;
   }
}

bool Model::unselectTriangle( unsigned t )
{
   if ( t < m_triangles.size() )
   {
      bool old = m_triangles[t]->m_selected;
      m_triangles[ t ]->m_selected = false;
      bool o = setUndoEnabled( false );
      selectVerticesFromTriangles();
      setUndoEnabled( o );

      MU_Select * undo = new MU_Select( SelectTriangles );
      undo->setSelectionDifference( t, false, old );
      sendUndo( undo );

      return true;
   }
   else
   {
      return false;
   }
}

bool Model::unselectGroup( unsigned m )
{
   LOG_PROFILE();

   if ( m >= 0 && m < m_groups.size() )
   {
      bool old = m_groups[m]->m_selected;
      m_groups[ m ]->m_selected = false;

      bool o = setUndoEnabled( false );

      list<int> tris = getGroupTriangles( m );
      list<int>::iterator it;
      for ( it = tris.begin(); it != tris.end(); it++ )
      {
         m_triangles[ *it ]->m_selected = false;
      }
      selectVerticesFromTriangles();

      setUndoEnabled( o );

      MU_Select * undo = new MU_Select( SelectGroups );
      undo->setSelectionDifference( m, false, old );
      sendUndo( undo );

      return true;
   }
   else
   {
      return false;
   }
}

bool Model::unselectBoneJoint( unsigned j )
{
   if ( j < m_joints.size() )
   {
      bool old = m_joints[j]->m_selected;
      m_joints[ j ]->m_selected = false;

      MU_Select * undo = new MU_Select( SelectJoints );
      undo->setSelectionDifference( j, false, old );
      sendUndo( undo );

      return true;
   }
   else
   {
      return false;
   }
}

bool Model::isVertexSelected( unsigned v )
{
   LOG_PROFILE();

   if ( v < m_vertices.size() )
   {
      return m_vertices[v]->m_selected;
   }
   else
   {
      return false;
   }
}

bool Model::isTriangleSelected( unsigned v )
{
   LOG_PROFILE();

   if ( v < m_triangles.size() )
   {
      return m_triangles[v]->m_selected;
   }
   else
   {
      return false;
   }
}

bool Model::isGroupSelected( unsigned v )
{
   LOG_PROFILE();

   if ( v < m_groups.size() )
   {
      return m_groups[v]->m_selected;
   }
   else
   {
      return false;
   }
}

bool Model::isBoneJointSelected( unsigned j )
{
   LOG_PROFILE();

   if ( j < m_joints.size() )
   {
      return m_joints[j]->m_selected;
   }
   else
   {
      return false;
   }
}

bool Model::selectVerticesInVolume( bool select, int c1, int c2, double a1, double b1, double a2, double b2, SelectionTest * test )
{
   LOG_PROFILE();

   beginSelectionDifference();

   if ( a1 > a2 )
   {
      double temp = a2;
      a2 = a1;
      a1 = temp;
   }

   if ( b1 > b2 )
   {
      double temp = b2;
      b2 = b1;
      b1 = temp;
   }

   if ( m_animationMode )
   {
      if ( m_animationMode == ANIMMODE_SKELETAL )
      {
         for ( unsigned v = 0; v < m_vertices.size(); v++ )
         {
            if ( m_keyframeVertices[v].m_coord[c1] >= a1 && m_keyframeVertices[v].m_coord[c1] <= a2 
                  && m_keyframeVertices[v].m_coord[c2] >= b1 && m_keyframeVertices[v].m_coord[c2] <= b2 )
            {
               if ( test )
                  m_vertices[v]->m_selected = test->shouldSelect( m_vertices[v] ) ? select : m_vertices[v]->m_selected;
               else
                  m_vertices[v]->m_selected = select;
            }
         }
      }
      else
      {
         FrameAnimVertex * vertex;
         for ( unsigned v = 0; v < m_vertices.size(); v++ )
         {
            vertex = (*m_frameAnims[m_currentAnim]->m_frameVertices[m_currentFrame])[ v ];
            if ( vertex->m_coord[c1] >= a1 && vertex->m_coord[c1] <= a2 
                  && vertex->m_coord[c2] >= b1 && vertex->m_coord[c2] <= b2 )
            {
               if ( test )
                  m_vertices[v]->m_selected = test->shouldSelect( m_vertices[v] ) ? select : m_vertices[v]->m_selected;
               else
                  m_vertices[v]->m_selected = select;
            }
         }
      }
   }
   else
   {
      for ( unsigned v = 0; v < m_vertices.size(); v++ )
      {
         if (  m_vertices[v]->m_visible
               && m_vertices[v]->m_coord[c1] >= a1 && m_vertices[v]->m_coord[c1] <= a2 
               && m_vertices[v]->m_coord[c2] >= b1 && m_vertices[v]->m_coord[c2] <= b2 )
         {
            if ( test )
               m_vertices[v]->m_selected = test->shouldSelect( m_vertices[v] ) ? select : m_vertices[v]->m_selected;
            else
               m_vertices[v]->m_selected = select;
         }
      }
   }
   endSelectionDifference();

   return true;
}

bool Model::selectTrianglesInVolume( bool select, int c1, int c2, double x1, double y1, double x2, double y2, bool connected, SelectionTest * test )
{
   LOG_PROFILE();

   beginSelectionDifference();

   unsigned i;
   for ( i = 0; i < m_vertices.size(); i++ )
   {
      m_vertices[i]->m_marked2 = false;
   }

   for ( i = 0; i < m_triangles.size(); i++ )
   {
      m_triangles[i]->m_marked2 = false;
   }

   if ( x1 > x2 )
   {
      double temp = x2;
      x2 = x1;
      x1 = temp;
   }

   if ( y1 > y2 )
   {
      double temp = y2;
      y2 = y1;
      y1 = temp;
   }

   unsigned t;
   for ( t = 0; t < m_triangles.size(); t++ )
   {
      if ( !test || (test && test->shouldSelect(m_triangles[t]) )) 
      {
         if ( test && m_triangles[t]->m_selected )
         {
            log_debug( "triangle %d tested positive and is selected\n", t );
         }

         if ( m_triangles[t]->m_visible )
         {
            bool above = false;
            bool below = false;

            int v;
            Vertex *vert[3];

            // 0. Assign vert to triangle's verticies 
            // 1. Check for vertices within the selection volume in the process
            for ( v = 0; v < 3; v++ )
            {
               vert[v] = m_vertices[ m_triangles[t]->m_vertexIndices[v] ];
            }
            for ( v = 0; v < 3; v++ )
            {
               if (  vert[v]->m_coord[c1] >= x1 && vert[v]->m_coord[c1] <= x2 
                     && vert[v]->m_coord[c2] >= y1 && vert[v]->m_coord[c2] <= y2 )
               {
                  // A vertex of the triangle is within the selection area
                  m_triangles[t]->m_selected = select;

                  vert[0]->m_marked2 = true;
                  vert[1]->m_marked2 = true;
                  vert[2]->m_marked2 = true;
                  goto next_triangle; // next triangle
               }
            }

            // 2. Find intersections between triangle edges and selection edges
            // 3. Also, check to see if the selection box is completely within triangle

            double m[3];
            double b[3];
            double *coord[3][2];
            m[0] = (vert[0]->m_coord[c2] - vert[1]->m_coord[c2]) / (vert[0]->m_coord[c1] - vert[1]->m_coord[c1]) ;
            coord[0][0] = vert[0]->m_coord;
            coord[0][1] = vert[1]->m_coord;
            m[1] = (vert[0]->m_coord[c2] - vert[2]->m_coord[c2]) / (vert[0]->m_coord[c1] - vert[2]->m_coord[c1]) ;
            coord[1][0] = vert[0]->m_coord;
            coord[1][1] = vert[2]->m_coord;
            m[2] = (vert[1]->m_coord[c2] - vert[2]->m_coord[c2]) / (vert[1]->m_coord[c1] - vert[2]->m_coord[c1]) ;
            coord[2][0] = vert[1]->m_coord;
            coord[2][1] = vert[2]->m_coord;

            b[0] = vert[0]->m_coord[c2] - ( m[0] * vert[0]->m_coord[c1] );
            b[1] = vert[2]->m_coord[c2] - ( m[1] * vert[2]->m_coord[c1] );
            b[2] = vert[2]->m_coord[c2] - ( m[2] * vert[2]->m_coord[c1] );

            for ( int line = 0; line < 3; line++ )
            {
               double y;
               double x;
               double xmin;
               double xmax;
               double ymin;
               double ymax;

               if ( coord[line][0][c1] < coord[line][1][c1] )
               {
                  xmin = coord[line][0][c1];
                  xmax = coord[line][1][c1];
               }
               else
               {
                  xmin = coord[line][1][c1];
                  xmax = coord[line][0][c1];
               }

               if ( coord[line][0][c2] < coord[line][1][c2] )
               {
                  ymin = coord[line][0][c2];
                  ymax = coord[line][1][c2];
               }
               else
               {
                  ymin = coord[line][1][c2];
                  ymax = coord[line][0][c2];
               }

               if ( x1 >= xmin && x1 <= xmax )
               {
                  y = m[line] * x1 + b[line];
                  if ( y >= y1 && y <= y2 )
                  {
                     m_triangles[t]->m_selected = select;

                     vert[0]->m_marked2 = true;
                     vert[1]->m_marked2 = true;
                     vert[2]->m_marked2 = true;
                     goto next_triangle; // next triangle
                  }

                  if ( y > y1 )
                  {
                     above = true;
                  }
                  if ( y < y1 )
                  {
                     below = true;
                  }
               }

               if ( x2 >= xmin && x2 <= xmax )
               {
                  y = m[line] * x2 + b[line];
                  if ( y >= y1 && y <= y2 )
                  {
                     m_triangles[t]->m_selected = select;

                     vert[0]->m_marked2 = true;
                     vert[1]->m_marked2 = true;
                     vert[2]->m_marked2 = true;
                     goto next_triangle; // next triangle
                  }
               }

               if ( y1 >= ymin && y1 <= ymax )
               {
                  if ( coord[line][0][c1] == coord[line][1][c1] )
                  {
                     if ( coord[line][0][c1] >= x1 && coord[line][0][c1] <= x2 )
                     {
                        m_triangles[t]->m_selected = select;

                        vert[0]->m_marked2 = true;
                        vert[1]->m_marked2 = true;
                        vert[2]->m_marked2 = true;
                        goto next_triangle; // next triangle
                     }
                  }
                  else
                  {
                     x = (y1 - b[line]) / m[line];
                     if ( x >= x1 && x <= x2 )
                     {
                        m_triangles[t]->m_selected = select;

                        vert[0]->m_marked2 = true;
                        vert[1]->m_marked2 = true;
                        vert[2]->m_marked2 = true;
                        goto next_triangle; // next triangle
                     }
                  }
               }

               if ( y2 >= ymin && y2 <= ymax )
               {
                  if ( coord[line][0][c1] == coord[line][1][c1] )
                  {
                     if ( coord[line][0][c1] >= x1 && coord[line][0][c1] <= x2 )
                     {
                        m_triangles[t]->m_selected = select;

                        vert[0]->m_marked2 = true;
                        vert[1]->m_marked2 = true;
                        vert[2]->m_marked2 = true;
                        goto next_triangle; // next triangle
                     }
                  }
                  else
                  {
                     x = (y2 - b[line]) / m[line];
                     if ( x >= x1 && x <= x2 )
                     {
                        m_triangles[t]->m_selected = select;

                        vert[0]->m_marked2 = true;
                        vert[1]->m_marked2 = true;
                        vert[2]->m_marked2 = true;
                        goto next_triangle; // next triangle
                     }
                  }
               }
            }

            if ( above && below )
            {
               // There was an intersection above and below the selection area,
               // This means we're inside the triangle, so add it to our selection list
               m_triangles[t]->m_selected = select;

               vert[0]->m_marked2 = true;
               vert[1]->m_marked2 = true;
               vert[2]->m_marked2 = true;
               goto next_triangle; // next triangle
            }

next_triangle:
            ; // because we need a statement after a label
         }
      }
   }

   if ( connected )
   {
      bool found = true;
      while ( found )
      {
         found = false;
         for ( t = 0; t < m_triangles.size(); t++ )
         {
            int count = 0;
            for ( unsigned v = 0; v < 3; v++ )
            {
               if ( m_vertices[ m_triangles[t]->m_vertexIndices[v] ]->m_marked2 )
               {
                  count++;
               }
            }

            if ( count > 0 && 
                  (count < 3 || m_triangles[t]->m_selected != select) ) 
            {
               found = true;

               m_triangles[t]->m_selected = select;

               for ( unsigned v = 0; v < 3; v++ )
               {
                  m_vertices[ m_triangles[t]->m_vertexIndices[v] ]->m_marked2 = true;
               }
            }
         }
      }
   }

   selectVerticesFromTriangles();

   endSelectionDifference();
   return true;
}

bool Model::selectGroupsInVolume( bool select, int c1, int c2, double x1, double y1, double x2, double y2, SelectionTest * test )
{
   LOG_PROFILE();

   beginSelectionDifference();

   selectTrianglesInVolume( select, c1, c2, x1, y1, x2, y2, false, test );

   if ( select )
   {
      selectGroupsFromTriangles( false );
   }
   else
   {
      selectGroupsFromTriangles( true );
   }

   endSelectionDifference();

   return true;
}

bool Model::selectBoneJointsInVolume( bool select, int c1, int c2, double a1, double b1, double a2, double b2, SelectionTest * test )
{
   LOG_PROFILE();

   beginSelectionDifference();

   if ( a1 > a2 )
   {
      double temp = a2;
      a2 = a1;
      a1 = temp;
   }

   if ( b1 > b2 )
   {
      double temp = b2;
      b2 = b1;
      b1 = temp;
   }

   for ( unsigned j = 0; j < m_joints.size(); j++ )
   {
      if ( m_joints[j]->m_visible
           && m_joints[j]->m_final.get( 3, c1 ) >= a1 && m_joints[j]->m_final.get( 3, c1 ) <= a2 
           && m_joints[j]->m_final.get( 3, c2 ) >= b1 && m_joints[j]->m_final.get( 3, c2 ) <= b2 )
      {
         if ( test )
            m_joints[j]->m_selected = test->shouldSelect( m_joints[j] ) ? select : m_joints[j]->m_selected;
         else
            m_joints[j]->m_selected = select;
      }
   }
   endSelectionDifference();

   return true;
}

bool Model::selectInVolumeXY( double x1, double y1, double x2, double y2, SelectionTest * test )
{
   LOG_PROFILE();

   switch ( m_selectionMode )
   {
      case SelectVertices:
         return selectVerticesInVolume ( true, 0, 1, x1, y1, x2, y2, test );
         break;
      case SelectTriangles:
         return selectTrianglesInVolume ( true, 0, 1, x1, y1, x2, y2, false, test );
         break;
      case SelectConnected:
         return selectTrianglesInVolume ( true, 0, 1, x1, y1, x2, y2, true, test );
         break;
      case SelectGroups:
         return selectGroupsInVolume ( true, 0, 1, x1, y1, x2, y2, test );
         break;
      case SelectJoints:
         return selectBoneJointsInVolume ( true, 0, 1, x1, y1, x2, y2, test );
         break;
      default:
         break;
   }
   return true;
}

bool Model::selectInVolumeXZ( double x1, double z1, double x2, double z2, SelectionTest * test )
{
   LOG_PROFILE();

   switch ( m_selectionMode )
   {
      case SelectVertices:
         return selectVerticesInVolume ( true, 0, 2, x1, z1, x2, z2, test );
         break;
      case SelectTriangles:
         return selectTrianglesInVolume ( true, 0, 2, x1, z1, x2, z2, false, test );
         break;
      case SelectConnected:
         return selectTrianglesInVolume ( true, 0, 2, x1, z1, x2, z2, true, test );
         break;
      case SelectGroups:
         return selectGroupsInVolume ( true, 0, 2, x1, z1, x2, z2, test );
         break;
      case SelectJoints:
         return selectBoneJointsInVolume ( true, 0, 2, x1, z1, x2, z2, test );
         break;
      default:
         break;
   }
   return true;
}

bool Model::selectInVolumeYZ( double y1, double z1, double y2, double z2, SelectionTest * test )
{
   LOG_PROFILE();

   switch ( m_selectionMode )
   {
      case SelectVertices:
         return selectVerticesInVolume ( true, 1, 2, y1, z1, y2, z2, test );
         break;
      case SelectTriangles:
         return selectTrianglesInVolume ( true, 1, 2, y1, z1, y2, z2, false, test );
         break;
      case SelectConnected:
         return selectTrianglesInVolume ( true, 1, 2, y1, z1, y2, z2, true, test );
         break;
      case SelectGroups:
         return selectGroupsInVolume ( true, 1, 2, y1, z1, y2, z2, test );
         break;
      case SelectJoints:
         return selectBoneJointsInVolume ( true, 1, 2, y1, z1, y2, z2, test );
         break;
      default:
         break;
   }
   return true;
}

bool Model::unselectInVolumeXY( double x1, double y1, double x2, double y2, SelectionTest * test )
{
   LOG_PROFILE();

   switch ( m_selectionMode )
   {
      case SelectVertices:
         return selectVerticesInVolume ( false, 0, 1, x1, y1, x2, y2, test );
         break;
      case SelectTriangles:
         return selectTrianglesInVolume ( false, 0, 1, x1, y1, x2, y2, false, test );
         break;
      case SelectConnected:
         return selectTrianglesInVolume ( false, 0, 1, x1, y1, x2, y2, true, test );
         break;
      case SelectGroups:
         return selectGroupsInVolume ( false, 0, 1, x1, y1, x2, y2, test );
         break;
      case SelectJoints:
         return selectBoneJointsInVolume ( false, 0, 1, x1, y1, x2, y2, test );
         break;
      default:
         break;
   }
   return true;
}

bool Model::unselectInVolumeXZ( double x1, double z1, double x2, double z2, SelectionTest * test )
{
   LOG_PROFILE();

   switch ( m_selectionMode )
   {
      case SelectVertices:
         return selectVerticesInVolume ( false, 0, 2, x1, z1, x2, z2, test );
         break;
      case SelectTriangles:
         return selectTrianglesInVolume ( false, 0, 2, x1, z1, x2, z2, false, test );
         break;
      case SelectConnected:
         return selectTrianglesInVolume ( false, 0, 2, x1, z1, x2, z2, true, test );
         break;
      case SelectGroups:
         return selectGroupsInVolume ( false, 0, 2, x1, z1, x2, z2, test );
         break;
      case SelectJoints:
         return selectBoneJointsInVolume ( false, 0, 2, x1, z1, x2, z2, test );
         break;
      default:
         break;
   }
   return true;
}

bool Model::unselectInVolumeYZ( double y1, double z1, double y2, double z2, SelectionTest * test )
{
   LOG_PROFILE();

   switch ( m_selectionMode )
   {
      case SelectVertices:
         return selectVerticesInVolume ( false, 1, 2, y1, z1, y2, z2, test );
         break;
      case SelectTriangles:
         return selectTrianglesInVolume ( false, 1, 2, y1, z1, y2, z2, false, test );
         break;
      case SelectConnected:
         return selectTrianglesInVolume ( false, 1, 2, y1, z1, y2, z2, true, test );
         break;
      case SelectGroups:
         return selectGroupsInVolume ( false, 1, 2, y1, z1, y2, z2, test );
         break;
      case SelectJoints:
         return selectBoneJointsInVolume ( false, 1, 2, y1, z1, y2, z2, test );
         break;
      default:
         break;
   }
   return true;
}

void Model::selectVerticesFromTriangles()
{
   LOG_PROFILE();

   unselectAllVertices();

   for ( unsigned t = 0; t < m_triangles.size(); t++ )
   {
      if ( m_triangles[t]->m_selected )
      {
         for ( int v = 0; v < 3; v++ )
         {
            m_vertices[ m_triangles[t]->m_vertexIndices[v] ]->m_selected = true;
         }
      }
   }
}

void Model::selectTrianglesFromGroups()
{
   LOG_PROFILE();

   unselectAllTriangles();

   for ( unsigned g = 0; g < m_groups.size(); g++ )
   {
      if ( m_groups[g]->m_selected )
      {
         for ( unsigned t = 0; t < m_groups[g]->m_triangleIndices.size(); t++ )
         {
            if ( m_triangles[ m_groups[g]->m_triangleIndices[t] ]->m_visible )
            {
               m_triangles[ m_groups[g]->m_triangleIndices[t] ]->m_selected = true;
            }
         }
      }
   }

   selectVerticesFromTriangles();
}

void Model::selectTrianglesFromVertices( bool all )
{
   LOG_PROFILE();

   unselectAllTriangles();

   for ( unsigned t = 0; t < m_triangles.size(); t++ )
   {
      if ( m_triangles[t]->m_visible )
      {
         int count = 0;
         for ( int v = 0; v < 3; v++ )
         {
            if ( m_vertices[ m_triangles[t]->m_vertexIndices[v] ]->m_selected )
            {
               count++;
            }

         }
         if ( all )
         {
            if ( count == 3 )
            {
               m_triangles[t]->m_selected = true;
            }
         }
         else
         {
            if ( count > 0 )
            {
               m_triangles[t]->m_selected = true;
            }
         }
      }
   }

   // Unselect vertices who don't have a triangle selected
   if ( all )
   {
      unselectAllVertices();
      selectVerticesFromTriangles();
   }
}

void Model::selectGroupsFromTriangles( bool all )
{
   LOG_PROFILE();

   unselectAllGroups();

   for ( unsigned g = 0; g < m_groups.size(); g++ )
   {
      unsigned count = 0;
      for ( unsigned t = 0; t < m_groups[g]->m_triangleIndices.size(); t++ )
      {
         if ( m_triangles[ m_groups[g]->m_triangleIndices[t] ]->m_selected )
         {
            count++;
         }
      }

      if ( all )
      {
         if ( count == m_groups[g]->m_triangleIndices.size() )
         {
            m_groups[g]->m_selected = true;
         }
         else
         {
            m_groups[g]->m_selected = false;
         }
      }
      else
      {
         if ( count > 0 )
         {
            m_groups[g]->m_selected = true;
         }
         else
         {
            m_groups[g]->m_selected = false;
         }
      }
   }

   // Unselect vertices who don't have a triangle selected
   unselectAllTriangles();
   selectTrianglesFromGroups();
}

bool Model::invertSelection()
{
   LOG_PROFILE();

   beginSelectionDifference();
   switch ( m_selectionMode )
   {
      case SelectVertices:
         for ( unsigned v = 0; v < m_vertices.size(); v++ )
         {
            if ( m_vertices[v]->m_visible )
            {
               m_vertices[v]->m_selected = m_vertices[v]->m_selected ? false : true;
            }
         }
         break;

      case SelectTriangles:
         for ( unsigned t = 0; t < m_triangles.size(); t++ )
         {
            if ( m_triangles[t]->m_visible )
            {
               m_triangles[t]->m_selected = m_triangles[t]->m_selected ? false : true;
            }
         }
         selectVerticesFromTriangles();
         break;

      case SelectGroups:
         for ( unsigned g = 0; g < m_groups.size(); g++ )
         {
            m_groups[g]->m_selected = m_groups[g]->m_selected ? false : true;
         }
         selectTrianglesFromGroups();
         break;

      case SelectJoints:
         for ( unsigned j = 0; j < m_joints.size(); j++ )
         {
            m_joints[j]->m_selected = m_joints[j]->m_selected ? false : true;
         }
         break;

      default:
         break;
   }
   endSelectionDifference();

   return true;
}

void Model::beginSelectionDifference()
{
   LOG_PROFILE();

   if ( m_undoEnabled )
   {
      unsigned t;
      for ( t = 0; t < m_vertices.size(); t++ )
      {
         m_vertices[t]->m_marked = m_vertices[t]->m_selected;
      }
      for ( t = 0; t < m_triangles.size(); t++ )
      {
         m_triangles[t]->m_marked = m_triangles[t]->m_selected;
      }
      for ( t = 0; t < m_groups.size(); t++ )
      {
         m_groups[t]->m_marked = m_groups[t]->m_selected;
      }
      for ( t = 0; t < m_joints.size(); t++ )
      {
         m_joints[t]->m_marked = m_joints[t]->m_selected;
      }
   }
}

void Model::endSelectionDifference()
{
   LOG_PROFILE();

   {
      MU_Select * undo = new MU_Select( SelectVertices );
      for ( unsigned t = 0; t < m_vertices.size(); t++ )
      {
         if ( m_vertices[t]->m_selected != m_vertices[t]->m_marked )
         {
            undo->setSelectionDifference( t, m_vertices[t]->m_selected, m_vertices[t]->m_marked );
         }
      }
      if ( undo->diffCount() > 0 )
      {
         sendUndo( undo );
      }
      else
      {
         undo->release();
      }
   }
   {
      MU_Select * undo = new MU_Select( SelectTriangles );
      for ( unsigned t = 0; t < m_triangles.size(); t++ )
      {
         if ( m_triangles[t]->m_selected != m_triangles[t]->m_marked )
         {
            undo->setSelectionDifference( t, m_triangles[t]->m_selected, m_triangles[t]->m_marked );
         }
      }
      if ( undo->diffCount() > 0 )
      {
         sendUndo( undo );
      }
      else
      {
         undo->release();
      }
   }
   {
      MU_Select * undo = new MU_Select( SelectGroups );
      for ( unsigned t = 0; t < m_groups.size(); t++ )
      {
         if ( m_groups[t]->m_selected != m_groups[t]->m_marked )
         {
            undo->setSelectionDifference( t, m_groups[t]->m_selected, m_groups[t]->m_marked );
         }
      }
      if ( undo->diffCount() > 0 )
      {
         sendUndo( undo );
      }
      else
      {
         undo->release();
      }
   }
   {
      MU_Select * undo = new MU_Select( SelectJoints );
      for ( unsigned t = 0; t < m_joints.size(); t++ )
      {
         if ( m_joints[t]->m_selected != m_joints[t]->m_marked )
         {
            undo->setSelectionDifference( t, m_joints[t]->m_selected, m_joints[t]->m_marked );
         }
      }
      if ( undo->diffCount() > 0 )
      {
         sendUndo( undo );
      }
      else
      {
         undo->release();
      }
   }
}

list<int> Model::getSelectedVertices()
{
   list<int> vertices;
   for ( unsigned t = 0; t < m_vertices.size(); t++ )
   {
      if ( m_vertices[t]->m_selected )
      {
         vertices.push_back( t );
      }
   }

   return vertices;
}

list<int> Model::getSelectedTriangles()
{
   list<int> triangles;
   for ( unsigned t = 0; t < m_triangles.size(); t++ )
   {
      if ( m_triangles[t]->m_selected )
      {
         triangles.push_back( t );
      }
   }

   return triangles;
}

list<int> Model::getSelectedGroups()
{
   list<int> groups;
   for ( unsigned t = 0; t < m_groups.size(); t++ )
   {
      if ( m_groups[t]->m_selected )
      {
         groups.push_back( t );
      }
   }

   return groups;
}

list<int> Model::getSelectedBoneJoints()
{
   list<int> joints;
   for ( unsigned t = 0; t < m_joints.size(); t++ )
   {
      if ( m_joints[t]->m_selected )
      {
         joints.push_back( t );
      }
   }

   return joints;
}

unsigned Model::getSelectedVertexCount()
{
   unsigned c = m_vertices.size();
   unsigned count = 0;

   for ( unsigned v = 0; v < c; v++ )
   {
      if ( m_vertices[v]->m_selected )
      {
         count++;
      }
   }

   return count;
}

unsigned Model::getSelectedTriangleCount()
{
   unsigned c = m_triangles.size();
   unsigned count = 0;

   for ( unsigned v = 0; v < c; v++ )
   {
      if ( m_triangles[v]->m_selected )
      {
         count++;
      }
   }

   return count;
}

unsigned Model::getSelectedBoneJointCount()
{
   unsigned c = m_joints.size();
   unsigned count = 0;

   for ( unsigned v = 0; v < c; v++ )
   {
      if ( m_joints[v]->m_selected )
      {
         count++;
      }
   }

   return count;
}

bool Model::parentJointSelected( int joint )
{
   while ( m_joints[joint]->m_parent >= 0 )
   {
      joint = m_joints[joint]->m_parent;

      if ( m_joints[joint]->m_selected )
      {
         return true;
      }
   }
   return false;
}

bool Model::directParentJointSelected( int joint )
{
   int p = m_joints[joint]->m_parent;
   if ( p >= 0 )
   {
      return m_joints[ p ]->m_selected;
   }
   return false;
}

bool Model::getSelectedBoundingRegion( double *x1, double *y1, double *z1, double *x2, double *y2, double *z2 )
{
   if ( x1 && y1 && z1 && x2 && y2 && z2 )
   {
      int visible = 0;
      bool havePoint = false;
      *x1 = *y1 = *z1 = *x2 = *y2 = *z2 = 0.0;

      for ( unsigned v = 0; v < m_vertices.size(); v++ )
      {
         if ( m_vertices[v]->m_visible && m_vertices[v]->m_selected )
         {
            if ( havePoint )
            {
               if ( m_vertices[v]->m_coord[0] < *x1 )
               {
                  *x1 = m_vertices[v]->m_coord[0];
               }
               if ( m_vertices[v]->m_coord[0] > *x2 )
               {
                  *x2 = m_vertices[v]->m_coord[0];
               }
               if ( m_vertices[v]->m_coord[1] < *y1 )
               {
                  *y1 = m_vertices[v]->m_coord[1];
               }
               if ( m_vertices[v]->m_coord[1] > *y2 )
               {
                  *y2 = m_vertices[v]->m_coord[1];
               }
               if ( m_vertices[v]->m_coord[2] < *z1 )
               {
                  *z1 = m_vertices[v]->m_coord[2];
               }
               if ( m_vertices[v]->m_coord[2] > *z2 )
               {
                  *z2 = m_vertices[v]->m_coord[2];
               }
            }
            else
            {
               *x1 = *x2 = m_vertices[v]->m_coord[0];
               *y1 = *y2 = m_vertices[v]->m_coord[1];
               *z1 = *z2 = m_vertices[v]->m_coord[2];
               havePoint = true;
            }
            visible++;
         }
      }

      for ( unsigned j = 0; j < m_joints.size(); j++ )
      {
         if ( m_joints[j]->m_selected )
         {
            double coord[3];
            m_joints[j]->m_absolute.getTranslation( coord );

            if ( havePoint )
            {
               if ( coord[0] < *x1 )
               {
                  *x1 = coord[0];
               }
               if ( coord[0] > *x2 )
               {
                  *x2 = coord[0];
               }
               if ( coord[1] < *y1 )
               {
                  *y1 = coord[1];
               }
               if ( coord[1] > *y2 )
               {
                  *y2 = coord[1];
               }
               if ( coord[2] < *z1 )
               {
                  *z1 = coord[2];
               }
               if ( coord[2] > *z2 )
               {
                  *z2 = coord[2];
               }
            }
            else
            {
               *x1 = *x2 = coord[0];
               *y1 = *y2 = coord[1];
               *z1 = *z2 = coord[2];
               havePoint = true;
            }

            visible++;
         }
      }

      return ( visible != 0 ) ? true : false;
   }

   return false;
}

bool Model::unselectAllVertices()
{
   LOG_PROFILE();

   for ( unsigned v = 0; v < m_vertices.size(); v++ )
   {
      m_vertices[ v ]->m_selected = false;
   }
   return true;
}

bool Model::unselectAllTriangles()
{
   for ( unsigned t = 0; t < m_triangles.size(); t++ )
   {
      m_triangles[ t ]->m_selected = false;
   }
   return true;
}

bool Model::unselectAllGroups()
{
   for ( unsigned m = 0; m < m_groups.size(); m++ )
   {
      m_groups[ m ]->m_selected = false;
   }
   return true;
}

bool Model::unselectAllBoneJoints()
{
   for ( unsigned j = 0; j < m_joints.size(); j++ )
   {
      m_joints[ j ]->m_selected = false;
   }
   return true;
}

bool Model::unselectAll()
{
   LOG_PROFILE();

   beginSelectionDifference();

   unselectAllVertices();
   unselectAllTriangles();
   unselectAllGroups();
   unselectAllBoneJoints();

   endSelectionDifference();
   return true;
}

#endif // MM3D_EDIT
