/*  Misfit Model 3D
 * 
 *  Copyright (c) 2004-2005 Kevin Worcester
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, 
 *  USA.
 *
 *  See the COPYING file for full license text.
 */


#include "torustool.h"

#include "pixmap/torustool.xpm"

#include "model.h"
#include "glmath.h"
#include "log.h"
#include "modelstatus.h"
#include "glmath.h"
#include "weld.h"

#include "mq3macro.h"
#include "mq3compat.h"

#include <math.h>
#include <vector>

#include <qwidget.h>

using std::vector;
using std::list;

TorusTool::TorusTool()
   : m_segments( 8 ),
     m_sides( 8 ),
     m_width( 50 )
{
}

TorusTool::~TorusTool()
{
}

void TorusTool::activated( int arg, Model * model, QMainWindow * mainwin )
{
   log_debug( "torus activated\n" );
   m_widget = new TorusToolWidget( this, mainwin );
#ifdef HAVE_QT4
   ////mainwin->addDockWindow( m_widget, DockBottom );
#endif
   m_widget->show();
}

void TorusTool::deactivated()
{
   m_widget->close();
}

void TorusTool::mouseButtonDown( Parent * parent, int buttonState, int x, int y )
{
   Model * model = parent->getModel();

   model->unselectAll();

   m_polys.clear();
   m_polys.setModel( model );

   int index = 0;
   double pos[3] = {0,0,0};

   if ( !parent->getXValue( x, y, &pos[0] ) ) { index = 1; }
   if ( !parent->getYValue( x, y, &pos[1] ) ) { index = 2; }
   if ( !parent->getZValue( x, y, &pos[2] ) ) { index = 0; }

   m_startX = pos[0];
   m_startY = pos[1];
   m_startZ = pos[2];

   double rx = 0.0;
   double ry = 0.0;

   double cx = 0.0;
   double cy = 0.0;
   double cz = 0.0;

   m_diameter = (double) m_width / 100.0;
   double rad = 1.0 - (m_diameter / 2.0);

   for ( unsigned t = 0; t <= m_segments; t++ )
   {
      double angle = (PI * 2) * ((double) t / (double) m_segments );
      rx = cos( angle );
      ry = sin( angle );

      for ( unsigned n = 0; n <= m_sides; n++ )
      {
         angle = (PI * 2) * ((double) n / (double) m_sides );
         double c = cos( angle ) * m_diameter * 0.5;
         cz = sin( angle );

         cx = (rx * rad) + (rx * c);
         cy = (ry * rad) + (ry * c);

         cx += 1.0;
         cy += 1.0;
         cx /= 2.0;
         cy /= 2.0;

         switch ( index )
         {
            case 0:
               m_polys.addVertex( cx, cy, cz );
               break;
            case 1:
               m_polys.addVertex( cz, cy, cx );
               break;
            default:
               m_polys.addVertex( cx, cz, cy );
               break;
         }
      }

      if ( t > 0 )
      {
         unsigned vbase1 = model->getVertexCount() - ((m_sides+1) * 2);
         unsigned vbase2 = model->getVertexCount() -  (m_sides+1);

         unsigned i;
         for ( i = 0; i < m_sides; i++ )
         {
            m_polys.addTriangle( vbase1 + i,     vbase1 + i + 1, vbase2 + i );
            m_polys.addTriangle( vbase1 + i + 1, vbase2 + i + 1, vbase2 + i );
         }
      }
   }

   m_polys.startCoordinates( m_startX, m_startY, m_startZ );

   m_polys.selectVertices();
   m_polys.selectTriangles();

   parent->updateAllViews();

   model_status( model, StatusNormal, STATUSTIME_SHORT, "Torus created" );
}

void TorusTool::mouseButtonUp( Parent * parent, int buttonState, int x, int y )
{
   Model * model = parent->getModel();
   weldSelectedVertices( model );
   m_polys.clear();
   parent->updateAllViews();
}

void TorusTool::mouseButtonMove( Parent * parent, int buttonState, int x, int y )
{
   int index = 0;
   double pos[3] = {0,0,0};

   if ( !parent->getXValue( x, y, &pos[0] ) ) { index = 1; }
   if ( !parent->getYValue( x, y, &pos[1] ) ) { index = 2; }
   if ( !parent->getZValue( x, y, &pos[2] ) ) { index = 0; }

   double xdiff = pos[0] - m_startX;
   double ydiff = pos[1] - m_startY;
   double zdiff = pos[2] - m_startZ;

   bool invert = true;

   switch ( index )
   {
      case 0:
         if ( m_circle )
         {
            double diff = sqrt( fabs(xdiff * ydiff) );
            ydiff = (ydiff < 0.0) ? -diff : diff;
            xdiff = (xdiff < 0.0) ? -diff : diff;
         }

         zdiff = ( fabs( xdiff ) > fabs( ydiff ) ) 
            ? fabs( xdiff ) 
            : fabs( ydiff );
         zdiff *= (m_diameter * 0.25);

         m_polys.updateDimensions( 
               xdiff, ydiff, zdiff );

         if ( ( xdiff < 0.0 && ydiff < 0.0 )
               || ( xdiff > 0.0 && ydiff > 0.0 ) )
         {
            invert = false;
         }
         break;
      case 1:
         if ( m_circle )
         {
            double diff = sqrt( fabs(zdiff * ydiff) );
            ydiff = (ydiff < 0.0) ? -diff : diff;
            zdiff = (zdiff < 0.0) ? -diff : diff;
         }

         xdiff = ( fabs( zdiff ) > fabs( ydiff ) ) 
            ? fabs( zdiff ) 
            : fabs( ydiff );
         xdiff *= (m_diameter * 0.25);

         m_polys.updateDimensions( 
               xdiff, ydiff, zdiff );

         if ( ( zdiff > 0.0 && ydiff < 0.0 )
               || ( zdiff < 0.0 && ydiff > 0.0 ) )
         {
            invert = false;
         }
         break;
      default:
         if ( m_circle )
         {
            double diff = sqrt( fabs(xdiff * zdiff) );
            zdiff = (zdiff < 0.0) ? -diff : diff;
            xdiff = (xdiff < 0.0) ? -diff : diff;
         }

         ydiff = ( fabs( xdiff ) > fabs( zdiff ) ) 
            ? fabs( xdiff ) 
            : fabs( zdiff );
         ydiff *= (m_diameter * 0.25);

         m_polys.updateDimensions( 
               xdiff, ydiff, zdiff );

         if ( ( xdiff > 0.0 && zdiff < 0.0 )
               || ( xdiff < 0.0 && zdiff > 0.0 ) )
         {
            invert = false;
         }
         break;
   }

   if ( invert == m_polys.areNormalsInverted() )
   {
      m_polys.invertNormals();
   }

   parent->updateAllViews();
}

const char ** TorusTool::getPixmap()
{
   return (const char **) torustool_xpm;
}

void TorusTool::setSegmentsValue( int newValue )
{
   m_segments = newValue;
}

void TorusTool::setSidesValue( int newValue )
{
   m_sides = newValue;
}

void TorusTool::setWidthValue( int newValue )
{
   m_width = newValue;
}

void TorusTool::setCircleValue( bool newValue )
{
   m_circle = newValue;
}

