/*  Misfit Model 3D
 * 
 *  Copyright (c) 2004-2005 Kevin Worcester
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, 
 *  USA.
 *
 *  See the COPYING file for full license text.
 */


#include "ellipsetool.h"

#include "pixmap/ellipsetool.xpm"

#include "model.h"
#include "glmath.h"
#include "log.h"
#include "modelstatus.h"

#include "mq3macro.h"
#include "mq3compat.h"

#include <math.h>
#include <vector>

#include <qwidget.h>

using std::vector;
using std::list;

EllipsoidTool::EllipsoidTool()
   : m_smoothness( 2 ),
     m_isSphere( false )
{
}

EllipsoidTool::~EllipsoidTool()
{
}

void EllipsoidTool::activated( int arg, Model * model, QMainWindow * mainwin )
{
   log_debug( "ellipse activated\n" );
   m_widget = new EllipsoidToolWidget( this, mainwin );
#ifdef HAVE_QT4
   //mainwin->addDockWindow( m_widget, DockBottom );
#endif
   m_widget->show();
}

void EllipsoidTool::deactivated()
{
   m_widget->close();
}

void EllipsoidTool::mouseButtonDown( Parent * parent, int buttonState, int x, int y )
{
   Model * model = parent->getModel();

   model->unselectAll();

   // Make top and bottom vertices
   int v1 = model->addVertex( 0,  1, 0 );
   int v2 = model->addVertex( 0, -1, 0 );

   log_debug( "added vertices: %d and %d\n", v1, v2 );

   // Make center vertices
   double offset = sin( 30 * PIOVER180 );
   double adjust = cos( 30 * PIOVER180 );

   vector<int> top;
   vector<int> bot;

   for ( int t = 0; t < 5; t++ )
   {
      double xoff = sin ( ( t * 72) * PIOVER180 ) * adjust;
      double zoff = cos ( ( t * 72) * PIOVER180 ) * adjust;

      int v = model->addVertex( xoff, offset, zoff );
      top.push_back( v );

      xoff = sin ( ( t * 72 + 36) * PIOVER180 ) * adjust;
      zoff = cos ( ( t * 72 + 36) * PIOVER180 ) * adjust;

      v = model->addVertex( xoff, -offset, zoff );
      bot.push_back( v );
   }

   // Create top and bottom faces
   model->selectTriangle( model->addTriangle( v1, top.back(),  top.front() ) );
   model->selectTriangle( model->addTriangle( v2, bot.front(), bot.back()  ) );

   vector<int>::iterator it1;
   vector<int>::iterator it2;
   vector<int>::reverse_iterator rit1;
   vector<int>::reverse_iterator rit2;

   for ( it1 = top.begin(); ; it1++ )
   {
      it2 = it1;
      it2++;

      if ( it2 == top.end() )
      {
         break;
      }

      model->selectTriangle( model->addTriangle( v1, *it1, *it2 ) );
   }

   for ( rit1 = bot.rbegin(); ; rit1++ )
   {
      rit2 = rit1;
      rit2++;

      if ( rit2 == bot.rend() )
      {
         break;
      }

      model->selectTriangle( model->addTriangle( v2, *rit1, *rit2 ) );
   }

   model->selectTriangle( model->addTriangle( top[0], bot[0], top[1] ) );
   model->selectTriangle( model->addTriangle( top[1], bot[1], top[2] ) );
   model->selectTriangle( model->addTriangle( top[2], bot[2], top[3] ) );
   model->selectTriangle( model->addTriangle( top[3], bot[3], top[4] ) );
   model->selectTriangle( model->addTriangle( top[4], bot[4], top[0] ) );

   model->selectTriangle( model->addTriangle( bot[1], top[1], bot[0] ) );
   model->selectTriangle( model->addTriangle( bot[2], top[2], bot[1] ) );
   model->selectTriangle( model->addTriangle( bot[3], top[3], bot[2] ) );
   model->selectTriangle( model->addTriangle( bot[4], top[4], bot[3] ) );
   model->selectTriangle( model->addTriangle( bot[0], top[0], bot[4] ) );

   for ( unsigned i = 0; i < m_smoothness; i++ )
   {
      model->subdivideSelectedTriangles();
   }

   list<int> verts;
   list<int>::iterator vit;

   verts = model->getSelectedVertices();

   for ( vit = verts.begin(); vit != verts.end(); vit++ )
   {
      EllipseVertices ev;

      ev.v = *vit;
      model->getVertexCoords( (*vit), ev.coords );

      double len = sqrt( ev.coords[0]*ev.coords[0] 
            + ev.coords[1]*ev.coords[1] 
            + ev.coords[2]*ev.coords[2] );

      ev.coords[0] = ev.coords[0] / len;
      ev.coords[1] = ev.coords[1] / len;
      ev.coords[2] = ev.coords[2] / len;

      m_vertices.push_back( ev );
   }

   int index = 0;
   double pos[3] = {0,0,0};
   double rad[3] = {0,0,0};

   if ( !parent->getXValue( x, y, &pos[0] ) ) { index = 1; }
   if ( !parent->getYValue( x, y, &pos[1] ) ) { index = 2; }
   if ( !parent->getZValue( x, y, &pos[2] ) ) { index = 0; }

   m_startX = pos[0];
   m_startY = pos[1];
   m_startZ = pos[2];

   updateVertexCoords( model, pos[0], pos[1], pos[2], rad[0], rad[1], rad[2] );

   parent->updateAllViews();

   model_status( model, StatusNormal, STATUSTIME_SHORT, "Ellipsoid created" );
}

void EllipsoidTool::mouseButtonUp( Parent * parent, int buttonState, int x, int y )
{
   //Model * model = parent->getModel();
   m_vertices.clear();
}

void EllipsoidTool::mouseButtonMove( Parent * parent, int buttonState, int x, int y )
{
   Model * model = parent->getModel();

   int index = 0;
   double pos[3] = {0,0,0};
   double rad[3] = {0,0,0};

   if ( !parent->getXValue( x, y, &pos[0] ) ) { index = 1; }
   if ( !parent->getYValue( x, y, &pos[1] ) ) { index = 2; }
   if ( !parent->getZValue( x, y, &pos[2] ) ) { index = 0; }

   if ( m_isSphere )
   {
      double sphereRadius = 0;
      double a, b;

      switch ( index )
      {
         case 0:
            a = fabs((m_startX - pos[0]) / 2);
            b = fabs((m_startY - pos[1]) / 2);
            //sphereRadius = sqrt( a*a + b*b );
            sphereRadius = (a < b) ? a : b;
            break;
         case 1:
            b = fabs((m_startY - pos[1]) / 2);
            a = fabs((m_startZ - pos[2]) / 2);
            //sphereRadius = sqrt( a*a + b*b );
            sphereRadius = (a < b) ? a : b;
            break;
         case 2:
            a = fabs((m_startX - pos[0]) / 2);
            b = fabs((m_startZ - pos[2]) / 2);
            //sphereRadius = sqrt( a*a + b*b );
            sphereRadius = (a < b) ? a : b;
            break;
         default:
            break;
      }

      rad[0] = rad[1] = rad[2] = sphereRadius;
   }
   else
   {
      switch ( index )
      {
         case 0:
            rad[0] = fabs(m_startX - pos[0]) / 2;
            rad[1] = fabs(m_startY - pos[1]) / 2;
            rad[2] = ( rad[0] < rad[1] ) ? rad[0] : rad[1];
            break;
         case 1:
            rad[1] = fabs(m_startY - pos[1]) / 2;
            rad[2] = fabs(m_startZ - pos[2]) / 2;
            rad[0] = ( rad[2] < rad[1] ) ? rad[2] : rad[1];
            break;
         case 2:
            rad[0] = fabs(m_startX - pos[0]) / 2;
            rad[2] = fabs(m_startZ - pos[2]) / 2;
            rad[1] = ( rad[0] < rad[2] ) ? rad[0] : rad[2];
            break;
         default:
            break;
      }
   }

   pos[0] = (pos[0] + m_startX) / 2;
   pos[1] = (pos[1] + m_startY) / 2;
   pos[2] = (pos[2] + m_startZ) / 2;

   updateVertexCoords( model, pos[0], pos[1], pos[2], rad[0], rad[1], rad[2] );

   parent->updateAllViews();
}

const char ** EllipsoidTool::getPixmap()
{
   return (const char **) ellipsetool_xpm;
}

void EllipsoidTool::updateVertexCoords( Model * model, double x, double y, double z,
      double xrad, double yrad, double zrad )
{
   list<EllipseVertices>::iterator it;

   for ( it = m_vertices.begin(); it != m_vertices.end(); it++ )
   {
      model->moveVertex( (*it).v, 
            (*it).coords[0]*xrad + x, 
            (*it).coords[1]*yrad + y, 
            (*it).coords[2]*zrad + z );
   }
}

void EllipsoidTool::setSmoothnessValue( int newValue )
{
   m_smoothness = newValue;
}

void EllipsoidTool::setSphere( bool o )
{
   m_isSphere = o;
}
