/*  Misfit Model 3D
 * 
 *  Copyright (c) 2004-2005 Kevin Worcester
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, 
 *  USA.
 *
 *  See the COPYING file for full license text.
 */


#ifndef __MVIEWPORT_H
#define __MVIEWPORT_H

#include <qgl.h>
#include "tool.h"
#include "decal.h"
#include "texture.h"
#include "glxfont.h"

#include <qimage.h>
#include <list>
#include <string>

using std::list;

class Model;
class Decal;
class Toolbox;
class QTimer;

typedef list<Decal *> DecalList;

class ModelViewport : public QGLWidget, public Tool::Parent
{
   Q_OBJECT
   public:
      enum
      {
         MAX_VERTICAL_UNITS = 32767,
         MAX_HORIZONTAL_UNITS = 32767
      };

      typedef enum 
      {
         ViewWireframe,
         ViewFlat,
         ViewSmooth,
         ViewTexture,
         ViewAlpha
      } ViewOptions;

      typedef enum
      {
         ScrollButtonPan,
         ScrollButtonLeft,
         ScrollButtonRight,
         ScrollButtonUp,
         ScrollButtonDown,
         ScrollButtonMAX
      } ScrollButtonE;


      ModelViewport( QWidget * parent = NULL, const char * name = "" );
      virtual ~ModelViewport();

      void freeTextures();
      double getZoomLevel() { return m_zoomLevel; };

      void frameArea( double x1, double y1, double z1, double x2, double y2, double z2 );
      void zoomIn();
      void zoomOut();

      void scrollUp();
      void scrollDown();
      void scrollLeft();
      void scrollRight();

      void setModel( Model * model ) { m_model = model; };
      void setToolbox( Toolbox * toolbox ) { m_toolbox = toolbox; };
      
      int constructButtonState( QMouseEvent * e );

      // Tool::Parent methods

      Model * getModel() { return m_model; };
      ViewDirection getViewDirection() { return m_viewDirection; };
      void updateView();
      void update3dView();
      void updateAllViews() { emit modelUpdated(); };

      bool getXValue( int x, int y, double * val );
      bool getYValue( int x, int y, double * val );
      bool getZValue( int x, int y, double * val );

      void addDecal( Decal * decal );
      void removeDecal( Decal * decal );

      void copyContentsToTexture( Texture * tex );
      
   signals:
      void zoomLevelChanged( const QString & zoomStr );
      void viewDirectionChanged( int dir );
      void modelUpdated();

   public slots:

      void viewChangeEvent( int dir );
      void setZoomLevel( double zoomLevel );

      void wireframeEvent();
      void flatEvent();
      void smoothEvent();
      void textureEvent();
      void alphaEvent();

      void scrollTimeout();

   protected slots:
      void wheelEvent( QWheelEvent * e );
      void mouseMoveEvent( QMouseEvent * e );
      void mousePressEvent( QMouseEvent * e );
      void mouseReleaseEvent( QMouseEvent * e );
      void keyPressEvent( QKeyEvent * e );
      void focusInEvent( QFocusEvent * e );
      void focusOutEvent( QFocusEvent * e );
      //void dragEnterEvent( QDragMoveEvent * e );

   protected:
      void initializeGL();
      void paintGL();
      void resizeGL( int w, int h );

      void checkGlErrors();

      void updateBackground();

      void adjustViewport();
      void setViewportDraw();
      void setViewportOverlay();

      void drawGridLines();
      void drawOrigin();
      void drawBackground();
      void drawOverlay();

      void makeTextureFromImage( const QImage & i, GLuint & t );

      Model * m_model;

      ViewDirection m_viewDirection;

      double m_centerX;
      double m_centerY;
      double m_rotX;
      double m_rotY;
      double m_rotZ;
      double m_width;
      double m_height;
      double m_zoomLevel;
      double m_far;
      double m_near;
      double m_farOrtho;
      double m_nearOrtho;

      int m_viewportWidth;
      int m_viewportHeight;

      GLuint    m_backgroundTexture;
      Texture * m_texture;
      std::string m_backgroundFile;

      QTimer        * m_scrollTimer;
      bool            m_inOverlay;
      ScrollButtonE m_overlayButton;
      GLuint m_scrollTextures[2];

      // Font stuff
      int m_displayList;
      GlxFontData * m_glxFontData;

      bool m_capture;
      bool m_texturesLoaded;
      ViewOptions m_viewOptions;

      QPoint m_scrollStartPosition;
      QColor m_backColor;

      DecalList m_decals;

      Toolbox * m_toolbox;
};

#endif // __MVIEWPORT_H
