/*  Misfit Model 3D
 * 
 *  Copyright (c) 2004-2005 Kevin Worcester
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, 
 *  USA.
 *
 *  See the COPYING file for full license text.
 */


#include "model.h"
#include "log.h"
#include "msg.h"
#include "texmgr.h"
#include "texture.h"

#include <math.h>
#include <map>

#ifdef MM3D_EDIT
#include "modelundo.h"
#endif // MM3D_EDIT

int Model::equivalent( const Model * model, int compareMask, double tolerance )
{
   int matchVal = 0;

   unsigned numVertices    = m_vertices.size();
   unsigned numTriangles   = m_triangles.size();
   unsigned numGroups      = m_groups.size();
   unsigned numJoints      = m_joints.size();
   unsigned numTextures    = m_materials.size();
   unsigned numFrameAnims  = m_frameAnims.size();
   unsigned numSkelAnims   = m_skelAnims.size();

   unsigned t = 0;
   unsigned v = 0;
   if ( compareMask & CompareGeometry )
   {
      if ( numVertices == model->m_vertices.size() && numTriangles == model->m_triangles.size() )
      {
         bool match = true;
         for ( v = 0; match && v < numVertices; v++ )
         {
            for ( unsigned i = 0; i < 3; i++ )
            {
               if ( fabs( m_vertices[v]->m_coord[i] - model->m_vertices[v]->m_coord[i]) > tolerance )
               {
                  match = false;
                  log_debug( "match failed at vertex coordinates %d[%d]\n", v, i );
               }
            }
         }

         for ( t = 0; match && t < numTriangles; t++ )
         {
            if (  m_triangles[t]->m_vertexIndices[0] != model->m_triangles[t]->m_vertexIndices[0]
               || m_triangles[t]->m_vertexIndices[1] != model->m_triangles[t]->m_vertexIndices[1]
               || m_triangles[t]->m_vertexIndices[2] != model->m_triangles[t]->m_vertexIndices[2] )
            {
               match = false;
               log_debug( "match failed at triangle %d\n", t );
               log_debug( "   %d %d %d\n", 
                     m_triangles[t]->m_vertexIndices[0], 
                     m_triangles[t]->m_vertexIndices[1], 
                     m_triangles[t]->m_vertexIndices[2] );
               log_debug( "   %d %d %d\n", 
                     model->m_triangles[t]->m_vertexIndices[0], 
                     model->m_triangles[t]->m_vertexIndices[1], 
                     model->m_triangles[t]->m_vertexIndices[2] );
            }
         }

         if ( match )
         {
            matchVal |= CompareGeometry;
            matchVal |= CompareFaces;
         }
      }
   }
   else if ( compareMask & CompareFaces )
   {
      if ( numTriangles == model->m_triangles.size() )
      {
         for ( t = 0; t < numTriangles; t++ )
         {
            m_triangles[t]->m_marked = false;
            model->m_triangles[t]->m_marked = false;
         }

         for ( t = 0; t < numTriangles; t++ )
         {
            for ( unsigned i = 0; i < model->m_triangles.size(); i++ )
            {
               if ( !model->m_triangles[i]->m_marked )
               {
                  bool match = true;
                  for ( unsigned j = 0; j < 3; j++ )  // triangle vertex
                  {
                     Vertex * v1 = m_vertices[ m_triangles[t]->m_vertexIndices[j] ];
                     Vertex * v2 = model->m_vertices[ model->m_triangles[i]->m_vertexIndices[j] ];
                     for ( unsigned k = 0; k < 3; k++ ) // vertex coord
                     {
                        if (   fabs( v1->m_coord[k] - v2->m_coord[k] ) > tolerance )
                        {
                           match = false;
                        }
                     }
                  }
                  if ( match )
                  {
                     m_triangles[t]->m_marked = true;
                     model->m_triangles[i]->m_marked = true;
                     break; // break inner loop
                  }
               }
            }
         }

         bool match = true;
         for ( t = 0; t < numTriangles; t++ )
         {
            if ( !m_triangles[t]->m_marked )
            {
               match = false;
               log_debug( "match failed at triangle %d, no match for face\n", t );
            }
         }
         if ( match )
         {
            matchVal |= CompareFaces;
         }
      }
   }

   if ( compareMask & CompareGroups )
   {
      std::list<int> l1;
      std::list<int> l2;
      std::list<int>::iterator it1;
      std::list<int>::iterator it2;

      bool match = false;

      if ( getGroupCount() == model->getGroupCount() )
      {
         unsigned int t = 0;
         unsigned int gcount = getGroupCount();
         unsigned int tcount = getTriangleCount();
         match = true;

         for ( unsigned int g = 0; match && g < gcount; g++ )
         {
            for ( t = 0; t < tcount; t++ )
            {
               m_triangles[t]->m_marked = false;
            }

            std::list<int> l1 = getGroupTriangles( g );
            std::list<int> l2 = model->getGroupTriangles( g );

            if ( l1.size() == l2.size() )
            {
               for ( it1  = l1.begin(); match && it1 != l1.end(); it1++ )
               {
                  if ( m_triangles[ *it1 ]->m_marked )
                  {
                     log_debug( "match failed in groups, triangle %d is in group %d twice\n", (*it1), g );
                  }
                  else
                  {
                     m_triangles[ *it1 ]->m_marked = true;
                  }
               }
               for ( it2  = l2.begin(); match && it2 != l2.end(); it2++ )
               {
                  if ( !m_triangles[ *it2 ]->m_marked )
                  {
                     log_debug( "match failed in groups, triangle %d is in group %d for rhs but not lhs\n", (*it2), g );
                  }
                  else
                  {
                     m_triangles[ *it2 ]->m_marked = false;
                  }
               }
            }
            else
            {
               log_debug( "match failed in groups, triangle count %d != %d for group %d\n", l1.size(), l2.size(), g );
               match = false;
            }
         }
      }

      if ( match )
      {
         matchVal |= CompareGroups;
      }
   }

   if ( compareMask & CompareSkeleton )
   {
      if ( numJoints == model->m_joints.size() && numVertices == model->m_vertices.size() )
      {
         bool match = true;

         for ( unsigned j = 0; match && j < numJoints; j++ )
         {
            if ( m_joints[j]->m_parent != model->m_joints[j]->m_parent )
            {
               match = false;
               log_debug( "match failed at joint parent for joint %d\n", j );
            }
            for ( unsigned i = 0; i < 3; i++ )
            {
               if ( fabs( m_joints[j]->m_localRotation[i] - model->m_joints[j]->m_localRotation[i] ) > tolerance )
               {
                  log_debug( "match failed at joint rotation for joint %d\n", j );
                  match = false;
               }
               if ( fabs( m_joints[j]->m_localTranslation[i] - model->m_joints[j]->m_localTranslation[i] ) > tolerance )
               {
                  log_debug( "match failed at joint translation for joint %d\n", j );
                  match = false;
               }
            }
         }

         for ( v = 0; match && v < numVertices; v++ )
         {
            if ( m_vertices[v]->m_boneId != model->m_vertices[v]->m_boneId )
            {
               log_debug( "match failed at vertex joint assignment for vertex %d\n", v );
               match = false;
            }
         }

         if ( match )
         {
            matchVal |= CompareSkeleton;
         }
      }
   }

   // TODO want to check texture contents?
   if ( compareMask & CompareTextures )
   {
      if (     numTextures  == model->m_materials.size() 
            && numTriangles == model->m_triangles.size() 
            && numGroups    == model->m_groups.size()     )
      {
         bool match = true;

         for ( t = 0; t < numTextures; t++ )
         {
            for ( unsigned i = 0; i < 4; i++ )
            {
               if ( fabs( m_materials[t]->m_ambient[i] - model->m_materials[t]->m_ambient[i] ) > tolerance )
               {
                  log_debug( "match failed at texture ambient for texture %d\n", t );
                  match = false;
               }
               if ( fabs( m_materials[t]->m_diffuse[i] - model->m_materials[t]->m_diffuse[i] ) > tolerance )
               {
                  log_debug( "match failed at texture diffuse for texture %d\n", t );
                  match = false;
               }
               if ( fabs( m_materials[t]->m_specular[i] - model->m_materials[t]->m_specular[i] ) > tolerance )
               {
                  log_debug( "match failed at texture specular for texture %d\n", t );
                  match = false;
               }
               if ( fabs( m_materials[t]->m_emissive[i] - model->m_materials[t]->m_emissive[i] ) > tolerance )
               {
                  log_debug( "match failed at texture emissive for texture %d\n", t );
                  match = false;
               }

            }

            if ( m_materials[t]->m_shininess != model->m_materials[t]->m_shininess )
            {
               log_debug( "match failed at texture shininess for texture %d\n", t );
               match = false;
            }
         }

         for ( t = 0; match && t < numTriangles; t++ )
         {
            for ( unsigned i = 0; i < 3; i++ )
            {
               if ( fabs( m_triangles[t]->m_s[i] - model->m_triangles[t]->m_s[i] ) > tolerance )
               {
                  log_debug( "match failed at texture coordinates for triangle %d\n", t );
                  match = false;
               }
            }
         }

         for ( unsigned g = 0; match && g < numGroups; g++ )
         {
            if ( m_groups[g]->m_smooth != model->m_groups[g]->m_smooth )
            {
               log_debug( "match failed at group smoothness group %d\n", g );
               match = false;
            }
            if ( m_groups[g]->m_materialIndex != model->m_groups[g]->m_materialIndex )
            {
               log_debug( "match failed at group texture for group %d\n", g );
               match = false;
            }

            unsigned numGroupTriangles = m_groups[g]->m_triangleIndices.size();
            if ( numGroupTriangles == model->m_groups[g]->m_triangleIndices.size() )
            {
               for ( t = 0; match && t < numGroupTriangles; t++ )
               {
                  if ( m_groups[g]->m_triangleIndices[t] != model->m_groups[g]->m_triangleIndices[t] )
                  {
                     log_debug( "match failed at group triangle assignments for group %d triangle %d\n", g, t );
                     match = false;
                  }
               }
            }
            else
            {
               log_debug( "match failed at group triangle count for group %d\n", g );
               match = false;
            }
         }

         if ( match )
         {
            matchVal |= CompareTextures;
         }
      }
      else
      {
         log_debug( "not comparing because of primitive mismatch\n" );
         log_debug( " (%d/%d) (%d/%d) (%d/%d)\n",
               numTextures,  model->m_materials.size(),
               numTriangles, model->m_triangles.size(), 
               numGroups,    model->m_groups.size() );
      }
   }

   if ( compareMask & CompareAnimData )
   {
      if (     numFrameAnims == model->m_frameAnims.size()
            && numSkelAnims == model->m_skelAnims.size()    )
      {
         bool match = true;

         for ( t = 0; match && t < numFrameAnims; t++ )
         {
            if ( m_frameAnims[t]->m_frameVertices.size() != model->m_frameAnims[t]->m_frameVertices.size() )
            {
               log_debug( "match failed at frame animation frame count for frame anim %d\n", t );
               match = false;
            }
            if ( fabs( m_frameAnims[t]->m_fps - model->m_frameAnims[t]->m_fps ) > tolerance )
            {
               log_debug( "match failed at frame animation fps for frame anim %d\n", t );
               match = false;
            }

            for ( unsigned i = 0; match && i < m_frameAnims[t]->m_frameVertices.size(); i++ )
            {
               for ( v = 0; match && v < numVertices; v++ )
               {
                  for ( unsigned n = 0; n < 3; n++ )
                  {
                     if (   fabs( (*m_frameAnims[t]->m_frameVertices[i])[v]->m_coord[n]
                          - (*model->m_frameAnims[t]->m_frameVertices[i])[v]->m_coord[n] ) > tolerance )
                     {
                        log_debug( "match failed at frame animation coords for frame anim %d, vertex %d\n", t, v );
                        match = false;
                     }
                  }
               }
            }
         }

         for ( t = 0; match && t < numSkelAnims; t++ )
         {
            if ( m_skelAnims[t]->m_frameCount != model->m_skelAnims[t]->m_frameCount )
            {
               log_debug( "match failed at skel animation frame count for skel anim %d\n", t );
               match = false;
            }
            if ( fabs( m_skelAnims[t]->m_fps - model->m_skelAnims[t]->m_fps ) > tolerance )
            {
               log_debug( "match failed at skel animation fps for skel anim %d\n", t );
               match = false;
            }

            for ( unsigned j = 0; j < m_skelAnims[t]->m_jointKeyframes.size(); j++ )
            {
               for ( unsigned i = 0; i < m_skelAnims[t]->m_jointKeyframes[j].size(); i++ )
               {
                  if (    m_skelAnims[t]->m_jointKeyframes[j][i]->m_isRotation 
                       != model->m_skelAnims[t]->m_jointKeyframes[j][i]->m_isRotation )
                  {
                     log_debug( "match failed at skel animation keyframe rotation for skel anim %d joint %d, keyframe %i\n", t, j, i );
                     match = false;
                  }
                  if (   fabs( m_skelAnims[t]->m_jointKeyframes[j][i]->m_time 
                       - model->m_skelAnims[t]->m_jointKeyframes[j][i]->m_time ) > tolerance )
                  {
                     log_debug( "match failed at skel animation keyframe time for skel anim %d joint %d, keyframe %i\n", t, j, i );
                     match = false;
                  }
                  for ( unsigned n = 0; n < 3; n++ )
                  {
                     if (   fabs( m_skelAnims[t]->m_jointKeyframes[j][i]->m_parameter[n] 
                              - model->m_skelAnims[t]->m_jointKeyframes[j][i]->m_parameter[n] ) > tolerance )
                     {
                        log_debug( "match failed at skel animation keyframe parameter for skel anim %d joint %d, keyframe %i\n", t, j, i );
                        match = false;
                     }
                  }
               }
            }
         }

         if ( match )
         {
            matchVal |= CompareAnimData;
            matchVal |= CompareAnimSets; 
         }
      }
   }

   if ( (compareMask & CompareAnimSets) && !(matchVal & CompareAnimData) )
   {
      if (     numFrameAnims == model->m_frameAnims.size()
            && numSkelAnims == model->m_skelAnims.size()    )
      {
         bool match = true;

         for ( t = 0; match && t < numFrameAnims; t++ )
         {
            if ( m_frameAnims[t]->m_frameVertices.size() != model->m_frameAnims[t]->m_frameVertices.size() )
            {
               log_debug( "match failed at frame animation frame count for frame anim %d\n", t );
               match = false;
            }
            if ( fabs( m_frameAnims[t]->m_fps - model->m_frameAnims[t]->m_fps ) > tolerance )
            {
               log_debug( "match failed at frame animation fps for frame anim %d\n", t );
               match = false;
            }
         }

         for ( t = 0; match && t < numSkelAnims; t++ )
         {
            if ( m_skelAnims[t]->m_frameCount != model->m_skelAnims[t]->m_frameCount )
            {
               log_debug( "match failed at skel animation frame count for skel anim %d\n", t );
               match = false;
            }
            if ( fabs( m_skelAnims[t]->m_fps - model->m_skelAnims[t]->m_fps ) > tolerance )
            {
               log_debug( "match failed at skel animation fps for skel anim %d\n", t );
               match = false;
            }
         }

         if ( match )
         {
            matchVal |= CompareAnimSets;
         }
      }
   }

   if ( compareMask & CompareMeta )
   {
      if ( numGroups == model->m_groups.size()
            && numTextures == model->m_materials.size()
            && numJoints == model->m_joints.size()
            && numSkelAnims == model->m_skelAnims.size()
            && numFrameAnims == model->m_frameAnims.size() )
      {
         bool match = true;

         for ( t = 0; match && t < numGroups; t++ )
         {
            if ( strcmp( m_groups[t]->m_name.c_str(), model->m_groups[t]->m_name.c_str() ) != 0 )
            {
               match = false;
            }
         }

         for ( t = 0; match && t < numTextures; t++ )
         {
            if ( strcmp( m_materials[t]->m_name.c_str(), model->m_materials[t]->m_name.c_str() ) != 0 )
            {
               match = false;
            }
            if ( strcmp( m_materials[t]->m_filename.c_str(), model->m_materials[t]->m_filename.c_str() ) != 0 )
            {
               match = false;
            }
            if ( strcmp( m_materials[t]->m_alphaFilename.c_str(), model->m_materials[t]->m_alphaFilename.c_str() ) != 0 )
            {
               match = false;
            }
         }

         for ( t = 0; match && t < numJoints; t++ )
         {
            if ( strcmp( m_joints[t]->m_name.c_str(), model->m_joints[t]->m_name.c_str() ) != 0 )
            {
               match = false;
            }
         }

         for ( t = 0; match && t < numSkelAnims; t++ )
         {
            if ( strcmp( m_skelAnims[t]->m_name.c_str(), model->m_skelAnims[t]->m_name.c_str() ) != 0 )
            {
               match = false;
            }
         }

         for ( t = 0; match && t < numFrameAnims; t++ )
         {
            if ( strcmp( m_frameAnims[t]->m_name.c_str(), model->m_frameAnims[t]->m_name.c_str() ) != 0 )
            {
               match = false;
            }
         }

         if ( match )
         {
            matchVal |= CompareMeta;
         }
      }
   }

   matchVal &= compareMask;
   return matchVal;
}

#ifdef MM3D_EDIT

bool Model::mergeAnimations( Model * model )
{
   if ( m_animationMode )
   {
      return false;
   }

   unsigned count = model->getAnimCount( ANIMMODE_SKELETAL );
   unsigned ac1 = getAnimCount( ANIMMODE_SKELETAL );

   if ( count == 0 )
   {
      msg_warning( "Model contains no skeletal animations" );
      return false;
   }

   unsigned j1 = getBoneJointCount();
   unsigned j2 = model->getBoneJointCount();


   if ( j1 != j2 )
   {
      msg_warning( "Model skeletons do not match" );
      return false;
   }

   for ( unsigned j = 0; j < j1; j++ )
   {
      if ( m_joints[ j ]->m_parent != model->m_joints[j]->m_parent )
      {
         msg_warning( "Model skeletons do not match" );
         return false;
      }
   }

   forceAddOrDelete( true );

   // Do skeletal add
   {
      for ( unsigned n = 0; n < count; n++ )
      {
         unsigned framecount = model->getAnimFrameCount( ANIMMODE_SKELETAL, n );

         unsigned index = addAnimation( ANIMMODE_SKELETAL, model->getAnimName( ANIMMODE_SKELETAL, n ) );
         setAnimFrameCount( ANIMMODE_SKELETAL, index, framecount );
         setAnimFPS( ANIMMODE_SKELETAL, n, model->getAnimFPS( ANIMMODE_SKELETAL, n ) );

         SkelAnim * sa = model->m_skelAnims[n];

         for ( unsigned j = 0; j < sa->m_jointKeyframes.size(); j++ )
         {
            for ( unsigned k = 0; k < sa->m_jointKeyframes[j].size(); k++ )
            {
               Keyframe * kf = sa->m_jointKeyframes[j][k];

               setSkelAnimKeyframe( ac1 + n, kf->m_frame, j, kf->m_isRotation,
                     kf->m_parameter[0], kf->m_parameter[1], kf->m_parameter[2] );
            }
         }
      }
   }

   invalidateNormals();

   forceAddOrDelete( true );

   return true;
}

bool Model::mergeModels( Model * model, bool textures, AnimationMerge animations, bool emptyGroups, double * trans, double * rot )
{
   if ( m_animationMode )
   {
      return false;
   }

   forceAddOrDelete( true );

   Matrix mat;

   if ( rot )
   {
      log_debug( "merge rotation: %f, %f, %f\n", rot[0], rot[1], rot[2] );
      mat.setRotation( rot );
   }

   if ( trans )
   {
      log_debug( "merge translation: %f, %f, %f\n", trans[0], trans[1], trans[2] );
      mat.setTranslation( trans );
   }

   unsigned vertbase   = 0;
   unsigned tribase    = 0;
   unsigned grpbase    = 0;
   unsigned jointbase  = 0;
   unsigned matbase    = 0;

   unsigned n = 0;
   unsigned count = 0;

   std::map<int,int> m_groupMap;

   vertbase   = m_vertices.size();
   tribase    = m_triangles.size();
   grpbase    = m_groups.size();
   jointbase  = m_joints.size();
   matbase    = m_materials.size();

   unselectAll();

   count = model->m_vertices.size();
   for ( n = 0; n < count; n++ )
   {
      Vertex * vert = model->m_vertices[n];
      Vector vec( vert->m_coord );
      vec = vec * mat;
      addVertex( vec.get(0), vec.get(1), vec.get(2) );
   }

   count = model->m_triangles.size();
   for ( n = 0; n < count; n++ )
   {
      Triangle * tri = model->m_triangles[n];

      addTriangle( tri->m_vertexIndices[0] + vertbase,
            tri->m_vertexIndices[1] + vertbase, tri->m_vertexIndices[2] + vertbase );
   }

   count = model->m_groups.size();
   for ( n = 0; n < count; n++ )
   {
      if ( emptyGroups || !model->getGroupTriangles(n).empty() )
      {
         const char * name = model->getGroupName( n );
         m_groupMap[n] = addGroup( name );
      }
   }

   for ( n = 0; n < count; n++ )
   {
      uint8_t val = model->getGroupSmooth( n );
      setGroupSmooth( m_groupMap[n], val );
   }

   count = model->m_joints.size();
   if ( count > 0 )
   {
      model->setupJoints();
      for ( n = 0; n < count; n++ )
      {
         Joint * joint = model->m_joints[n];
         double rot[3];
         double tran[3];

         Matrix jabs = joint->m_absolute * mat;

         jabs.getRotation( rot );
         jabs.getTranslation( tran );

         int parent = joint->m_parent;

         if ( parent >= 0 )
         {
            parent += jointbase;
         }

         addBoneJoint( joint->m_name.c_str(), tran[0], tran[1], tran[2],
               rot[0], rot[1], rot[2], parent );
      }

      for ( n = 0; n < count; n++ )
      {
         list<int> verts = model->getBoneJointVertices( n );
         list<int>::iterator it;
         for ( it = verts.begin(); it != verts.end(); it++ )
         {
            setVertexBoneJoint( (*it) + vertbase, n + jointbase );
         }
      }
   }

   if ( textures )
   {
      TextureManager * texmgr = TextureManager::getInstance();

      count = model->getTextureCount();
      for ( n = 0; n < count; n++ )
      {
         const char * name = model->getTextureFilename( n );
         Texture * newtex = texmgr->getTexture( name );

         addTexture( newtex );
      }

      for ( n = 0; n < count; n++ )
      {
         float val[4] = { 0.0, 0.0, 0.0, 0.0 };
         float shin = 0.0;

         model->getTextureAmbient(  n, val );
         setTextureAmbient(  n + matbase, val );
         model->getTextureDiffuse(  n, val );
         setTextureDiffuse(  n + matbase, val );
         model->getTextureEmissive( n, val );
         setTextureEmissive( n + matbase, val );
         model->getTextureSpecular( n, val );
         setTextureSpecular( n + matbase, val );

         model->getTextureShininess( n, shin );
         setTextureShininess( n + matbase, shin );
      }

      count = model->m_groups.size();
      for ( n = 0; n < count; n++ )
      {
         int val = model->getGroupTextureId( n );
         setGroupTextureId( m_groupMap[n], val + matbase );
      }

      count = model->getTriangleCount();
      float s = 0.0;
      float t = 0.0;
      for ( n = 0; n < count; n++ )
      {
         for ( unsigned i = 0; i < 3; i++ )
         {
            model->getTextureCoords( n,    i, s, t );
            setTextureCoords( n + tribase, i, s, t );
         }
      }

      for ( n = 0; n < count; n++ )
      {
         int val = model->getTriangleGroup( n );
         if ( val >= 0 )
         {
            addTriangleToGroup( m_groupMap[val], n + tribase );
         }
      }
   }

   bool frameAnimsNeeded = (getAnimCount( ANIMMODE_FRAME ) > 0 );

   if ( frameAnimsNeeded )
   {
      setFrameAnimVertexCount( m_vertices.size() );
   }

   unsigned oldcount = getAnimCount( ANIMMODE_FRAME );
   if ( animations != AM_NONE )
   {
      // Do frame merge if possible
      unsigned ac1 = getAnimCount( ANIMMODE_FRAME );
      unsigned ac2 = model->getAnimCount( ANIMMODE_FRAME );

      bool match = false;

      if ( animations == AM_MERGE && ac1 == ac2 )
      {
         match = true; // Have to check frame counts too

         unsigned a = 0;

         for ( a = 0; match && a < ac1; a++ )
         {
            unsigned fc1 = getAnimFrameCount( ANIMMODE_FRAME, a );
            unsigned fc2 = model->getAnimFrameCount( ANIMMODE_FRAME, a );

            if ( fc1 != fc2 )
            {
               match = false;
            }
         }

         if ( match )
         {
            for ( a = 0; a < ac1; a++ )
            {
               unsigned fc1 = getAnimFrameCount( ANIMMODE_FRAME, a );
               unsigned vertcount = model->m_vertices.size();

               for ( unsigned f = 0; f < fc1; f++ )
               {
                  for ( unsigned v = 0; v < vertcount; v++ )
                  {
                     double coord[3] = { 0, 0, 0 };
                     model->getFrameAnimVertexCoords( a, f, v, coord[0], coord[1], coord[2] );
                     Vector vec( coord );
                     vec = vec * mat;
                     setFrameAnimVertexCoords( a, f, v + vertbase, vec.get(0), vec.get(1), vec.get(2) );
                  }
               }
            }

            frameAnimsNeeded = false;
         }
      }

      // Do frame add otherwise
      if ( !match || animations == AM_ADD )
      {
         count = model->getAnimCount( ANIMMODE_FRAME );
         for ( n = 0; n < count; n++ )
         {
            unsigned framecount = model->getAnimFrameCount( ANIMMODE_FRAME, n );

            unsigned index = addAnimation( ANIMMODE_FRAME, model->getAnimName( ANIMMODE_FRAME, n ) );
            setAnimFrameCount( ANIMMODE_FRAME, index, framecount );

            unsigned vertcount = model->m_vertices.size();

            for ( unsigned f = 0; f < framecount; f++ )
            {
               for ( unsigned v = 0; v < vertcount; v++ )
               {
                  double coord[3] = { 0, 0, 0 };
                  model->getFrameAnimVertexCoords( n, f, v, coord[0], coord[1], coord[2] );
                  Vector vec( coord );
                  vec = vec * mat;
                  setFrameAnimVertexCoords( index, f, v + vertbase, vec.get(0), vec.get(1), vec.get(2) );
               }
            }
         }
      }

      // Do skeletal merge if possible
      ac1 = getAnimCount( ANIMMODE_SKELETAL );
      ac2 = model->getAnimCount( ANIMMODE_SKELETAL );

      match = false;
      if ( ac1 == ac2 && animations == AM_MERGE )
      {
         match = true; // Still need to check frame count

         unsigned a = 0;

         for ( a = 0; match && a < ac1; a++ )
         {
            unsigned fc1 = getAnimFrameCount( ANIMMODE_SKELETAL, a );
            unsigned fc2 = model->getAnimFrameCount( ANIMMODE_SKELETAL, a );

            if ( fc1 != fc2 )
            {
               match = false;
            }
         }

         if ( match )
         {
            for ( a = 0; a < ac1; a++ )
            {
               SkelAnim * sa = model->m_skelAnims[a];

               for ( unsigned j = 0; j < sa->m_jointKeyframes.size(); j++ )
               {
                  for ( unsigned k = 0; k < sa->m_jointKeyframes[j].size(); k++ )
                  {
                     Keyframe * kf = sa->m_jointKeyframes[j][k];

                     setSkelAnimKeyframe( a, kf->m_frame, j + jointbase, kf->m_isRotation,
                           kf->m_parameter[0], kf->m_parameter[1], kf->m_parameter[2] );
                  }
               }
            }
         }
      }

      // Do skeletal add otherwise
      if ( !match || animations == AM_ADD )
      {
         count = model->getAnimCount( ANIMMODE_SKELETAL );
         for ( n = 0; n < count; n++ )
         {
            unsigned framecount = model->getAnimFrameCount( ANIMMODE_SKELETAL, n );

            unsigned index = addAnimation( ANIMMODE_SKELETAL, model->getAnimName( ANIMMODE_SKELETAL, n ) );
            setAnimFrameCount( ANIMMODE_SKELETAL, index, framecount );
            setAnimFPS( ANIMMODE_SKELETAL, n, model->getAnimFPS( ANIMMODE_SKELETAL, n ) );

            SkelAnim * sa = model->m_skelAnims[n];

            for ( unsigned j = 0; j < sa->m_jointKeyframes.size(); j++ )
            {
               for ( unsigned k = 0; k < sa->m_jointKeyframes[j].size(); k++ )
               {
                  Keyframe * kf = sa->m_jointKeyframes[j][k];

                  setSkelAnimKeyframe( ac1 + n, kf->m_frame, j + jointbase, kf->m_isRotation,
                        kf->m_parameter[0], kf->m_parameter[1], kf->m_parameter[2] );
               }
            }
         }
      }
   }

   if ( frameAnimsNeeded )
   {
      // We have frame anims that don't have our new vertices.  
      // Must add them

      count = oldcount; // Only need to adjust original frame anims
      for ( n = 0; n < count; n++ )
      {
         unsigned framecount = getAnimFrameCount( ANIMMODE_FRAME, n );
         unsigned vertcount = model->m_vertices.size();

         for ( unsigned v = 0; v < vertcount; v++ )
         {
            double coord[3] = { 0, 0, 0 };
            model->getVertexCoords( v, coord );

            Vector vec( coord );
            vec = vec * mat;

            for ( unsigned f = 0; f < framecount; f++ )
            {
               setFrameAnimVertexCoords( n, f, v + vertbase, vec.get(0), vec.get(1), vec.get(2) );
            }
         }
      }
   }

   invalidateNormals();

   forceAddOrDelete( true );

   return true;
}

#endif // MM3D_EDIT
