#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <mx/mx.h>
#include "pakviewer.h"
#include "mdx.h"
#include "GlWindow.h"
#include "mdxviewer.h"


int
pak_ExtractFile (const char *pakFile, const char *lumpName, char *outFile)
{
	FILE *file = fopen (pakFile, "rb");
	if (!file)
		return 0;

	int ident, dirofs, dirlen;

	fread (&ident, sizeof (int), 1, file);
	if (ident != (int) (('K' << 24) + ('C' << 16) + ('A' << 8) + 'P'))
	{
		fclose (file);
		return 0;
	}

	fread (&dirofs, sizeof (int), 1, file);
	fread (&dirlen, sizeof (int), 1, file);

	fseek (file, dirofs, SEEK_SET);
	int numLumps = dirlen / 64;

	for (int i = 0; i < numLumps; i++)
	{
		char name[56];
		int filepos, filelen;

		fread (name, 56, 1, file);
		fread (&filepos, sizeof (int), 1, file);
		fread (&filelen, sizeof (int), 1, file);

		if (!strcmp (name, lumpName))
		{
			FILE *out = fopen (outFile, "wb");
			if (!out)
			{
				fclose (file);
				return 0;
			}

			fseek (file, filepos, SEEK_SET);

			while (filelen--)
				fputc (fgetc (file), out);

			fclose (out);
			fclose (file);

			return 1;
		}
	}

	fclose (file);

	return 0;
}



PAKViewer::PAKViewer (mxWindow *window)
: mxWindow (window, 0, 0, 0, 0)
{
	strcpy (d_pakFile, "");
	strcpy (d_currLumpName, "");
	tvPAK = new mxTreeView (this, 0, 0, 0, 0, IDC_PAKVIEWER);
	pmMenu = new mxPopupMenu ();
	pmMenu->add ("Load Model", 1);
	pmMenu->add ("Merge Model", 2);
	pmMenu->addSeparator ();
	pmMenu->add ("Load Background", 3);
	pmMenu->add ("Load Water", 4);
	pmMenu->addSeparator ();
	pmMenu->add ("Extract...", 5);

	setLoadEntirePAK (false);

	tab = new mxTab (this, 0, 0, 0, 0);
	tab->add (tvPAK, "PAK");
}



PAKViewer::~PAKViewer ()
{
	closePAKFile ();
}



void
_makeTempFileName (char *str, const char *prefix)
{
	char path2[256];

	GetTempPath (256, path2);

	strcpy (str, path2);
	strcat (str, "/mdxtemp");
	strcat (str, prefix);
}



int
PAKViewer::handleEvent (mxEvent *event)
{
	switch (event->event)
	{
	case mxEvent::Action:
	{
		switch (event->action)
		{
		case IDC_PAKVIEWER: // tvPAK
			if (event->flags & mxEvent::RightClicked)
			{
				OnPAKViewer ();
				bool isMdx = strstr (d_currLumpName, ".mdx") != 0;
				bool isPcx = strstr (d_currLumpName, ".pcx") != 0;
				bool isTga = strstr (d_currLumpName, ".tga") != 0;
				pmMenu->setEnabled (1, isMdx || isPcx);
				pmMenu->setEnabled (2, isMdx || isPcx);
				pmMenu->setEnabled (3, isPcx || isTga);
				pmMenu->setEnabled (4, isPcx || isTga);

				int ret = pmMenu->popup (tvPAK, event->x, event->y);
				switch (ret)
				{
				case 1:
				case 2:
					if (isMdx)
						OnLoadModel (ret - 1);
					else if (isTga)
						OnLoadTexture (ret == 1 ? TEXTURE_BACKGROUND:TEXTURE_WATER);
					break;

				case 3:
				case 4:
					if (isTga)
						OnLoadTexture (ret ==  3 ? TEXTURE_BACKGROUND:TEXTURE_WATER);
					break;

				case 5:
					OnExtract ();
				}
			}

			else if (event->flags & mxEvent::DoubleClicked)
			{
				OnPAKViewer ();
				char e[16];

				_splitpath (d_currLumpName, 0, 0, 0, e);
				
					if (!_stricmp (e, ".mdx"))
						OnLoadModel (0);

					else if (!_stricmp (e, ".tga"))
						OnLoadTexture (TEXTURE_BACKGROUND);

					else if (!_stricmp (e, ".wav"))
						OnPlaySound ();

					return 1;
			}

			return 1;
		} // event->action
	} // mxEvent::Action
	break;

	case mxEvent::Size:
	{
		tab->setBounds (0, 0, event->width, event->height);
	} // mxEvent::Size
	break;

	} // event->event

	return 1;
}



int
PAKViewer::OnPAKViewer ()
{
	mxTreeViewItem *tvi = tvPAK->getSelectedItem ();
	if (tvi)
	{
		strcpy (d_currLumpName, tvPAK->getLabel (tvi));

		// find the full lump name
		mxTreeViewItem *tviParent = tvPAK->getParent (tvi);
		char tmp[128];
		while (tviParent)
		{
			strcpy (tmp, d_currLumpName);
			strcpy (d_currLumpName, tvPAK->getLabel (tviParent));
			strcat (d_currLumpName, "/");
			strcat (d_currLumpName, tmp);
			tviParent = tvPAK->getParent (tviParent);
		}

		if (!d_loadEntirePAK)
		{
			// finally insert "models/"
			strcpy (tmp, d_currLumpName);
			strcpy (d_currLumpName, "models/");
			strcat (d_currLumpName, tmp);
		}
	}

	return 1;
}



int
PAKViewer::OnLoadModel (int pos)
{
	static char str2[256];
	static char ext[256];

	_splitpath (d_currLumpName, 0, 0, 0, ext);
	bool isMDX = (!_stricmp (ext, ".mdx"));
	char prefix[16];

	if (isMDX)
		sprintf (prefix, "%s%d", "mdx.mdx", pos);
	else
		sprintf (prefix, "%s%d", "mdx.tga", pos);

	_makeTempFileName (str2, prefix);

	// extract the file from the pak file and give it a temp name
	if (!pak_ExtractFile (d_pakFile, d_currLumpName, str2))
	{
		mxMessageBox (this, "Error extracting from PAK file.", "MDX Viewer", MX_MB_OK | MX_MB_ERROR);
		return 1;
	}

	// now load the things
	if (isMDX)
	{
		mdx_model_t *model = g_mdxViewer->glw->loadModel (str2, pos);
		if (model)
		{
			if (pos == 0)
				g_mdxViewer->initAnimation (model, -1);
			g_mdxViewer->setModelInfo (model, pos);

			// try to load skin
			if (model->header.numSkins > 0)
			{
				sprintf (prefix, "%s%d", "mdx.tga", pos);
				_makeTempFileName (str2, prefix);
				if (pak_ExtractFile (d_pakFile, model->skins[0], str2))
				{
					if (g_mdxViewer->glw->loadTexture (str2, pos == 0 ? TEXTURE_MODEL:TEXTURE_WEAPON))
					{
						g_mdxViewer->setRenderMode (3);
					}
				}
			}

			if (pos == 0)
				g_mdxViewer->centerModel ();

			g_mdxViewer->glw->redraw ();
		}
		else
			mxMessageBox (this, "Error loading model.", "MDX Viewer", MX_MB_OK | MX_MB_ERROR);
	}
	else // must be skin file
	{
		if (g_mdxViewer->glw->loadTexture (str2, pos == 0 ? TEXTURE_MODEL:TEXTURE_WEAPON))
		{
			g_mdxViewer->setRenderMode (3);
			g_mdxViewer->glw->redraw ();
		}
		else
			mxMessageBox (this, "Error loading skin.", "MDX Viewer", MX_MB_OK | MX_MB_ERROR);
	}

	return 1;
}



int
PAKViewer::OnLoadTexture (int pos)
{
	char str2[128];
	char prefix[16];
	bool isTarga = (strstr (d_currLumpName, ".tga") != 0);

	sprintf (prefix, "%d%s", pos, isTarga ? ".tga":".tga");
	_makeTempFileName (str2, prefix);

	// extract the file from the pak file and give it a temp name
	if (!pak_ExtractFile (d_pakFile, d_currLumpName, str2))
	{
		mxMessageBox (this, "Error extracting from PAK file.", "MDX Viewer", MX_MB_OK | MX_MB_ERROR);
		return 1;
	}

	// now load the things
	if (g_mdxViewer->glw->loadTexture (str2, pos))
	{
		if (pos == TEXTURE_BACKGROUND)
		{
			g_mdxViewer->cbBackground->setChecked (true);
			g_mdxViewer->glw->setFlag (F_BACKGROUND, true);
		}
		else if (pos == TEXTURE_WATER)
		{
			g_mdxViewer->cbWater->setChecked (true);
			g_mdxViewer->glw->setFlag (F_WATER, true);
		}
		g_mdxViewer->setRenderMode (3);
		g_mdxViewer->glw->redraw ();
	}
	else
		mxMessageBox (this, "Error loading skin.", "MDX Viewer", MX_MB_OK | MX_MB_ERROR);

	return 1;
}



int
PAKViewer::OnPlaySound ()
{
#ifdef WIN32
	static char str2[256];
	char suffix[16] = "";

	// stop any playing sound
	PlaySound (0, 0, SND_FILENAME | SND_ASYNC);

	sprintf (suffix, "%d%s", 44, ".wav");

	_makeTempFileName (str2, suffix);

	if (!pak_ExtractFile (d_pakFile, d_currLumpName, str2))
	{
		mxMessageBox (this, "Error extracting from PAK file.", "MDX Viewer", MX_MB_OK | MX_MB_ERROR);
		return 1;
	}

	PlaySound (str2, 0, SND_FILENAME | SND_ASYNC);

#endif
	return 1;
}



int
PAKViewer::OnExtract ()
{
	char *ptr = (char *) mxGetSaveFileName (this, "", "*.*");
	if (ptr)
	{
		if (!pak_ExtractFile (d_pakFile, d_currLumpName, ptr))
			mxMessageBox (this, "Error extracting from PAK file.", "MDX Viewer", MX_MB_OK | MX_MB_ERROR);
	}

	return 1;
}



int
_compare(const void *arg1, const void *arg2)
{
	if (strchr ((char *) arg1, '/') && !strchr ((char *) arg2, '/'))
		return -1;

	else if (!strchr ((char *) arg1, '/') && strchr ((char *) arg2, '/'))
		return 1;

	else
		return strcmp ((char *) arg1, (char *) arg2);
}



bool
PAKViewer::openPAKFile (const char *pakFile)
{
	FILE *file = fopen (pakFile, "rb");
	if (!file)
		return false;

	int ident, dirofs, dirlen;

	// check for id
	fread (&ident, sizeof (int), 1, file);
	if (ident != (int) (('K' << 24) + ('C' << 16) + ('A' << 8) + 'P'))
	{
		fclose (file);
		return false;
	}

	// load lumps
	fread (&dirofs, sizeof (int), 1, file);
	fread (&dirlen, sizeof (int), 1, file);
	int numLumps = dirlen / 64;

	fseek (file, dirofs, SEEK_SET);
	lump_t *lumps = new lump_t[numLumps];
	if (!lumps)
	{
		fclose (file);
		return false;
	}

	fread (lumps, sizeof (lump_t), numLumps, file);
	fclose (file);

	qsort (lumps, numLumps, sizeof (lump_t), _compare);

	// save pakFile for later
	strcpy (d_pakFile, pakFile);

	tvPAK->remove (0);

	char namestack[32][32];
	mxTreeViewItem *tvistack[32];
	for (int k = 0; k < 32; k++)
	{
		strcpy (namestack[k], "");
		tvistack[k] = 0;
	}

	for (int i = 0; i < numLumps; i++)
	{
		if (d_loadEntirePAK || !strncmp (lumps[i].name, "models", 6))
		{
			char *tok;
			if (d_loadEntirePAK)
				tok = &lumps[i].name[0];
			else
				tok = &lumps[i].name[7];

			int i = 1;
			while (tok)
			{
				char *end = strchr (tok, '/');
				if (end)
					*end = '\0';

				if (strcmp (namestack[i], tok))
				{
					strcpy (namestack[i], tok);

					tvistack[i] = tvPAK->add (tvistack[i - 1], tok);

					for (int j = i + 1; j < 32; j++)
					{
						strcpy (namestack[j], "");
						tvistack[j] = 0;
					}
				}

				++i;

				if (end)
					tok = end + 1;
				else
					tok = 0;
			}
		}
	}

	delete[] lumps;

	return true;
}



void
PAKViewer::closePAKFile ()
{
	strcpy (d_pakFile, "");
	tvPAK->remove (0);
}
