/*****************************************************************
Copyright (c) 1998 Jawed Karim <jkarim@students.uiuc.edu>
All rights reserved.

This source code has been provided to you under the condition
that you adhere to the following terms:

1. YOU MAY USE THIS SOURCE CODE PRIVATELY WITHOUT RESTRICTIONS.

2. REDISTRIBUTIONS OF MODIFICATIONS OF THIS PROGRAM IN BINARY OR
   IN SOURCE CODE FORM ARE NOT PERMITTED.

3. REDISTRIBUTIONS OF THIS SOURCE CODE ARE ONLY PERMITTED IF
   THE SOURCE CODE REMAINS COMPLETELY UNCHANGED AND ALL THE
   FILES WHICH WERE IN THE ORIGINAL DISTRIBUTION ARE INCLUDED.

4. ALL SOFTWARE USING SECTIONS OF THIS SOURCE CODE MUST GIVE
   EXPLICIT ACKNOWLEDGMENT TO JAWED KARIM IN THE PROGRAM
   ITSELF AS WELL AS IN THE DOCUMENTATION.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
INTERRUPTION).
*****************************************************************/

#include <stdio.h>
#include <iostream.h>

#include <windows.h>
#include <commctrl.h>
#include <windowsx.h>
#include <wininet.h>

#include <GL/gl.h>
#include <GL/glu.h>

#include "resource.h"
#include "fileio.h"
#include "inter.h"
#include "pcx.h"
#include "slidebar.h"
#include "md2.h"
#include "wingl.h"

const int iRenderWindowX = 400;
const int iRenderWindowY = 400;

CSlidebar frameSlider;
CMD2 md2Model;
COpenGLWindow glWindow (iRenderWindowX, iRenderWindowY);
CImage imageTexture;

int WINAPI WinMain (HINSTANCE hInstance, HINSTANCE hPrevInstance, PSTR szCmdLine, int iCmdShow)
{
	static char szAppName[] = "Quake2 Model Viewer" ;
	MSG         msg ;
	WNDCLASSEX  wndclass ;

	wndclass.cbSize        = sizeof (wndclass) ;
	wndclass.style         = CS_HREDRAW | CS_VREDRAW ;
	wndclass.lpfnWndProc   = WndProc ;
	wndclass.cbClsExtra    = 0 ;
	wndclass.cbWndExtra    = 0 ;
	wndclass.hInstance	   = hInstanceGlobal = hInstance ;
	wndclass.hIcon         = LoadIcon (NULL, IDI_APPLICATION) ;
	hIconGlobal = LoadIcon(hInstanceGlobal, MAKEINTRESOURCE(IDI_ICON1));
	wndclass.hCursor       = LoadCursor (NULL, IDC_ARROW) ;
	wndclass.hbrBackground = (HBRUSH) GetStockObject (WHITE_BRUSH) ;
	wndclass.lpszMenuName  = NULL ;
	wndclass.lpszClassName = szAppName ;
	wndclass.hIconSm       = LoadIcon (NULL, IDI_APPLICATION) ;

	RegisterClassEx (&wndclass) ;
	LoadLibrary("RICHED32.DLL");
	InitCommonControls();

	RegisterClassEx (&wndclass) ;

	wndclass.lpfnWndProc = GraphicsProc;
	wndclass.cbWndExtra = 0;
	wndclass.hIcon		= NULL;
	wndclass.lpszClassName = "Graphics";
	wndclass.hIconSm = NULL;
	wndclass.hCursor = LoadCursor (NULL, IDC_CROSS) ;

	RegisterClassEx (&wndclass) ;

	hDialog = CreateDialog (hInstance, MAKEINTRESOURCE (IDD_MAINDIALOG), NULL, (DLGPROC)DlgProc);

	frameSlider.Initialize (GetDlgItem (hDialog, IDC_SLIDER));

	SendMessage (GetDlgItem (hDialog, IDC_TEXTURE), BM_SETCHECK, (WPARAM)BST_UNCHECKED, 0);
	SendMessage (GetDlgItem (hDialog, IDC_WIREFRAME), BM_SETCHECK, (WPARAM)BST_CHECKED, 0);

	ShowWindow (hDialog, iCmdShow) ;
	UpdateWindow (hDialog) ;

	while (GetMessage (&msg, NULL, 0, 0))
	{
		TranslateMessage (&msg) ;
		DispatchMessage (&msg) ;
	}
	return msg.wParam ;
}

LRESULT CALLBACK WndProc (HWND hwnd, UINT iMsg, WPARAM wParam, LPARAM lParam)
{
	switch (iMsg)
    {
		case WM_DESTROY :
			md2Model.Destroy ();
			imageTexture.Destroy ();
			PostQuitMessage (0) ;
			return 0 ;

		case WM_QUERYDRAGICON:
			return (LRESULT) hIconGlobal;
	}

	return DefWindowProc (hwnd, iMsg, wParam, lParam) ;
}

BOOL CALLBACK DlgProc (HWND hDlg, UINT iMsg, WPARAM wParam, LPARAM lParam)
{
	NM_UPDOWN *pUpDown;
	char szLongFilename[255];
	char szShortFilename[255];
	int iState;

	switch (iMsg) {

		case WM_NOTIFY:

			pUpDown	= (NM_UPDOWN FAR *) lParam;

			switch (LOWORD (wParam)) {

				case IDC_SPIN:

					frameSlider.setFocus();

					if (pUpDown->iDelta > 0)
						frameSlider.setPosition(frameSlider.getPosition() - 1);

					if (pUpDown->iDelta < 0)
						frameSlider.setPosition(frameSlider.getPosition() + 1);

					redraw (frameSlider.getPosition());

					break;
			}
			return 0;

		case WM_HSCROLL:

			switch (LOWORD (wParam)) {

				case TB_ENDTRACK:

					redraw (frameSlider.getPosition());

					break;

				default:
					break;
			} 

		case WM_INITDIALOG:
			SendMessage (hDlg, WM_SETICON, TRUE, (LPARAM)hIconGlobal);
			SendMessage (hDlg, WM_SETICON, FALSE, (LPARAM)hIconGlobal);

			return TRUE;

		case WM_PAINT:
			return 0;

		case WM_COMMAND:

			if (HIWORD (wParam) == BN_CLICKED)
			{
				iState = SendMessage (GetDlgItem (hDlg, IDC_WIREFRAME), BM_GETCHECK, 0, 0);
				if (iState == BST_CHECKED)
					render_mode = WIREFRAME;

				iState = SendMessage (GetDlgItem (hDlg, IDC_TEXTURE), BM_GETCHECK, 0, 0);
				if (iState == BST_CHECKED)
					render_mode = TEXTURE;

				if (((HWND)lParam) == (GetDlgItem (hDlg, IDC_WIREFRAME)) ||
					((HWND)lParam) == (GetDlgItem (hDlg, IDC_TEXTURE)))
					ChangeRenderMode ();
			}

			switch (LOWORD (wParam))
			{
				case IDM_MENU_ITEM_HELP:
					DialogBox (hInstanceGlobal, MAKEINTRESOURCE (IDD_ABOUT), hDlg, (DLGPROC)HelpDlgProc);
					return 0;

				case IDM_MENU_ITEM_EXIT: case WM_DESTROY:
					md2Model.Destroy();
					imageTexture.Destroy ();

					PostQuitMessage (0);
					return TRUE;

				case IDM_MENU_ITEM_RELOAD_PCX:

					imageTexture.Destroy ();

					if ((imageTexture.Read (imageTexture.m_szCurrentFile)) == 0)
					{
						imageTexture.Image2GLTexture ();

						redraw (frameSlider.getPosition());
						init ();
					}
					else
						MessageBox (hDlg, "There was a problem loading the PCX file.", "File Error", MB_OK | MB_ICONEXCLAMATION);

					return 0;

				case IDM_MENU_ITEM_OPEN_PCX:
					szLongFilename[0] = '\0';
					szShortFilename[0] = '\0';

					if (LoadPCXFileOpenDlg (hDialog, szLongFilename, szShortFilename) == TRUE) {

						imageTexture.Destroy();

						if ((imageTexture.Read (szLongFilename)) == 0)
						{
							imageTexture.Image2GLTexture ();

							SendMessage (GetDlgItem (hDlg, IDC_WIREFRAME), BM_SETCHECK, (WPARAM)BST_UNCHECKED, 0);

							SendMessage (GetDlgItem (hDlg, IDC_TEXTURE), BM_SETCHECK, (WPARAM)BST_CHECKED, 0);
							render_mode = TEXTURE;

							init ();

							EnableMenuItem (GetMenu (hDialog), IDM_MENU_ITEM_RELOAD_PCX, MF_ENABLED);
						}
						else
							MessageBox (hDlg, "There was a problem loading the PCX file.", "File Error", MB_OK | MB_ICONEXCLAMATION);
					}

					return 0;


				case IDM_MENU_ITEM_OPEN:
					szLongFilename[0] = '\0';
					szShortFilename[0] = '\0';

					if (LoadMD2FileOpenDlg (hDialog, szLongFilename, szShortFilename) == TRUE) {

						md2Model.Destroy();
						imageTexture.Destroy ();

						if (hGraphics != NULL)
							DestroyWindow (hGraphics);

						if (md2Model.Read (szLongFilename)) {

							hGraphics = CreateWindowEx (WS_EX_OVERLAPPEDWINDOW, "Graphics", NULL,
								WS_OVERLAPPEDWINDOW | WS_VISIBLE | WS_CAPTION,
								390, 30, iRenderWindowX, iRenderWindowY,
								hDlg, NULL, (HINSTANCE) GetWindowLong (hDlg, GWL_HINSTANCE), NULL) ;

							frameSlider.setRange (0, md2Model.getnumberFrames() - 1);
							frameSlider.setPosition (0);

							rot[0] = rot[1] = trans[0] = trans[1] = trans[2] = 0.0;

							init();

							EnableControls (1);

						}
						else
							MessageBox (hDlg, "There was a problem loading the MD2 file.", "File Error", MB_OK | MB_ICONEXCLAMATION);
					}

					return 0;
			}
			break;

		case WM_QUERYDRAGICON:
			return (LRESULT) hIconGlobal;
			return 0;
	}
	return FALSE ;
}

void init (void)
{
	glClearColor (0.0, 0.0, 0.0, 0.0);

	glMatrixMode(GL_PROJECTION);
	glLoadIdentity ();

	gluPerspective (40, 1, 40, 500);

	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity ();

	glTranslatef(0.0F, 0.0F, -70.0F);

	glRotatef(-90.0F, 1.0F, 0.0F, 0.0F);
	glRotatef(90.0F, 0.0F, 0.0F, 1.0F);

	glEnable(GL_DEPTH_TEST);
	glEnable(GL_LIGHT0);

	glCullFace(GL_BACK);

	glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);


	glTexImage2D(GL_TEXTURE_2D, 0, 4, imageTexture.m_iscaledWidth, imageTexture.m_iscaledHeight, 0, GL_RGBA, GL_UNSIGNED_BYTE, imageTexture.glTexture);

	ChangeRenderMode ();
}

void redraw (int frame)
{
	int i;
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	glPushMatrix();

	glTranslatef(-trans[2], -trans[0], trans[1]);
	glTranslatef(0, -trans[0], trans[1]);
	glRotatef(-rot[0], 0.0f, 1.0f, 0.0f);
	glRotatef(rot[1], 0.0f, 0.0f, 1.0f);

	for (i = 0; i < md2Model.getnumberTriangles(); i++) {
		glBegin(GL_TRIANGLES);

		glTexCoord2f((md2Model.m_index_list[i].a_s)/(float)imageTexture.m_iWidth, (md2Model.m_index_list[i].a_t)/(float)imageTexture.m_iHeight);
		glVertex3f (md2Model.m_frame_list[frame].vertex[md2Model.m_index_list[i].a].x,
					md2Model.m_frame_list[frame].vertex[md2Model.m_index_list[i].a].y,
					md2Model.m_frame_list[frame].vertex[md2Model.m_index_list[i].a].z);

		glTexCoord2f((md2Model.m_index_list[i].b_s)/(float)imageTexture.m_iWidth, (md2Model.m_index_list[i].b_t)/(float)imageTexture.m_iHeight);
		glVertex3f (md2Model.m_frame_list[frame].vertex[md2Model.m_index_list[i].b].x,
					md2Model.m_frame_list[frame].vertex[md2Model.m_index_list[i].b].y,
					md2Model.m_frame_list[frame].vertex[md2Model.m_index_list[i].b].z);

		glTexCoord2f((md2Model.m_index_list[i].c_s)/(float)imageTexture.m_iWidth, (md2Model.m_index_list[i].c_t)/(float)imageTexture.m_iHeight);
		glVertex3f (md2Model.m_frame_list[frame].vertex[md2Model.m_index_list[i].c].x,
					md2Model.m_frame_list[frame].vertex[md2Model.m_index_list[i].c].y,
					md2Model.m_frame_list[frame].vertex[md2Model.m_index_list[i].c].z);

		glEnd();
	}

	glPopMatrix();
	glFlush();
    SwapBuffers(glWindow.GetDC());
}

void ChangeRenderMode (void)
{
	if (render_mode == WIREFRAME) {
		glDisable(GL_CULL_FACE);
		glDisable(GL_TEXTURE_2D);
		glDisable (GL_LIGHTING);

		glColor3f (0.0, 1.0, 0.0);
		glPolygonMode (GL_FRONT_AND_BACK, GL_LINE);
	}
	else if (render_mode == TEXTURE) {
		glEnable(GL_CULL_FACE);
		glEnable(GL_TEXTURE_2D);
		glDisable (GL_LIGHTING);

		glColor3f (0.0F, 0.0F, 0.0F);
		glPolygonMode (GL_FRONT, GL_FILL);
		glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL);
	}

	redraw (frameSlider.getPosition());
}

void resize (int x, int y)
{

	glViewport(0, 0, x, y);
}

static void update (int state, int ox, int nx, int oy, int ny)
{
	int dx = ox - nx;
	int dy = ny - oy;

	switch(state) {
		case PAN:
			trans[0] -= dx / 5.0f;
			trans[1] -= dy / 5.0f;
			break;

		case ROTATE:
			rot[0] += (dy * 180.0f) / 500.0f;
			rot[1] -= (dx * 180.0f) / 500.0f;
			#define clamp(x) x = x > 360.0f ? x-360.0f : x < -360.0f ? x+=360.0f : x
			clamp(rot[0]);
			clamp(rot[1]);
			break;
	
		case ZOOM:
			trans[2] -= (dx+dy) / 4.0f;
			break;
	}
}

LRESULT APIENTRY GraphicsProc (HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	static int omx, omy, mx, my;
	static GLuint    state   = 0;

	int x, y;

    switch (message) {

		case WM_LBUTTONDOWN:
		case WM_RBUTTONDOWN:
			SetCapture(hGraphics);
			mx = LOWORD(lParam);
			my = HIWORD(lParam);

			if (message == WM_LBUTTONDOWN)
				state = PAN;

			if (message == WM_RBUTTONDOWN)
				state = ROTATE;

			if ((wParam & MK_RBUTTON) && (wParam & MK_LBUTTON))
				state = ZOOM;

			break;

		case WM_LBUTTONUP:
		case WM_RBUTTONUP:
			ReleaseCapture();
			state = 0;
			break;

		case WM_MOUSEMOVE:
			if (state) {
				omx = mx;
				omy = my;
				mx = LOWORD(lParam);
				my = HIWORD(lParam);
				if(mx & 1 << 15) mx -= (1 << 16);
				if(my & 1 << 15) my -= (1 << 16);
				update (state, omx, mx, omy, my);
				PostMessage(hGraphics, WM_PAINT, 0, 0);
			}
			break;

		case WM_CREATE:
			glWindow.Create (hWnd);

			break;

		case WM_DESTROY:
			glWindow.Destroy (hWnd);

			EnableControls (0);
			break;

		case WM_SIZE:
			x = LOWORD (lParam);
			y = HIWORD (lParam);

			if (glWindow.GLRCValid()) {

				glWindow.SetSize (x, y);

				resize(x, y);
				return 0;
			}

			break;

		case WM_PALETTECHANGED:
			if (glWindow.GLRCValid() && glWindow.PaletteValid() && (HWND) wParam != hWnd) {

				glWindow.RedoPalette ();
				redraw(frameSlider.getPosition());

				break;
			}
			break;

		case WM_QUERYNEWPALETTE:

			if (glWindow.GLRCValid() && glWindow.PaletteValid()) {

				glWindow.RedoPalette ();
				redraw(frameSlider.getPosition());

				return TRUE;
			}
			break;

		case WM_PAINT:
        {
            PAINTSTRUCT ps;
            BeginPaint(hWnd, &ps);

            if (glWindow.GLRCValid()) {
                redraw(frameSlider.getPosition());
            }

            EndPaint(hWnd, &ps);
            return 0;
        }
        break;

		case WM_KEYDOWN :

			switch (wParam)
			{
				case VK_HOME :
					rot[0] = rot[1] = trans[0] = trans[1] = trans[2] = 0.0;
					break;

				case VK_RIGHT:
					update (ROTATE, 0, 10, 0, 0);
					break;

				case VK_LEFT:
					update (ROTATE, 0, -10, 0, 0);
					break;

				case VK_UP:
					update (ROTATE, 0, 0, 0, -10);
					break;

				case VK_DOWN:
					update (ROTATE, 0, 0, 0, 10);
					break;

				case VK_ADD:
					trans[2] += 10;
					break;

				case VK_SUBTRACT:
					trans[2] -= 10;
					break;
			}
			redraw (frameSlider.getPosition());

			return 0;

		default:
			break;
	}
	return DefWindowProc(hWnd, message, wParam, lParam);
}

BOOL CALLBACK HelpDlgProc (HWND hDlg, UINT iMsg, WPARAM wParam, LPARAM lParam)
{
	switch (iMsg)
	{
        case WM_COMMAND :

			switch (LOWORD (wParam))
			{
				case IDCANCEL :
				case IDOK :

					EndDialog (hDlg, 0) ;
					return TRUE ;
			}
			break ;
    }
	return FALSE ;
}

void EnableControls (int status)
{
	if (status == 0) {

		EnableWindow (GetDlgItem (hDialog, IDC_SLIDER), FALSE);
		EnableWindow (GetDlgItem (hDialog, IDC_SPIN), FALSE);

		EnableWindow (GetDlgItem (hDialog, IDC_WIREFRAME), FALSE);
		EnableWindow (GetDlgItem (hDialog, IDC_TEXTURE), FALSE);

		EnableMenuItem (GetMenu (hDialog), IDM_MENU_ITEM_OPEN_PCX, MF_GRAYED);

		EnableMenuItem (GetMenu (hDialog), IDM_MENU_ITEM_RELOAD_PCX, MF_GRAYED);
	}

	if (status == 1) {

		EnableWindow (GetDlgItem (hDialog, IDC_SLIDER), TRUE);
		EnableWindow (GetDlgItem (hDialog, IDC_SPIN), TRUE);

		EnableWindow (GetDlgItem (hDialog, IDC_WIREFRAME), TRUE);
		EnableWindow (GetDlgItem (hDialog, IDC_TEXTURE), TRUE);

		EnableMenuItem (GetMenu (hDialog), IDM_MENU_ITEM_OPEN_PCX, MF_ENABLED);
	}
}
