# ***** BEGIN GPL LICENSE BLOCK *****

# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software Foundation,
# Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

# ***** END GPL LICENCE BLOCK *****

'''
This script is an importer and exporter for the Kingpin "MDX" file format.

The frames are named <frameName><N> with :<br>
 - <N> the frame number<br>
 - <frameName> the name choosen at the last marker
                (or 'frame' if the last marker has no name or if there is no last marker)

Skins are set using image textures in materials, if it is longer than 63 characters it is truncated.

Thanks to Bob Holcomb for MD2_NORMALS taken from his exporter.
Thanks to David Henry for the documentation about the MD2 file format.

================
previous authors
================
DarkRain
Bob Holcomb
Sebastian Lieberknecht
Dao Nguyen
Bernd Meyer
Damien Thebault
Erwan Mathieu
Takehiko Nawata


hypov8
======
v1.1.1  (tested in blender 2.79)
inital mdx suport for blender 2.79
bassed on md2 1.1.1 script with import glCommands support
added skin search path for texcture not im nodel folder
added multi part player model bbox fix. all parts must be visable in sceen
fixed texture issue in glCommands. not checking for uv match, just vertex id
v1.1.2
switched vertex normals X/Y
fixed a bad indent causing a crash

v1.1.3
fixed import animation bug. edit mode caused curuption
timeline matches length on imported animated model
clean up format to pep8
merged both mdx and md2 into 1 script file
combined import md2/mdx into 1 menu item
combined export md2/mdx into 1 menu item

todo:
suport for multi part model. for dm location damage and shootables

'''

bl_info = {
    "name": "Kingpin Model format (md2/mdx)",
    "description": "Importer and exporter for Kingpin file format (.mdx)",
    "author": "Kingpin MDX support by hypov8",
    "tracker_url": "www.kingpin.info",
    "version": (1, 1, 3),
    "blender": (2, 63, 0),
    "location": "File > Import/Export > Kingpin MDX",
    "warning": "",  # used for warning icon and text in addons panel
    "support": 'COMMUNITY',
    "category": "Import-Export"}

import bpy
from bpy.props import (
    StringProperty,
    BoolProperty,
    EnumProperty,
    IntProperty,
)

from bpy_extras.io_utils import ExportHelper, ImportHelper, unpack_list, unpack_face_list
from bpy_extras.image_utils import load_image

from math import pi
from mathutils import Matrix

import struct
import random
import os
import shutil

MDX_MAX_TRIANGLES = 4096
MDX_MAX_VERTS = 2048
MDX_MAX_FRAMES = 1024  # 512 - 1024 is an UFO:AI extension.
MDX_MAX_SKINS = 32
MDX_MAX_SKINNAME = 63
MDX_NORMALS = ((-0.525731, 0.000000, 0.850651),
               (-0.442863, 0.238856, 0.864188),
               (-0.295242, 0.000000, 0.955423),
               (-0.309017, 0.500000, 0.809017),
               (-0.162460, 0.262866, 0.951056),
               (0.000000, 0.000000, 1.000000),
               (0.000000, 0.850651, 0.525731),
               (-0.147621, 0.716567, 0.681718),
               (0.147621, 0.716567, 0.681718),
               (0.000000, 0.525731, 0.850651),
               (0.309017, 0.500000, 0.809017),
               (0.525731, 0.000000, 0.850651),
               (0.295242, 0.000000, 0.955423),
               (0.442863, 0.238856, 0.864188),
               (0.162460, 0.262866, 0.951056),
               (-0.681718, 0.147621, 0.716567),
               (-0.809017, 0.309017, 0.500000),
               (-0.587785, 0.425325, 0.688191),
               (-0.850651, 0.525731, 0.000000),
               (-0.864188, 0.442863, 0.238856),
               (-0.716567, 0.681718, 0.147621),
               (-0.688191, 0.587785, 0.425325),
               (-0.500000, 0.809017, 0.309017),
               (-0.238856, 0.864188, 0.442863),
               (-0.425325, 0.688191, 0.587785),
               (-0.716567, 0.681718, -0.147621),
               (-0.500000, 0.809017, -0.309017),
               (-0.525731, 0.850651, 0.000000),
               (0.000000, 0.850651, -0.525731),
               (-0.238856, 0.864188, -0.442863),
               (0.000000, 0.955423, -0.295242),
               (-0.262866, 0.951056, -0.162460),
               (0.000000, 1.000000, 0.000000),
               (0.000000, 0.955423, 0.295242),
               (-0.262866, 0.951056, 0.162460),
               (0.238856, 0.864188, 0.442863),
               (0.262866, 0.951056, 0.162460),
               (0.500000, 0.809017, 0.309017),
               (0.238856, 0.864188, -0.442863),
               (0.262866, 0.951056, -0.162460),
               (0.500000, 0.809017, -0.309017),
               (0.850651, 0.525731, 0.000000),
               (0.716567, 0.681718, 0.147621),
               (0.716567, 0.681718, -0.147621),
               (0.525731, 0.850651, 0.000000),
               (0.425325, 0.688191, 0.587785),
               (0.864188, 0.442863, 0.238856),
               (0.688191, 0.587785, 0.425325),
               (0.809017, 0.309017, 0.500000),
               (0.681718, 0.147621, 0.716567),
               (0.587785, 0.425325, 0.688191),
               (0.955423, 0.295242, 0.000000),
               (1.000000, 0.000000, 0.000000),
               (0.951056, 0.162460, 0.262866),
               (0.850651, -0.525731, 0.000000),
               (0.955423, -0.295242, 0.000000),
               (0.864188, -0.442863, 0.238856),
               (0.951056, -0.162460, 0.262866),
               (0.809017, -0.309017, 0.500000),
               (0.681718, -0.147621, 0.716567),
               (0.850651, 0.000000, 0.525731),
               (0.864188, 0.442863, -0.238856),
               (0.809017, 0.309017, -0.500000),
               (0.951056, 0.162460, -0.262866),
               (0.525731, 0.000000, -0.850651),
               (0.681718, 0.147621, -0.716567),
               (0.681718, -0.147621, -0.716567),
               (0.850651, 0.000000, -0.525731),
               (0.809017, -0.309017, -0.500000),
               (0.864188, -0.442863, -0.238856),
               (0.951056, -0.162460, -0.262866),
               (0.147621, 0.716567, -0.681718),
               (0.309017, 0.500000, -0.809017),
               (0.425325, 0.688191, -0.587785),
               (0.442863, 0.238856, -0.864188),
               (0.587785, 0.425325, -0.688191),
               (0.688191, 0.587785, -0.425325),
               (-0.147621, 0.716567, -0.681718),
               (-0.309017, 0.500000, -0.809017),
               (0.000000, 0.525731, -0.850651),
               (-0.525731, 0.000000, -0.850651),
               (-0.442863, 0.238856, -0.864188),
               (-0.295242, 0.000000, -0.955423),
               (-0.162460, 0.262866, -0.951056),
               (0.000000, 0.000000, -1.000000),
               (0.295242, 0.000000, -0.955423),
               (0.162460, 0.262866, -0.951056),
               (-0.442863, -0.238856, -0.864188),
               (-0.309017, -0.500000, -0.809017),
               (-0.162460, -0.262866, -0.951056),
               (0.000000, -0.850651, -0.525731),
               (-0.147621, -0.716567, -0.681718),
               (0.147621, -0.716567, -0.681718),
               (0.000000, -0.525731, -0.850651),
               (0.309017, -0.500000, -0.809017),
               (0.442863, -0.238856, -0.864188),
               (0.162460, -0.262866, -0.951056),
               (0.238856, -0.864188, -0.442863),
               (0.500000, -0.809017, -0.309017),
               (0.425325, -0.688191, -0.587785),
               (0.716567, -0.681718, -0.147621),
               (0.688191, -0.587785, -0.425325),
               (0.587785, -0.425325, -0.688191),
               (0.000000, -0.955423, -0.295242),
               (0.000000, -1.000000, 0.000000),
               (0.262866, -0.951056, -0.162460),
               (0.000000, -0.850651, 0.525731),
               (0.000000, -0.955423, 0.295242),
               (0.238856, -0.864188, 0.442863),
               (0.262866, -0.951056, 0.162460),
               (0.500000, -0.809017, 0.309017),
               (0.716567, -0.681718, 0.147621),
               (0.525731, -0.850651, 0.000000),
               (-0.238856, -0.864188, -0.442863),
               (-0.500000, -0.809017, -0.309017),
               (-0.262866, -0.951056, -0.162460),
               (-0.850651, -0.525731, 0.000000),
               (-0.716567, -0.681718, -0.147621),
               (-0.716567, -0.681718, 0.147621),
               (-0.525731, -0.850651, 0.000000),
               (-0.500000, -0.809017, 0.309017),
               (-0.238856, -0.864188, 0.442863),
               (-0.262866, -0.951056, 0.162460),
               (-0.864188, -0.442863, 0.238856),
               (-0.809017, -0.309017, 0.500000),
               (-0.688191, -0.587785, 0.425325),
               (-0.681718, -0.147621, 0.716567),
               (-0.442863, -0.238856, 0.864188),
               (-0.587785, -0.425325, 0.688191),
               (-0.309017, -0.500000, 0.809017),
               (-0.147621, -0.716567, 0.681718),
               (-0.425325, -0.688191, 0.587785),
               (-0.162460, -0.262866, 0.951056),
               (0.442863, -0.238856, 0.864188),
               (0.162460, -0.262866, 0.951056),
               (0.309017, -0.500000, 0.809017),
               (0.147621, -0.716567, 0.681718),
               (0.000000, -0.525731, 0.850651),
               (0.425325, -0.688191, 0.587785),
               (0.587785, -0.425325, 0.688191),
               (0.688191, -0.587785, 0.425325),
               (-0.955423, 0.295242, 0.000000),
               (-0.951056, 0.162460, 0.262866),
               (-1.000000, 0.000000, 0.000000),
               (-0.850651, 0.000000, 0.525731),
               (-0.955423, -0.295242, 0.000000),
               (-0.951056, -0.162460, 0.262866),
               (-0.864188, 0.442863, -0.238856),
               (-0.951056, 0.162460, -0.262866),
               (-0.809017, 0.309017, -0.500000),
               (-0.864188, -0.442863, -0.238856),
               (-0.951056, -0.162460, -0.262866),
               (-0.809017, -0.309017, -0.500000),
               (-0.681718, 0.147621, -0.716567),
               (-0.681718, -0.147621, -0.716567),
               (-0.850651, 0.000000, -0.525731),
               (-0.688191, 0.587785, -0.425325),
               (-0.587785, 0.425325, -0.688191),
               (-0.425325, 0.688191, -0.587785),
               (-0.425325, -0.688191, -0.587785),
               (-0.587785, -0.425325, -0.688191),
               (-0.688191, -0.587785, -0.425325))


class MDX:
    ''' model class'''

    def __init__(self, options):
        self.options = options
        self.object = None
        return

    def setObject(self, object):
        self.object = object
        self.mesh = self.object.to_mesh(bpy.context.scene, True, 'PREVIEW')

    def makeObject(self):
        print("Creating mesh", end='')
        # Create the mesh
        mesh = bpy.data.meshes.new(self.name)

        # Prepare vertices and faces
        mesh.vertices.add(self.numVerts)
        mesh.tessfaces.add(self.numTris)
        print('.', end='')

        # Verts
        mesh.vertices.foreach_set("co", unpack_list(self.frames[0]))
        # mesh.transform(Matrix.Rotation(-pi / 2, 4, 'Z'))
        print('.', end='')

        # Tris
        # mesh.tessfaces.foreach_set("vertices_raw", unpack_face_list([face[0] for face in self.tris])) #hypov8 disabled
        print('.', end='')

        # Skins
        mesh.tessface_uv_textures.new()
        if self.numSkins > 0:
            # material = bpy.data.materials.new(self.name)
            material = bpy.data.materials.new(self.skins[0])  # hypov8 use mdx internal name
            for skin in self.skins:
                skinImg = Util.loadImage(skin, self.filepath)
                if skinImg is None:
                    # skinImg = bpy.data.images.new(os.path.join(self.filepath, skin), self.skinWidth, self.skinHeight)
                    skinImg = bpy.data.images.new(skin, self.skinWidth, self.skinHeight)  # hypov8 sno extra data
                skinImg.mapping = 'UV'
                skinImg.name = skin
                skinTex = bpy.data.textures.new(skin, type='IMAGE')  # self.name + #hypov8 no extra data
                skinTex.image = skinImg
                matTex = material.texture_slots.add()
                matTex.texture = skinTex
                matTex.texture_coords = 'UV'
                matTex.use_map_color_diffuse = True
                matTex.use_map_alpha = True
                matTex.uv_layer = mesh.tessface_uv_textures[0].name
                material.use_shadeless = True  # hypov8
            mesh.materials.append(material)
        print('.', end='')

        # hypov8 new face/uv creator. seams an issue with foreach_set. suspect its trying to load a poly not tri
        for idx in range(len(self.tris)):
            tri = self.tris[idx][0]  # (v1ID, v2ID, v3ID)
            face = self.tris[idx][1]  # (v1UVID, v2UVID, v3UVID)
            uvX = self.uvs[face[0]]
            uvY = self.uvs[face[1]]
            uvZ = self.uvs[face[2]]
            mesh.tessface_uv_textures[0].data[idx].uv_raw = (
                uvX[0], uvX[1], uvY[0], uvY[1], uvZ[0], uvZ[1], 0.0, 0.0)
            mesh.tessfaces[idx].vertices_raw = (tri[0], tri[1], tri[2], 0)
            # hypov8 add: smooth all faces
            mesh.tessfaces[idx].use_smooth = True

        # Tris
        # mesh.tessface_uv_textures[0].data.foreach_set("uv_raw",
        #  unpack_list([self.uvs[i] for i in unpack_face_list([face[1] for face in self.tris])])) #hypov8 disable
        if self.numSkins > 0:
            image = mesh.materials[0].texture_slots[0].texture.image
            if image is not None:
                for uv in mesh.tessface_uv_textures[0].data:
                    uv.image = image
        print('.', end='')

        mesh.validate()
        mesh.update()
        obj = bpy.data.objects.new(mesh.name, mesh)
        base = bpy.context.scene.objects.link(obj)
        bpy.context.scene.objects.active = obj
        base.select = True
        print("Done")

        # Animate
        if self.options.fImportAnimation and self.numFrames > 1:
            obj.use_shape_key_edit_mode = True  # fix v1.1.3
            if self.options.fAddTimeline:
                lastFName = ""
                bpy.data.scenes[0].timeline_markers.clear()

            for i, frame in enumerate(self.frames):
                progressStatus = i / self.numFrames * 100
                bpy.context.scene.frame_set(i)
                key = obj.shape_key_add(name=("frame_%i" % i), from_mix=False)
                # mesh.vertices.foreach_set("co", unpack_list(frame))  # 2.79
                key.data.foreach_set("co", unpack_list(frame))  # fix v1.1.3
                # mesh.transform(Matrix.Rotation(-pi / 2, 4, 'Z'))

                if self.options.fAddTimeline:  # import frame names
                    tmp_str = self.frame_names[i].rstrip(b'0123456789')
                    if lastFName != tmp_str:
                        bpy.data.scenes[0].timeline_markers.new(tmp_str.decode('utf-8'), frame=i)
                        lastFName = tmp_str

                if i > 0:
                    obj.data.shape_keys.key_blocks[i].value = 1.0
                    obj.data.shape_keys.key_blocks[i].keyframe_insert("value", frame=i)
                    obj.data.shape_keys.key_blocks[i].value = 0.0
                    obj.data.shape_keys.key_blocks[i].keyframe_insert("value", frame=i - 1)
                    if i < len(self.frames) - 1:
                        obj.data.shape_keys.key_blocks[i].keyframe_insert("value", frame=i + 1)
                print("Animating - progress: %3i%%\r" % int(progressStatus), end='')

            # set sceen timeline to match imported model
            bpy.context.scene.frame_start = 0
            bpy.context.scene.frame_end = self.numFrames - 1

            print("Animating - progress: 100%.")

        bpy.context.scene.update()
        print("Model imported")

    def write(self, filePath):
        '''    Export model    '''
        self.bbox = []  # mdx
        self.skinWidth, self.skinHeight, skins = Util.getSkins(self.mesh, self.options.eTextureNameMethod)
        if self.skinWidth < 1:
            self.skinWidth = 64
        if self.skinHeight < 1:
            self.skinHeight = 64
        self.numSkins = len(skins)
        self.numVerts = len(self.mesh.vertices)
        self.numUV, uvList, uvDict = self.buildTexCoord()  # hypov8 todo: fix this
        self.numTris = len(self.mesh.tessfaces)

        # mdx
        self.numGLCmds = 1 + self.buildGLcommands()

        self.numFrames = 1
        if self.options.fExportAnimation:
            self.numFrames = 1 + self.fEndFrame - self.fStartFrame

        self.frameSize = struct.calcsize("<6f16s") + struct.calcsize("<4B") * self.numVerts

        if self.isMdx:
            self.numSfxDefines = 0  # mdx
            self.numSfxEntries = 0  # mdx
            self.numSubObjects = 1  # mdx

            self.ofsSkins = struct.calcsize("<23i")  # mdx
            self.ofsTris = self.ofsSkins + struct.calcsize("<64s") * self.numSkins
            self.ofsFrames = self.ofsTris + struct.calcsize("<6H") * self.numTris
            self.ofsGLCmds = self.ofsFrames + self.frameSize * self.numFrames

            self.ofsVertexInfo = self.ofsGLCmds + struct.calcsize("<i") * self.numGLCmds  # mdx
            self.ofsSfxDefines = self.ofsVertexInfo + struct.calcsize("<i") * self.numVerts  # mdx
            self.ofsSfxEntries = self.ofsSfxDefines  # mdx
            self.ofsBBoxFrames = self.ofsSfxEntries  # mdx
            self.ofsDummyEnd = self.ofsBBoxFrames + struct.calcsize("<6i") * self.numFrames  # mdx
            self.ofsEnd = self.ofsDummyEnd  # + struct.calcsize("<i")
        else:
            self.ofsSkins = struct.calcsize("<17i")
            self.ofsUV = self.ofsSkins + struct.calcsize("<64s") * self.numSkins
            self.ofsTris = self.ofsUV + struct.calcsize("<2h") * self.numUV
            self.ofsFrames = self.ofsTris + struct.calcsize("<6H") * self.numTris
            self.ofsGLCmds = self.ofsFrames + self.frameSize * self.numFrames
            self.ofsEnd = self.ofsGLCmds + struct.calcsize("<i") * self.numGLCmds

        file = open(filePath, "wb")
        try:
            # ####################
            # ### write header ###
            if self.isMdx:
                data = struct.pack("<23i",  # mdx
                                   self.ident,
                                   self.version,
                                   self.skinWidth,
                                   self.skinHeight,
                                   self.frameSize,
                                   self.numSkins,
                                   self.numVerts,
                                   self.numTris,
                                   self.numGLCmds,
                                   self.numFrames,
                                   self.numSfxDefines,  # mdx
                                   self.numSfxEntries,  # mdx
                                   self.numSubObjects,  # mdx
                                   self.ofsSkins,
                                   self.ofsTris,
                                   self.ofsFrames,
                                   self.ofsGLCmds,
                                   self.ofsVertexInfo,  # mdx
                                   self.ofsSfxDefines,  # mdx
                                   self.ofsSfxEntries,  # mdx
                                   self.ofsBBoxFrames,  # mdx
                                   self.ofsDummyEnd,  # mdx
                                   self.ofsEnd)
            else:
                data = struct.pack("<17i",
                                   self.ident,
                                   self.version,
                                   self.skinWidth,
                                   self.skinHeight,
                                   self.frameSize,
                                   self.numSkins,
                                   self.numVerts,
                                   self.numUV,  # number of texture coordinates
                                   self.numTris,
                                   self.numGLCmds,
                                   self.numFrames,
                                   self.ofsSkins,
                                   self.ofsUV,
                                   self.ofsTris,
                                   self.ofsFrames,
                                   self.ofsGLCmds,
                                   self.ofsEnd)

            file.write(data)

            # #############################
            # ### write skin file names ###
            for iSkin, (skinPath, skinName) in enumerate(skins):
                sourcePath = bpy.path.abspath(skinPath)

                if self.options.fCopyTextureSxS:
                    destPath = os.path.join(os.path.dirname(filePath), os.path.basename(sourcePath))
                    print("Copying texture %s to %s" % (sourcePath, destPath))
                    try:
                        shutil.copy(sourcePath, destPath)
                    except (FileNotFoundError, NameError):
                        print("Copying texture %s to %s failed." % (sourcePath, destPath))
                    if self.options.eTextureNameMethod == 'FILEPATH':
                        skinName = destPath

                if len(skinName) > MDX_MAX_SKINNAME:
                    print("WARNING: The texture name '%s' is too long. It was automatically truncated." % skinName)
                    if self.options.eTextureNameMethod == 'FILEPATH':
                        skinName = os.path.basename(skinName)

                data = struct.pack("<64s", bytes(skinName[0:MDX_MAX_SKINNAME], encoding="utf8"))
                file.write(data)  # skin name

            # ###############################
            # ### write software uv index ###
            if not self.isMdx:
                for uv in uvList:
                    data = struct.pack("<2h",
                                       int(uv[0] * self.skinWidth),
                                       int((1 - uv[1]) * self.skinHeight)
                                       )
                    file.write(data)  # uv
            del uvList

            # #################################
            # ### write triangle index data ###
            for face in self.mesh.tessfaces:
                # 0,2,1 for good cw/ccw
                data = struct.pack("<3H",  # ### write vert indices ###
                                   face.vertices[0],
                                   face.vertices[2],
                                   face.vertices[1]
                                   )
                file.write(data)  # vert index
                uvs = self.mesh.tessface_uv_textures.active.data[face.index].uv
                data = struct.pack("<3H",   # ### write tex cord indices ###
                                   uvDict[(uvs[0][0], uvs[0][1])],
                                   uvDict[(uvs[2][0], uvs[2][1])],
                                   uvDict[(uvs[1][0], uvs[1][1])],
                                   )
                file.write(data)  # uv index
            del uvDict

            # ####################
            # ### write frames ###
            if self.options.fExportAnimation and self.numFrames > 1:
                timeLineMarkers = []
                for marker in bpy.context.scene.timeline_markers:
                    timeLineMarkers.append(marker)

                # sort the markers. The marker with the frame number closest to 0 will be the first marker in the list.
                # The marker with the biggest frame number will be the last marker in the list
                timeLineMarkers.sort(key=lambda marker: marker.frame)
                markerIdx = 0

                # delete markers at same frame positions
                if len(timeLineMarkers) > 1:
                    markerFrame = timeLineMarkers[len(timeLineMarkers) - 1].frame
                    for i in range(len(timeLineMarkers) - 2, -1, -1):
                        if timeLineMarkers[i].frame == markerFrame:
                            del timeLineMarkers[i]
                        else:
                            markerFrame = timeLineMarkers[i].frame

                # calculate shared bounding box
                if self.options.fUseSharedBoundingBox:
                    self.bbox_min = None
                    self.bbox_max = None
                    for frame in range(self.fStartFrame, self.fEndFrame + 1):
                        bpy.context.scene.frame_set(frame)
                        self.calcSharedBBox()
                fNameIdx = 1
                for frame in range(self.fStartFrame, self.fEndFrame + 1):
                    frameIdx = frame - self.fStartFrame + 1
                    # Display the progress status of the export in the console
                    progressStatus = frameIdx / self.numFrames * 100
                    print("Export progress: %3i%%\r" % int(progressStatus), end='')

                    bpy.context.scene.frame_set(frame)

                    if len(timeLineMarkers) != 0:
                        if markerIdx + 1 != len(timeLineMarkers):
                            if frame >= timeLineMarkers[markerIdx + 1].frame:
                                markerIdx += 1
                                fNameIdx = 1
                            else:
                                fNameIdx += 1
                        name = timeLineMarkers[markerIdx].name + ('%02d' % fNameIdx)
                    else:
                        name = "frame" + str(frameIdx)

                    self.outFrame(file, name)
            else:
                if self.options.fUseSharedBoundingBox:
                    self.bbox_min = None
                    self.bbox_max = None
                    self.calcSharedBBox()
                self.outFrame(file)
                # self.bbox.append(getBBoxFrame(self))

            # gl commands
            # if not self.options.fSkipGLCommands: #mdx force glCommands
            for glCmd in self.glCmdList:
                if self.isMdx:
                    data = struct.pack("<iL", glCmd[0], 0)  # mdx add object id
                else:
                    data = struct.pack("<i", glCmd[0])
                file.write(data)

                for cmd in glCmd[1]:
                    data = struct.pack("<ffI", cmd[0], cmd[1], cmd[2])
                    file.write(data)
            # NULL GLCommand
            data = struct.pack("<I", 0)
            file.write(data)

            if self.isMdx:
                # ofsVertexInfo #mdx
                for i in range(self.numVerts):  # self.mesh.tessfaces:
                    data = struct.pack("<i", 1)  # fill as object #1
                    file.write(data)  # vert index

                # ofsSfxDefines #mdx
                # ofsSfxEntries #mdx

                # ofsBBoxFrames #mdx
                for i in range(self.numFrames):
                    data = struct.pack("<6f",
                                       self.bbox[i][0],
                                       self.bbox[i][1],
                                       self.bbox[i][2],
                                       self.bbox[i][3],
                                       self.bbox[i][4],
                                       self.bbox[i][5],
                                       )
                    # data = struct.pack("<3f", self.mesh.vertices[0].co[0],
                    #           self.mesh.vertices[0].co[1], self.mesh.vertices[0].co[2])
                    # file.write(data) # bbox min
                    # data = struct.pack("<3f", self.mesh.vertices[0].co[0],
                    #           self.mesh.vertices[0].co[1], self.mesh.vertices[0].co[2])
                    file.write(data)

            # ofsDummyEnd #mdx

        finally:
            file.close()
        print("Export progress: 100% - Model exported.")

    def read(self, filePath):
        ''' open .md2 file and read contents '''
        self.name = os.path.splitext(os.path.basename(filePath))[0]
        ext = os.path.splitext(os.path.basename(filePath))[1]

        if ext != '.md2' and ext != '.mdx':
            raise RuntimeError("ERROR: File not md2 or mdx")
            return False
        else:
            if ext == '.mdx':
                self.isMdx = True
                self.ident = 1481655369
                self.version = 4
            else:
                self.isMdx = False
                self.ident = 844121161
                self.version = 8

        self.filepath = filePath
        self.name = os.path.splitext(os.path.basename(filePath))[0]
        self.skins = []
        self.uvs = []
        self.tris = []
        self.frames = []
        self.frame_names = []  # 1.1.3

        print("Reading: %s" % self.filepath, end='')
        progressStatus = 0.0
        inFile = open(self.filepath, "rb")
        try:
            if self.isMdx:
                buff = inFile.read(struct.calcsize("<23i"))
                data = struct.unpack("<23i", buff)

                if data[0] != self.ident or data[1] != self.version:
                    raise NameError("Invalid MDX file")

                self.skinWidth = max(1, data[2])
                self.skinHeight = max(1, data[3])
                # framesize
                self.numSkins = data[5]
                self.numVerts = data[6]
                self.numTris = data[7]
                self.numGLCmds = data[8]
                if self.options.fImportAnimation:
                    self.numFrames = data[9]
                else:
                    self.numFrames = 1
                self.ofsSkins = data[13]
                self.ofsTris = data[14]
                self.ofsFrames = data[15]
                self.ofsGLCmds = data[16]
            else:
                buff = inFile.read(struct.calcsize("<17i"))
                data = struct.unpack("<17i", buff)

                if data[0] != self.ident or data[1] != self.version:
                    raise NameError("Invalid MD2 file")

                self.skinWidth = max(1, data[2])
                self.skinHeight = max(1, data[3])
                # framesize
                self.numSkins = data[5]
                self.numVerts = data[6]
                self.numUV = data[7]
                self.numTris = data[8]
                self.numGLCmds = data[9]  # hypo add:

                if self.options.fImportAnimation:
                    self.numFrames = data[10]
                else:
                    self.numFrames = 1
                self.ofsSkins = data[11]
                self.ofsUV = data[12]
                self.ofsTris = data[13]
                self.ofsFrames = data[14]
                self.ofsGLCmds = data[15]  # hypo add:

            #
            # Skins
            if self.numSkins > 0:
                inFile.seek(self.ofsSkins, 0)
                for i in range(self.numSkins):
                    buff = inFile.read(struct.calcsize("<64s"))
                    data = struct.unpack("<64s", buff)
                    self.skins.append(Util.asciiz(data[0].decode("utf-8", "replace")))
            print('.', end='')

            # UV (software 1byte texture cords)
            if self.isMdx is False and self.numGLCmds <= 1:
                print(" (Model does not have GLCommands) ")
                inFile.seek(self.ofsUV, 0)
                for i in range(self.numUV):
                    buff = inFile.read(struct.calcsize("<2h"))
                    data = struct.unpack("<2h", buff)
                    # self.uvs.append((data[0] / self.skinWidth, 1 - (data[1] / self.skinHeight)))
                    self.uvs.append((data[0] / self.skinWidth, 1 - (data[1] / self.skinHeight)))  # hypo add: index0
                print('.', end='')

                # Tris (non GLCommand)
                inFile.seek(self.ofsTris, 0)
                for i in range(self.numTris):
                    buff = inFile.read(struct.calcsize("<6H"))
                    data = struct.unpack("<6H", buff)
                    self.tris.append(((data[0], data[2], data[1]), (data[3], data[5], data[4])))  # hypo add
                print('.', end='')

            else:
                # =====================================================================================
                # UV GLCommands
                inFile.seek(self.ofsGLCmds, 0)
                uvIdx = 0

                def readGLVertex(inFile):
                    buff = inFile.read(struct.calcsize("<2f1l"))
                    data = struct.unpack("<2f1l", buff)
                    s = data[0]
                    t = 1.0 - data[1]  # flip Y
                    idx = data[2]
                    return (s, t, idx)

                for glx in range(self.numGLCmds):  # wont get to this number. todo while 1:?
                    if self.isMdx is True:
                        buff = inFile.read(struct.calcsize("<2l"))
                        data = struct.unpack("<2l", buff)
                    else:
                        buff = inFile.read(struct.calcsize("<l"))
                        data = struct.unpack("<l", buff)
                    # read strip
                    if data[0] >= 1:
                        numStripVerts = data[0]
                        v2 = readGLVertex(inFile)
                        v3 = readGLVertex(inFile)
                        self.uvs.append((v2[0], v2[1]))
                        self.uvs.append((v3[0], v3[1]))
                        uvIdx += 2
                        for i in range(1, (numStripVerts - 1), 1):
                            v1 = v2[:]  # new ref
                            v2 = v3[:]  # new ref
                            v3 = readGLVertex(inFile)
                            self.uvs.append((v3[0], v3[1]))
                            uvIdx += 1
                            if (i % 2) == 0:
                                self.tris.append(
                                    ((v1[2], v2[2], v3[2]), (uvIdx - 3, uvIdx - 2, uvIdx - 1)))
                            else:
                                self.tris.append(
                                    ((v3[2], v2[2], v1[2]), (uvIdx - 1, uvIdx - 2, uvIdx - 3)))
                    # read fan
                    elif data[0] <= -1:
                        numFanVerts = -data[0]
                        v1 = readGLVertex(inFile)
                        v3 = readGLVertex(inFile)
                        centreVert = uvIdx
                        self.uvs.append((v1[0], v1[1]))
                        self.uvs.append((v3[0], v3[1]))
                        uvIdx += 2
                        for i in range(1, (numFanVerts - 1), 1):
                            v2 = v3[:]  # new ref
                            v3 = readGLVertex(inFile)
                            uvIdx += 1
                            self.uvs.append((v3[0], v3[1]))
                            self.tris.append(
                                ((v3[2], v2[2], v1[2]), (uvIdx - 1, uvIdx - 2, centreVert)))
                    else:
                        print("-= done gl =-")
                        break
                print('.', end='')
                # ===================================================================================

            # Frames
            inFile.seek(self.ofsFrames, 0)
            for i in range(self.numFrames):
                buff = inFile.read(struct.calcsize("<6f16s"))
                data = struct.unpack("<6f16s", buff)
                verts = []
                for j in range(self.numVerts):
                    buff = inFile.read(struct.calcsize("<4B"))
                    vert = struct.unpack("<4B", buff)
                    verts.append((data[0] * vert[0] + data[3],
                                  data[1] * vert[1] + data[4],
                                  data[2] * vert[2] + data[5]))
                self.frames.append(verts)
                tmp_str = data[6].split(b'\x00')  # 1.1.3
                self.frame_names.append(tmp_str[0])  # 1.1.3

            print('.', end='')
        finally:
            inFile.close()
        print("Done")

    def calcSharedBBox(self):
        mesh = self.object.to_mesh(bpy.context.scene, True, 'PREVIEW')

        mesh.transform(self.object.matrix_world)
        # mesh.transform(Matrix.Rotation(pi / 2, 4, 'Z'))

        min = [mesh.vertices[0].co[0],
               mesh.vertices[0].co[1],
               mesh.vertices[0].co[2]]
        max = [mesh.vertices[0].co[0],
               mesh.vertices[0].co[1],
               mesh.vertices[0].co[2]]

        for vert in mesh.vertices:
            for i in range(3):
                if vert.co[i] < min[i]:
                    min[i] = vert.co[i]
                if vert.co[i] > max[i]:
                    max[i] = vert.co[i]

        if self.bbox_min is None:
            self.bbox_min = [min[0], min[1], min[2]]
            self.bbox_max = [max[0], max[1], max[2]]
        else:
            for i in range(3):
                if self.bbox_min[i] > min[i]:
                    self.bbox_min[i] = min[i]
                if self.bbox_max[i] < max[i]:
                    self.bbox_max[i] = max[i]

    def outFrame(self, file, frameName="frame"):
        mesh = self.object.to_mesh(bpy.context.scene, True, 'PREVIEW')

        mesh.transform(self.object.matrix_world)
        # mesh.transform(Matrix.Rotation(pi / 2, 4, 'Z'))

        if not self.options.fUseSharedBoundingBox:
            # ##### compute the bounding box ###############
            min = [9999.0, 9999.0, 9999.0]
            max = [-9999.0, -9999.0, -9999.0]

            if not self.options.fIsPlayerModel:
                for vert in mesh.vertices:
                    for i in range(3):
                        if vert.co[i] < min[i]:
                            min[i] = vert.co[i]
                        if vert.co[i] > max[i]:
                            max[i] = vert.co[i]
            # mdx PPM hypov8
            else:
                # calculate every 'visable' vertex in sceen to get player bbox
                # cant be used when using bbox for every frame
                for tmpObj in bpy.context.visible_objects:
                    if not tmpObj.type == 'MESH':
                        continue
                    tobj = tmpObj.to_mesh(bpy.context.scene, True, 'PREVIEW')
                    tobj.transform(tmpObj.matrix_world)
                    # tobj.transform(Matrix.Rotation(pi / 2, 4, 'Z'))

                    for vert in tobj.vertices:
                        for i in range(3):
                            if vert.co[i] < min[i]:
                                min[i] = vert.co[i]
                            if vert.co[i] > max[i]:
                                max[i] = vert.co[i]
            # print("bbox min({})\nbbox max({})\n".format(min, max))
            ########################################
        else:
            min = self.bbox_min
            max = self.bbox_max

        if self.isMdx:
            # mdx bbox
            bboxMin = [9999, 9999, 9999]
            bboxMax = [-9999, -9999, -9999]
            for vert in mesh.vertices:
                for i in range(3):
                    if vert.co[i] < bboxMin[i]:
                        bboxMin[i] = vert.co[i]
                    if vert.co[i] > bboxMax[i]:
                        bboxMax[i] = vert.co[i]
            self.bbox.append([bboxMin[0], bboxMin[1], bboxMin[2],
                              bboxMax[0], bboxMax[1], bboxMax[2]])

        # BL: some caching to speed it up:
        # -> sd_ gets the vertices between [0 and 255]
        #    which is our important quantization.
        sdx = (max[0] - min[0]) / 255.0
        sdy = (max[1] - min[1]) / 255.0
        sdz = (max[2] - min[2]) / 255.0
        isdx = 255.0 / (max[0] - min[0])
        isdy = 255.0 / (max[1] - min[1])
        isdz = 255.0 / (max[2] - min[2])

        # note about the scale: self.object.scale is already applied via matrix_world
        data = struct.pack("<6f16s",
                           # writing the scale of the model
                           sdx,
                           sdy,
                           sdz,
                           # now the initial offset (= min of bounding box)
                           min[0],
                           min[1],
                           min[2],
                           # and finally the name.
                           bytes(frameName, encoding="utf8"))

        file.write(data)  # frame header

        for vert in mesh.vertices:
            # find the closest normal for every vertex
            for iN in range(162):
                # hypov8 no longer inverted x/y is this ok?
                dot = vert.normal[0] * MDX_NORMALS[iN][0] + \
                    vert.normal[1] * MDX_NORMALS[iN][1] + \
                    vert.normal[2] * MDX_NORMALS[iN][2]

                if iN == 0 or dot > maxDot:
                    maxDot = dot
                    bestNormalIndex = iN

            # and now write the normal.
            data = struct.pack("<4B",
                               int((vert.co[0] - min[0]) * isdx),
                               int((vert.co[1] - min[1]) * isdy),
                               int((vert.co[2] - min[2]) * isdz),
                               bestNormalIndex)

            file.write(data)  # write vertex and normal

    def buildTexCoord(self):
        # Create an UV coord dictionary to avoid duplicate entries and save space
        meshTextureFaces = self.mesh.tessface_uv_textures.active.data
        uvList = []
        uvDict = {}
        uvCount = 0
        for uv in [(uvs[0], uvs[1]) for meshTextureFace in meshTextureFaces for uvs in meshTextureFace.uv]:
            if uv not in uvDict.keys():
                uvList.append(uv)
                uvDict[uv] = uvCount
                uvCount += 1

        return uvCount, uvList, uvDict

    def findStripLength(self, startTri, startVert):
        meshTextureFaces = self.mesh.tessface_uv_textures.active.data
        numFaces = len(self.mesh.tessfaces)

        self.cmdVerts = []
        self.cmdTris = []
        self.cmdUV = []
        self.used[startTri] = 2

        self.cmdVerts.append(self.mesh.tessfaces[startTri].vertices_raw[startVert % 3])
        self.cmdVerts.append(self.mesh.tessfaces[startTri].vertices_raw[(startVert + 2) % 3])
        self.cmdVerts.append(self.mesh.tessfaces[startTri].vertices_raw[(startVert + 1) % 3])
        self.cmdUV.append(meshTextureFaces[startTri].uv[startVert % 3])
        self.cmdUV.append(meshTextureFaces[startTri].uv[(startVert + 2) % 3])
        self.cmdUV.append(meshTextureFaces[startTri].uv[(startVert + 1) % 3])

        stripCount = 1
        self.cmdTris.append(startTri)
        m1 = self.mesh.tessfaces[startTri].vertices_raw[(startVert + 2) % 3]
        m2 = self.mesh.tessfaces[startTri].vertices_raw[(startVert + 1) % 3]
        u1 = meshTextureFaces[startTri].uv[(startVert + 2) % 3]  # hypov8 add:
        u2 = meshTextureFaces[startTri].uv[(startVert + 1) % 3]  # hypov8 add:

        for triCounter in range(startTri + 1, numFaces):
            for k in range(3):
                # print("uv===({}, {} ) ({}, {})".format(u1[0], u1[1],
                #       meshTextureFaces[triCounter].uv[k][0], meshTextureFaces[triCounter].uv[k][1]))
                if((self.mesh.tessfaces[triCounter].vertices_raw[k] == m1) and
                   (self.mesh.tessfaces[triCounter].vertices_raw[(k + 1) % 3] == m2) and
                   (meshTextureFaces[triCounter].uv[k][0] == u1[0]) and
                   (meshTextureFaces[triCounter].uv[k][1] == u1[1])):  # hypov8 add: make sure uv also match

                    if(self.used[triCounter] == 0):
                        if(stripCount % 2 == 1):  # is this an odd tri
                            m1 = self.mesh.tessfaces[triCounter].vertices_raw[(k + 2) % 3]
                            u1 = meshTextureFaces[triCounter].uv[(k + 2) % 3]
                        else:
                            m2 = self.mesh.tessfaces[triCounter].vertices_raw[(k + 2) % 3]
                            u2 = meshTextureFaces[triCounter].uv[(k + 2) % 3]

                        self.cmdVerts.append(self.mesh.tessfaces[triCounter].vertices_raw[(k + 2) % 3])
                        self.cmdUV.append(meshTextureFaces[triCounter].uv[(k + 2) % 3])
                        stripCount += 1
                        self.cmdTris.append(triCounter)

                        self.used[triCounter] = 2
                        triCounter = startTri + 1  # restart looking

        # clear used counter
        for usedCounter in range(numFaces):
            if self.used[usedCounter] == 2:
                self.used[usedCounter] = 0

        return stripCount

    def findFanLength(self, startTri, startVert):
        meshTextureFaces = self.mesh.tessface_uv_textures.active.data
        numFaces = len(self.mesh.tessfaces)

        self.cmdVerts = []
        self.cmdTris = []
        self.cmdUV = []
        self.used[startTri] = 2

        self.cmdVerts.append(self.mesh.tessfaces[startTri].vertices_raw[startVert % 3])
        self.cmdVerts.append(self.mesh.tessfaces[startTri].vertices_raw[(startVert + 2) % 3])
        self.cmdVerts.append(self.mesh.tessfaces[startTri].vertices_raw[(startVert + 1) % 3])
        self.cmdUV.append(meshTextureFaces[startTri].uv[startVert % 3])
        self.cmdUV.append(meshTextureFaces[startTri].uv[(startVert + 2) % 3])
        self.cmdUV.append(meshTextureFaces[startTri].uv[(startVert + 1) % 3])

        fanCount = 1
        self.cmdTris.append(startTri)
        m2 = self.mesh.tessfaces[startTri].vertices_raw[(startVert + 0) % 3]
        m1 = self.mesh.tessfaces[startTri].vertices_raw[(startVert + 1) % 3]
        u2 = meshTextureFaces[startTri].uv[(startVert + 0) % 3]  # hypov8 add:
        u1 = meshTextureFaces[startTri].uv[(startVert + 1) % 3]  # hypov8 add:

        for triCounter in range(startTri + 1, numFaces):
            for k in range(3):
                if((self.mesh.tessfaces[triCounter].vertices_raw[k] == m1) and
                   (self.mesh.tessfaces[triCounter].vertices_raw[(k + 1) % 3] == m2)and
                   (meshTextureFaces[triCounter].uv[k][0] == u1[0]) and
                   (meshTextureFaces[triCounter].uv[k][1] == u1[1])):  # hypov8 add: make sure uv also match

                    if(self.used[triCounter] == 0):
                        m1 = self.mesh.tessfaces[triCounter].vertices_raw[(k + 2) % 3]
                        u1 = meshTextureFaces[startTri].uv[(k + 2) % 3]  # hypov8 add:
                        self.cmdVerts.append(self.mesh.tessfaces[triCounter].vertices_raw[(k + 2) % 3])
                        self.cmdUV.append(meshTextureFaces[triCounter].uv[(k + 2) % 3])
                        fanCount += 1
                        self.cmdTris.append(triCounter)

                        self.used[triCounter] = 2
                        triCounter = startTri + 1  # restart looking

        # clear used counter
        for usedCounter in range(numFaces):
            if self.used[usedCounter] == 2:
                self.used[usedCounter] = 0

        return fanCount

    def buildGLcommands(self):
        numFaces = len(self.mesh.tessfaces)
        self.used = [0] * numFaces
        numCommands = 0
        self.glCmdList = []

        for triCounter in range(numFaces):
            if self.used[triCounter] == 0:
                # intialization
                bestLength = 0
                bestType = 0
                bestVerts = []
                bestTris = []
                bestUV = []

                for startVert in range(3):
                    cmdLength = self.findFanLength(triCounter, startVert)
                    if (cmdLength > bestLength):
                        bestType = 1
                        bestLength = cmdLength
                        bestVerts = self.cmdVerts
                        bestTris = self.cmdTris
                        bestUV = self.cmdUV

                    cmdLength = self.findStripLength(triCounter, startVert)
                    if (cmdLength > bestLength):
                        bestType = 0
                        bestLength = cmdLength
                        bestVerts = self.cmdVerts
                        bestTris = self.cmdTris
                        bestUV = self.cmdUV

                # mark tris as used
                for usedCounter in range(bestLength):
                    self.used[bestTris[usedCounter]] = 1

                cmd = []
                if bestType == 0:  # strip
                    num = bestLength + 2
                else:  # fan
                    num = (-(bestLength + 2))

                numCommands += 1
                if self.isMdx:  # mdx
                    numCommands += 1

                for cmdCounter in range(bestLength + 2):
                    # (u,v) in blender -> (u,1-v)
                    cmd.append((bestUV[cmdCounter][0], 1.0 - bestUV[cmdCounter][1], bestVerts[cmdCounter]))
                    numCommands += 3

                self.glCmdList.append((num, cmd))

        del self.used, bestVerts, bestUV, bestTris, self.cmdVerts, self.cmdUV, self.cmdTris
        return numCommands


class Util:
    # deletes an object from Blender (remove + unlink)
    @staticmethod
    def deleteObject(object):
        bpy.context.scene.objects.unlink(object)
        bpy.data.objects.remove(object)

    # duplicates the given object and returns it
    @staticmethod
    def duplicateObject(object):
        # backup the current object selection and current active object
        selObjects = bpy.context.selected_objects[:]
        actObject = bpy.context.active_object

        # deselect all selected objects
        bpy.ops.object.select_all(action='DESELECT')
        # select the object which we want to duplicate
        object.select = True

        # duplicate the selected object
        bpy.ops.object.duplicate()

        # the duplicated object is automatically selected
        copyObj = bpy.context.selected_objects[0]

        # select all objects which have been previously selected and make active the previous active object
        bpy.context.scene.objects.active = actObject
        for obj in selObjects:
            obj.select = True

        return copyObj

    @staticmethod
    def applyModifiers(object):
        if len(object.modifiers) == 0:
            return object

        modifier = object.modifiers.new('Triangulate-Export', 'TRIANGULATE')
        mesh = object.to_mesh(bpy.context.scene, True, 'PREVIEW')
        modifiedObj = bpy.data.objects.new(mesh.name, mesh)
        bpy.context.scene.objects.link(modifiedObj)
        object.modifiers.remove(modifier)

        return modifiedObj

    # returns the mesh of the object and return object.data (mesh)
    @staticmethod
    def triangulateMesh(object):
        modifier = object.modifiers.new('Triangulate-Export', 'TRIANGULATE')

    @staticmethod
    def getSkins(mesh, method):
        skins = []
        width = -1
        height = -1
        for material in mesh.materials:
            for texSlot in material.texture_slots:
                if not texSlot or texSlot.texture.type != 'IMAGE':
                    continue
                if any(texSlot.texture.image.filepath in skin for skin in skins):
                    continue
                if method == 'BASENAME':
                    texname = os.path.basename(texSlot.texture.image.filepath)
                elif method == 'FILEPATH':
                    texname = texSlot.texture.image.filepath
                elif method == 'SHADENAME':  # mdx
                    texname = texSlot.texture.name  # mdx
                else:
                    texname = texSlot.texture.image.name
                skins.append((texSlot.texture.image.filepath, texname))
                if texSlot.texture.image.size[0] > width:
                    width = texSlot.texture.image.size[0]
                if texSlot.texture.image.size[1] > height:
                    height = texSlot.texture.image.size[1]
        return width, height, skins

    @staticmethod
    def loadImage(mdxPath, filePath):
        # Handle ufoai skin name format
        fileName = os.path.basename(mdxPath)
        # if mdxPath[0] == '.':
        #    for ext in ['.png', '.jpg', '.jpeg']:
        #        fileName = mdxPath[1:] + ext
        #        if os.path.isfile(os.path.join(os.path.dirname(mdxPath), fileName)):
        #            break
        #        elif os.path.isfile(os.path.join(os.path.dirname(filePath), fileName)):
        #            break
        #    else:
        #        fileName = mdxPath[1:]
        image = load_image(fileName, os.path.dirname(mdxPath), recursive=False)
        if image is not None:
            return image
        image = load_image(fileName, os.path.dirname(filePath), recursive=False)
        if image is not None:
            return image
        # mdx
        idxModels = filePath.find("models\\")
        idxPlayer = filePath.find("players\\")
        idxTextur = filePath.find("textures\\")

        outname = ""
        if idxModels >= 1:
            if filePath[0] == "/":
                outname = filePath[1:idxModels]
            else:
                outname = filePath[0:idxModels]
        elif idxPlayer >= 1:
            if filePath[0] == "/":
                outname = filePath[1:idxPlayer]  # trim
            else:
                outname = filePath[0:idxPlayer]  # trim

        idxModels = filePath.find("models\\")
        idxPlayer = filePath.find("players\\")
        idxTextur = filePath.find("textures\\")

        fullpath = outname + mdxPath
        fullpath = bpy.path.native_pathsep(fullpath)
        image = load_image(fileName, os.path.dirname(fullpath), recursive=False)
        print("\nfileName={}".format(fileName))
        print("path={}".format(fullpath))
        print("basepath={}".format(os.path.basename(fullpath)))
        print("dirname={}".format(os.path.dirname(fullpath)))
        if image is not None:
            return image

        print("image failed!!!")

        return None

    @staticmethod
    def asciiz(s):
        for i, c in enumerate(s):
            if ord(c) == 0:
                return s[:i]


class ObjectInfo:
    def __init__(self, object):
        self.vertices = -1
        self.faces = 0
        self.status = ('', '')
        # self.frames = 1 + self.properties.fEndFrame - self.properties.fStartFrame
        self.isMesh = object and object.type == 'MESH'

        if self.isMesh:
            originalObject = object
            mesh = object.data

            self.skinWidth, self.skinHeight, self.skins = Util.getSkins(mesh, 'DATANAME')

            try:
                # apply the modifiers
                object = Util.applyModifiers(object)
                mesh = object.data

                mesh.update(calc_tessface=True)
                self.status = (str(len(mesh.vertices)) + " vertices", str(len(mesh.tessfaces)) + " faces")
                self.faces = len(mesh.tessfaces)
                self.vertices = len(mesh.vertices)
                self.isUnwrapped = len(mesh.tessface_uv_textures) > 0

            finally:
                if object.name != originalObject.name:
                    originalObject.select = True
                    bpy.context.scene.objects.active = originalObject
                    Util.deleteObject(object)
            print(originalObject.name + ": ", self.status)


class Export_MDX(bpy.types.Operator, ExportHelper):
    '''Export selection to Kingpin file format (md2/mdx)'''
    bl_idname = "export_kingpin.mdx"
    bl_label = "Export Kingpin Model (md2, mdx)"
    filename_ext = ".md2"  # md2 used later

    fExportAnimation = BoolProperty(
        name="Export animation",
        description="Export all frames",
        default=False)
    typesExportSkinNames = [
        ('SHADENAME', "Shader Name", "Use Texture name from slot"),  # mdx
        ('DATANAME', "Data block name",
         "Use image datablock names"),
        ('FILEPATH', "File path", "Use image file paths"),
        ('BASENAME', "File name", "Use image file names"),
    ]
    eTextureNameMethod = EnumProperty(
        name="Skin names",
        description="Choose skin naming method",
        items=typesExportSkinNames,
    )
    fCopyTextureSxS = BoolProperty(
        name="Copy texture(s) next to .mdx",
        description="Try to copy textures to mdx directory (won't overwrite files)",
        default=False
    )
    fUseSharedBoundingBox = BoolProperty(
        name="Use shared bounding box across frames",
        description="Calculate a shared bounding box from all frames\n" +
                    "Used to avoid wobbling in static vertices but wastes resolution",
        default=False
    )
    fIsPlayerModel = BoolProperty(
        name="Multy Part Player Model",  # mdx
        description="Use all visible sceen object to create bounding box size.\n" +
        "This fixes seam algment issues in players",
        default=False
    )
    filter_glob = StringProperty(  # 2.8
        default="*.md2;*.mdx",
        options={'HIDDEN'},
    )
    fStartFrame = IntProperty(
        name="Start Frame",
        description="Animated model start frame",
        min=0,
        max=1024,
        default=0,
    )
    fEndFrame = IntProperty(
        name="End Frame",
        description="Animated model end frame",
        min=0,
        max=1024,
        default=40,
    )

    check_extension = False  # 2.8 allow typing md2/mdx

    def __init__(self):
        self.properties.fStartFrame = bpy.context.scene.frame_start
        self.properties.fEndFrame = bpy.context.scene.frame_end
        try:
            self.object = bpy.context.selected_objects[0]
            # go into object mode before we start the actual export procedure
            bpy.ops.object.mode_set(mode='OBJECT', toggle=False)
            self.info = ObjectInfo(self.object)
        except IndexError:
            self.object = None

    def execute(self, context):
        ''' Export model '''
        props = self.properties
        filePath = self.filepath

        ext = os.path.splitext(os.path.basename(filePath))[1]
        if ext != '.md2' and ext != '.mdx':
            raise RuntimeError("ERROR: File not md2 or mdx")

        object = self.object
        originalObject = object

        self.frames = 1 + self.fEndFrame - self.fStartFrame

        if self.frames > MDX_MAX_FRAMES and self.fExportAnimation:
            raise RuntimeError("There are too many frames (%i), at most %i are supported in mdx"
                               % (self.frames, MDX_MAX_FRAMES))

        object = Util.duplicateObject(object)
        Util.triangulateMesh(object)

        object.data.update(calc_tessface=True)

        # save the current frame to reset it after export
        if self.fExportAnimation:
            frame = bpy.context.scene.frame_current

        try:
            mdx = MDX(self)
            if ext == '.mdx':
                mdx.isMdx = True
                mdx.ident = 1481655369
                mdx.version = 4
            else:
                filePath = bpy.path.ensure_ext(filePath, self.filename_ext)
                mdx.isMdx = False
                mdx.ident = 844121161
                mdx.version = 8

            mdx.fStartFrame = self.properties.fStartFrame
            mdx.fEndFrame = self.properties.fEndFrame

            mdx.setObject(object)
            mdx.write(filePath)
        finally:
            if object.name != originalObject.name:
                originalObject.select = True
                bpy.context.scene.objects.active = originalObject
                Util.deleteObject(object)
            if self.fExportAnimation:
                bpy.context.scene.frame_set(frame)

        self.report({'INFO'}, "Model '%s' exported" % originalObject.name)
        return {'FINISHED'}

    def draw(self, context):
        layout = self.layout

        layout.prop(self, "fExportAnimation")
        layout.prop(self, "fIsPlayerModel")
        sub = layout.column()
        sub.enabled = self.fExportAnimation
        sub.prop(self, "fStartFrame", )
        sub.prop(self, "fEndFrame")
        layout.prop(self, "eTextureNameMethod")
        layout.prop(self, "fUseSharedBoundingBox")

    def invoke(self, context, event):
        if not context.selected_objects:
            self.report({'ERROR'}, "Please, select an object to export!")
            return {'CANCELLED'}
        # check constrains
        if len(bpy.context.selected_objects) > 1:
            self.report(
                {'ERROR'}, "Please, select exactly one object to export!")
            return {'CANCELLED'}

        if not self.info.isMesh:
            self.report({'ERROR'}, "Only meshes can be exported (selected object is of type '%s')"
                        % (self.object.type))
            return {'CANCELLED'}
        if self.info.faces > MDX_MAX_TRIANGLES:
            self.report({'ERROR'},
                        "Object has too many (triangulated) faces (%i), at most %i are supported in mdx"
                        % (self.info.faces, MDX_MAX_TRIANGLES))
            return {'CANCELLED'}
        if self.info.vertices > MDX_MAX_VERTS:
            self.report({'ERROR'},
                        "Object has too many vertices (%i), at most %i are supported in mdx"
                        % (self.info.vertices, MDX_MAX_VERTS))
            return {'CANCELLED'}
        if not self.info.isUnwrapped:
            self.report({'ERROR'}, "Mesh must be unwrapped")
            return {'CANCELLED'}
        if len(self.info.skins) < 1:
            self.report({'ERROR'}, "There must be at least one skin")
            return {'CANCELLED'}
        if len(self.info.skins) > MDX_MAX_SKINS:
            self.report({'ERROR'}, "There are too many skins (%i), at most %i are supported in mdx"
                        % (len(self.info.skins), MDX_MAX_SKINS))
            return {'CANCELLED'}

        wm = context.window_manager
        wm.fileselect_add(self)
        return {'RUNNING_MODAL'}


class Import_MDX(bpy.types.Operator, ImportHelper):
    '''Import Kingpin format file (md2/mdx)'''
    bl_idname = "import_kingpin.mdx"
    bl_label = "Import Kingpin Model (md2/mdx)"

    filename_ext = ".mdx"  # 2.8
    fImportAnimation = BoolProperty(
        name="Import animation",
        description="Import all frames",
        default=True
    )
    fAddTimeline = BoolProperty(
        name="Import animation names",
        description="Import animation frame names to time line\n" +
        "WARNING: Removes all existing marker frame names",
        default=False,
    )

    filter_glob = StringProperty(  # 2.8
        default="*.md2;*.mdx",
        options={'HIDDEN'},
    )

    def __init__(self):
        if len(bpy.context.selected_objects) > 0:
            # go into object mode before we start the actual import procedure
            bpy.ops.object.mode_set(mode='OBJECT', toggle=False)

    def execute(self, context):
        # deselect all objects
        bpy.ops.object.select_all(action='DESELECT')

        mdx = MDX(self)
        mdx.read(self.filepath)
        mdx.makeObject()

        bpy.context.scene.update()
        self.report({'INFO'}, "File '%s' imported" % self.filepath)
        return {'FINISHED'}

    def draw(self, context):
        layout = self.layout
        layout.prop(self, "fImportAnimation")
        sub = layout.column()
        sub.enabled = self.fImportAnimation
        sub.prop(self, "fAddTimeline", )


def menu_func_export(self, context):
    self.layout.operator(Export_MDX.bl_idname, text="Kingpin Model (md2/mdx)")


def menu_func_import(self, context):
    self.layout.operator(Import_MDX.bl_idname, text="Kingpin Model (md2/mdx)")


def register():
    bpy.utils.register_module(__name__)
    bpy.types.INFO_MT_file_export.append(menu_func_export)
    bpy.types.INFO_MT_file_import.append(menu_func_import)


def unregister():
    bpy.utils.unregister_module(__name__)
    bpy.types.INFO_MT_file_export.remove(menu_func_export)
    bpy.types.INFO_MT_file_import.remove(menu_func_import)


if __name__ == "__main__":
    register()
