			''
      '' This tool written in FreeBASIC. Use & modify at your own risks. - 2021, ACC
      ''
      

      #DEFINE MAX_VERTICES    2048
			#DEFINE MAX_TEXCOORDS   2044
			#DEFINE MAX_TRIANGLES   2048
			#DEFINE MAX_GLPACKETS   4096
      #DEFINE MAX_SKINS         16

			#DEFINE err_no_handle           -1
			#DEFINE err_no_found            -2
			#DEFINE err_no_specified        -3
			#DEFINE err_general_io          -4
			#DEFINE err_frame_out_of_range  -5
			#DEFINE err_not_MD2             -6
			#DEFINE err_not_version_8       -7
			#DEFINE err_not_MDX             -8
			#DEFINE err_not_version_4       -9

			#MACRO memSet(x, y)
			x = CALLOCATE(SIZEOF(TYPEOF(*x)) * y)
			#ENDMACRO

			#MACRO memZString(x, y)
			x = CALLOCATE(SIZEOF(TYPEOF(*x)) * (LEN(y) + 1))
			*x = y
			#ENDMACRO

			#MACRO memFree(x)
			IF x THEN
				DEALLOCATE(x): x = 0
			END IF
			#ENDMACRO

			TYPE vec2_t
				x AS SINGLE                   ' Horizontal coordinate
				y AS SINGLE                   ' Vertical coordinate
			END TYPE

			TYPE vec3_t
				x AS SINGLE                   ' X coordinate (left-right)
				y AS SINGLE                   ' Y coordinate (depth)
				z AS SINGLE                   ' Z coordinate (height)
			END TYPE

			''
			'' MD2
			''
			TYPE md2_t
				AS INTEGER ident              ' IDP2 (&h32504449)
				AS INTEGER version            ' must be 8
				AS INTEGER skinWidth          ' skin width, in pixels
				AS INTEGER skinHeight         ' skin height, in pixels
				AS INTEGER frameSize          ' size of a frame, in bytes
				AS INTEGER numSkins           ' number of skins
				AS INTEGER numVertices        ' number of vertices per frame
				AS INTEGER numSt              ' number of texture coordinates (not in MDX)
				AS INTEGER numTris            ' number of triangles
				AS INTEGER numGlCmds          ' number of packets in the OpenGL stream, in integers
				AS INTEGER numFrames          ' number of frames
				AS INTEGER ofsSkins           ' offset to skin data
				AS INTEGER ofsSt              ' offset to texture coordinate data (not in MDX)
				AS INTEGER ofsTris            ' offset to triangle data
				AS INTEGER ofsFrames          ' offset to frame data
				AS INTEGER ofsGlCmds          ' offset to OpenGL stream
				AS INTEGER ofsEnd             ' offset to end of file
			END TYPE

			''
			'' MDX
			''
			TYPE mdx_t
				AS INTEGER ident              ' IDPX (&h58504449)
				AS INTEGER version            ' must be 8
				AS INTEGER skinWidth          ' skin width, in pixels
				AS INTEGER skinHeight         ' skin height, in pixels
				AS INTEGER frameSize          ' size of a frame, in bytes
				AS INTEGER numSkins           ' number of skins
				AS INTEGER numVertices        ' number of vertices per frame
				AS INTEGER numTris            ' number of triangles
				AS INTEGER numGlCmds          ' number of packets in the OpenGL stream, in integers
				AS INTEGER numFrames          ' number of frames
				AS INTEGER numSfxDefines
				AS INTEGER numSfxEntries
				AS INTEGER numSubObjects
				AS INTEGER ofsSkins           ' offset to skin data
				AS INTEGER ofsTris            ' offset to triangle data
				AS INTEGER ofsFrames          ' offset to frame data
				AS INTEGER ofsGlCmds          ' offset to OpenGL stream
				AS INTEGER ofsVertexInfo
				AS INTEGER ofsSfxDefines
				AS INTEGER ofsSfxEntries
				AS INTEGER ofsBBoxFrames
				AS INTEGER ofsDummyEnd
				AS INTEGER ofsEnd             ' offset to end of file
			END TYPE

			''
			'' Script
			''
			TYPE scriptV_t
				AS vec3_t org                 ' origin (coordinate)
				AS SINGLE n                   ' normal (processing)
				AS UBYTE  nIndex              ' normal (index in list)
			END TYPE

			TYPE scriptT_t
				AS SHORT  p1                  ' point 1 (index)
				AS SHORT  p2                  ' point 2 (index)
				AS SHORT  p3                  ' point 3 (index)
				AS SHORT  t1                  ' texture point 1 (index)
				AS SHORT  t2                  ' texture point 1 (index)
				AS SHORT  t3                  ' texture point 1 (index)
				AS vec3_t n                   ' normal (vector)
				AS SINGLE s                   ' area/surface
			END TYPE

			TYPE script_t
				AS scriptV_t PTR vertex       ' Each 3D vertex
				AS vec2_t    PTR texCoord     ' Each 2D vertex
				AS scriptT_t PTR triangle     ' Each triangle
				AS INTEGER   PTR GLPackets    ' OpenGL stream
				AS ZSTRING   PTR skinFile     ' Skin filename
				AS USHORT        skinWidth    ' Skin length, in pixels
				AS USHORT        skinHeight   ' Skin height, in pixels
				AS ZSTRING   PTR outputFile   ' Output file (MD2)
				AS vec3_t        min          ' Bounding box
				AS vec3_t        max          ' Bounding box
				AS vec3_t        box          ' Bounding box
				AS USHORT        numVertices  ' Number of 3D vertices
				AS USHORT        numTexCoords ' Number of 2D vertices
				AS USHORT        numTris ' Number of triangles
				AS USHORT        numGLPackets ' Number of GLPackets
				AS USHORT        numSkins     ' Number of skins

				DECLARE SUB DESTROY()         ' Manual free-all
				DECLARE CONSTRUCTOR ()        ' Constructor
				DECLARE DESTRUCTOR ()         ' Destructor
			END TYPE

			SUB script_t.DESTROY()
				memFree(this.vertex)
				memFree(this.texCoord)
				memFree(this.triangle)
				memFree(this.GLPackets)
				memFree(this.outputFile)
				memFree(this.skinFile)
				this.skinWidth = 0
				this.skinHeight = 0
				this.numVertices = 0
				this.numTexCoords = 0
				this.numTris = 0
				this.numGLPackets = 0
				this.numSkins = 0
			END SUB

			CONSTRUCTOR script_t()
				memSet(this.vertex, MAX_VERTICES)
				memSet(this.texCoord, MAX_TEXCOORDS)
				memSet(this.triangle, MAX_TRIANGLES)
				memSet(this.GLPackets, MAX_GLPACKETS)
        memSet(this.skinFile, MAX_SKINS * 64)
			END CONSTRUCTOR

			DESTRUCTOR script_t()
				this.destroy()
			END DESTRUCTOR

			'' OPERATOR OVERLOAD (COMPARE VEC3_T)
			OPERATOR = (BYREF lhs AS vec3_t, BYREF rhs AS vec3_t) AS INTEGER
				RETURN ((lhs.x = rhs.x) AND (lhs.y = rhs.y) AND (lhs.z = rhs.z))
			END OPERATOR

			'' OPERATOR OVERLOAD (COMPARE VEC2_T)
			OPERATOR = (BYREF lhs AS vec2_t, BYREF rhs AS vec2_t) AS INTEGER
				RETURN ((lhs.x = rhs.x) AND (lhs.y = rhs.y))
			END OPERATOR

			DECLARE FUNCTION loadMD2(filename AS STRING, BYREF frameId AS INTEGER, BYREF script AS script_t) AS INTEGER
			DECLARE FUNCTION loadMDX(filename AS STRING, BYREF frameId AS INTEGER, BYREF script AS script_t) AS INTEGER
			DECLARE FUNCTION saveScript(filename AS STRING, script AS script_t) AS INTEGER
      DECLARE FUNCTION saveMD2(BYREF filename AS STRING, BYREF script AS script_t) AS INTEGER
			DECLARE FUNCTION vecNormalize(BYREF v AS vec3_t) AS vec3_t
			DECLARE FUNCTION trisSurface(BYREF p1 AS vec3_t PTR, BYREF p2 AS vec3_t PTR, BYREF p3 AS vec3_t PTR) AS SINGLE
			DECLARE FUNCTION trisNormal(BYREF p1 AS vec3_t PTR, BYREF p2 AS vec3_t PTR, BYREF p3 AS vec3_t PTR) AS vec3_t
			DECLARE FUNCTION vecSubtract(BYREF lhs AS vec3_t, BYREF rhs AS vec3_t) AS vec3_t
			DECLARE FUNCTION loadScript(BYREF filename AS STRING, BYREF target AS script_t) AS INTEGER
      DECLARE SUB strClean(BYREF msg AS STRING)
			DECLARE SUB vecCopy(BYREF lhs AS vec3_t, BYREF rhs AS vec3_t)

			DIM AS script_t script
			DIM AS INTEGER  lastErr = any, mode = any, frame = 0
			DIM AS STRING   source, target

			source = COMMAND(1)
			frame = VAL(COMMAND(2))
			IF (LEN(source) = 0) THEN END
			target = "outscript.txt"

			SELECT CASE UCASE(RIGHT(source, LEN(source) - INSTRREV(source, ".")))
			CASE "MD2"
        mode = 1
			CASE "MDX"
        mode = 2
			CASE "TXT"
        mode = 3
			CASE ELSE
				PRINT "Format unsupported"
        SLEEP: END
			END SELECT

			' get source
			SELECT CASE mode
				CASE 1
					lastErr = loadMD2(source, frame, script)
				CASE 2
					lastErr = loadMDX(source, frame, script)
				CASE 3
					lastErr = loadScript(source, script)
			END SELECT
			IF lastErr THEN PRINT "Loading - error code: ";lastErr:SLEEP:END

			' write target
			SELECT CASE mode
			CASE 1, 2
				lastErr = saveScript(target, script)
			CASE 3
				IF (script.outputFile = 0) THEN
					memZString(script.outputFile, "tris.md2")
				END IF
				lastErr = saveMD2(*script.outputFile, script)
			END SELECT
			IF lastErr THEN PRINT "Saving - error code: ";lastErr:SLEEP:END

			script.destroy()

			SLEEP

			normalList:
				DATA -0.525731, 0.000000, 0.850651,  -0.442863, 0.238856, 0.864188
				DATA -0.295242, 0.000000, 0.955423,  -0.309017, 0.500000, 0.809017
				DATA -0.162460, 0.262866, 0.951056,   0.000000, 0.000000, 1.000000
				DATA  0.000000, 0.850651, 0.525731,  -0.147621, 0.716567, 0.681718
				DATA  0.147621, 0.716567, 0.681718,   0.000000, 0.525731, 0.850651
				DATA  0.309017, 0.500000, 0.809017,   0.525731, 0.000000, 0.850651
				DATA  0.295242, 0.000000, 0.955423,   0.442863, 0.238856, 0.864188
				DATA  0.162460, 0.262866, 0.951056,  -0.681718, 0.147621, 0.716567
				DATA -0.809017, 0.309017, 0.500000,  -0.587785, 0.425325, 0.688191
				DATA -0.850651, 0.525731, 0.000000,  -0.864188, 0.442863, 0.238856
				DATA -0.716567, 0.681718, 0.147621,  -0.688191, 0.587785, 0.425325
				DATA -0.500000, 0.809017, 0.309017,  -0.238856, 0.864188, 0.442863
				DATA -0.425325, 0.688191, 0.587785,  -0.716567, 0.681718,-0.147621
				DATA -0.500000, 0.809017,-0.309017,  -0.525731, 0.850651, 0.000000
				DATA  0.000000, 0.850651,-0.525731,  -0.238856, 0.864188,-0.442863
				DATA  0.000000, 0.955423,-0.295242,  -0.262866, 0.951056,-0.162460
				DATA  0.000000, 1.000000, 0.000000,   0.000000, 0.955423, 0.295242
				DATA -0.262866, 0.951056, 0.162460,   0.238856, 0.864188, 0.442863
				DATA  0.262866, 0.951056, 0.162460,   0.500000, 0.809017, 0.309017
				DATA  0.238856, 0.864188,-0.442863,   0.262866, 0.951056,-0.162460
				DATA  0.500000, 0.809017,-0.309017,   0.850651, 0.525731, 0.000000
				DATA  0.716567, 0.681718, 0.147621,   0.716567, 0.681718,-0.147621
				DATA  0.525731, 0.850651, 0.000000,   0.425325, 0.688191, 0.587785
				DATA  0.864188, 0.442863, 0.238856,   0.688191, 0.587785, 0.425325
				DATA  0.809017, 0.309017, 0.500000,   0.681718, 0.147621, 0.716567
				DATA  0.587785, 0.425325, 0.688191,   0.955423, 0.295242, 0.000000
				DATA  1.000000, 0.000000, 0.000000,   0.951056, 0.162460, 0.262866
				DATA  0.850651,-0.525731, 0.000000,   0.955423,-0.295242, 0.000000
				DATA  0.864188,-0.442863, 0.238856,   0.951056,-0.162460, 0.262866
				DATA  0.809017,-0.309017, 0.500000,   0.681718,-0.147621, 0.716567
				DATA  0.850651, 0.000000, 0.525731,   0.864188, 0.442863,-0.238856
				DATA  0.809017, 0.309017,-0.500000,   0.951056, 0.162460,-0.262866
				DATA  0.525731, 0.000000,-0.850651,   0.681718, 0.147621,-0.716567
				DATA  0.681718,-0.147621,-0.716567,   0.850651, 0.000000,-0.525731
				DATA  0.809017,-0.309017,-0.500000,   0.864188,-0.442863,-0.238856
				DATA  0.951056,-0.162460,-0.262866,   0.147621, 0.716567,-0.681718
				DATA  0.309017, 0.500000,-0.809017,   0.425325, 0.688191,-0.587785
				DATA  0.442863, 0.238856,-0.864188,   0.587785, 0.425325,-0.688191
				DATA  0.688191, 0.587785,-0.425325,  -0.147621, 0.716567,-0.681718
				DATA -0.309017, 0.500000,-0.809017,   0.000000, 0.525731,-0.850651
				DATA -0.525731, 0.000000,-0.850651,  -0.442863, 0.238856,-0.864188
				DATA -0.295242, 0.000000,-0.955423,  -0.162460, 0.262866,-0.951056
				DATA  0.000000, 0.000000,-1.000000,   0.295242, 0.000000,-0.955423
				DATA  0.162460, 0.262866,-0.951056,  -0.442863,-0.238856,-0.864188
				DATA -0.309017,-0.500000,-0.809017,  -0.162460,-0.262866,-0.951056
				DATA  0.000000,-0.850651,-0.525731,  -0.147621,-0.716567,-0.681718
				DATA  0.147621,-0.716567,-0.681718,   0.000000,-0.525731,-0.850651
				DATA  0.309017,-0.500000,-0.809017,   0.442863,-0.238856,-0.864188
				DATA  0.162460,-0.262866,-0.951056,   0.238856,-0.864188,-0.442863
				DATA  0.500000,-0.809017,-0.309017,   0.425325,-0.688191,-0.587785
				DATA  0.716567,-0.681718,-0.147621,   0.688191,-0.587785,-0.425325
				DATA  0.587785,-0.425325,-0.688191,   0.000000,-0.955423,-0.295242
				DATA  0.000000,-1.000000, 0.000000,   0.262866,-0.951056,-0.162460
				DATA  0.000000,-0.850651, 0.525731,   0.000000,-0.955423, 0.295242
				DATA  0.238856,-0.864188, 0.442863,   0.262866,-0.951056, 0.162460
				DATA  0.500000,-0.809017, 0.309017,   0.716567,-0.681718, 0.147621
				DATA  0.525731,-0.850651, 0.000000,  -0.238856,-0.864188,-0.442863
				DATA -0.500000,-0.809017,-0.309017,  -0.262866,-0.951056,-0.162460
				DATA -0.850651,-0.525731, 0.000000,  -0.716567,-0.681718,-0.147621
				DATA -0.716567,-0.681718, 0.147621,  -0.525731,-0.850651, 0.000000
				DATA -0.500000,-0.809017, 0.309017,  -0.238856,-0.864188, 0.442863
				DATA -0.262866,-0.951056, 0.162460,  -0.864188,-0.442863, 0.238856
				DATA -0.809017,-0.309017, 0.500000,  -0.688191,-0.587785, 0.425325
				DATA -0.681718,-0.147621, 0.716567,  -0.442863,-0.238856, 0.864188
				DATA -0.587785,-0.425325, 0.688191,  -0.309017,-0.500000, 0.809017
				DATA -0.147621,-0.716567, 0.681718,  -0.425325,-0.688191, 0.587785
				DATA -0.162460,-0.262866, 0.951056,   0.442863,-0.238856, 0.864188
				DATA  0.162460,-0.262866, 0.951056,   0.309017,-0.500000, 0.809017
				DATA  0.147621,-0.716567, 0.681718,   0.000000,-0.525731, 0.850651
				DATA  0.425325,-0.688191, 0.587785,   0.587785,-0.425325, 0.688191
				DATA  0.688191,-0.587785, 0.425325,  -0.955423, 0.295242, 0.000000
				DATA -0.951056, 0.162460, 0.262866,  -1.000000, 0.000000, 0.000000
				DATA -0.850651, 0.000000, 0.525731,  -0.955423,-0.295242, 0.000000
				DATA -0.951056,-0.162460, 0.262866,  -0.864188, 0.442863,-0.238856
				DATA -0.951056, 0.162460,-0.262866,  -0.809017, 0.309017,-0.500000
				DATA -0.864188,-0.442863,-0.238856,  -0.951056,-0.162460,-0.262866
				DATA -0.809017,-0.309017,-0.500000,  -0.681718, 0.147621,-0.716567
				DATA -0.681718,-0.147621,-0.716567,  -0.850651, 0.000000,-0.525731
				DATA -0.688191, 0.587785,-0.425325,  -0.587785, 0.425325,-0.688191
				DATA -0.425325, 0.688191,-0.587785,  -0.425325,-0.688191,-0.587785
				DATA -0.587785,-0.425325,-0.688191,  -0.688191,-0.587785,-0.425325


''
'' Load MD2 file into script structure
''
PRIVATE FUNCTION loadMD2(filename AS STRING, BYREF frameId AS INTEGER, BYREF script AS script_t) AS INTEGER
	DIM AS    md2_t     head
	DIM AS   vec3_t     scale, translate
	DIM AS UINTEGER     ff = FREEFILE, anyInt = ANY, i = ANY
	DIM AS    UBYTE PTR xPtr = ANY
	DIM AS   vec3_t PTR v = ANY

	' Open file
	ff = FREEFILE
	IF (LEN(filename) = 0) THEN RETURN err_no_specified
	IF (ff = 0) THEN RETURN err_no_handle
	IF (LEN(DIR(filename, &h23)) = 0) THEN RETURN err_no_found
	IF OPEN(filename FOR BINARY AS #ff) THEN RETURN err_general_io

	' Test header
	GET #ff,, head
	IF (head.ident <> &h32504449) THEN CLOSE #ff: RETURN err_not_MD2
	IF (head.version <> 8) THEN CLOSE #ff: RETURN err_not_version_8
	IF ((frameId >= head.numFrames) OR (frameId < 0)) THEN RETURN err_frame_out_of_range

	' Get basic info from header
	script.numVertices = head.numVertices
	script.numGLPackets = head.numGlCmds
	script.skinWidth = head.skinWidth
	script.skinHeight = head.skinHeight

	' Get skins list
	script.numSkins = head.numSkins
	xPtr = CPTR(UBYTE PTR, script.skinFile)
	GET #ff, head.ofsSkins + 1, *xPtr, 64 * script.numSkins

	' Get specified frame
	SEEK #ff, head.ofsFrames + 1 + (frameId * head.frameSize)
	GET #ff,, scale               ' get Scale vector
	GET #ff,, translate           ' get Offset vector
	SEEK #ff, SEEK(ff) + 16       ' skip Frame Name
	xPtr = CPTR(UBYTE PTR, @anyInt)
	FOR i = 0 TO head.numVertices - 1
		GET #ff, , anyInt
		v = @script.vertex[i].org
		v->x = xPtr[0] * scale.x + translate.x
		v->y = xPtr[1] * scale.y + translate.y
		v->z = xPtr[2] * scale.z + translate.z
	NEXT i

	' Get GLCommands (contains strips and UVs)
	GET #ff, head.ofsGlCmds + 1, *script.GLPackets, script.numGLPackets
	CLOSE #ff

	FUNCTION = 0
END FUNCTION

''
'' Load MDX file into script structure
''
PRIVATE FUNCTION loadMDX(filename AS STRING, BYREF frameId AS INTEGER, BYREF script AS script_t) AS INTEGER
	DIM AS    mdx_t     head
	DIM AS   vec3_t     scale, translate
	DIM AS UINTEGER     ff = FREEFILE, anyInt = ANY, i = ANY, pkSrc = 0, pkDst = 0, pkSize = ANY
	DIM AS    UBYTE PTR xPtr = ANY
	DIM AS   vec3_t PTR v = ANY

	' Open file
	ff = FREEFILE
	IF (LEN(filename) = 0) THEN RETURN err_no_specified
	IF (ff = 0) THEN RETURN err_no_handle
	IF (LEN(DIR(filename, &h23)) = 0) THEN RETURN err_no_found
	IF OPEN(filename FOR BINARY AS #ff) THEN RETURN err_general_io

	' Test header
	GET #ff,, head
	IF (head.ident <> &h58504449) THEN CLOSE #ff: RETURN err_not_MDX
	IF (head.version <> 4) THEN CLOSE #ff: RETURN err_not_version_4
	IF ((frameId >= head.numFrames) OR (frameId < 0)) THEN RETURN err_frame_out_of_range

	' Get basic info from header
	script.numVertices = head.numVertices
	script.numGLPackets = head.numGlCmds
	script.skinWidth = head.skinWidth
	script.skinHeight = head.skinHeight

	' Get skins list
	script.numSkins = head.numSkins
	xPtr = CPTR(UBYTE PTR, script.skinFile)
	GET #ff, head.ofsSkins + 1, *xPtr, 64 * script.numSkins

	' Get specified frame
	SEEK #ff, head.ofsFrames + 1 + (frameId * head.frameSize)
	GET #ff,, scale               ' get Scale vector
	GET #ff,, translate           ' get Offset vector
	SEEK #ff, SEEK(ff) + 16       ' skip Frame Name
	xPtr = CPTR(UBYTE PTR, @anyInt)
	FOR i = 0 TO head.numVertices - 1
		GET #ff, , anyInt
		v = @script.vertex[i].org
		v->x = xPtr[0] * scale.x + translate.x
		v->y = xPtr[1] * scale.y + translate.y
		v->z = xPtr[2] * scale.z + translate.z
	NEXT i

	' Get GLCommands (contains strips, UVs and submodel number)
	GET #ff, head.ofsGlCmds + 1, *script.GLPackets, script.numGLPackets
	CLOSE #ff

	' Remove SubObjectIndex field (1 entry, right after the vertex count)
	pkSrc = 0: pkDst = 0
	DO
		script.GLPackets[pkDst] = script.GLPackets[pkSrc]
		pkSize = ABS(script.GLPackets[pkSrc])
		IF (pkSize) THEN
			pkSrc += 2 ' skip count & SubObjectIndex
			pkDst += 1 ' write to next
			FOR i = 0 TO pkSize - 1
				script.GLPackets[pkDst]     = script.GLPackets[pkSrc]
				script.GLPackets[pkDst + 1] = script.GLPackets[pkSrc + 1]
				script.GLPackets[pkDst + 2] = script.GLPackets[pkSrc + 2]
				pkSrc += 3
				pkDst += 3
			NEXT i
		'ELSE
			'EXIT DO
		END IF
	LOOP WHILE pkSize ' UNTIL (pkSrc >= script.numGLPackets)
	script.numGLPackets = pkDst

	FUNCTION = 0
END FUNCTION

''
'' Save to script file
''
PRIVATE FUNCTION saveScript(filename AS STRING, script AS script_t) AS INTEGER
	DIM AS INTEGER      pkOfst = 0, pkSize = ANY
	DIM AS INTEGER      pkS = ANY, pkT = ANY, pkV = ANY
	DIM AS UINTEGER     i = ANY
	DIM AS   vec3_t PTR v = ANY

	PRINT "Writing script file..."
	OPEN filename FOR OUTPUT AS #1

	' Write skins
	IF (script.numSkins < 2) THEN
		PRINT #1, "mdlSkin " + script.skinFile[0] + " " + STR(script.skinWidth) + " " + STR(script.skinHeight)
	ELSE
		PRINT #1, "mdlSkinSize " + STR(script.skinWidth) + " " + STR(script.skinHeight)
		FOR i = 0 TO script.numSkins - 1
			PRINT #1, "mdlSkinFile " + script.skinFile[64 * i]
		NEXT i
	END IF
	PRINT #1, ""

	' Convert GlCommands, and use the coordinates found in the vertex list
	PRINT #1, "mdlScale 1 1 1"
	PRINT #1, "mdlOffset 0 0 0"
	DO
		PRINT #1, ""
		' packet size
		SELECT CASE script.GLPackets[pkOfst]
		CASE IS < 0
			pkSize = -script.GLPackets[pkOfst]
			PRINT #1, "glCommand fan // " + STR(pkSize)
		CASE IS > 0
			pkSize = script.GLPackets[pkOfst]
			PRINT #1, "glCommand strip // " + STR(pkSize)
		CASE ELSE
			EXIT DO
		END SELECT
		pkOfst += 1 ' skipping count entry
		' get each vertex
		FOR i = 0 TO pkSize - 1
			pkS = CPTR(SINGLE PTR, script.GLPackets)[pkOfst] * script.skinwidth
			pkT = CPTR(SINGLE PTR, script.GLPackets)[pkOfst + 1] * script.skinheight
			pkV = script.GLPackets[pkOfst + 2]
			v = @script.vertex[pkV].org
			PRINT #1, v->x, v->y, v->z, pkS, pkT
			pkOfst += 3
		NEXT i
	LOOP UNTIL (pkOfst >= script.numGLPackets)
	CLOSE #1

	RETURN 0
END FUNCTION

''
'' CLEAN STRING (REMOVE EXCESS SPACES, DISCARD COMMENTS)
''
PRIVATE SUB strClean(BYREF msg AS STRING)
	DIM AS STRING    outStr = ""
	DIM AS UBYTE     prev = 32
	DIM AS UBYTE PTR xPtr = ANY

	IF INSTR(msg, "//") THEN msg = LEFT(msg, INSTR(msg,"//") - 1)
	IF LEN(msg) THEN
		xPtr = CPTR(UBYTE PTR, STRPTR(msg))
		FOR i AS UINTEGER = 0 TO LEN(msg) - 1
			IF (xPtr[i] = 9) THEN xPtr[i] = 32
			IF ((prev <> 32) OR (xPtr[i] <> 32)) THEN outStr = outStr + CHR(xPtr[i])
			prev = xPtr[i]
		NEXT i

		msg = RTRIM(outStr)
	END IF
END SUB

''
'' RETURN UNIT VECTOR
''
PRIVATE FUNCTION vecNormalize(BYREF v AS vec3_t) AS vec3_t
	DIM AS SINGLE length = SQR(v.x * v.x + v.y * v.y + v.z * v.z)
	RETURN TYPE<vec3_t>(v.x / length, v.y / length, v.z / length)
END FUNCTION

''
'' COMPUTE AREA OF TRIANGLE
''
PRIVATE FUNCTION trisSurface(BYREF p1 AS vec3_t PTR, BYREF p2 AS vec3_t PTR, BYREF p3 AS vec3_t PTR) AS SINGLE
	RETURN SQR( _
						((p2->x * p1->y) - (p3->x * p1->y) - (p1->x * p2->y) + (p3->x * p2->y) + (p1->x * p3->y) - (p2->x * p3->y))^2 + _
						((p2->x * p1->z) - (p3->x * p1->z) - (p1->x * p2->z) + (p3->x * p2->z) + (p1->x * p3->z) - (p2->x * p3->z))^2 + _
						((p2->y * p1->z) - (p3->y * p1->z) - (p1->y * p2->z) + (p3->y * p2->z) + (p1->y * p3->z) - (p2->y * p3->z))^2) * 0.50
END FUNCTION

''
'' COMPUTE NORMAL OF TRIANGLE SURFACE, RETURN UNIT VECTOR
''
PRIVATE FUNCTION trisNormal(BYREF p1 AS vec3_t PTR, BYREF p2 AS vec3_t PTR, BYREF p3 AS vec3_t PTR) AS vec3_t
	DIM AS vec3_t v, w, t

	' get delta of two edges
	v.x = p1->x - p2->x
	v.y = p1->y - p2->y
	v.z = p1->z - p2->z

	w.x = p3->x - p1->x
	w.y = p3->y - p1->y
	w.z = p3->z - p1->z

	' cross product
	t.x = v.y * w.z - v.z * w.y
	t.y = v.z * w.x - v.x * w.z
	t.z = v.x * w.y - v.y * w.x

	' done unit vector
	RETURN vecNormalize(t)
END FUNCTION

''
'' COPY A 3D VECTOR
''
PRIVATE SUB vecCopy(BYREF lhs AS vec3_t, BYREF rhs AS vec3_t)
	lhs.x = rhs.x
	lhs.y = rhs.y
	lhs.z = rhs.z
END SUB

''
'' SUBTRACT TWO 3D VECTORS, RETURN RESULT
''
PRIVATE FUNCTION vecSubtract(BYREF lhs AS vec3_t, BYREF rhs AS vec3_t) AS vec3_t
	RETURN TYPE<vec3_t>(lhs.x - rhs.x, lhs.y - rhs.y, lhs.z - rhs.z)
END FUNCTION

''
'' LOAD AND PROCESS SCRIPT FILE
''
PRIVATE FUNCTION loadScript(BYREF filename AS STRING, BYREF target AS script_t) AS INTEGER
	DIM AS vec3_t        mdlScale, mdlOffset
	DIM AS vec3_t        tempVec3
	DIM AS vec3_t        normal
	DIM AS vec3_t        mdlRotOrg
	DIM AS vec2_t        tempVec2
	DIM AS SINGLE        mdlRotDeg(2), mdlRotSin(2), mdlRotCos(2)
	DIM AS SINGLE        nearDist = ANY, dist = ANY, fTmp = ANY
	DIM AS SINGLE        a = ANY, b = ANY
	DIM AS INTEGER       ff = ANY, i = ANY, j = ANY, glMode = ANY, ofs = ANY
	DIM AS INTEGER       numLine = 0, numWords = ANY
	DIM AS INTEGER       remFlag = 0
	DIM AS SHORT         index
	DIM AS STRING        msg
	DIM AS UBYTE     PTR bPtr = ANY
	DIM AS ZSTRING   PTR sPtr = ANY
	DIM AS INTEGER   PTR glStream = ANY, glHead = ANY
	DIM AS scriptT_t PTR tri = ANY
	DIM AS vec3_t    PTR v1 = ANY, v2 = ANY, v3 = ANY

	#MACRO readNumber (x)
	DO WHILE (ofs < LEN(msg))
		IF (LEN(sPtr[ofs])) THEN
			x = VAL(sPtr[ofs])
			ofs += LEN(sPtr[ofs]) + 1
			EXIT DO
		ELSE
			ofs += 1
		END IF
	LOOP
	#ENDMACRO

	#MACRO readString (x)
	DO WHILE (ofs < LEN(msg))
		IF (LEN(sPtr[ofs])) THEN
			x = CALLOCATE(SIZEOF(TYPEOF(*x)) * (LEN(sPtr[ofs]) + 1))
			*x = sPtr[ofs]
			ofs += LEN(sPtr[ofs]) + 1
			EXIT DO
		ELSE
			ofs += 1
		END IF
	LOOP
	#ENDMACRO

	' assume something went wrong
	FUNCTION = -1

  ' Open file
	ff = FREEFILE
	IF (LEN(filename) = 0) THEN RETURN err_no_specified
	IF (ff = 0) THEN RETURN err_no_handle
	IF (LEN(DIR(filename, &h23)) = 0) THEN RETURN err_no_found
	IF OPEN(filename FOR INPUT AS #ff) THEN RETURN err_general_io

	' set defaults, just to be sure
	mdlScale = TYPE<vec3_t>(1.00, 1.00, 1.00)
	mdlOffset = TYPE<vec3_t>(0.00, 0.00, 0.00)

	DO
		' read one line from the file
		numLine += 1
		LINE INPUT #ff, msg
		strClean(msg) ' make string safe
		IF (LEN(msg) = 0) THEN CONTINUE DO

		' split words
		msg += " "
		bPtr = CPTR(UBYTE PTR, STRPTR(msg))
		FOR i = 0 TO LEN(msg) - 1
			IF (bPtr[i] = 32) THEN bPtr[i] = 0
		NEXT i

		' parse instructions
		sPtr = CPTR(ZSTRING PTR, STRPTR(msg))
		ofs = LEN(sPtr[0]) + 1
		IF remFlag THEN
			IF (UCASE(sPTR[0]) = "**COMMENT_STOP**") THEN remFlag xor= 1
			CONTINUE DO
		END IF

		SELECT CASE UCASE(sPtr[0])
			CASE "**COMMENT_START**"
				remFlag xor= 1

			CASE "FILEOUTPUT"
				readString(target.outputFile)

			CASE "MDLSCALE"
				readNumber(mdlScale.X)
				readNumber(mdlScale.Y)
				readNumber(mdlScale.Z)

			CASE "MDLOFFSET"
				readNumber(mdlOffset.X)
				readNumber(mdlOffset.Y)
				readNumber(mdlOffset.Z)

			CASE "MDLROTATE"
				readNumber(mdlRotOrg.X)  ' rotation origin
				readNumber(mdlRotOrg.Y)
				readNumber(mdlRotOrg.Z)
				readNumber(mdlRotDeg(0)) ' degrees around X
				readNumber(mdlRotDeg(1)) ' degrees around Y
				readNumber(mdlRotDeg(2)) ' degrees around Z
				FOR i AS INTEGER = 0 TO 2
					mdlRotDeg(i) = mdlRotDeg(i) * 0.01745329252
					mdlRotSin(i) = SIN(mdlRotDeg(i))
					mdlRotCos(i) = COS(mdlRotDeg(i))
				NEXT i

			'' this should be deprecated ''
			CASE "MDLSKIN"
				DO WHILE (ofs < LEN(msg))
					IF (LEN(sPtr[ofs])) THEN
						target.skinFile[target.numSkins * 64] = sPtr[ofs]
						target.numSkins += 1
						ofs += LEN(sPtr[ofs]) + 1
						EXIT DO
					ELSE
						ofs += 1
					END IF
				LOOP
				readNumber(target.skinWidth)
				readNumber(target.skinHeight)

			'' new skin declaration style ''
			CASE "MDLSKINSIZE"
				readNumber(target.skinWidth)
				readNumber(target.skinHeight)
			
			'' new skin declaration style ''
			CASE "MDLSKINFILE"
				DO WHILE (ofs < LEN(msg))
					IF (LEN(sPtr[ofs])) THEN
						target.skinFile[target.numSkins * 64] = sPtr[ofs]
						target.numSkins += 1
						ofs += LEN(sPtr[ofs]) + 1
						EXIT DO
					ELSE
						ofs += 1
					END IF
				LOOP

			CASE "GLCOMMAND"
				glHead = @target.glPackets[target.numGlPackets] ' point to vertex count
				target.numGLPackets += 1 ' add one packet for vertex count
				glStream = @target.glPackets[target.numGlPackets] ' point to next packet
				IF (UCASE(sPtr[ofs]) = "STRIP") THEN glMode = 1 ELSE glMode = -1

			CASE ELSE
				' test for numerals
				numWords = 0
				FOR i = 0 TO LEN(msg) - 1
					SELECT CASE bPtr[i]
						CASE 44
							bPtr[i] = 46 ' convert "," to "." and continue
						CASE 0
							numWords += 1 ' count string-terminators
						CASE 48 TO 57, 43, 45, 46
							' ignore numbers minus, plus and decimal
						CASE ELSE
							PRINT "LINE ";numLine;": expected vertices!"
							CLOSE #ff
							EXIT FUNCTION
					END SELECT
				NEXT i
				IF (numWords <> 5) THEN
					PRINT "LINE ";numLine;": expected 5 values (X Y Z U V)"
					CLOSE #ff
					EXIT FUNCTION
				END IF

				ofs = 0

				' read vertex origin
				readNumber(tempVec3.x) 
				readNumber(tempVec3.y) 
				readNumber(tempVec3.z)

				' rotate vertex coordinates
				IF (mdlRotDeg(0)) THEN ' around X
					a = tempVec3.y - mdlRotOrg.y
					b = tempVec3.z - mdlRotOrg.z
					tempVec3.y = (a * mdlRotCos(0)) - (b * mdlRotSin(0))
					tempVec3.z = (a * mdlRotSin(0)) + (b * mdlRotCos(0))
				END IF
				IF (mdlRotDeg(1)) THEN ' around Y
					a = tempVec3.x - mdlRotOrg.x
					b = tempVec3.z - mdlRotOrg.z
					tempVec3.x = (a * mdlRotCos(1)) + (b * mdlRotSin(1))
					tempVec3.z = -(a * mdlRotSin(1)) + (b * mdlRotCos(1))
				END IF
				IF (mdlRotDeg(2)) THEN ' around Z
					a = tempVec3.x - mdlRotOrg.x
					b = tempVec3.y - mdlRotOrg.y
					tempVec3.x = (a * mdlRotCos(2)) - (b * mdlRotSin(2))
					tempVec3.y = (a * mdlRotSin(2)) + (b * mdlRotCos(2))
				END IF

				' offset vertex coordinates
				tempVec3.x = (tempVec3.x * mdlScale.x) + mdlOffset.x
				tempVec3.y = (tempVec3.y * mdlScale.y) + mdlOffset.y
				tempVec3.z = (tempVec3.z * mdlScale.z) + mdlOffset.z

				' read texturecoordinates
				readNumber(tempVec2.x)
				readNumber(tempVec2.y)

				' get texCoord index or add to list
				index = -1
				FOR i = 0 TO target.numTexCoords - 1
					IF (tempVec2 = target.texCoord[i]) THEN
						index = i
						EXIT FOR
					END IF
				NEXT i
				IF (index = -1) THEN
					index = target.numTexCoords
					target.texCoord[target.numTexCoords] = tempVec2
					target.numtexCoords += 1
				END IF

				' temporary representation of UV (normalize later)
				glStream[0] = index

				' get vertex index or add to list
				index = -1
				FOR i = 0 TO target.numVertices - 1
					IF (tempVec3 = target.vertex[i].org) THEN
						index = i
						EXIT FOR
					END IF
				NEXT i
				IF (index = -1) THEN
					index = target.numVertices
					target.vertex[target.numVertices].org = tempVec3
					target.numVertices += 1
				END IF

				' add vertex to glPacket
				glStream[2] = CINT(index)

				' move ahead, update vertex count in glHead
				target.numGlPackets += 3
				glStream = @glStream[3]
				*glHead += glMode
		END SELECT
	LOOP UNTIL EOF(ff)
	CLOSE #ff

	' add an end-of-stream GLPacket
	target.numGLPackets += 1

	' rebuild triangles
	index = 0
	DO
		glStream = @target.glPackets[index] ' index of first vertex
		SELECT CASE target.glPackets[index]
			CASE IS < 0 ' FAN (0 1 2 - 0 2 3 - 0 3 4...)
				FOR i = 1 TO ABS(target.glPackets[index]) - 2
					ofs = i * 3
					tri = @target.triangle[target.numTris]
					' vertices index
					tri->p1 = glStream[3]
					tri->p2 = glStream[3 + ofs]
					tri->p3 = glStream[6 + ofs]
					' texture coordinates index
					tri->t1 = glStream[1]
					tri->t2 = glStream[1 + ofs]
					tri->t3 = glStream[4 + ofs]
					' add triangle
					target.numTris += 1
				NEXT i
			CASE IS > 0 ' STRIP (0 1 2 - 1 2 3 - 2 3 4...)
				FOR i = 0 TO target.glPackets[index] - 3
					ofs = i * 3
					tri = @target.triangle[target.numTris]
					' vertices index
					tri->p1 = glStream[3 + ofs]
					tri->t1 = glStream[1 + ofs]
					IF (i AND 1) THEN
						tri->p2 = glStream[9 + ofs]
						tri->p3 = glStream[6 + ofs]
						tri->t2 = glStream[7 + ofs]
						tri->t3 = glStream[4 + ofs]
					ELSE
						tri->p2 = glStream[6 + ofs]
						tri->p3 = glStream[9 + ofs]
						tri->t2 = glStream[4 + ofs]
						tri->t3 = glStream[7 + ofs]
					END IF
					' add triangle
					target.numTris += 1
				NEXT i
			CASE ELSE
				EXIT DO
		END SELECT
		index += ABS(target.glPackets[index]) * 3 + 1
	LOOP

	' compute normal and area of triangles
	FOR i = 0 TO target.numTris - 1
		tri = @target.triangle[i]
		v1 = @target.vertex[tri->p1].org
		v2 = @target.vertex[tri->p2].org
		v3 = @target.vertex[tri->p3].org

		tri->n = trisNormal(v1, v2, v3)
		tri->s = trisSurface(v1, v2, v3)
	NEXT i

	' normalize coordinates in glStream
	index = 0
	DO
		glStream = @target.glPackets[index + 1] ' index of first vertex
		IF (target.glPackets[index] = 0) THEN EXIT DO
		FOR i = 0 TO ABS(target.glPackets[index]) - 1
			ofs = glStream[0]
			*CPTR(SINGLE PTR, @glStream[0]) = target.texCoord[ofs].x / target.skinWidth
			*CPTR(SINGLE PTR, @glStream[1]) = target.texCoord[ofs].y / target.skinHeight
			glStream = @glStream[3]
		NEXT i
		index += ABS(target.glPackets[index]) * 3 + 1
	LOOP

	' get min-max
	vecCopy(target.min, target.vertex[0].org)
	vecCopy(target.max, target.vertex[0].org)
	FOR i = 1 TO target.numVertices - 1
		tempVec3 = target.vertex[i].org
		IF (target.min.X > tempVec3.x) THEN target.min.X = tempVec3.x
		IF (target.min.Y > tempVec3.y) THEN target.min.Y = tempVec3.y
		IF (target.min.Z > tempVec3.z) THEN target.min.Z = tempVec3.z
		IF (target.max.X < tempVec3.x) THEN target.max.X = tempVec3.x
		IF (target.max.Y < tempVec3.y) THEN target.max.Y = tempVec3.y
		IF (target.max.Z < tempVec3.z) THEN target.max.Z = tempVec3.z
	NEXT i
	target.box = vecSubtract(target.max, target.min)

	' get vertices normal
	FOR i = 0 TO target.numVertices - 1

		' add normal of neighbor triangles
		tempVec3 = TYPE<vec3_t>(0, 0, 0)
		FOR j = 0 TO target.numTris - 1
			tri = @target.triangle[j]
			IF ((tri->p1 = i) OR (tri->p2 = i) OR (tri->p3 = i)) THEN
				tempVec3.x += tri->n.x * tri->s
				tempVec3.y += tri->n.y * tri->s
				tempVec3.z += tri->n.z * tri->s
			END IF
		NEXT j
		tempVec3 = vecNormalize(tempVec3)

		' find nearest normal in the look up table
		RESTORE normalList
		nearDist = 32767
		FOR j = 0 TO 165
			READ normal.x, normal.y, normal.z
			dist = SQR( (tempVec3.x - normal.x)^2 + (tempVec3.y - normal.y)^2 + (tempVec3.z - normal.z)^2 )
			IF (dist < nearDist) THEN
				nearDist = dist
				target.vertex[i].nIndex = j
			END IF
		NEXT j
	NEXT i

	' done
	RETURN 0
END FUNCTION

''
'' Save processed script to MD2
''
PRIVATE FUNCTION saveMD2(BYREF filename AS STRING, BYREF script AS script_t) AS INTEGER
	DIM AS INTEGER      ff = FREEFILE
	DIM AS md2_t        md2
	DIM AS STRING * 16  frameName
	DIM AS UINTEGER     i = ANY, j = ANY
	DIM AS vec3_t PTR   fPtr

	' ready header
	WITH md2
		' signature
		.ident = &h32504449
		.version = 8

		' skin size
		.skinWidth = script.skinWidth
		.skinHeight = script.skinHeight

		' stats
		.numSkins = script.numSkins         ' data size: 64 * num_skins
		.numVertices = script.numVertices   ' data size: no associated block 
		.numST = script.numTexCoords        ' data size: 4 * num_st
		.numGlCmds = script.numGlPackets    ' data size: variable (length in integers)
		.numFrames = 1                      ' data size: num_frames * (12 + 12 + 16 + 4 * num_vertices)

		' stats: triangle count
		.numTris = script.numTris     ' data size: 12 * numTris

		' stats: frame size
		.frameSize = 12 + 12 + 16 + 4 * .numVertices

		' offsets
		.ofsSkins  = SIZEOF(TYPEOF(md2))
		.ofsSt     = .ofsSkins    + 64 * .numSkins
		.ofsTris   = .ofsSt       + 4  * .numSt
		.ofsFrames = .ofsTris     + 12 * .numTris
		.ofsGlCmds = .ofsFrames   + .numFrames * .frameSize
		.ofsEnd    = .ofsGlCmds   + 4 * .numGlCmds

		' summary
		PRINT "         ID: "; MKL(.ident)
		PRINT "    version:"; .version
		PRINT "  skinWidth:"; .skinWidth
		PRINT " skinHeight:"; .skinHeight
		PRINT "  frameSize:"; .frameSize
		PRINT
		PRINT "   numSkins:"; .numSkins
		PRINT "numVertices:"; .numVertices
		PRINT "      numSt:"; .numSt
		PRINT "    numTris:"; .numTris
		PRINT "  numGlCmds:"; .numGlCmds
		PRINT "  numFrames:"; .numFrames
		PRINT
		PRINT "   ofsSkins:"; .ofsSkins
		PRINT "      ofsSt:"; .ofsSt
		PRINT "    ofsTris:"; .ofsTris
		PRINT "  ofsFrames:"; .ofsFrames
		PRINT "  ofsGlCmds:"; .ofsGlCmds
		PRINT "     ofsEnd:"; .ofsEnd
	END WITH

  IF (LEN(DIR(filename, &h23))) then kill filename
	OPEN filename FOR BINARY AS #ff
	PUT #ff,, md2 ' header

	' ofs_skins
	PUT #ff,, *CPTR(UBYTE PTR, script.skinFile), (script.numSkins * 64)

	' ofs_st
	FOR i = 0 TO md2.numSt - 1
		PUT #ff,, CSHORT(script.texCoord[i].x)
		PUT #ff,, CSHORT(script.texCoord[i].y)
	NEXT i

	' ofs_tris (software renderer & GTKRadiant)
	FOR i = 0 TO script.numTris - 1
		' vertices
		PUT #ff,, CSHORT(script.triangle[i].p1)
		PUT #ff,, CSHORT(script.triangle[i].p2)
		PUT #ff,, CSHORT(script.triangle[i].p3)
		' uv
		PUT #ff,, CSHORT(script.triangle[i].t1)
		PUT #ff,, CSHORT(script.triangle[i].t2)
		PUT #ff,, CSHORT(script.triangle[i].t3)
	NEXT i

	' ofs_frames
	PUT #ff,, CSNG(script.box.X / 255)
	PUT #ff,, CSNG(script.box.Y / 255)
	PUT #ff,, CSNG(script.box.Z / 255)
	PUT #ff,, CSNG(script.min.X)
	PUT #ff,, CSNG(script.min.Y)
	PUT #ff,, CSNG(script.min.Z)
	frameName = "default"
	PUT #ff,,frameName
	FOR i = 0 TO script.numVertices - 1
		' packed coordinates
		fPtr = @script.vertex[i].org
		PUT #ff,, CUBYTE(((fPtr->x - script.min.X) / script.box.X) * 255)
		PUT #ff,, CUBYTE(((fPtr->y - script.min.Y) / script.box.Y) * 255)
		PUT #ff,, CUBYTE(((fPtr->z - script.min.Z) / script.box.Z) * 255)
		' normal
		PUT #ff,, script.vertex[i].nIndex
	NEXT i

	' ofs_opengl
	PUT #ff,, *script.glPackets, script.numGlPackets

	CLOSE #ff
  
	RETURN 0
END FUNCTION
