#include "StdAfx.h"
#include "3DSTranslator.h"

#include "MBaseObject.h"
#include "MEditableMesh.h"
#include "MMeshShape.h"
#include "MSystemManager.h"
#include "MLight.h"

//#include <io.h>
#include <stdio.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <assert.h>

#if USE_LIB3DS
#include <set>
#include <list>

#include "lib3ds/light.h"
#include "lib3ds/material.h"
#include "lib3ds/camera.h"
#include "lib3ds/mesh.h"
#endif

#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

// 3DS tags and related defines
#define MAIN3DS			0x4D4D
#define EDIT3DS			0x3D3D  // this is the start of the editor config

#if USE_LIB3DS
#else
// 3DS tags and related defines
#define MAIN3DS			0x4D4D
#define VER3DS				0x0002
#define EDIT3DS			0x3D3D  // this is the start of the editor config
#define EDIT_CONFIG		0x3d3e
#define EDIT_CONFIG1		0x0100
#define EDIT_OBJECT		0x4000
#define OBJ_TRIMESH		0x4100
#define TRI_VERTEXL		0x4110
#define TRANS_MATRIX		0x4160
#define TRI_FACEL1		0x4120
#define FACE_SMOOTHING	0x4150
#define KF_DATA			0xb000
#define KF_HEAD			0xb00a
#define KF_SEG_INFO		0xb008
#define KF_CURTIME		0xb009
#define OBJECT_NODE_TAG	0xb002
#define NODE_ID			0xb030
#define NODE_HDR			0xb010
#define PIVOT				0xb013
#define POS_TRACK_TAG	0xb020
#define ROT_TRACK_TAG	0xb021
#define SCL_TRACK_TAG   0xb022
#define NO_PARENT			0xffff

#define BLOCK_SIZE	   4096
#endif


//---------------------------------------------------------------------------------

M3DSTranslator::M3DSTranslator() {
#if USE_LIB3DS
	file = NULL;
#endif
}

M3DSTranslator::~M3DSTranslator() {
}

// Class related
MTranslatorPtr M3DSTranslator::createNew() const {
   M3DSTranslator   *NewTranslator;

   NewTranslator = new M3DSTranslator;

   return NewTranslator;
}

bool M3DSTranslator::canImportFile(MStr Filename) {
   FILE		*hFile;
   WORD     FileID;

   Filename.Replace('/', '\\');
  
   hFile = fopen(Filename,"rb");

   if (!hFile)
      return false;

   fread(&FileID,sizeof(WORD),1,hFile);

   fclose(hFile);

   if (FileID != MAIN3DS && FileID != EDIT3DS)
      return false;

   return true;
}

#if USE_LIB3DS

void M3DSTranslator::
BuildLight(Lib3dsLight *light)
{
	MLight *newLight = new MLight();
	char sbuf[64];
	
	sprintf(sbuf, "%f %f %f", light->color[0], light->color[1], light->color[2]);
	newLight->setParamByName("Col", sbuf);
	sprintf(sbuf, "%f", light->multiplier);
	// newLight->setParamByName("SpCol", "0.5 0.5 0.5"); // No specular color in 3ds files
	newLight->setParamByName("Mul", sbuf);
	
	sprintf(sbuf, "%f %f %f", light->position[0], light->position[1], light->position[2]);
    newLight->setParamByName("Tsl", sbuf);
	
	if (light->spot_light)
	{
		// What do we do with spot lights?
		// For now, it gets turned into an omni light
	}
	
  newLight->setName(MStr(light->name) + "Shape");

	MSceneObjectPtr sceneObj = new MSceneObject(newLight);
	sceneObj->setName(light->name);
  mScene->addObject(sceneObj);
}

void M3DSTranslator::
BuildCamera(Lib3dsCamera *cam)
{
	// Not implemented
}

void M3DSTranslator::
BuildTexture(Lib3dsMaterial *mat)
{
	char sbuf[64];
	char matname[16];
	MMaterialPtr material = new MMaterial;

	strcpy(matname, mat->name);
	// CleanupName(matname, mat->name, (sizeof(mat->name) / sizeof(mat->name[0])) - 1);

#if _DEBUG
	sprintf(sbuf, "Create texture, '%s'\n", matname);
	OutputDebugString(sbuf);
#endif

	sprintf(sbuf, "%f %f %f", mat->ambient[0], mat->ambient[1], mat->ambient[2]);
	material->setParamByName("Amb", sbuf);
	sprintf(sbuf, "%f %f %f", mat->diffuse[0], mat->diffuse[1], mat->diffuse[2]);
	material->setParamByName("Diff", sbuf);
	sprintf(sbuf, "%f %f %f", mat->specular[0], mat->specular[1], mat->specular[2]);
	material->setParamByName("Spec", sbuf);
	material->setParamByName("Emm", "0.0 0.0 0.0");	// No emission in 3ds materials
	sprintf(sbuf, "%f", mat->shininess);
	material->setParamByName("Shin", sbuf);
	material->setParamByName("Opac", "1.0");	// Ignore opacity maps

	// See if there is an image map for this material
	if (mat->texture1_map.percent > 0.0)
	{
		material->setParamByName("DiffFile", mat->texture1_map.name);
		// Doesn't work: material->setDiffuseMapFilename(mat->texture.map.name);	
	}

    material->setName(matname);
    mScene->addObject(material);
}

class stringCmp : public std::string {
public:
	bool operator ()(const std::string &x, const std::string &y)
	{
		return strcmp(x.c_str(), y.c_str()) < 0;
	}
};

class CMaterialName {
protected:
	std::string name;
public:
	CMaterialName(char *str) : name(str) { }
	const char *c_str() {
		return name.c_str();
	}
	bool operator <(const CMaterialName &y)
	{
		return strcmp(name.c_str(), y.name.c_str()) < 0;
	}
	friend bool operator <(const CMaterialName &x, const CMaterialName &y)
	{
		return strcmp(x.name.c_str(), y.name.c_str()) < 0;
	}
};

// Count material names in a mesh.  Needed for LIB3DS, as it stores
// the material name in every single face.
// class CNameSet : public std::set< CMaterialName > {
class CNameSet : public std::set< std::string, stringCmp > {
};
static CNameSet *GetMatNames(Lib3dsMesh *mesh)
{
	CNameSet *nameSet = new CNameSet;
	int i;
	for (i=0;i<mesh->faces;i++) {
		// Lib3dsFace &face = mesh->faceL[i];
		// nameSet->insert(CMaterialName(mesh->faceL[i].material));
		nameSet->insert(mesh->faceL[i].material);
	}
	
	return nameSet;
}
class CNameCountPair : public std::pair< const std::string *, int> {
};

static int
GetMaterialCount(Lib3dsMaterial *materials)
{
	Lib3dsMaterial *tmat;
	int i;

	for (tmat=materials,i=0;tmat!=NULL;tmat=tmat->next,i++)
		/* Intentionally left blank */
		;

	return i;
}

// Translate the structures used for reading 3DS into those used by
// the rest of the renderer
bool M3DSTranslator::
convert_to_meshes()
{
#if _DEBUG
	char sbuf[256];
#endif
	char meshname[64];
	MSceneObjectPtr *hMeshObjs;
	long num_vert, num_uvs, num_tris;
	int i, j, cnt;
	MVector3 *verts, *uvs;
	MTriangle *faces;
	std::list<MEditableMesh *> MeshHandles;

	// Step through all the mesh objects, making them individually
	unsigned long objcnt;
	Lib3dsMesh *tmesh = file->meshes;
	for (objcnt=0;tmesh != NULL;tmesh=tmesh->next)
		objcnt++;
	tmesh = file->meshes;
	hMeshObjs = new MSceneObjectPtr[objcnt];

	for (i=0,cnt=0,tmesh=file->meshes;tmesh!=NULL;i++,tmesh=tmesh->next,i++) {
		Lib3dsMesh *meshobj = tmesh;

		int nverts, nfaces;
		nverts = meshobj->points;
		nfaces = meshobj->faces;
		if ((nverts < 3) || (nfaces < 1)) {
			// No vertices or faces associated with this mesh object.
			// Just skip to the next one.
			continue;
		}

		strcpy(meshname, meshobj->name);
		// CleanupName(meshname, meshobj->name, (sizeof(meshobj->name) / sizeof(meshobj->name[0])) - 1);

#if _DEBUG
		sprintf(sbuf, "Process object '%s'\n", meshname);
		OutputDebugString(sbuf);
#endif

		int bound_faces = 0;
		int nmats = 0;

		// Get the set of material names used in this object (lib3ds doesn't
		// do it for us, like 3dsftk does).
		CNameSet *nameSet = GetMatNames(meshobj);
		nmats = nameSet->size();
		bound_faces = nfaces;
		// Figure out how many faces reference each material
		int nmatCounts;
		CNameCountPair *matCounts = new CNameCountPair[nmats];
		CNameSet::iterator nsItr;
		for (nsItr=nameSet->begin(),i=0;nsItr!=nameSet->end();nsItr++,i++) {
			matCounts[i].first = &(*nsItr);
			matCounts[i].second = 0;
		}

		for (nmatCounts=0;nmatCounts<nmats;nmatCounts++) {
		}

#if 0
		// TBD: split meshes by material
		if ((nmats > 1) && (bound_faces == nfaces)) {
			// We need to split this object into multiple pieces.  We do this if all
			// the faces are accounted for in the list of bindings
			std::list<MTreeObjectNodePtr> MeshSubHandles;
			MEditableMesh *sub_object = new MEditableMesh();
#if _DEBUG
			OutputDebugString("Split by material\n");
#endif
			sub_object->setName(meshname + "Mesh");
			for (j=0;j<nmats;j++) {
				num_vert = 3 * meshobj->matarray[j].nfaces;
				num_tris = meshobj->matarray[j].nfaces;

				if (num_vert <= 0 || num_tris <= 0) {
					// No faces or vertices associated with this material.
					// Therefore we don't need to create a mesh.
					continue;
				}

				verts = new MVector3[num_vert];
				if (meshobj->ntextverts == meshobj->nvertices) {
					num_uvs = num_vert;
					uvs = new MVector3[num_vert];
				} else {
					num_uvs = 0;
					uvs = NULL;
				}
				faces = new MTriangle[num_tris];
				// Make the list of triangles, faces, and u/v maps
				int k, u, v, f;
				for (k=0,u=0,v=0,f=0;k<num_tris;k++) {
					faces[k].m_Vertices[0] = f++;
					faces[k].m_Vertices[1] = f++;
					faces[k].m_Vertices[2] = f++;
					int p0 = meshobj->facearray[meshobj->matarray[j].faceindex[k]].v1;
					int p1 = meshobj->facearray[meshobj->matarray[j].faceindex[k]].v2;
					int p2 = meshobj->facearray[meshobj->matarray[j].faceindex[k]].v3;
					verts[v].x = meshobj->vertexarray[p0].x;
					verts[v].y = meshobj->vertexarray[p0].y;
					verts[v].z = meshobj->vertexarray[p0].z;
					v++;
					verts[v].x = meshobj->vertexarray[p1].x;
					verts[v].y = meshobj->vertexarray[p1].y;
					verts[v].z = meshobj->vertexarray[p1].z;
					v++;
					verts[v].x = meshobj->vertexarray[p2].x;
					verts[v].y = meshobj->vertexarray[p2].y;
					verts[v].z = meshobj->vertexarray[p2].z;
					v++;
					if (uvs != NULL) {
						uvs[u].x = meshobj->textarray[p0].u;
						uvs[u].y = 1.0 - meshobj->textarray[p0].v;
						uvs[u].z = 0.0;
						u++;
						uvs[u].x = meshobj->textarray[p1].u;
						uvs[u].y = 1.0 - meshobj->textarray[p1].v;
						uvs[u].z = 0.0;
						u++;
						uvs[u].x = meshobj->textarray[p1].u;
						uvs[u].y = 1.0 - meshobj->textarray[p1].v;
						uvs[u].z = 0.0;
						u++;
					}
				}
				sub_object->addVertsAndTriangles(verts, faces, num_vert, num_tris);

				if (uvs != NULL) {
					// Build a mesh to hold u/v coordinates
					MEditableMesh *uvMesh = new MEditableMesh();
					uvMesh->addVertsAndTriangles(uvs, faces, num_vert, num_tris);
					sub_object->setTextureMesh(MEditableMeshPtr(uvMesh));
				}

				delete[] faces;
				delete[] verts;
				if (uvs != NULL) {
					delete[] uvs;
				}
				// Assign the material: meshobj->matarray[j].name
				char matname[64];
				strcpy(matname, matarray[j].name);
				// CleanupName(matname, meshobj->matarray[j].name, (sizeof(meshobj->matarray[j].name) / sizeof(char3ds)) - 1);
				MMaterialPtr Material = AZTEC_CAST(MMaterial, mScene->getObjectList()->findObject(matname));
				sub_object->setTextureMaterial(Material);
			
				// Ready for the next legitimate object

				MeshSubHandles.push_back(mScene->addObject(sub_object));
			}
			if (MeshSubHandles.size() > 1) {
				// Since we have more than one thing in this mesh, we add them all to
				// an empty transform node.
				MSceneObjectPtr EmptyObject = new MSceneObject;
				MTreeObjectNodePtr TransNode;
				
				EmptyObject->setName(meshname);				
				TransNode = mScene->addObject(EmptyObject);
				hMeshObjs[cnt] = TransNode;
				
				MBaseObjectTreePtr objList = mScene->getObjectList();
				// Go through the current Selected objects, and set the parent to the new transform node
				{
					MBaseObjectPtr Obj;
					MTreeObjectNodePtr ObjNode;
					for (std::list<MTreeObjectNodePtr>::iterator mj=MeshSubHandles.begin();mj!=MeshSubHandles.end();mj++) {
						// Add this mesh object to the transform node
						ObjNode = *mj;
						if (mj != NULL) {
              MSceneObjectPtr child = AZTEC_CAST(MSceneObject, ObjNode->getObject();
              child->setParent(EmptyObject);
						} else {
							OutputDebugString("Null child node\n");
						}
					}
					cnt++;
				}
			} else if (MeshSubHandles.size() == 1) {
				hMeshObjs[cnt++] = *(MeshSubHandles.begin());
			}
			MeshSubHandles.clear();
		} else {
#endif
			// Process this object as a single unit - either because all faces
			// share the same material, or because there isn't a material for
			// every face (including no material at all).
			MEditableMesh *Mesh = new MEditableMesh();
			Mesh->setName(MStr(meshname) + "Mesh");
			MMeshShapePtr meshShape = new MMeshShape(Mesh);
			meshShape->setName(MStr(meshname) + "Shape");

			num_tris = nfaces;
			num_vert = nverts;
			verts = new MVector3[num_vert];

			// Handle u/v vertices.  Can't handle if # of u/v coordinates
			// doesn't match.
			if (meshobj->texels == nverts) {
				num_uvs = nverts;
				uvs = new MVector3[num_vert];
			} else {
				num_uvs = 0;
				uvs = NULL;
			}

			for (j=0;j<nverts;j++) {
				verts[j].x = meshobj->pointL[j].pos[0];
				verts[j].y = meshobj->pointL[j].pos[1];
				verts[j].z = meshobj->pointL[j].pos[2];

				if (uvs != NULL) {
					uvs[j].x = meshobj->texelL[j][0];
					uvs[j].y = 1.0f - meshobj->texelL[j][1];
					uvs[j].z = 0.0f;
				}
			}
			// Make the list of triangles
			faces = new MTriangle[num_tris];
			for (j=0;j<num_tris;j++) {
				faces[j].setVertex(0, meshobj->faceL[j].points[0]);
				faces[j].setVertex(1, meshobj->faceL[j].points[1]);
				faces[j].setVertex(2, meshobj->faceL[j].points[2]);
			}

			Mesh->addVertsAndTriangles(verts, faces, num_vert, num_tris);

			if (uvs != NULL) {
				// Build a mesh to hold u/v coordinates
				MEditableMesh *uvMesh = new MEditableMesh();
				uvMesh->addVertsAndTriangles(uvs, faces, num_vert, num_tris);
				Mesh->setTextureMesh(MEditableMeshPtr(uvMesh));
			}

			delete[] faces;
			delete[] verts;
			if (uvs != NULL) {
				delete[] uvs;
			}

			MSceneObjectPtr sceneObj = new MSceneObject(meshShape);
			sceneObj->setName(meshname);
			if (mScene->addObject(sceneObj) != NULL) {
				hMeshObjs[cnt] = sceneObj;
			} else {
				hMeshObjs[cnt] = NULL;
			}

#if 1
			// TBD: Assign the material to the mesh
			if (nmats > 0) {
				char matname[64];
				const char *mname = (*nameSet->begin()).c_str();
				strcpy(matname, mname);
				// CleanupName(matname, mname, (sizeof(meshobj->matarray[0].name) / sizeof(char3ds)) - 1);
				MMaterialPtr Material = AZTEC_CAST(MMaterial, mScene->getObjectList()->findObject(matname));
				sceneObj->setTextureMaterial(Material);
			}
#endif
			
			// Ready for the next legitimate object
			if (hMeshObjs[cnt] != NULL) {
				cnt++;
			} else {
#if _DEBUG
				OutputDebugString("Bad mesh: ");
				OutputDebugString(meshname);
				OutputDebugString("\n");
#endif
			}
#if 0
		}
#endif

		// Free up any memory used for this object
		delete[] matCounts;
		delete nameSet;

		meshobj = meshobj->next;
	}

	if (cnt > 1) {
		// Since we have more than one thing in this mesh, we add them all to a CSG
		MSceneObjectPtr EmptyObject = new MSceneObject;
		MTreeObjectNodePtr TransNode;
		
		EmptyObject->setName("Mesh");				
		TransNode = mScene->addObject(EmptyObject);
		
		MBaseObjectTreePtr objList = mScene->getObjectList();
		for (i=0;i<cnt;i++) {
			// Add this mesh object to the CSG
      MSceneObjectPtr child = AZTEC_CAST(MSceneObject, hMeshObjs[i]);

      if (child != NULL) {
        child->setParent(EmptyObject);
      }
		}
	}

	if (cnt > 1) {
		// Select the composite object
	} else if (cnt == 1) {
		// Select the object
	}

	delete[] hMeshObjs;

	return true;
}      


#else
//int M3DSTranslator::ParseTriMeshChunk(int hFile, MMesh *Mesh, int &BytesRead, int Length)
//{
//}

int M3DSTranslator::ParseObjectChunk(int hFile, MScenePtr Scene, int &BytesRead, int Length) {
   MSceneObjectPtr sceneObj = new MSceneObject();
   MEditableMeshPtr Mesh;
   MMeshShapePtr meshShape;
   bool DoneVerts, DoneTris;
   int BytesLeft;

   DoneVerts = false;
   DoneTris = false;

   Mesh = new MEditableMesh;
   meshShape = new MMeshShape(Mesh);

   sceneObj->setShapeObject(meshShape);

   Scene->addObject(Mesh);
   Scene->addObject(meshShape);

   {
      MSystemManagerPtr SysMan;
      SysMan = Aztec::getSystemManager();

      SysMan->logOutput("ParseObjectChunk()");
   }
   sceneObj->setName(Scene->generateUniqueName("meshObj"));
   Mesh->setName(Scene->generateUniqueName("meshObjMesh"));
   meshShape->setName(Scene->generateUniqueName("meshObjShape"));

//   if (TagID == EDIT_OBJECT)
   {
     char     TempName[BLOCK_SIZE];
     int      Num;
     
     Num = 0;
     do {
       read(hFile, &TempName[Num], 1);
       Num ++;
       BytesRead++;
     } while (TempName[Num-1] != '\x0');
     
     sceneObj->setName(Scene->generateUniqueName(TempName));
     Mesh->setName(Scene->generateUniqueName(MStr(TempName) + "Mesh"));
     meshShape->setName(Scene->generateUniqueName(MStr(TempName) + "Shape"));
     
   }

   while (BytesRead < Length) {
      WORD     TagID;
      DWORD    Size;

      read(hFile, &TagID, sizeof(WORD));
      BytesRead += sizeof(WORD);
   
      if (eof(hFile)) {
         goto Done;
      }

      read(hFile, &Size, sizeof(DWORD));
      BytesRead += sizeof(DWORD);

      BytesLeft = Size-6;

      if (TagID == OBJ_TRIMESH)
         continue;

      if (TagID == TRI_VERTEXL) {
         MVector3 *Verts;
         WORD        NumVerts, i;

         read(hFile, &NumVerts, sizeof(WORD));
         BytesRead += sizeof(WORD);
         BytesLeft -= sizeof(WORD);


         Verts = new MVector3[NumVerts];

         for (i = 0; i< NumVerts; i++)
         {
            read(hFile, &Verts[i].x, sizeof(float));
            read(hFile, &Verts[i].y, sizeof(float));
            read(hFile, &Verts[i].z, sizeof(float));

            BytesRead += sizeof(float)*3;
            BytesLeft -= sizeof(float)*3;
         }

         lseek(hFile, BytesLeft, SEEK_CUR);

         Mesh->addVertexArray(Verts, NumVerts);

         delete[] Verts;
         DoneVerts = true;

      } else if (TagID == TRI_FACEL1) {  
         MTriangle      *Tris;
         WORD           NumTris, i, j;

         read(hFile, &NumTris, sizeof(WORD));
         BytesRead += sizeof(WORD);
         BytesLeft -= sizeof(WORD);

         Tris = new MTriangle[NumTris];

         for (i = 0; i < NumTris; i++)
         {
            for (j=0; j<4; j++)
            {
               if (j != 4)
               {
                  WORD     Vert;
                  read(hFile, &Vert, sizeof(WORD));
                  BytesRead += sizeof(WORD);
                  BytesLeft -= sizeof(WORD);
                  Tris[i].m_Vertices[j] = Vert;
               }
            }
         }

         lseek(hFile, BytesLeft, SEEK_CUR);

         Mesh->addTriangleArray(Tris, NumTris);

         delete[] Tris;
         DoneTris = true;
      } else {
         lseek(hFile, BytesLeft, SEEK_CUR);
         BytesRead += BytesLeft;
      }

   }

Done:
   if (DoneVerts && DoneTris) {
     Mesh->calculateNormals();
     Scene->addObject(sceneObj);
   }

   Mesh = NULL;

   return Length;
}

int M3DSTranslator::Parse3DSChunk(int hFile, MScenePtr Scene, int &BytesRead) {
   WORD     TagID;
   DWORD    Size;
   int      Result, BytesLeft;

   Result = 0;

   if (eof(hFile))
      return -1;

   read(hFile, &TagID, sizeof(WORD));
   BytesRead += sizeof(WORD);
   
   if (eof(hFile))
      return -1;

   read(hFile, &Size, sizeof(DWORD));
   BytesRead += sizeof(DWORD);

   BytesLeft = Size - 6;

   {
      MSystemManagerPtr SysMan;
      SysMan = Aztec::getSystemManager();

      SysMan->logOutput("Parsing Chunk %x size %i", TagID, Size);
   }

   if (TagID == EDIT_OBJECT) {
      int Res, ChunkBytes;

      ChunkBytes = 0;

      Res = ParseObjectChunk(hFile, Scene, ChunkBytes, Size);

      if (Res < 0)
         return Res;

      BytesRead += Res;
      BytesLeft -= Res;
   } else if (TagID == MAIN3DS || TagID == EDIT3DS) {
      while (BytesLeft > 0)
      {
         int Res;
         Res = Parse3DSChunk(hFile, Scene, BytesRead);

         if (Res < 0)
            return Res;
         BytesRead += Res;
         BytesLeft -= Res;
      }

      Result = Size;
      goto Done;
   } else {
      // Skip all other chunks
      while (BytesLeft > 0)
      {
         lseek(hFile, BytesLeft, SEEK_CUR);

         BytesRead += BytesLeft;
         BytesLeft -= BytesLeft;

      }
      Result = Size;
      goto Done;
   }


Done:
   return Result;
}
#endif

bool M3DSTranslator::importFile(MStr Filename, MScenePtr Scene) {
   MSystemManagerPtr SysMan;
   bool result = true;

   SysMan = Aztec::getSystemManager();
   assert(SysMan != NULL);

   Filename.Replace('/', '\\');

#if USE_LIB3DS
	mScene = Scene;
	file = lib3ds_file_load(Filename.c_str());
	if (file == NULL)
		return false;

	lib3ds_file_eval(file, 0.0f);

	// Walk through all materials
	Lib3dsMaterial *material = file->materials;
	while (material != NULL) {
		BuildTexture(material);
		material = material->next;
	}

	// Collect up all cameras
	Lib3dsCamera *camera = file->cameras;
	while (camera != NULL) {
		BuildCamera(camera);
		camera = camera->next;
	}

	// Get all omnidirectional (point) lights
	Lib3dsLight *light = file->lights;
	while (light != NULL) {
		BuildLight(light);
		light = light->next;
	}

	// Pull all the meshes and convert them to Aztec form.
	result = convert_to_meshes();
	if (!result) {
		char sbuf[256];
		sprintf(sbuf, "No valid meshes found in '%s'", Filename.c_str());
#ifdef _DEBUG
		OutputDebugString(sbuf);
#endif
	}

	lib3ds_file_free(file);
#else
   int               hFile, Result, BytesRead;
   WORD              FileID;

   hFile = open((LPCTSTR)Filename,O_BINARY | O_RDONLY);

   if (hFile == -1)
      return false;


   read(hFile, &FileID, sizeof(WORD));
   if (FileID != MAIN3DS && FileID != EDIT3DS) {
      close(hFile);
      return false;
   }

   lseek(hFile, 0, SEEK_SET);
   BytesRead = 0;
   Result = Parse3DSChunk(hFile, Scene, BytesRead);

   close(hFile);

   result = true;
#endif

   return result;
}

bool M3DSTranslator::exportFile(MStr Filename, MScenePtr Scene) {
   return false;
}


