#include "Stdafx.h"

#include "PrimParticle.h"
#include "ParticleActionParam.h"
#include "ParticleScript.h"

#include "MBaseObject.h"
#include "MShapeObject.h"
#include "MMesh.h"

#include "MScene.h"

#include <stdio.h>
#include <assert.h>

#include <MSystemManager.h>

#include <params/MParameterFactory.h>

#include "scripting/MJavaScriptSupport.h"

#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

#ifndef WIN32
#define OutputDebugString(x) printf(x)
#endif

extern Aztec::MSystemManager *primSysMan;

MPrimitiveParticle::MPrimitiveParticle()
{
	addParameter(m_InMeshParam = Aztec::MParameterFactory::createObject("inMesh", "inMesh", "Control Mesh"));

	maxParticlesPtr = Aztec::MParameterFactory::createInteger("maxParticles", "maxParticles", "Max Particles");
	stepRatePtr = Aztec::MParameterFactory::createFloat("stepRate", "stepRate", "Calculation Rate");
	loopAnimPtr = Aztec::MParameterFactory::createBoolean("loopAnim", "loopAnim", "Looping Animation");

	addParameter(maxParticlesPtr);
	addParameter(stepRatePtr);
	addParameter(loopAnimPtr);

	setParamByName("MeshName", "");
	setParamByName("maxParticles", "1000");
	setParamByName("stepRate", "0.05");		// Default to 1/20'th of a second between calculations
	setParamByName("loopAnim", "false");

	// Add a parameter that reflects the particle actions to Aztec (primarily for making
	// scripting easier).
	Aztec::MParameterObjectPtr actionParam = new Aztec::ParticleActionParam(this);
    addParameter(actionParam);
    actionParam->setVisible(false);

	isPlaying = false;

	lastEvaluatedTime = -1.0;

	setFlag(OBJECTFLAG_NOCOMPONENTS);

#if 1
	// Set up a default list of actions
	int i;
	actionType action;

	action.action = actionType::velocityDomain;
	float cylarray[] = {(float)PDCylinder, 0.01f, 0.0, 0.35f, 0.01f, 0.0, 0.37f, 0.021f, 0.019f};
	action.m_params.clear();
	for (i=0;i<(sizeof(cylarray)/sizeof(cylarray[0]));i++) {
		action.m_params.push_back(cylarray[i]);
	}
	m_ActionList.push_back(action);

	action.action = actionType::colorDomain;
	float colorarray[] = {1.0, PDLine, 0.8f, 0.9f, 1.0, 1.0, 1.0, 1.0};
	action.m_params.clear();
	for (i=0;i<(sizeof(colorarray)/sizeof(colorarray[0]));i++) {
		action.m_params.push_back(colorarray[i]);
	}
	m_ActionList.push_back(action);

	action.action = actionType::size;
	float sizearray[] = {PDPoint, 1.5f, 1.5f, 1.5f};
	action.m_params.clear();
	for (i=0;i<(sizeof(sizearray)/sizeof(sizearray[0]));i++) {
		action.m_params.push_back(sizearray[i]);
	}
	m_ActionList.push_back(action);

	action.action = actionType::source;
	float sourcearray[] = {100, PDLine, 0.0, 0.0, 0.401f, 0.0, 0.0, 0.405f};
	action.m_params.clear();
	for (i=0;i<(sizeof(colorarray)/sizeof(colorarray[0]));i++) {
		action.m_params.push_back(sourcearray[i]);
	}
	m_ActionList.push_back(action);

	action.action = actionType::gravity;
	float gravityarray[] = {0.0, 0.0, -0.01f};
	action.m_params.clear();
	for (i=0;i<(sizeof(gravityarray)/sizeof(gravityarray[0]));i++) {
		action.m_params.push_back(gravityarray[i]);
	}
	m_ActionList.push_back(action);

	action.action = actionType::bounce;
	float bouncearray[] = {-0.05f, 0.35f, 0, PDDisc, 0, 0, 0,  0, 0, 1,  5, 0};
	action.m_params.clear();
	for (i=0;i<(sizeof(bouncearray)/sizeof(bouncearray[0]));i++) {
		action.m_params.push_back(bouncearray[i]);
	}
	m_ActionList.push_back(action);

	action.action = actionType::sink;
	float sinkarray[] = {false, PDPlane, 0, 0, -3, 0, 0, 1};
	action.m_params.clear();
	for (i=0;i<(sizeof(sinkarray)/sizeof(sinkarray[0]));i++) {
		action.m_params.push_back(sinkarray[i]);
	}
	m_ActionList.push_back(action);
#endif
}

MPrimitiveParticle::~MPrimitiveParticle()
{
	maxParticlesPtr = NULL;
	stepRatePtr = NULL;
	loopAnimPtr = NULL;
}

const char *MPrimitiveParticle::actionEnumToString(MPrimitiveParticle::actionType::actionEnum a)
{
	switch (a) {
	case actionType::color: return "color";
	case actionType::colorDomain: return "colorDomain";
	case actionType::size: return "size";
	case actionType::sizeDomain: return "sizeDomain";
	case actionType::startingAge: return "startingAge";
	case actionType::timeStep: return "timeStep";
	case actionType::velocity: return "velocity";
	case actionType::velocityDomain: return "velocityDomain";
	case actionType::vertexB: return "vertexB";
	case actionType::vertexBTracks: return "vertexBTracks";
	case actionType::avoid: return "avoid";
	case actionType::bounce: return "bounce";
	case actionType::copyVertexB: return "copyVertexB";
	case actionType::damping: return "damping";
	case actionType::explosion: return "explosion";
	case actionType::follow: return "follow";
	case actionType::gravitate: return "gravitate";
	case actionType::gravity: return "gravity";
	case actionType::jet: return "jet";
	case actionType::killOld: return "killOld";
	case actionType::matchVelocity: return "matchVelocity";
	case actionType::move: return "move";
	case actionType::orbit: return "orbit";
	case actionType::randomAccel: return "randomAccel";
	case actionType::randomDisplace: return "randomDisplace";
	case actionType::randomVelocity: return "randomVelocity";
	case actionType::restore: return "restore";
	case actionType::sink: return "sink";
	case actionType::sinkVelocity: return "sinkVelocity";
	case actionType::source: return "source";
	case actionType::speedLimit: return "speedLimit";
	case actionType::targetColor: return "targetColor";
	case actionType::targetSize: return "targetSize";
	case actionType::targetVelocity: return "targetVelocity";
	case actionType::vertex: return "vertex";
	case actionType::vortex: return "vortex";
	default:
		return "unknownAction";
	}
}

float MPrimitiveParticle::actionEnumToFloat(MPrimitiveParticle::actionType::actionEnum a)
{
	return (float)a;
}

MPrimitiveParticle::actionType::actionEnum MPrimitiveParticle::floatToActionEnum(float f)
{
	return (actionType::actionEnum)((int)f);
}

Aztec::MBaseObjectPtr MPrimitiveParticle::createNew() {
	MPrimitiveParticle *NewObj;
	
	NewObj = new MPrimitiveParticle;

	NewObj->m_ParamList->setFromList(getParamList());

  return NewObj;
}


void MPrimitiveParticle::updateKey(int Time, DWORD Channel, bool CreateKey)
{
}

void MPrimitiveParticle::onParameterChange(Aztec::MParameterObjectPtr param)
{
}

Aztec::MMeshPtr MPrimitiveParticle::convertToMesh()
{
	return NULL;
}

void MPrimitiveParticle::evaluateColor(const particleParamType &entries)
{
	float alpha = entries[1];
	float red = entries[2];
	float green = entries[3];
	float blue = entries[4];

	pColor(red, green, blue, alpha);
}

void MPrimitiveParticle::evaluateColorD(const particleParamType &entries)
{
	float alpha;
	float r1, g1, b1;
	float r2, g2, b2;
	float r3, g3, b3;

	int i = 0;	// Index into parameters

	PDomainEnum domain = PDomainEnum((int)entries[i++]);

	switch (domain) {
	case PDPoint:
		alpha = entries[i++];
		r1 = entries[i++]; g1 = entries[i++]; b1 = entries[i++];
		r2 = g2 = b2 = 0.0;
		r3 = g3 = b3 = 0.0;
		break;
	case PDLine:
		alpha = entries[i++];
		r1 = entries[i++]; g1 = entries[i++]; b1 = entries[i++];
		r2 = entries[i++]; g2 = entries[i++]; b2 = entries[i++];
		r3 = g3 = b3 = 0.0;
		break;
	case PDTriangle:
		alpha = entries[i++];
		r1 = entries[i++]; g1 = entries[i++]; b1 = entries[i++];
		r2 = entries[i++]; g2 = entries[i++]; b2 = entries[i++];
		r3 = entries[i++]; g3 = entries[i++]; b3 = entries[i++];
		break;
	case PDDisc:
		break;
	default:
		break;
	}

	pColorD(alpha, domain, r1, g1, b1, r2, g2, b2, r3, g3, b3);
}
//
// Order of parameters by domain type.  The value of entries[startIndex] defines
// the type of the domain.  After the domain type, 
//    Domain types      Value (papi.h) Parameters                       Description
//    -----------------------------------------------------------------------------------------
//       PDPoint          0             (x, y, z)                        Center point
//       PDLine           1             (x0, y0, z0), (x1, y1, z1)       End points of line
//       PDTriangle       2             (x0, y0, z0), ..., (x2, y2, z2)  Vertices of triangle
//       PDPlane          3             (x0, y0, z0), (nx, ny, nz)       Point on plane, Normal
//       PDBox            4             (x0, y0, z0), (x1, y1, z1)       Opposite corners of box
//       PDSphere         5             (x0, y0, z0), r0, r1             Center, inner and outer radii
//       PDCylinder       6
//       PDCone           7
//       PDBlob           8
//       PDDisc           9
//       PDRectangle     10
void MPrimitiveParticle::getDomainEntries(PDomainEnum domain, int startIndex,
										  const particleParamType &entries, float *a)
{
	int i = startIndex;

	// Set to 0.0f to match Particle API defaults.
	for (int j=0;j<9;j++) {
		a[j] = 0.0f;
	}

	// Endpoint/Center of the domain (all domains start with a point).
	a[0] = entries[i++]; a[1] = entries[i++]; a[2] = entries[i++];

	// Figure out the rest of the parameters based on the domain
	switch (domain) {
	case PDPoint:
		break;
	case PDLine:
		// Other point in the line
		a[3] = entries[i++]; a[4] = entries[i++]; a[5] = entries[i++];
		break;
	case PDTriangle:
		// Second and third points in the triangle
		a[3] = entries[i++]; a[4] = entries[i++]; a[5] = entries[i++];
		a[6] = entries[i++]; a[7] = entries[i++]; a[8] = entries[i++];
		break;
	case PDPlane:
		// Normal to the plane
		a[3] = entries[i++]; a[4] = entries[i++]; a[5] = entries[i++];
		break;
	case PDBox:
		// Other corner of the box
		a[3] = entries[i++]; a[4] = entries[i++]; a[5] = entries[i++];
		break;
	case PDSphere:
		// Inner and outer radii
		a[3] = entries[i++]; a[4] = entries[i++];
		break;
	case PDCylinder:
		// Other endpoint of cylinder
		a[3] = entries[i++]; a[4] = entries[i++]; a[5] = entries[i++];
		// Inner and outer radii
		a[6] = entries[i++]; a[7] = entries[i++];
		break;
	case PDCone:
		// Base of cone (previous point was the apex)
		a[3] = entries[i++]; a[4] = entries[i++]; a[5] = entries[i++];
		// Inner and outer radii
		a[6] = entries[i++]; a[7] = entries[i++];
		break;
	case PDBlob:
		// Standard deviation of the blob
		a[3] = entries[i++];
		break;
	case PDDisc:
		// Normal to the plane of the disc
		a[3] = entries[i++]; a[4] = entries[i++]; a[5] = entries[i++];
		// Inner and outer radii
		a[6] = entries[i++]; a[7] = entries[i++];
		break;
	case PDRectangle:
		// Basis vectors for the rectangle (relative to the origin point)
		a[3] = entries[i++]; a[4] = entries[i++]; a[5] = entries[i++];
		a[6] = entries[i++]; a[7] = entries[i++]; a[8] = entries[i++];
		break;
	default:
		break;
	}
}

void MPrimitiveParticle::evaluateKillOld(const particleParamType &entries)
{
	// Generic domain parameters.
	float age_limit = entries[0];
	bool kill_less_than = (entries[1] == 0.0f ? false : true);

	pKillOld(age_limit, kill_less_than);
}

void MPrimitiveParticle::evaluateVelocity(const particleParamType &entries)
{
	// Generic domain parameters.
	float x = entries[0], y = entries[1], z = entries[2];

	pVelocity(x, y, z);
}

void MPrimitiveParticle::evaluateVelocityDomain(const particleParamType &entries)
{
	// Generic domain parameters.
	float a[9];

	PDomainEnum domain = PDomainEnum((int)entries[0]);

	getDomainEntries(domain, 1, entries, a);

	pVelocityD(domain, a[0], a[1], a[2], a[3], a[4], a[5], a[6], a[7], a[8]);
}

void MPrimitiveParticle::evaluateSize(const particleParamType &entries)
{
	// Set the size of the particle using the x value only.
	// TBD: use the other size entries.
	float size = entries[1];

	pSize(size);
}

void MPrimitiveParticle::evaluateSizeD(const particleParamType &entries)
{
	// Generic domain parameters.
	float a[9];

	PDomainEnum domain = PDomainEnum((int)entries[0]);

	getDomainEntries(domain, 1, entries, a);

	pSizeD(domain, a[0], a[1], a[2], a[3], a[4], a[5], a[6], a[7], a[8]);
}

void MPrimitiveParticle::evaluateAvoid(const particleParamType &entries)
{
	float magnitude = entries[0];
	float epsilon = entries[1];
	float look_ahead = entries[2];

	// Generic domain parameters.
	float a[9];

	PDomainEnum domain = PDomainEnum((int)entries[3]);

	getDomainEntries(domain, 4, entries, a);

	pAvoid(magnitude, epsilon, look_ahead, domain, a[0], a[1], a[2], a[3], a[4], a[5], a[6], a[7], a[8]);
}

void MPrimitiveParticle::evaluateBounce(const particleParamType &entries)
{
	float friction = entries[0];
	float resilience = entries[1];
	float cutoff = entries[2];

	// Generic domain parameters.
	float a[9];

	PDomainEnum domain = PDomainEnum((int)entries[3]);

	getDomainEntries(domain, 4, entries, a);

	pBounce(friction, resilience, cutoff, domain, a[0], a[1], a[2], a[3], a[4], a[5], a[6], a[7], a[8]);
}

void MPrimitiveParticle::evaluateSource(const particleParamType &entries)
{
	float rate = entries[0];

	// Generic domain parameters.
	float a[9];

	PDomainEnum domain = PDomainEnum((int)entries[1]);

	getDomainEntries(domain, 2, entries, a);

	pSource(rate, domain, a[0], a[1], a[2], a[3], a[4], a[5], a[6], a[7], a[8]);
}

void MPrimitiveParticle::evaluateGravity(const particleParamType &entries)
{
	// Gravity is a direction, so just pull off the first three values
	// (skipping the domain, which is unused).
	float g0 = entries[0];
	float g1 = entries[1];
	float g2 = entries[2];

	pGravity(g0, g1, g2);
}

void MPrimitiveParticle::evaluateSink(const particleParamType &entries)
{
	bool kill_inside = (entries[0] == 0.0f ? false : true);
	// Generic domain parameters.
	float a[9];

	PDomainEnum domain = PDomainEnum((int)entries[1]);

	getDomainEntries(domain, 2, entries, a);

	pSink(kill_inside, domain, a[0], a[1], a[2], a[3], a[4], a[5], a[6], a[7], a[8]);
}

void MPrimitiveParticle::evaluateVortex(const particleParamType &entries)
{
	int paramCnt = entries.size();
	if (paramCnt < 6) {
		// Not enough parameters for this action.  TBD: report error
		return;
	}

	float cx, cy, cz, ax, ay, az;
	float mag = 1.0f, eps = P_EPS, rad = P_MAXFLOAT;

	cx = entries[0]; cy = entries[1]; cz = entries[2];
	ax = entries[3]; ay = entries[4]; az = entries[5];
	if (paramCnt > 6) {
		mag = entries[6];
		if (paramCnt > 7) {
			eps = entries[7];
			if (paramCnt > 8) {
				rad = entries[8];
			}
		}
	}

	pVortex(cx, cy, cz, ax, ay, az, mag, eps, rad);
}

void MPrimitiveParticle::evaluateActions(Aztec::MSceneObjectPtr sceneObj, Aztec::MSceneViewFlags ViewFlags)
{
	actionListType::iterator actionItr;
	
	for (actionItr=m_ActionList.begin();actionItr!=m_ActionList.end();actionItr++) {
		const actionType &action = *actionItr;

		switch (action.action) {
		case actionType::avoid: evaluateAvoid(action.m_params); break;
		case actionType::bounce: evaluateBounce(action.m_params); break;
		case actionType::color: evaluateColor(action.m_params); break;
		case actionType::colorDomain: evaluateColorD(action.m_params); break;
		case actionType::gravity: evaluateGravity(action.m_params); break;
		case actionType::sink: evaluateSink(action.m_params); break;
		case actionType::size: evaluateSize(action.m_params); break;
		case actionType::sizeDomain: evaluateSizeD(action.m_params); break;
		case actionType::source: evaluateSource(action.m_params); break;
		case actionType::velocity: evaluateVelocity(action.m_params); break;
		case actionType::velocityDomain: evaluateVelocityDomain(action.m_params); break;
#if 0
		case actionType::startingAge: return "startingAge";
		case actionType::timeStep: return "timeStep";
		case actionType::vertexB: return "vertexB";
		case actionType::vertexBTracks: return "vertexBTracks";
		case actionType::copyVertexB: return "copyVertexB";
		case actionType::damping: return "damping";
		case actionType::explosion: return "explosion";
		case actionType::follow: return "follow";
		case actionType::gravitate: return "gravitate";
		case actionType::jet: return "jet";
#endif
		case actionType::killOld: evaluateKillOld(action.m_params);
#if 0
		case actionType::matchVelocity: return "matchVelocity";
		case actionType::move: return "move";
		case actionType::orbit: return "orbit";
		case actionType::randomAccel: return "randomAccel";
		case actionType::randomDisplace: return "randomDisplace";
		case actionType::randomVelocity: return "randomVelocity";
		case actionType::restore: return "restore";
		case actionType::sinkVelocity: return "sinkVelocity";
		case actionType::speedLimit: return "speedLimit";
		case actionType::targetColor: return "targetColor";
		case actionType::targetSize: return "targetSize";
		case actionType::targetVelocity: return "targetVelocity";
		case actionType::vertex: return "vertex";
#endif
		case actionType::vortex: evaluateVortex(action.m_params);
		default:
			OutputDebugString("Can't handle action: ");
			OutputDebugString(actionEnumToString(action.action));
			OutputDebugString("\n");
		}
	}

	// Move particles to their new positions.
	pMove();
}

bool MPrimitiveParticle::drawObject(const Aztec::MBaseObjectPtr &baseObj, Aztec::MSceneViewFlags ViewFlags)
{
	Aztec::MParameterObjectPtr Param;

	// TBD: Draw representations of the pieces of the particle system (sources, bouncers, ...)

	// Draw the particles themselves.
	if (!ViewFlags.m_Playing) {
		if (isPlaying) {
			// Last time we checked, we were playing an animation.  Since the user has
			// stopped doing that, we need to terminate the particle system.

			// Until I can figure out a failure code for pGenParticleGroups(), I'm going to assume
			// it worked, and hence I need to get rid of it.
			pDeleteParticleGroups(particleHandle, 1);

			isPlaying = false;
		}

		// If not actively playing, then no need to draw anything.
		return true;
	}

	// Get the time
	long currentTick = getTime();
	Aztec::MScenePtr scene = primSysMan->getScene();
	float currentSec = (float)scene->tickToSeconds(currentTick);
	long startTime = scene->getStartTime();
	long endTime = scene->getEndTime();

	float timeStep;
	if (!stepRatePtr->getValueFloat(timeStep)) {
		// Failed to read parameter value.
		assert(false);
		return false;
	}

	// Get the object that represents the particle emitter (unused)
	Param = m_ParamList->getParameter("inMesh");

	if (!isPlaying) {
		// We are starting the effect.  Generate a particle group.
		lastEvaluatedTime = currentSec;

		int maxParticles;
		if (!maxParticlesPtr->getValueInteger(maxParticles)) {
			// Failed to read parameter value - abort processing.  TBD: how should user be
			// alerted that something bad happened?
			assert(false);
			return false;
		}

		// Docs don't say if there is an error code for pGenParticleGroups() - so we don't know
		// if the allocation fails.
		particleHandle = pGenParticleGroups(1, maxParticles);
		pCurrentGroup(particleHandle);

		isPlaying = true;
	}

	// Perform a time step on the particle system
	Aztec::MSceneObjectPtr sceneObj = AZTEC_CAST(Aztec::MSceneObject, baseObj);

	// The rate of display to the user may not match the steps at which we want to evaluate
	// the particle system.  This loop will evaluate until the particle system time is at least
	// as much as the current Aztec time.
	for (;lastEvaluatedTime <= currentSec;lastEvaluatedTime+=timeStep) {
		evaluateActions(sceneObj, ViewFlags);
	}

	// Draw the particles
	pDrawGroupp(GL_POINTS);

	return true;
}

class MParticleScriptSupport :
	public Aztec::MJavaScriptSupport
{
public:
	MParticleScriptSupport(void) { }
	~MParticleScriptSupport(void) { }

	JSObject *getJavaScriptObject(JSContext *cx, JSObject *obj, Aztec::MNamedObjectPtr nobj) {
		MParticlePtr pptr = AZTEC_CAST(MPrimitiveParticle, nobj);
		if (pptr != NULL) {
			return Aztec::newParticle(cx, obj, pptr);
		} else {
			return NULL;
		}
	}
};

Aztec::MScriptingSupportPtr MPrimitiveParticle::getScriptObject()
{
	return new MParticleScriptSupport();
}
