#include <AztecMainPCH.h>
#include <tools/MIKBoneTool.h>

// AztecLib includes
#include <animation/MIKControllerCCD.h>
#include <MUIManager.h>
#include <MBone.h>

// AztecMain includes
#include <MDLGlobs.h>
#include <DLGGlobs.h>

MIKBoneTool::MIKBoneTool()
{
  setName("ikBoneTool");
  currentState = PickingFirstBone;
}

void MIKBoneTool::initialise() {
  firstBone = NULL;
  secondBone = NULL;
  pickedBone = NULL;
  currentState = PickingFirstBone;

  MSystemManager::getInstance()->logOutput("Click on the bone that will be the start of the IK chain, then press ENTER");
}

bool MIKBoneTool::finish() {
  if (currentState == PickingFirstBone) {
    firstBone = pickedBone;
    currentState = PickingSecondBone;
    g_Scene->selectObject(pickedBone, false);

    // we need to update the display because our selection has changed.
    g_MainDlg->PostMessage(MM_UPDATEVIEWPORTS, MMC_UPDATE_ALL, 0);
    g_MainDlg->SendMessage(MM_UPDATECHANNELBAR, 0, 0);

    MSystemManager::getInstance()->logOutput("Click on the bone that will be the last bone of the IK chain, then press ENTER");

    // return false because we have other things we need to pick.
    return false;
  }

  if (currentState == PickingSecondBone) {
    secondBone = pickedBone;
    currentState = DonePicking;
    g_Scene->selectObject(pickedBone, false);

    createIKObject();

    MSystemManager::getInstance()->logOutput("");

    // return true because we are finished with this tool.
    return true;
  }

  // if we are in any other state, we just finish the tool as per normal
  return true;
}

bool MIKBoneTool::cancel() {
  // if we are cancelling, then just use the default behaviour.
  return MToolType::cancel();
}

int MIKBoneTool::DrawTool(bool Select, MShiftState ShiftState, MBaseViewWndPtr View)
{
  return TOOLRESULT_DRAWALL;
}

int MIKBoneTool::onMouseDown(int X, int Y, const MShiftState &Shift)
{
  MToolType::onMouseDown(X, Y, Shift);
  
  // Check to see if the mouse buttons are down.
  COpenGLWnd     *GLWnd;
  AztecFlags     OldMode;
  
  GLWnd = AZTEC_CAST(COpenGLWnd, g_CurView);
  
  if (GLWnd == NULL) {
    return TOOLRESULT_DRAWNONE;
  }
  
  // check to see if all the mouse buttons are up
  if (!Shift.m_Left) {
    return TOOLRESULT_DRAWNONE;
  }
  
  // select all the objects in the selection
  
  OldMode = MUIManager::getComponentMode();
  MUIManager::setComponentMode(MUIManager::OBJECT_MODE);

  GLWnd->performSelection(X, Y, &m_InputList);
  
  MUIManager::setComponentMode(OldMode);
  
  // This is the bone that we have selected.
  MSceneObjectPtr closestBone;
  float minDistance = 10000000000;
  
  // iterate over the list, and get out all the selection items, and find 
  // the closest one.
  MBaseObjectPtr Obj;
  
  for (int i = 0; i < m_InputList.size(); ++i) {
    MSelectionItem *selItem = &m_InputList[i];
    
    // make sure our selection didn't have any components,
    // since we are just after whole.
    if (selItem->getNumComponents() == 0) {
      MSceneObjectPtr selectedObject;
      
      selectedObject = AZTEC_CAST(MSceneObject, selItem->getObject());
      
      if (selectedObject == NULL) {
        continue;
      }

      MBoneObjectPtr bone = AZTEC_CAST(MBoneObject, selectedObject->getShapeObject());

      if (bone == NULL) {
        continue;
      }
      
      // now that we definately have a bone, find the closest one that 
      // we clicked on
      MVector3 boneCentre = g_Scene->objectToWorldSpace(MBaseObjectPtr(selectedObject), MVector3(0,0,0));

      if ((boneCentre - m_DownRay.Org).length2() < minDistance) {
        minDistance = (boneCentre - m_DownRay.Org).length2();
        closestBone = selectedObject;
      }
  
    }
  }

  // deselect the old bone that we had.
  g_Scene->selectObject(pickedBone, false);
  pickedBone = closestBone;
  // selec the new bone
  g_Scene->selectObject(pickedBone);

  return true;
}

void MIKBoneTool::createIKObject() {
  // if we dont have two bones, then give up
  if (firstBone == NULL || secondBone == NULL) {
    return;
  }

  // create the end effector for the ik chain.
  MVector3 effectorLocation = g_Scene->objectToWorldSpace(MBaseObjectPtr(secondBone), MVector3(0,0,0));

  MSceneObjectPtr endEffector = new MSceneObject();
  endEffector->setName("Effector");
  endEffector->findParameter("Translate")->setValueVector(effectorLocation);
  endEffector->findParameter("drawAxis")->setValueBoolean(true);

  g_Scene->addObject(endEffector);

  // now create the actual ik solver.
  MIKControllerCCD *ik = new MIKControllerCCD();
  MIKControllerPtr ikPtr = ik;

  ik->setName("IKController");

  ik->getEndEffectorParameter()->setValue(endEffector);
  ik->getStartBoneParameter()->setValue(firstBone);
  ik->getEndBoneParameter()->setValue(secondBone);

  g_Scene->addObject(ikPtr);

  g_Scene->selectNone();
  g_Scene->selectObject(endEffector);
}
