
#ifndef SCENE_H
#define SCENE_H

#include "ModelGeneric.h"
          
namespace Aztec {

  class MSelectionItem;
  class MScene;

  typedef MRefCountedPtr<MSelectionItem> MSelectionItemPtr;
  typedef MRefCountedPtr<MScene> MScenePtr;

}

#include "MSceneObject.h"
#include "MBaseObject.h"
#include "MNamedObject.h"
#include "MListsTrees.h"
#include "MMaterial.h"
#include "MTimeSegment.h"
#include "MMath.h"

#include <vector>

#include <MSelectionItem.h>
#include <MBaseUndoNode.h>

namespace Aztec {
  
  //----------------------------------------------------------------------------------------
  //  MScene
  //----------------------------------------------------------------------------------------
  class MGENEXPORT MScene : public MBaseObject {
  protected:
    MBaseObjectTreePtr m_Objects;
    MBaseObjectListPtr m_SelectedObj;   // For selected objects, not components.

    std::vector<MSceneObjectPtr> m_Lights;
    std::vector<MTimeSegmentPtr> m_TimeSegments;
    
  public:
    MScene();
    ~MScene();

    /**
     * Gets the main scene that we are dealing with. 
     */
    static MScenePtr getGlobalScene();

    /**
     * This sets the main scene that we deal with.
     */
    static void setGlobalScene(const MScenePtr &scene);

    // Class related
    virtual MStr getClassName() {return MStr("MScene");};
    virtual MStr getParentClassName() {return MStr("MBaseObject");};
    virtual MBaseObjectPtr createNew();
    
    // Time related

    virtual void setTime(int Time);

    /**
     * This sets The current starting range of time.
     *
     * @param startTime This is the time we are starting from
     *                  measured in Ticks (1/3600th of a second).
     */
    void setStartTime(int startTime);
    /**
     * This gets the current start time, measured in ticks.
     *
     * @return the Current Starting time.
     */
    int getStartTime();
    /**
     * This sets The current ending range of time.
     *
     * @param startTime This is the time we are starting from
     *                  measured in Ticks (1/3600th of a second).
     */
    void setEndTime(int endTime);
    /**
     * This gets the current ending time, measured in ticks.
     *
     * @return the Current Starting time.
     */
    int getEndTime();
    /**
     * This sets the time range, from startTime to endTime
     * both of which are measured in ticks (1/3600th of a second).
     *
     * @param startTime The new Start time.
     * @param endTime The new End time.
     */
    void setTimeRange(int startTime, int endTime);

    /**
     * This gets the current frames per second setting.
     * This defaults to 30 FPS.
     *
     * @return The current number of Frames per second.
     */
    int getFramesPerSecond() const;

    /**
     * This sets the current Frames Per Second.
     *
     * @param the new number of Frames per Second
     */
    void setFramesPerSecond(int fps);

    /**
     * This converts ticks to a frame number
     *
     * @param tick The number of ticks.
     * @return The equivilent number of Frames.
     */
    inline double tickToFrame(int tick) const {
      return (double)tick * (double)getFramesPerSecond() / (double)getTicksPerSecond();
    }

    /**
     * This converts a number of Frames to a tick count.
     *
     * @param frame The number of Frames.
     * @return The equivilent number of Ticks.
     */
    inline int frameToTick(double frame) const {
      return (int) ( frame * (double)getTicksPerSecond() / (double)getFramesPerSecond() );
    }

    /**
     * This converts a tick number to the number of seconds.
     *
     * @param tick The number of Ticks.
     * @return The equivilent number of Seconds.
     */
    inline double tickToSeconds(int tick) const {
      return (double)tick / (double)getTicksPerSecond();
    }

    /**
     * This converts the number of seconds to the number of ticks.
     *
     * @param seconds The number of Seconds.
     * @return The equivilent number of Ticks.
     */
    inline int SecondsToTick(double seconds) const {
      return (int) (seconds * (double)getTicksPerSecond());
    }

    /**
     * This gets the number of ticks that occur in one frame. This should
     * be used in preference to using scene->frameToTick(1) to work this
     * value out.
     */
    inline float getTicksPerFrame() const {
      return ((float)getTicksPerSecond() / (float)getFramesPerSecond());
    }

    /**
     * This gets the number of ticks that occur every second. This value
     * is constance and independant of frame rate.
     */
    inline int getTicksPerSecond() const {
      return 3600;
    }

    // Object related
    MBaseObjectTreePtr getObjectList() {return m_Objects;};
    void setObjectList(MBaseObjectTreePtr Tree);
    MBaseObjectListPtr getSelectedObjectList() {return m_SelectedObj;};

    MTreeObjectNodePtr addObject(MNamedObjectPtr Obj, MTreeObjectNodePtr Parent = NULL);
    MTreeObjectNodePtr addObjectSpecial(MTimeSegmentPtr segment, MTreeObjectNodePtr parent = NULL);
    
    int deleteObject(MNamedObjectPtr Obj, bool DeleteChildren = false);
    int deleteObjectNode(MTreeObjectNodePtr Node, bool DeleteChildren = false);
    int deleteObjectSpecial(MMaterialPtr Material, bool DeleteChildren = false);
    int deleteObjectSpecial(MTimeSegmentPtr Light, bool DeleteChildren = false);

    // List maintenenace functions
    void rebuildLightList();
    void rebuildMaterialList();
    void rebuildSelectList();
    
    // Deletes all objects in the scene
    void clearScene();
    
    // Object Related Stuff
    int updateObject(MBaseObjectPtr Obj, bool UpdateDependants = true);
    int updateSelectedObjectList();
    void setObjectChanged(MBaseObject *obj);
    
    // Remove all objects from the Selected Object list
    int selectNone();

    /**
     * This selects or deselects the given object. If we are currently in a 
     * component editing mode, the newly selected object is also placed in 
     * that component mode.
     * 
     * Normally, if an object is the child of an already selected object, 
     * then the selection request is ignored, because non-intuitive things 
     * can happen when trying to do move operations of two objects in the 
     * same parent/child heirarchy. The third argument allows this behaviour 
     * to be disabled, and the object selected regardless of what else is 
     * selected.
     */
    int selectObject(MBaseObjectPtr Obj, bool TargetSel = true, bool ignoreParentSelection = false);
    int getNumSelectedObjects() {return m_SelectedObj->getCount();};
    
    int drawScene(MSceneViewFlags ViewFlags, std::vector<MSelectionItem> *OutputList);
    int processSelection(UINT *SelectBuf, 
                         int NumHits, 
                         const std::vector<MSceneObjectPtr> &ObjArray, 
                         std::vector<MSelectionItem> *OutputList);
    
    void setupGLLights();
    
    MVector3 getSelectionCentre(int MaxCount = -1);
    bool anythingSelected();
    
    MTransformObjectPtr getTransform(MBaseObjectPtr Obj);
    MTransformObjectPtr getTransform(MTreeObjectNodePtr Node);
    MTransformObjectPtr getTransformParent(MBaseObjectPtr Obj);
    MTransformObjectPtr getTransformParent(MTreeObjectNodePtr Node);
    
    // Goes up the Tree and concatenates all the transforms using gl* commands.
    void GLTransform(MBaseObjectPtr Obj, DWORD XFormChan = TRANSFORM_CHAN_ALL);
    void GLTransform(MTreeObjectNodePtr Node, DWORD XFormChan = TRANSFORM_CHAN_ALL);
    void GLTransform(DWORD XFormChan = TRANSFORM_CHAN_ALL);
    
    void getWorldTransformMatrix(MTreeObjectNodePtr Node, float* Mat);
    void getWorldTransformMatrix(MTreeObjectNodePtr Node, MMatrix4 &Mat);
    MVector3 getWorldTransformTranslate(MTreeObjectNodePtr Node);
    
    // Coordinate conversion functions
    
    // convert from WorldSpace into ObjectSpace coordinates
    MVector3 worldToObjectSpace(MBaseObjectPtr Obj, const MVector3 &Vec);
    MVector3 worldToObjectSpace(MTreeObjectNodePtr ObjNode, const MVector3 &Vec);
    
    // Convert from Object Space into World Space cooridnates
    MVector3 objectToWorldSpace(MBaseObjectPtr Obj, const MVector3 &Vec);
    MVector3 objectToWorldSpace(MTreeObjectNodePtr ObjNode, const MVector3 &Vec);

    /**
     * This creates an empty scene object, essentially just
     * a way of grouping things.
     */
    MSceneObjectPtr createSceneObject(const std::string &name = "");

    /**
     * Creates a light.
     */
    MSceneObjectPtr createLight(const std::string &name = "");

    /**
     * Creates an animatable and editable mesh
     */
    MSceneObjectPtr createMesh(const std::string &name = "");

    /**
     * Creates an of the given type.
     * The type can be a friendly name, such as Light, Cube, or GeoSphere, 
     * or it can be the proper classname, such as MPrimitiveCube.
     * If the type is a 'group', then it simply create an empty scene
     * object, ready to use. If the type given is not a shape type, then it 
     * just creates a new object of that type, and returns that, without the
     * Scene Object wrapper.
     * 
     * @param primType The type of the primitive to create.
     * @return a named object which contains and object of the appropriate 
     *         type.
     */
    MNamedObjectPtr createObject(const std::string &primType, std::string name = "");

    // Time segment functions
    /**
     * Gets the number of time segments defined
     * @return the number of Time Segments.
     */
    int getTimeSegmentCount();

    /**
     * Gets the Time Segment at the given index.
     *
     * @param index The index of the Time Segment to get.
     * @return The Time Segment at the given index.
     */
    MTimeSegmentPtr getTimeSegment(int index);

  protected:
    bool doUpdateObject(MBaseObjectPtr obj);

    int m_DoObjectUpdates;
    bool m_PendingUpdates;
    bool m_DoingUpdate;

    /**
     * This is the starting time for our current time range. It
     * is measured in Ticks (1/3600th of a second)
     */
    int m_StartTime;

    /**
     * This is the ending time for our current time range. It
     * is measured in Ticks (1/3600th of a second)
     */
    int m_EndTime;

    /**
     * This is the current Frames Per Second
     */
    int m_FPS;
  };

}
#endif 
