#ifndef MKeyableValue_Header 
#define MKeyableValue_Header

#include "ModelGeneric.h"

namespace Aztec {

  class MKeyableValue;
  typedef MRefCountedPtr<MKeyableValue> MKeyableValuePtr;

}

#include "MVector3Value.h"

#include <vector>

namespace Aztec {

  /**
   * This is an interface class for a list of Keys. Each type
   * of key list implements this interface.
   */
  class MGENEXPORT MKeyableValue : public virtual MValue {
  public:
    /**
     * Does this value have a key set at a given time.
     */
    virtual bool hasKeySet(long time) = 0;

    /**
     * This takes all keys that are on or after the start time and before the 
     * end time, and shifts them forward by the given amount.
     *
     * @param onOrAfterTime The time that each key that is on or after this 
     *                      time will be shifted.
     * @param beforeTime Any key before this time will be shifted
     * @param amount The amount of time that each key will be shifted by. This
     *               can be a negative number to move them backwards.
     */
    virtual void shiftKeys(long onOrAfterTime, long beforeTime, long amount) = 0;

    /**
     * This shifts the selected keys by the given amount.
     *
     * @param amount The amount of time to shift each key by.
     */
    virtual void shiftSelectedKeys(long amount) = 0;

    /**
     * This gets all the times of the keys, and places them into a
     * container.
     *
     * @param keyTimes This is a vector which recieves all the 
     *        times of the keys set in this keyable value.
     */
    virtual void getKeyTimes(std::vector<int> &keyTimes) = 0;

    /**
     * This gets the times of the keys that are selected, and places them into 
     * a container.
     *
     * @param keyTimes This is a vector which recieves all the times of the 
     *                 keys set in this keyable value.
     */
    virtual void getSelectedKeyTimes(std::vector<int> &keyTimes) = 0;

    /**
     * Delete key set at the given time.
     */
    virtual void deleteKey(long time) = 0;

    /**
     * Deletes all the selected keys
     */
    virtual void deleteSelectedKeys() = 0;

    /**
     * Quick test to see if there are any keys set.
     *
     * @return true if there are keys set, false if there are zero keys.
     */
    virtual bool hasKeys() = 0;
  };
  

}

#endif 
