// Create gear tooth
function
create_tooth(gear_angle, tooth_angle, center,
    outer_pt, inner_pt, edge_pts, pts_index)
{
  var mx = new AztecMatrix4;

  mx.rotZ(gear_angle - 0.3 * tooth_angle);
  mx.transform(inner_pt, edge_pts[pts_index+0]);
  var pt = edge_pts[pts_index+0];
  pt.add(center);
  edge_pts[pts_index+0].add(center);

  mx.rotZ(gear_angle - 0.19 * tooth_angle);
  mx.transform(outer_pt, edge_pts[pts_index+1]);
  edge_pts[pts_index+1].add(center);

  mx.rotZ(gear_angle + 0.19 * tooth_angle);
  mx.transform(outer_pt, edge_pts[pts_index+2]);
  edge_pts[pts_index+2].add(center);

  mx.rotZ(gear_angle + 0.3 * tooth_angle);
  mx.transform(inner_pt, edge_pts[pts_index+3]);
  edge_pts[pts_index+3].add(center);
}

// Create gear
// Parameters:
//   teeth: number of teeth on the gear - must be a multiple of 4 
//   outer_radius: distance from center to top of gear tooth
//   inner_radius: distance from center to bottom of gear tooth
//   thickness: thickness of the gear. Bottom will be at -thickness
function
create_gear(teeth, outer_radius, inner_radius, thickness)
{
  // Have Aztec build us an empty mesh
  var meshObj = Scene.createMesh();
  var mesh = meshObj.getMesh();

  var center = new AztecVector3;
  var next_side, num_side, num_teeth ;
  var gear_angle, tooth_angle ;

  gear_pts = new Array(4 * teeth + 1);
  for (var i=0;i<4 * teeth;i++) {
    gear_pts[i] = new AztecVector3;
  }
  gear_pts[4 * teeth] = center;  // Add the center vertex to the array

  outer_pt = new AztecVector3(outer_radius, 0.0, 0.0);
  inner_pt = new AztecVector3(inner_radius, 0.0, 0.0);
    
  tooth_angle = 2.0 * Math.PI / teeth;
    
  // output gear top
  for (num_teeth=0;num_teeth<teeth;num_teeth++) {
    gear_angle = 2.0 * Math.PI * num_teeth / teeth;
    create_tooth(gear_angle, tooth_angle, center, outer_pt, inner_pt,
        gear_pts, num_teeth * 4);
  }

  // Build triangles
  var tris = new Array(4 * teeth);
  var triIndex = 0;
  var centerIndex = 4 * teeth;
  var vindex = 0;
  for (num_teeth=0;num_teeth<teeth;num_teeth++,vindex+=4) {
    if (num_teeth == 0) {
      // Wrap around from last to first
      tris[triIndex++] = new AztecTriangle(teeth * 4 - 1, 0, centerIndex);
    } else {
      tris[triIndex++] = new AztecTriangle(vindex-1, vindex, centerIndex);
    }
    tris[triIndex++] = new AztecTriangle(vindex, vindex+1, vindex+2);
    tris[triIndex++] = new AztecTriangle(vindex, vindex+2, vindex+3);
    tris[triIndex++] = new AztecTriangle(vindex, vindex+3, centerIndex);
  }

  // Emit top polygon
  mesh.addVertsAndTriangles(gear_pts, tris, 4 * teeth + 1, triIndex);

  // Move all gear points down
  for (var i=0;i<teeth*4+1;i++) {
    gear_pts[i].z -= thickness;
  }
  // Flip triangle order so their natural normal points down
  for (var i=0;i<triIndex;i++) {
    t = tris[i];
    t.set(t.i0, t.i2, t.i1);
  }
  // Emit bottom polygon
  mesh.addVertsAndTriangles(gear_pts, tris, 4 * teeth + 1, triIndex);

  // Move all gear points back up
  for (var i=0;i<teeth*4+1;i++) {
    gear_pts[i].z = 0;
  }

  // output sides of the teeth
  side_pts = new Array(4);
  side_pts[0] = new AztecVector3; side_pts[1] = new AztecVector3;
  side_pts[2] = new AztecVector3; side_pts[3] = new AztecVector3;
  var sideTris = new Array(2);
  sideTris[0] = new AztecTriangle(0, 1, 2);
  sideTris[1] = new AztecTriangle(0, 2, 3);
  for (num_side=0;num_side<4*teeth;num_side++) {
    next_side = (num_side + 1) % ( 4 * teeth);
    side_pts[0].set(gear_pts[num_side]);
    side_pts[1].set(gear_pts[num_side]);
    side_pts[1].z -= thickness;
    side_pts[2].set(gear_pts[next_side]);
    side_pts[2].z -= thickness;
    side_pts[3].set(gear_pts[next_side]);

    mesh.addVertsAndTriangles(side_pts, sideTris, 4, 2);
  }

  // Clean up redundant vertices and compute normals for all tris
  mesh.weldVertices(0, 0.0);
  mesh.calculateNormals();
}

// Make a single gear
function makeGear() {
  create_gear(32, 10, 9, 1);
}
