#include <Aztec3DPCH.h>

#include <views/AztecViewManager.h>

#include <gui/MGridLayout.h>
#include <gui/MBorderLayout.h>
#include <gui/MTextField.h>
#include <gui/MLabel.h>

#include <controls/ChannelBar.h>

#include <MScene.h>
#include <MDAGraph.h>
#include <params/MKeyParameter.h>

#include <algorithm>

namespace AztecGUI {
                

  class ChannelBarLayout : public Aztec::MLayoutManager {
  private:
    typedef std::vector<Aztec::MComponentPtr> Children;
    Children children;

  public:

    void addComponent(Aztec::MComponentPtr component) {
      children.push_back(component);
    }

    void addComponent(Aztec::MComponentPtr component, Aztec::MConstraint constraints) {
      children.push_back(component);
    }

    void removeComponent(Aztec::MComponentPtr component) {
      Children::iterator i = std::find(children.begin(), children.end(), component);
      if (i != children.end()) {
        children.erase(i);
      }
    }

    void layoutContainer(Aztec::MContainerPtr container) {
      Aztec::MSize2D containerSize = container->getSize();
      int yPos = 0;

      for (Children::iterator i = children.begin(); i != children.end(); ++i) {
        Aztec::MSize2D preferredSize = (*i)->getPreferredSize();

        (*i)->setPosition(0, yPos, containerSize.getWidth(), preferredSize.getHeight());

        yPos += preferredSize.getHeight();
      }
    }

    Aztec::MSize2D getMinimumSize(Aztec::MContainerPtr container) {
      // the minimum size is the largest minimum width of all the 
      // components, and all the heights of each component added together.

      int maxWidth = 0;
      int height = 0;
      for (Children::iterator i = children.begin(); i != children.end(); ++i) {
        Aztec::MSize2D size = (*i)->getMinimumSize();
        if (size.getWidth() > maxWidth) {
          maxWidth = size.getWidth();
        }
        height += size.getHeight();
        
      }

      return Aztec::MSize2D(maxWidth, height);
    }

    Aztec::MSize2D getPreferredSize(Aztec::MContainerPtr container) {
      // the preferred size is the largest preferred width of all the 
      // components, and all the preferred heights of each component added together.

      int maxWidth = 0;
      int height = 0;
      for (Children::iterator i = children.begin(); i != children.end(); ++i) {
        Aztec::MSize2D size = (*i)->getPreferredSize();
        if (size.getWidth() > maxWidth) {
          maxWidth = size.getWidth();
        }
        height += size.getHeight();
        
      }

      return Aztec::MSize2D(maxWidth, height);
    }

    Aztec::MConstraint getConstraint(const Aztec::MComponentPtr &component) {
      return 0;
    }

  };

  typedef Aztec::MRefCountedPtr<ChannelBarLayout> ChannelBarLayoutPtr;




  static ChannelBar *g_Instance = NULL;

  ChannelBar::ChannelBar() {
    g_Instance = this;
  }


  ChannelBar::~ChannelBar() {
    g_Instance = NULL;
  }

  ChannelBar* ChannelBar::getInstance() {
    return g_Instance;
  }

  static void getDispalyableNodes(const Aztec::MNamedObjectPtr &object, std::set<Aztec::MDAGNode*> &nodes) {
    for (Aztec::MDAGraphUpstreamIterator i(&*object); !i.atEnd(); ++i) {
      nodes.insert(*i);
    }
  }

  void ChannelBar::showParametersFor(const Aztec::MNamedObjectPtr &object) {
    if (curObject == object) {
      // check to see if the inputs have changed since last time. If they 
      // have, we must redraw the whole thing.
      std::set<Aztec::MDAGNode*> inputs;
      getDispalyableNodes(object, inputs);
      if (curObjectInputs == inputs) {

        // make sure we have updated values.
        for (unsigned i = 0; i < parameters.size(); ++i) {
          parameters[i]->updateEditorValue();
        }

        return;
      } else {
        curObjectInputs = inputs;
      }
    } else {
      getDispalyableNodes(object, curObjectInputs);
    }
    curObject = object;

    if (scrolledContainer != NULL) {
      scroller->removeComponent(scrolledContainer);
    }

    parameters.clear();
    objects.clear();
    scrolledContainer = NULL;

    if (object != NULL) {
      scrolledContainer = new Aztec::MContainer();
      scrolledContainer->setLayoutManager(new ChannelBarLayout());
      scroller->addComponent(scrolledContainer, Aztec::MBorderLayout::CENTRE);
      
      addParameters(object);
    }

    // we couldn't do a doLayout() on this container, because the scroller 
    // doesn't change size, and it thinks that nothing has changed (since 
    // the size doesn't change)
    scroller->doLayout();
    refresh();
  }

  Aztec::MSize2D ChannelBar::getMinimumSize() {
    return Aztec::MSize2D(200,50);
  }
  
  void ChannelBar::onCreate() {
    setLayoutManager(new Aztec::MBorderLayout(0,0));

    menu = new Aztec::MMenuBar();
    scroller = new Aztec::MScrolledContainer();
    scroller->setVisibility(Aztec::MScrolledContainer::HorizontalScroller, Aztec::MScrolledContainer::Never);
    addComponent(menu, Aztec::MBorderLayout::NORTH);
    addComponent(scroller, Aztec::MBorderLayout::CENTRE);

#ifdef _WIN32
    menu->addItem(new Aztec::MMenuItem("Selected"));
    menu->addItem(new Aztec::MMenuItem("Edit"));
#else
    // Quick hack to work around a few bugs in Qt/AGUI
    menu->createImpl();

    Aztec::MMenuPtr sel_dummy = new Aztec::MMenu();
    Aztec::MMenuPtr edit_dummy = new Aztec::MMenu();

    sel_dummy->addItem(new Aztec::MMenuItem("dummy", ""));
    edit_dummy->addItem(new Aztec::MMenuItem("dummy", ""));

    Aztec::MMenuItemPtr sel = new Aztec::MMenuItem("Selected", sel_dummy);
    Aztec::MMenuItemPtr edit = new Aztec::MMenuItem("Edit", edit_dummy);

    menu->addItem(sel);
    menu->addItem(edit);
#endif
  }

  Aztec::MLabelPtr createLabel(const std::string &text, float shade = 0.6) {
    Aztec::MLabelPtr label = new Aztec::MLabel(text);

    label->setBackgroundColour(Aztec::MColour(shade, shade, shade));
    return label;
  }

  void ChannelBar::addParameters(const Aztec::MNamedObjectPtr &object) {
    doAddParameters(object);


    scrolledContainer->addComponent(createLabel("Shape"));

    // check to see if this is a SceneObject. If it is, add in the shapes.
    Aztec::MSceneObjectPtr sceneObject = AZTEC_CAST(Aztec::MSceneObject, object);
    if (sceneObject != NULL) {
      doAddParameters(sceneObject->getShapeObject());
    }

    Aztec::MSceneObjectPtr sceneParent;

    if (sceneObject != NULL) {
      sceneParent = sceneObject->getParent();
    }

    scrolledContainer->addComponent(createLabel("Inputs"));

    // now add all the inputs
    std::vector<Aztec::MDAGNode*> ignoreList;
    ignoreList.push_back(&*sceneParent);
    for (Aztec::MDAGraphUpstreamIterator it(&*object, ignoreList); !it.atEnd(); ++it) {
      Aztec::MNamedObject *inputNamedObj = AZTEC_CAST(Aztec::MNamedObject, *it);
      
      if (inputNamedObj != NULL && inputNamedObj != &*object) {
        doAddParameters(inputNamedObj);
      }
    }

  }

  void ChannelBar::doAddParameters(const Aztec::MNamedObjectPtr &object) {
    if (object == NULL || objects.find(object) != objects.end()) {
      return;
    }

    scrolledContainer->addComponent(createLabel(object->getName().c_str(), 0.8));
    objects.insert(object);

    Aztec::MContainerPtr cont = new Aztec::MContainer();
    
    scrolledContainer->addComponent(cont);
    
    Aztec::MParameterObjectListPtr list = object->getParamList();
    
    // first count the number of visible parameters.
    int numVisible = 0;
    for (int i = 0; i < list->getNumParams(); ++i) {
      Aztec::MParameterObjectPtr parameter = list->getParameter(i);
      if (parameter->isVisible()) {
        ++numVisible;
      }
    }
    
    cont->setLayoutManager(new Aztec::MGridLayout(2,numVisible));
    for (int i = 0; i < list->getNumParams(); ++i) {
      Aztec::MParameterObjectPtr parameter = list->getParameter(i);
      // don't display certain parameters
      if (parameter->isVisible()) {
        
        // create the label.
        Aztec::MLabelPtr label = new Aztec::MLabel();
        label->setValue(parameter->getFriendlyName().c_str());
        label->setVisible(true);
        
        cont->addComponent(label);
        
        ChannelParameterPtr channelParam = new ChannelParameter(parameter);
        channelParam->createEditor(cont);
        channelParam->updateEditorValue();
        parameters.push_back(channelParam);
      }
    }
  }


  class ChannelTextField : public Aztec::MTextField {
  public:
    ChannelTextField(ChannelParameter *param) {
      this->param = param;
    }

    bool onAccepted() {
      // if the value is accepted, post the changes back to the parameter.
      param->getParameter()->setValueString(getValue().c_str());
      AztecViewManager::redrawAllViews();

      return false;
    }
    bool onCancelled() {
      // if we hit escape, revert the value.
      param->updateEditorValue();
      return false;
    }
    
    ChannelParameter *param;
  };

  ChannelParameter::ChannelParameter() {
  }

  ChannelParameter::ChannelParameter(const Aztec::MParameterObjectPtr &param) {
    parameter = param;
    paramName = param->getFriendlyName();

  }

  Aztec::MComponentPtr ChannelParameter::createEditor(const Aztec::MContainerPtr &container) {
    // make the editor
    Aztec::MTextFieldPtr field = new ChannelTextField(this);
    container->addComponent(field);

    editor = field;

    MStr str;
    parameter->getValueString(str);
    field->setValue(str.c_str());
    field->setVisible(true);

    return editor;
  }

  void ChannelParameter::updateEditorValue() {
    MStr str;
    parameter->getValueString(str);
    editor->setValue(str.c_str());

    // Check to see if we have an animated parameter. If we do, colour in the 
    // editor to match what is going on.
    // 
    // If we have some keys, make it light pink, if we are on a keyed value, 
    // make it daker, otherwise make it the normal colour.
    Aztec::MKeyParameterPtr keyParam = AZTEC_CAST(Aztec::MKeyParameter, parameter);

    if (keyParam != NULL) {
      if (keyParam->getKeyableValue()->hasKeySet(Aztec::MScene::getGlobalScene()->getTime())) {
        editor->setBackgroundColour(Aztec::MColour(1.0, 0.5, 0.5, 0.65));
      } else if (keyParam->getKeyableValue()->hasKeys()) {
        editor->setBackgroundColour(Aztec::MColour(1.0, 0.5, 0.5, 0.25));
      } else {
        // Use 100% transparent colour so the colour is the default one for editors.
        editor->setBackgroundColour(Aztec::MColour(1.0, 0.5, 0.5, 0.0));
      }
    }
  }

  std::string ChannelParameter::getParameterName() {
    return paramName;
  }

  Aztec::MParameterObjectPtr ChannelParameter::getParameter() {
    return parameter;
  }

  Aztec::MComponentPtr ChannelParameter::getEditor() {
    return editor;
  }

}

