#include <Aztec3DPCH.h>
#include <controls/AztecGLCanvas.h>

// Aztec2 includes
#include <views/AztecViewManager.h>
#include <views/Aztec3DView.h>
#include <tools/MSelectTool.h>

// AztecGUI includes
#include <tools/MToolManager.h>
#include <config/UIConfig.h>

// AztecLib includes
#include <MSystemManager.h>
#include <scripting/MScriptInterp.h>

#include <algorithm>
#include <iostream>
#include <math.h>
#include <assert.h>
#include <GL/glu.h>

namespace AztecGUI {

  using namespace Aztec;

  AztecGLCanvas::AztecGLCanvas(AztecView *parentView) 
    : AztecGLFontCanvas(), parentView(parentView)
  {
    aspectRatio = 1.0;

    gridBasisX.set(1,0,0);
    gridBasisY.set(0,1,0);
    gridVisible = true;
    undoCount = 0;
    gridMajorSpacing = 5; 
    gridSpacing = 1.0;
    gridRange = 100.0;
    menuPopping = false;
	snapViaComponentCP = 1;

  }

  AztecGLCanvas::~AztecGLCanvas() {
	  {
	}
  }
  
  // utility functions
  float AztecGLCanvas::getAspectRatio() {
    return aspectRatio;
  }

  void AztecGLCanvas::getSelection(float left, float top, float right, float bottom, std::vector<Aztec::MSelectionItem> &items) {
    items.clear();
  }
  
  void AztecGLCanvas::getSelection(float x, float y, std::vector<Aztec::MSelectionItem> &items) {
    items.clear();
  }

  Aztec::MRay AztecGLCanvas::getRay(int x, int y) {

    // TODO: Change this so it does not use open gl to computer the coordinates, but does it by itself.
    double   worldX,worldY,worldZ;
    double   ModelMat[16], ProjMat[16];
    int      ViewInfo[4];
    
    MVector3 V1, V2;
    MRay     Ray;
    makeCurrent();

    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glLoadIdentity();
    doCameraTransform();
    glGetDoublev(GL_PROJECTION_MATRIX, ProjMat);
    glGetDoublev(GL_MODELVIEW_MATRIX, ModelMat);
    glGetIntegerv(GL_VIEWPORT, ViewInfo);
    glPopMatrix();
//    glGetIntegerv(GL_VIEWPORT, ViewInfo);
    
    gluUnProject(x, ViewInfo[3] - y, 0.0, ModelMat, ProjMat, ViewInfo, &worldX, &worldY, &worldZ);
    V1.set((float)worldX,(float)worldY,(float)worldZ);
    gluUnProject(x, ViewInfo[3] - y, 0.5, ModelMat, ProjMat, ViewInfo, &worldX, &worldY, &worldZ);
    V2.set((float)worldX,(float)worldY,(float)worldZ);
    
    Ray.set(V1, V2, 1.0f);
    
    return Ray;
  }

  std::string AztecGLCanvas::getViewGroup() const {
    return parentView->getViewGroup();
  }

  void AztecGLCanvas::glAdjustMatrixForSelection(float x, float y) {
    GLint       ViewportInfo[4];
    
    int         PickWidth, PickHeight;
    
    PickWidth = 5;
    PickHeight = 5;
    
    glGetIntegerv(GL_VIEWPORT, ViewportInfo);
    
    gluPickMatrix(x, ViewportInfo[3] - y, PickWidth,PickHeight, ViewportInfo);
  }

  void AztecGLCanvas::glAdjustMatrixForSelection(float left, float top, float right, float bottom) {
    GLint       ViewportInfo[4];
    
    int         width, height, x, y;
    
    x = (left + right) / 2;
    y = (top + bottom) / 2;
    width = fabs(left - right);
    height = fabs(top - bottom);

    // we want an absolute value of the width and height, which must also be greater than 0.
    if (width < 1) width = 1;
    if (height < 1) height = 1;

    glGetIntegerv(GL_VIEWPORT, ViewportInfo);
    
    gluPickMatrix(x, ViewportInfo[3] - y, width,height, ViewportInfo);
  }

  void AztecGLCanvas::setGridBasis(const Aztec::MVector3 &gridX, const Aztec::MVector3 &gridY) {
    gridBasisX = gridX;
    gridBasisY = gridY;
  }

  Aztec::MPlane AztecGLCanvas::getGridPlane() const {
    Aztec::MVector3 normal = gridBasisX / gridBasisY;
    normal.normalize();

    return Aztec::MPlane(MVector3(0,0,0), normal);
  }

  bool AztecGLCanvas::getGridVisible() const {
    return gridVisible;
  }

  void AztecGLCanvas::setGridVisible(bool visible) {
    gridVisible = visible;
  }


  float AztecGLCanvas::getGridRange() {
    return gridRange;
  }
  
  void AztecGLCanvas::setGridRange(float range){
    gridRange = range;
  }

  float AztecGLCanvas::getGridSpacing(){
    return gridSpacing;
  }
  void AztecGLCanvas::setGridSpacing(float spacing){
    gridSpacing = spacing;
  }
	
  int AztecGLCanvas::getGridMajorSpacing(){
    return gridMajorSpacing;
  }
  void AztecGLCanvas::setGridMajorSpacing(int spacing){
    gridMajorSpacing = spacing;
  }

  int AztecGLCanvas::getSnapViaComponentCP(){
	  return snapViaComponentCP;
  }


  void AztecGLCanvas::setSnapViaComponentCP(int state){
	snapViaComponentCP = state;
  }




  bool AztecGLCanvas::onResize(int newWidth, int newHeight) {

    calculateAspectRatio();

    return MGLCanvas::onResize(newWidth, newHeight);
  }


  bool AztecGLCanvas::onPaint() {
    AztecGLFontCanvas::onPaint();

    draw3D();

    return true;
  }

  bool AztecGLCanvas::onMousePressed(const MMouseEvent &event) {
    AztecViewPtr view = AztecView::getViewForComponent(event.getComponent());

    assert(view != NULL);
    AztecViewManager::setCurrentView(view);

    // check to see if there are no keyboard modifiers. If there aren't, assume we are popping up a menu
    if (event.getType() == MMouseEvent::RBUTTON_DOWN &&
        event.getShiftState().leftMouseDown == false &&
        event.getShiftState().middleMouseDown == false &&
        event.getShiftState().isKeyboardUp()) 
    {
      // first we try and cancel the current tool. If the current tool returns true to indicate it is completely cancelled, then we popup the menu. If it returns false, that means it is still in progres, so we do not popup the menu.
      MToolTypePtr tool = MToolManager::getInstance()->GetTool(view->getViewGroup());

      if (tool != NULL && !tool->inProgress()) {
        // Display the context menu if no keys have been pressed
        // and no mouse buttons are down.
        menuPopping = true;
        view->displayContextMenu(event);  
        return true;
      }
    } 
    
    if (event.getType() == MMouseEvent::RBUTTON_DOWN) {
      // check to see if the tool is doing anything, if it is, cancel it.
      MToolTypePtr tool = MToolManager::getInstance()->GetTool(view->getViewGroup());

      if (tool != NULL && tool->inProgress()) {
        tool->cancel();
        // redraw all the views.
        AztecViewManager::redrawAllViews();
        return true;
      }
    }

    AztecViewManager::disableViewUpdating();

    // start an undo block each time the mouse is pressed.
    ++undoCount;
    Aztec::MSystemManager::getInstance()->getUndoManager()->beginUndo("Action");

    mouseDownPos = event.getPos();

    setMouseCapture(true);

    // first get the tool which suits this particular combination of shortcut keys
    MToolTypePtr temporaryTool = MToolManager::getInstance()->getToolByName(UIConfig::getQuickToolFor(getViewGroup(), event.getShiftState()));

    // if we don't have a shortcut tool, see if our current tool requires one
    if (temporaryTool == NULL) {
      // see if our current tool requires selection. If it does, and we have none, switch to the select tool.
      if (MToolManager::getInstance()->GetTool(getViewGroup())->RequiresSelection()) {
        if (Aztec::MScene::getGlobalScene()->getNumSelectedObjects() == 0) {
          temporaryTool = view->getSelectTool();
        }
      }
    }

    // after all our playing about, see if we have a new temporary tool
    // If we do, activate it, if we don't see if we have clicked on a tool manipulator.
    if (temporaryTool != NULL) {
      MToolManager::getInstance()->setTemporary(temporaryTool, getViewGroup());
    } else {
      int pickedManipulator = 0;

      // check to see if the user is using the default manipulator
      if (UIConfig::isGrabManipulator(event.getShiftState())) {
        pickedManipulator = MToolManager::getInstance()->GetTool(getViewGroup())->getDefaultManip();
      } else {
      
        glMatrixMode(GL_PROJECTION);
        glPushMatrix();
      
        initProjectionMatrix(smSelect,event.getX(),event.getY(),0,0);
        pickedManipulator = drawManipulators(true);
      
        glMatrixMode(GL_PROJECTION);
        glPopMatrix();
      
        glMatrixMode(GL_MODELVIEW);
      }      
      
      
      MToolManager::getInstance()->GetTool(getViewGroup())->m_PickedManip = pickedManipulator;
      
      if (pickedManipulator == 0 && MToolManager::getInstance()->GetTool(getViewGroup())->getDefaultManip() != -1) { // The user missed all the manipultors
        MToolManager::getInstance()->setTemporary(view->getSelectTool(), getViewGroup());
      }

    }

    bool returnValue = false;
    int result = MToolManager::getInstance()->GetTool(getViewGroup())->onMouseDown(event);

    if (result == TOOLRESULT_DRAWCURRENT) {
      AztecViewManager::redrawCurrentView();
      returnValue = true;
    } else if (result == TOOLRESULT_DRAWALL) {
      // update the displays.
      AztecViewManager::redrawAllViews();
      returnValue = true;
    }

    AztecViewManager::enableViewUpdating();

    return returnValue;
  }

  bool AztecGLCanvas::onMouseReleased(const MMouseEvent &event) {
    mouseUpPos = event.getPos();

    // if the popup menu is popping, we don't want any mouse handling.
    if (menuPopping) {
      menuPopping = false;
      return true;
    }

    AztecViewManager::disableViewUpdating();

    if (event.getShiftState().isMouseUp()) {
      setMouseCapture(false);
    }

    AztecViewPtr view = AztecView::getViewForComponent(event.getComponent());
    assert(view != NULL);
    AztecViewManager::setCurrentView(view);

    int result = MToolManager::getInstance()->GetTool(getViewGroup())->onMouseUp(event);

    if (event.getShiftState().isMouseUp()) {
      MToolManager::getInstance()->unsetTemporary(getViewGroup());
    }

    bool handled = false;

    if (result == TOOLRESULT_DRAWCURRENT) {
      AztecViewManager::redrawCurrentView();
      handled = true;
    } else if (result == TOOLRESULT_DRAWALL) {
      // update the displays.
      AztecViewManager::redrawAllViews();
      handled = true;
    } else {
      // Do a quick 'n dirty check to see if the mouse didn't move too much
      MRect2D rc(mouseDownPos+MPoint2D(-5,-5), mouseDownPos+MPoint2D(5,5));

    }


    // finish an undo block each time the mouse is released.
    if (undoCount > 0) {
      Aztec::MSystemManager::getInstance()->getUndoManager()->endUndo();
      --undoCount;
    }

    AztecViewManager::enableViewUpdating();

    return handled;
  }

  bool AztecGLCanvas::onMouseMove(const MMouseEvent &event) {
    int result = MToolManager::getInstance()->GetTool(getViewGroup())->onMouseMove(event);

    if (result == TOOLRESULT_DRAWCURRENT) {
      refresh();
      return true;
    } else if (result == TOOLRESULT_DRAWALL) {
      // update the displays.
      AztecViewManager::redrawAllViews();
      return true;
    }

    return false;
  }

  bool AztecGLCanvas::onMouseWheel(const Aztec::MMouseEvent &event) {
    if (event.getType() == Aztec::MMouseEvent::WHEEL_UP) {
      Aztec::MScriptInterpreter::getInstance()->ExecuteScript("Scene.viewZoomIn();");
      return true;
    } else if (event.getType() == Aztec::MMouseEvent::WHEEL_DOWN) {
      Aztec::MScriptInterpreter::getInstance()->ExecuteScript("Scene.viewZoomOut();");
      return true;
    } else {
      return false;
    }
  }

  void AztecGLCanvas::calculateAspectRatio() {
    int bigValue;
    int smallValue;  // Was trying to use variable name 'small' but this was #defined as char in c:\Program Files\Microsoft Visual Studio\VC98\Include\RPCNDR.H line 39. Weird!!!

    getBigAndSmallViewSize(bigValue, smallValue);

    aspectRatio = (float)bigValue/(float)smallValue;
  }

  void AztecGLCanvas::getBigAndSmallViewSize(int &bigSize, int &smallSize) {
    bigSize = getSize().getWidth();
    smallSize = getSize().getHeight();

    if (smallSize > bigSize) {
      std::swap(bigSize,smallSize);
    }
  }

  MToolTypePtr AztecGLCanvas::getToolFor(const MShiftState &shift) {
    MToolTypePtr tool = MToolManager::getInstance()->getToolByName(UIConfig::getQuickToolFor(getViewGroup(), shift));

    if (tool == NULL) {
      tool = MToolManager::getInstance()->GetTool(getViewGroup());
    }

    return tool;
  }

}

