#include <AztecMainPCH.h>
#include "BaseOpenGLWnd.h"

#include <gl/glu.h>
#include <math.h>

#include "MdlGlobs.h"

#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


class ViewGLCanvas : public GLCanvas {
public:
  ViewGLCanvas(MBaseOpenGLWnd *view) {
    baseView = view;
  }

  // GLCanvas methods
  void onMouseUp(int x, int y, MShiftState Shift) {
  }

  void onMouseDown(int x, int y, MShiftState Shift) {
  }

  void onMouseMove(int x, int y, MShiftState Shift) {
  }

  LRESULT DefWindowProc(UINT message, WPARAM wParam, LPARAM lParam) {
    if (message == WM_LBUTTONDOWN ||
      message == WM_MBUTTONDOWN ||
      message == WM_RBUTTONDOWN ||
      message == WM_LBUTTONUP ||
      message == WM_MBUTTONUP ||
      message == WM_RBUTTONUP ||
      message == WM_MOUSEMOVE) 
    {
      // first convert the mouse coordinates from local to parent space.

      POINT p;
      p.x = (short int)LOWORD(lParam);
      p.y = (short int)HIWORD(lParam);
      ClientToScreen(&p);
      baseView->ScreenToClient(&p);

      lParam = MAKELPARAM(p.x, p.y);

      return baseView->SendMessage(message, wParam, lParam);
    }

    return GLCanvas::DefWindowProc(message, wParam, lParam);
  }

private:
  MBaseOpenGLWnd *baseView;
};


MBaseOpenGLWnd::MBaseOpenGLWnd() {
  needToCreateFonts = true;
  m_ViewInfo[0] = m_ViewInfo[1] = m_ViewInfo[2] = m_ViewInfo[3] = 0;
  m_WindowRect.left = 0;
  m_WindowRect.top = 0;
  m_WindowRect.right = 10;
  m_WindowRect.bottom = 10;
  glCanvas = NULL;
}

MBaseOpenGLWnd::~MBaseOpenGLWnd() {
  if (glCanvas != NULL) {
    delete glCanvas;
  }
}

void MBaseOpenGLWnd::ViewCreate() {
  MBaseViewWnd::ViewCreate();
  glCanvas = new ViewGLCanvas(this);
  glCanvas->Create(WS_VISIBLE, m_WindowRect, this);
}

int MBaseOpenGLWnd::InitOpenGL(HWND hWnd) {
  bool createFonts = true;
  
  // Find other views in the lists.
  MBaseObjectPtr Obj;
  g_ViewList.getList()->beginIteration();
  while (( Obj = g_ViewList.getList()->getNext() ) != NULL ) {
    MBaseOpenGLWndPtr wnd;
    
    wnd = AZTEC_CAST(MBaseOpenGLWnd, Obj);
    
    if (wnd == NULL || wnd->glCanvas == NULL) {
      continue;
    }
    
    createFonts = true;
    if (wnd->glCanvas->getGLContext() != glCanvas->getGLContext()) {
      if (wglShareLists(wnd->glCanvas->getGLContext(), glCanvas->getGLContext()) == FALSE) {
        g_SysMan->logOutput("MBaseOpenGLWnd::InitOpenGL() - Error on sharing display lists - %x", GetLastError());
      } else {
        createFonts = false;
        break;
      }
    } 
  }     
  g_ViewList.getList()->endIteration();
  
  GLMakeCurrent();
  
  if (createFonts) {
    glCanvas->createFonts(GLVIEW_FONTSTART);
  }

  return 0;	// Everything A Ok;
}

int MBaseOpenGLWnd::KillOpenGL()
{
  glCanvas->killOpenGL();
  return 1;
}


int MBaseOpenGLWnd::GLMakeCurrent() {
  return glCanvas->makeCurrent();
}


int MBaseOpenGLWnd::GLResizeToClient() {

  if (glCanvas != NULL) {
    glCanvas->GetClientRect(&m_WindowRect);
    if (glCanvas->makeCurrent() != GLCanvas::GL_OK) {
      return 0;
    }

    AdjustProjectionMatrix();
  }

  
  return 0;
}

int MBaseOpenGLWnd::GLSwapBuffers() {
  return glCanvas->swapBuffers();
}

void MBaseOpenGLWnd::DrawView() {
  MBaseViewWnd::DrawView();
}

void MBaseOpenGLWnd::createFonts(AztecFlags fontStart) {
  if (::IsWindowVisible(m_hWnd) == TRUE) {
    BOOL result;
    HDC hDC;
    
    hDC = ::GetDC(m_hWnd);

    HGDIOBJ oldFont = ::SelectObject(hDC, ::GetStockObject(DEFAULT_GUI_FONT));
   
    // don't create the fonts if the it has already been made.
    if (!glIsList(fontStart)) {

      result = ::wglUseFontBitmaps(hDC, 0, 255, fontStart);

      if (result == FALSE) {
        DWORD err;
    
        err = ::GetLastError();
        g_SysMan->logOutput("MBaseOpenGLWnd::createFonts() - Error creating Fonts (%i)", err);
      } else {
        needToCreateFonts = false;
      }

    }
  
    ::SelectObject(hDC, oldFont);
    ::ReleaseDC(m_hWnd, hDC);
  }
}

float MBaseOpenGLWnd::GetViewAspect() {
  long big, Small;
  
  if (m_WindowRect.right>m_WindowRect.bottom) {
    big = m_WindowRect.right;
    Small = m_WindowRect.bottom;
  } else {
    big = m_WindowRect.bottom;
    Small = m_WindowRect.right;
  }
  
  return (float)big/Small;
}

void MBaseOpenGLWnd::AdjustProjectionMatrix(MSelectMethod PickMode, 
                                            int left, int top,
                                            int right, int bottom) 
{
  MToolTypePtr CurTool;
  
  CurTool = g_ToolMan.GetTool();
  
  GLMakeCurrent();
  glMatrixMode(GL_PROJECTION);
  
  glViewport(0,0,m_WindowRect.right, m_WindowRect.bottom);
  
  glLoadIdentity();
  
  m_Aspect = GetViewAspect();
  
  // set up the picking projection if necessary.
  if (PickMode != smNone) {
    GLint       ViewportInfo[4];
    
    int         PickWidth, PickHeight, PickX, PickY;
    
    PickWidth = 3;
    PickHeight = 3;
    
    PickX = left;
    PickY = top;
    
    if (PickMode == smSelectBox || PickMode == smDeselectBox) {
      PickX = (left + right) / 2;
      PickY = (top + bottom) / 2;
      PickWidth = (left - right);
      PickHeight = (top - bottom);
      
      if (PickWidth < 0) {
        PickWidth = -PickWidth;
      }
      
      if (PickHeight < 0) {
        PickHeight = -PickHeight;
      }
      
      if (PickWidth == 0) {
        PickWidth = 1;
      }
      
      if (PickHeight == 0) {
        PickHeight = 1;
      }
      
    }
    
    glGetIntegerv(GL_VIEWPORT, ViewportInfo);
    
    gluPickMatrix(PickX, ViewportInfo[3] - PickY, PickWidth,PickHeight, ViewportInfo);
  }
}

void MBaseOpenGLWnd::AdjustScaleFactor() {
}

float MBaseOpenGLWnd::GetScaleFactor() {
  return 1.0;
}

void MBaseOpenGLWnd::UpdateXFormMatricies() {
}

void MBaseOpenGLWnd::DoViewportTransform() {
}


MVector3 MBaseOpenGLWnd::ScreenToWorldSpace(int X, int Y, const MPlane &ConstrainPlane) {
  MRay     Ray;
  
  Ray = GetViewRay(X, Y);
  
  return Ray.intersectWithPlane(ConstrainPlane);
}

MRay MBaseOpenGLWnd::GetViewRay(int X, int Y) {
  double   x,y,z;
  double   ModelMat[16], ProjMat[16];
  int      ViewInfo[4];
  
  MVector3 V1, V2;
  MRay     Ray;
  GLMakeCurrent();
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  glLoadIdentity();
  DoViewportTransform();
  glGetDoublev(GL_PROJECTION_MATRIX, ProjMat);
  glGetDoublev(GL_MODELVIEW_MATRIX, ModelMat);
  glGetIntegerv(GL_VIEWPORT, ViewInfo);
  glPopMatrix();
  glGetIntegerv(GL_VIEWPORT, ViewInfo);
  
  gluUnProject(X, ViewInfo[3] - Y, 0.0, ModelMat, ProjMat, ViewInfo, &x, &y, &z);
  V1.set((float)x,(float)y,(float)z);
  gluUnProject(X, ViewInfo[3] - Y, 0.5, ModelMat, ProjMat, ViewInfo, &x, &y, &z);
  V2.set((float)x,(float)y,(float)z);
  
  Ray.set(V1, V2, 1.0f);
  
  return Ray;
}

MVector3 MBaseOpenGLWnd::getViewNormal() {
  MVector3 result;
  m_ViewXFormInv.transform(MVector3(0,0,1),result);
  
  result.x = -result.x;
  result.y = -result.y;
  result.z = result.z;

  return result;
}


void MBaseOpenGLWnd::DrawViewExtras() {
  
}


int MBaseOpenGLWnd::DrawCube(float Size) {
  float    Size2;
  int      i;
  
  Size2 = Size / 2;
  
  for (i=0;i<6;i++) {
    glPushMatrix();
    
    switch (i) {
    case 0:  glRotatef(0,0,0,0); break;
    case 1:  glRotatef(90,1,0,0); break;
    case 2:  glRotatef(180,1,0,0); break;
    case 3:  glRotatef(270,1,0,0); break;
    case 4:  glRotatef(90,0,1,0); break;
    case 5:  glRotatef(270,0,1,0); break;
    }
    
    glBegin(GL_QUADS);
    
    glVertex3f( Size2,  Size2,  Size2);
    glVertex3f(-Size2,  Size2,  Size2);
    glVertex3f(-Size2, -Size2,  Size2);
    glVertex3f( Size2, -Size2,  Size2);
    
    glEnd();
    
    glPopMatrix();
  }
  return 1;
}

int MBaseOpenGLWnd::DrawArrowHead(float Length, float Radius) {
  float Color[4], Ang;
  
  glGetFloatv(GL_CURRENT_COLOR, Color);
  
  // Draw a fan of tris beginning from the top of the arrow head
  glBegin(GL_TRIANGLE_FAN);
  
  glColor4fv(Color);
  glVertex3f(Length,0,0);
  for (Ang=0;Ang<=360;Ang+=10) {
    float posy, posz, factor;
    
    posy = (float)(Radius*cos(Ang/180.0*M_PI));
    posz = (float)(Radius*sin(Ang/180.0*M_PI));
    factor = (float)((fabs(Ang-180))/360);
    
    glColor4f(Color[0]*factor, Color[1]*factor, Color[2]*factor, Color[3]);
    glVertex3f(0, posy, posz);
  }
  
  glEnd();
  
  glBegin(GL_TRIANGLE_FAN);
  
  // Draw a fan of tris beginning for the flat section of the arrow head
  glColor4fv(Color);
  glVertex3f(0,0,0);
  for (Ang=0;Ang<=360;Ang+=10) {
    float posy, posz, factor;
    posy = (float)(Radius*cos(Ang/180.0*M_PI));
    posz = (float)(Radius*sin(Ang/180.0*M_PI));
    factor = (float)((fabs(Ang-180))/360);
    
    glColor4f(Color[0]*factor, Color[1]*factor, Color[2]*factor, Color[3]);
    glVertex3f(0, posy, posz);
  }
  
  glEnd();
  
  return 1;
}

int MBaseOpenGLWnd::DrawAxisIcon(float Size, float Alpha, float LineWidth, bool Select, AztecFlags Flags) {
  MVector3    XVec, YVec;
  float       OldLineWidth;
  glMatrixMode(GL_MODELVIEW);
  
  //   glEnable(GL_DEPTH_TEST);
  glGetFloatv(GL_LINE_WIDTH, &OldLineWidth);
  glLineWidth(LineWidth);
  int   Num;
  
  glDisable(GL_CULL_FACE);
  glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
  
  // for do a for loop, from 1 to 4, for each the
  // axes. 1 = X, 2 = Y, 3 = Z, 4 = that yellow thing in the middle.
  for (Num = 1; Num <= 4; Num++) {
    if (!(Flags & (1 << (Num-1))))
      continue;
    
    if (Select) {
      glPushName(Num);
    }
    
    glPushMatrix();
    
    if (Num == 1) {
      glColor4f(1,0,0,Alpha);
      if (Flags & DRAWAXIS_XGRAY) {
        glColor4f(0,1,1,Alpha);
      }
    }
    if (Num == 2) {
      glColor4f(0,1,0,Alpha);
      glRotatef(90, 0,0,1);
      if (Flags & DRAWAXIS_YGRAY) {
        glColor4f(0,1,1,Alpha);
      }
    }
    if (Num == 3) {
      glColor4f(0,0,1,Alpha);
      glRotatef(-90, 0,1,0);
      if (Flags & DRAWAXIS_ZGRAY) {
        glColor4f(0,1,1,Alpha);
      }
    }
    if (Num == 4) {
      if (Flags & DRAWAXIS_ARROWS) {     // Do we want to draw the yellow square?

        // determine what axis to draw the square on
        if ((Flags & DRAWAXIS_MIDDLE_TYPE) == DRAWAXIS_MIDDLE_NORMAL_X) {
          XVec.set(0, Size*0.1, 0);
          YVec.set(0, 0, Size*0.1);
        } else if ((Flags & DRAWAXIS_MIDDLE_TYPE) == DRAWAXIS_MIDDLE_NORMAL_Y) {
          XVec.set(Size*0.1, 0, 0);
          YVec.set(0, 0, Size*0.1);
        } else if ((Flags & DRAWAXIS_MIDDLE_TYPE) == DRAWAXIS_MIDDLE_NORMAL_Z) {
          XVec.set(Size*0.1, 0, 0);
          YVec.set(0, Size*0.1, 0);
        } else {
          float    Mat[16];
          
          glGetFloatv(GL_MODELVIEW_MATRIX, Mat);
          
          XVec.set(Mat[0], Mat[4], Mat[8]);
          YVec.set(Mat[1], Mat[5], Mat[9]);
          
          XVec.normalize();
          YVec.normalize();
          XVec *= (float)(Size*0.1);
          YVec *= (float)(Size*0.1);
        }
        
        glColor4f(1,1,0,Alpha);
        if (Flags & DRAWAXIS_WGRAY) {
          glColor4f(0,1,1,Alpha);
        }
        
        glLineStipple(1, 0x5555);
        glEnable(GL_LINE_STIPPLE);
        
        glScalef(1.1f,1.1f,1.1f);
        
        glLineWidth(1);
        
        if (Select) {
          glBegin(GL_QUADS);
        } else {
          glBegin(GL_LINE_LOOP);
        }
        
        glVertex3f(-XVec.x - YVec.x, -XVec.y - YVec.y, -XVec.z - YVec.z);
        glVertex3f( XVec.x - YVec.x,  XVec.y - YVec.y,  XVec.z - YVec.z);
        glVertex3f( XVec.x + YVec.x,  XVec.y + YVec.y,  XVec.z + YVec.z);
        glVertex3f(-XVec.x + YVec.x, -XVec.y + YVec.y, -XVec.z + YVec.z);
        
        glEnd();
      }            
      glLineWidth(LineWidth);
      
      glDisable(GL_LINE_STIPPLE);
      
      glPopMatrix();
      
      if (Select) {
        glPopName();
      }
      break;
    }
    
    glBegin(GL_LINES);
    glVertex3f(0,0,0);
    glVertex3f(0.7f*Size,0,0  );
    glEnd();
    
    if (Flags & DRAWAXIS_ARROWS) {     // Draw funky arrow heads
      glPushMatrix();
      glTranslatef((float)(Size*0.7),0,0);
      DrawArrowHead((float)(Size*0.3), (float)(Size*0.1));
      glPopMatrix();
    }
    
    if (Flags & DRAWAXIS_LABELS) {
      // Draw X Y Z labels at the end of the sticks
      char  Str[5];
      
      Str[0] = 'X'+Num-1;
      glRasterPos3f(Size, 0,0);
      glListBase(GLVIEW_FONTSTART);
      glCallLists(1, GL_UNSIGNED_BYTE, Str);
    }
    
    glPopMatrix();
    
    if (Select) {
      glPopName();
    }
   }
   
   glLineWidth(OldLineWidth);
   
   return 0;
}

int MBaseOpenGLWnd::DrawRotateIcon(float Size, float Alpha, float LineWidth, bool Select) {
  float       OldLineWidth;
  MVector3    XVec, YVec, ZVec, ModelX, ModelY, ModelZ;
  glMatrixMode(GL_MODELVIEW);
  
  glPushMatrix();
  glGetFloatv(GL_LINE_WIDTH, &OldLineWidth);
  glLineWidth(LineWidth);
  
  // Draw circle in xy plane, xz, and zy plane
  {
    float    Mat[16];
    
    glGetFloatv(GL_MODELVIEW_MATRIX, Mat);
    
    ModelX.set(Mat[0], Mat[4], Mat[8]);
    ModelY.set(Mat[1], Mat[5], Mat[9]);
    ModelZ.set(Mat[2], Mat[6], Mat[10]);
    
    ModelX.normalize();
    ModelY.normalize();
    ModelZ.normalize();
    
    Mat[0] = ModelX.x; Mat[4] = ModelX.y; Mat[8] = ModelX.z;
    Mat[1] = ModelY.x; Mat[5] = ModelY.y; Mat[9] = ModelY.z;
    Mat[2] = ModelZ.x; Mat[6] = ModelZ.y; Mat[10] = ModelZ.z;
    //      glLoadMatrixf(Mat);
  }
  
  
  float    Ang, Color[4];
  int      Num;
  
  Color[3] = Alpha;
  
  glDisable(GL_CULL_FACE);
  glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
  
  // for do a for loop, from 1 to 4, for each the
  // axes. 1 = X, 2 = Y, 3 = Z, 4 = that yellow thing in the middle.
  for (Num = 1; Num <= 4; Num++) {
    if (Select) {
      glPushName(Num);
    }
    
    glPushMatrix();
    
    if (Num == 1) {
      Color[0] = 1.0;
      Color[1] = 0.0;
      Color[2] = 0.0;
      XVec.set(0,Size,0);
      YVec.set(0,0,Size);
    }
    if (Num == 2) {
      Color[0] = 0.0;
      Color[1] = 1.0;
      Color[2] = 0.0;
      XVec.set(Size,0,0);
      YVec.set(0,0,Size);
    }
    if (Num == 3) {
      Color[0] = 0.0;
      Color[1] = 0.0;
      Color[2] = 1.0;
      XVec.set(Size,0,0);
      YVec.set(0,Size,0);
    }
    if (Num == 4) {
      Color[0] = 1.0;
      Color[1] = 1.0;
      Color[2] = 0.0;
      
      {
        float    Mat[16];
        
        glGetFloatv(GL_MODELVIEW_MATRIX, Mat);
        
        XVec.set(Mat[0], Mat[4], Mat[8]);
        YVec.set(Mat[1], Mat[5], Mat[9]);
        
        XVec.normalize();
        YVec.normalize();
        XVec *= Size;
        YVec *= Size;
      }
      glLineStipple(1, 0x5555);
      glEnable(GL_LINE_STIPPLE);
      
      glScalef(1.1f,1.1f,1.1f);
    }
    
    glBegin(GL_LINE_LOOP);
    
    // Draw a fan of tris beginning from the top of the arrow head
    glColor4fv(Color);
    for (Ang=0;Ang<=360;Ang+=10) {
      float       posy, posz;
      MVector3    Pos;
      
      posy = (float)cos(Ang/180.0*M_PI);
      posz = (float)sin(Ang/180.0*M_PI);
      
      glColor4f(Color[0], Color[1], Color[2], Alpha);
      
      Pos = posy * XVec;
      Pos += posz * YVec;
      
      if (Num != 4 && Pos * ModelZ < 0)
        glColor4f(0, 0, 0, Alpha/4);
      
      glVertex3f(Pos.x, Pos.y, Pos.z);
      //         glVertex3f(0, posy, posz);
    }
    glEnd();
    
    glPopMatrix();
    
    if (Select)
      glPopName();
    
  }
  
  glDisable(GL_LINE_STIPPLE);
  
  glLineWidth(OldLineWidth);
  glPopMatrix();
  
  return 0;
}

int MBaseOpenGLWnd::DrawScaleIcon(float Size, float Alpha, float LineWidth, bool Select) {
  float OldLineWidth;
  glMatrixMode(GL_MODELVIEW);
  
  glGetFloatv(GL_LINE_WIDTH, &OldLineWidth);
  glLineWidth(LineWidth);
  int   Num;
  
  glDisable(GL_CULL_FACE);
  glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
  
  // for do a for loop, from 1 to 4, for each the
  // axes. 1 = X, 2 = Y, 3 = Z, 4 = that yellow thing in the middle.
  for (Num = 1; Num <= 4; Num++)
  {
    if (Select) {
      glPushName(Num);
    }
    
    glPushMatrix();
    
    if (Num == 1) {
      glColor4f(1,0,0,Alpha);
    }
    if (Num == 2) {
      glColor4f(0,1,0,Alpha);
      glRotatef(90, 0,0,1);
    }
    if (Num == 3) {
      glColor4f(0,0,1,Alpha);
      glRotatef(-90, 0,1,0);
    }
    if (Num == 4) {
      glColor4f(1,1,0,Alpha);
      
      DrawCube((float)(Size*0.2));
      
      if (Select) {
        glPopName();
      }
      
      glPopMatrix();
      
      break;
    }
    
    glBegin(GL_LINES);
    glVertex3f(0,0,0);
    glVertex3f((float)(0.8*Size),0,0  );
    glEnd();
    
    // Draw funky arrow heads
    glPushMatrix();
    glTranslatef((float)(Size*0.9),0,0);
    DrawCube((float)(Size*0.2));
    glPopMatrix();
    
    glPopMatrix();
    
    if (Select)
      glPopName();
  }
  
  glLineWidth(OldLineWidth);
  
  return 0;
}

void MBaseOpenGLWnd::GLDrawText(float x, float y, const MStr &str) {
  glRasterPos2d(x,y);
  glListBase(GLVIEW_FONTSTART);
  glCallLists(str.GetLength(), GL_UNSIGNED_BYTE, (LPCTSTR)str);
}

void MBaseOpenGLWnd::GLDrawText(float x, float y, int maxWidth, const MStr &str) {
  HDC   dc;
  SIZE  size;
  char  buf[4096];
  dc = ::GetDC(m_hWnd);
  
  // copy the string, including the null terminator.
  memcpy(buf, (LPCTSTR)str, str.GetLength()+1);
  
  ::GetTextExtentPoint32(dc, (LPCTSTR)str, str.GetLength(), &size);
  
  // if we go beyond the extents we have to adjust it
  if (size.cx > maxWidth) {
    int   ellStart;
    
    ellStart = str.GetLength()-1;
    
    // keep adjusting the text until we have an acceptable size
    while (ellStart > 0 && size.cx > maxWidth) {
      // copy the ellipses
      strcpy(&buf[ellStart], "...");
      ::GetTextExtentPoint32(dc, buf, ellStart+2, &size);
      ellStart--;
    }
    
    // if the ellipses takes up the entire text, don't bother drawing anything
    if (ellStart == 0) {
      ::ReleaseDC(m_hWnd, dc);
      return;
    }
    glRasterPos2d(x,y);
    glListBase(GLVIEW_FONTSTART);
    glCallLists(ellStart+4, GL_UNSIGNED_BYTE, buf);
  } else {
    glRasterPos2d(x,y);
    glListBase(GLVIEW_FONTSTART);
    glCallLists(str.GetLength(), GL_UNSIGNED_BYTE, (LPCTSTR)str);
  }
  
  
  ::ReleaseDC(m_hWnd, dc);
}


