#include <AztecMainPCH.h>

#include "resource.h"

#include "MenuItem.h"
#include "Keyboard.h"
#include "MdlConst.h"

#include "DlgGlobs.h"   // Global dialog variables.
#include "MdlGlobs.h"
#include "MdlMsgs.h"

#include "KeyFuncGeneral.h"
#include "KeyFuncMeshEdit.h"

#include "UndoGeneral.h"

#include "MEditableMesh.h"
#include "MAnimMesh.h"
#include "MMeshShape.h"
#include <mesh/MMeshModifier.h>

#include "MUIManager.h"

using std::vector;

#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

void InitMeshEditActionList(MActionListType *AL) {
  g_SysMan->logOutput("Initialising Mesh Edit Action List");
  
  // General
  AL->Add("KMeshFlipNormals", "Flip Normals", "Mesh", KMeshFlipNormals, g_MainDlg, ID_MESH_FACES_FLIPNORMALS);
  AL->Add("KMeshGrabUVs", "Grab UV Coordinates", "Mesh", KMeshGrabUVs, g_MainDlg, ID_MESH_GRAB_UVS);
  
  AL->Add("KMeshVerticesWeld", "Weld Vertices", "Mesh", KMeshVerticesWeld, g_MainDlg, ID_MESH_POINTS_WELD);
  AL->Add("KMeshVerticesSelectIsolated", "Select Isolated Vertices", "Mesh", KMeshVerticesSelectIsolated, g_MainDlg, ID_MESH_POINTS_SELECTISOLATED);
  AL->Add("KMeshVerticesSelectConnected", "Select Connected Vertices", "Mesh", KMeshVerticesSelectConnected, g_MainDlg, ID_MESH_POINTS_SELECTCONNECTED);
  AL->Add("KMeshVerticesDeleteIsolated", "Delete Isolated Vertices", "Mesh", KMeshVerticesDeleteIsolated, g_MainDlg, ID_MESH_POINTS_DELETEISOLATED);
  
  AL->Add("KMeshFacesCollapse", "Collapse Faces", "Mesh", KMeshFacesCollapse, g_MainDlg, ID_MESH_FACES_COLLAPSE);
  AL->Add("KMeshFacesDetach", "Detach Faces", "Mesh", KMeshFacesDetach, g_MainDlg, ID_MESH_FACES_DETACH);
  AL->Add("KMeshCollapseEdges", "Collapse Edges", "Mesh", KMeshEdgesCollapse, g_MainDlg, ID_MESH_EDGES_COLLAPSE);
  
  AL->Add("KMeshCollapseToMesh", "Collapse To Mesh", "Mesh", KMeshCollapseToMesh, g_MainDlg, ID_EDIT_COLLAPSETOMESH);
  
  AL->Add("KMeshEditCreateSubdivSurface", "Create Subdivided Surface", "Mesh", KMeshEditCreateSubdivSurface, g_MainDlg, ID_MESHEDIT_CREATESUBDIVIDEDOBJECT);

  AL->Add("combineMeshes", "Combine Selected Meshes", "Mesh", KCombineSelectedMeshes, g_MainDlg, ID_MESHEDIT_COMBINEMESHES);
  AL->Add("separateMeshes", "Seperate Selected Faces in Objects", "Mesh", KSeparateSelectedMeshes, g_MainDlg, ID_MESHEDIT_SEPARATEMESHES);
  AL->Add("flattenComponents", "Flattens the selection to flat plane", "Mesh", KFlattenComponents, g_MainDlg, ID_MESHEDIT_FLATTENCOMPONENTS);
}

int KMeshFlipNormals() {
	MBaseObjectTreePtr ObjTree;
	MTreeObjectNodePtr ObjNode;
	MBaseObjectPtr Obj;
	MSceneObjectPtr SceneObj;
	MEditableMeshPtr ObjMesh;
	bool FlipChildNormals = false;
	
	MParameterObjectPtr Var;
	Var = g_ProgSet.GetToolParameter("KMeshFlipNormals", "FlipChildNormals"); 
	if (Var != NULL) {
		int val;
		Var->getValueInteger(val);
		FlipChildNormals = (val != 0);
    Var = NULL;
	}
	
	ObjTree = g_SysMan->getScene()->getObjectList();
	
	ObjTree->beginIteration();
	while ((Obj = ObjTree->getNext()) != NULL) {
		ObjNode = ObjTree->getCurrentNode();
		if (Obj->isFlagged(OBJECTFLAG_SELECTED) || 
			(ObjNode->isChildOfFlagged(OBJECTFLAG_SELECTED) && FlipChildNormals)) 
		{
			SceneObj = AZTEC_CAST(MSceneObject, Obj);
			if (SceneObj == NULL) {
				// Note: Should this really be an error?  Suppose a material is selected,
				// then it should just be ignored, right?
				// g_SysMan->logOutput("Error: Flip Normals failed, only operates on Scene Objects");
				continue;
			}
			
			MShapeObjectPtr shapeObj = SceneObj->getShapeObject();
			if (shapeObj == NULL) {
				g_SysMan->logOutput("Error: Flip Normals failed on '%s', only operates on Objects with a Shape and Mesh output", (LPCTSTR)SceneObj->getName());
				continue;
			}
			
			ObjMesh = AZTEC_CAST(MEditableMesh, shapeObj->convertToMesh());
			
			if (ObjMesh == NULL) {
				g_SysMan->logOutput("Error: Flip Normals failed on '%s', only operates on Objects with a Mesh output", (LPCTSTR)SceneObj->getName());
				continue;
			}
			
			AztecFlags FlagsReqd;
			
			FlagsReqd = 0xFF;
			
			if (ObjMesh->isInComponentMode()) {
				if (ObjMesh->isInComponentMode(MComponentisedObject::FACET_TYPE)) {
					FlagsReqd = TRIANGLE_SELECTED;
				} else {
					FlagsReqd = 0;
				}
			}
			
			ObjMesh->flipNormals(FlagsReqd);
		}
	}
	ObjTree->endIteration();
	
	g_MainDlg->SendMessage(MM_UPDATEVIEWPORTS, MMC_UPDATE_ALL, 0);
	g_MainDlg->SendMessage(MM_UPDATECHANNELBAR, 0, 0);
	
	return 1;
}


int KMeshGrabUVs() {
  MBaseObjectPtr Obj;
  MVector3       ViewNormal;
  MMatrix4       ViewMatrix;
  int            i,n, FirstTime = 0;
  COpenGLWnd     *CurGLView;
  
  
  ViewNormal.set(0,0,1);
  
  // Get the viewing normal from the current view.
  CurGLView = AZTEC_CAST(COpenGLWnd, g_CurView);
  ViewMatrix.identity();
  
  if (CurGLView) {
    CurGLView->UpdateXFormMatricies();
    ViewMatrix = MMatrix4(CurGLView->m_ModelMat);
    
    // We have to transpose it, because opengl uses a different
    // row majorness than us.
    ViewMatrix.transpose();

    ViewNormal.set(ViewMatrix.m[0][2], ViewMatrix.m[1][2], ViewMatrix.m[2][2] );
    ViewNormal.normalize();
  }  
  
  g_Scene->getObjectList()->beginIteration();
  
  while (( Obj = g_Scene->getObjectList()->getNext() ) != NULL ) {
    if (!Obj->isFlagged(OBJECTFLAG_SELECTED)) {
      continue;
    }
    
    MSceneObjectPtr sceneObject;
    MTreeObjectNodePtr sceneObjectNode;
    MMeshPtr Mesh, SkinMesh;
    MEditableMeshPtr editableSkinMesh;
    MMatrix4 worldMatrix, objToViewMatrix;
    
    sceneObject = AZTEC_CAST(MSceneObject, Obj);
    
    if (sceneObject == NULL) {
      continue;
    }
    
    if (sceneObject->getShapeObject() != NULL) {
      Mesh = sceneObject->getShapeObject()->convertToMesh();
    }
    
    if (Mesh == NULL) {
      continue;
    }
    
    sceneObjectNode = g_Scene->getObjectList()->getCurrentNode(); 
    g_Scene->getWorldTransformMatrix(sceneObjectNode, worldMatrix);

    objToViewMatrix = worldMatrix;
    objToViewMatrix *= ViewMatrix;

    ViewNormal.set(objToViewMatrix.m[0][2], objToViewMatrix.m[1][2], objToViewMatrix.m[2][2] );
    ViewNormal.normalize();

    SkinMesh = Mesh->getTextureMesh();
    editableSkinMesh = AZTEC_CAST(MEditableMesh, SkinMesh);;
    
    // if we do not have a skin mesh, add one in.
    if (SkinMesh == NULL) {
      editableSkinMesh = new MEditableMesh;
      editableSkinMesh->setFromMesh(Mesh);
      Mesh->setTextureMesh(editableSkinMesh);
      FirstTime = 1;
      
      // otherwise check to see if we have an editable one
    } else if (editableSkinMesh == NULL) {
      // TODO: We should add in an edit mesh modifier here
      editableSkinMesh = new MEditableMesh;
      editableSkinMesh->setFromMesh(Mesh);
      Mesh->setTextureMesh(editableSkinMesh);
      FirstTime = 1;
    }
    
    {      
      bool     NothingSelected = true;
      
      // go through the triangles of the mesh, any that are selected, flag them, then separate the faces.
      // Must do this twice, once for front facing, and once for back facing
      
      // Check to see if nothing is selected first. If there is nothing, just assume all is selected
      for (i=0; i<Mesh->getNumTris(); i++) {
        MTriangle const *MeshTri;
        
        MeshTri = Mesh->getTriangle(i);
        
        if (MeshTri->isFlagged(TRIANGLE_SELECTED)) {
          NothingSelected = false;
          break;
        }
      }
      
      for (i=0; i<Mesh->getNumTris(); i++) {
        MTriangle const *MeshTri;
        
        MeshTri = Mesh->getTriangle(i);
        
        if (Mesh->isTriangleFlagged(i, TRIANGLE_SELECTED) || NothingSelected) {
          if (ViewNormal * MeshTri->getNormal() > 0.0) {
            editableSkinMesh->setTriangleFlag(i, TRIANGLE_FLAGFORCHANGE);
            editableSkinMesh->unsetTriangleFlag(i, TRIANGLE_FLAGFORCHANGE2);
          } else {
            editableSkinMesh->unsetTriangleFlag(i, TRIANGLE_FLAGFORCHANGE);
            editableSkinMesh->setTriangleFlag(i, TRIANGLE_FLAGFORCHANGE2);
          }
        } else {
          editableSkinMesh->unsetTriangleFlag(i, TRIANGLE_FLAGFORCHANGE);
          editableSkinMesh->setTriangleFlag(i, TRIANGLE_FLAGFORCHANGE2);
        }
      }
      editableSkinMesh->separateFaces(TRIANGLE_FLAGFORCHANGE);
      editableSkinMesh->separateFaces(TRIANGLE_FLAGFORCHANGE2);
      
    }
    
    MVector3    Min, Max, Size;
    float       LargeSize;
    
    // Get the minimum and maximum mesh extents, using the camera transform.
    Mesh->getMeshExtents(Min, Max, objToViewMatrix);
    Size = Max - Min;
    
    LargeSize = Size.x;
    if (Size.y > LargeSize) {
      LargeSize = Size.y;
    }
    if (Size.z > LargeSize) {
      LargeSize = Size.z;
    }
    
    LargeSize = LargeSize * 2;    // Double the size so we can tile a front/back thing
    
    // Now go through and assign uv coords to the mesh
    for (i=0; i<Mesh->getNumTris(); i++) {
      MTriangle  const *MeshTri;
      
      MeshTri = Mesh->getTriangle(i);
      editableSkinMesh->unsetTriangleFlag(i, TRIANGLE_SELECTED);
      if (editableSkinMesh->isTriangleFlagged(i, TRIANGLE_FLAGFORCHANGE | TRIANGLE_FLAGFORCHANGE2)) {
        int meshVertIndex;
        int skinVertIndex;
        editableSkinMesh->setTriangleFlag(i, TRIANGLE_SELECTED);
        
        for (n = 0; n<3; n++) {
          MVector3    Pos;
          
          meshVertIndex = Mesh->getTriVert(i, n);
          skinVertIndex = editableSkinMesh->getTriVert(i, n);
          
          Pos = (objToViewMatrix*Mesh->getVertexPosition(meshVertIndex)) - Min;
          Pos *= (float)(1.0 / LargeSize);
          if (editableSkinMesh->isTriangleFlagged(i, TRIANGLE_FLAGFORCHANGE2))
            Pos.x += 0.5f;
          
          Pos.y = 1.0f - Pos.y;
          
          editableSkinMesh->setVertexPosition(skinVertIndex, Pos);
        }
        
      } else if (FirstTime) {
        int skinVertIndex;
        for (n = 0; n<3; n++) {
          skinVertIndex = editableSkinMesh->getTriVert(i, n);
          editableSkinMesh->setVertexPosition(skinVertIndex, MVector3(0,0,0));
        }
      }
    }
   }
   g_Scene->getObjectList()->endIteration();
   
   g_MainDlg->SendMessage(MM_UPDATEVIEWPORTS, MMC_UPDATE_ALL, 0);
   g_MainDlg->SendMessage(MM_UPDATECHANNELBAR, 0, 0);
   
   return 1;
}

int KMeshVerticesWeld() {
  MBaseObjectPtr Obj;
  MSceneObjectPtr SceneObj;
  MEditableMeshPtr Mesh;
  
  g_Scene->getObjectList()->beginIteration();
  while (( Obj = g_Scene->getObjectList()->getNext() ) != NULL ) {
    if (!Obj->isFlagged(OBJECTFLAG_SELECTED)) {
      continue;
    }
    
    SceneObj = AZTEC_CAST(MSceneObject, Obj);
    
    if (SceneObj == NULL) {
      continue;
    }
    
    Mesh = AZTEC_CAST(MEditableMesh, SceneObj->getShapeObject()->convertToMesh());
    
    if (Mesh == NULL) {
      continue;
    }
    
    Mesh->collapseVertices(VERTEX_SELECTED);
  }
  g_Scene->getObjectList()->endIteration();
  
  g_MainDlg->SendMessage(MM_UPDATEVIEWPORTS, MMC_UPDATE_ALL, 0);
  g_MainDlg->SendMessage(MM_UPDATECHANNELBAR, 0, 0);
  return 1;
}

int KMeshVerticesSelectIsolated() {
  g_SysMan->logOutput("Error: Function not implmented");
  return 1;
}


bool selectTouchingComponents(MBaseObjectPtr obj, AztecFlags component) {
  // iterate over all the selected objects that are in component mode.
  MSceneObjectPtr sceneObj = AZTEC_CAST(MSceneObject, obj);
  MComponentisedObjectPtr compObj;
  
  if (sceneObj != NULL) {
    compObj = sceneObj->getComponentObject();
  } else {
    compObj = AZTEC_CAST(MComponentisedObject, obj);
  }
  
  if (compObj != NULL) {
    return compObj->growFlagged(MComponentisedObject::POINT_TYPE);
  }
  
  return false;
}


int KMeshVerticesSelectConnected() {
  // iterate over all the selected objects that are in component mode.
  MBaseObjectPtr obj;
  g_Scene->getObjectList()->beginIteration();
  while (( obj = g_Scene->getObjectList()->getNext() ) != NULL ) {
    
    MSceneObjectPtr sceneObj = AZTEC_CAST(MSceneObject, obj);
    MComponentisedObjectPtr compObj;
    
    if (sceneObj != NULL) {
      compObj = sceneObj->getComponentObject();
    } else {
      compObj = AZTEC_CAST(MComponentisedObject, obj);
    }
    
    if (compObj != NULL) {
      compObj->flagConnected(MComponentisedObject::POINT_TYPE);
    }
  }
  g_Scene->getObjectList()->endIteration();
  
  
  return 1;
}

int KMeshVerticesDeleteIsolated()
{
  g_SysMan->logOutput("Error: Function not implmented");
  return 1;
}


int KMeshCollapseFaces() {
  g_SysMan->logOutput("Error: Function not implmented");
  return 1;
}

int KMeshFacesDetach() {
  MBaseObjectPtr Obj;
  MSceneObjectPtr SceneObj;
  MEditableMeshPtr Mesh;
  
  g_Scene->getObjectList()->beginIteration();
  while (( Obj = g_Scene->getObjectList()->getNext() ) != NULL ) {
    if (!Obj->isFlagged(OBJECTFLAG_SELECTED)) {
      continue;
    }
    
    SceneObj = AZTEC_CAST(MSceneObject, Obj);
    
    if (SceneObj == NULL) {
      continue;
    }
    
    Mesh = AZTEC_CAST(MEditableMesh, SceneObj->getShapeObject()->convertToMesh());
    
    if (Mesh == NULL) {
      continue;
    }
    
    Mesh->separateFaces(TRIANGLE_SELECTED);
  }
  g_Scene->getObjectList()->endIteration();
  
  g_MainDlg->SendMessage(MM_UPDATEVIEWPORTS, MMC_UPDATE_ALL, 0);
  g_MainDlg->SendMessage(MM_UPDATECHANNELBAR, 0, 0);
  return 1;
}

int KMeshFacesCollapse()
{
  g_SysMan->logOutput("Error: Function not implmented");
  return 1;
}

int KMeshEdgesCollapse() {
  g_SysMan->logOutput("Error: Function not implmented");
  return 1;
}

int KMeshCollapseToMesh() {
  MBaseObjectPtr Obj;
  MSceneObjectPtr SceneObj;
  vector<MNamedObjectPtr> objsToAdd;
  
  g_Scene->getObjectList()->beginIteration();
  
  while (( Obj = g_Scene->getObjectList()->getNext() ) != NULL ) {
    MShapeObjectPtr srcShape;
    MMeshPtr SrcMesh;
    if (!Obj->isFlagged(OBJECTFLAG_SELECTED)) {
      continue;
    }
    
    SceneObj = AZTEC_CAST(MSceneObject, Obj);
    if (SceneObj == NULL) {
      continue;
    }
    
    srcShape = SceneObj->getShapeObject();
    if (srcShape != NULL) {
      SrcMesh = AZTEC_CAST(MMesh, srcShape->convertToMesh());
    }
    
    if (SrcMesh == NULL) {
      continue;
    }
    
    
    MAnimMeshPtr NewMesh;
    MMeshShapePtr meshShape;
    
    NewMesh = new MAnimMesh;
    NewMesh->setFrom(SrcMesh);
    
    NewMesh->setName( SceneObj->getName() + "Mesh" );
    
    meshShape = new MMeshShape(NewMesh);
    meshShape->setName(SceneObj->getName() + "Shape");
    
    SceneObj->setShapeObject(meshShape);

    MComponentisedObjectPtr compObj = SceneObj->getComponentObject();
    
    if (compObj != NULL) {
      compObj->setComponentMode(MUIManager::getComponentMode());
    }
    
    objsToAdd.push_back(NewMesh);
    objsToAdd.push_back(meshShape);
  }
  g_Scene->getObjectList()->endIteration();
  
  for (int n = 0; n < objsToAdd.size(); ++n) {
    g_Scene->addObject(objsToAdd[n]);
  }
  objsToAdd.clear();
  
  g_MainDlg->SendMessage(MM_UPDATEVIEWPORTS, MMC_UPDATE_ALL, 0);
  g_MainDlg->SendMessage(MM_UPDATECHANNELBAR, 0, 0);
  return 1;
}

int KMeshEditCreateSubdivSurface() {
  // attempt to create a subdivided object for each of the selected objects.
  MBaseObjectPtr obj;
  MSceneObjectPtr sceneObj;
  
  g_Scene->getObjectList()->beginIteration();
  
  while (( obj = g_Scene->getObjectList()->getNext() ) != NULL ) {
    sceneObj = AZTEC_CAST(MSceneObject, obj);
    if (sceneObj == NULL || !obj->isFlagged(OBJECTFLAG_SELECTED)) {
      continue;
    }
    
    MTreeObjectNodePtr objNode = g_Scene->getObjectList()->getCurrentNode();
    MMeshModifierPtr meshModifier;
    
    meshModifier = AZTEC_CAST(MMeshModifier, MSystemManager::getInstance()->getPluginManager()->createObjectFromDefault("MPrimitiveSubdiv"));
    
    // if we couldn't make the sub divided object, quit.
    if (meshModifier == NULL) {
      g_SysMan->logOutput("Could not create a Subdivided object, the Primtive Class couldn't be found.");
      g_Scene->getObjectList()->endIteration();
      return 0;
    }
    
    // make the old object wire frame only
    sceneObj->setParamByName("drawWire", "yes");

    MShapeObjectPtr shapeObj = new MMeshShape();
    MSceneObjectPtr newSceneObj = new MSceneObject();
    shapeObj->setName(sceneObj->getName() + "Shape");
    newSceneObj->setName(sceneObj->getName() + "Subdiv");
    newSceneObj->setShapeObject(shapeObj);
    meshModifier->setName(sceneObj->getName() + "SubdivCreator");

    meshModifier->getInputParameter()->setInputParameter(sceneObj->findParameter("shape"));
    shapeObj->findParameter("inMesh")->setInputParameter(meshModifier->getOutputParameter());

    g_Scene->addObject(newSceneObj, objNode);
    g_Scene->addObject(meshModifier);
  
    g_Scene->selectNone();
    g_Scene->selectObject(sceneObj);

  }
  g_Scene->getObjectList()->endIteration();
  
  g_MainDlg->SendMessage(MM_UPDATEVIEWPORTS, MMC_UPDATE_ALL, 0);
  g_MainDlg->SendMessage(MM_UPDATECHANNELBAR, 0, 0);
  return 1;
}

int KCombineSelectedMeshes() {
  
  MMeshShapePtr newShapeObject; // this is the new shape object
  MEditableMeshPtr newMesh; // this is the new mesh
  vector<MSceneObjectPtr> srcMeshes;
  MSceneObjectPtr masterObj; // this is the object to receive all our meshes

  // get the meshes we will be using as inputs
  {
    MBaseObjectPtr obj;
    MSceneObjectPtr sceneObj;
    g_Scene->getSelectedObjectList()->beginIteration();

    while ((obj = g_Scene->getSelectedObjectList()->getNext()) != NULL) {
      sceneObj = AZTEC_CAST(MSceneObject, obj);

      // we can only operate on scene objects and those with shape objects
      if (sceneObj == NULL || sceneObj->getShapeObject() == NULL) {
        continue;
      }

      // if we haven't got a master object yet, make it so
      if (masterObj == NULL) {
        masterObj = sceneObj;
      }

      srcMeshes.push_back(sceneObj);
    }

    g_Scene->getSelectedObjectList()->endIteration();
  }

  // if we only have one or no input meshes, abort.
  if (srcMeshes.size() <= 1) {
    g_SysMan->logOutput("Error: At least two objects with meshes must be selected.");
    return 0;
  }

  newShapeObject = new MMeshShape();
  newMesh = new MEditableMesh();

  MMatrix4 destWorldInv, srcWorld, transform;

  g_Scene->getWorldTransformMatrix( g_Scene->getObjectList()->findObject(masterObj), destWorldInv);
  destWorldInv.inverse();
  for (std::vector<MSceneObjectPtr>::iterator it = srcMeshes.begin(); it != srcMeshes.end(); ++it) {
    MMeshPtr srcMesh = (*it)->getShapeObject()->convertToMesh();
    MVector3 *verts;
    MTriangle *tris;

    verts = new MVector3[srcMesh->getNumVerts()];
    tris = new MTriangle[srcMesh->getNumTris()];

    g_Scene->getWorldTransformMatrix( g_Scene->getObjectList()->findObject(*it), srcWorld);
    transform = srcWorld * destWorldInv;

    // loop over the vertices and triangles
    for (int n = 0; n < srcMesh->getNumVerts(); ++n) {
      verts[n] = transform * srcMesh->getVertexPosition(n);
    }

    for (int n = 0; n < srcMesh->getNumTris(); ++n) {
      tris[n] = *srcMesh->getTriangle(n);
    }

    newMesh->addVertsAndTriangles(verts, tris, srcMesh->getNumVerts(), srcMesh->getNumTris());

    if (*it != masterObj) {
      g_Scene->deleteObject(*it);
    }

    delete[] verts;
    delete[] tris;
  }

  newShapeObject->setName(  masterObj->getName() + "Shape" );
  newMesh->setName(  masterObj->getName() + "Mesh" );

  g_Scene->addObject(newShapeObject);
  g_Scene->addObject(newMesh);

  newShapeObject->setMeshObject(newMesh);
  masterObj->setShapeObject(newShapeObject);

  g_MainDlg->SendMessage(MM_UPDATEVIEWPORTS, MMC_UPDATE_ALL, 0);
  g_MainDlg->SendMessage(MM_UPDATECHANNELBAR, 0, 0);
  return 1;
}

int KSeparateSelectedMeshes() {
  if ( Aztec::MUIManager::getComponentMode() != Aztec::MComponentisedObject::FACET_TYPE) {
    g_SysMan->logOutput("Error: Must be in face mode to separate meshes.");
  }

  MBaseObjectPtr obj;
  MSceneObjectPtr sceneObj;
  g_Scene->getSelectedObjectList()->beginIteration();
  vector<MSceneObjectPtr> objectsToAdd;

  MSceneObjectPtr newObject;
  MMeshShapePtr newMeshShape;
  MEditableMeshPtr newMesh;

  newObject = new MSceneObject();
  newObject->setName("meshObj");
  newMeshShape = new MMeshShape();
  newMeshShape->setName("meshObjShape");

  newMesh = new MEditableMesh();
  newMesh->setName("meshObjMesh");
  newMeshShape->setMeshObject(newMesh);
  newObject->setShapeObject(newMeshShape);


  while ((obj = g_Scene->getSelectedObjectList()->getNext()) != NULL) {
    sceneObj = AZTEC_CAST(MSceneObject, obj);

    // we can only operate on scene objects and those with mesh shape objects
    if (sceneObj == NULL || AZTEC_CAST(MMeshShape, sceneObj->getShapeObject()) == NULL) {
      continue;
    }
    if ( sceneObj->getComponentObject() == NULL || 
      !sceneObj->getComponentObject()->isInComponentMode(MComponentisedObject::FACET_TYPE) ) {
      continue;
    }

    MEditableMeshPtr srcMesh = AZTEC_CAST(MEditableMesh, AZTEC_CAST(MMeshShape, sceneObj->getShapeObject())->getMeshObject());
    MVector3 *newVerts;
    MTriangle *newTris;
    MMatrix4 worldXForm;

    g_Scene->getWorldTransformMatrix(g_Scene->getObjectList()->findObject(sceneObj), worldXForm);

    vector<int> vertIndicies;

    vertIndicies.resize(srcMesh->getNumVerts());

    // clear the flags on the vertices
    srcMesh->unsetVertexFlags(VERTEX_FLAGFORCHANGE);

    // loop over the triangles and flag touching verts
    srcMesh->flagVerticesOnTris(TRIANGLE_SELECTED, VERTEX_FLAGFORCHANGE);

    // loop over the vertices and map from old index numbers to new ones
    int vertexCount = 0;
    int triangleCount = srcMesh->getNumTrisFlagged(TRIANGLE_SELECTED);

    for (int i = 0; i < srcMesh->getNumVerts(); ++i) {
      if (srcMesh->isVertexFlagged(i, VERTEX_FLAGFORCHANGE)) {
        vertIndicies[i] = vertexCount;
        ++vertexCount;
      } else {
        vertIndicies[i] = -1;
      }
    }

    newVerts = new MVector3[vertexCount];
    newTris = new MTriangle[triangleCount];


    // loop over the vertices, and copy them appropriately.
    for (int i = 0; i < srcMesh->getNumVerts(); ++i) {
      if (vertIndicies[i] != -1) {
        MVector3 pos = newVerts[vertIndicies[i]] = srcMesh->getVertexPosition(i);
        pos = worldXForm * pos;
        newVerts[vertIndicies[i]] = pos;
      }
    }

    // loop over the triangles, and adjust the vertices
    triangleCount = 0;
    for (int i = 0; i < srcMesh->getNumTris(); ++i) {
      MTriangle const *t = srcMesh->getTriangle(i);

      if (t->isFlagged(TRIANGLE_SELECTED)) {
        newTris[triangleCount] = *t;
        newTris[triangleCount].setVertex(0, vertIndicies[newTris[triangleCount].getVertex(0)]);
        newTris[triangleCount].setVertex(1, vertIndicies[newTris[triangleCount].getVertex(1)]);
        newTris[triangleCount].setVertex(2, vertIndicies[newTris[triangleCount].getVertex(2)]);
        ++triangleCount;
      }
    }

    newMesh->addVertsAndTriangles(newVerts, newTris, vertexCount, triangleCount);

    srcMesh->deleteTriangleFlag(VERTEX_SELECTED);
    sceneObj->getComponentObject()->setComponentMode(MComponentisedObject::OBJECT_TYPE);

    delete[] newVerts;
    delete[] newTris;
  }
  g_Scene->getSelectedObjectList()->endIteration();

  g_Scene->addObject(newObject);

  g_Scene->selectNone();
  g_Scene->selectObject(newObject);

  newMesh->calculateNormals();
  newMesh->setTriangleFlags(TRIANGLE_SELECTED);

  g_MainDlg->SendMessage(MM_UPDATEVIEWPORTS, MMC_UPDATE_ALL, 0);
  g_MainDlg->SendMessage(MM_UPDATECHANNELBAR, 0, 0);
  return 1;
}

int KFlattenComponents () {
  if ( Aztec::MUIManager::getComponentMode() != Aztec::MComponentisedObject::FACET_TYPE && 
       Aztec::MUIManager::getComponentMode() != Aztec::MComponentisedObject::POINT_TYPE ) {
    g_SysMan->logOutput("Error: Must be in Facet or Point mode to flatten components.");
  }

  MBaseObjectPtr obj;
  MSceneObjectPtr sceneObj;
  
  g_Scene->getObjectList()->beginIteration();
  
  while (( obj = g_Scene->getObjectList()->getNext() ) != NULL ) {
    MShapeObjectPtr srcShape;
    MEditableMeshPtr srcMesh;

    if (!obj->isFlagged(OBJECTFLAG_SELECTED)) {
      continue;
    }
    
    sceneObj = AZTEC_CAST(MSceneObject, obj);
    if (sceneObj == NULL) {
      continue;
    }
    
    srcShape = sceneObj->getShapeObject();
    if (srcShape != NULL) {
      srcMesh = AZTEC_CAST(MEditableMesh, srcShape->convertToMesh());
    }
    
    if (srcMesh == NULL) {
      continue;
    }

    // now we go over the components, and flatten them.
    MVector3 normal, centre;
    int numSelected = 0;

    if (Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::FACET_TYPE) {
      for (int n = 0; n < srcMesh->getNumTris(); ++n) {
        MTriangle const *tri = srcMesh->getTriangle(n);

        if (tri->isFlagged(TRIANGLE_SELECTED)) {
          normal += tri->getNormal();
          centre += tri->getCentre();
          ++numSelected;
        }

      }

      normal.normalize();
      centre /= numSelected;

      // now that we have our normal and centre, we can flatten the faces
      for (int n = 0; n < srcMesh->getNumTris(); ++n) {
        MTriangle const *tri = srcMesh->getTriangle(n);

        if (tri->isFlagged(TRIANGLE_SELECTED)) {
          MVector3 newPos, oldPos, diff;

          for (int i = 0; i < 3; ++i) {
            oldPos = srcMesh->getVertexPosition(tri->getVertex(i));
            diff = oldPos - centre;
            diff = normal * ((oldPos - centre) * normal);
            newPos = oldPos - diff;
            srcMesh->setVertexPosition(tri->getVertex(i), newPos);
          }
        }

      }

    } else if (Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::POINT_TYPE) {
      for (int n = 0; n < srcMesh->getNumVerts(); ++n) {
        if (srcMesh->isVertexFlagged(n, VERTEX_SELECTED)) {
          normal += srcMesh->getVertexNormal(n);
          centre += srcMesh->getVertexPosition(n);
          ++numSelected;
        }

      }

      normal.normalize();
      centre /= numSelected;

      // now that we have our normal and centre, we can flatten the faces
      for (int n = 0; n < srcMesh->getNumVerts(); ++n) {
        if (srcMesh->isVertexFlagged(n, VERTEX_SELECTED)) {
          MVector3 newPos, oldPos, diff;

          oldPos = srcMesh->getVertexPosition(n);
          diff = oldPos - centre;
          diff = normal * ((oldPos - centre) * normal);
          newPos = oldPos - diff;
          srcMesh->setVertexPosition(n, newPos);
        }
      }
    }
  
    srcMesh->calculateNormals();
  }

  g_Scene->getObjectList()->endIteration();

  g_MainDlg->SendMessage(MM_UPDATEVIEWPORTS, MMC_UPDATE_ALL, 0);
  g_MainDlg->SendMessage(MM_UPDATECHANNELBAR, 0, 0);
  return 1;
}
