#include "StdAfx.h"

#include <math.h>
#include "ModelTypes.h"
#include "scripting/ScriptRGBAFloat.h"
#include "scripting/ScriptVector3.h"
#include "scripting/ScriptFuncs.h"

#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

namespace Aztec {  

static JSBool
set(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	MRGBAFloat *vec = (MRGBAFloat *)JS_GetInstancePrivate(cx, obj, &aztecRGBAFloat_class, NULL);
	if (vec == NULL || argc < 1 || argc > 4) {
		return JS_FALSE;
	}

	if (argc == 1) {
		// Either an array of doubles or another RGBAFloat
		if (JSVAL_IS_OBJECT(*argv)) {

			// First see if it is another RGBAFloat
			JSObject *obj2 = JSVAL_TO_OBJECT(*argv);
			MRGBAFloat *vec2 = (MRGBAFloat *)JS_GetInstancePrivate(cx, obj2, &aztecRGBAFloat_class, NULL);
			if (vec2 != NULL) {
				*vec = *vec2;
				*rval = JSVAL_VOID;
				return JS_TRUE;
			}

			// See if the argument is an array of doubles
			jsuint lengthp;
			if (JS_IsArrayObject(cx, obj2) == JS_TRUE &&
				JS_GetArrayLength(cx, obj2, &lengthp) == JS_TRUE &&
				lengthp >= 4) {
				jsval v1, v2, v3, v4;
        bool successful = false;

				if (JS_GetElement(cx, obj2, 0, &v1) == JS_TRUE &&
				    JS_GetElement(cx, obj2, 1, &v2) == JS_TRUE &&
				    JS_GetElement(cx, obj2, 2, &v3) == JS_TRUE &&
				    JSVAL_IS_NUMBER(v1) && 
				    JSVAL_IS_NUMBER(v2) &&
				    JSVAL_IS_NUMBER(v3) ) {

					double r, g, b, a;

					if (JS_GetElement(cx, obj2, 3, &v4) == JS_TRUE && 
					    JSVAL_IS_NUMBER(v4)) {
						// If retrieving the value fails, default to the value of 1.0
						if (!getDoubleVal(cx, v4, a)) {
							a = 1.0;
						}
					}

					if (getDoubleVal(cx, v1, r) &&
					    getDoubleVal(cx, v2, g) &&
					    getDoubleVal(cx, v3, b)) {
						vec->set((float)r, (float)g, (float)b, (float)a);
					}
				}

				if (!successful) {
					*rval = JSVAL_VOID;
					return JS_TRUE;
				}
			}
		}
	} else if (argc == 3) {
		// If there are three arguments and all three are doubles, then
		// assign them to the components of the vector.
		if (JSVAL_IS_NUMBER(*argv) && JSVAL_IS_NUMBER(*(argv+1)) &&
			JSVAL_IS_NUMBER(*(argv+2))) {
			double r, g, b;
			if (getDoubleVal(cx, *argv, r) &&
				getDoubleVal(cx, *(argv+1), g) &&
				getDoubleVal(cx, *(argv+2), b)) {
				vec->set((float)r, (float)g, (float)b, 1.0);
				*rval = JSVAL_VOID;
				return JS_TRUE;
			}
		}
	} else if (argc == 4) {
		// If there are three arguments and all three are doubles, then
		// assign them to the components of the vector.
		if (JSVAL_IS_NUMBER(*argv) && JSVAL_IS_NUMBER(*(argv+1)) &&
			JSVAL_IS_NUMBER(*(argv+2)) && JSVAL_IS_NUMBER(*(argv+3))) {
			double r, g, b, a;
			if (getDoubleVal(cx, *argv, r) &&
				getDoubleVal(cx, *(argv+1), g) &&
				getDoubleVal(cx, *(argv+2), b) &&
				getDoubleVal(cx, *(argv+3), a)) {
				vec->set((float)r, (float)g, (float)b, (float)a);
				*rval = JSVAL_VOID;
				return JS_TRUE;
			}
		}
	}

	// Don't know how to convert the arguments into the components of
	// a RGBAFloat.
	return JS_FALSE;
}

static JSPropertySpec aztecRGBAFloat_props[] = {
	{"r",	0,	JSPROP_ENUMERATE},
	{"g",	1,	JSPROP_ENUMERATE},
	{"b",	2,	JSPROP_ENUMERATE},
	{"a",	3,	JSPROP_ENUMERATE},
	{0}
};

static JSFunctionSpec aztecRGBAFloat_methods[] = {
	{"set",				set,		1},
	{0}
};

static JSBool
aztecRGBAFloat_getProperty(JSContext *cx, JSObject *obj, jsval id, jsval *vp)
{
	MRGBAFloat *vec = (MRGBAFloat *)JS_GetInstancePrivate(cx, obj, &aztecRGBAFloat_class, NULL);
	if (vec == NULL) {
		return JS_FALSE;
	}

	if (JSVAL_IS_STRING(id)) {
		char *propStr = JS_GetStringBytes(JS_ValueToString(cx, id));
		if (strcmp(propStr, "r") == 0) {
			return JS_NewDoubleValue(cx, vec->x, vp);
		} else if (strcmp(propStr, "g") == 0) {
			return JS_NewDoubleValue(cx, vec->y, vp);
		} else if (strcmp(propStr, "b") == 0) {
			return JS_NewDoubleValue(cx, vec->z, vp);
		} else if (strcmp(propStr, "a") == 0) {
			return JS_NewDoubleValue(cx, vec->w, vp);
		} else {
			return JS_TRUE;
		}
	} else if (JSVAL_IS_INT(id)) {
		int index = JSVAL_TO_INT(id);
		if (index == 0) {
			return JS_NewDoubleValue(cx, vec->x, vp);
		} else if (index == 1) {
			return JS_NewDoubleValue(cx, vec->y, vp);
		} else if (index == 2) {
			return JS_NewDoubleValue(cx, vec->z, vp);
		} else if (index == 3) {
			return JS_NewDoubleValue(cx, vec->w, vp);
		} else {
			return JS_TRUE;
		}
	} else {
		return JS_FALSE;
	}
}

static JSBool
aztecRGBAFloat_setProperty(JSContext *cx, JSObject *obj, jsval id, jsval *vp)
{
	MRGBAFloat *vec = (MRGBAFloat *)JS_GetInstancePrivate(cx, obj, &aztecRGBAFloat_class, NULL);
	if (vec == NULL) {
		return JS_FALSE;
	}

	double t;
	if (!getDoubleVal(cx, *vp, t)) {
		return JS_FALSE;
	}

	if (JSVAL_IS_STRING(id)) {
		char *propStr = JS_GetStringBytes(JS_ValueToString(cx, id));
		if (strcmp(propStr, "r") == 0) {
			vec->x = (float)t;
			return JS_TRUE;
		} else if (strcmp(propStr, "g") == 0) {
			vec->y = (float)t;
			return JS_TRUE;
		} else if (strcmp(propStr, "b") == 0) {
			vec->z = (float)t;
			return JS_TRUE;
		} else if (strcmp(propStr, "a") == 0) {
			vec->w = (float)t;
			return JS_TRUE;
		} else {
			return JS_FALSE;
		}
	} else if (JSVAL_IS_INT(id)) {
		int index = JSVAL_TO_INT(id);
		if (index == 0) {
			vec->x = (float)t;
			return JS_TRUE;
		} else if (index == 1) {
			vec->y = (float)t;
			return JS_TRUE;
		} else if (index == 2) {
			vec->z = (float)t;
			return JS_TRUE;
		} else if (index == 3) {
			vec->w = (float)t;
			return JS_TRUE;
		} else {
			return JS_FALSE;
		}
	} else {
		return JS_FALSE;
	}
}

static JSBool
aztecRGBAFloat_convert(JSContext *cx, JSObject *obj, JSType typ, jsval *vp)
{
	MRGBAFloat *vec = (MRGBAFloat *)JS_GetInstancePrivate(cx, obj, &aztecRGBAFloat_class, NULL);
	if (vec == NULL) {
		return JS_FALSE;
	}
	
    switch (typ) {
    case JSTYPE_OBJECT:
        *vp = OBJECT_TO_JSVAL(obj);
        return JS_TRUE;
		
    case JSTYPE_VOID:
    case JSTYPE_STRING:
		{
			char buf[128];
			sprintf(buf, "%f %f %f %f", vec->x, vec->y, vec->z, vec->w);
			JSString *str = JS_NewStringCopyZ(cx, buf);
			*vp = STRING_TO_JSVAL(str);
			return JS_TRUE;
		}
		
	case JSTYPE_FUNCTION:
	case JSTYPE_NUMBER:
    case JSTYPE_BOOLEAN:
    default:
        return JS_FALSE;
    }
}

static void
aztecRGBAFloat_finalize(JSContext *cx, JSObject *obj)
{
	MRGBAFloat *vec = (MRGBAFloat *)JS_GetInstancePrivate(cx, obj, &aztecRGBAFloat_class, NULL);
	if (vec != NULL) {
		delete vec;
	}
}

JSClass aztecRGBAFloat_class = { 
	"AztecRGBAFloat", JSCLASS_HAS_PRIVATE, 
		JS_PropertyStub, JS_PropertyStub,					// add/del property
		aztecRGBAFloat_getProperty, aztecRGBAFloat_setProperty,	// get/set property
		JS_EnumerateStub, JS_ResolveStub,
		aztecRGBAFloat_convert, aztecRGBAFloat_finalize
};

static JSBool aztecRGBAFloat_constructor(JSContext *cx, JSObject *obj, uintN argc,
                             jsval *argv, jsval *rval)
{
	MRGBAFloat *vec = new MRGBAFloat();
	if (vec == NULL) {
		return NULL;
	}

	return JS_SetPrivate(cx, obj, vec);
}

JSObject *newRGBAFloat(JSContext *cx)
{
	JSObject *jsvec = JS_NewObject(cx, &aztecRGBAFloat_class, NULL, JS_GetGlobalObject(cx));

	if (jsvec == NULL) {
		return NULL;
	}

	MRGBAFloat *pvec = new MRGBAFloat();
	JS_SetPrivate(cx, jsvec, pvec);

	return jsvec;
}

void addRGBAFloatClass(JSContext *cx)
{
	// Add a RGBAFloat class (reflection of MRGBAFloat) to the global namespace
    JS_InitClass(cx, JS_GetGlobalObject(cx), NULL,
		&aztecRGBAFloat_class, aztecRGBAFloat_constructor, 0,
		aztecRGBAFloat_props, aztecRGBAFloat_methods, 0, 0);

}

}
