#include "StdAfx.h"

#include <math.h>
#include "MEditableMesh.h"
#include "scripting/ScriptMesh.h"
#include "scripting/ScriptVector3.h"
#include "scripting/ScriptVertex.h"
#include "scripting/ScriptRGBAFloat.h"
#include "scripting/ScriptTriangle.h"

#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

namespace Aztec {

class AztecMeshInfo {
public:
	AztecMeshInfo() {
		mesh = NULL;
	}
	~AztecMeshInfo() {
		mesh = NULL;
	}

	MMeshPtr mesh;
};

static JSBool
	addVertex(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	AztecMeshInfo *minfo = (AztecMeshInfo *)JS_GetInstancePrivate(cx, obj, &aztecMesh_class, NULL);
	if (minfo == NULL || argc < 1) {
		return JS_TRUE;
	}

	MEditableMesh *mesh = AZTEC_CAST(MEditableMesh, minfo->mesh);
	if (mesh == NULL) {
		JS_ReportError(cx, "Attempt to add vertex to non-editable mesh");
		return JS_FALSE;
	}

	uintN i;
	for (i=0;i<argc;i++) {
		jsval arg = argv[i];
		if (JSVAL_IS_OBJECT(arg)) {
			JSObject *obj = JSVAL_TO_OBJECT(arg);
			MVector3 *vec = (MVector3 *)JS_GetInstancePrivate(cx, obj, &aztecVector3_class, NULL);
			if (vec != NULL) {
				int r = mesh->addVertex(vec->x, vec->y, vec->z);
				*rval = INT_TO_JSVAL(r);
				return JS_TRUE;
			}

			MVertex *vert = (MVertex *)JS_GetInstancePrivate(cx, obj, &aztecVertex_class, NULL);
			if (vert != NULL) {
				int r = mesh->addVertexArray(&vert->getPos());
				*rval = INT_TO_JSVAL(r);
				return JS_TRUE;
			}
		}
	}
	*rval = JSVAL_VOID;

	return JS_TRUE;
}

static JSBool
	addVertsAndTriangles(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	AztecMeshInfo *minfo = (AztecMeshInfo *)JS_GetInstancePrivate(cx, obj, &aztecMesh_class, NULL);
	if (minfo == NULL || (argc != 2 && argc != 4)) {
		return JS_TRUE;
	}

	MEditableMesh *mesh = AZTEC_CAST(MEditableMesh, minfo->mesh);
	if (mesh == NULL) {
		JS_ReportError(cx, "Attempt to add vertex to non-editable mesh");
		return JS_FALSE;
	}
	jsval arg0 = argv[0];
	jsval arg1 = argv[1];

	if (!JSVAL_IS_OBJECT(arg0) || !JSVAL_IS_OBJECT(arg1)) {
		JS_ReportError(cx, "Arguments to addVertsAndTriangles([verts], [tris]) must be arrays");
		return JS_FALSE;
	}

	JSObject *vobj = JSVAL_TO_OBJECT(arg0);
	JSObject *tobj = JSVAL_TO_OBJECT(arg1);
	if (JS_IsArrayObject(cx, vobj) == JS_FALSE ||
		JS_IsArrayObject(cx, tobj) == JS_FALSE) {
			JS_ReportError(cx, "Arguments to addVertsAndTriangles([verts], [tris]) must be arrays");
			return JS_FALSE;
		}

		// Get lengths of the two arrays
		jsuint lengthv, lengtht;
		if (JS_GetArrayLength(cx, vobj, &lengthv) == JS_FALSE ||
			JS_GetArrayLength(cx, tobj, &lengtht) == JS_FALSE) {
				JS_ReportError(cx, "Arguments to addVertsAndTriangles([verts], [tris]) must be arrays");
				return JS_FALSE;
			}

			// Allocate storage for the vertices and triangles
			MVector3 *Verts = new MVector3[lengthv];
			MInternalTriangle *Tris = new MInternalTriangle[lengtht];

			// Pull vertices and triangles from the JavaScript objects into arrays.
			// Validate the types as we go.
			jsuint index;

			// Walk through the vertex array pulling elements and making sure each
			// element is either a vector3 or a vertex
			for (index=0;index<lengthv;index++) {
				jsval vval;
				if ((JS_GetElement(cx, vobj, index, &vval) == JS_FALSE) ||
					!JSVAL_IS_OBJECT(vval)) {
						// Failed to get an array element or element isn't an object
						delete[] Verts;
						delete[] Tris;
						JS_ReportError(cx, "Vertex element %d is not a vector or vertex", index);
						return JS_FALSE;
					}

					JSObject *vecObj = JSVAL_TO_OBJECT(vval);
					MVector3 *vec = (MVector3 *)JS_GetInstancePrivate(cx, vecObj, &aztecVector3_class, NULL);
					if (vec != NULL) {
						// This is a valid vector, so copy the value in 
						Verts[index] = *vec;
					} else {
						MVertex *vert = (MVertex *)JS_GetInstancePrivate(cx, vecObj, &aztecVertex_class, NULL);
						if (vert == NULL) {
							// Not a vector3 or a vertex.
							delete[] Verts;
							delete[] Tris;
							JS_ReportError(cx, "Vertex element %d is not a vector or vertex", index);
							return JS_FALSE;
						}

						Verts[index] = vert->getPos();
					}
			}

			// Walk through the vertex array pulling elements and making sure each
			// element is either a vector3 or a vertex
			for (index=0;index<lengtht;index++) {
				jsval tval;
				if ((JS_GetElement(cx, tobj, index, &tval) == JS_FALSE) ||
					!JSVAL_IS_OBJECT(tval)) {
						// Failed to get an array element or element isn't an object
						delete[] Verts;
						delete[] Tris;
						JS_ReportError(cx, "Triangle element %d is not a triangle type", index);
						return JS_FALSE;
					}

					JSObject *triObj = JSVAL_TO_OBJECT(tval);
					MInternalTriangle *tri = (MInternalTriangle *)JS_GetInstancePrivate(cx, triObj, &aztecTriangle_class, NULL);
					if (tri == NULL) {
						// Not a triangle.
						delete[] Verts;
						delete[] Tris;
						JS_ReportError(cx, "Triangle element %d is not a triangle type", index);
						return JS_FALSE;
					}

					Tris[index] = *tri;
			}

			mesh->addVertsAndTriangles(Verts, Tris, lengthv, lengtht);

			delete[] Verts;
			delete[] Tris;

			*rval = JSVAL_VOID;

			return JS_TRUE;
}

static JSBool
	addTriangle(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	AztecMeshInfo *minfo = (AztecMeshInfo *)JS_GetInstancePrivate(cx, obj, &aztecMesh_class, NULL);
	if ((minfo == NULL) || ((argc != 3) && (argc != 1))) {
		return JS_TRUE;
	}

	// Make sure it is a mesh we are allowed to edit
	MEditableMesh *mesh = AZTEC_CAST(MEditableMesh, minfo->mesh);
	if (mesh == NULL) {
		JS_ReportError(cx, "Attempt to add vertex to non-editable mesh");
		return JS_FALSE;
	}

	if (argc == 3) {
		jsval arg1 = *argv;
		jsval arg2 = *(argv+1);
		jsval arg3 = *(argv+2);

		// See if we are adding three vertices
		if (JSVAL_IS_OBJECT(arg1) && JSVAL_IS_OBJECT(arg2) && JSVAL_IS_OBJECT(arg3)) {
			MVertex *v0 = (MVertex *)JS_GetInstancePrivate(cx, JSVAL_TO_OBJECT(arg1), &aztecVertex_class, NULL);
			MVertex *v1 = (MVertex *)JS_GetInstancePrivate(cx, JSVAL_TO_OBJECT(arg2), &aztecVertex_class, NULL);
			MVertex *v2 = (MVertex *)JS_GetInstancePrivate(cx, JSVAL_TO_OBJECT(arg3), &aztecVertex_class, NULL);
			if ((v0 == NULL) || (v1 == NULL) || (v2 == NULL)) {
				// Passing in three vertices?
				JS_ReportError(cx, "Invalid arguments to addTriangle().  Must be three indices, or three Vertices.");
				return JS_FALSE;
			}

			mesh->addTriangle(v0->getPos(), v1->getPos(), v2->getPos());
			*rval = JSVAL_VOID;
			return JS_TRUE;
		}

		// See if we are adding a triangle that references existing vertices (by index)
		if (!JSVAL_IS_NUMBER(arg1) || !JSVAL_IS_NUMBER(arg2) ||
			!JSVAL_IS_NUMBER(arg3)) {
				JS_ReportError(cx, "Triangle vertex indices must be numeric");
				return JS_FALSE;
			}

			int i0 = (JSVAL_IS_INT(arg1) ? JSVAL_TO_INT(arg1) : (int)*JSVAL_TO_DOUBLE(arg1));
			int i1 = (JSVAL_IS_INT(arg2) ? JSVAL_TO_INT(arg2) : (int)*JSVAL_TO_DOUBLE(arg2));
			int i2 = (JSVAL_IS_INT(arg3) ? JSVAL_TO_INT(arg3) : (int)*JSVAL_TO_DOUBLE(arg3));

			int numV = mesh->getNumVerts();
			if (i0 < 0 || i1 < 0 || i2 < 0) {
				JS_ReportError(cx, "Triangle vertices must be greater than 0");
				return JS_FALSE;
			}

			mesh->addTriangle(i0, i1, i2);
			*rval = JSVAL_VOID;
			return JS_TRUE;
	} else if (argc == 1) {
		if (!JSVAL_IS_OBJECT(*argv)) {
			// Passing in three vertices?
			JS_ReportError(cx, "Invalid arguments to addTriangle()");
			return JS_FALSE;
		}

		MInternalTriangle *tri = (MInternalTriangle *)JS_GetInstancePrivate(cx, JSVAL_TO_OBJECT(*argv), &aztecTriangle_class, NULL);
		if (tri != NULL) {
			mesh->addTriangleArray(tri);
			*rval = JSVAL_VOID;
			return JS_TRUE;
		}
	}

	*rval = JSVAL_VOID;

	return JS_TRUE;
}

static JSBool
	calculateNormals(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	AztecMeshInfo *minfo = (AztecMeshInfo *)JS_GetInstancePrivate(cx, obj, &aztecMesh_class, NULL);
	if ((minfo == NULL) || (argc != 0)) {
		return JS_TRUE;
	}

	MMesh *mesh = AZTEC_CAST(MMesh, minfo->mesh);
	if (mesh == NULL) {
		JS_ReportError(cx, "Attempt to get vertex from non-mesh");
		return JS_FALSE;
	}

	mesh->calculateNormals();
	*rval = JSVAL_VOID;

	return JS_TRUE;

}

static JSBool
	getVertex(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	AztecMeshInfo *minfo = (AztecMeshInfo *)JS_GetInstancePrivate(cx, obj, &aztecMesh_class, NULL);
	if (minfo == NULL || argc < 1) {
		return JS_TRUE;
	}

	MMesh *mesh = AZTEC_CAST(MMesh, minfo->mesh);
	if (mesh == NULL) {
		JS_ReportError(cx, "Attempt to get vertex from non-mesh");
		return JS_FALSE;
	}

	if (!JSVAL_IS_NUMBER(*argv)) {
		JS_ReportError(cx, "Vertex index must be an number");
		return JS_FALSE;
	}

	int i0 = (JSVAL_IS_INT(*argv) ? JSVAL_TO_INT(*argv) : (int)*JSVAL_TO_DOUBLE(*argv));
	MVertex vert =	mesh->getVertex(i0);

	JSObject *jsvert = newVertex(cx);
	MVertex *tvert = (MVertex *)JS_GetInstancePrivate(cx, jsvert, &aztecVertex_class, NULL);
	if (tvert == NULL) {
		JS_ReportError(cx, "Failed to create a valid vertex object");
		return JS_FALSE;
	}

	// Set the value of the return object to the value of the mesh vertex
	*tvert = vert;

	*rval = OBJECT_TO_JSVAL(jsvert);

	return JS_TRUE;
}

class TriangleGetter {
public:
	static MInternalTriangle* getTriangle(const MMeshPtr &mesh, int triIndex) {
		return mesh->m_Tris[triIndex];
	}
};

static JSBool
	getTriangle(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	AztecMeshInfo *minfo = (AztecMeshInfo *)JS_GetInstancePrivate(cx, obj, &aztecMesh_class, NULL);
	if (minfo == NULL || argc < 1) {
		return JS_TRUE;
	}

	MMesh *mesh = AZTEC_CAST(MMesh, minfo->mesh);
	if (mesh == NULL) {
		JS_ReportError(cx, "Attempt to get triangle from non-mesh");
		return JS_FALSE;
	}

	if (!JSVAL_IS_NUMBER(*argv)) {
		JS_ReportError(cx, "Triangle index must be an number");
		return JS_FALSE;
	}

	int i0 = (JSVAL_IS_INT(*argv) ? JSVAL_TO_INT(*argv) : (int)*JSVAL_TO_DOUBLE(*argv));
	MInternalTriangle *tri = TriangleGetter::getTriangle(mesh, i0);

	if (tri == NULL) {
		JS_ReportError(cx, "Invalid triangle index");
		return JS_FALSE;
	}

	JSObject *jstri = newTriangle(cx);
	MInternalTriangle *ttri = (MInternalTriangle *)JS_GetInstancePrivate(cx, jstri, &aztecTriangle_class, NULL);
	if (ttri == NULL) {
		JS_ReportError(cx, "Failed to create a valid triangle object");
		return JS_FALSE;
	}

	// Set the value of the return object to the value of the mesh vertex
	*ttri = *tri;

	*rval = OBJECT_TO_JSVAL(jstri);

	return JS_TRUE;
}

static JSBool
	extrudeFaces(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	AztecMeshInfo *minfo = (AztecMeshInfo *)JS_GetInstancePrivate(cx, obj, &aztecMesh_class, NULL);
	if ((minfo == NULL) || (argc != 3)) {
		return JS_TRUE;
	}

	// Make sure it is a mesh we are allowed to edit
	MEditableMesh *mesh = AZTEC_CAST(MEditableMesh, minfo->mesh);
	if (mesh == NULL) {
		JS_ReportError(cx, "Attempt to add vertex to non-editable mesh");
		return JS_FALSE;
	}

	jsval arg0 = *argv;
	jsval arg1 = *(argv+1);
	jsval arg2 = *(argv+2);

	if (!JSVAL_IS_NUMBER(arg0) || !JSVAL_IS_NUMBER(arg1) || !JSVAL_IS_OBJECT(arg2)) {
		JS_ReportError(cx, "Invalid arguments to extrudeFaces(int, float, Vector3)");
		return JS_FALSE;
	}

	MVector3 *vec = (MVector3 *)JS_GetInstancePrivate(cx, JSVAL_TO_OBJECT(arg2), &aztecVector3_class, NULL);
	if (vec == NULL) {
		JS_ReportError(cx, "Invalid arguments to extrudeFaces(int, float, Vector3)");
		return JS_FALSE;
	}
	int flags = (JSVAL_IS_INT(arg0) ? JSVAL_TO_INT(arg0) : (int)*JSVAL_TO_DOUBLE(arg0));
	float amount = (JSVAL_IS_INT(arg0) ? (float)JSVAL_TO_INT(arg0) : (float)*JSVAL_TO_DOUBLE(arg0));

	int r = mesh->extrudeFaces(flags, amount, *vec);

	*rval = INT_TO_JSVAL(r);

	return JS_TRUE;
}

static JSBool
	collapseVertices(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	AztecMeshInfo *minfo = (AztecMeshInfo *)JS_GetInstancePrivate(cx, obj, &aztecMesh_class, NULL);
	if ((minfo == NULL) || (argc != 1)) {
		return JS_TRUE;
	}

	// Make sure it is a mesh we are allowed to edit
	MEditableMesh *mesh = AZTEC_CAST(MEditableMesh, minfo->mesh);
	if (mesh == NULL) {
		JS_ReportError(cx, "Attempt to add vertex to non-editable mesh");
		return JS_FALSE;
	}

	jsval arg0 = *argv;

	if (!JSVAL_IS_NUMBER(arg0)) {
		JS_ReportError(cx, "Invalid arguments to collapseVertices(int)");
		return JS_FALSE;
	}

	int flags = (JSVAL_IS_INT(arg0) ? JSVAL_TO_INT(arg0) : (int)*JSVAL_TO_DOUBLE(arg0));

	int r = mesh->collapseVertices(flags);

	*rval = INT_TO_JSVAL(r);

	return JS_TRUE;
}

static JSBool
	weldVertices(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	AztecMeshInfo *minfo = (AztecMeshInfo *)JS_GetInstancePrivate(cx, obj, &aztecMesh_class, NULL);
	if ((minfo == NULL) || (argc != 2)) {
		return JS_TRUE;
	}

	// Make sure it is a mesh we are allowed to edit
	MEditableMesh *mesh = AZTEC_CAST(MEditableMesh, minfo->mesh);
	if (mesh == NULL) {
		JS_ReportError(cx, "Attempt to add vertex to non-editable mesh");
		return JS_FALSE;
	}

	jsval arg0 = *argv;
	jsval arg1 = *(argv+1);

	if (!JSVAL_IS_NUMBER(arg0) || !JSVAL_IS_NUMBER(arg1)) {
		JS_ReportError(cx, "Invalid arguments to weldVertices(int, float)");
		return JS_FALSE;
	}

	int flags = (JSVAL_IS_INT(arg0) ? JSVAL_TO_INT(arg0) : (int)*JSVAL_TO_DOUBLE(arg0));
	float threshold = (JSVAL_IS_INT(arg0) ? (float)JSVAL_TO_INT(arg0) : (float)*JSVAL_TO_DOUBLE(arg0));

	int r = mesh->weldVertices(flags, threshold);

	*rval = INT_TO_JSVAL(r);

	return JS_TRUE;
}

static JSBool
	centerMesh(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	AztecMeshInfo *minfo = (AztecMeshInfo *)JS_GetInstancePrivate(cx, obj, &aztecMesh_class, NULL);
	if ((minfo == NULL) || (argc != 0)) {
		return JS_TRUE;
	}

	// Make sure it is a mesh we are allowed to edit
	MEditableMesh *mesh = AZTEC_CAST(MEditableMesh, minfo->mesh);
	if (mesh == NULL) {
		JS_ReportError(cx, "Attempt to add vertex to non-editable mesh");
		return JS_FALSE;
	}

	mesh->centerMesh();

	*rval = JSVAL_VOID;

	return JS_TRUE;
}

static JSBool
	scaleMesh(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	AztecMeshInfo *minfo = (AztecMeshInfo *)JS_GetInstancePrivate(cx, obj, &aztecMesh_class, NULL);
	if ((minfo == NULL) || (argc != 1)) {
		return JS_TRUE;
	}

	// Make sure it is a mesh we are allowed to edit
	MEditableMesh *mesh = AZTEC_CAST(MEditableMesh, minfo->mesh);
	if (mesh == NULL) {
		JS_ReportError(cx, "Attempt to add vertex to non-editable mesh");
		return JS_FALSE;
	}

	jsval arg0 = *argv;

	if (!JSVAL_IS_NUMBER(arg0)) {
		JS_ReportError(cx, "Invalid arguments to scaleMesh(float)");
		return JS_FALSE;
	}

	float factor = (JSVAL_IS_INT(arg0) ? (float)JSVAL_TO_INT(arg0) : (float)*JSVAL_TO_DOUBLE(arg0));

	mesh->scaleMesh(factor);

	*rval = JSVAL_VOID;

	return JS_TRUE;
}

static JSBool
	flipNormals(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	AztecMeshInfo *minfo = (AztecMeshInfo *)JS_GetInstancePrivate(cx, obj, &aztecMesh_class, NULL);
	if ((minfo == NULL) || (argc != 1)) {
		return JS_TRUE;
	}

	// Make sure it is a mesh we are allowed to edit
	MEditableMesh *mesh = AZTEC_CAST(MEditableMesh, minfo->mesh);
	if (mesh == NULL) {
		JS_ReportError(cx, "Attempt to flip normals on a non-editable mesh");
		return JS_FALSE;
	}

	jsval arg0 = *argv;

	if (!JSVAL_IS_NUMBER(arg0)) {
		JS_ReportError(cx, "Invalid arguments to flipNormals(int)");
		return JS_FALSE;
	}

	int flags = (JSVAL_IS_INT(arg0) ? JSVAL_TO_INT(arg0) : (int)*JSVAL_TO_DOUBLE(arg0));

	int r = mesh->flipNormals(flags);

	*rval = INT_TO_JSVAL(r);

	return JS_TRUE;
}

static JSFunctionSpec aztecMesh_methods[] = {
	{"addVertex",				addVertex,				1},
	{"addVertsAndTriangles",	addVertsAndTriangles,	1},
	{"addTriangle",				addTriangle,			1},
	{"calculateNormals",		calculateNormals,		0},
	{"centerMesh",				centerMesh,				0},
	{"collapseVertices",		collapseVertices,		1},
	{"extrudeFaces",			extrudeFaces,			3},
	{"flipNormals",				flipNormals,			1},
	{"getTriangle",				getTriangle,			1},
	{"getVertex",				getVertex,				1},
	{"scaleMesh",				scaleMesh,				1},
	{"weldVertices",			weldVertices,			2},
	{0}
};

static JSPropertySpec aztecMesh_props[] = {
	{"numVerts",				0,	JSPROP_ENUMERATE},
	{"numTris",					1,	JSPROP_ENUMERATE},
	{"VERTEX_SELECTED",			2,	JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"VERTEX_VISIBLE",			3,	JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"VERTEX_FLAGFORCHANGE",	4,	JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"VERTEX_DRAW_FEEDBACK",	5,	JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"TRIANGLE_SELECTED",		6,	JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"TRIANGLE_VISIBLE",		7,	JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"TRIANGLE_FLAGFORCHANGE",	8,	JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"TRIANGLE_FLAGFORCHANGE2",	9,	JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"EDGE_SELECTED",			10,	JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"EDGE_VISIBLE",			11,	JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{0}
};

static JSBool
	aztecMesh_getProperty(JSContext *cx, JSObject *obj, jsval id, jsval *vp)
{
	AztecMeshInfo *minfo = (AztecMeshInfo *)JS_GetInstancePrivate(cx, obj, &aztecMesh_class, NULL);
	if (minfo == NULL || minfo->mesh == NULL) {
		return JS_FALSE;
	}

	if (JSVAL_IS_STRING(id)) {
		char *propStr = JS_GetStringBytes(JS_ValueToString(cx, id));
		if (strcmp(propStr, "numVerts") == 0) {
			jsint numV = minfo->mesh->getNumVerts();
			*vp = INT_TO_JSVAL(numV);
			return JS_TRUE;
		} else if (strcmp(propStr, "numTris") == 0) {
			jsint numT = minfo->mesh->getNumTris();
			*vp = INT_TO_JSVAL(numT);
			return JS_TRUE;
		} else if (strcmp(propStr, "VERTEX_SELECTED") == 0) {
			*vp = INT_TO_JSVAL(VERTEX_SELECTED);
			return JS_TRUE;
		} else if (strcmp(propStr, "VERTEX_VISIBLE") == 0) {
			*vp = INT_TO_JSVAL(VERTEX_VISIBLE);
			return JS_TRUE;
		} else if (strcmp(propStr, "VERTEX_FLAGFORCHANGE") == 0) {
			*vp = INT_TO_JSVAL(VERTEX_FLAGFORCHANGE);
			return JS_TRUE;
		} else if (strcmp(propStr, "VERTEX_DRAW_FEEDBACK") == 0) {
			*vp = INT_TO_JSVAL(VERTEX_DRAW_FEEDBACK);
			return JS_TRUE;
		} else if (strcmp(propStr, "TRIANGLE_SELECTED") == 0) {
			*vp = INT_TO_JSVAL(TRIANGLE_SELECTED);
			return JS_TRUE;
		} else if (strcmp(propStr, "TRIANGLE_VISIBLE") == 0) {
			*vp = INT_TO_JSVAL(TRIANGLE_VISIBLE);
			return JS_TRUE;
		} else if (strcmp(propStr, "TRIANGLE_FLAGFORCHANGE") == 0) {
			*vp = INT_TO_JSVAL(TRIANGLE_FLAGFORCHANGE);
			return JS_TRUE;
		} else if (strcmp(propStr, "TRIANGLE_FLAGFORCHANGE2") == 0) {
			*vp = INT_TO_JSVAL(TRIANGLE_FLAGFORCHANGE2);
			return JS_TRUE;
		} else if (strcmp(propStr, "EDGE_SELECTED") == 0) {
			*vp = INT_TO_JSVAL(EDGE_SELECTED);
			return JS_TRUE;
		} else if (strcmp(propStr, "EDGE_VISIBLE") == 0) {
			*vp = INT_TO_JSVAL(EDGE_VISIBLE);
			return JS_TRUE;
		}

		// May be a built-in method.
		for (int m=0;aztecMesh_methods[m].name != NULL;m++) {
			if (!strcmp(propStr, aztecMesh_methods[m].name)) {
				// Yup - is a built-in function.
				JSFunction *jfunc =
					JS_NewFunction(cx, aztecMesh_methods[m].call,
					aztecMesh_methods[m].nargs,
					0, obj, propStr);
				JSObject *jfobj = JS_GetFunctionObject(jfunc);
				*vp = OBJECT_TO_JSVAL(jfobj);
				return JS_TRUE;
			}
		}

		JS_ReportError(cx, "%s is not defined", propStr);
		return JS_FALSE;
	} else if (JSVAL_IS_INT(id)) {
		int index = JSVAL_TO_INT(id);
		switch (index) {
case 0:
	*vp = INT_TO_JSVAL(minfo->mesh->getNumVerts());
	return JS_TRUE;
case 1:
	*vp = INT_TO_JSVAL(minfo->mesh->getNumTris());
	return JS_TRUE;
case 2:
	*vp = INT_TO_JSVAL(VERTEX_SELECTED);
	return JS_TRUE;
case 3:
	*vp = INT_TO_JSVAL(VERTEX_VISIBLE);
	return JS_TRUE;
case 4:
	*vp = INT_TO_JSVAL(VERTEX_FLAGFORCHANGE);
	return JS_TRUE;
case 5:
	*vp = INT_TO_JSVAL(VERTEX_DRAW_FEEDBACK);
	return JS_TRUE;
case 6:
	*vp = INT_TO_JSVAL(TRIANGLE_SELECTED);
	return JS_TRUE;
case 7:
	*vp = INT_TO_JSVAL(TRIANGLE_VISIBLE);
	return JS_TRUE;
case 8:
	*vp = INT_TO_JSVAL(TRIANGLE_FLAGFORCHANGE);
	return JS_TRUE;
case 9:
	*vp = INT_TO_JSVAL(TRIANGLE_FLAGFORCHANGE2);
	return JS_TRUE;
case 10:
	*vp = INT_TO_JSVAL(EDGE_SELECTED);
	return JS_TRUE;
case 11:
	*vp = INT_TO_JSVAL(EDGE_VISIBLE);
	return JS_TRUE;
default:
	break;
		}
	}
	return JS_TRUE;
}

static JSBool
	aztecMesh_setProperty(JSContext *cx, JSObject *obj, jsval id, jsval *vp)
{
	AztecMeshInfo *minfo = (AztecMeshInfo *)JS_GetInstancePrivate(cx, obj, &aztecMesh_class, NULL);
	if (minfo == NULL) {
		return JS_FALSE;
	}

	return JS_FALSE;
}

static void
	aztecMesh_finalize(JSContext *cx, JSObject *obj)
{
	AztecMeshInfo *minfo = (AztecMeshInfo *)JS_GetInstancePrivate(cx, obj, &aztecMesh_class, NULL);
	if (minfo != NULL) {
		minfo->mesh = NULL;
		delete minfo;
		JS_SetPrivate(cx, obj, NULL);
	}
}

JSClass aztecMesh_class = { 
	"AztecMesh", JSCLASS_HAS_PRIVATE, 
		JS_PropertyStub, JS_PropertyStub,				// add/del property
		aztecMesh_getProperty, aztecMesh_setProperty,	// get/set property
		JS_EnumerateStub, JS_ResolveStub,
		JS_ConvertStub, aztecMesh_finalize
};

static JSBool aztecMesh_constructor(JSContext *cx, JSObject *obj, uintN argc,
	jsval *argv, jsval *rval)
{
	AztecMeshInfo *minfo = new AztecMeshInfo();
	if (minfo == NULL) {
		return NULL;
	}
	minfo->mesh = new MEditableMesh();

	return JS_SetPrivate(cx, obj, minfo);
}

JSObject *newMesh(JSContext *cx, MMeshPtr mesh)
{
	JSObject *jsmesh = JS_NewObject(cx, &aztecMesh_class, NULL, JS_GetGlobalObject(cx));

	if (jsmesh == NULL) {
		return NULL;
	}

	AztecMeshInfo *minfo = new AztecMeshInfo();
	if (minfo == NULL) {
		return NULL;
	}
	minfo->mesh = mesh;
	JS_SetPrivate(cx, jsmesh, minfo);

	return jsmesh;
}

void addMeshClass(JSContext *cx)
{
	// Add a Vertex class (reflection of MVertex3) to the global namespace
	JSObject *iobj =
		JS_InitClass(cx, JS_GetGlobalObject(cx), NULL,
		&aztecMesh_class, aztecMesh_constructor, 0,
		aztecMesh_props, aztecMesh_methods, 0, 0);

}

}
