#include <StdAfx.h>

#include <misc/ResourceManager.h>

#include <stdlib.h>
#include <vector>
#include <map>
#include <fstream>

namespace Aztec {

#ifdef _WIN32
  static char DIR_SEP = '\\';
#else
  static char DIR_SEP = '/';
#endif

  typedef std::vector<std::string> StringVector;
  typedef std::map<std::string, std::string> StringMap;

  static StringVector resourcePaths;
  static StringMap recentResources;
  

  void ResourceManager::addResourcePath(const std::string &path) {
    resourcePaths.push_back(path);
  }

  static bool fileExists(const std::string &filename) {
    if (filename.length() == 0) {
      return false;
    }

    std::ifstream in(filename.c_str());

    return in.is_open();
  }

  static std::string fixSlashes(const std::string &rhs) {
    std::string result = rhs;
    // change all the slashes to be platform dependant.
    for (unsigned int i = 0; i < result.length(); ++i) {
      if (result[i] == '/' || result[i] == '\\') {
        result[i] = DIR_SEP;
      }
    }

    return result;
  }

  static std::string appendPath(const std::string &left, const std::string &right) {
    // copy the left hand side.
    std::string result = fixSlashes(left);

    // if it doesn't end in a directory separator, add one in.
    if (result.length() > 0 && result[result.length() - 1] != DIR_SEP) {
      result += DIR_SEP;
    }

    result += right;

    return result;
  }

  static const char* getEnvVar(const std::string &var) {
    char *result = getenv(var.c_str());

    return (result != NULL) ? result : "";
  }

  static std::string getRealPath(const std::string &path) {
    // checks to see if the path contains an environment variable. If it does, expand it out.
    std::string result = path;
    int index = std::string::npos;

    // find any $( 's
    while ((index = result.find("$(")) != std::string::npos) {

      // find the matching closing bracket
      int endIndex = result.find(")", index + 1);

      // expand the environment var
      result = result.substr(0, index) + getEnvVar(result.substr(index + 2, endIndex - index)) + result.substr(endIndex + 1);
    }

    return result;
  }


  std::string ResourceManager::locateResource(const std::string &resource) {
    std::string realResource = fixSlashes(resource);

    // try to see if a cached version exists.
    {
      std::string cachedPath = recentResources[realResource];
      if (fileExists(cachedPath)) {
        return cachedPath;
      }
    }

    // try and find the resource
    for (unsigned int i = 0; i < resourcePaths.size(); ++i) {
      std::string path = appendPath(getRealPath(resourcePaths[i]), realResource);
      if (fileExists(path)) {
        // cache our path for future reference.
        recentResources[realResource] = path;
        return path;
      }
    }

    // as a last resort, try the plain filename.
    if (fileExists(resource)) {
      recentResources[realResource] = realResource;
      return realResource;
    }

    return "";
  }


}

