#include "StdAfx.h"

#include <MVertex.h>
#include <MEditableMesh.h>

namespace Aztec {

  MVertex::MVertex() : mesh(NULL), index(-1) { 
  }

  MVertex::MVertex(const MVertex &src) 
    : mesh(src.mesh), index(src.index), pos(src.pos) { 
  }

  int MVertex::getIndex() {
    return index;
  }

  MVector3 MVertex::getPos() {
    if (mesh != NULL) {
      return mesh->getVertexPosition(index);
    } else {
      return pos;
    }
  }

  void MVertex::setPos(const MVector3 &pos) {
    if (mesh != NULL) {
      // try to cast the mesh as an editable mesh so we can change the value.
      MEditableMesh *editMesh = AZTEC_CAST(MEditableMesh, mesh);

#ifdef DEBUG
      assert(editMesh != NULL);
#endif
      if (editMesh != NULL) {
        editMesh->setVertexPosition(index, pos);
      }
    } else {
      this->pos = pos;
    }
  }

  float MVertex::getX() {
    return getPos().x;
  }

  float MVertex::getY() {
    return getPos().y;
  }

  float MVertex::getZ() {
    return getPos().z;
  }

  
  MVector3 MVertex::getNormal() {
    if (mesh != NULL) {
      return mesh->getVertexNormal(index);
    } else {
      return MVector3(0,0,1);
    }
  }

  void MVertex::setNormal(const MVector3 &normal) {
    if (mesh != NULL) {
      // try to cast the mesh as an editable mesh so we can change the value.
      MEditableMesh *editMesh = AZTEC_CAST(MEditableMesh, mesh);

#ifdef DEBUG
      assert(editMesh != NULL);
#endif
      if (editMesh != NULL) {
        editMesh->setVertexNormal(index, normal);
      }
    }
  }

  MRGBAFloat MVertex::getFeedbackColour() {
    if (mesh != NULL) {
      return mesh->getComponentFeedbackColour(MComponentisedObject::POINT_TYPE, index);
    } else {
      return MRGBAFloat();
    }
  }

  void MVertex::setFeedbackColour(const MRGBAFloat &colour) {
    if (mesh != NULL) {
      mesh->setComponentFeedbackColour(MComponentisedObject::POINT_TYPE, index, colour);
    }
  }

  MVertex& MVertex::operator=(const MVertex &rhs) {
    // check to see if our src has a mesh
    if (rhs.mesh != NULL) {
      // if it does, just copy the mesh details.
      mesh = rhs.mesh;
      index = rhs.index;
    } else {
      // if it doesn't, check to see if we have a mesh
      if (mesh != NULL) {
        // if we do have a mesh, assign the location of the source vertex to 
        // this one.
        MEditableMesh *editMesh = AZTEC_CAST(MEditableMesh, mesh);
        
#ifdef DEBUG
        assert(editMesh != NULL);
#endif
        if (editMesh != NULL) {
          editMesh->setVertexPosition(index, rhs.pos);
        }
      } else {
        // if we don't, just assign the location of the source vertex to 
        // our temporary location
        pos = rhs.pos;
      }
    }
    return *this;
  }

  MVertex::MVertex(MMesh *newMesh, int newIndex)
    : mesh(newMesh), index(newIndex)
  {
  }

}
