#include "StdAfx.h"

#include "MDAGNode.h"
#include "MDAGraph.h"

#include <vector>
#include <algorithm>
#include <stdio.h>

#include "MSystemManager.h"

#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


namespace Aztec {

  std::string getName(MDAGNode *node) {

    MNamedObject *namedObj = AZTEC_CAST(MNamedObject, node);
    MParameterObject *paramObj = AZTEC_CAST(MParameterObject, node);
    MParameterObjectList *paramList = AZTEC_CAST(MParameterObjectList, node);

    char name[1024];

    if (namedObj != NULL) {
      strcpy(name, (namedObj->getClassName() + "." + namedObj->getName()).c_str());
    } else if (paramObj != NULL) {
      std::string nameStr;
    
      if (paramObj->getOwner() != NULL) {
        namedObj = AZTEC_CAST(MNamedObject, paramObj->getOwner());

        if (namedObj != NULL) {
          nameStr += namedObj->getName();
          nameStr += ".";
        }
      }
    
      nameStr += paramObj->getLongName();

      strcpy(name, nameStr.c_str());
    } else if (paramList != NULL) {
      std::string nameStr;

      namedObj = AZTEC_CAST(MNamedObject, paramList->getOwner());

      if (namedObj != NULL) {
        nameStr += namedObj->getName();
      } else {
        sprintf(name, "%x", node);
        nameStr += name;
      }
      nameStr += ".paramList";

      strcpy(name, nameStr.c_str());
    } else {
      sprintf(name, "%x.%s", node, node->getClassName().c_str());
    }

    return name;
  }


  static unsigned int globalDagID = 1;

  MDAGNode::MDAGNode() {
    dagNodeID = globalDagID++;
  }

  MDAGNode::~MDAGNode() {
    // iterate over our inputs, and remove us as
    // an output for that object
    NodeList::iterator it;

    for (it = inputNodes.begin(); it != inputNodes.end(); ++it) {
      (*it)->doRemoveOutput(this);
    }

    // iterate over our outputs, and remove us as
    // an input for that object
    for (it = outputNodes.begin(); it != outputNodes.end(); ++it) {
      (*it)->doRemoveInput(this);
    }

    inputNodes.clear();
    outputNodes.clear();
  }


  bool MDAGNode::addInput(MDAGNode *node) {
    if (node == NULL) {
      return false;
    }
    // we cannot add the given input to this node if 
    // we are outputting to it.
    if (MDAGraph::doesNodeOutputTo(this, node)) {
#ifdef _DEBUG
//      MSystemManager::getInstance()->logOutput("Failed to add input %x to %x", node, this);
      MSystemManager::getInstance()->logOutput("Failed to add input %s to %s", getName(node).c_str(), getName(this).c_str());
#endif
      return false;
    }

    doAddInput(node);
    if (node != NULL) {
      node->doAddOutput(this);
    }

    // we have gained an input, so we have changed
    flagOutputs(OBJECTFLAG_NEEDS_UPDATE);
  
    return true;
  }

  bool MDAGNode::addOutput(MDAGNode *node) {
    if (node == NULL) {
      return false;
    }
    // we cannot add the node as an output if
    // that node is outputting to us.
    if (MDAGraph::doesNodeOutputTo(node, this)) {
#ifdef _DEBUG
      MSystemManager::getInstance()->logOutput("Failed to add output %s to %s", getName(node).c_str(), getName(this).c_str());
#endif
      return false;
    }

    doAddOutput(node);
    if (node != NULL) {
      node->doAddInput(this);
    }

    // they have gained an input, so we have changed
    node->flagOutputs(OBJECTFLAG_NEEDS_UPDATE);

    return true;
  }

  void MDAGNode::removeInput(MDAGNode *node) {
    if (node == NULL) {
      return;
    }
    doRemoveInput(node);
    if (node != NULL) {
      node->doRemoveOutput(this);
    }

    // we have lost an input, so we have changed
    flagOutputs(OBJECTFLAG_NEEDS_UPDATE);
  
  }

  void MDAGNode::removeOutput(MDAGNode *node) {
    if (node == NULL) {
      return;
    }
    doRemoveOutput(node);
    if (node != NULL) {
      node->doRemoveInput(this);
    }

    // the output object has lost an input so it has chanaged.
    node->flagOutputs(OBJECTFLAG_NEEDS_UPDATE);
  }

  int MDAGNode::getInputCount() {
    return inputNodes.size();
  }

  void MDAGNode::getInputs(MDAGNode **inps) {
    NodeList::iterator it;

    for (it = inputNodes.begin(); it != inputNodes.end(); ++it) {
      *inps = *it;
      ++inps;
    }
  }

  int MDAGNode::getOutputCount() {
    return outputNodes.size();
  }

  void MDAGNode::getOutputs(MDAGNode **outs) {
    NodeList::iterator it;

    for (it = outputNodes.begin(); it != outputNodes.end(); ++it) {
      *outs = *it;
      ++outs;
    }
  }

  void MDAGNode::flagOutputs(AztecFlags flag) {
    MDAGraph::flagOutputs(this, flag);
  }

  MBaseObjectPtr MDAGNode::createNew() {
    MDAGNodePtr newNode = new MDAGNode;

    newNode->setFrom(this);

    return newNode;
  }

  void MDAGNode::setFrom(MBaseObjectPtr srcObj) {
    doEnsureUndo(nodeUndo, this);

    MBaseObject::setFrom(srcObj);

    MDAGNode *node = AZTEC_CAST(MDAGNode, srcObj);
    NodeList::iterator it;

    if (node != NULL) {
      for (it = node->inputNodes.begin(); it != node->inputNodes.end(); ++it) {
        addInput(*it);
      }
      for (it = node->inputNodes.begin(); it != node->inputNodes.end(); ++it) {
        addOutput(*it);
      }
    }
  }

  bool MDAGNode::doUpdateObject() {
    MBaseObject::doUpdateObject();

    // loop over all our inputs, and update them
    NodeList::iterator it;

    // we create a coyp of the inputs because during the update process
    // nodes may be added or removed, thus invalidating our iterator. If
    // we have a copy instead, that problem is solved.
    NodeList l = inputNodes;
    
    for (it = l.begin(); it != l.end(); ++it) {
      (*it)->updateObject();
    }

    return true;
  }


  const WORD DAG_Version = 0x0101;

  static inline void doInsert(MDAGNode::NodeList &list, MDAGNode *node) {
    MDAGNode::NodeList::iterator it;

    it = std::lower_bound(list.begin(), list.end(), node);

    if (it == list.end()) {
      list.push_back(node);
    } else if (*it != node) {
      list.insert(it, node);
    }

    // list.insert(node);
  }

  static inline void doRemove(MDAGNode::NodeList &list, MDAGNode *node) {
    MDAGNode::NodeList::iterator it;

    it = std::lower_bound(list.begin(), list.end(), node);

    if (it != list.end() && *it == node) {
      list.erase(it);
    }

    // list.erase(node);
  }

  void MDAGNode::doAddInput(MDAGNode *node) {
    doEnsureUndo(nodeUndo, this);
    doInsert(inputNodes, node);
  }

  void MDAGNode::doAddOutput(MDAGNode *node) {
    doEnsureUndo(nodeUndo, this);
    doInsert(outputNodes, node);
  }

  void MDAGNode::doRemoveInput(MDAGNode *node) {
    doEnsureUndo(nodeUndo, this);
    doRemove(inputNodes, node);
  }

  void MDAGNode::doRemoveOutput(MDAGNode *node) {
    doEnsureUndo(nodeUndo, this);
    doRemove(outputNodes, node);
  }

  // MUndoableObject methods
  void MDAGNode::finishWithUndoNode() {
    nodeUndo = NULL;
    MBaseObject::finishWithUndoNode();
  }

  template<class FROM, class TO>
  void copyVector(TO &lhs, const FROM &rhs) {
    lhs.resize(rhs.size());
    for (unsigned i = 0; i < rhs.size(); ++i) {
      lhs[i] = &*rhs[i];
    }
  }

  template <class A, class B>
  void swapVector(A &lhs, B &rhs) {
    std::vector<MDAGNodePtr> temp = lhs;
    copyVector(lhs, rhs);
    copyVector(rhs, temp);
  }

  MDAGNode::Undo::Undo(MDAGNode *node) {
    object = node;
    copyVector(storedInputs, node->inputNodes);
    copyVector(storedOutputs, node->outputNodes);
  }

  MUndoableObject* MDAGNode::Undo::getObject() {
    return &*object;
  }

  void MDAGNode::Undo::undo() {
    swapVector(storedInputs, object->inputNodes);
    swapVector(storedOutputs, object->outputNodes);
  }

  void MDAGNode::Undo::redo() {
    swapVector(storedInputs, object->inputNodes);
    swapVector(storedOutputs, object->outputNodes);
  }


}
