#include <AztecGUICommonPCH.h>

#include <gui/win32/MButtonImpl.h>
#include <gui/win32/MContainerImpl.h>
#include <gui/win32/MAppImpl.h>
#include <config/UIConfig.h>

#include <assert.h>

namespace Aztec {

  MButton::MButton( const std::string &caption, 
                    const std::string &command, 
                    MSize2D prefSize) 
    : MComponent(caption),
      m_Caption(caption),
      m_Command(command),
      minButSize(-1,-1),
      buttonDown(false)
  {

    if (prefSize.getWidth() != -1 || prefSize.getHeight() != -1) {
      setPreferredSize(prefSize);
      setMinimumSize(prefSize);
    }

    if (command != "") {
      std::string friendlyName = AztecGUI::UIConfig::getFriendlyName(command);
      if (friendlyName == "") {
        AztecGUI::UIConfig::addFriendlyCommand(caption, command, "Unsorted");
      }
    }
  }

  MButton::~MButton() {
  }

  //
  // Here is a sample subclass function.
  //
  static LONG FAR PASCAL SubClassFunc(   HWND hWnd,
                 WORD Message,
                 WORD wParam,
                 LONG lParam)
  {
    MButtonPtr button = AZTEC_CAST(MButton, MApp::getInstance()->getComponent(hWnd));

    assert(button != NULL);

     //
     // When the focus is in an edit control inside a dialog box, the
     //  default ENTER key action will not occur.
     //
    if (Message == WM_KEYDOWN) {
      bool handled = button->handleKeyEvent(MKeyEvent(MKeyEvent::KEY_DOWN, ::GetTickCount(), button->getKeyCodeFromVK(wParam), button->getStateFromAsync(), button));
    } else if (Message == WM_KEYUP) {
      bool handled = button->handleKeyEvent(MKeyEvent(MKeyEvent::KEY_UP, ::GetTickCount(), button->getKeyCodeFromVK(wParam), button->getStateFromAsync(), button));
    } else if (Message == WM_NCPAINT) {
      // BUG: For some reason, when a region passes through this window proc, 
      // and then NC_PAINT message is handled, the control's frame isn't 
      // always redrawn. The hack is to set wParam to 1, which means the 
      // entire frame should be drawn.
      wParam = 1;
    }
    
    return CallWindowProc(button->lpfnOldWndProc, hWnd, Message, wParam, lParam);
  }



  bool MButton::createImpl() {
    MApp *app;
    HWND parentHWND = NULL;

    // Get the parent windows hwnd, so we can create our button.
    if (m_Parent != NULL) {
      parentHWND = m_Parent->getHWnd();
    }

    app = MApp::getInstance();

    // create the actual windows control.
    DWORD bitmapflag = image != NULL ? BS_BITMAP : 0;

    m_hWnd = ::CreateWindow("BUTTON", m_Caption.c_str(), WS_CHILD | BS_PUSHLIKE | BS_CHECKBOX | bitmapflag,
                            0, 10, 0, 10, parentHWND, NULL, app->getHInstance(), NULL);

    // if it succeeded, update everything.
    if (m_hWnd != 0) {
      // now subclass us
      lpfnOldWndProc = (WNDPROC)SetWindowLong(m_hWnd, GWL_WNDPROC, (DWORD) SubClassFunc);

      // Register this control and hwnd with the application.
      app->registerComponent(m_hWnd, this);

      // Change the font of the button.
      ::SendMessage(m_hWnd, WM_SETFONT, (DWORD)::GetStockObject(DEFAULT_GUI_FONT), 0);

      // reset the size since the font has changed.
      minButSize = MSize2D(-1,-1);

      ::SendMessage(m_hWnd, BM_SETCHECK, buttonDown ? BST_CHECKED : BST_UNCHECKED, 0);
      if (image != NULL) {
        setImage(image);
      }


      // Call the crete event handler.
      onCreate();

      // show the button, and update the screen.
      setVisible(true);
      ::UpdateWindow(m_hWnd);

      return true;
    }

    return false;
  }


  void MButton::setCaption(const std::string &caption) {
    m_Caption = caption;
    if (m_hWnd != NULL) {
      ::SetWindowText(m_hWnd, caption.c_str());
    }
    minButSize = MSize2D(-1,-1);
    // TODO: tell the container that we may have changed size.
  }

  std::string MButton::getCaption() {
    return m_Caption;
  }

  void MButton::setCommand(const std::string &command) {
    m_Command = command;
  }

  std::string MButton::getCommand() {
    return m_Command;
  }

  bool MButton::onClick() {
    return false;
  }

  void MButton::setButtonDown(bool buttonDown) {
    ::SendMessage(m_hWnd, BM_SETCHECK, buttonDown ? BST_CHECKED : BST_UNCHECKED, 0);
    this->buttonDown = buttonDown;
  }

  bool MButton::getButtonDown() {
    return buttonDown;
  }

  void MButton::setImage(const Aztec::MImagePtr &i) {
    image = i;

    if (m_hWnd != NULL) {
      HDC dc = ::GetDC(m_hWnd);

      if (dc != NULL) {
        bitmap.setFromImage(image, dc);

        ::SendMessage(m_hWnd, BM_SETIMAGE, IMAGE_BITMAP, (LPARAM)bitmap.m_hBitmap);

        // For some reason this code to change the button style DOES NOT WORK. Why is that?
        // This means that button images must be set before the button is created, otherwise it wont work.
        /*

        if (bitmap.m_hBitmap != NULL) {
          ::SendMessage(m_hWnd, BM_SETSTYLE, (BS_PUSHLIKE | BS_CHECKBOX) | BS_BITMAP, TRUE);
        } else {
          ::SendMessage(m_hWnd, BM_SETSTYLE, (BS_PUSHLIKE | BS_CHECKBOX) & ~BS_BITMAP, TRUE);
        }
        */

        ::ReleaseDC(m_hWnd, dc);
      } else {
        bitmap.setFromImage(NULL, NULL);
      }
    }
  }

  void MButton::addListener(const MButtonListenerPtr &listener) {
    listeners.push_back(listener);
  }


  MSize2D MButton::getMinimumSize() {
    updateMinButSize();

    return MSize2D::getLargest(minButSize, MComponent::getMinimumSize());
  }

  bool MButton::isMinimumSizeSet() {
    return true;
  }

  void MButton::fireClickListeners() {
    for (int i = 0; i < listeners.size(); ++i) {
      listeners[i]->onClick(this);
    }
  }
  
  void MButton::updateMinButSize() {
    if (minButSize.getWidth() == -1 || minButSize.getHeight() == -1) {
      if (image != NULL) {
        minButSize.setWidth(image->getWidth() + 4);
        minButSize.setHeight(image->getHeight() + 4);
      } else {
        HDC hdc = ::GetDC(m_hWnd);

        if (hdc != NULL) {
          SIZE size;
          ::GetTextExtentPoint(hdc, m_Caption.c_str(), m_Caption.size(), &size);
          minButSize.setWidth(size.cx);
          minButSize.setHeight(size.cy + 5);

          ::ReleaseDC(m_hWnd, hdc);
        }
      }
    }
  }

}

