
// AztecLib includes
#include <AztecGUICommonPCH.h>
#include <misc/ResourceManager.h>

// AztecGUI includes
#include <gui/util/MMenuFactory.h>

// Std includes
#include <istream>
#include <fstream>
#include <assert.h>


namespace Aztec {

  static void parseCommand(const std::string &command, 
                           const MMenuShellPtr &parentMenu, 
                           std::istream &reader, 
                           const MMenuItem::ListenerPtr &listener);

  static void parseCommand(const std::string &command, 
                           const MMenuShellPtr &parentMenu, 
                           int &index,
                           std::istream &reader, 
                           const MMenuItem::ListenerPtr &listener);

  MMenuFactory::MMenuFactory() {
  }

  static std::string readQuotedString(std::istream &reader) {
    
    std::string result;

    char ch;
    bool started = false;
    bool quoted = false;
    bool done = false;

    while (!done) {
      ch = reader.get();

      if (reader.eof()) {
        done = true;
        continue;
      }

      if (!started) {
        if (!isspace(ch)) {
          started = true;
        }
      }

      if (started) {
        // check for a quote
        if (ch == '\"') {
          // if we have already encoutered a quote, this must be the end of it all.
          if (quoted) {
            quoted = false;
            done = true;
          } else {
            quoted = true;
          }
          continue;
        } else if (!quoted && isspace(ch)) {
          done = true;
          continue;
        } else {
          result += ch;
        }
      }

    }

    return result;
  }

  static void parseMenu(MMenuPtr parentMenu, std::istream &reader, const MMenuItem::ListenerPtr &listener) {
    std::string openBrace;
    std::string command;

    reader >> openBrace;
    
    assert(openBrace == "{");

    reader >> command;

    while (command != "}") {
      parseCommand(command, parentMenu, reader, listener);
      reader >> command;
    }
  }

  class ListeningMenu : public MMenu {
  public:
    ListeningMenu(const MMenuItem::ListenerPtr &newListener)
      : MMenu()
    {
      listener = newListener;
    }

    bool onItemClick(const MMenuItemPtr &item) {
      if (listener != NULL) {
        listener->onClick(item);
      }
      return false;
    }

    MMenuItem::ListenerPtr listener;
  };

  static MMenuPtr createMenu(const MMenuShellPtr &parentMenu, int index, 
                             const std::string &menuName, 
                             const MMenuItem::ListenerPtr &listener) {
    MMenuPtr menu = listener != NULL ? new ListeningMenu(listener) : new MMenu;
    MMenuItemPtr item = new MMenuItem(menuName, "", menu);

  #ifndef _WIN32
    menu->init();
  #endif

    if (parentMenu != NULL) {
      if (index != -1) {
        parentMenu->addItem(item, index);
      } else {
        parentMenu->addItem(item);
      }
    }

    return menu;
  }

  static std::string removeQuotes(const std::string &src) {
    std::string result;
    unsigned int from;
	int to;

    for (from = 0; from < src.size(); ++from) {
      if (src[from] != ' ' && src[from] != '\"') {
        break;
      }
    }
    for (to = src.size() - 1; to > from; --to) {
      if (src[to] != ' ' && src[to] != '\"') {
        break;
      }
    }

    if (to == from) return "";

    return src.substr(from, to - from + 1);
  }

  static MMenuItemPtr parseMenuItem(const MMenuShellPtr &parentMenu, std::istream &reader, const MMenuItem::ListenerPtr &listener) {
    std::string menuText;
    std::string menuCommand;

    std::string cmd;
    std::string openBrace;
    bool enabled = true;
    std::string optionCmd;

    menuText = readQuotedString(reader);

    reader >> openBrace;

    assert(openBrace == "{");

    reader >> cmd;

    while (cmd != "}") {
      if (cmd == "command") {
        std::getline(reader, menuCommand);
        menuCommand = removeQuotes(menuCommand);
        if (menuCommand == "") {
          enabled = false;
        }
      } else if (cmd == "disabled") {
        enabled = false;
      } else if (cmd == "option") {
        std::getline(reader, optionCmd);
        optionCmd = removeQuotes(optionCmd);
      } else {
        parseCommand(cmd, parentMenu, reader, listener);
      }
      reader >> cmd;
    }

    MMenuItemPtr result = new MMenuItem(menuText, menuCommand);      
    if (optionCmd.length() > 0) {
      result->setOptionCommand(optionCmd);
    }
    result->setEnabled(enabled);
    parentMenu->addItem(result);

    return result;

  }

  static void parseCommand(const std::string &command, const MMenuShellPtr &parentMenu, int &index, std::istream &reader, const MMenuItem::ListenerPtr &listener) {
    if (command == "menu") {
      std::string menuName = readQuotedString(reader);
      
      MMenuPtr subMenu = createMenu(parentMenu, index, menuName, listener);
      parseMenu(subMenu, reader, listener);

    } else if (command == "item") {
      // otherwise we have a regular item to add

      MMenuItemPtr item = parseMenuItem(parentMenu, reader, listener);
    } else if (command == "separator") {
      parentMenu->addSeparator();
    }


  }

  static void parseCommand(const std::string &command, const MMenuShellPtr &parentMenu, std::istream &reader, const MMenuItem::ListenerPtr &listener) {
    if (command == "menu") {
      std::string menuName = readQuotedString(reader);
      
      MMenuPtr subMenu = createMenu(parentMenu, -1, menuName, listener);
      parseMenu(subMenu, reader, listener);

    } else if (command == "item") {
      // otherwise we have a regular item to add

      MMenuItemPtr item = parseMenuItem(parentMenu, reader, listener);
    } else if (command == "separator") {
      parentMenu->addSeparator();
    }

  }

  MMenuPtr MMenuFactory::loadMenu(const std::string &filename, const MMenuItem::ListenerPtr &listener) {
    MMenuPtr result = new MMenu;

    loadMenu(result, filename, listener);

    return result;
  }

  void MMenuFactory::loadMenu(const MMenuShellPtr &menu, int index, const std::string &filename, const MMenuItem::ListenerPtr &listener) {
    // Use the resource manager to find the correct location of the requested file.
    std::ifstream reader(ResourceManager::locateResource(filename).c_str());

    if (!reader.is_open()) {
      return;
    }

    while (!reader.eof()) {
      std::string command;
      
      reader >> command;

      parseCommand(command, menu, index, reader, listener);
    }

    return;
  }

  void MMenuFactory::loadMenu(const MMenuShellPtr &menu, const std::string &filename, const MMenuItem::ListenerPtr &listener) {
    loadMenu(menu, -1, filename, listener);
  }

}



