#include "AztecGUICommonPCH.h"

#include "AztecGUIMPluginManager.h"

#include <assert.h>

#ifdef WIN32

#include <io.h>
#include <direct.h>

#else

#include <sys/unistd.h>
#include <dlfcn.h>
#include <dirent.h>

#endif

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

namespace AztecGUI {

  class PluginDLL {
  public:
    typedef MPluginManager::ParameterPluginList ParameterPluginList;

    PluginDLL(HINSTANCE lib, 
              GetGUIPluginCountFunc getPluginCountFunction, 
              GetGUIPluginFunc getPluginFunction) 
    {
      dll = lib;
      getCountFunc = getPluginCountFunction;
      getPluginFunc = getPluginFunction;
    }
    PluginDLL(const PluginDLL &src) 
      : dll(src.dll), 
        getCountFunc(src.getCountFunc), 
        getPluginFunc(src.getPluginFunc),
        plugins(src.plugins)
    {
    }

    void freeDLL() {
      plugins.clear();
      FreeLibrary(dll);
      dll = NULL;
    }

    int getCount() {
      return getCountFunc();
    }

    MParameterPluginPtr getPlugin(int index) {
      return getPluginFunc(index);
    }

    bool loadPlugins() {
      if (getCount() == 0) {
        return false;
      }

      for (int i = 0; i < getCount(); ++i) {
        MParameterPluginPtr plugin = getPlugin(i);
        if (plugin != NULL) {
          plugins.push_back(plugin);
        }
      }

      return true;
    }

    ParameterPluginList plugins;

  private:

    HINSTANCE dll;
    GetGUIPluginCountFunc getCountFunc;
    GetGUIPluginFunc getPluginFunc;

  };

  /**
   * This is the real plugin manager class. The class presented in the DLL is
   * just an interface into this one, to avoid exposing unnecessary
   * implementation details.
   */
  class PluginManagerImpl {
  public:
    typedef MPluginManager::ParameterPluginList ParameterPluginList;

    PluginManagerImpl() {
    }

    ~PluginManagerImpl() {
    }
    
    void initialise() {
    }

    void cleanup() {
      for (unsigned int index = 0; index < plugins.size(); ++index) {
        plugins[index].freeDLL();
      }

      plugins.clear();
    }

    void loadPlugins(const std::string &directory, const std::string &fileMask) {
      typedef std::vector<std::string> FilenameList;
      
      FilenameList filenames;
      
      int Count = 0;
      
#ifdef WIN32
      _finddata_t    FindData;
      int            hFind, Result;
      
      std::string actualSearchMask = directory + "\\" + fileMask;
      hFind = _findfirst(actualSearchMask.c_str(), &FindData);
      
      if (hFind != -1) {
        Result = 0;
      } else {
        Result = 1;
      }
      
      while (Result == 0)
      {
        filenames.push_back(FindData.name);
        Result = _findnext(hFind, &FindData);
      }
      
      _findclose(hFind);
#else
      // TODO: use proper regular expression matching here.
      MStr expr = fileMask;
      expr.RemoveLeading(*);
      expr.RemoveTrailing(*);
      
      DIR *dirStream = opendir(directory.c_str());
      struct dirent *dir;
      
      dir = readdir(dirStream);
      
      while (dir != NULL) {
        MStr filename(dir->d_name);
        if (filename.findSubstring(expr) != -1) {
          filenames.push_back(filename);
        }                  
        
        dir = readdir(dirStream);
      }
      
      closedir(dirStream);
      
#endif
      
      FilenameList::iterator it;
      for (it = filenames.begin(); it != filenames.end(); ++it) {
        loadDLL(*it);
      }
    }

    void getPluginsForObject(const Aztec::MBaseObjectPtr &object, ParameterPluginList &result) {
      // loop over all the known plugins, and get any matches that we 
      // might have.
      PluginList::iterator DLLit;
      for (DLLit = plugins.begin(); DLLit != plugins.end(); ++DLLit) {
        for (unsigned int index = 0; index < (*DLLit).plugins.size(); ++index) {
          if ((*DLLit).plugins[index]->isUIForObject(object)) {
            result.push_back((*DLLit).getPlugin(index));
          }
        }
      }
    }

  protected:
    typedef std::vector< PluginDLL > PluginList;

    PluginList plugins;

    void loadDLL(const std::string filename) {
      HINSTANCE lib;
      lib = LoadLibrary(filename.c_str());

      if (lib == NULL) {
        return;
      }

      GetGUIPluginFunc getFunc;
      GetGUIPluginCountFunc getCountFunc;

      getFunc = (GetGUIPluginFunc)GetProcAddress(lib, "getGUIPlugin");

      if (getFunc == NULL) {
        FreeLibrary(lib);
        return;
      }

      getCountFunc = (GetGUIPluginCountFunc)GetProcAddress(lib, "getGUIPluginCount");

      if (getCountFunc == NULL) {
        FreeLibrary(lib);
        return;
      }

      PluginDLL dll(lib, getCountFunc, getFunc);

      if (!dll.loadPlugins()) {
        dll.freeDLL();
        return;
      }

      plugins.push_back(dll);
    }
  };

  static PluginManagerImpl *pluginMan = NULL;

  void MPluginManager::initialise() {
    assert(pluginMan == NULL);

    pluginMan = new PluginManagerImpl();
    pluginMan->initialise();
  }

  void MPluginManager::cleanup() {
    if (pluginMan != NULL) {
      pluginMan->cleanup();
      delete pluginMan;
      pluginMan = NULL;
    }
  }

  void MPluginManager::loadPlugins(const std::string &directory, const std::string &fileMask) {
    assert(pluginMan != NULL);
    pluginMan->loadPlugins(directory, fileMask);
  }

  void MPluginManager::getPluginsForObject(const Aztec::MBaseObjectPtr &object, ParameterPluginList &result) {
    assert(pluginMan != NULL);
    pluginMan->getPluginsForObject(object, result);
  }

}
