#ifndef MShiftState_Header
#define MShiftState_Header

#include <gui/MGuiBase.h>

namespace Aztec {

  /**
   * This class represents the state of the 'modifier' keys when an event 
   * happens. It stores which mouse buttons are down, and also which of the 
   * three modifier keys (ctrl, alt and shift) are down. Using this, we can do
   * simple checking for each event to see how to respond to the user.
   */
  class AZTECGUI_EXPORT MShiftState {
  public:
    MShiftState() :
      leftMouseDown(false) ,
      middleMouseDown(false) ,
      rightMouseDown(false) ,
      altPressed(false) ,
      ctrlPressed(false) ,
      shiftPressed(false) 
    {
    }

    MShiftState(const MShiftState &rhs) 
      :       leftMouseDown(rhs.leftMouseDown) ,
      middleMouseDown(rhs.middleMouseDown) ,
      rightMouseDown(rhs.rightMouseDown) ,
      altPressed(rhs.altPressed) ,
      ctrlPressed(rhs.ctrlPressed) ,
      shiftPressed(rhs.shiftPressed) 
    {
    }

    MShiftState(bool left, bool mid, bool right, 
                bool alt, bool ctrl, bool shift) 
      :       leftMouseDown(left) ,
      middleMouseDown(mid) ,
      rightMouseDown(right),
      altPressed(alt) ,
      ctrlPressed(ctrl) ,
      shiftPressed(shift) 
    {
    }


    bool leftMouseDown : 1;
    bool middleMouseDown : 1;
    bool rightMouseDown : 1;
    bool altPressed : 1;
    bool ctrlPressed : 1;
    bool shiftPressed : 1;

    /// Returns true if nothing is pressed, and false if anything at all is pressed.
    bool isEmpty() const {
      return !(leftMouseDown || middleMouseDown || rightMouseDown || altPressed || ctrlPressed || shiftPressed);
    }

    /// Returns true if there are no mouse buttons pressed.
    bool isMouseUp() const {
      return !(leftMouseDown || middleMouseDown || rightMouseDown);
    }

    /// Returns true if there are no keyboard modifier keys pressed.
    bool isKeyboardUp() const {
      return !(altPressed || ctrlPressed || shiftPressed);;
    }

    /// Less than operator to allow it to be sorted in stl sets and what not.
    bool operator<(const MShiftState &rhs) const {

      // The general sorting of this is that false < true
      if (leftMouseDown != rhs.leftMouseDown) {
        return leftMouseDown == false ? true : false;
      } else if (middleMouseDown != rhs.middleMouseDown) {
        return middleMouseDown == false ? true : false;
      } else if (rightMouseDown != rhs.rightMouseDown) {
        return rightMouseDown == false ? true : false;
      } else if (altPressed != rhs.altPressed) {
        return altPressed == false ? true : false;
      } else if (ctrlPressed != rhs.ctrlPressed) {
        return ctrlPressed == false ? true : false;
      } else if (shiftPressed != rhs.shiftPressed) {
        return shiftPressed == false ? true : false;
      } else {
        // if we have gotten here, then everything is equal, so return false.
        return false;
      }

    }
  };

}


#endif

