#include <Aztec3DPCH.h>
#include <tools/SlideEdgeTool.h>


// Aztec2 includes
#include <views/Aztec3DView.h>
#include <utils/AztecGLUtils.h>
#include <config/UIConfig.h>
#include <views/AztecViewManager.h>
#include <utils/SceneFunctions.h>
#include <functions/mesh/MeshFunctions.h>

// AztecLib includes
#include <MUIManager.h>
#include <MScene.h>
#include <MEditableMesh.h>
#include <misc/MSceneHelper.h>
#include <MSystemManager.h>

// standard includes
#include <assert.h>

namespace AztecGUI {
  
  SlideEdgeTool::SlideEdgeTool()
  {
    m_RequiresSel = true;
    dragging = false;
  }

  std::string SlideEdgeTool::getName() {
    return "toolEdgeSlide";
  }

  void drawEdges(const Aztec::MScenePtr &scene, 
                    const Aztec::MTreeObjectNodePtr &node,
                    const Aztec::MSceneObjectPtr &sceneObj, 
                    const Aztec::MEditableMeshPtr &mesh)
  {
    glLineWidth(5);
    glBegin(GL_LINES);

    // draw any edges that are selected.
    for (int i = 0; i < mesh->getNumTris(); ++i) {
      for (int edge = 0; edge < 3; ++edge) {
        if (mesh->isTriangleEdgeFlagged(i, edge, EDGE_SELECTED)) {
          int a, b;
          mesh->getVertsFromEdge(i, edge, &a, &b);

          Aztec::MVector3 ap = mesh->getVertexPosition(a);
          Aztec::MVector3 bp = mesh->getVertexPosition(b);

          // convert these into world space.
          ap = scene->objectToWorldSpace(node, ap);
          bp = scene->objectToWorldSpace(node, bp);

          glVertex3fv((float*)&ap);
          glVertex3fv((float*)&bp);
        }
      }
    }

    glEnd();
  }

  void drawPoints(const Aztec::MScenePtr &scene, 
                    const Aztec::MTreeObjectNodePtr &node,
                    const Aztec::MSceneObjectPtr &sceneObj, 
                    const Aztec::MEditableMeshPtr &mesh)
  {
    glDisable(GL_DEPTH_TEST);
    glColor3f(0.1, 0.1, 1.0);
    glPointSize(5);
    glEnable(GL_POINT_SMOOTH);
    glBegin(GL_POINTS);

    // draw any edges that are selected.
    for (int i = 0; i < mesh->getNumTris(); ++i) {
      for (int edge = 0; edge < 3; ++edge) {
        if (mesh->isTriangleEdgeFlagged(i, edge, EDGE_SELECTED)) {
          int a, b;
          mesh->getVertsFromEdge(i, edge, &a, &b);

          Aztec::MVector3 point = 0.5 * (mesh->getVertexPosition(a) + mesh->getVertexPosition(b));

          // convert these into world space.
          point = scene->objectToWorldSpace(node, point);

          glVertex3fv((float*)&point);
        }
      }
    }

    glEnd();
    glDisable(GL_POINT_SMOOTH);
    glEnable(GL_DEPTH_TEST);

  }

  int SlideEdgeTool::drawTool(bool select, const Aztec::MComponentPtr &component)
  {
    if (select) {
      glPushName(1);
      applyToEditableMeshes(Aztec::MScene::getGlobalScene(), Aztec::MScene::getGlobalScene()->getObjectList(), ifSelectedCriteria, drawEdges);
      glPopName();    
    } else {
      applyToEditableMeshes(Aztec::MScene::getGlobalScene(), Aztec::MScene::getGlobalScene()->getObjectList(), ifSelectedCriteria, drawPoints);
    }
    return 0;
  }

  void SlideEdgeTool::initialise() {
    dragging = false;
  }

  bool SlideEdgeTool::finish() {
    // finishing has no meaning for this tool, so return false;
    return false;
  }

  bool SlideEdgeTool::cancel() {
    if (dragging) {
      return false;
    } else {
      return true;
    }
  
  }

  bool SlideEdgeTool::inProgress() {
    return dragging;
  }

  class FindClosestEdge {
  public:
    FindClosestEdge(Aztec::MRay r, int *a, int *b, 
      Aztec::MTreeObjectNodePtr *node,
      Aztec::MSceneObjectPtr *sceneObj,
      Aztec::MEditableMeshPtr *mesh)
      : ray(r),
        bestNode(node),
        bestSceneObj(sceneObj),
        bestMesh(mesh)
    {
      bestA = a;
      bestB = b;
      *bestA = -1;
      *bestB = -1;
      bestDist = 0;
    }

    int *bestA, *bestB;
    Aztec::MTreeObjectNodePtr *bestNode;
    Aztec::MSceneObjectPtr *bestSceneObj;
    Aztec::MEditableMeshPtr *bestMesh;
    Aztec::MRay ray;
    double bestDist;

    void operator()(const Aztec::MScenePtr &scene, 
      const Aztec::MTreeObjectNodePtr &node,
      const Aztec::MSceneObjectPtr &sceneObj, 
      const Aztec::MEditableMeshPtr &mesh)
    {
      for (int i = 0; i < mesh->getNumTris(); ++i) {
        for (int edge = 0; edge < 3; ++edge) {
          if (mesh->isTriangleEdgeFlagged(i, edge, EDGE_SELECTED)) {
            int a, b;
            mesh->getVertsFromEdge(i, edge, &a, &b);
            
            Aztec::MVector3 ap = mesh->getVertexPosition(a);
            Aztec::MVector3 bp = mesh->getVertexPosition(b);
            
            // convert these into world space.
            ap = scene->objectToWorldSpace(node, ap);
            bp = scene->objectToWorldSpace(node, bp);

            // construct a ray for this edge;
            Aztec::MRay edgeRay(ap, bp-ap);

            double edgeDist = 0.0;
            double dist = Aztec::MMath::distanceBetween(edgeRay, ray, &edgeDist, NULL);

            if (edgeDist >= 0.0 && edgeDist <= 1.0) {
              if (*bestA == -1 || dist < bestDist) {
                *bestA = a;
                *bestB = b;
                *bestNode = node;
                bestDist = dist;
                *bestMesh = mesh;
                *bestSceneObj = sceneObj;
              }
            }
          }
        }
      }
    }
  };

  static Aztec::MRay getEdgeRay(const Aztec::MMeshPtr &mesh, int a, int b, bool normalise = true) {
    Aztec::MVector3 ap = mesh->getVertexPosition(a);
    Aztec::MVector3 bp = mesh->getVertexPosition(b);

    Aztec::MVector3 dir = bp - ap;

    if (normalise) {
      dir.normalize();
    }
    return Aztec::MRay(ap, dir);
  }


  static void getSelectedEdgeRays(const Aztec::MMeshPtr &mesh, SlideEdgeTool::EdgeRayMap &edges) {
    for (int i = 0; i < mesh->getNumTris(); ++i) {
      for (int edge = 0; edge < 3; ++edge) {
        if (mesh->isTriangleEdgeFlagged(i, edge, EDGE_SELECTED)) {
          int a, b;
          mesh->getVertsFromEdge(i, edge, &a, &b);

          edges[SlideEdgeTool::SlideEdge(a, b)] = getEdgeRay(mesh, a, b);
        }
      }
    }
  }




  int SlideEdgeTool::onMouseDown(const Aztec::MMouseEvent &event)
  {
    MXYZToolType::onMouseDown(event);

    int result = TOOLRESULT_DRAWNONE;

    // see if we have a picked manipulator.
    // if we do, then start the dragging process
    if (m_PickedManip != -1) {

      AztecGLCanvasPtr glWnd = AztecGLView::getGLCanvasFor(event.getComponent());

      // find our closest edge.
      applyToEditableMeshes(Aztec::MScene::getGlobalScene(), 
                            Aztec::MScene::getGlobalScene()->getObjectList(), 
                            ifSelectedCriteria, 
                            FindClosestEdge(glWnd->getRay(event.getX(), event.getY()), &draggingPointA, &draggingPointB,
                            &draggingNode, &draggingSceneObj, &draggingMesh));

      if (draggingPointA != -1 && draggingPointB != -1) {
        dragging = true;
        draggingMesh->takeSnapshot();

        getSelectedEdgeRays(draggingMesh, edgeRays);
      }

      result = TOOLRESULT_DRAWALL;
    }
    
    return TOOLRESULT_DRAWALL;
  }

  Aztec::MVector3 getWorldPos(const Aztec::MSceneObjectPtr &sceneObj, 
                             const Aztec::MMeshPtr &mesh,
                             int vertex) {

    return Aztec::MScene::getGlobalScene()->objectToWorldSpace(Aztec::MBaseObjectPtr(sceneObj), mesh->getVertexPosition(vertex));
  }

  void getTriPlaneAndDirection(Aztec::MSceneObjectPtr sceneObj, Aztec::MMeshPtr mesh, int tri, int a, int b, Aztec::MPlane &plane, Aztec::MVector3 &dir) {
    int edgeIndex = mesh->getTriangleEdge(tri, a, b);

    assert(edgeIndex != -1);

    a = mesh->getTriangleVertex(tri, edgeIndex);
    b = mesh->getTriangleVertex(tri, (edgeIndex + 1) % 3);
    int c = mesh->getTriangleVertex(tri, (edgeIndex + 2) % 3);

    Aztec::MVector3 ab, bc;

    ab = getWorldPos(sceneObj, mesh, b) - getWorldPos(sceneObj, mesh, a);
    bc = getWorldPos(sceneObj, mesh, c) - getWorldPos(sceneObj, mesh, b);

    ab.normalize();
    bc.normalize();

    plane.set(getWorldPos(sceneObj, mesh, b), ab / bc);
    dir.set((ab/bc)/ab);
    dir.normalize();
  }

  static int getPoint(const std::vector<int> &p, int index) {
    return p[(index+p.size()) % p.size()];
  }

  static double getDistanceAlong(const Aztec::MRay &a, const Aztec::MRay &b) {
    double along;
    Aztec::MMath::distanceBetween(a, b, &along);
    return along;
  }

  static void getNeighbouringPoints(const Aztec::MMeshPtr &mesh, int from, int to, int &a, int &b) {
    std::vector<int> points;
    mesh->getVerticesConnecting(from, points);

    // now find the to vertex in the points
    for (int i = 0; i < points.size(); ++i) {
      if (points[i] == to) {
        a = getPoint(points, i-1);
        b = getPoint(points, i+1);
        return;
      }
    }

    a = -1;
    b = -1;
  }

  static void fixPoints(const Aztec::MEditableMeshPtr &mesh, int from, int to, SlideEdgeTool::EdgeRayMap &rays) {
    std::vector<int> others;
    mesh->getVerticesConnecting(to, others);

    mesh->setVertexFlag(to, VERTEX_FLAGFORCHANGE);

    for (int i = 0; i < others.size(); ++i) {
      // we need to adjust any selected edges to make sure they line up properly.
      // But we ignore any edge that has the end vertex flagged as already adjusted.
      if (!mesh->isVertexFlagged(others[i], VERTEX_FLAGFORCHANGE)) {
        if (mesh->isEdgeFlagged(to, others[i], EDGE_SELECTED)) {
          // if the edge is flagged, we need to move the vertex others[i] so 
          // the edge to->others[i] is parallel to the original ray.
          //
          // To achieve this, we get the two edges that neighbour the 
          // others[i]->to edge, and find which edge makes a best match to 
          // slide the vertex along.
          int left, right;
          getNeighbouringPoints(mesh, others[i], to, left, right);

          // if we have no valid edges, then just return out.
          if (left == -1 && right == -1) {
            return;
          }

          Aztec::MRay leftRay = (left != -1) ? getEdgeRay(mesh, others[i], left, false) : Aztec::MRay();
          Aztec::MRay rightRay = (right != -1) ? getEdgeRay(mesh, others[i], right, false) : Aztec::MRay();

          Aztec::MRay edgeRay = rays[SlideEdgeTool::SlideEdge(to, others[i])];

          edgeRay.Org = mesh->getVertexPosition(to);
          // see which ray does the best intersecting.
          double leftDistance = (left != -1) ? getDistanceAlong(leftRay, edgeRay) : -1.0;
          double rightDistance = (right != -1) ? getDistanceAlong(rightRay, edgeRay) : -1.0;

          if (left != -1 && leftDistance >= 0) {
            mesh->setVertexPosition(others[i], leftRay.Org + leftRay.Dir * leftDistance);
          } else if (right != -1) {
            mesh->setVertexPosition(others[i], rightRay.Org + rightRay.Dir * rightDistance);
          }
          mesh->setVertexFlag(others[i], VERTEX_FLAGFORCHANGE);

          fixPoints(mesh, to, others[i], rays);
        }
      }
    }
  }

  int SlideEdgeTool::onMouseMove(const Aztec::MMouseEvent &event) {
    MXYZToolType::onMouseMove(event);
    if (dragging) {
      assert(draggingMesh != NULL);

      draggingMesh->retreiveSnapshot();
      draggingMesh->unsetVertexFlags(VERTEX_FLAGFORCHANGE);

      Aztec::MVector3 aPos = getWorldPos(draggingSceneObj, draggingMesh, draggingPointA);
      Aztec::MVector3 bPos = getWorldPos(draggingSceneObj, draggingMesh, draggingPointB);


      // get the triangles on the selected edge
      std::vector<int> tris;
      draggingMesh->getTrianglesWithEdge(draggingPointA, draggingPointB, tris);

      int onTri = -1;
      Aztec::MPlane plane;
      Aztec::MVector3 dir;

      for (int i = 0; i < tris.size(); ++i) {
        getTriPlaneAndDirection(draggingSceneObj, draggingMesh, tris[i], draggingPointA, draggingPointB, plane, dir); 

        // check to see if our ray intersections on the side of the edge that the triangle is on.
        if ((getCurrentRay().intersectWithPlane(plane) - aPos) * dir > 0) {
          onTri = tris[i];
          break;
        }
      }
      
      // if onTri isn't -1, it means we have a valid drag target
      if (onTri != -1) {
        Aztec::MVector3 intersectPoint = getCurrentRay().intersectWithPlane(plane) - bPos;
        Aztec::MVector3 ba = aPos - bPos;
        ba.normalize();
        Aztec::MVector3 bc = dir;

        float bAmount = ba * intersectPoint;
        float cAmount = bc * intersectPoint;


        Aztec::MVector3 mesh_ba = draggingMesh->getVertexPosition(draggingPointA) - draggingMesh->getVertexPosition(draggingPointB);
        Aztec::MVector3 mesh_bc = 
          draggingMesh->getVertexPosition(
            draggingMesh->getTriangleVertex(onTri, 
              (draggingMesh->getTriangleEdge(onTri, draggingPointA, draggingPointB) + 2)%3
            )
          ) - draggingMesh->getVertexPosition(draggingPointB);

        mesh_ba.normalize();
        mesh_bc.normalize();

        mesh_bc = (mesh_ba/mesh_bc)/mesh_ba;
        mesh_bc.normalize();

        Aztec::MVector3 mesh_pos = mesh_ba * bAmount + mesh_bc * cAmount + draggingMesh->getVertexPosition(draggingPointB);

        
        // get the polygon that we are dealing with.
        std::vector<int> points;
        draggingMesh->getPolygonVerticesFromTriangle(onTri, points);

        // find where our edge is.
        Aztec::MRay aray, bray;

        for (int i = 0; i < points.size(); ++i) {
          if (getPoint(points, i) == draggingPointA) {
            int other;
            if (getPoint(points, i-1) == draggingPointB) {
              other = getPoint(points, i+1);
            } else {
              other = getPoint(points, i-1);
            }
                  
            aray = getEdgeRay(draggingMesh, draggingPointA, other);
          }
          if (getPoint(points, i) == draggingPointB) {
            int other;
            if (getPoint(points, i-1) == draggingPointA) {
              other = getPoint(points, i+1);
            } else {
              other = getPoint(points, i-1);
            }
                  
            bray = getEdgeRay(draggingMesh, draggingPointB, other);
          }
        }
        Aztec::MRay edgeRay = getEdgeRay(draggingMesh, draggingPointA, draggingPointB);
        edgeRay.Org = mesh_pos;

        double along_a = getDistanceAlong(aray, edgeRay);
        double along_b = getDistanceAlong(bray, edgeRay);

        draggingMesh->setVertexPosition(draggingPointA, aray.Org + aray.Dir * along_a);
        draggingMesh->setVertexPosition(draggingPointB, bray.Org + bray.Dir * along_b);

        draggingMesh->setVertexFlag(draggingPointA, VERTEX_FLAGFORCHANGE);
        draggingMesh->setVertexFlag(draggingPointB, VERTEX_FLAGFORCHANGE);

        fixPoints(draggingMesh, draggingPointA, draggingPointB, edgeRays);
        fixPoints(draggingMesh, draggingPointB, draggingPointA, edgeRays);
      }



      return TOOLRESULT_DRAWALL;
    } else {
      return TOOLRESULT_DRAWNONE;
    }

  }

  int SlideEdgeTool::onMouseUp(const Aztec::MMouseEvent &event) {
    MXYZToolType::onMouseUp(event);
    if (dragging) {
      dragging = false;

      return TOOLRESULT_DRAWALL;
    } else {
      return TOOLRESULT_DRAWNONE;
    }
  }

  int SlideEdgeTool::getDefaultManip() {
    return 2;
  }

}