#include <Aztec3DPCH.h>
#include <tools/MXYZToolType.h>

// Aztec2 includes
#include <utils/AztecGLUtils.h>
#include <views/SelectingView.h>
#include <views/Aztec3DView.h>

// AztecLib includes
#include <MUIManager.h>
#include <MTransformObject.h>
#include <MListsTrees.h>
#include <MSystemManager.h>
#include <MEditableMesh.h>

// standard includes
#include <math.h>

namespace AztecGUI {

  using Aztec::MMeshPtr;
  using Aztec::MVector3;
  using Aztec::MUIManager;
  using Aztec::MComponentisedObject;
  using Aztec::MTriangle;
  using Aztec::MMatrix4;
  using Aztec::MBaseObjectPtr;
  using Aztec::MSceneObjectPtr;
  using Aztec::MSceneObject;
  using Aztec::MShapeObjectPtr;
  using Aztec::MShapeObject;
  using Aztec::MTreeObjectNodePtr;
  using Aztec::MTransformObjectPtr;
  using Aztec::MScenePtr;
  using Aztec::MSystemManager;



  // Goes through the vertices of a given mesh if they will be altered by an action
  void MXYZToolType::MarkMeshVerticesComponents(MMeshPtr Mesh, Aztec::AztecFlags CompMode) {
    if (Mesh == NULL)
      return;
  
    if (CompMode == Aztec::MComponentisedObject::POINT_TYPE) {
      for (int i = Mesh->getNumVerts()-1; i >= 0; i--) {
        if (Mesh->isVertexFlagged(i, VERTEX_SELECTED)) {
          Mesh->setVertexFlag(i, VERTEX_FLAGFORCHANGE);
        } else {
          Mesh->unsetVertexFlag(i, VERTEX_FLAGFORCHANGE);
        }
      }
    }
    if (CompMode == Aztec::MComponentisedObject::FACET_TYPE) {
      Mesh->unsetVertexFlags(VERTEX_FLAGFORCHANGE);

      for (int i = Mesh->getNumTris()-1; i >= 0; i--) {
        if (Mesh->isTriangleFlagged(i, TRIANGLE_SELECTED)) {
          for (int n=0; n<3; n++) {
            Mesh->setVertexFlag(Mesh->getTriangleVertex(i, n), VERTEX_FLAGFORCHANGE);
          }
        }
      }
    
    }
    if (CompMode == Aztec::MComponentisedObject::EDGE_TYPE) {
      int         i;
    
      Mesh->unsetVertexFlags(VERTEX_FLAGFORCHANGE);
      for (i=Mesh->getNumTris()-1; i>=0; i--) {
        for (int edge = 0; edge < 3; ++edge) {
          if (Mesh->isTriangleEdgeFlagged(i, edge, EDGE_SELECTED)) {
            Mesh->setVertexFlag(Mesh->getTriangleVertex(i, edge), VERTEX_FLAGFORCHANGE);
            Mesh->setVertexFlag(Mesh->getTriangleVertex(i, (edge+1) % 3), VERTEX_FLAGFORCHANGE);
          }
        }
      }
    
    }
  }

  void MXYZToolType::PrepareSelectedObjects(MVector3 &Origin, MMatrix4 &SelObjXFormMat) {
    MBaseObjectPtr BaseSelObj;
    MSceneObjectPtr SelObj;
    MTreeObjectNodePtr SelObjNode;
    MTransformObjectPtr XForm;
  
    MScenePtr scene = MSystemManager::getInstance()->getScene();

    BaseSelObj = scene->getSelectedObjectList()->getTail();
  
  
    scene->getSelectedObjectList()->beginIteration();
  
    while (( BaseSelObj = scene->getSelectedObjectList()->getNext() ) != NULL )
    {
      SelObjNode = scene->getObjectList()->findObject(BaseSelObj);
    
      SelObj = AZTEC_CAST(MSceneObject, BaseSelObj);
    
      if (SelObj != NULL) {
        prepareSceneObject(SelObj);
      
        {
          MTreeObjectNodePtr ParentNode;
        
          ParentNode = NULL;
        
          if (SelObjNode != NULL) {
            ParentNode = SelObjNode->getParent();
          }
        
          XForm = scene->getTransform(SelObjNode);
        
          if (XForm != NULL) {
            // Tell the Transformobject to keep a backup set of values for reference.
            XForm->holdValues();
            Origin = scene->getSelectionCentre();
          }
          scene->getWorldTransformMatrix(SelObjNode, SelObjXFormMat);
        }
      }
    }   
    scene->getSelectedObjectList()->endIteration();
  }

  void MXYZToolType::UpdateKeys(const MStr &UndoName, Aztec::AztecFlags KeyFlags, bool holdValues)
  {
    MBaseObjectPtr BaseObj;
    MSceneObjectPtr Obj;
    MScenePtr scene = MSystemManager::getInstance()->getScene();

    scene->getObjectList()->beginIteration();
    while (( BaseObj = scene->getObjectList()->getNext() ) != NULL ) {
      Obj = AZTEC_CAST(MSceneObject, BaseObj);
      if (Obj == NULL) {
        continue;
      }
    
      Aztec::MComponentisedObjectPtr compObj = Obj->getComponentObject();
    
      if (compObj != NULL && compObj->isInComponentMode()) {
        Aztec::MEditableComponentisedObjectPtr editableCompObj = Obj->getEditableComponentObject();
        editableCompObj->updateComponentAnimation(
          editableCompObj->getComponentMode(), 
          Aztec::MComponentisedObject::COMPONENT_SELECTED,
          scene->getTime(),
          Aztec::MSystemManager::getInstance()->getSettings()->m_Animate);

      } else if (Obj->isFlagged(OBJECTFLAG_SELECTED)) {
        Aztec::MTransformObjectPtr XForm;
      
        XForm = Obj->getTransformObject();
      
        if (XForm != NULL) {
          XForm->updateKey(Obj->getTime(), KeyFlags, Aztec::MSystemManager::getInstance()->getSettings()->m_Animate);
          if (holdValues) {
            XForm->holdValues();
          }
        }
      }
    }
    scene->getObjectList()->endIteration();
  
  }

  void MXYZToolType::getPlaneParams(const AztecGLCanvasPtr &canvas, MVector3 &Origin, MVector3 &Dir, MVector3 &ViewNorm, MMatrix4 &XFormMat)
  {
    canvas->getInverseTransform().transform(MVector3(0,0,1),ViewNorm);
  
    ViewNorm.x = -ViewNorm.x;
    ViewNorm.y = -ViewNorm.y;
    ViewNorm.z = ViewNorm.z;
  
    PrepareSelectedObjects(Origin, XFormMat);
    MMatrix4 mat(m_AxisTransform);
    mat.inverse();
    if (m_PickedManip == 1) // X Axis
    {
      Dir.set(0,0,1);
      if (fabs(Dir * ViewNorm) < 0.01)
        Dir.set(0,1,0);
      Dir = m_AxisTransform * Dir;
    }
    else if (m_PickedManip == 2) // Y Axis
    {
      Dir.set(0,0,1);
      if (fabs(Dir * ViewNorm) < 0.01)
        Dir.set(1,0,0);
      Dir = m_AxisTransform * Dir;
    }
    else if (m_PickedManip == 3) // Z Axis
    {
      Dir.set(1,0,0);
      if (fabs(Dir * ViewNorm) < 0.01) {
        Dir.set(0,1,0);
      }
      Dir = m_AxisTransform * Dir;
    } else {
      Dir.set(ViewNorm);
    }
  }

  void MXYZToolType::getPlaneParams(const AztecGLCanvasPtr &canvas, MConstraintType &con, const MMatrix4 &axisTransform, Aztec::AztecFlags flags) {
    MVector3 origin;
    MMatrix4 XFormMat;
  
    PrepareSelectedObjects(origin, XFormMat);

    if (m_PickedManip == 1) {
      con.setConstraint(Aztec::MRay(origin, axisTransform * Aztec::MVector3(1,0,0)));
    } else if (m_PickedManip == 2) {
      con.setConstraint(Aztec::MRay(origin, axisTransform * Aztec::MVector3(0,1,0)));
    } else if (m_PickedManip == 3) {
      con.setConstraint(Aztec::MRay(origin, axisTransform * Aztec::MVector3(0,0,1)));
    } else {
      if ((flags & DRAWAXIS_MIDDLE_TYPE) == DRAWAXIS_MIDDLE_NORMAL_X) {
        con.setConstraint(Aztec::MPlane(origin, axisTransform*MVector3(1,0,0)));
      } else if ((flags & DRAWAXIS_MIDDLE_TYPE) == DRAWAXIS_MIDDLE_NORMAL_Y) {
        con.setConstraint(Aztec::MPlane(origin, axisTransform*MVector3(0,1,0)));
      } else if ((flags & DRAWAXIS_MIDDLE_TYPE) == DRAWAXIS_MIDDLE_NORMAL_Z) {
        con.setConstraint(Aztec::MPlane(origin, axisTransform*MVector3(0,0,1)));
      } else {
        Aztec::MVector3 ViewNorm = canvas->getViewNormal();
        con.setConstraint(Aztec::MPlane(origin, ViewNorm));
      }
    }
  }

  void MXYZToolType::getPlaneParams(const AztecGLCanvasPtr &canvas, MConstraintType &con, const Aztec::MMatrix4 &axisTransform, const Aztec::MVector3 &origin, Aztec::AztecFlags flags) {

    MVector3 fakeOrigin;
    MMatrix4 XFormMat;
  
    PrepareSelectedObjects(fakeOrigin, XFormMat);

    if (m_PickedManip == 1) {
      con.setConstraint(Aztec::MRay(origin, axisTransform * Aztec::MVector3(1,0,0)));
    } else if (m_PickedManip == 2) {
      con.setConstraint(Aztec::MRay(origin, axisTransform * Aztec::MVector3(0,1,0)));
    } else if (m_PickedManip == 3) {
      con.setConstraint(Aztec::MRay(origin, axisTransform * Aztec::MVector3(0,0,1)));
    } else {
      if ((flags & DRAWAXIS_MIDDLE_TYPE) == DRAWAXIS_MIDDLE_NORMAL_X) {
        con.setConstraint(Aztec::MPlane(origin, axisTransform*MVector3(1,0,0)));
      } else if ((flags & DRAWAXIS_MIDDLE_TYPE) == DRAWAXIS_MIDDLE_NORMAL_Y) {
        con.setConstraint(Aztec::MPlane(origin, axisTransform*MVector3(0,1,0)));
      } else if ((flags & DRAWAXIS_MIDDLE_TYPE) == DRAWAXIS_MIDDLE_NORMAL_Z) {
        con.setConstraint(Aztec::MPlane(origin, axisTransform*MVector3(0,0,1)));
      } else {
        Aztec::MVector3 ViewNorm = canvas->getViewNormal();
        con.setConstraint(Aztec::MPlane(origin, ViewNorm));
      }
    }
  }


  int MXYZToolType::onMouseDown(const Aztec::MMouseEvent &event) {
    M3DToolType::onMouseDown(event);
    m_DownVec = m_Constraint.getConstrainedPoint(getCurrentRay());
    m_CurVec = m_Constraint.getConstrainedPoint(getCurrentRay());
    return TOOLRESULT_DRAWNONE;
  }

  int MXYZToolType::onMouseUp(const Aztec::MMouseEvent &event) {
    M3DToolType::onMouseUp(event);
    m_UpVec = m_Constraint.getConstrainedPoint(getCurrentRay());
    m_CurVec = m_Constraint.getConstrainedPoint(getCurrentRay());
    return TOOLRESULT_DRAWNONE;
  }

  int MXYZToolType::onMouseMove(const Aztec::MMouseEvent &event) {
    M3DToolType::onMouseMove(event);
    m_CurVec = m_Constraint.getConstrainedPoint(getCurrentRay());
    return TOOLRESULT_DRAWNONE;
  }

  void MXYZToolType::getAxisMatrix(const AztecGLCanvasPtr &canvas) {
    Aztec::MScenePtr scene = Aztec::MScene::getGlobalScene();

    if (Aztec::MUIManager::getAxisMode() == Aztec::MUIManager::WORLD_AXIS) {
      m_AxisTransform.identity();
    } else if (Aztec::MUIManager::getAxisMode() == Aztec::MUIManager::SCREEN_AXIS) {
      if (canvas == NULL) {
        m_AxisTransform.identity();
      } else {
        m_AxisTransform = MMatrix4(canvas->getModelView());
      }
    } else if (MUIManager::getAxisMode() == MUIManager::OBJECT_AXIS) {
      Aztec::MSceneObjectPtr sceneObj;
      Aztec::MBaseObjectPtr obj;

      sceneObj = AZTEC_CAST(MSceneObject, scene->getSelectedObjectList()->getTail());

      if (Aztec::MUIManager::getAxisMode() == MUIManager::PARENT_AXIS && sceneObj != NULL) {
        sceneObj = sceneObj->getParent();
      }

      if (sceneObj != NULL) {
        scene->getWorldTransformMatrix( scene->getObjectList()->findObject(sceneObj), m_AxisTransform );
      } else {
        m_AxisTransform.identity();
      }
    } else if (MUIManager::getAxisMode() == MUIManager::LOCAL_AXIS) {

      MSceneObjectPtr sceneObj;
      MBaseObjectPtr obj;

      sceneObj = AZTEC_CAST(MSceneObject, scene->getSelectedObjectList()->getTail());

      if (MUIManager::getAxisMode() == MUIManager::PARENT_AXIS && sceneObj != NULL) {
        sceneObj = sceneObj->getParent();
      }

      if (sceneObj != NULL) {
        // if we are not in component mode, use the object's axis, otherwise
        // use the components to figure things out.
        if (MUIManager::getComponentMode() == Aztec::MComponentisedObject::OBJECT_TYPE) {
          scene->getWorldTransformMatrix( scene->getObjectList()->findObject(sceneObj), m_AxisTransform );
        } else {
          MMatrix4 objXForm;
          MVector3 avgNormal(0,0,0.001f); // just a little bit off centre to allow no selections to not cause problems.
          int compType = MUIManager::getComponentMode();

          scene->getWorldTransformMatrix( scene->getObjectList()->findObject(sceneObj), objXForm );

          Aztec::MComponentisedObjectPtr compObj = sceneObj->getComponentObject();
          if (compObj != NULL) {
          // iterate over the object's components
            for (int n = 0; n < compObj->getComponentCount(compType); ++n) {
              if (!compObj->isComponentFlagged(compType, n)) {
                continue;
              }

              avgNormal += compObj->getComponentNormal(compType, n);
            }
          }

          avgNormal.normalize();

          // now we have an average normal, work out the matrix
          // which transform the z axis onto that normal
          MMatrix4 xform;
          {
            MVector3 zAxis(0,0,1), axis;
            xform.identity();
            float angle(acos(zAxis * avgNormal));

            if (fabs(angle) > 0.01) {
              axis = zAxis / avgNormal;
              axis.normalize();

              // rotate with vector takes radians
              xform.rotateWithVector(axis, angle);
            }
          }

          m_AxisTransform = xform * objXForm;

        }
      } else {
        m_AxisTransform.identity();
      }
    } else {
      // just a catchall here
      m_AxisTransform.identity();

    }

    m_AxisTransform.normaliseRotation();
    m_AxisTransform.m[0][3] = 0;
    m_AxisTransform.m[1][3] = 0;
    m_AxisTransform.m[2][3] = 0;
    m_AxisTransform.m[3][3] = 1;
    m_AxisTransform.m[3][2] = 0;
    m_AxisTransform.m[3][1] = 0;
    m_AxisTransform.m[3][0] = 0;

    m_AxisTransformInv = m_AxisTransform;
    m_AxisTransformInv.inverse();

  }

  void MXYZToolType::updateVectors() {
  }

  Aztec::MVector3 MXYZToolType::getSelectionCentre(const Aztec::MComponentPtr &component) {
    AztecViewPtr view = AztecView::getViewForComponent(component);
    SelectingView *selectView = AZTEC_CAST(SelectingView, view);

    if (selectView != NULL) {
      return selectView->getSelectionCentre();
    } else {
      return Aztec::MScene::getGlobalScene()->getSelectionCentre();
    }
  }

  bool MXYZToolType::anythingSelected(const Aztec::MComponentPtr &component) {
    AztecViewPtr view = AztecView::getViewForComponent(component);
    SelectingView *selectView = AZTEC_CAST(SelectingView, view);

    if (selectView != NULL) {
      return selectView->anythingSelected();
    } else {
      return Aztec::MScene::getGlobalScene()->anythingSelected();
    }
  }

  void MXYZToolType::prepareSceneObject(const Aztec::MSceneObjectPtr &sceneObj) {
    // set up which vertices need to be moved if in component mode
    Aztec::MShapeObjectPtr outObj = sceneObj->getShapeObject();
    Aztec::MComponentisedObjectPtr compObj = sceneObj->getComponentObject();
    if (compObj != NULL && compObj->isInComponentMode()) {
      Aztec::MEditableComponentisedObjectPtr editCompObj = sceneObj->getEditableComponentObject();
      if (editCompObj != NULL) {
        editCompObj->storeComponentPositions(editCompObj->getComponentMode());
      
        Aztec::MEditableMeshPtr mesh = AZTEC_CAST(Aztec::MEditableMesh, sceneObj->getShapeObject()->convertToMesh());
        if (mesh != NULL) {
          MarkMeshVerticesComponents(mesh, mesh->getComponentMode());

          Aztec::MEditableComponentisedObjectPtr textureMesh = AZTEC_CAST(Aztec::MEditableComponentisedObject, mesh->getTextureMesh());
          if (textureMesh != NULL) {
            textureMesh->storeComponentPositions(MUIManager::getComponentMode());
          }

        }
      }
    }
  }

  bool MXYZToolType::is3DView(const Aztec::MComponentPtr &component) {
    AztecViewPtr view = AztecView::getViewForComponent(component);
    Aztec3DViewPtr view3D = AZTEC_CAST(Aztec3DView, view);

    if (view3D != NULL) {
      return true;
    } else {
      return false;
    }

  }


  //------------------
  //  MConstraintType
  //------------------
  void MConstraintType::setConstraint(const Aztec::MPlane &plane) {
    constraintType = ctPlane;
    m_Plane = plane;
  }

  void MConstraintType::setConstraint(const Aztec::MRay &ray) {
    constraintType = ctRay;
    m_Ray = ray;
  }

  Aztec::MVector3 MConstraintType::getConstrainedPoint(const Aztec::MRay &srcRay) {
    if (constraintType == ctPlane) {
      return srcRay.intersectWithPlane(m_Plane);
    } else if (constraintType == ctRay) {
      double dist;

      Aztec::MMath::distanceBetween(m_Ray, srcRay, &dist);

      return m_Ray.Org + dist * m_Ray.Dir;
    }

    return MVector3();
  }

  Aztec::MVector3 MConstraintType::getConstraintNormal() {
    if (constraintType == ctPlane) {
      return Aztec::MVector3(m_Plane.A, m_Plane.B, m_Plane.C);
    } else if (constraintType == ctRay) {
      return m_Ray.Dir;
    }

    return Aztec::MVector3();
  }


}

