#include <Aztec3DPCH.h>

#include "tools/ChamferTool.h"

// Aztec2 includes
#include <views/Aztec3DView.h>
#include <utils/AztecGLUtils.h>
#include <config/UIConfig.h>
#include <functions/mesh/MeshFunctions.h>

// AztecLib includes
#include "MSystemManager.h"
#include "MEditableMesh.h"
#include "MAnimMesh.h"
#include "MUIManager.h"
#include <MComponentisedObject.h>

// standard includes
#include <math.h>
#include <GL/gl.h>
#include <set>
#include <assert.h>

namespace AztecGUI {
  using std::set;

  class DistanceTooFar {
  };

  ChamferTool::ChamferTool(bool solid)
  {
    m_RequiresSel = true;
    chamferDistance = 0.0f;
    lastGoodChamferDistance = 0;
    chamferVector.set(1,0,0);
    doingChamfer = false;

    solidChamfer = solid;

  }

  std::string ChamferTool::getName() {
    return solidChamfer ? "toolSolidChamfer" : "toolChamfer";
  }

  static std::vector<Aztec::MVector3> rectPoints;

  int ChamferTool::drawTool(bool select, const Aztec::MComponentPtr &component) {
    // we want to draw an arrow at the centre of the selection, 
    // and and pointing in the average normal direction.

    // update our normal. If nothing was selected, we don't draw anything
    if (!doingChamfer && !updateNormal()) {
      return 1;
    }

    AztecGLCanvasPtr GLWnd = AztecGLView::getGLCanvasFor(component);

    if (GLWnd == NULL) {
      return 0;
    }
  
    if (rectPoints.size() > 0) {
      glBegin(GL_QUADS);
      for (unsigned i = 0; i < rectPoints.size(); ++i) {
        glVertex3fv((float*)&rectPoints[i]);
      }
      glEnd();
    }

    glPushAttrib(GL_ENABLE_BIT);
    glDisable(GL_DEPTH_TEST);
  
  
    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glLoadIdentity();
  
    // Perform the viewport transformation
  
    GLWnd->doCameraTransform();
    glMatrixMode(GL_MODELVIEW);
  
    glPushMatrix();

    Aztec::MScene::getGlobalScene()->GLTransform(Aztec::MScene::getGlobalScene()->getSelectedObjectList()->getHead());

    glPushMatrix();

    float ScaleFact = GLWnd->getScalingFactor(m_CurrentPos);
    glTranslatef(m_CurrentPos.x, m_CurrentPos.y, m_CurrentPos.z);
  
    glScalef(ScaleFact, ScaleFact, ScaleFact);
  
    glMultMatrixf((float*)toolTransform.m);

    // draw the axis icon
    glDrawAxisIcon(UIConfig::get3DWidgetSize(), 0.5f, 2.0f, select, DRAWAXIS_Z | DRAWAXIS_ARROWS);

    // if we are extruding the faces, draw an arrow where we started from.
    if (doingChamfer) {
      glPopMatrix();
      glPushMatrix();

      ScaleFact = GLWnd->getScalingFactor(m_StartPos);
      glTranslatef(m_StartPos.x, m_StartPos.y, m_StartPos.z);
      glScalef(ScaleFact, ScaleFact, ScaleFact);
      glMultMatrixf((float*)toolTransform.m);

      glDrawAxisIcon(UIConfig::get3DWidgetSize(), 0.2f, 2.0f, false, DRAWAXIS_ALLGRAY | DRAWAXIS_Z | DRAWAXIS_ARROWS);
    }
  
    glPopMatrix();
    glPopMatrix();
    glPopMatrix();
  
    glPopAttrib();
  
    return 1;
  }

  bool ChamferTool::updateNormal() {
    Aztec::MSceneObjectPtr sceneObj;
    Aztec::MEditableMeshPtr meshObj;
  
    sceneObj = AZTEC_CAST(Aztec::MSceneObject, Aztec::MScene::getGlobalScene()->getSelectedObjectList()->getHead());

    if (sceneObj != NULL && sceneObj->getShapeObject() != NULL) {
      meshObj = AZTEC_CAST(Aztec::MEditableMesh, sceneObj->getShapeObject()->convertToMesh());
    }

    if (meshObj == NULL) {
      return false;
    }

    Aztec::MVector3 Norm, centre;
    int n, count;
  
    count = 0;
    Norm.set(0,0,0);

    if (Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::OBJECT_TYPE) {
      return false;
    }

    int numComps = meshObj->getComponentCount(Aztec::MUIManager::getComponentMode());

    // go through and extract the average normal and centre
    for(n = 0; n < numComps; n++) {
      if (!meshObj->isComponentFlagged(Aztec::MUIManager::getComponentMode(), n)) {
        continue;
      }

      centre += meshObj->getComponentCentre(Aztec::MUIManager::getComponentMode(), n);
      Norm += meshObj->getComponentNormal(Aztec::MUIManager::getComponentMode(), n);
      count++;
    }
    if (count) {
      Norm.normalize();
      centre /= (float)count;
    } else {
      return false;
    }
  
    m_StartPos = centre;
    m_CurrentPos = m_StartPos;
    m_ExtrudeCentre = centre;
    chamferVector = Norm;

    // transform the drawing to out z axis lines up with the normal.
    {
      Aztec::MVector3  x,y,z;

      toolTransform.identity();
      z = chamferVector;

      // construct two otrhonormal vectors
      y.set(-z.y, z.z, -z.x);
      x = z.crossProduct(y);
      y = z.crossProduct(x);

      toolTransform.m[0][0] = x.x; toolTransform.m[0][1] = x.y; toolTransform.m[0][2] = x.z;
      toolTransform.m[1][0] = y.x; toolTransform.m[1][1] = y.y; toolTransform.m[1][2] = y.z;
      toolTransform.m[2][0] = z.x; toolTransform.m[2][1] = z.y; toolTransform.m[2][2] = z.z;
    }

    return true;
  }

  const int getPoint(const std::vector<int> &points, int index) {
    return points[(index + points.size()) % points.size()];
  }

  const bool getPoint(const std::vector<bool> &points, int index) {
    return points[(index + points.size()) % points.size()];
  }

  inline Aztec::MVector3 getEdgeVector(const Aztec::MEditableMeshPtr &mesh, int from, int to) {
    const Aztec::MVector3 a = mesh->getVertexPosition(from);
    const Aztec::MVector3 b = mesh->getVertexPosition(to);
    Aztec::MVector3 n = (b-a);
    n.normalize();
    return n;
  }

  Aztec::MVector3 blend(const Aztec::MEditableMeshPtr &mesh, int from, int to, float distance) {
    const Aztec::MVector3 a = mesh->getVertexPosition(from);
    const Aztec::MVector3 b = mesh->getVertexPosition(to);
    Aztec::MVector3 n = (b-a);

    if (distance < 0.0 || distance >= n.length()) {
      throw DistanceTooFar();
    }

    n.normalize();
    return mesh->getVertexPosition(from) + distance * getEdgeVector(mesh, from, to);
  }

  typedef std::pair<int, int> ChamferEdge;

  typedef std::map<ChamferEdge, int> NewPointMap;


  inline int getNewPoint(ChamferEdge edge, const NewPointMap &newVerts) {
    NewPointMap::const_iterator i = newVerts.find(edge);
    if (i != newVerts.end()) {
      return i->second;
    } else {
      return -1;
    }
  }

  inline int getNewPoint(int from, int to, const NewPointMap &newVerts) {
    return getNewPoint(ChamferEdge(from, to), newVerts);
  }

  inline void addNewPoint(ChamferEdge edge, int newVertex, NewPointMap &newVerts) {
    newVerts[edge] = newVertex;
  }

  inline void addNewPoint(int from, int to, int newVertex, NewPointMap &newVerts) {
    newVerts[ChamferEdge(from,to)] = newVertex;
  }

  inline int splitEdge(const Aztec::MEditableMeshPtr &mesh, int from, int to, float dist, NewPointMap &newVerts) {
    ChamferEdge edge(from, to);

    int newPointIndex = getNewPoint(edge, newVerts);
    if (newPointIndex == -1) {
      // check to see if this new vertex overlaps a vertex on the edge going the other way.
      Aztec::MVector3 pos = blend(mesh, edge.first, edge.second, dist);

      ChamferEdge otherEdge(to, from);
      int otherPointIndex = getNewPoint(otherEdge, newVerts);
      if (otherPointIndex != -1) {
        if ((mesh->getVertexPosition(otherPointIndex) - mesh->getVertexPosition(edge.first)).length() < 
          (pos - mesh->getVertexPosition(edge.first)).length()) {
          throw DistanceTooFar();
        }
      }
      
      int newVertex = mesh->getNumVerts();
      addNewPoint(edge, newVertex, newVerts);
      mesh->addVertex(pos.x, pos.y, pos.z);
      return newVertex;
    } else {
      return newPointIndex;
    }
  }

  typedef std::vector<int> Polygon;
  typedef std::list<Polygon> PolygonList;

  bool polyEdgeInSet(const Polygon &poly, int index, const std::set<Edge> &edges) {
    return edges.find(Edge(getPoint(poly, index), getPoint(poly, index+1))) != edges.end();
  }

  template <class T> 
  bool polyEdgeInSet(T i, const std::set<Edge> &edges) {
    int a = *i;
    ++i;
    int b = *i;
    return edges.find(Edge(a, b)) != edges.end();
  }

  void splitupPoly(const Aztec::MEditableMeshPtr &mesh, 
                   const Polygon &poly, 
                   const Aztec::MVector3 &polyNormal,
                   const std::set<Edge> &selectedEdges, 
                   float dist, 
                   NewPointMap &newVerts, 
                   PolygonList &newPolygons,
                   bool makePoly) 
  {
    std::vector<bool> selected;
    std::vector<bool> good;
    selected.resize(poly.size());
    good.resize(poly.size());

    Polygon newPoly = poly;
    // iterate over the points, and add in any new ones.
    for (int i = 0; i < poly.size(); ++i) {
      good[i] = selected[i] = polyEdgeInSet(poly, i, selectedEdges);
      // if the edge is selected, split up the surrounding two.
      if (selected[i]) {
        
        int a = getPoint(poly, i-1);
        int b = getPoint(poly, i);
        int c = getPoint(poly, i+1);
        int d = getPoint(poly, i+2);

        // we have three edges
        // a--b--c--d

        Aztec::MRay ba(mesh->getVertexPosition(b), getEdgeVector(mesh, b, a));
        Aztec::MRay cd(mesh->getVertexPosition(c), getEdgeVector(mesh, c, d));


        Aztec::MVector3 un = polyNormal / getEdgeVector(mesh, b, c);

        Aztec::MRay newbc(mesh->getVertexPosition(b) + dist * un, getEdgeVector(mesh, b, c));

        double tempDist;

        Aztec::MMath::distanceBetween(ba, newbc, &tempDist);
        if (fabs(getEdgeVector(mesh, b, a) * getEdgeVector(mesh, b, c)) < 0.9999 &&
            fabs(getEdgeVector(mesh, c, d) * getEdgeVector(mesh, b, c)) < 0.9999) 
        {
          newPoly[i] = splitEdge(mesh, b, a, tempDist, newVerts);
          newPoly[(i+1)%newPoly.size()] = splitEdge(mesh, c, d , tempDist, newVerts);
        } else {
          good[i] = false;
        }
      }
    }

    // now check to see if any consecutive edges are selected. If they are, we need to add in an extra vertex for them
    // add in a quad for the corner face.
    if (makePoly) {
      for (int i = 0; i < poly.size(); ++i) {
        if (getPoint(selected, i) && getPoint(selected, i-1)) {
          // add in a vertex that bisects the angle prev->from->to
          Aztec::MVector3 edgea = getEdgeVector(mesh, getPoint(poly,i-1), getPoint(poly, i)); 
          Aztec::MVector3 edgeb = getEdgeVector(mesh, getPoint(poly, i), getPoint(poly,i+1)); 
          Aztec::MVector3 z = polyNormal;
          Aztec::MVector3 un = z / edgea;
          Aztec::MVector3 vn = z / edgeb;
          Aztec::MVector3 normal = un+vn;
          normal.normalize();

          Aztec::MVector3 y = normal;
          Aztec::MVector3 x = y / z;

          Aztec::MVector3 offsetA = dist * un;

          float amount = (offsetA * x) / (edgea * x);
          float newy = (offsetA * y) - amount * (edgea * y);

          Aztec::MVector3 pos = newy * y + mesh->getVertexPosition(getPoint(poly, i));

          mesh->addVertex(pos.x, pos.y, pos.z);
          newPoly[i] = mesh->getNumVerts() - 1;


          Polygon corner;
          corner.push_back(getPoint(poly, i));
          corner.push_back(getNewPoint(getPoint(poly, i), getPoint(poly,i+1), newVerts));
          corner.push_back(getPoint(newPoly, i));
          corner.push_back(getNewPoint(getPoint(poly, i), getPoint(poly,i-1), newVerts));
          newPolygons.push_back(corner);
        }
      }


    }

    if (makePoly) {
      // now go and make some polygons for the empty parts
      for (int i = 0; i < poly.size(); ++i) {
        // if the edge is selected, then we definitely have to make a new quad
        if (getPoint(selected, i) && getPoint(good, i)) {
          int selectCount = 0;
          int a = getPoint(poly, i);
          int b = getPoint(poly, i+1);
          int c = getPoint(newPoly, i+1);
          int d = getPoint(newPoly, i);

          // if the edge before us is selected, then we need to adjust the vertices to use.
          if (getPoint(selected, i-1)) {
            a = getNewPoint(getPoint(poly, i), getPoint(poly, i+1), newVerts);
            assert(a != -1);
          }
          // if the edge after us is selected, then we need to adjust the vertices to use.
          if (getPoint(selected, i+1)) {
            b = getNewPoint(getPoint(poly, i+1), getPoint(poly, i), newVerts);
            assert(b != -1);
          }

          newPolygons.push_back(Polygon());
          Polygon &p = newPolygons.back();
          p.push_back(a);

          // check to see if we have another vert to attach to.
          int a1 = getNewPoint(a, b, newVerts);
          if (a1 != -1) p.push_back(a1);

          int b1 = getNewPoint(b, a, newVerts);
          if (b1 != -1) p.push_back(b1);

          p.push_back(b);
          p.push_back(c);
          p.push_back(d);
        }
      }

      newPolygons.push_back(newPoly);
    }
  }

  static void doChamfer(const Aztec::MEditableMeshPtr &mesh, Aztec::AztecFlags flags, float dist) {

    std::set<int> inputVerts;
    std::set<Edge> inputEdges;
    NewPointMap newVerts;
    PolygonList newPolygons;

    // loop over all the tris, and find the vertices we need to handle.
    for (int tri = 0; tri < mesh->getNumTris(); ++tri) {
      for (int edge = 0; edge < 3; ++edge) {
        int a, b;
        mesh->getVertsFromEdge(tri, edge, &a, &b);
        if (mesh->isTriangleEdgeFlagged(tri, edge, flags)) {
          mesh->getVertsFromEdge(tri, edge, &a, &b);
          inputVerts.insert(a);
          inputVerts.insert(b);
          inputEdges.insert(Edge(a,b));


          // see how many edges eminate from the one vertex. If there is more than four, 
        }
      }
    }

    // unset all the triangle flags so we can work with them
    mesh->unsetTriangleFlags(TRIANGLE_FLAGFORCHANGE);

    std::vector<Polygon> polygons;
    std::vector<Aztec::MVector3> normals;

    // iterate over all the triangle. Any triangle that has a selected edge, take 
    // its polygon, and split it up into bits
    for (int tri = 0; tri < mesh->getNumTris(); ++tri) {
      for (int edge = 0; edge < 3; ++edge) {
        if (mesh->isTriangleEdgeFlagged(tri, edge, flags) && !mesh->isTriangleFlagged(tri, TRIANGLE_FLAGFORCHANGE)) {
          Polygon poly;
          mesh->getPolygonVerticesFromTriangle(tri, poly);
          mesh->flagPolygon(tri, TRIANGLE_FLAGFORCHANGE);
          polygons.push_back(poly);
          normals.push_back(mesh->getTriangleNormal(tri));
          splitupPoly(mesh, poly, mesh->getTriangleNormal(tri), inputEdges, dist, newVerts, newPolygons, false); 
          break;
        }
      }
    }

    for (int i = 0; i < polygons.size(); ++i) {
      splitupPoly(mesh, polygons[i], normals[i], inputEdges, dist, newVerts, newPolygons, true); 
    }

    // delete the triangles we are about to retriangulate.
    mesh->deleteTriangleFlag(TRIANGLE_FLAGFORCHANGE);

    // split all the edges up
    while (newVerts.size() > 0) {
      NewPointMap::iterator it = newVerts.begin();

      // split this edge
      int from = it->first.first;
      int to = it->first.second;
      int a = it->second;

      NewPointMap::iterator other = newVerts.find(ChamferEdge(to, from));

      // first make the new edges from->a->to
      mesh->divideEdgeWithVertex(from, to, a);

      // now if we have a second point, make the edges a->b->to
      if (other != newVerts.end()) {
        mesh->divideEdgeWithVertex(a, to, other->second);
        newVerts.erase(other);
      }

      newVerts.erase(it);
    }


    // now go and make some new triangles
    int polyCount = 0;
    for (PolygonList::iterator it = newPolygons.begin(); it != newPolygons.end(); ++it) {
      mesh->triangulatePolygon(*it);
      polyCount++;
    }


  }

  int ChamferTool::onMouseDown(const Aztec::MMouseEvent &event) {
    MXYZToolType::onMouseDown(event);

    Aztec::MSceneObjectPtr sceneObj;
    Aztec::MEditableMeshPtr MeshObj;
  
    sceneObj = AZTEC_CAST(Aztec::MSceneObject, Aztec::MScene::getGlobalScene()->getSelectedObjectList()->getHead());

    if (sceneObj == NULL) {
      Aztec::MSystemManager::getInstance()->logOutput("Error: Face Extude requires one scene object to work with.");
      return TOOLRESULT_DRAWNONE;
    }
    MeshObj = AZTEC_CAST(Aztec::MEditableMesh, sceneObj->getShapeObject()->convertToMesh());
    if (MeshObj == NULL) {
      Aztec::MSystemManager::getInstance()->logOutput("Error: Face Extude requires Mesh objects to work with.");
      return TOOLRESULT_DRAWNONE;
    }

    if (!updateNormal()) {
      doingChamfer = false;
      Aztec::MSystemManager::getInstance()->logOutput("Error: Face Extrude requires some faces selected.");
    
      return TOOLRESULT_DRAWNONE;
    }

    doingChamfer = true;
    lastBadChamferDistance = 50000;
    lastGoodChamferDistance = 0.0;
    {
      Aztec::MSceneObjectPtr sceneObj;
      Aztec::MEditableMeshPtr MeshObj;
    
      sceneObj = AZTEC_CAST(Aztec::MSceneObject, Aztec::MScene::getGlobalScene()->getSelectedObjectList()->getHead());

      if (sceneObj == NULL) {
        Aztec::MSystemManager::getInstance()->logOutput("Error: Face Extude requires one scene object to work with.");
        return TOOLRESULT_DRAWNONE;
      }
      MeshObj = AZTEC_CAST(Aztec::MEditableMesh, sceneObj->getShapeObject()->convertToMesh());
      if (MeshObj == NULL) {
        Aztec::MSystemManager::getInstance()->logOutput("Error: Face Extude requires Mesh objects to work with.");
        return TOOLRESULT_DRAWNONE;
      }
    
    }
  

    Aztec::MRay viewRay, normalRay;
    Aztec::MVector3  e1,e2;

    e2 = Aztec::MScene::getGlobalScene()->objectToWorldSpace(Aztec::MBaseObjectPtr(MeshObj), chamferVector);



    normalRay.set(m_StartPos, e2);
    viewRay = getCurrentRay();

    Aztec::MMath::distanceBetween(normalRay, viewRay, &chamferDistance, NULL);
    mouseStartedPos = m_StartPos + normalRay.Dir * chamferDistance;

    chamferDistance = 0.0;

    // if we are in point mode, we get split the dges around the selected points
    // and move the one point.
    if (Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::POINT_TYPE) {

    // extrude the faces if that is the mode we are in
    } else if (Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::FACET_TYPE) {

      // For face mode we have to make it extract the border of faces

    } else if (Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::EDGE_TYPE) {

      MeshObj->takeSnapshot();

      // We need to build up a list of edges that are selected, and build the 
      // list according to the way the edges are connected. This is important 
      // because we need a consistent way of determining if we are dealing with 
      // the left or right hand side of an edge.
      
      // unset all the edge flags we might be able to use.
      for (int t = 0; t < MeshObj->getNumTris(); ++t) {
        for (int e = 0; e < 3; ++e) {
          MeshObj->unsetEdgeFlag(t, e, EDGE_FLAGFORCHANGE);
        }
      }

//      splitEdgesAroundEdges(MeshObj, EDGE_SELECTED, 0.5);
    }
  
    return TOOLRESULT_DRAWALL;
  }


  int ChamferTool::onMouseMove(const Aztec::MMouseEvent &event) {
    MXYZToolType::onMouseMove(event);
    // if we are extruding, keep track of how the mouse has moved etc.
  
    if (doingChamfer) {
      Aztec::MSceneObjectPtr sceneObj;
      Aztec::MEditableMeshPtr MeshObj;
    
      sceneObj = AZTEC_CAST(Aztec::MSceneObject, Aztec::MScene::getGlobalScene()->getSelectedObjectList()->getHead());

      if (sceneObj == NULL) {
        Aztec::MSystemManager::getInstance()->logOutput("Error: Face Extude requires one scene object to work with.");
        return TOOLRESULT_DRAWNONE;
      }
      MeshObj = AZTEC_CAST(Aztec::MEditableMesh, sceneObj->getShapeObject()->convertToMesh());
      if (MeshObj == NULL) {
        Aztec::MSystemManager::getInstance()->logOutput("Error: Face Extude requires Mesh objects to work with.");
        return TOOLRESULT_DRAWNONE;
      }
    
      AztecGLCanvasPtr glWnd = AztecGLView::getGLCanvasFor(event.getComponent());
    
      if (glWnd == NULL) {
        chamferDistance = (float)(m_DownPos.x - m_CurPos.x) / 25.0f;
      } else {
        Aztec::MRay viewRay, normalRay;
        Aztec::MVector3  e1,e2;

        e2 = Aztec::MScene::getGlobalScene()->objectToWorldSpace(Aztec::MBaseObjectPtr(MeshObj), chamferVector);



        normalRay.set(mouseStartedPos, e2);
        viewRay = glWnd->getRay(event.getX(), event.getY());

        Aztec::MMath::distanceBetween(normalRay, viewRay, &chamferDistance, NULL);
        chamferDistance = -chamferDistance;
      }

      m_CurrentPos = m_StartPos - chamferDistance * chamferVector;

      if (chamferDistance < 0) chamferDistance = 0;
      if (chamferDistance > lastBadChamferDistance) chamferDistance = lastBadChamferDistance;


      if (Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::EDGE_TYPE) {
        bool chamferFailed = true;
        int failedCount = 0;
        do {
          float targetChamferDistance = chamferDistance;
          try {
            MeshObj->retreiveSnapshot();
            doChamfer(MeshObj, EDGE_SELECTED, chamferDistance);
            lastGoodChamferDistance = chamferDistance;
            chamferFailed = false;
          } catch (DistanceTooFar d) {
            // if it failed, then we have to do the chamfer again, but with the last good distance.
            lastBadChamferDistance = chamferDistance;
            chamferDistance = 0.5 * (lastGoodChamferDistance + lastBadChamferDistance);
            ++failedCount;
//            doChamfer(MeshObj, EDGE_SELECTED, chamferDistance);
          }

          if (failedCount >= 20) {
            chamferDistance = lastGoodChamferDistance;
          }
          if (failedCount > 20) {
            MeshObj->retreiveSnapshot();
            break;
          }
        } while (chamferFailed);
      }

      MeshObj->calculateNormals();
    
      return TOOLRESULT_DRAWALL;
    }
  
    return TOOLRESULT_DRAWNONE;
  }

  int ChamferTool::onMouseUp(const Aztec::MMouseEvent &event) {
    MXYZToolType::onMouseUp(event);

    if (doingChamfer) {
      doingChamfer = false;
    }
  
    return TOOLRESULT_DRAWALL;
  }

}
