#include <Aztec3DPCH.h>

#include <functions/edit/GraphFunctions.h>

// Aztec2 includes
#include <views/AztecViewManager.h>
#include <views/AztecGraphView.h>

// AztecLib includes
#include <MScene.h>
#include <MSystemManager.h>
#include <MUIManager.h>
#include <MListsTrees.h>
#include <MEditableMesh.h>


namespace AztecGUI {
  
  static int graphMergeSelectedKeys(const StringVector &args, std::string &result) {
    AztecGraphViewPtr graphView = AZTEC_CAST(AztecGraphView, AztecViewManager::getCurrentView());
  
    if (graphView == NULL) {
      Aztec::MSystemManager::getInstance()->logOutput("Cannot merge selected keys when the current view is not a Graph View");
      return FunctionManager::FAIL;
    }

    GraphComponent *graph = graphView->getGraphComponent();
    std::vector<GraphPair> graphValues;
    
    graph->getGraphValues(graphValues);
    
    // iterate over the graph values, and try and merge selected keys
    for (int index = 0; index < graphValues.size(); ++index) {
      MFloatValuePtr value = graphValues[index].value;
      MFloatKeyList *keyList = AZTEC_CAST(MFloatKeyList, value);
      
      // if we have a valid key list, work with that.
      if (keyList != NULL) {
        // loop over the selected keys, and merge the two together
        std::vector<int> keyTimes;
        
        keyList->getSelectedKeyTimes(keyTimes);
        
        // if we have no selected keys, or just one, then we just continue 
        // on to the next keylist.
        if (keyTimes.size() <= 1) {
          continue;
        }
        
        long avgTime = 0;
        float avgValue = 0.0;
        
        // now we calculate the average time for the selected keys.
        for (int keyIndex = 0; keyIndex < keyTimes.size(); ++keyIndex) {
          avgTime += keyTimes[keyIndex];
          avgValue += keyList->getValueAtTime(keyTimes[keyIndex]);
        }
        avgTime /= keyTimes.size();
        avgValue /= keyTimes.size();
        
        // now round the time off to the granularity
        avgTime = ((int)(0.5 + (float)avgTime / keyList->getGranularity()))*keyList->getGranularity();
        
        // now remove all the keys except for the first one.
        for (int keyIndex = keyTimes.size() - 1; keyIndex > 0; --keyIndex) {
          keyList->deleteKey(keyTimes[keyIndex]);
        }
        
        MFloatKeyPtr key = AZTEC_CAST(MFloatKey, keyList->getKeyAtTime(keyTimes[0]));
        
        key->setKeyTime(avgTime);
        key->setValue(avgValue);
      }
      
    }
    
    AztecViewManager::redrawAllViews();

    return FunctionManager::SUCCEED;
  }
  
  static int editDelete(const StringVector &args, std::string &result) {
    DeletingView *deleter = AZTEC_CAST(DeletingView, AztecViewManager::getCurrentView());
    
    // try the viewport specific action
    if (deleter != NULL) {
      deleter->deleteSelected();
      AztecViewManager::redrawCurrentView();
      return FunctionManager::SUCCEED;
    } else {
      // otherwise we have failed, so do nothing.
      return FunctionManager::FAIL;
    }

  }

  void registerGraphFunctions(FunctionManager &man) {

    man.registerFunction("graphMergeSelectedKeys", graphMergeSelectedKeys);

  }


}

