
#include <Aztec3DPCH.h>
#include <iostream>

#include <controls/MultiSplitter.h>
#include <controls/FourWaySplitterMethod.h>


namespace AztecGUI {

  class SplitterLayout : public Aztec::MLayoutManager {
  public:
    SplitterLayout(MultiSplitter *splitter) {
      split = splitter;
    }

    void addComponent(Aztec::MComponentPtr component) {
      children.push_back(component);
    }

    void addComponent(Aztec::MComponentPtr component, Aztec::MConstraint constraints) {
  #ifdef _WIN32
//  #warning Please read about Linux/GCC/RH9 issue + workaround in $(aztec2)/controls/MultiSplitter.cpp!
     /**
      * Using Windows and MSVC6 (sp5), the addComponent(component) method from
      * above is executed from within the MainWindow class (see controls/MainWindow.cpp)
      * to add new views.
      * However, on Linux (rh9) using GCC 3.2, this method is called instead, and I have
      * absolutely no idea how it's possible that a method with 2 args is called when
      * asking for the method with only 1 argument (as seen above).
      *
      * I suspect that GCC 3.2 has some problems here..
      * (3.3 or 3.4 will fortunately have a new C++ parser and other new stuff, perhaps that'll help?)
      *
      * (rve - 20 May, 2003)
      */
      int index = constraints.m_Value - 1;
      if (children.size() <= index) {
        children.resize(index + 1);
      }
      children[index] = component;
  #else
  #warning Please read about Linux/GCC/RH9 issue + workaround in $(aztec2)/controls/MultiSplitter.cpp!
      // Temporary workaround.. do what the real 1 argumented addComponent() would do.
      // Might want to retry using GCC 3.3 or 3.4 when it's released.
      children.push_back(component);
  #endif
    }

    void removeComponent(Aztec::MComponentPtr component) {
      for (int i = 0; i < children.size(); ++i) {
        if (children[i] == component) {
          children[i] = NULL;
          return;
        }
      }
    }

    void layoutContainer(Aztec::MContainerPtr container) {
      for (int i = 0; i < children.size(); ++i) {
        if (i < split->getSplitterMethod()->getPaneCount()) {
          if (children[i] != NULL) {
            Aztec::MRect2D rect = split->getSplitterMethod()->getRectFor(split, i);
            if (rect.getWidth() <= 0 || rect.getHeight() <= 0) {
              children[i]->setVisible(false);
            } else {
              children[i]->setPosition(rect.x1, rect.y1);
              children[i]->setSize(rect.getWidth(), rect.getHeight());
              children[i]->setVisible(true);
            }
          }
        } else {
          children[i]->setVisible(false);
        }
      }
    }

    Aztec::MSize2D getMinimumSize(Aztec::MContainerPtr container) {
      return Aztec::MSize2D(50,50);
    }

    Aztec::MSize2D getPreferredSize(Aztec::MContainerPtr container) {
      return Aztec::MSize2D(50, 50);
    }

    Aztec::MConstraint getConstraint(const Aztec::MComponentPtr &component) {
      // the splitter layout doesn't have constraints, so return an empty one.
      for (int i = 0; i < children.size(); ++i) {
        if (children[i] == component) {
          // return index + 1 so the constraint never looks like an empty constraint
          return Aztec::MConstraint(i + 1);
        }
      }
      return Aztec::MConstraint();
    }

  private:
    MultiSplitter *split;
    std::vector<Aztec::MComponentPtr> children;
  };

#ifdef _WIN32
  static void drawLine(HDC dc, int x1, int y1, int x2, int y2) {
    ::MoveToEx(dc, x1, y1, NULL);
    ::LineTo(dc, x2, y2);
  }
  static void draw3DRect(HDC dc, int left, int top, int width, int height) {

    int x1, y1, x2 ,y2;

    x1 = left-1;
    y1 = top-1;
    x2 = left + width;
    y2 = top + height;

    HPEN oldPen, pen;
    HBRUSH brush = ::GetSysColorBrush(COLOR_3DFACE);

    pen = ::CreatePen(PS_SOLID, 1, ::GetSysColor(COLOR_BTNSHADOW));
    oldPen = (HPEN)::SelectObject(dc, pen);
    drawLine(dc, x1-1,y1-1,x2,y1-1);
    drawLine(dc, x1-1,y2,x1-1,y1-1);
    ::DeleteObject(pen);

    pen = ::CreatePen(PS_SOLID, 1, ::GetSysColor(COLOR_3DDKSHADOW));
    ::SelectObject(dc, pen);
    drawLine(dc,x1,y1,x2,y1);
    drawLine(dc,x1,y2,x1,y1);
    ::DeleteObject(pen);

    pen = ::CreatePen(PS_SOLID, 1, ::GetSysColor(COLOR_BTNHIGHLIGHT));
    ::SelectObject(dc, pen);
    drawLine(dc,x2,y1,x2,y2);
    drawLine(dc,x2,y2,x1,y2);
    ::DeleteObject(pen);

    pen = ::CreatePen(PS_SOLID, 1, ::GetSysColor(COLOR_3DLIGHT));
    ::SelectObject(dc, pen);
    drawLine(dc,x2+1,y1,x2+1,y2+1);
    drawLine(dc,x2+1,y2+1,x1,y2+1);
    ::DeleteObject(pen);

    ::SelectObject(dc, oldPen);
  }

  static void draw3DRect(HDC dc, const Aztec::MRect2D &rect) {
    draw3DRect(dc, rect.x1, rect.y1, rect.getWidth(), rect.getHeight());
  }
#endif

  MultiSplitter::MultiSplitter()
  {
    draggingHit = SplitterMethod::NOTHING;
    method = new FourWaySplitter();
    needsUpdating = false;
    componentMaximised = false;
  }

  MultiSplitter::~MultiSplitter() {
    delete method;
  }


  void MultiSplitter::setSplitterMethod(SplitterMethod *method) {
    this->method = method;

    doLayout();
    refresh();
  }

  SplitterMethod *MultiSplitter::getSplitterMethod() const {
    return method;
  }

  void MultiSplitter::toggleMaximisedComponent(const Aztec::MComponentPtr &component) {
    if (componentMaximised) {
      componentMaximised = false;
      method->maximisePane(-1);
    } else {
      Aztec::MConstraint constraint = getLayoutManager()->getConstraint(component);
      method->maximisePane(constraint.m_Value - 1);
      if (constraint.m_Value > 0) {
        componentMaximised = true;
      }
    }
    doLayout();
  }

  void MultiSplitter::onCreate() {
    setLayoutManager(new SplitterLayout(this));
  }


  bool MultiSplitter::onPaint()
  {
#ifdef _WIN32 
    // fill in the whole rectangle.
    RECT rect;
    Aztec::MSize2D r = getClientSize();
    rect.left = r.getLeft();
    rect.top = r.getTop();
    rect.right = r.getRight();
    rect.bottom = r.getBottom();

    ::FillRect(paintDC, &rect, ::GetSysColorBrush(COLOR_3DFACE));

    // now iterate over the views, and paint in the borders
    for (int i = 0; i < method->getPaneCount(); ++i) {
      draw3DRect(paintDC, method->getRectFor(this, i));
    }

#else

#endif
    return true;
  }
  
  bool MultiSplitter::onMouseMove(const Aztec::MMouseEvent &event) {

    if (draggingHit == SplitterMethod::NOTHING) {
      SplitterMethod::HitTestResultEnum result = method->hitTest(this, event.getX(), event.getY());
      setMouseCursor(method->getCursorForHit(result));
    } else {
      if (event.getPos() != lastMouse) {
        method->doDrag(this, draggingHit, event.getX(), event.getY());
        doLayout();
        refresh();
        needsUpdating = true;
        lastMouse = event.getPos();
      }
    }

    return true;
  }

  bool MultiSplitter::onMousePressed(const Aztec::MMouseEvent &event) {
    setMouseCapture(true);
    draggingHit = method->hitTest(this, event.getX(), event.getY());

    if (draggingHit != SplitterMethod::NOTHING) {
      method->beginDrag(this, draggingHit, event.getX(), event.getY());
    }

    return true;
  }

  bool MultiSplitter::onMouseReleased(const Aztec::MMouseEvent &event) {
    if (draggingHit != SplitterMethod::NOTHING) {
      method->endDrag(this, draggingHit, event.getX(), event.getY());
      draggingHit = SplitterMethod::NOTHING;
      doLayout();
      refresh();
    }

    setMouseCapture(false);
    return true;
  }


}

