#include "StdAfx.h"

#include "ASETranslator.h"

#include "MBaseObject.h"
#include "MSystemManager.h"
#include "MMeshShape.h"

#include <stdio.h>
#include <assert.h>

#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


MASETranslator::MASETranslator() {
  m_TabCount = 0;
}

MASETranslator::~MASETranslator() {
}

// Class related
MTranslatorPtr MASETranslator::createNew() const {
  MASETranslator   *NewTranslator;
  
  NewTranslator = new MASETranslator;
  
  return NewTranslator;
}

int MASETranslator::SeparateLine(char *&Buffer, MStr &TagName, MStr &Parameter) {
  if (!Buffer)
    return 0;
  
  char  *Src;
  int   Result;
  
  Src = Buffer;
  
  // Skip past any spaces
  while (*Src == ' ' || *Src == '\t')
    Src++;
  
  if (*Src != '*')
  {
    TagName = " ";
    TagName[0] = *Src;
    return 0;
  }
  
  char     *Start, *End;
  
  Start = Src;
  while (*Src != ' ' && *Src != '\t')
    Src++;
  
  End = Src - 1;
  *Src = '\x0';
  TagName = Start;
  *Src = ' ';
  
  while (*Src == ' ' || *Src == '\t')
    Src++;
  Start = Src;
  
  while (*Src != '\n')
    Src++;
  *Src = '\x0';
  Parameter = Start;
  *Src = '\n';
  
  Result = Src - Buffer + 1;
  if (*Src == '\n')
    Src++;
  
  Buffer = Src;
  
  return Result;
}

bool MASETranslator::canImportFile(MStr Filename) {
  FILE     *hFile;
  char     Data[512], *Buf;
  
  Filename.Replace('/', '\\');
  
  hFile = fopen(Filename,"rt");

  if (!hFile)
    return false;
  
  fread(Data,512,1,hFile);
  
  fclose(hFile);
  
  
  MStr  TagName, Param;
  
  Buf = Data;
  SeparateLine(Buf, TagName, Param);
  
  if (TagName.compareNoCase("*3DSMAX_ASCIIEXPORT") == 0)
    return true;
  
  return false;
}

void MASETranslator::AdvanceLine(char *&Buf) {
  while (*Buf != '\n' && *Buf != '\x0')
    Buf++;
  
  if (*Buf == '\n')
    Buf++;
}

bool MASETranslator::ParseFile(char *Buffer, int Length, MScenePtr Scene) {
  MSystemManagerPtr SysMan;
  char              *Buf, *End;
  MStr              Tag, Param;
  int               Result;
  
  SysMan = Aztec::getSystemManager();
  
  SysMan->logOutput("MASETranslator: Beginning file translate");
  
  Buf = Buffer;
  End = Buffer+ Length;
  
  // Check for the initial tag indicating a ase file.
  {
    Result = SeparateLine(Buf, Tag, Param);
    
    if (Result == 0)
      return 0;
    
    if (Tag.compareNoCase("*3DSMAX_ASCIIEXPORT") != 0)
    {
      SysMan->logOutput("MASETranslator: *3DSMAX_ASCIIEXPORT not found");
      return 0;
    }
  }
  
  
  while (Buf < End)
  {
    Result = SeparateLine(Buf, Tag, Param);
    
    if (Result == 0)
    {
      // if it has failed, check to see if it is a closing brace
      if (Tag.compareNoCase("}") == 0)
      {
        AdvanceLine(Buf);
        continue;
      }
      
      return false;
    }
    
    if (Tag.compareNoCase( ASETAG_COMMENT ) == 0)
    {
      SysMan->logOutput("MASETranslator: Comment '%s'", (LPCTSTR)Param);
    }
    else if (Tag.compareNoCase( ASETAG_SCENE ) == 0)
    {  
      if (!ParseSceneTag(Buf, End, Scene))
        return false;
    }
    else if (Tag.compareNoCase( ASETAG_GEOMOBJECT ) == 0)
    {  
      if (!ParseGeomObjectTag(Buf, End, Scene))
        return false;
    }
    else if (Tag.compareNoCase( ASETAG_MATERIAL_LIST ) == 0)
    {  
      if (!ParseMaterialListTag(Buf, End, Scene))
        return false;
    }
    else if (Param.compareNoCase("{"))
    {
      // Unknown tag. If we have encountered a set of bracers
      if (!ParseSkipBlock(Buf, End))
      {
        SysMan->logOutput("ASE Import: Error - Matching closing brace not found.");
      }
    }
    
  }
  
  return true;
}

bool MASETranslator::ParseSkipBlock(char *&Buffer, char *End) {
  int   CurLevel = 1;    // Start at 1 level, since we assume a { has already been encountered.
  
  while (Buffer < End)
  {  
    if (*Buffer == '{')
      CurLevel++;
    
    if (*Buffer == '}')
    {
      CurLevel--;
      if (CurLevel == 0)
        return true;
    }
    
    Buffer ++;
  }
  
  return false;
  
}

bool MASETranslator::ParseSceneTag(char *&Buffer, char *End, MScenePtr Scene)
{
  MStr              Tag, Param;
  int               Result;
  MSystemManagerPtr SysMan;
  
  SysMan = Aztec::getSystemManager();
  
  AdvanceLine(Buffer);
  
  // keep going until we hit a closing brace or pass the end of the string
  while (Buffer < End)
  {
    Result = SeparateLine(Buffer, Tag, Param);
    
    if (Result == 0)
    {
      // if it has failed, check to see if it is a closing brace
      if (Tag.compareNoCase("}") == 0)
      {
        AdvanceLine(Buffer);
        return true;
      }
      return false;
    }
    
    if (Tag.compareNoCase("*SCENE_FIRSTFRAME") == 0)
    {
      int      FirstFrame;
      
      FirstFrame = Param.ToInt();
      
      SysMan->logOutput("ASE Import: First Frame %i", FirstFrame);
    }
    if (Tag.compareNoCase("*SCENE_LASTFRAME") == 0)
    {
      int      FirstFrame;
      
      FirstFrame = Param.ToInt();
      
      SysMan->logOutput("ASE Import: Last Frame %i", FirstFrame);
    }
  }
  
  if (Buffer >= End)
    return false;
  
  return true;
}

bool MASETranslator::ParseGeomObjectTag(char *&Buffer, char *End, MScenePtr Scene) {
  MStr Tag, Param;
  int Result;
  MSceneObjectPtr sceneObj;
  MEditableMeshPtr Mesh;
  MMeshShapePtr meshShape;

  sceneObj = new MSceneObject;
  Mesh = new MEditableMesh;
  meshShape = new MMeshShape(Mesh);

  sceneObj->setShapeObject(meshShape);

  Scene->addObject(sceneObj);
  Scene->addObject(Mesh);
  Scene->addObject(meshShape);
  
  // keep going until we hit a closing brace or pass the end of the string
  while (Buffer < End) {
    Result = SeparateLine(Buffer, Tag, Param);
    
    if (Result == 0) {
      // if it has failed, check to see if it is a closing brace
      if (Tag.compareNoCase("}") == 0) {
        AdvanceLine(Buffer);
        Scene->addObject(Mesh);
        return true;
      }
      return false;
    }
    
    if (Tag.compareNoCase("*NODE_NAME") == 0) {
      // replace quotes and tabs with spaces
      Param.Replace('\"', ' ');
      Param.Replace('\t', ' ');
      Param.Replace('\n', ' ');
      Param.Replace('\x0d', ' ');
      sceneObj->setName(Param);
      Mesh->setName(Param + "Mesh");
      meshShape->setName(Param + "Shape");

    } else if (Tag.compareNoCase("*NODE_TM") == 0) {
      ParseNodeTMTag(Buffer, End, Scene, sceneObj);
    } else if (Tag.compareNoCase("*MESH") == 0) {
      ParseMeshTag(Buffer, End, Scene, Mesh);
    }
  }
  
  if (Buffer >= End)
    return false;
  
  return true;
}

bool MASETranslator::ParseMaterialListTag(char *&Buffer, char *End, MScenePtr Scene) {
  MStr           Tag, Param;
  int            Result;
  
  MStr           CurMatStr;
  int            MatCount = -1, CurMatInt = -1;
  
  // keep going until we hit a closing brace or pass the end of the string
  while (Buffer < End)
  {
    Result = SeparateLine(Buffer, Tag, Param);
    
    if (Result == 0)
    {
      // if it has failed, check to see if it is a closing brace
      if (Tag.compareNoCase("}") == 0)
      {
        AdvanceLine(Buffer);
        return true;
      }
      return false;
    }
    
    CurMatStr.Format("*MATERIAL %i", CurMatInt);
    
    if (Tag.compareNoCase("*MATERIAL_COUNT") == 0)
    {
      // replace quotes and tabs with spaces
      MatCount = Param.ToInt();
    }
    else if (Tag.compareNoCase(CurMatStr) == 0)
    {
      // Parse the actual material info.
    }
    
    if (MatCount != -1)
      CurMatInt++;
  }
  
  if (Buffer >= End)
    return false;
  
  return true;
}


bool MASETranslator::ParseNodeTMTag(char *&Buffer, char *End, MScenePtr Scene, MSceneObjectPtr Obj) {
  MStr     Tag, Param;
  int      Result;
  
  // keep going until we hit a closing brace or pass the end of the string
  while (Buffer < End)
  {
    Result = SeparateLine(Buffer, Tag, Param);
    
    if (Result == 0)
    {
      // Check to see if we are parsing a }, if we are, go past it, since this signifies the end
      // of the NodeTMTag
      if (Tag.compareNoCase("}") == 0)
      {
        AdvanceLine(Buffer);
        return true;
      }
      return false;
    }
    
    if (Tag.compareNoCase("*NODE_NAME") == 0)
    {
      // replace quotes and tabs with spaces
      Param.Replace('\"', ' ');
      Param.Replace('\t', ' ');
      Param.Replace('\n', ' ');
      Param.Replace('\x0d', ' ');
      Obj->setName(Param);

    } else if (Tag.compareNoCase("*TM_POS") == 0) {
      MVector3    Pos;
      
      Pos.convertFromString(Param);
      
      Obj->getTransformObject()->setTranslateVector(Pos);
      Obj->getTransformObject()->updateKey(Scene->getTime());

    } else if (Tag.compareNoCase("*TM_ROTAXIS") == 0) {
      MVector3    Rot;
      
      Rot.convertFromString(Param);
      
      Obj->getTransformObject()->setRotateVector(Rot);
      Obj->getTransformObject()->updateKey(Scene->getTime());

    } else if (Tag.compareNoCase("*TM_SCALE") == 0) {
      MVector3    Scale;
      
      Scale.convertFromString(Param);
      
      Obj->getTransformObject()->setScaleVector(Scale);
      Obj->getTransformObject()->updateKey(Scene->getTime());
    }
  }
  
  if (Buffer >= End)
    return false;
  
  return true;
}

bool MASETranslator::ParseMeshTag(char *&Buffer, char *End, MScenePtr Scene, MEditableMeshPtr Mesh) {
  MStr              Tag, Param;
  int               Result;
  MSystemManagerPtr SysMan;
  
  int         NumXYZ, getNumTris, NumST, NumSTTris;
  MVector3 *Verts, *TextureVerts;
  MTriangle   *Tris, *TextureTris;
  
  SysMan = Aztec::getSystemManager();
  
  NumXYZ = -1;
  getNumTris = -1;
  NumST = -1;
  
  Verts = NULL;
  TextureVerts = NULL;
  TextureTris = NULL;
  Tris = NULL;
  
  AdvanceLine(Buffer);
  
  // keep going until we hit a closing brace or pass the end of the string
  while (Buffer < End)
  {
    Result = SeparateLine(Buffer, Tag, Param);
    
    if (Result == 0)
    {
      // if it has failed, check to see if it is a closing brace
      if (Tag.compareNoCase("}") == 0)
      {
        AdvanceLine(Buffer);
        if (Verts && Tris)
        {
          Mesh->addVertexArray(Verts, NumXYZ);
          Mesh->addTriangleArray(Tris, getNumTris);
          Mesh->calculateNormals();
          
          if (TextureVerts) {
            MEditableMeshPtr TextureMesh;
            TextureMesh = new MEditableMesh;
            TextureMesh->addVertexArray(TextureVerts, NumST);
            
            if (TextureTris)
              TextureMesh->addTriangleArray(TextureTris, NumSTTris);
            else
              TextureMesh->addTriangleArray(Tris, getNumTris);
            
            Mesh->setTextureMesh(TextureMesh);
            
            delete[] TextureVerts;
            
            if (TextureTris)
              delete[] TextureTris;
          }
          
          delete[] Verts;
          delete[] Tris;
        }
        return true;
      }
      return false;
    }
    
    if (Tag.compareNoCase("*TIME_VALUE") == 0) {
      Scene->setTime( Param.ToInt() );
    } else if (Tag.compareNoCase("*MESH_NUMVERTEX") == 0) {
      NumXYZ = Param.ToInt();
    } else if (Tag.compareNoCase("*MESH_NUMFACES") == 0) {
      getNumTris = Param.ToInt();
    } else if (Tag.compareNoCase("*MESH_NUMTVERTEX") == 0) {
      NumST = Param.ToInt();
    } else if (Tag.compareNoCase("*MESH_NUMTVFACES") == 0) {
      NumSTTris = Param.ToInt();
    } else if (Tag.compareNoCase("*MESH_VERTEX_LIST") == 0) {
      if (NumXYZ != -1) {
        Verts = new MVector3[NumXYZ];
        ParseVertexListTag(Buffer, End, Scene, Verts);
      } else {
        SysMan->logOutput("ASE Import: Error - Vertex list found without number of vertices.");
      }
    } else if (Tag.compareNoCase("*MESH_FACE_LIST") == 0) {
      if (getNumTris != -1) {
        Tris = new MTriangle[getNumTris];
        ParseTriangleListTag(Buffer, End, Scene, Tris);
      } else {
        SysMan->logOutput("ASE Import: Error - Face list found without number of Faces.");
      }
    } else if (Tag.compareNoCase("*MESH_TVERTLIST") == 0) {
      if (NumST != -1) {
        TextureVerts = new MVector3[NumST];
        ParseTextureVertexListTag(Buffer, End, Scene, TextureVerts);
      } else {
        SysMan->logOutput("ASE Import: Error - Texture Vertex list found without number of vertices.");
      }
    } else if (Tag.compareNoCase("*MESH_TFACELIST") == 0) {
      if (NumSTTris != -1) {
        TextureTris = new MTriangle[NumSTTris];
        ParseTextureTriangleListTag(Buffer, End, Scene, TextureTris);
      } else {
        SysMan->logOutput("ASE Import: Error - Texture Face list found without number of Faces.");
      }
    }
  }
  
  if (Buffer >= End)
    return false;
  
  return true;
}

bool MASETranslator::ParseVertexListTag(char *&Buffer, char *End, MScenePtr Scene, MVector3* Verts) {
  MStr     Tag, Param;
  int      Result;
  
  // keep going until we hit a closing brace or pass the end of the string
  while (Buffer < End)
  {
    Result = SeparateLine(Buffer, Tag, Param);
    
    if (Result == 0)
    {
      // if it has failed, check to see if it is a closing brace
      if (Tag.compareNoCase("}") == 0)
      {
        AdvanceLine(Buffer);
        return true;
      }
      return false;
    }
    
    if (Tag.compareNoCase("*MESH_VERTEX") == 0)
    {
      int      VertNum;
      MVector3 *Vert;
      MStr     VertParams[4];
      
      if (Param.Separate(VertParams, 4) == 4)
      {
        VertNum = VertParams[0].ToInt();
        Vert = &Verts[VertNum];
        
        Vert->set(VertParams[1].ToFloat(), VertParams[2].ToFloat(), VertParams[3].ToFloat());
      }
    }
  }
  
  if (Buffer >= End)
    return false;
  
  return true;
}

bool MASETranslator::ParseTriangleListTag(char *&Buffer, char *End, MScenePtr Scene, MTriangle *Tris) {
  MStr     Tag, Param;
  int      Result;
  
  // keep going until we hit a closing brace or pass the end of the string
  while (Buffer < End)
  {
    Result = SeparateLine(Buffer, Tag, Param);
    
    if (Result == 0)
    {
      // if it has failed, check to see if it is a closing brace
      if (Tag.compareNoCase("}") == 0)
      {
        AdvanceLine(Buffer);
        return true;
      }
      return false;
    }
    
    if (Tag.compareNoCase("*MESH_FACE") == 0)
    {
      int         TriNum, NumParams, CurParam;
      MTriangle   *Tri;
      MStr        TriParams[25], *CurStr;
      
      if (NumParams = Param.Separate(TriParams, 25))
      {
        TriNum = TriParams[0].ToInt();
        Tri = &Tris[TriNum];
        
        // Go through the parameters, and sort through what we want
        CurParam = 1;
        CurStr = TriParams + 1;
        while (CurParam < NumParams)
        {
          int triIndex = -1;
          if (CurStr->compareNoCase("A:") == 0) {
            triIndex = 0;
          }
          if (CurStr->compareNoCase("B:") == 0) {
            triIndex = 1;
          }
          if (CurStr->compareNoCase("C:") == 0) {
            triIndex = 2;
          }

          if (triIndex != -1) {
            CurStr++;
            CurParam++;
            Tri->setVertex(triIndex, CurStr->ToInt());
            
            CurStr++;
            CurParam++;
            continue;
          }

          CurStr++;
          CurParam++;
        }
      }
    }
  }
  
  if (Buffer >= End)
    return false;
  
  return true;
}

bool MASETranslator::ParseTextureVertexListTag(char *&Buffer, char *End, MScenePtr Scene, MVector3* Verts) {
  MStr     Tag, Param;
  int      Result;
  
  // keep going until we hit a closing brace or pass the end of the string
  while (Buffer < End)
  {
    Result = SeparateLine(Buffer, Tag, Param);
    
    if (Result == 0)
    {
      // if it has failed, check to see if it is a closing brace
      if (Tag.compareNoCase("}") == 0)
      {
        AdvanceLine(Buffer);
        return true;
      }
      return false;
    }
    
    if (Tag.compareNoCase("*MESH_TVERT") == 0)
    {
      int      VertNum;
      MVector3 *Vert;
      MStr     VertParams[4];
      
      if (Param.Separate(VertParams, 4) == 4)
      {
        VertNum = VertParams[0].ToInt();
        Vert = &Verts[VertNum];
        
        Vert->set(VertParams[1].ToFloat(), VertParams[2].ToFloat(), VertParams[3].ToFloat());
      }
    }
  }
  
  if (Buffer >= End)
    return false;
  
  return true;
}

bool MASETranslator::ParseTextureTriangleListTag(char *&Buffer, char *End, 
                                                 MScenePtr Scene, MTriangle *Tris) {
  MStr     Tag, Param;
  int      Result;
  
  // keep going until we hit a closing brace or pass the end of the string
  while (Buffer < End) {
    Result = SeparateLine(Buffer, Tag, Param);
    
    if (Result == 0) {
      // if it has failed, check to see if it is a closing brace
      if (Tag.compareNoCase("}") == 0) {
        AdvanceLine(Buffer);
        return true;
      }
      return false;
    }
    
    if (Tag.compareNoCase("*MESH_TFACE") == 0) {
      int         TriNum, NumParams;
      MTriangle   *Tri;
      MStr        TriParams[25];
      
      if (NumParams = Param.Separate(TriParams, 25)) {
        TriNum = TriParams[0].ToInt();
        Tri = &Tris[TriNum];
        
        Tri->setVertex(0, TriParams[1].ToInt());
        Tri->setVertex(1, TriParams[2].ToInt());
        Tri->setVertex(2, TriParams[3].ToInt());
      }
    }
  }
  
  if (Buffer >= End)
    return false;
  
  return true;
}


bool MASETranslator::importFile(MStr Filename, MScenePtr Scene) {
  FILE              *hFile;
  MSystemManagerPtr SysMan;
  char              *Buffer;
  int               BufferLen, Result;
  long				hFileSize;
  
  SysMan = Aztec::getSystemManager();
  assert(SysMan != NULL);
  
  Filename.Replace('/', '\\');
  
  hFile = fopen(Filename,"rt");
  
  if (!hFile )
    return false;
  
  //Get file size
  fseek(hFile,0,SEEK_END);
  hFileSize=ftell(hFile);
  fseek(hFile,0,SEEK_SET);

  Buffer = new char[hFileSize+1];
  
  BufferLen = fread(Buffer,1, hFileSize, hFile);
  
  fclose(hFile);
  
  Buffer[BufferLen] = '\x0';
  
  Result = ParseFile(Buffer, BufferLen, Scene);
  
  delete[] Buffer;
  
  return (Result != 0);
}

int MASETranslator::ASEWrite(FILE *hFile, const MStr &Str) {
  return ASEWrite(hFile, (LPCTSTR)Str);
}

int MASETranslator::ASEWrite(FILE *hFile, const char *Str) {
  char  *S;
  int   Result;
  
  S = (char*)Str;
  
  while (*S) {
    if (*S == '}') {
      m_TabCount--;
    }
    S++;
  }
  
  
  for (int n=0; n<m_TabCount; n++) {
    fprintf(hFile,"\t");
  }
  
  Result = fprintf(hFile, Str);
  
  S = (char*)Str;
  
  while (*S) {
    if (*S == '{') {
      m_TabCount++;
    }
    S++;
  }
  
  return Result;
}

bool MASETranslator::exportFile(MStr Filename, MScenePtr Scene) {
  // Open an ascii file, and start writing to it
  FILE      *hFile;

  hFile = fopen(Filename,"wt");
  
  if (!hFile)
    return false;
  
  m_TabCount = 0;
  
  // write out the header for the file
  ASEWrite(hFile, "*3DSMAX_ASCIIEXPORT\t200\n");
  ASEWrite(hFile, "*COMMENT \"Aztec AsciiExport Version  0.1\"\n");
  
  // write out the scene information
  ASEWrite(hFile, "*SCENE {\n");
  ASEWrite(hFile, MStr().Format("*SCENE_FILENAME \"%s\"\n", (LPCTSTR)Filename));
  ASEWrite(hFile, MStr().Format("*SCENE_FIRSTFRAME %i\n", 0));
  ASEWrite(hFile, MStr().Format("*SCENE_LASTFRAME %i\n", 100));
  ASEWrite(hFile, MStr().Format("*SCENE_FRAMESPEED %i\n", 30));
  ASEWrite(hFile, MStr().Format("*SCENE_TICKSPERFRAME %i\n", 30));
  ASEWrite(hFile, MStr().Format("*SCENE_BACKGROUND_STATIC %f\t%f\t%f\n", 0.0, 0.0, 0.0));
  ASEWrite(hFile, MStr().Format("*SCENE_AMBIENT_STATIC %f\t%f\t%f\n", 0.0, 0.0, 0.0));
  
  ASEWrite(hFile, "}\n");
  
  
  // Write out the material list
  
  // TODO: output actual materials
  ASEWrite(hFile, "*MATERIAL_LIST {\n");
  ASEWrite(hFile, MStr().Format("*MATERIAL_COUNT %i\n", 0));
  ASEWrite(hFile, "}\n");
  
  
  // go through every object in the scene, and write out the objects
  // todo: do not just write out meshes, write out actual object data
  MBaseObjectPtr Obj;
  MSceneObjectPtr SceneObj;
  MMeshPtr MeshObj;
  MTransformObjectPtr XFormObj;
  
  Scene->setTime(0);
  Scene->getObjectList()->beginIteration();
  
  // set the time back to the start of the animation
  while ((Obj = Scene->getObjectList()->getNext()) != NULL) {
    SceneObj = AZTEC_CAST(MSceneObject, Obj);
    if (SceneObj == NULL) {
      continue;
    }

    MShapeObjectPtr shapeObj = SceneObj->getShapeObject();

    if (shapeObj == NULL) {
      continue;
    }

    MeshObj = shapeObj->convertToMesh();
    if (MeshObj == NULL) {
      continue;
    }
    
    {
      ASEWrite(hFile, "*GEOMOBJECT {\n");
      
      ASEWrite(hFile, MStr().Format("*NODE_NAME \"%s\"\n", (LPCTSTR)SceneObj->getName()));
      
      XFormObj = SceneObj->getTransformObject();
      if (XFormObj != NULL) {
        MMatrix4    Matrix;
        MVector3    TranslateVec, ScaleVec;
        
        TranslateVec = XFormObj->getTranslateVector(Scene->getTime());
        ScaleVec = XFormObj->getScaleVector(Scene->getTime());
        
        XFormObj->getTransformMatrix(Matrix);
        
        ASEWrite(hFile, "*NODE_TM {\n");
        
        ASEWrite(hFile, MStr().Format("*NODE_NAME \"%s\"\n", (LPCTSTR)SceneObj->getName()));
        ASEWrite(hFile, MStr().Format("*INHERIT_POS %i %i %i\n", 1, 1, 1));
        ASEWrite(hFile, MStr().Format("*INHERIT_ROT %i %i %i\n", 1, 1, 1));
        ASEWrite(hFile, MStr().Format("*INHERIT_SCL %i %i %i\n", 1, 1, 1));
        
        ASEWrite(hFile, MStr().Format("*TM_ROW0 %f %f %f\n", Matrix.m[0][0], Matrix.m[0][1], Matrix.m[0][2])) ;
        ASEWrite(hFile, MStr().Format("*TM_ROW1 %f %f %f\n", Matrix.m[1][0], Matrix.m[1][1], Matrix.m[1][2])) ;
        ASEWrite(hFile, MStr().Format("*TM_ROW2 %f %f %f\n", Matrix.m[2][0], Matrix.m[2][1], Matrix.m[2][2])) ;
        ASEWrite(hFile, MStr().Format("*TM_ROW3 %f %f %f\n", Matrix.m[3][0], Matrix.m[3][1], Matrix.m[3][2])) ;
        
        ASEWrite(hFile, MStr().Format("*TM_POS %f %f %f\n", TranslateVec.x, TranslateVec.y, TranslateVec.z));
        ASEWrite(hFile, MStr().Format("*TM_SCALE %f %f %f\n", ScaleVec.x, ScaleVec.y, ScaleVec.z));
        ASEWrite(hFile, MStr().Format("*TM_SCALEAXIS 0.0 0.0 0.0\n"));
        
        ASEWrite(hFile, "}\n");
      }
      
      if (MeshObj != NULL) {
        Scene->setTime(0);
        ASEWrite(hFile, "*MESH {\n");
        
        ASEWrite(hFile, MStr().Format("*TIMEVALUE %i\n", 0));
        ASEWrite(hFile, MStr().Format("*MESH_NUMVERTEX %i\n", MeshObj->getNumVerts()));
        ASEWrite(hFile, MStr().Format("*MESH_NUMFACES %i\n", MeshObj->getNumTris()));
        
        // Write out the vertex list
        {
          int      n;
          ASEWrite(hFile, "*MESH_VERTEX_LIST {\n");
          for (n=0; n<MeshObj->getNumVerts(); n++)
          {
            MVector3 Vert = MeshObj->getVertexPosition(n);
            
            ASEWrite(hFile, MStr().Format("*MESH_VERTEX  %i\t%.4f\t%.4f\t%.4f\n", n, Vert.x, Vert.y, Vert.z));
          }
          
          ASEWrite(hFile, "}\n");
        }
        
        // Write out the face list
        {
          ASEWrite(hFile, "*MESH_FACE_LIST {\n");
          for (int n = 0; n < MeshObj->getNumTris(); n++) {
            ASEWrite(hFile, MStr().Format("*MESH_FACE  %i: A: %i B: %i C: %i AB: 1 BC: 1 CA: 1 \n", 
              n, 
              MeshObj->getTriangleVertex(n, 0), MeshObj->getTriangleVertex(n, 1), MeshObj->getTriangleVertex(n, 2)));
          }
          
          ASEWrite(hFile, "}\n");
        }
        
        // Write out the texture mesh list if necessary.
        {
          MMeshPtr TextureMesh;
          TextureMesh = MeshObj->getTextureMesh();
          
          if (TextureMesh != NULL) {
            ASEWrite(hFile, MStr().Format("*NUMTVERTEX %i\n", TextureMesh->getNumVerts()));
            
            // Write out the texture vertex list
            {
              int      n;
              ASEWrite(hFile, "*MESH_TVERTLIST {\n");
              for (n=0; n<MeshObj->getNumVerts(); n++)
              {
                MVector3 Vert = MeshObj->getVertexPosition(n);

                ASEWrite(hFile, MStr().Format("*MESH_TVERT  %i\t%.4f\t%.4f\t%.4f\n", n, Vert.x, Vert.y, Vert.z));
              }
              
              ASEWrite(hFile, "}\n"); // End *MESH_TVERTLIST
            }
            
            // Write out the texture face list
            {
              int      n;
              ASEWrite(hFile, "*MESH_TFACELIST{\n");
              for (n=0; n<MeshObj->getNumTris(); n++) {
                ASEWrite(hFile, MStr().Format("*MESH_TFACE  %i\t%i\t%i\t%i\n", n, 
                  MeshObj->getTriangleVertex(n, 0), MeshObj->getTriangleVertex(n, 1), MeshObj->getTriangleVertex(n, 2)));
              }
              ASEWrite(hFile, "}\n"); // End *MESH_TFACELIST
            }
            
          }
          
        }
        
        ASEWrite(hFile, "}\n"); // End *MESH
      }
      ASEWrite(hFile, "*PROP_CASTSHADOW 1\n");
      ASEWrite(hFile, "*PROP_RECVSHADOW 1\n");
      
      ASEWrite(hFile, "}\n"); // End *GEOMOBJECT
      }
      
      MeshObj = NULL;
   }
   Scene->getObjectList()->endIteration();
   
   
   fclose(hFile);
   
   return false;
}
