#include "Stdafx.h"

#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <stdio.h>
#include <math.h>
#include <assert.h>

#include "RangTree.h"

#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

#ifndef WIN32
#define OutputDebugString(x) printf(x)
#endif 

static char sbuf[512];

#if !defined( EPSILON )
#define EPSILON 1.0e-6
#endif

#if 1
#define USE_MESSAGE_FILE 0

#if USE_MESSAGE_FILE
static FILE *message_file = NULL;
#endif

#if _DEBUG
#define DEBUG_TREES 1
#define DEBUG_LEAVES 0
#else
#define DEBUG_TREES 0
#define DEBUG_LEAVES 0
#endif

static void
message(char *fmt, ...)
{
   va_list ap;
   va_start(ap, fmt);
#if USE_MESSAGE_FILE
	vfprintf(message_file, fmt, ap);
#else
	vsprintf(sbuf, fmt, ap);
	OutputDebugString(sbuf);
#endif
   va_end(ap);
}

Range_Tree::node23leaf::node23leaf(RangeKey &key, void *data)
{
	this->key = key;
	this->data = new range_data(data);
}

void Range_Tree::
Show()
{
#if USE_MESSAGE_FILE
	message_file = fopen("particle.out", "wt");
#endif
	message("Range tree:\n");
	show_two_three(Root, 0);
	message("\n");
#if USE_MESSAGE_FILE
	fclose(message_file);
#endif
}

static float
clampzero(float x)
{
	if (fabs(x) < 1.0e-4)
		return 0.0;
	else
		return x;
}

static void
printrange(Range3D range)
{
	message("[%g,%g,%g],[%g,%g,%g]",
			clampzero(range[0][0]), clampzero(range[0][1]),
			clampzero(range[0][2]), clampzero(range[1][0]),
			clampzero(range[1][1]), clampzero(range[1][2]));
}

void Range_Tree::
show_two_three(nodeptr tree, int depth)
{
	int i;
	
	for (i=0;i<depth;i++)
		message(" ");
	if (tree == NULL)
		message("NULL\n");
	else if (tree->kind == node23::LEAF_NODE) {
		message("l(<%.10g,%.10g,%.10g> ",
			tree->leaf->key[0], tree->leaf->key[1], tree->leaf->key[2]);
		printrange(tree->range);
		if (tree->leaf->data->next != NULL) {
			range_data *data = tree->leaf->data;
			int cnt;
			for (cnt=0;data!=NULL;data=data->next)
				cnt++;
			message(") - %d\n", cnt);
		} else
			message(")\n");
	} else if (tree->kind == node23::TWO_NODE) {
		message("n2(<%g,%g,%g> ",
			clampzero(tree->key[0][0]), clampzero(tree->key[0][1]), clampzero(tree->key[0][2]));
		printrange(tree->range);
	    message(")\n");
		show_two_three(tree->ptrs[0], depth+1);
		show_two_three(tree->ptrs[1], depth+1);
		}
	else {
		message("n3(<%g,%g,%g> ",
			clampzero(tree->key[0][0]), clampzero(tree->key[0][1]), clampzero(tree->key[0][2]),
			clampzero(tree->key[1][0]), clampzero(tree->key[1][1]), clampzero(tree->key[1][2]));
		printrange(tree->range);
		message(")\n");
		show_two_three(tree->ptrs[0], depth+1);
		show_two_three(tree->ptrs[1], depth+1);
		show_two_three(tree->ptrs[2], depth+1);
		}
}

#endif

Range_Tree::node23::node23(RangeKey &key, Range3D &range, void *data)
{
	kind = LEAF_NODE;
	this->leaf   = new node23leaf(key, data);
	this->link   = this->leaf;
	this->range  = range;
}

// Comparisons of boxes are made based on their lower left corner
int Range_Tree::
compare(RangeKey &a, RangeKey &b)
{
	float m0, m1;
	m0 = a[0];
	m1 = b[0];
	if (m0 < m1)
		return -1;
	else if (m0 > m1)
		return 1;
	m0 = a[1];
	m1 = b[1];
	if (m0 < m1)
		return -1;
	else if (m0 > m1)
		return 1;
	m0 = a[2];
	m1 = b[2];
	if (m0 < m1)
		return -1;
	else if (m0 > m1)
		return 1;
	else
		return 0;
}

RangeKey &Range_Tree::
least_key(node23 *branch)
{
	assert(branch != NULL);
#if 0
	if (branch->kind ==  node23::LEAF_NODE)
		return branch->key[0];
	else
		return least_key(branch->ptrs[0]);
#else
	return branch->link->key;
#endif
}

// This function pushes a new entry down the tree.  If the result of pushing
// it down splits a node into two nodes, then the pointers to the new nodes
// plus their range is returned (along with a return result of 1).
int Range_Tree::
split(nodeptr intree, RangeKey &key, Range3D &range, void *data,
	  nodeptr *left_tree, nodeptr *right_tree, nodeptr *leaf)
{
	int result;
	nodeptr temp1, new_tree1, new_tree2;
	range_data *trangedata;

	if (intree == NULL)
		result = 0;
	else if (intree->kind == node23::LEAF_NODE) {
		result = compare(key, intree->leaf->key);
#if 0
		if (result != 0 &&
			fabs(key[0] - intree->leaf->key[0]) < 1.0e-6 &&
			fabs(key[1] - intree->leaf->key[1]) < 1.0e-6 &&
			fabs(key[2] - intree->leaf->key[2]) < 1.0e-6) {
			sprintf(sbuf, "Compare Error: (%g,%g,%g)\n",
				key[0] - intree->leaf->key[0],
				key[1] - intree->leaf->key[1],
				key[2] - intree->leaf->key[2]);
			OutputDebugString(sbuf);
		}
#endif
		if (result < 0) {
			// New box is below the existing one
			*left_tree  = new node23(key, range, data);
			*right_tree = intree;
			*leaf = *left_tree;
			result = 1;
		} else if (result == 0) {
			// Complete overlap - push it on
			intree->range += range;
			trangedata = new range_data(data);
			trangedata->next = intree->leaf->data;
			intree->leaf->data = trangedata;
			*leaf = intree;
			result = 0;
		} else {
			// New box is above the existing one
			*left_tree  = intree;
			*right_tree = new node23(key, range, data);
			*leaf = *right_tree;
			result = 1;
		}
	} else if (intree->kind == node23::TWO_NODE)
		// Don't do anything with a 2-tree
		result = 0;

	// If we get here, then we are comparing the new range/data pair against
	// a THREE_NODE. 
	else if (compare(key, intree->key[0]) < 0) {
		// New box is below the key for the upper branch
		if (split(intree->ptrs[0], key, range, data, &new_tree1, &new_tree2, leaf)) {
			// Right tree will be the two rightmost nodes in the existing tree
			*right_tree = new node23(node23::TWO_NODE);
			(*right_tree)->ptrs[0] = intree->ptrs[1];
			(*right_tree)->ptrs[1] = intree->ptrs[2];
			(*right_tree)->link    = intree->ptrs[1]->link;
			(*right_tree)->key[0]  = least_key(intree->ptrs[2]);
			(*right_tree)->range = intree->ptrs[1]->range;
			(*right_tree)->range += intree->ptrs[2]->range;

			// Left tree will be the new node + the leftmost node in the existing tree
			*left_tree = intree; // We are reusing the node allocated to "intree"
			intree->kind = node23::TWO_NODE;
			intree->ptrs[0] = new_tree1;
			intree->ptrs[1] = new_tree2;
			intree->link    = new_tree1->link;
			intree->key[0] = least_key(new_tree2);
			intree->range = new_tree1->range;
			intree->range += new_tree2->range;

			result = 1;
			}
		else {
			intree->range += intree->ptrs[0]->range;
			result = 0;
		}
	} else if (compare(key, intree->key[1]) < 0) {
		// New box is between the two bounding boxes for this node
		if (split(intree->ptrs[1], key, range, data, &new_tree1, &new_tree2, leaf)) {
			temp1 = intree->ptrs[0];

			*right_tree = new node23(node23::TWO_NODE);
			(*right_tree)->ptrs[0] = new_tree2;
			(*right_tree)->ptrs[1] = intree->ptrs[2];
			(*right_tree)->link    = new_tree2->link;
			(*right_tree)->key[0] = least_key(intree->ptrs[2]);
			(*right_tree)->range = new_tree2->range;
			(*right_tree)->range += intree->ptrs[2]->range;

			*left_tree = intree;
			intree->kind = node23::TWO_NODE;
			intree->ptrs[0] = temp1;
			intree->ptrs[1] = new_tree1;
			intree->link    = temp1->link;
			intree->key[0] = least_key(new_tree1);
			intree->range = temp1->range;
			intree->range += new_tree1->range;

			result = 1;
			}
		else {
			intree->range += intree->ptrs[1]->range;
			result = 0;
		}
	} else {
		if (split(intree->ptrs[2], key, range, data, &new_tree1, &new_tree2, leaf)) {
			*left_tree = new node23(node23::TWO_NODE);
			(*left_tree)->ptrs[0] = intree->ptrs[0];
			(*left_tree)->ptrs[1] = intree->ptrs[1];
			(*left_tree)->link    = intree->ptrs[0]->link;
			(*left_tree)->key[0] = least_key(intree->ptrs[1]);
			(*left_tree)->range = intree->ptrs[0]->range;
			(*left_tree)->range += intree->ptrs[1]->range;

			*right_tree = intree;
			intree->kind = node23::TWO_NODE;
			intree->ptrs[0] = new_tree1;
			intree->ptrs[1] = new_tree2;
			intree->link    = new_tree1->link;
			intree->key[0] = least_key(new_tree2);
			(*right_tree)->range = new_tree1->range;
			(*right_tree)->range += new_tree2->range;
			result = 1;
		} else {
			intree->range += intree->ptrs[2]->range;
			result = 0;
		}
	}

#if DEBUG_TREES
	if (result) {
		switch ((*left_tree)->kind) {
		case node23::LEAF_NODE:
#if DEBUG_LEAVES
			Range3D box;
			for (trangedata=(*left_tree)->leaf->data;trangedata!=NULL;trangedata=trangedata->next) {
				((PolyTriPtr)trangedata->data)->compute_bbox(box);
				assert((*left_tree)->range.Contains(box, (float)EPSILON));
			}
#endif
			break;
		case node23::TWO_NODE:
			assert((*left_tree)->range.Contains((*left_tree)->ptrs[0]->range, (float)EPSILON));
			assert((*left_tree)->range.Contains((*left_tree)->ptrs[1]->range, (float)EPSILON));
			assert((*left_tree)->key[0] == (*left_tree)->ptrs[1]->link->key);
			break;
		case node23::THREE_NODE:
			assert((*left_tree)->range.Contains((*left_tree)->ptrs[0]->range, (float)EPSILON));
			assert((*left_tree)->range.Contains((*left_tree)->ptrs[1]->range, (float)EPSILON));
			assert((*left_tree)->range.Contains((*left_tree)->ptrs[2]->range, (float)EPSILON));
			assert((*left_tree)->key[0] == (*left_tree)->ptrs[1]->link->key);
			assert((*left_tree)->key[1] == (*left_tree)->ptrs[2]->link->key);
			break;
		}
		switch ((*right_tree)->kind) {
		case node23::LEAF_NODE:
#if DEBUG_LEAVES
			Range3D box;
			for (trangedata=(*right_tree)->leaf->data;trangedata!=NULL;trangedata=trangedata->next) {
				((PolyTriPtr)trangedata->data)->compute_bbox(box);
				assert((*right_tree)->range.Contains(box, (float)EPSILON));
			}
#endif
			break;
		case node23::TWO_NODE:
			assert((*right_tree)->range.Contains((*right_tree)->ptrs[0]->range, (float)EPSILON));
			assert((*right_tree)->range.Contains((*right_tree)->ptrs[1]->range, (float)EPSILON));
			assert((*right_tree)->key[0] == (*right_tree)->ptrs[1]->link->key);
			break;
		case node23::THREE_NODE:
			assert((*right_tree)->range.Contains((*right_tree)->ptrs[0]->range, (float)EPSILON));
			assert((*right_tree)->range.Contains((*right_tree)->ptrs[1]->range, (float)EPSILON));
			assert((*right_tree)->range.Contains((*right_tree)->ptrs[2]->range, (float)EPSILON));
			assert((*right_tree)->key[0] == (*right_tree)->ptrs[1]->link->key);
			assert((*right_tree)->key[1] == (*right_tree)->ptrs[2]->link->key);
			break;
		}
	} else {
		if (intree != NULL) {
			switch (intree->kind) {
			case node23::LEAF_NODE:
#if DEBUG_LEAVES
				Range3D box;
				for (trangedata=intree->leaf->data;trangedata!=NULL;trangedata=trangedata->next) {
					((PolyTriPtr)trangedata->data)->compute_bbox(box);
					assert(intree->range.Contains(box, (float)EPSILON));
				}
#endif
				break;
			case node23::TWO_NODE:
				assert(intree->range.Contains(intree->ptrs[0]->range, (float)EPSILON));
				assert(intree->range.Contains(intree->ptrs[1]->range, (float)EPSILON));
				assert(intree->key[0] == intree->ptrs[1]->link->key);
				break;
			case node23::THREE_NODE:
				assert(intree->range.Contains(intree->ptrs[0]->range, (float)EPSILON));
				assert(intree->range.Contains(intree->ptrs[1]->range, (float)EPSILON));
				assert(intree->range.Contains(intree->ptrs[2]->range, (float)EPSILON));
				assert(intree->key[0] == intree->ptrs[1]->link->key);
				assert(intree->key[1] == intree->ptrs[2]->link->key);
				break;
			}
		}
	}
	assert(!result || compare((*left_tree)->link->key, (*right_tree)->link->key) < 0);
#endif

	return result;
}

int Range_Tree::
insert3(nodeptr intree, RangeKey &key, Range3D &range, void *data,
		nodeptr *outtree, nodeptr *leaf)
{
	int result;
	nodeptr temp1;
	nodeptr new_tree1, new_tree2;

	if (intree == NULL) {
		*outtree = new node23(key, range, data);
		*leaf = *outtree;
		result = 1;
	} else if (intree->kind == node23::LEAF_NODE) {
		result = 0;
	} else if (intree->kind == node23::TWO_NODE) {
		if (compare(key, intree->key[0]) < 0) {
			if (insert3(intree->ptrs[0], key, range, data, &new_tree1, leaf)) {
				intree->ptrs[0] = new_tree1;
				intree->link = new_tree1->link;
				intree->range += new_tree1->range;
				*outtree = intree;
				result = 1;
			} else if (split(intree->ptrs[0], key, range, data, &new_tree1, &new_tree2, leaf)) {
				temp1 = intree->ptrs[1];
				intree->ptrs[0] = new_tree1;
				intree->ptrs[1] = new_tree2;
				intree->ptrs[2] = temp1;
				intree->key[1] = intree->key[0];
				intree->key[0] = least_key(new_tree2);
				intree->link = new_tree1->link;
				intree->range = new_tree1->range;
				intree->range += new_tree2->range;
				intree->range += temp1->range;
				intree->kind = node23::THREE_NODE;
				*outtree = intree;
				result = 1;
			} else {
				intree->link = intree->ptrs[0]->link;
				intree->range += intree->ptrs[0]->range;
				result = 0;
			}
		} else {
			if (insert3(intree->ptrs[1], key, range, data, &new_tree1, leaf)) {
				intree->ptrs[1] = new_tree1;
				intree->key[0] = least_key(new_tree1);
				intree->range = intree->ptrs[0]->range;
				intree->range += new_tree1->range;
				*outtree = intree;
				result = 1;
			} else if (split(intree->ptrs[1], key, range, data, &new_tree1, &new_tree2, leaf)) {
				intree->ptrs[1] = new_tree1;
				intree->ptrs[2] = new_tree2;
				intree->key[0] = least_key(new_tree1);
				intree->key[1] = least_key(new_tree2);
				intree->range = intree->ptrs[0]->range;
				intree->range += new_tree1->range;
				intree->range += new_tree2->range;
				intree->kind = node23::THREE_NODE;
				*outtree = intree;
				result = 1;
			} else {
				intree->range += intree->ptrs[1]->range;
				result = 0;
			}
		}
	// If we get here, we are inserting into a THREE_NODE
	} else if (compare(key, intree->key[0]) < 0)
		if (insert3(intree->ptrs[0], key, range, data, &new_tree1, leaf)) {
			intree->ptrs[0] = new_tree1;
			intree->link = new_tree1->link;
			intree->range = new_tree1->range;
			intree->range += intree->ptrs[1]->range;
			intree->range += intree->ptrs[2]->range;
			*outtree = intree;
			result = 1;
		} else {
			intree->link = intree->ptrs[0]->link;
			intree->range += intree->ptrs[0]->range;
			result = 0;
		}
	else if (compare(key, intree->key[1]) < 0)
		if (insert3(intree->ptrs[1], key, range, data, &new_tree1, leaf)) {
			intree->ptrs[1] = new_tree1;
			intree->key[0] = least_key(new_tree1);
			intree->range = intree->ptrs[0]->range;
			intree->range += intree->ptrs[2]->range;
			intree->range += new_tree1->range;
			*outtree = intree;
			result = 1;
		} else {
			intree->range += intree->ptrs[1]->range;
			result = 0;
		}
	else {
		if (insert3(intree->ptrs[2], key, range, data, &new_tree1, leaf)) {
			intree->ptrs[2] = new_tree1;
			intree->key[1] = least_key(new_tree1);
			intree->range = intree->ptrs[0]->range;
			intree->range += intree->ptrs[1]->range;
			intree->range += new_tree1->range;
			*outtree = intree;
			result = 1;
		} else {
			intree->range += intree->ptrs[2]->range;
			result = 0;
		}
	}

#if DEBUG_TREES
	if (result) {
		switch ((*outtree)->kind) {
		case node23::LEAF_NODE:
#if DEBUG_LEAVES
			{
			Range3D box;
			range_data *trangedata;
			for (trangedata=(*outtree)->leaf->data;trangedata!=NULL;trangedata=trangedata->next) {
				((PolyTriPtr)trangedata->data)->compute_bbox(box);
				assert((*outtree)->range.Contains(box, (float)EPSILON));
			}
			}
			assert((*outtree)->link == (*outtree)->leaf);
#endif
			break;
		case node23::TWO_NODE:
			assert((*outtree)->range.Contains((*outtree)->ptrs[0]->range, (float)EPSILON));
			assert((*outtree)->range.Contains((*outtree)->ptrs[1]->range, (float)EPSILON));
			assert(compare((*outtree)->key[0], least_key((*outtree)->ptrs[1])) == 0);
			assert((*outtree)->link == (*outtree)->ptrs[0]->link);
			assert((*outtree)->key[0] == (*outtree)->ptrs[1]->link->key);
			break;
		case node23::THREE_NODE:
			assert((*outtree)->range.Contains((*outtree)->ptrs[0]->range, (float)EPSILON));
			assert((*outtree)->range.Contains((*outtree)->ptrs[1]->range, (float)EPSILON));
			assert((*outtree)->range.Contains((*outtree)->ptrs[2]->range, (float)EPSILON));
			assert(compare((*outtree)->key[0], least_key((*outtree)->ptrs[1])) == 0);
			assert(compare((*outtree)->key[1], least_key((*outtree)->ptrs[2])) == 0);
			assert(compare((*outtree)->key[0], (*outtree)->key[1]) < 0);
			assert((*outtree)->link == (*outtree)->ptrs[0]->link);
			assert((*outtree)->key[0] == (*outtree)->ptrs[1]->link->key);
			assert((*outtree)->key[1] == (*outtree)->ptrs[2]->link->key);
			break;
		}
	} else {
		switch (intree->kind) {
		case node23::LEAF_NODE:
#if DEBUG_LEAVES
			{
			Range3D box;
			range_data *trangedata;
			for (trangedata=intree->leaf->data;trangedata!=NULL;trangedata=trangedata->next) {
				((PolyTriPtr)trangedata->data)->compute_bbox(box);
				assert(intree->range.Contains(box, (float)EPSILON));
			}
			}
			assert(intree->link == intree->leaf);
#endif
			break;
		case node23::TWO_NODE:
			assert(intree->range.Contains(intree->ptrs[0]->range, (float)EPSILON));
			assert(intree->range.Contains(intree->ptrs[1]->range, (float)EPSILON));
			assert(compare(intree->key[0], least_key(intree->ptrs[1])) == 0);
			assert(intree->link == intree->ptrs[0]->link);
			assert(intree->key[0] == intree->ptrs[1]->link->key);
			break;
		case node23::THREE_NODE:
			assert(intree->range.Contains(intree->ptrs[0]->range, (float)EPSILON));
			assert(intree->range.Contains(intree->ptrs[1]->range, (float)EPSILON));
			assert(intree->range.Contains(intree->ptrs[2]->range, (float)EPSILON));
			assert(compare(intree->key[0], least_key(intree->ptrs[1])) == 0);
			assert(compare(intree->key[1], least_key(intree->ptrs[2])) == 0);
			assert(compare(intree->key[0], intree->key[1]) < 0);
			assert(intree->link == intree->ptrs[0]->link);
			assert(intree->key[0] == intree->ptrs[1]->link->key);
			assert(intree->key[1] == intree->ptrs[2]->link->key);
			break;
		}
	}
#endif

	return result;
}

Range_Tree::rangedataptr Range_Tree::
two_three_least(nodeptr intree)
{
	if (intree == NULL)
		return NULL;
	else if (intree->kind == node23::LEAF_NODE)
		return intree->leaf->data;
	else if (intree->kind == node23::TWO_NODE)
		return two_three_least(intree->ptrs[0]);
	else
		return two_three_least(intree->ptrs[0]);
}

// Delete a range/data pair from a 2-3 tree.
Range_Tree::delete_result Range_Tree::
two_three_delete(nodeptr intree, RangeKey &key, Range3D &range,
				 void *value, nodeptr *outtree)
{
	nodeptr tree1, tree2, tree3, tree4;
	delete_result result;

	*outtree = intree;

	if (intree == NULL)
		return DeleteFailed;

	if (intree->kind == node23::LEAF_NODE) {
		if (compare(key, intree->leaf->key) == 0) {
			rangedataptr root, last;
			bool found = false;
			for (last=root=intree->leaf->data;root!=NULL && !found;) {
				if (root->data == value) {
					// Found the one we were looking for
					cleanup(root->data);
					if (last == root)
						intree->leaf->data = root->next;
					else
						last->next = root->next;
					delete root;
					if (intree->leaf->data == NULL) {
						delete intree->leaf;
						delete intree;
						*outtree = NULL;
						result = DeletedBranch;
						found = true;
					} else {
						*outtree = intree;
						result = DeleteSucceeded;
						found = true;
					}
				} else {
					last = root;
					root = root->next;
				}
			}
			if (!found)
				result = DeleteFailed;
		} else
			result = DeleteFailed;
	} else if (intree->kind == node23::TWO_NODE) {
		if (compare(key, intree->key[0]) < 0) {
			switch (two_three_delete(intree->ptrs[0], key, range, value, &tree1)) {
			case DeleteFailed:
				result = DeleteFailed;
				break;
			case DeleteSucceeded:
				intree->ptrs[0] = tree1;
				intree->link = tree1->link;
				intree->range = tree1->range;
				intree->range += intree->ptrs[1]->range;
				*outtree = intree;
				result = DeleteSucceeded;
				break;
			case DeletedBranch:
				if (tree1 == NULL) {
					// (1, 1) -> leaf
					*outtree = intree->ptrs[1];
					delete intree;
					result = DeletedBranch;
				} else if (intree->ptrs[1]->kind == node23::TWO_NODE) {
					tree2 = intree->ptrs[1];
					tree3 = tree2->ptrs[0];
					tree4 = tree2->ptrs[1];
					
					tree2->kind = node23::THREE_NODE;
					tree2->ptrs[0] = tree1;
					tree2->ptrs[1] = tree3;
					tree2->ptrs[2] = tree4;
					tree2->link = tree1->link;
					tree2->key[0] = least_key(tree3);
					tree2->key[1] = least_key(tree4);
					tree2->range = tree1->range;
					tree2->range += tree3->range;
					tree2->range += tree4->range;
					*outtree = tree2;
					delete intree;
					result = DeletedBranch;
				} else {
					// Split one from the right and add it to the left
					tree2 = intree->ptrs[1];
					tree3 = new node23(node23::TWO_NODE); // intree->ptrs[0]; - can't reuse a deleted node...
					tree3->ptrs[0] = tree1;
					tree3->ptrs[1] = tree2->ptrs[0];
					tree3->link = tree1->link;
					tree3->key[0] = least_key(tree2->ptrs[0]);
					tree3->range = tree1->range;
					tree3->range += tree3->ptrs[1]->range;
					intree->ptrs[0] = tree3;
					intree->link = tree3->link;

					tree1 = tree2->ptrs[1];
					tree3 = tree2->ptrs[2];
					tree2->kind = node23::TWO_NODE;
					tree2->ptrs[0] = tree1;
					tree2->ptrs[1] = tree3;
					tree2->link = tree1->link;
					tree2->key[0] = least_key(tree3);
					tree2->range = tree1->range;
					tree2->range += tree3->range;

					intree->key[0] = least_key(tree2);
					intree->range = intree->ptrs[0]->range;
					intree->range += tree2->range;
					*outtree = intree;
					result = DeleteSucceeded;
				}
				break;
			}
		} else {
			switch (two_three_delete(intree->ptrs[1], key, range, value, &tree1)) {
			case DeleteFailed:
				result = DeleteFailed;
				break;
			case DeleteSucceeded:
				intree->ptrs[1] = tree1;
				intree->key[0] = least_key(tree1);
				intree->range = intree->ptrs[0]->range;
				intree->range += intree->ptrs[1]->range;
				*outtree = intree;
				result = DeleteSucceeded;
				break;
			case DeletedBranch:
				if (tree1 == NULL) {
					*outtree = intree->ptrs[0];
					delete intree;
					result = DeletedBranch;
				} else if (intree->ptrs[0]->kind == node23::TWO_NODE) {
					// Take the node from the right and insert in the left
					tree2 = intree->ptrs[0];
					tree3 = tree2->ptrs[0];
					tree4 = tree2->ptrs[1];
					tree2->kind = node23::THREE_NODE;
					tree2->ptrs[0] = tree3;
					tree2->ptrs[1] = tree4;
					tree2->ptrs[2] = tree1;
					tree2->link = tree3->link;
					tree2->key[0] = least_key(tree4);
					tree2->key[1] = least_key(tree1);
					tree2->range = tree3->range;
					tree2->range += tree4->range;
					tree2->range += tree1->range;
					*outtree = tree2;
					delete intree;
					result = DeletedBranch;
				} else {
					// Split one from the left and add it to the right
					tree2 = intree->ptrs[0];
					tree3 = new node23(node23::TWO_NODE); // intree->ptrs[1]; - can't reuse a deleted node
					tree3->ptrs[0] = tree2->ptrs[2];
					tree3->ptrs[1] = tree1;
					tree3->link = tree3->ptrs[0]->link;
					tree3->key[0] = least_key(tree1);
					tree3->range = tree3->ptrs[0]->range;
					tree3->range += tree3->ptrs[1]->range;
					intree->ptrs[1] = tree3;

					tree2->kind = node23::TWO_NODE;
					tree2->range = tree2->ptrs[0]->range;
					tree2->range += tree2->ptrs[1]->range;

					intree->range = tree2->range;
					intree->range += tree3->range;
					intree->key[0] = least_key(tree3);
					*outtree = intree;
					result = DeleteSucceeded;
				}
				break;
			}
		}
	} else if (compare(key, intree->key[0]) < 0) {
		switch (two_three_delete(intree->ptrs[0], key, range, value, &tree1)) {
		case DeleteFailed:
			result = DeleteFailed;
			break;
		case DeleteSucceeded:
			intree->link = intree->ptrs[0]->link;
			*outtree = intree;
			result = DeleteSucceeded;
			break;
		case DeletedBranch:
			if (tree1 == NULL) {
				tree1 = intree->ptrs[1];
				tree2 = intree->ptrs[2];
				intree->kind = node23::TWO_NODE;
				intree->ptrs[0] = tree1;
				intree->ptrs[1] = tree2;
				intree->link = tree1->link;
				intree->key[0] = least_key(tree2);
				intree->range = tree1->range;
				intree->range += tree2->range;
				*outtree = intree;
				result = DeleteSucceeded;
			} else if (intree->ptrs[1]->kind == node23::TWO_NODE) {
				// Take the node from the left and insert into the middle
				tree2 = intree->ptrs[1];
				tree3 = tree2->ptrs[0];
				tree4 = tree2->ptrs[1];
				tree2->kind = node23::THREE_NODE;
				tree2->ptrs[0] = tree1;
				tree2->ptrs[1] = tree3;
				tree2->ptrs[2] = tree4;
				tree2->link = tree1->link;
				tree2->key[0] = least_key(tree3);
				tree2->key[1] = least_key(tree4);
				tree2->range = tree1->range;
				tree2->range += tree3->range;
				tree2->range += tree4->range;
				tree3 = intree->ptrs[2];
				intree->kind = node23::TWO_NODE;
				intree->ptrs[0] = tree2;
				intree->ptrs[1] = tree3;
				intree->link = tree2->link;
				intree->key[0] = least_key(tree3);
				intree->range = tree2->range;
				intree->range += tree3->range;
				*outtree = intree;
				result = DeleteSucceeded;
			} else {
				// Split off one from the middle and attach it to the left
				tree2 = intree->ptrs[1];
				tree3 = new node23(node23::TWO_NODE); // intree->ptrs[0]; - can't reuse a deleted node
				tree3->kind = node23::TWO_NODE;
				tree3->ptrs[0] = tree1;
				tree3->ptrs[1] = tree2->ptrs[0];
				tree3->link = tree1->link;
				tree3->key[0] = least_key(tree3->ptrs[1]);
				tree3->range = tree1->range;
				tree3->range += tree3->ptrs[1]->range;
				intree->link = tree1->link;
				intree->ptrs[0] = tree3;

				tree3 = tree2->ptrs[1];
				tree4 = tree2->ptrs[2];
				tree2->kind = node23::TWO_NODE;
				tree2->ptrs[0] = tree3;
				tree2->ptrs[1] = tree4;
				tree2->link = tree3->link;
				tree2->key[0] = least_key(tree4);
				tree2->range = tree3->range;
				tree2->range += tree4->range;
				intree->key[0] = least_key(tree2);
				*outtree = intree;
				result = DeleteSucceeded;
			}
			break;
		}
	} else if (compare(key, intree->key[1]) < 0) {
		switch (two_three_delete(intree->ptrs[1], key, range, value, &tree1)) {
		case DeleteFailed:
			result = DeleteFailed;
			break;
		case DeleteSucceeded:
			intree->ptrs[1] = tree1;
			intree->key[0] = least_key(tree1);
			intree->range = intree->ptrs[0]->range;
			intree->range += intree->ptrs[1]->range;
			intree->range += intree->ptrs[2]->range;
			*outtree = intree;
			result = DeleteSucceeded;
			break;
		case DeletedBranch:
			if (tree1 == NULL) {
				// Make it into a two node.  No loss of height
				tree1 = intree->ptrs[0];
				tree2 = intree->ptrs[2];
				intree->kind = node23::TWO_NODE;
				intree->ptrs[0] = tree1;
				intree->ptrs[1] = tree2;
				intree->link = tree1->link;
				intree->key[0] = least_key(tree2);
				intree->range = intree->ptrs[0]->range;
				intree->range += intree->ptrs[1]->range;
				*outtree = intree;
				result = DeleteSucceeded;
			} else if (intree->ptrs[0]->kind == node23::TWO_NODE) {
				// Take the node from the middle and insert it to the left
				tree2 = intree->ptrs[0];
				tree3 = tree2->ptrs[0];
				tree4 = tree2->ptrs[1];
				tree2->kind = node23::THREE_NODE;
				tree2->ptrs[0] = tree3;
				tree2->ptrs[1] = tree4;
				tree2->ptrs[2] = tree1;
				tree2->link = tree3->link;
				tree2->key[0] = least_key(tree4);
				tree2->key[1] = least_key(tree1);
				tree2->range = tree3->range;
				tree2->range += tree4->range;
				tree2->range += tree1->range;
				tree3 = intree->ptrs[2];
				intree->kind = node23::TWO_NODE;
				intree->ptrs[0] = tree2;
				intree->ptrs[1] = tree3;
				intree->link = tree2->link;
				intree->key[0] = least_key(tree3);
				intree->range = tree2->range;
				intree->range += tree3->range;
				*outtree = intree;
				result = DeleteSucceeded;
			} else {
				// Split off one from the left and attach it to the middle
				tree2 = intree->ptrs[0];
				tree3 = new node23(node23::TWO_NODE); // intree->ptrs[1];
				tree3->kind = node23::TWO_NODE;
				tree3->ptrs[0] = tree2->ptrs[2];
				tree3->ptrs[1] = tree1;
				tree3->link = tree3->ptrs[0]->link;
				tree3->key[0] = least_key(tree1);
				tree3->range = tree3->ptrs[0]->range;
				tree3->range += tree1->range;
				intree->ptrs[1] = tree3;

				tree3 = tree2->ptrs[0];
				tree4 = tree2->ptrs[1];
				tree2->kind = node23::TWO_NODE;
				tree2->ptrs[0] = tree3;
				tree2->ptrs[1] = tree4;
				tree2->link = tree3->link;
				tree2->key[0] = least_key(tree4);
				tree2->range = tree3->range;
				tree2->range += tree4->range;

				intree->link = tree2->link;
				intree->key[0] = least_key(intree->ptrs[1]);
				intree->range = intree->ptrs[0]->range;
				intree->range += intree->ptrs[1]->range;
				intree->range += intree->ptrs[2]->range;
				*outtree = intree;
				result = DeleteSucceeded;
			}
			break;
		}
	} else {
		switch (two_three_delete(intree->ptrs[2], key, range, value, &tree1)) {
		case DeleteFailed:
			result = DeleteFailed;
			break;
		case DeleteSucceeded:
			intree->ptrs[2] = tree1;
			intree->key[1] = least_key(tree1);
			intree->range = intree->ptrs[0]->range;
			intree->range += intree->ptrs[1]->range;
			intree->range += intree->ptrs[2]->range;
			*outtree = intree;
			result = DeleteSucceeded;
			break;
		case DeletedBranch:
			if (tree1 == NULL) {
				// Collapse a three way branch to a two way branch
				tree1 = intree->ptrs[0];
				tree2 = intree->ptrs[1];
				intree->kind = node23::TWO_NODE;
				intree->ptrs[0] = tree1;
				intree->ptrs[1] = tree2;
				intree->link = tree1->link;
				intree->key[0] = least_key(tree2);
				intree->range = tree1->range;
				intree->range += tree2->range;
				*outtree = intree;
				result = DeleteSucceeded;
			} else if (intree->ptrs[1]->kind == node23::TWO_NODE) {
				// Take the node from the right and insert it into the middle
				tree2 = intree->ptrs[1];
				tree3 = tree2->ptrs[0];
				tree4 = tree2->ptrs[1];
				tree2->kind = node23::THREE_NODE;
				tree2->ptrs[0] = tree3;
				tree2->ptrs[1] = tree4;
				tree2->ptrs[2] = tree1;
				tree2->link = tree3->link;
				tree2->key[0] = least_key(tree4);
				tree2->key[1] = least_key(tree1);
				tree2->range = tree3->range;
				tree2->range += tree4->range;
				tree2->range += tree1->range;

				intree->kind = node23::TWO_NODE;
				intree->ptrs[1] = tree2;
				intree->key[0] = least_key(tree2);
				intree->range = intree->ptrs[0]->range;
				intree->range += tree2->range;
				*outtree = intree;
				result = DeleteSucceeded;
			} else {
				// Split off one from the middle and attach it to the right
				tree2 = intree->ptrs[1];
				tree3 = new node23(node23::TWO_NODE); // intree->ptrs[2]; - can't reuse a deleted node
				tree3->ptrs[0] = tree2->ptrs[2];
				tree3->ptrs[1] = tree1;
				tree3->link = tree3->ptrs[0]->link;
				tree3->key[0] = least_key(tree1);
				tree3->range = tree3->ptrs[0]->range;
				tree3->range += tree1->range;
				intree->ptrs[2] = tree3;

				tree3 = tree2->ptrs[0];
				tree4 = tree2->ptrs[1];
				tree2->kind = node23::TWO_NODE;
				tree2->ptrs[0] = tree3;
				tree2->ptrs[1] = tree4;
				tree2->link = tree3->link;
				tree2->key[0] = least_key(tree4);
				tree2->range = tree3->range;
				tree2->range += tree4->range;
				intree->key[0] = least_key(intree->ptrs[1]);
				intree->key[1] = least_key(intree->ptrs[2]);
				intree->range = intree->ptrs[0]->range;
				intree->range += intree->ptrs[1]->range;
				intree->range += intree->ptrs[2]->range;
				*outtree = intree;
				result = DeleteSucceeded;
			}
			break;
		}
	}

#if DEBUG_TREES
	if (result != DeleteFailed) {
		if (*outtree != NULL) {
			switch ((*outtree)->kind) {
			case node23::LEAF_NODE:
#if DEBUG_LEAVES
				{
				Range3D box;
				range_data *trangedata;
				for (trangedata=(*outtree)->leaf->data;trangedata!=NULL;trangedata=trangedata->next) {
					((PolyTriPtr)trangedata->data)->compute_bbox(box);
					assert((*outtree)->range.Contains(box, (float)EPSILON));
				}
				}
				assert((*outtree)->link == (*outtree)->leaf);
#endif
				break;
			case node23::TWO_NODE:
				assert((*outtree)->range.Contains((*outtree)->ptrs[0]->range, (float)EPSILON));
				assert((*outtree)->range.Contains((*outtree)->ptrs[1]->range, (float)EPSILON));
				assert((*outtree)->link == (*outtree)->ptrs[0]->link);
				assert((*outtree)->key[0] == (*outtree)->ptrs[1]->link->key);
				assert(compare((*outtree)->key[0], least_key((*outtree)->ptrs[1])) == 0);
				break;
			case node23::THREE_NODE:
				assert((*outtree)->range.Contains((*outtree)->ptrs[0]->range, (float)EPSILON));
				assert((*outtree)->range.Contains((*outtree)->ptrs[1]->range, (float)EPSILON));
				assert((*outtree)->range.Contains((*outtree)->ptrs[2]->range, (float)EPSILON));
				assert((*outtree)->link == (*outtree)->ptrs[0]->link);
				assert((*outtree)->key[0] == (*outtree)->ptrs[1]->link->key);
				assert((*outtree)->key[1] == (*outtree)->ptrs[2]->link->key);
				assert(compare((*outtree)->key[0], least_key((*outtree)->ptrs[1])) == 0);
				assert(compare((*outtree)->key[1], least_key((*outtree)->ptrs[2])) == 0);
				assert(compare((*outtree)->key[0], (*outtree)->key[1]) < 0);
				break;
			}
		}
	} else {
		sprintf(sbuf, "Failed to delete %p", value);
		OutputDebugString(sbuf);
	}
#endif
	return result;
}

// Retrieve the value for a name from a 2-3 tree. */

// Insert data that covers a range/box in 3D
Range_Tree::nodeptr Range_Tree::
Insert(Range3D &range, void *data)
{
	nodeptr left_tree, right_tree;
	nodeptr leaf = NULL;
	RangeKey key(range[0][0], range[0][1], range[0][2]);

#if defined(_DEBUG) && 0
sprintf(sbuf, "Insert: %p\n", data);
OutputDebugString(sbuf);
#endif
	if (insert3(Root, key, range, data, &left_tree, &leaf)) {
		Root = left_tree;
	} else if (split(Root, key, range, data, &left_tree, &right_tree, &leaf)) {
		Root = new node23(node23::TWO_NODE);
		Root->ptrs[0] = left_tree;
		Root->link = left_tree->link;
		Root->key[0] = least_key(right_tree);
		Root->range = left_tree->range;
		Root->range += right_tree->range;
		Root->ptrs[1] = right_tree;
	}
#if DEBUG_TREES
	switch (Root->kind) {
	case node23::LEAF_NODE:
#if DEBUG_LEAVES
		Range3D box;
		range_data *trangedata;
		for (trangedata=Root->leaf->data;trangedata!=NULL;trangedata=trangedata->next) {
			((PolyTriPtr)trangedata->data)->compute_bbox(box);
			assert(Root->range.Contains(box, (float)EPSILON));
		}
#endif
		break;
	case node23::TWO_NODE:
		assert(Root->range.Contains(Root->ptrs[0]->range, (float)EPSILON));
		assert(Root->range.Contains(Root->ptrs[1]->range, (float)EPSILON));
		assert(Root->key[0] == Root->ptrs[1]->link->key);
		break;
	case node23::THREE_NODE:
		assert(Root->range.Contains(Root->ptrs[0]->range, (float)EPSILON));
		assert(Root->range.Contains(Root->ptrs[1]->range, (float)EPSILON));
		assert(Root->range.Contains(Root->ptrs[2]->range, (float)EPSILON));
		assert(Root->key[0] == Root->ptrs[1]->link->key);
		assert(Root->key[1] == Root->ptrs[2]->link->key);
		break;
	}
#endif

	entry_count++;
#if defined(_DEBUG) && 0
	Show();
	OutputDebugString("\n");
#endif

	return leaf;
}

// Delete an element from the range tree
void Range_Tree::
Delete(Range3D &range, void *value)
{
	RangeKey key(range[0][0], range[0][1], range[0][2]);
#if defined(_DEBUG) && 0
sprintf(sbuf, "Delete: %p\n", value);
OutputDebugString(sbuf);
#endif
	nodeptr tree1;
	switch (two_three_delete(Root, key, range, value, &tree1)) {
	case DeleteFailed:
#if defined(_DEBUG)
		sprintf(sbuf, "Failed to delete %p \n", value);
		OutputDebugString(sbuf);
		sprintf(sbuf, "   (%g,%g,%g) (%g,%g,%g) from\n",
			range[0][0], range[0][1], range[0][2],
			range[1][0], range[1][1], range[1][2]);
		OutputDebugString(sbuf);
		Show();
#endif
		break;
	case DeleteSucceeded:
		Root = tree1;
		entry_count--;
		break;
	case DeletedBranch:
		// delete Root;
		Root = tree1;
		entry_count--;
		break;
	}
}

// Delete an element from the range tree - this one is optimized for the
// case where the leaf that holds the value is known.
void Range_Tree::
Delete(nodeptr leaf, void *value)
{
	// TBD: implement the deletion
}

void Range_Tree::
shread_two_three(nodeptr intree)
{
	rangedataptr tnode1, tnode2;

	if (intree == NULL)
		return;
	else if (intree->kind == node23::LEAF_NODE) {
		// Remove all entries
		for (tnode1=intree->leaf->data;tnode1!=NULL;) {
			tnode2 = tnode1;
			tnode1 = tnode1->next;
			cleanup(tnode2->data); // User supplied deallocation
			// Do we need this: delete[] tnode2->key; ?
			delete tnode2;
		}
		delete intree->leaf;
	} else if (intree->kind == node23::TWO_NODE) {
		shread_two_three(intree->ptrs[0]);
		shread_two_three(intree->ptrs[1]);
	} else {
		shread_two_three(intree->ptrs[0]);
		shread_two_three(intree->ptrs[1]);
		shread_two_three(intree->ptrs[2]);
	}
	delete intree;
}

// User supplied tree manipulation - default is to do nothing
void Range_Tree::
process(Range3D &range, void *value, void *data)
{
	;
}

// User supplied cleanup routine - default is to do nothing
void Range_Tree::
cleanup(void *data)
{
	;
}

// Here we get to some implementation specfic uses of the 2-3 tree.
// Access each entry in the tree and do something with it...
void Range_Tree::
process_two_three(nodeptr tree, Range3D &range, void *data)
{
	if (tree == NULL)
		// No data - should probably throw an assert...
		return;
	else if (!tree->range.Intersects(range))
		// This tree doesn't overlap the range we are interested in
		return;
	else if (tree->kind == node23::LEAF_NODE)
		process(range, tree->leaf->data, data);
	else if (tree->kind == node23::TWO_NODE) {
		process_two_three(tree->ptrs[0], range, data);
		process_two_three(tree->ptrs[1], range, data);
	} else {
		process_two_three(tree->ptrs[0], range, data);
		process_two_three(tree->ptrs[1], range, data);
		process_two_three(tree->ptrs[2], range, data);
	}
}

void Range_Tree::
process_all_nodes(nodeptr tree, void *data)
{
	if (tree == NULL)
		return;
	else if (tree->kind == node23::LEAF_NODE)
		process(tree->range, tree->leaf->data, data);
	else if (tree->kind == node23::TWO_NODE) {
		process_all_nodes(tree->ptrs[0], data);
		process_all_nodes(tree->ptrs[1], data);
	} else {
		process_all_nodes(tree->ptrs[0], data);
		process_all_nodes(tree->ptrs[1], data);
		process_all_nodes(tree->ptrs[2], data);
	}
}

// Visit every entry in the symbol tree (in order), applying the function "process"
// to each token as it is encountered
void Range_Tree::
Process(Range3D &range, void *data)
{
	process_two_three(Root, range, data);
}

// Visit every entry in the symbol tree (in order), applying the function "process"
// to each token as it is encountered
void Range_Tree::
ProcessAll(void *data)
{
	process_all_nodes(Root, data);
}

int Range_Tree::
exhaustive_search(nodeptr tree, void *value)
{
	if (tree == NULL)
		return 0;
	else if (tree->kind == node23::LEAF_NODE) {
		int count = 0;
		rangedataptr root;
		for (root=tree->leaf->data;root!=NULL;root=root->next)
			if (root->data == value)
				count++;
		return count;
	} else if (tree->kind == node23::TWO_NODE) {
		return exhaustive_search(tree->ptrs[0], value) +
				exhaustive_search(tree->ptrs[1], value);
	} else {
		return exhaustive_search(tree->ptrs[0], value) +
				exhaustive_search(tree->ptrs[1], value) +
				exhaustive_search(tree->ptrs[2], value);
	}
}

int Range_Tree::
InTree(void *value)
{
	return exhaustive_search(Root, value);
}

Range_Tree::
Range_Tree() {
	Root = NULL;
	entry_count = 0;
}

// Clean up the tree.  Since the calling program may have allocated memory for the
// token, we let it call a processing function as we are deleting each element
// from the tree.
Range_Tree::
~Range_Tree()
{
	shread_two_three(Root);
	Root = NULL;
}
