#include "StdAfx.h"
#include "PovrayTranslator.h"

#include <MBaseObject.h>
#include <MSystemManager.h>
#include <MAnimMesh.h>
#include <MMeshShape.h>
#include <HelperFuncs.h>
#include <MMath.h>

#include <string.h>
#include <stdio.h>
#include <stdlib.h>


/********************/

#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif



PovrayTranslator::PovrayTranslator()
{
}

PovrayTranslator::~PovrayTranslator()
{
}

static std::string toString(MVector3 &vec) {
  char buf[1024];
  sprintf(buf, "%.3f,%.3f,%.3f", vec.x, vec.y, vec.z);
  return buf;
}


MTranslatorPtr PovrayTranslator::createNew() const {
  PovrayTranslator   *NewTranslator = new PovrayTranslator();
  
  return NewTranslator;
}

std::string PovrayTranslator::getFilter() { 
  return "*.POV"; 
}
std::string PovrayTranslator::getFilterDescription() { 
  return "Persistence Of Vision scenes"; 
}

bool PovrayTranslator::exportFile(const MStr &Filename,
                                  const MScenePtr &scene,
                                  const MParameterObjectListPtr &options) 
{
  MSceneObjectPtr camera;
  if (options != NULL) {
    MStr cameraName;
    options->getParameter("camera")->getValueString(cameraName);
    camera = AZTEC_CAST(MSceneObject, scene->getObjectList()->findObject(cameraName));
  }

  FILE *out;

  out = fopen(Filename.c_str(), "wt");

  // if we couldn't open the file, just bail out.
  if (out == NULL) {
    return false;
  }

  objectsWritten.clear();


  fprintf(out, "// This is a POV scene file generated by Aztec 3D Modeller.\n");
  fprintf(out, "// http://aztec.sourceforge.net/.\n");
  fprintf(out, "\n");

  // this is the version of pov ray that is required. Since we will be using 
  // meshes, version 3.5 will probably be appropriate.
  fprintf(out, "#version 3.5\n\n");
  
  fprintf(out, "#include \"colors.inc\"\n\n");
  fprintf(out, "#include \"textures.inc\"\n\n");

  // TODO: when Aztec gets a camera type, make it so the camera info is
  // written out like the other objects.

  float cameraDistance = 50;
  MVector3 cameraAngles(-30,0,-135);
  MVector3 cameraFocus(0,0,0);

  if (camera == NULL) {
    fprintf(out, "camera {\n"
                 "   direction y\n"
                 "   up <0,0,1>\n"
                 "   right <1.333,0,0>\n"
                 "\n"
                 "   // 0, distance from focus, 0\n"
                 "   location <0,-%.3f,0>\n"
                 "\n"
                 "   // elevation, roll, orbit angles\n"
                 "   rotate <%.3f,%.3f,%.3f>\n"
                 "\n"
                 "   // this is the focus point that the camera is looking at.\n"
                 "   translate <%.3f,%.3f,%.3f>\n"
                 "}\n",
                 cameraDistance,
                 cameraAngles.x,cameraAngles.y,cameraAngles.z,
                 cameraFocus.x, cameraFocus.y, cameraFocus.z);
  } else {
    cameraAngles = camera->getRotateVector(scene->getTime());
    cameraFocus = camera->getTranslateVector(scene->getTime());
    fprintf(out, "camera {\n"
                 "  direction y\n"
                 "  up <0,0,1>\n"
                 "  right <1.333,0,0>\n"
                 "  location <0,0,0>\n");
    fprintf(out, "  rotate <0,0,%.3f>\n", cameraAngles.z);
    fprintf(out, "  rotate <0,%.3f,0>\n", cameraAngles.y);
    fprintf(out, "  rotate <%.3f,0,0>\n", cameraAngles.x);
    fprintf(out, "  translate <%s>\n", toString(cameraFocus).c_str());
    fprintf(out, "}\n");
  }

  // loop over all the objects and write them out.
  writeOutObjects(out, scene);

  fflush(out);
  fclose(out);

  return true;
}

bool PovrayTranslator::canImport() {
  return false;
}

bool PovrayTranslator::canExport() {
  return true;
}

bool PovrayTranslator::canImportFile(MStr Filename) {
  return false;
}

MStr PovrayTranslator::getClassName() {
  return "PovrayTranslator";
}

bool PovrayTranslator::importFile(MStr Filename, MScenePtr Scene) {
  return false;
}

bool PovrayTranslator::exportFile(MStr Filename, MScenePtr Scene) {
  return exportFile(Filename, Scene, NULL);
}

void PovrayTranslator::writeOutObjects(FILE *out, const MScenePtr &scene) {
  MBaseObjectPtr object;

  scene->getObjectList()->beginIteration();
  while ((object = scene->getObjectList()->getNext()) != NULL) {
    writeSceneShape(out, scene, AZTEC_CAST(MSceneObject, object));    
  }
  scene->getObjectList()->endIteration();

  scene->getObjectList()->beginIteration();
  while ((object = scene->getObjectList()->getNext()) != NULL) {
    MSceneObjectPtr sceneObject = AZTEC_CAST(MSceneObject, object);

    // if we don't have a scene object, do nothing
    if (sceneObject == NULL) {
      continue;
    }

    // do up the tree until we reach the parent.
    while (sceneObject->getParent() != NULL) {
      sceneObject = sceneObject->getParent();
    }
    
    writeSceneObject(out, scene, sceneObject);    
  }
  scene->getObjectList()->endIteration();
}

void PovrayTranslator::writeSceneShape(FILE *out, const MScenePtr &scene, const MSceneObjectPtr &object) {
  if (object == NULL) {
    return;
  }

  if (object->getShapeObject() != NULL) {
    fprintf(out, "#declare %s = ", object->getShapeObject()->getName().c_str());
    writeShapeObject(out, scene, object->getShapeObject());
  }
  fprintf(out, "\n");

}

void PovrayTranslator::writeSceneObject(FILE *out, const MScenePtr &scene, const MSceneObjectPtr &object) {
  // if we don't have an object, or if we have already written it out,
  // don't do anything.
  if (object == NULL || objectsWritten.find(object) != objectsWritten.end()) {
    return;
  }

  MVector3 translation = object->getTranslateVector(scene->getTime());
  MVector3 rotation = object->getRotateVector(scene->getTime());
  MVector3 scale = object->getScaleVector(scene->getTime());
  MVector3 colour;
  object->findParameter("viewCol")->getValueVector(colour);

  fprintf(out, "object {\n");

  fprintf(out, "union {\n");

  if (object->getShapeObject() != NULL) {
    fprintf(out, "object {\n");
    fprintf(out, "  %s\n", object->getShapeObject()->getName().c_str());
    fprintf(out, "}\n");
  }

  MTreeObjectNodePtr node = scene->getObjectList()->findObject(object);

  if (node != NULL) {
    node = node->getFirstChild();
    while (node != NULL) {
      writeSceneObject(out, scene, AZTEC_CAST(MSceneObject, node->getObject()));

      node = node->getNextSibling();
    }
  }

  // TODO: write out the children as well.
  fprintf(out, "}\n");

  fprintf(out, "  scale <%s>\n", toString(scale).c_str());
  fprintf(out, "  rotate <0,0,%.3f>\n", rotation.z);
  fprintf(out, "  rotate <0,%.3f,0>\n", rotation.y);
  fprintf(out, "  rotate <%.3f,0,0>\n", rotation.x);
  fprintf(out, "  translate <%s>\n", toString(translation).c_str());

  fprintf(out, "  pigment { color <%.3f,%.3f,%.3f> }\n", colour.x, colour.y, colour.z);
  fprintf(out, "  finish { Dull }\n");
    
  fprintf(out, "}\n");
  fprintf(out, "\n");

  objectsWritten.insert(object);
}

void PovrayTranslator::writeShapeObject(FILE *out, const MScenePtr &scene, const MShapeObjectPtr &shape) {
  // we can't do anything if we don't have a shape.
  if (shape == NULL) {
    return;
  }

  // check for a light.
  if (AZTEC_CAST(MLight, shape) != NULL) {
    writeLightObject(out, scene, AZTEC_CAST(MLight, shape));
    return;
  }

  // if we haven't found a match, just write out the mesh.
  writeMeshObject(out, scene, shape->convertToMesh());

}

void PovrayTranslator::writeLightObject(FILE *out, const MScenePtr &scene, const MLightPtr &light) {
  // if we don't have light, we can't do anything.
  if (light == NULL) {
    return;
  }
  MVector3 colour;
  light->findParameter("Colour")->getValueVector(colour);

  // out position is already defined by the scene object, so we are at 
  // position 0,0,0
  fprintf(out, "light_source {\n");
  fprintf(out, "  <0,0,0>,\n");
  fprintf(out, "  rgb <%.3f,%.3f,%.3f>\n", colour.x, colour.y, colour.z);
  fprintf(out, "}\n");

}

void PovrayTranslator::writeMeshObject(FILE *out, const MScenePtr &scene, const MMeshPtr &mesh) {
  // we can't do anything if we don't have a mesh
  if (mesh == NULL) {
    fprintf(out, "sphere {\n");
    fprintf(out, "< 0, 0, 0 >, 0\n");
    fprintf(out, "}\n");
    return;
  }

  fprintf(out, "mesh2 { \n");

  // write out the vertices
  fprintf(out, "vertex_vectors {\n");
  fprintf(out, "%i", mesh->getNumVerts());
  for (int i = 0; i < mesh->getNumVerts(); ++i) {
    MVector3 pos = mesh->getVertexPosition(i);
    fprintf(out, ",\n<%s>", toString(pos).c_str());
  }
  fprintf(out, "\n}\n");

  // write out the normals
  fprintf(out, "normal_vectors {\n");
  fprintf(out, "%i", mesh->getNumVerts());
  for (int i = 0; i < mesh->getNumVerts(); ++i) {
    MVector3 pos = mesh->getVertexNormal(i);
    fprintf(out, ",\n<%s>", toString(pos).c_str());
  }
  fprintf(out, "\n}\n");

  // write out the triangles
  fprintf(out, "face_indices {\n");
  fprintf(out, "%i", mesh->getNumTris());
  for (int i = 0; i < mesh->getNumTris(); ++i) {
    MTriangle const *tri = mesh->getTriangle(i);
    fprintf(out, ",\n<%i,%i,%i>", tri->getVertex(0), tri->getVertex(1), tri->getVertex(2));
  }
  fprintf(out, "\n}\n");

  fprintf(out, "\n}\n");
}


