#include <AztecMainPCH.h>
#include "tools/ExtrudeTool.h"

#include "MSystemManager.h"
#include "MEditableMesh.h"
#include "MAnimMesh.h"
#include "MUIManager.h"

#include "MDLGlobs.h"
#include "DlgGlobs.h"
#include "MDLMsgs.h"

#include "KeyFuncGeneral.h"
#include "ModelGLConst.h"

// standard includes
#include <math.h>
#include <gl/gl.h>
#include <set>

using std::set;

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

int KToolExtrude() {
  if (MUIManager::getComponentMode() == MUIManager::OBJECT_MODE) {
    KModeFace();
  }
  g_ToolMan.PushTool(new MExtrudeToolType);
  g_MainDlg->PostMessage(MM_UPDATEVIEWPORTS, MMC_UPDATE_ALL, 0);
  return 1;
}

//--------------------
//  MExtrudeToolType
//--------------------
MVector3    m_ExtrudeVec;
float       m_Dist;

MExtrudeToolType::MExtrudeToolType()
{
  setName("toolExtrude");

  m_RequiresSel = true;
  m_Dist = 0.0f;
  m_ExtrudeVec.set(1,0,0);
  m_Extruding = false;
}

int MExtrudeToolType::DrawTool(bool Select, MShiftState ShiftState, MBaseViewWndPtr View) {
  // we want to draw an arrow at the centre of the selection, 
  // and and pointing in the average normal direction.

  // update our normal. If nothing was selected, we don't draw anything
  if (!m_Extruding && !updateExtrudeNormal()) {
    return 1;
  }

  // Get the current viewport and cast it to a gl wnd.
  COpenGLWndPtr  GLWnd;
  
  GLWnd = AZTEC_CAST(COpenGLWnd, View);
  
  if (GLWnd == NULL) {
    return 0;
  }
  
  glPushAttrib(GL_ENABLE_BIT);
  glDisable(GL_DEPTH_TEST);
  
  
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  glLoadIdentity();
  
  // Perform the viewport transformation
  
  GLWnd->DoViewportTransform();
  glMatrixMode(GL_MODELVIEW);
  
  glPushMatrix();

  g_Scene->GLTransform(g_Scene->getSelectedObjectList()->getHead());

  glPushMatrix();

  float ScaleFact = GLWnd->GetScaleFactor(m_CurrentPos);
  glTranslatef(m_CurrentPos.x, m_CurrentPos.y, m_CurrentPos.z);
  
  glScalef(ScaleFact, ScaleFact, ScaleFact);
  
  glMultMatrixf((float*)m_GLTransform.m);

  // draw the axis icon
  GLWnd->DrawAxisIcon(g_IconSize, 0.5f, 2.0f, Select, DRAWAXIS_Z | DRAWAXIS_ARROWS);

  // if we are extruding the faces, draw an arrow where we started from.
  if (m_Extruding) {
    glPopMatrix();
    glPushMatrix();

    ScaleFact = GLWnd->GetScaleFactor(m_StartPos);
    glTranslatef(m_StartPos.x, m_StartPos.y, m_StartPos.z);
    glScalef(ScaleFact, ScaleFact, ScaleFact);
    glMultMatrixf((float*)m_GLTransform.m);

    GLWnd->DrawAxisIcon(g_IconSize, 0.2f, 2.0f, false, DRAWAXIS_ALLGRAY | DRAWAXIS_Z | DRAWAXIS_ARROWS);
  }
  
  glPopMatrix();
  glPopMatrix();
  glPopMatrix();
  
  glPopAttrib();
  
  return 1;
}

bool MExtrudeToolType::updateExtrudeNormal() {
  MSceneObjectPtr sceneObj;
  MEditableMeshPtr meshObj;
  
  sceneObj = AZTEC_CAST(MSceneObject, g_Scene->getSelectedObjectList()->getHead());

  if (sceneObj != NULL && sceneObj->getShapeObject() != NULL) {
    meshObj = AZTEC_CAST(MEditableMesh, sceneObj->getShapeObject()->convertToMesh());
  }

  if (meshObj == NULL) {
    return false;
  }

  MVector3    Norm, centre;
  int         n, count;
  
  count = 0;
  Norm.set(0,0,0);

  if (MUIManager::getComponentMode() == MUIManager::OBJECT_MODE) {
    return false;
  }

  int numComps = meshObj->getComponentCount(MUIManager::getComponentMode());

  // go through and extract the average normal and centre
  for(n = 0; n < numComps; n++) {
    if (!meshObj->isComponentFlagged(MUIManager::getComponentMode(), n)) {
      continue;
    }

    centre += meshObj->getComponentCentre(MUIManager::getComponentMode(), n);
    Norm += meshObj->getComponentNormal(MUIManager::getComponentMode(), n);
    count++;
  }
  if (count) {
    Norm.normalize();
    centre /= (float)count;
  } else {
    return false;
  }
  
  m_StartPos = centre;
  m_CurrentPos = m_StartPos;
  m_ExtrudeCentre = centre;
  m_ExtrudeVec = Norm;

  // transform the drawing to out z axis lines up with the normal.
  {
    MVector3  x,y,z;

    m_GLTransform.identity();
    z = m_ExtrudeVec;

    // construct two otrhonormal vectors
    y.set(-z.y, z.z, -z.x);
    x = z.crossProduct(y);
    y = z.crossProduct(x);

    m_GLTransform.m[0][0] = x.x; m_GLTransform.m[0][1] = x.y; m_GLTransform.m[0][2] = x.z;
    m_GLTransform.m[1][0] = y.x; m_GLTransform.m[1][1] = y.y; m_GLTransform.m[1][2] = y.z;
    m_GLTransform.m[2][0] = z.x; m_GLTransform.m[2][1] = z.y; m_GLTransform.m[2][2] = z.z;
  }

  return true;
}

#define TEMP_MARK_TODO 0x80
#define TEMP_MARK_DONE 0x40

static void resetTempMarks(MEditableMeshPtr mesh) {
  // any edge that has already been split on this vertex will
  // be marked with the TEMP_MARK_TODO flag. Right now we unflag all the 
  // vertices in the mesh.

  mesh->unsetVertexFlags(TEMP_MARK_TODO | TEMP_MARK_DONE);
}

// 
/**
 * Splits the edges around the vertices of the mesh with the given flags.
 *
 * @param mesh The mesh whose edges we are splitting
 * @param flags The flags that the vertices have that we want to split
 * @param dist The distance from each vertex to split connected edges. 
 * @param distIsFraction If true, the dist parameter is considered to be
 *                       a fraction. If false, it is a distance in local space.
 * 
 */
static void splitEdgesAroundVerts(MEditableMeshPtr mesh, AztecFlags flags, float dist = 0.5, bool distIsFraction = true) {
  float distSqr = dist*dist;

  // loop over the vertices in the mesh.

  for (int i = mesh->getNumVerts() - 1; i >= 0 ; --i) {
    if (!mesh->isVertexFlagged(i, flags)) {
      continue;
    }

    resetTempMarks(mesh);

    // now loop over the triangles in the mesh and mark
    // any vertices that need splitting
    for (int t = mesh->getNumTris() - 1; t >= 0 ; --t) {
      for (int e = 0; e < 3; ++e) {
        if (mesh->getTriVert(t, e) == i) {
          mesh->setVertexFlag(mesh->getTriVert(t, (e+1)%3), TEMP_MARK_TODO);
          mesh->setVertexFlag(mesh->getTriVert(t, (e+2)%3), TEMP_MARK_TODO);
          break;
        }
      }
    }


    // now loop over the triangles in the mesh and split the edges
    for (int t = mesh->getNumTris() - 1; t >= 0 ; --t) {
      MTriangle const *tri = mesh->getTriangle(t);

      // this array holds wether or not we should do the edge.
      // 0 means no split.
      // 1 means a split from the first vertex
      // -1 means a split, but from the other vertex
      int doEdge[3];
      doEdge[0] = 0;
      doEdge[1] = 0;
      doEdge[2] = 0;

      // any edge that has already been split on this vertex will
      // be marked with the TRIANGLE_FLAGFORCHANGE2 flag.

      // the edges that we have to do is those that
      // start with this vertex.
      for (int e = 0; e < 3; ++e) {
        if (mesh->getTriVert(t, e) == i) {
          int vert1, vert2, vert3;

          vert1 = i;
          vert2 = mesh->getTriVert(t, (e+1)%3);
          vert3 = mesh->getTriVert(t, (e+2)%3);

          doEdge[(e+1)%3] = 0;
          if (mesh->isVertexFlagged(vert2, TEMP_MARK_TODO) && !mesh->isVertexFlagged(vert2, TEMP_MARK_DONE)) {
            doEdge[e] = 1;
          }
          if (mesh->isVertexFlagged(vert3, TEMP_MARK_TODO) && !mesh->isVertexFlagged(vert3, TEMP_MARK_DONE)) {
            doEdge[(e+2)%3] = -1;
          }
          break;
        }
      }

      // see if any edges are marked for splitting
      for (int e = 0; e < 3; ++e) {
        if (doEdge[e] == 0) {
          continue;
        }
        // if this edge is marked to be split, lets split it
        int vert1, vert2;

        vert1 = mesh->getTriVert(t, e);
        vert2 = mesh->getTriVert(t, (e+1)%3);

        float frac = 0;
        if (distIsFraction) {
          frac = dist;
        } else {
          MVector3 vertDist;
          float length;

          vertDist = mesh->getVertexPosition(vert1);
          vertDist -= mesh->getVertexPosition(vert2);

          length = vertDist.length();
          if (dist < length) {
            frac = dist/length;
          }
        }

        if (frac > 0) {
          if (doEdge[e] == 1) {
            mesh->divideEdge(t * 3 + e, frac);
          } else {
            mesh->divideEdge(t * 3 + e, 1.0 - frac);
          }
        }

        mesh->unsetVertexFlag(vert1, TEMP_MARK_TODO);
        mesh->unsetVertexFlag(vert2, TEMP_MARK_TODO);
        mesh->setVertexFlag(vert1, TEMP_MARK_DONE);
        mesh->setVertexFlag(vert2, TEMP_MARK_DONE);
      }

    }

  }
}

/**
 * Splits the edges around the vertices of the mesh with the given flags.
 *
 * @param mesh The mesh whose edges we are splitting
 * @param flags The flags that the vertices have that we want to split
 * @param dist The distance from each vertex to split connected edges. 
 * @param distIsFraction If true, the dist parameter is considered to be
 *                       a fraction. If false, it is a distance in local space.
 * 
 */
static void splitEdgesAroundEdges(MEditableMeshPtr mesh, AztecFlags flags, float dist = 0.5, bool distIsFraction = true) {
  float distSqr = dist*dist;

  // loop over the vertices in the mesh.

  for (int i = mesh->getNumVerts() - 1; i >= 0 ; --i) {
    int vert = i;

    // if the vertex we are on isn't flagged, then there is no way we 
    // should split it.
    if (!mesh->isVertexFlagged(i, flags)) {
      continue;
    }

    resetTempMarks(mesh);

    // now loop over the triangles in the mesh and mark
    // any vertices that need splitting
    for (int t = mesh->getNumTris() - 1; t >= 0 ; --t) {
      for (int e = 0; e < 3; ++e) {
        if (mesh->getTriVert(t, e) == i) {
          mesh->setVertexFlag(mesh->getTriVert(t, (e+1)%3), TEMP_MARK_TODO );
          mesh->setVertexFlag(mesh->getTriVert(t, (e+2)%3), TEMP_MARK_TODO );
          break;
        }
      }
    }


    // now loop over the triangles in the mesh and split the edges
    for (int t = mesh->getNumTris() - 1; t >= 0 ; --t) {

      // this array holds wether or not we should do the edge.
      // 0 means no split.
      // 1 means a split from the first vertex
      // -1 means a split, but from the other vertex
      int doEdge[3];
      doEdge[0] = 0;
      doEdge[1] = 0;
      doEdge[2] = 0;

      // any edge that has already been split on this vertex will
      // be marked with the TRIANGLE_FLAGFORCHANGE2 flag.

      // the edges that we have to do is those that
      // start with this vertex.
      for (int e = 0; e < 3; ++e) {
        if (mesh->getTriVert(t, e) == i) {
          int vert2, vert3;

          vert2 = mesh->getTriVert(t, (e+1)%3);
          vert3 = mesh->getTriVert(t, (e+2)%3);

          doEdge[(e+1)%3] = 0;
          if (mesh->isVertexFlagged(vert2, TEMP_MARK_TODO) && !mesh->isVertexFlagged(vert2, TEMP_MARK_DONE)) {
            doEdge[e] = 1;
          }
          if (mesh->isVertexFlagged(vert3, TEMP_MARK_TODO) && !mesh->isVertexFlagged(vert3, TEMP_MARK_DONE)) {
            doEdge[(e+2)%3] = -1;
          }
          break;
        }
      }

      // see if any edges are marked for splitting
      for (int e = 0; e < 3; ++e) {
        if (doEdge[e] == 0) {
          continue;
        }
        // if this edge is marked to be split, lets split it
        int vert1, vert2;

        vert1 = mesh->getTriVert(t, e);
        vert2 = mesh->getTriVert(t, (e+1)%3);

        if (mesh->isVertexFlagged(vert1, flags) && mesh->isVertexFlagged(vert2, flags)) {
          continue;
        }

        if (!(vert1 == vert || vert2 == vert)) {
          continue;
        }

        float frac = 0;
        if (distIsFraction) {
          frac = dist;
        } else {
          MVector3 vertDist;
          float length;

          vertDist = mesh->getVertexPosition(vert1);
          vertDist -= mesh->getVertexPosition(vert2);

          length = vertDist.length();
          if (dist < length) {
            frac = dist/length;
          }
        }

        if (frac > 0) {
          if (doEdge[e] == 1) {
            mesh->divideEdge(t * 3 + e, frac);
          } else {
            mesh->divideEdge(t * 3 + e, 1.0 - frac);
          }
        }

        mesh->unsetVertexFlag(vert1, TEMP_MARK_TODO);
        mesh->unsetVertexFlag(vert2, TEMP_MARK_TODO);
        mesh->setVertexFlag(vert1, TEMP_MARK_DONE);
        mesh->setVertexFlag(vert2, TEMP_MARK_DONE);
      }

    }

  }
}

int MExtrudeToolType::onMouseDown(int X, int Y, const MShiftState &Shift) {
  MSceneObjectPtr sceneObj;
  MEditableMeshPtr MeshObj;
  
  sceneObj = AZTEC_CAST(MSceneObject, g_Scene->getSelectedObjectList()->getHead());

  if (sceneObj == NULL) {
    g_SysMan->logOutput("Error: Face Extude requires one scene object to work with.");
    return TOOLRESULT_DRAWNONE;
  }
  MeshObj = AZTEC_CAST(MEditableMesh, sceneObj->getShapeObject()->convertToMesh());
  if (MeshObj == NULL) {
    g_SysMan->logOutput("Error: Face Extude requires Mesh objects to work with.");
    return TOOLRESULT_DRAWNONE;
  }

  if (!updateExtrudeNormal()) {
    m_Extruding = false;
    g_SysMan->logOutput("Error: Face Extrude requires some faces selected.");
    
    return TOOLRESULT_DRAWNONE;
  }

  m_Extruding = true;
  
  {
    COpenGLWndPtr glWnd;
    
    MSceneObjectPtr sceneObj;
    MEditableMeshPtr MeshObj;
    
    sceneObj = AZTEC_CAST(MSceneObject, g_Scene->getSelectedObjectList()->getHead());

    if (sceneObj == NULL) {
      g_SysMan->logOutput("Error: Face Extude requires one scene object to work with.");
      return TOOLRESULT_DRAWNONE;
    }
    MeshObj = AZTEC_CAST(MEditableMesh, sceneObj->getShapeObject()->convertToMesh());
    if (MeshObj == NULL) {
      g_SysMan->logOutput("Error: Face Extude requires Mesh objects to work with.");
      return TOOLRESULT_DRAWNONE;
    }
    
    glWnd = AZTEC_CAST(COpenGLWnd, g_CurView);
    
    if (glWnd == NULL) {
      m_StartDist = 0.0;
    } else {
      MRay viewRay, normalRay;
      MVector3  e1,e2;

      e1 = g_Scene->objectToWorldSpace(MBaseObjectPtr(MeshObj), m_ExtrudeCentre);
      e2 = g_Scene->objectToWorldSpace(MBaseObjectPtr(MeshObj), m_ExtrudeCentre + m_ExtrudeVec);

      normalRay.set(e1, e2-e1);
      viewRay = glWnd->GetViewRay(X, Y);

      MMath::distanceBetween(normalRay, viewRay, &m_StartDist, NULL);
    }
  }
  

  m_Dist = m_StartDist;

  // if we are in point mode, we get split the dges around the selected points
  // and move the one point.
  if (MUIManager::getComponentMode() == MUIManager::POINT_MODE) {
    // TODO: Make the radius of extrusion customisable.
    splitEdgesAroundVerts(MeshObj, VERTEX_SELECTED, 2, false);

    // now go over all the selected verts and mark them for change so we
    // move them with the tool
    for (int i = MeshObj->getNumVerts() - 1; i >= 0 ; --i) {
      if (MeshObj->isVertexFlagged(i, VERTEX_SELECTED)) {
        MeshObj->setVertexFlag(i, VERTEX_FLAGFORCHANGE);
      } else {
        MeshObj->unsetVertexFlag(i, VERTEX_FLAGFORCHANGE);
      }
    }

  // extrude the faces if that is the mode we are in
  } else if (MUIManager::getComponentMode() == MUIManager::FACET_MODE) {
    // Extrude the faces initially by a small amount.
    MeshObj->extrudeFaces(TRIANGLE_SELECTED, 0.1f, m_ExtrudeVec);
  } else if (MUIManager::getComponentMode() == MUIManager::EDGE_MODE) {
    MeshObj->unsetVertexFlags(VERTEX_FLAGFORCHANGE | VERTEX_SELECTED);

    for (int i = MeshObj->getNumTris() - 1; i >= 0 ; --i) {
      for (int e = 0; e < 3; ++e) {
        if (MeshObj->isTriangleEdgeFlagged(i, e, EDGE_SELECTED)) {
          MeshObj->setVertexFlag(MeshObj->getTriangleVertex(i, e), VERTEX_FLAGFORCHANGE | VERTEX_SELECTED);
          MeshObj->setVertexFlag(MeshObj->getTriangleVertex(i, (e+1)%3), VERTEX_FLAGFORCHANGE | VERTEX_SELECTED);
        }
      }
    }
    // TODO: Make the radius of extrusion customisable.
    // DOESN'T WORK WITH VEREX_FLAGFORCHANGE!!!! WHY NOT
    splitEdgesAroundEdges(MeshObj, VERTEX_SELECTED, 2, false);
    MeshObj->unsetVertexFlags(VERTEX_FLAGFORCHANGE);
    for (int i = MeshObj->getNumTris() - 1; i >= 0 ; --i) {
      for (int e = 0; e < 3; ++e) {
        if (MeshObj->isTriangleEdgeFlagged(i, e, EDGE_SELECTED)) {
          MeshObj->setVertexFlag(MeshObj->getTriangleVertex(i, e), VERTEX_FLAGFORCHANGE);
          MeshObj->setVertexFlag(MeshObj->getTriangleVertex(i, (e+1)%3), VERTEX_FLAGFORCHANGE);
        }
      }
    }
  }
  
  // Take a snapshot, so we can alter the distance of the vertices later on nice and easily.
  MeshObj->takeSnapshot();
  
  return TOOLRESULT_DRAWALL;
}


int MExtrudeToolType::onMouseMove(int X, int Y, const MShiftState &Shift) {
  // if we are extruding, keep track of how the mouse has moved etc.
  
  if (m_Extruding) {
    COpenGLWndPtr glWnd;
    MSceneObjectPtr sceneObj;
    MEditableMeshPtr MeshObj;
    
    sceneObj = AZTEC_CAST(MSceneObject, g_Scene->getSelectedObjectList()->getHead());

    if (sceneObj == NULL) {
      g_SysMan->logOutput("Error: Face Extude requires one scene object to work with.");
      return TOOLRESULT_DRAWNONE;
    }
    MeshObj = AZTEC_CAST(MEditableMesh, sceneObj->getShapeObject()->convertToMesh());
    if (MeshObj == NULL) {
      g_SysMan->logOutput("Error: Face Extude requires Mesh objects to work with.");
      return TOOLRESULT_DRAWNONE;
    }
    
    glWnd = AZTEC_CAST(COpenGLWnd, g_CurView);
    
    if (glWnd == NULL) {
      m_Dist = (float)(m_CurPos.x - m_DownPos.x) / 25.0f;
    } else {
      MRay viewRay, normalRay;
      MVector3  e1,e2;

      e1 = g_Scene->objectToWorldSpace(MBaseObjectPtr(MeshObj), m_ExtrudeCentre);
      e2 = g_Scene->objectToWorldSpace(MBaseObjectPtr(MeshObj), m_ExtrudeCentre + m_ExtrudeVec);

      normalRay.set(e1, e2-e1);
      viewRay = glWnd->GetViewRay(X, Y);

      MMath::distanceBetween(normalRay, viewRay, &m_Dist, NULL);
    }

    m_Dist -= m_StartDist;
    m_CurrentPos = m_StartPos + m_Dist * m_ExtrudeVec;


    MeshObj->retreiveSnapshot();
    
    // Move the marked vertices by the specified amount.
    
    for (int n=0; n<MeshObj->getNumVerts(); n++) {
      if (MeshObj->isVertexFlagged(n, VERTEX_FLAGFORCHANGE)) {
        MeshObj->setVertexPosition(n, MeshObj->getVertexPosition(n) + m_Dist * m_ExtrudeVec);
      }
    }

    MeshObj->calculateNormals();
    
    return TOOLRESULT_DRAWALL;
  }
  
  return TOOLRESULT_DRAWNONE;
}

int MExtrudeToolType::onMouseUp(int X, int Y, const MShiftState &Shift) {
  if (m_Extruding) {
    // If the extrusion didn't aceheive anything, just undo it.
    if (fabs(m_Dist) < 0.01) {
      Aztec::getSystemManager()->getUndoManager()->undo();
    } else {
      MBaseObjectPtr BaseObj;
      MSceneObjectPtr Obj;
  
      g_Scene->getObjectList()->beginIteration();
      while (( BaseObj = g_Scene->getObjectList()->getNext() ) != NULL ) {
        Obj = AZTEC_CAST(MSceneObject, BaseObj);
        if (Obj == NULL) {
          continue;
        }
    
        MTreeObjectNodePtr ObjNode;
        MComponentisedObjectPtr compObj = Obj->getComponentObject();
    
        ObjNode = g_Scene->getObjectList()->getCurrentNode();
    
        if (compObj != NULL && compObj->isInComponentMode()) {
          MAnimMeshPtr AnimMesh;
          AnimMesh = AZTEC_CAST(MAnimMesh, compObj);
          if (AnimMesh != NULL) {
            AnimMesh->updateKey(g_Scene->getTime(), g_SysMan->getSettings()->m_Animate);
          }
        }
      }
      g_Scene->getObjectList()->endIteration();
    }
    
    m_Extruding = false;
    updateExtrudeNormal();
  }
  
  return TOOLRESULT_DRAWALL;
}


