// GraphViewWnd.cpp : implementation file
//

#include <AztecMainPCH.h>

#include <views/GraphViewWnd.h>
#include <controls/GraphComponent.h>
#include <tools/MGraphSelectTool.h>

#include <main/MdlGlobs.h>

#include <math.h>

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


// CGraphComponent

BEGIN_MESSAGE_MAP(CGraphComponent, CWnd)
//{{AFX_MSG_MAP(COpenGLWnd)
ON_WM_SIZE()
ON_WM_DESTROY()
ON_WM_PAINT()
ON_WM_KEYDOWN()
ON_WM_CLOSE()
ON_WM_HSCROLL()
ON_WM_VSCROLL()
ON_WM_MOUSEWHEEL()
ON_WM_CREATE()
//}}AFX_MSG_MAP
END_MESSAGE_MAP()

CGraphComponent::CGraphComponent(MBaseViewWnd *parentView) {
  m_Dragging = false;
  minTick = g_Scene->getStartTime();
  maxTick = g_Scene->getEndTime();
  minValue = -100;
  maxValue = 100;
  this->parentView = parentView;
}

CGraphComponent::~CGraphComponent()
{

}

void CGraphComponent::renderXAxis() const {
  // TODO: thid should be changed so it can handle things other than
  // time as an input. There is no rush for this, as this will only be necessary
  // when KeyList's can have things other than time as an input.

  // draw the current time.
  glColor4f(0.1f, 0.4f, 0.9f,0.5f);
  glBegin(GL_QUADS);
  glVertex2f(g_Scene->getTime(), maxValue);
  glVertex2f(g_Scene->getTime() + g_Scene->getTicksPerFrame(), maxValue);
  glVertex2f(g_Scene->getTime() + g_Scene->getTicksPerFrame(), minValue);
  glVertex2f(g_Scene->getTime(), minValue);
  glEnd();

  // Draw the guides
  glBegin(GL_LINES);

  glColor3f(0,0,0);
  glVertex2f(minTick,maxValue);
  glVertex2f(minTick,minValue);

  glVertex2f(minTick,0);
  glVertex2f(maxTick,0);

  glEnd();


  // Draw the Timelines Text
  long frameMin, frameMax;

  frameMin = g_Scene->tickToFrame(minTick);
  frameMax = g_Scene->tickToFrame(maxTick);

  int frameStep = 1;
  {
    bool collision = true;
    HDC dc = ::GetDC(m_hWnd);

    while (collision) {
      RECT lastTextRect, textRect;
      SIZE size;

      lastTextRect.left = -5000;
      lastTextRect.right = -5000;
      lastTextRect.top = -10;
      lastTextRect.bottom = -10;

      collision = false;
      if (frameMin + frameStep <= frameMax) {
        for (int frame = frameMin; frame <= frameMax; frame += frameStep) {
          int tick = g_Scene->frameToTick(frame);
          float xPos, yPos;
          clientToGraph(0,15, xPos, yPos);

          MStr str;
          str.Format("%i", frame);
          ::GetTextExtentPoint32(dc, (LPCTSTR)str, str.GetLength(), &size);

          graphToClient(tick, yPos, textRect.left, textRect.bottom);

          textRect.right = textRect.left + size.cx + 3;
          textRect.top = textRect.bottom - size.cy;

          if (textRect.left < lastTextRect.right) {
            collision = true;
            break;
          }

          lastTextRect = textRect;
        }
      }

      if (collision) {
        if (frameStep == 1) {
          frameStep = 2;
        } else if (frameStep == 2) {
          frameStep = 5;
        } else if (frameStep < 30) {
          frameStep += 5;
        } else if (frameStep < 50) {
          frameStep += 20;
        } else {
          frameStep += 50;
        }
      }
    }

    ::ReleaseDC(m_hWnd, dc);
  }

  glLineWidth(1);

  // now we adjust our frame min to ensure that the frameStep will
  // fall onto the 0 mark.
  if (frameMin < 0) {
    int diff = (frameStep-frameMin) % frameStep;
    frameMin -= (frameStep - diff);
  }


  // now we render our actual time line.
  for (int frame = frameMin; frame <= frameMax; frame += frameStep) {
    int tick = g_Scene->frameToTick(frame);

    glColor3f(0.3f,0.3f,0.3f);
    glBegin(GL_LINES);
    glVertex2f(tick, maxValue);
    glVertex2f(tick, minValue);
    glEnd();

    float fx, fy;
    int ix, iy;
    clientToGraph(0,15, fx, fy);
    graphToClient(tick, fy, ix, iy);
    clientToGraph(ix + 2, iy , fx, fy);

    MStr str;
    str.Format("%i", frame);

    glColor3f(1,1,1);
    GLDrawText(fx, fy, str);
    
  }
}

void CGraphComponent::getKeysInRect(const RECT &rect, std::vector<ListKey> &keys) {
  keys.clear();
  
  float minValue, maxValue;
  float minTime, maxTime;

  clientToGraph(rect.left, rect.top, minTime, minValue);
  clientToGraph(rect.right, rect.bottom, maxTime, maxValue);

  if (minTime > maxTime) {
    std::swap(minTime, maxTime);
  }
  if (minValue > maxValue) {
    std::swap(minValue, maxValue);
  }

  // go through out current value lits, and try to see if any keys are underneath it.
  for (int i = 0; i < graphValues.size(); ++i) {
    MFloatKeyListPtr keyList = AZTEC_CAST(MFloatKeyList, graphValues[i].value);
    
    if (keyList != NULL) {
      // loop over the keys
      for (int keyIndex = 0; keyIndex < keyList->getKeyCount(); ++keyIndex) {
        float value;
        long time;

        time = keyList->getTimeAtIndex(keyIndex);
        value = keyList->getValueAtIndex(keyIndex);

        if (time >= minTime && time <= maxTime && 
            value >= minValue && value <= maxValue) {

          keys.push_back(std::make_pair(keyList, keyIndex));
        }
      }
    }
  }
}

void CGraphComponent::getKeysAtPoint(int x, int y, std::vector<ListKey> &keys, int radius) {
  RECT rect;

  rect.left = x - radius;
  rect.right = x + radius;
  rect.top = y - radius;
  rect.bottom = y + radius;

  getKeysInRect(rect, keys);
}

MFloatKeyListPtr CGraphComponent::getKeyListAtMouse(int mouseX, int mouseY) {
  for (int i = 0; i < graphValues.size(); ++i) {
    MFloatKeyListPtr keyList = AZTEC_CAST(MFloatKeyList, graphValues[i].value);

    if (keyList == NULL) {
      continue;
    }

    float x1,y1;
    float minY,maxY;

    clientToGraph(mouseX, mouseY+3, x1, minY);
    clientToGraph(mouseX, mouseY-3, x1, maxY);

    y1 = keyList->getValueAtTime(x1);

    // if we have a value that falls within our range, this is our baby.
    if (y1 >= minY && y1 <= maxY) {
      return keyList;
    }
  }

  return NULL;
}

void CGraphComponent::getGraphRange(float &min, float &max) {
  min = minTick;
  max = maxTick;
}

void CGraphComponent::setGraphRange(float min, float max) {

  // only bother zooming in if it will make a difference
  if (max - min > g_Scene->getTicksPerFrame()) {
    minTick = min;
    maxTick = max;
  }
}

void CGraphComponent::getValueRange(float &min, float &max) {
  min = minValue;
  max = maxValue;
}

void CGraphComponent::setValueRange(float min, float max) {
  minValue = min;
  maxValue = max;
}

void CGraphComponent::initialiseRendering() {
  makeCurrent();
  
  glEnable(GL_POINT_SMOOTH);
  glDisable(GL_LINE_SMOOTH);

  glDisable(GL_DEPTH_TEST);
  glDisable(GL_CULL_FACE);
  
  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glPointSize(4);
  
  // set up the drawing modes
  
  glLineWidth(2);
  
  RECT clientRect;
  ::GetClientRect(m_hWnd, &clientRect);

  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  glViewport(0,0,clientRect.right, clientRect.bottom);
  glTranslatef(-1.0f,0.0f,0.0f);
  glScalef(2.0/(maxTick-minTick), 2.0/(maxValue - minValue), 1);
  glTranslatef(-minTick,-(minValue + maxValue)/2.0,0);
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
  
  glClearColor(0.4f, 0.4f, 0.4f, 0.0f);
  glClear(GL_COLOR_BUFFER_BIT);
}
//  minTick / maxTick - minTick
void CGraphComponent::DrawView() {
  CRect clientRect;
  GetClientRect(&clientRect);
  if (clientRect.Width() <= 1 || clientRect.Height() <= 1) {
    return;
  }

  initialiseRendering();

  renderXAxis();
  renderYAxis();

  {
    MBaseObjectTreePtr Tree = g_Scene->getObjectList();
    MBaseObjectPtr BaseObj;
    MSceneObjectPtr Obj;
    RECT clientRect;
    
    ::GetClientRect(m_hWnd, &clientRect);

    for (int i = 0; i < graphValues.size(); ++i) {

      MFloatValuePtr value = graphValues[i].value;
      MVector3 colour = graphValues[i].colour;

      float timeStep;
      timeStep = (float)(maxTick - minTick) / (clientRect.right/2);

      glColor3f(colour.x, colour.y, colour.z);

      glBegin(GL_LINE_STRIP);
      float pos;
      for (float time = minTick; time < maxTick; time += timeStep) {
        pos = value->getValueAtTime(time);
        glVertex2f(time, pos);
      }
      glEnd();

      // Draw the control points on each spline.
      glPointSize(8);

      glBegin(GL_POINTS);
      MFloatKeyListPtr keyList = AZTEC_CAST(MFloatKeyList, value);

      if (keyList != NULL) {
        for (int listIndex = 0; listIndex < 3; ++listIndex) {
          for (int index = 0; index < keyList->getKeyCount(); ++index) {
            float pos;
            pos = keyList->getValueAtIndex(index);
            MKeyPtr key = keyList->getKeyAtIndex(index);

            if (key->isSelected()) {
              glColor3f(0.9f,0.9f,0.9f);
            } else {
              glColor3f(0.7f,0.7f,0.7f);
            }

            glVertex2f(key->getKeyTime(), pos);
          }
        }
        glEnd();
      
        glPointSize(4);
        glBegin(GL_POINTS);
        for (int listIndex = 0; listIndex < 3; ++listIndex) {
          for (int index = 0; index < keyList->getKeyCount(); ++index) {
            float pos;
            pos = keyList->getValueAtIndex(index);
            MKeyPtr key = keyList->getKeyAtIndex(index);

            if (key->isSelected()) {
              glColor3f(1,1,0);
            } else {
              glColor3f(0.2f,0.2f,0.2f);
            }

            glVertex2f(key->getKeyTime(), pos);
          }
        }
        glEnd();
      }

    }
  }

  MToolTypePtr tool = g_ToolMan.GetTool("graphView");

  if (tool != NULL) {
    tool->DrawTool(false, shiftState, parentView);
  }

  glFlush();
  glFinish();
  
  swapBuffers();

}

// MBaseViewWnd methods
void CGraphComponent::onMouseDown(int mouseX, int mouseY, MShiftState Shift) {
  MSystemManager::getInstance()->getUndoManager()->beginUndo("Graph Edit");
  shiftState = Shift;
  MToolTypePtr tool = g_ToolMan.GetTool("graphView");

  if (tool != NULL) {
    if (tool->RequiresSelection()) {
      bool doSelection = false;

      if (isAnythingSelected()) {
        // make sure that we have a key underneath us
        std::vector<ListKey> keys;
        getKeysAtPoint(mouseX, mouseY, keys);

        // set it up so we assume there are no selected keys underneath our cursor
        doSelection = true;

        // now loop over the keys under our mouse, and see if any are selected.
        // If one or more are selected, then we don't need to switch to the
        // selection tool, so we break out of the loop.
        for (int i = 0; i < keys.size(); ++i) {
          if (keys[i].first->getKeyAtIndex(keys[i].second)->isSelected()) {
            doSelection = false;
            break;
          }
        }
      } else {
        doSelection = true;
      }

      if (doSelection) {
        g_ToolMan.setTemporary(getSelectTool(), "graphView");
        tool = g_ToolMan.GetTool("graphView");
      }
    }

    parentView->handleToolResult(tool->onMouseDown(mouseX, mouseY, Shift));
  }
}

void CGraphComponent::onMouseUp(int mouseX, int mouseY, MShiftState Shift) {
  shiftState = Shift;
  MToolTypePtr tool = g_ToolMan.GetTool("graphView");

  if (tool != NULL) {
    parentView->handleToolResult(tool->onMouseUp(mouseX, mouseY, Shift));

    g_ToolMan.UnsetTemporary("graphView");
  }
  MSystemManager::getInstance()->getUndoManager()->endUndo();
}

void CGraphComponent::onMouseMove(int mouseX, int mouseY, MShiftState Shift) {
  shiftState = Shift;
  ::SetCursor(LoadCursor(NULL, IDC_ARROW));
  MToolTypePtr tool = g_ToolMan.GetTool("graphView");

  if (tool != NULL) {
    parentView->handleToolResult(tool->onMouseMove(mouseX, mouseY, Shift));
  }
}

MToolTypePtr CGraphComponent::getSelectTool() {
  return new MGraphSelectTool();
}

bool CGraphComponent::isAnythingSelected() {
  // go through out current value lits, and try to see if any keys are underneath it.
  for (int i = 0; i < graphValues.size(); ++i) {
    MFloatKeyListPtr keyList = AZTEC_CAST(MFloatKeyList, graphValues[i].value);
    
    if (keyList != NULL) {
      // loop over the keys
      for (int keyIndex = 0; keyIndex < keyList->getKeyCount(); ++keyIndex) {
        if (keyList->getKeyAtIndex(keyIndex)->isSelected()) {
          return true;
        }
      }
    }
  }
  return false;
}

void CGraphComponent::setGraphValues(const std::vector<GraphPair> &values) {
  graphValues = values;
}

void CGraphComponent::getGraphValues(std::vector<GraphPair> &values) {
  values = graphValues;
}

bool CGraphComponent::isCurrentView() {
  CGraphViewWnd *parent = dynamic_cast<CGraphViewWnd*>(GetParent());

  if (parent != NULL) {
    return parent->isCurrentView();
  }

  return false;
}

void CGraphComponent::setAsCurrentView() {
  CGraphViewWnd *parent = dynamic_cast<CGraphViewWnd*>(GetParent());

  if (parent != NULL) {
    parent->setAsCurrentView();
  }
}

DWORD CGraphComponent::ViewPopupMenu(int x, int y) {
  CGraphViewWnd *parent = dynamic_cast<CGraphViewWnd*>(GetParent());

  if (parent != NULL) {
    parent->ViewPopupMenu(x,y);
  }

  return 0;
}

int CGraphComponent::HandlePopupCommand(DWORD Cmd) {
  CGraphViewWnd *parent = dynamic_cast<CGraphViewWnd*>(GetParent());

  if (parent != NULL) {
    parent->HandlePopupCommand(Cmd);
  }

  return 0;
}

void CGraphComponent::renderYAxis() const {
  // Draw the guides
  glBegin(GL_LINES);

  glColor3f(0,0,0);
//  glVertex2f(minTick,maxValue);
//  glVertex2f(minTick,minValue);

  glVertex2f(0, minValue);
  glVertex2f(0, maxValue);

  glEnd();

  const float stepsToUse[]  = { 0.01f , 0.05f , 0.1f  , 0.2f  , 0.5f  , 1.0f  , 2.0f  , 5.0f  , 10.0f , 20.0f , 25.0f , 50.0f };
  const char *formatToUse[] = { "%.2f", "%.2f", "%.1f", "%.1f", "%.1f", "%.0f", "%.0f", "%.0f", "%.0f", "%.0f", "%.0f", "%.0f" };
  const int MAX_NUM_STEPS = 12;
  const char *valueFormat = formatToUse[0];
  bool cantDraw = false;

  // Draw the Timelines Text
  float valueStep = stepsToUse[0];
  {

    bool collision = true;
    int collisionCount = 0;
    HDC dc = ::GetDC(m_hWnd);

    while (collision) {
      RECT lastTextRect, textRect;
      SIZE size;

      lastTextRect.left = 0;
      lastTextRect.right = 10;
      lastTextRect.top = 5000;
      lastTextRect.bottom = 5000;

      collision = false;
      if (minValue + valueStep <= maxValue) {
        for (float value = minValue; value <= maxValue; value += valueStep) {
          float xPos, yPos;
          clientToGraph(0,15, xPos, yPos);

          MStr str;
          str.Format(valueFormat, value);
          ::GetTextExtentPoint32(dc, (LPCTSTR)str, str.GetLength(), &size);

          graphToClient(xPos, value, textRect.left, textRect.bottom);

          textRect.right = textRect.left + size.cx + 3;
          textRect.top = textRect.bottom - size.cy;

          // since the Y Axis will be drawn going up the screen we have to 
          // check to see if the bottom of the new rect is greater that the
          // top of the last rect.
          if (textRect.bottom >= lastTextRect.top) {
            collision = true;
            break;
          }

          lastTextRect = textRect;
        }
      }

      if (collision) {
        ++collisionCount;
        if (collisionCount >= MAX_NUM_STEPS) {
          collisionCount = MAX_NUM_STEPS - 1;
          valueStep += stepsToUse[collisionCount];
        } else {
          valueStep = stepsToUse[collisionCount];
        }

        valueFormat = formatToUse[collisionCount];

        // if we have a collision, but our step is great than our window 
        // height, we just return out o this function, because we can't 
        // possibly draw anything when things are like that.
        if (valueStep > maxValue - minValue) {
          cantDraw = true;
          break;
        }

      }
    }

    ::ReleaseDC(m_hWnd, dc);
  }

  if (cantDraw) {
    return;
  }

  glLineWidth(1);

  float valueMinToUse = minValue;
  // now we adjust our frame min to ensure that the frameStep will
  // fall onto the 0 mark.
  if (valueMinToUse < 0) {
    float diff = fmod(valueStep - minValue, valueStep);
    valueMinToUse -= (valueStep - diff);
  }


  // now we render our actual time line.
  for (float value = valueMinToUse; value <= maxValue; value += valueStep) {
    glColor3f(0.3f,0.3f,0.3f);
    glBegin(GL_LINES);
    glVertex2f(minTick, value);
    glVertex2f(maxTick, value);
    glEnd();

    float fx, fy;
    int ix, iy;
    clientToGraph(0,15, fx, fy);
    graphToClient(fx, value, ix, iy);
    clientToGraph(ix + 2, iy , fx, fy);

    // don't draw graph values if they are near the top.
    if (iy < 35) {
      continue;
    }

    MStr str;
    str.Format(valueFormat, value);

    glColor3f(1,1,1);
    GLDrawText(fx, fy, str);
    
  }
}

void CGraphComponent::OnSize(UINT nType, int cx, int cy) 
{
  GLCanvas::OnSize(nType, cx, cy);

}


void CGraphComponent::OnPaint() 
{
  CPaintDC PaintDc(this);
  
  DrawView();
  
}

void CGraphComponent::OnDestroy() {
  CWnd::OnDestroy();
  
}

void CGraphComponent::OnHScroll( UINT nSBCode, UINT nPos, CScrollBar* pScrollBar ) {

  float diff = 0;

  switch (nSBCode) {
  case SB_LINELEFT: 
    diff = -0.1*(maxTick - minTick);
    if (diff == 0) diff = -g_Scene->getTicksPerFrame();
    break;
  case SB_LINERIGHT: 
    diff = 0.1*(maxTick - minTick);
    if (diff == 0) diff = g_Scene->getTicksPerFrame();
    break;
  case SB_PAGELEFT: 
    diff = -0.9*(maxTick - minTick);
    if (diff == 0) diff = -g_Scene->getTicksPerFrame();
    break;
  case SB_PAGERIGHT: 
    diff = 0.9*(maxTick - minTick);
    if (diff == 0) diff = g_Scene->getTicksPerFrame();
    break;
  }

  if (diff != 0) {
    minTick += diff;
    maxTick += diff;
    DrawView();
  }
}

void CGraphComponent::OnVScroll( UINT nSBCode, UINT nPos, CScrollBar* pScrollBar ) {

  float diff = 0;

  switch (nSBCode) {
  case SB_LINELEFT: 
    diff = 0.1*(maxValue - minValue);
    if (diff == 0) diff = 0.1f;
    break;
  case SB_LINERIGHT: 
    diff = -0.1*(maxValue - minValue);
    if (diff == 0) diff = -0.1f;
    break;
  case SB_PAGELEFT: 
    diff = 0.9*(maxValue - minValue);
    if (diff == 0) diff = 1.0f;
    break;
  case SB_PAGERIGHT: 
    diff = -0.9*(maxValue - minValue);
    if (diff == 0) diff = -1.0f;
    break;
  }

  if (diff != 0) {
    minValue += diff;
    maxValue += diff;
    DrawView();
  }
}

BOOL CGraphComponent::OnMouseWheel(UINT nFlags, short zDelta, CPoint pt) {
  float middle = 0.5*(maxTick + minTick);
  float diff = maxTick - minTick;
  if (zDelta < 0) {
    diff *= 0.5;

    if (diff < 2) {
      diff = 2;
    }

  } else if (zDelta > 0) {
    diff *= 2;
  }
  
  minTick = middle - diff/2;
  maxTick = middle + diff/2;
  DrawView();
  
  return CWnd::OnMouseWheel(nFlags, zDelta, pt);
}

void CGraphComponent::clientToGraph(int ix, int iy, float &fx, float &fy) const {
  float glX, glY;

  RECT clientRect;

  ::GetClientRect(m_hWnd, &clientRect);

  glX = 2.0 * (float)ix / (float)clientRect.right - 1.0;
  glY = -2.0 * (float)iy / (float)clientRect.bottom + 1.0;

  fx = (glX + 1.0) / (2.0 / (maxTick - minTick)) + minTick;
  fy = glY / (2.0 / (maxValue - minValue)) + (minValue + maxValue)/2.0;
}

void CGraphComponent::graphToClient(float fx, float fy, int &ix, int &iy) const {
  float glX, glY;

  RECT clientRect;

  ::GetClientRect(m_hWnd, &clientRect);

  glX = (fx - minTick) * (2.0 / (maxTick - minTick)) - 1.0;
  glY = (fy - (minValue + maxValue)/2.0) * (2.0 / (maxValue - minValue));

  ix = (glX + 1.0) / (2.0 / (float)clientRect.right);
  iy = (glY - 1.0) / (-2.0 / (float)clientRect.bottom);
}

void CGraphComponent::graphToClient(float fx, float fy, long &ix, long &iy) const {
  float glX, glY;

  RECT clientRect;

  ::GetClientRect(m_hWnd, &clientRect);

  glX = (fx - minTick) * (2.0 / (maxTick - minTick)) - 1.0;
  glY = (fy - (minValue + maxValue)/2.0) * (2.0 / (maxValue - minValue));

  ix = (glX + 1.0) / (2.0 / (float)clientRect.right);
  iy = (glY - 1.0) / (-2.0 / (float)clientRect.bottom);
}

int CGraphComponent::OnCreate(LPCREATESTRUCT lpCreateStruct) 
{
  if (GLCanvas::OnCreate(lpCreateStruct) == -1) {
    return -1;
  }

  SCROLLINFO scrollInfo;

  scrollInfo.cbSize = sizeof(SCROLLINFO);
  scrollInfo.fMask = SIF_ALL;

  GetScrollInfo(SB_HORZ, &scrollInfo);

  scrollInfo.nMin = 0;
  scrollInfo.nMax = 100;
  scrollInfo.nPos = 50;

  SetScrollInfo(SB_HORZ, &scrollInfo);


  scrollInfo.cbSize = sizeof(SCROLLINFO);
  scrollInfo.fMask = SIF_ALL;

  GetScrollInfo(SB_VERT, &scrollInfo);

  scrollInfo.nMin = 0;
  scrollInfo.nMax = 100;
  scrollInfo.nPos = 50;

  SetScrollInfo(SB_VERT, &scrollInfo);
  

  return 0;
}
