#include "StdAfx.h"

#include <math.h>
#include "MMesh.h"
#include "scripting/ScriptVertex.h"
#include "scripting/ScriptVector3.h"
#include "scripting/ScriptRGBAFloat.h"
#include "scripting/ScriptFuncs.h"

#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

namespace Aztec {  

static JSBool
set(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	MVertex *vec = (MVertex *)JS_GetInstancePrivate(cx, obj, &aztecVertex_class, NULL);
	if (vec == NULL || argc < 1 || argc > 3) {
		return JS_FALSE;
	}

	if (argc == 1) {
		// Either an array of doubles or another Vertex
		if (JSVAL_IS_OBJECT(*argv)) {

			// First see if it is another Vertex
			JSObject *obj2 = JSVAL_TO_OBJECT(*argv);
			MVertex *vec2 = (MVertex *)JS_GetInstancePrivate(cx, obj2, &aztecVertex_class, NULL);
			if (vec2 != NULL) {
				*vec = *vec2;
				*rval = JSVAL_VOID;
				return JS_TRUE;
			}

			// See if the argument is an array of doubles
			jsuint lengthp;
			if (JS_IsArrayObject(cx, obj2) == JS_TRUE &&
				JS_GetArrayLength(cx, obj2, &lengthp) == JS_TRUE &&
				lengthp >= 3) {
				jsval v1, v2, v3;
				if (JS_GetElement(cx, obj2, 0, &v1) == JS_TRUE &&
					JS_GetElement(cx, obj2, 1, &v2) == JS_TRUE &&
					JS_GetElement(cx, obj2, 2, &v3) == JS_TRUE) {
					if (JSVAL_IS_NUMBER(v1) && JSVAL_IS_NUMBER(v2) &&
						JSVAL_IS_NUMBER(v3)) {
						double x, y, z;
						if (getDoubleVal(cx, v1, x) &&
							getDoubleVal(cx, v2, y) &&
							getDoubleVal(cx, v3, z)) {
							vec->setPos(MVector3((float)x, (float)y, (float)z));
							*rval = JSVAL_VOID;
							return JS_TRUE;
						}
					}

				}
			}
		}
	} else if (argc == 3) {
		// If there are three arguments and all three are doubles, then
		// assign them to the components of the vector.
		if (JSVAL_IS_NUMBER(*argv) && JSVAL_IS_NUMBER(*(argv+1)) &&
			JSVAL_IS_NUMBER(*(argv+2))) {
			double x, y, z;
			if (getDoubleVal(cx, *argv, x) &&
				getDoubleVal(cx, *(argv+1), y) &&
				getDoubleVal(cx, *(argv+2), z)) {
				vec->setPos(MVector3((float)x, (float)y, (float)z));
				*rval = JSVAL_VOID;
				return JS_TRUE;
			}
		}
	}

	// Don't know how to convert the arguments into the components of
	// a vector.
	return JS_FALSE;
}

static JSPropertySpec aztecVertex_props[] = {
	{"x",				0,	JSPROP_ENUMERATE},
	{"y",				1,	JSPROP_ENUMERATE},
	{"z",				2,	JSPROP_ENUMERATE},
	{"normal",			3,	JSPROP_ENUMERATE},
	{"feedbackColor",	4,	JSPROP_ENUMERATE},
	{0}
};

static JSFunctionSpec aztecVertex_methods[] = {
	{"set",				set,		1},
	{0}
};

static JSBool
aztecVertex_getProperty(JSContext *cx, JSObject *obj, jsval id, jsval *vp)
{
	MVertex *vec = (MVertex *)JS_GetInstancePrivate(cx, obj, &aztecVertex_class, NULL);
	if (vec == NULL) {
		return JS_FALSE;
	}

	if (JSVAL_IS_STRING(id)) {
		char *propStr = JS_GetStringBytes(JS_ValueToString(cx, id));
		if (strcmp(propStr, "x") == 0) {
			return JS_NewDoubleValue(cx, vec->getX(), vp);
		} else if (strcmp(propStr, "y") == 0) {
			return JS_NewDoubleValue(cx, vec->getY(), vp);
		} else if (strcmp(propStr, "z") == 0) {
			return JS_NewDoubleValue(cx, vec->getZ(), vp);
		} else if (strcmp(propStr, "normal") == 0) {
			// return normal member
			JSObject *jsnorm = newVector3(cx);
			MVector3 *nvec = (MVector3 *)JS_GetPrivate(cx, jsnorm);
			*nvec = vec->getNormal();

			*vp = OBJECT_TO_JSVAL(jsnorm);
			return JS_TRUE;
		} else if (strcmp(propStr, "feedbackColor") == 0) {
			// return feedback color member
			JSObject *jsfcolor= newRGBAFloat(cx);
			MRGBAFloat *fcolor = (MRGBAFloat *)JS_GetPrivate(cx, jsfcolor);
			*fcolor = vec->getFeedbackColour();

			*vp = OBJECT_TO_JSVAL(jsfcolor);
			return JS_TRUE;
		} else {
			return JS_TRUE;
		}
	} else if (JSVAL_IS_INT(id)) {
		int index = JSVAL_TO_INT(id);
		if (index == 0) {
			return JS_NewDoubleValue(cx, vec->getX(), vp);
		} else if (index == 1) {
			return JS_NewDoubleValue(cx, vec->getY(), vp);
		} else if (index == 2) {
			return JS_NewDoubleValue(cx, vec->getZ(), vp);
		} else if (index == 3) {
			// return normal member
			JSObject *jsnorm = newVector3(cx);
			MVector3 *nvec = (MVector3 *)JS_GetPrivate(cx, jsnorm);
			*nvec = vec->getNormal();

			*vp = OBJECT_TO_JSVAL(jsnorm);
			return JS_TRUE;
		} else if (index == 4) {
			// return feedback color member
			JSObject *jsfcolor= newRGBAFloat(cx);
			MRGBAFloat *fcolor = (MRGBAFloat *)JS_GetPrivate(cx, jsfcolor);
			*fcolor = vec->getFeedbackColour();

			*vp = OBJECT_TO_JSVAL(jsfcolor);
			return JS_TRUE;
		} else {
			return JS_TRUE;
		}
	} else {
		return JS_FALSE;
	}

	return JS_FALSE;
}

static JSBool
aztecVertex_setProperty(JSContext *cx, JSObject *obj, jsval id, jsval *vp)
{
	MVertex *vec = (MVertex *)JS_GetInstancePrivate(cx, obj, &aztecVertex_class, NULL);
	if (vec == NULL) {
		return JS_FALSE;
	}

	double t;
	if (JSVAL_IS_STRING(id)) {
		char *propStr = JS_GetStringBytes(JS_ValueToString(cx, id));
    MVector3 pos = vec->getPos();
		if (strcmp(propStr, "x") == 0) {
			if (getDoubleVal(cx, *vp, t)) {
        pos.x = (float)t;
        vec->setPos(pos);
				return JS_TRUE;
			}
		} else if (strcmp(propStr, "y") == 0) {
			if (getDoubleVal(cx, *vp, t)) {
        pos.y = (float)t;
        vec->setPos(pos);
				return JS_TRUE;
			}
		} else if (strcmp(propStr, "z") == 0) {
			if (getDoubleVal(cx, *vp, t)) {
        pos.z = (float)t;
        vec->setPos(pos);
				return JS_TRUE;
			}
		} else if (strcmp(propStr, "normal") == 0) {
			if (JSVAL_IS_OBJECT(*vp)) {
				// This is an object param, so it is ok to hand it an object.
				JSObject *jsobj = JSVAL_TO_OBJECT(*vp);
				void *priv = JS_GetInstancePrivate(cx, jsobj, &aztecVector3_class, NULL);
				if (priv != NULL) {
					MVector3 *nvec = (MVector3 *)priv;
					vec->setNormal(*nvec);
					return JS_TRUE;
				}
			}
		} else if (strcmp(propStr, "feedbackColor") == 0) {
			if (JSVAL_IS_OBJECT(*vp)) {
				// This is an object param, so it is ok to hand it an object.
				JSObject *jsobj = JSVAL_TO_OBJECT(*vp);
				void *priv = JS_GetInstancePrivate(cx, jsobj, &aztecRGBAFloat_class, NULL);
				if (priv != NULL) {
					MRGBAFloat *nvec = (MRGBAFloat *)priv;
					vec->setFeedbackColour(*nvec);
					return JS_TRUE;
				}
			}
		} else {
			return JS_FALSE;
		}
	} else if (JSVAL_IS_INT(id)) {
		int index = JSVAL_TO_INT(id);
    MVector3 pos = vec->getPos();
		if (index == 0) {
			if (getDoubleVal(cx, *vp, t)) {
        pos.x = (float)t;
        vec->setPos(pos);
				return JS_TRUE;
			}
		} else if (index == 1) {
			if (getDoubleVal(cx, *vp, t)) {
        pos.y = (float)t;
        vec->setPos(pos);
				return JS_TRUE;
			}
		} else if (index == 2) {
			if (getDoubleVal(cx, *vp, t)) {
        pos.z = (float)t;
        vec->setPos(pos);
				return JS_TRUE;
			}
		} else if (index == 3) {
			if (JSVAL_IS_OBJECT(*vp)) {
				JSObject *jsobj = JSVAL_TO_OBJECT(*vp);
				void *priv = JS_GetInstancePrivate(cx, jsobj, &aztecVector3_class, NULL);
				if (priv != NULL) {
					MVector3 *nvec = (MVector3 *)priv;
					vec->setNormal(*nvec);
					return JS_TRUE;
				}
			}
		} else if (index == 4) {
			if (JSVAL_IS_OBJECT(*vp)) {
				JSObject *jsobj = JSVAL_TO_OBJECT(*vp);
				void *priv = JS_GetInstancePrivate(cx, jsobj, &aztecRGBAFloat_class, NULL);
				if (priv != NULL) {
					MRGBAFloat *nvec = (MRGBAFloat *)priv;
					vec->setFeedbackColour(*nvec);
					return JS_TRUE;
				}
			}
		} else {
			return JS_FALSE;
		}
	} else {
		return JS_FALSE;
	}

	return JS_FALSE;
}

static JSBool
aztecVertex_convert(JSContext *cx, JSObject *obj, JSType typ, jsval *vp)
{
	MVertex *vec = (MVertex *)JS_GetInstancePrivate(cx, obj, &aztecVertex_class, NULL);
	if (vec == NULL) {
		return JS_FALSE;
	}
	
    switch (typ) {
    case JSTYPE_OBJECT:
        *vp = OBJECT_TO_JSVAL(obj);
        return JS_TRUE;
		
    case JSTYPE_VOID:
    case JSTYPE_STRING:
		{
			char buf[128];
			sprintf(buf, "%f %f %f", vec->getX(), vec->getY(), vec->getZ());
			JSString *str = JS_NewStringCopyZ(cx, buf);
			*vp = STRING_TO_JSVAL(str);
			return JS_TRUE;
		}
		
	case JSTYPE_FUNCTION:
	case JSTYPE_NUMBER:
    case JSTYPE_BOOLEAN:
    default:
        return JS_FALSE;
    }
}

static void
aztecVertex_finalize(JSContext *cx, JSObject *obj)
{
	MVertex *vec = (MVertex *)JS_GetInstancePrivate(cx, obj, &aztecVertex_class, NULL);
	if (vec != NULL) {
		delete vec;
	}
}

JSClass aztecVertex_class = { 
	"AztecVertex", JSCLASS_HAS_PRIVATE, 
		JS_PropertyStub, JS_PropertyStub,					// add/del property
		aztecVertex_getProperty, aztecVertex_setProperty,	// get/set property
		JS_EnumerateStub, JS_ResolveStub,
		aztecVertex_convert, aztecVertex_finalize
};

static JSBool aztecVertex_constructor(JSContext *cx, JSObject *obj, uintN argc,
                             jsval *argv, jsval *rval)
{
	MVertex *vec = new MVertex();
	if (vec == NULL) {
		return NULL;
	}

	return JS_SetPrivate(cx, obj, vec);
}

JSObject *newVertex(JSContext *cx)
{
	JSObject *jsvec = JS_NewObject(cx, &aztecVertex_class, NULL, JS_GetGlobalObject(cx));

	if (jsvec == NULL) {
		return NULL;
	}

	MVertex *pvec = new MVertex();
	JS_SetPrivate(cx, jsvec, pvec);

	return jsvec;
}

void addVertexClass(JSContext *cx)
{
	// Add a Vertex class (reflection of MVertex3) to the global namespace
    JS_InitClass(cx, JS_GetGlobalObject(cx), NULL,
		&aztecVertex_class, aztecVertex_constructor, 0,
		aztecVertex_props, aztecVertex_methods, 0, 0);

}

}
