#include "StdAfx.h"

#include <math.h>
#include "scripting/ScriptVector3.h"
#include "scripting/ScriptFuncs.h"

#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

namespace Aztec {

// TBD: use this instead of a direct pointer to a MVector3.  Potential
// problems if the MVector3 we point to gets deallocated prematurely.
class V3Val {
public:
	V3Val() {
		vec = new MVector3();
		copy = false;
	}
	V3Val(MVector3 *val, bool copyFlag) {
		if (copyFlag) {
			vec = val;
			copy = true;
		} else {
			vec = new MVector3();
			copy = false;
		}
	}
	~V3Val() {
		if (!copy && vec != NULL) {
			delete vec;
		}
	}

	MVector3 *vec;
	bool copy;	// Are we just holding a reference to an existing vec?
};

static JSBool
add(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	MVector3 *vec = (MVector3 *)JS_GetInstancePrivate(cx, obj, &aztecVector3_class, NULL);
	if (vec == NULL || argc == 0 || argc > 2) {
		return JS_FALSE;
	}

	if (argc == 1 && JSVAL_IS_OBJECT(*argv)) {
		JSObject *obj2 = JSVAL_TO_OBJECT(*argv);

		MVector3 *vec1 = (MVector3 *)JS_GetInstancePrivate(cx, obj2, &aztecVector3_class, NULL);
		if (vec != NULL) {
			// Transform the vector by this matrix.  Put the result back into
			// the vector.
			MVector3 rvec = *vec + *vec1;
			*vec = rvec;
			*rval = JSVAL_VOID;
			return JS_TRUE;
		}
	}

	if (argc == 2 && JSVAL_IS_OBJECT(*argv) && JSVAL_IS_OBJECT(*(argv+1))) {
		JSObject *obj2 = JSVAL_TO_OBJECT(*argv);
		JSObject *obj3 = JSVAL_TO_OBJECT(*(argv+1));

		MVector3 *vec1 = (MVector3 *)JS_GetInstancePrivate(cx, obj2, &aztecVector3_class, NULL);
		MVector3 *vec2 = (MVector3 *)JS_GetInstancePrivate(cx, obj3, &aztecVector3_class, NULL);

		if (vec1 != NULL && vec2 != NULL) {
			// Transform the vector by this matrix.  Put the result back into
			// the vector.
			*vec = *vec1 + *vec2;
			*rval = JSVAL_VOID;
			return JS_TRUE;
		}
	}

	// Two useful possibilities: matrix4 * matrix4 and matrix4 * vector4.

	return JS_NewDoubleValue(cx, 42.0, rval);
}

static JSBool
angle(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	MVector3 *vec = (MVector3 *)JS_GetInstancePrivate(cx, obj, &aztecVector3_class, NULL);
	if (vec == NULL) {
		return JS_FALSE;
	}

	if (argc == 1 && JSVAL_IS_OBJECT(*argv)) {
		JSObject *obj2 = JSVAL_TO_OBJECT(*argv);
		MVector3 *vec2 = (MVector3 *)JS_GetInstancePrivate(cx, obj2, &aztecVector3_class, argv);
		if (vec2 == NULL) {
			return JS_FALSE;
		}

		MVector3 av, bv;
		av = *vec;
		bv = *vec2;

		av.normalize();
		bv.normalize();

		double ang = acos(av.dotProduct(bv));

		return JS_NewDoubleValue(cx, ang, rval);
	} else {
		return JS_FALSE;
	}
}

static JSBool
dot(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	MVector3 *vec = (MVector3 *)JS_GetInstancePrivate(cx, obj, &aztecVector3_class, NULL);
	if (vec == NULL) {
		return JS_FALSE;
	} else if (argc != 1) {
		// Must do dot product with another vector
		return JS_FALSE;
	}

	if (JSVAL_IS_OBJECT(*argv)) {
		JSObject *obj2 = JSVAL_TO_OBJECT(*argv);
		MVector3 *vec2 = (MVector3 *)JS_GetInstancePrivate(cx, obj2, &aztecVector3_class, argv);
		if (vec2 == NULL) {
			return JS_FALSE;
		}

		return JS_NewDoubleValue(cx, vec->dotProduct(*vec2), rval);
	} else {
		return JS_FALSE;
	}
}

static JSBool
cross(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	MVector3 *vec = (MVector3 *)JS_GetInstancePrivate(cx, obj, &aztecVector3_class, NULL);
	if (vec == NULL) {
		return JS_FALSE;
	}

	if (argc == 1 && JSVAL_IS_OBJECT(*argv)) {
		JSObject *obj2 = JSVAL_TO_OBJECT(*argv);
		MVector3 *vec2 = (MVector3 *)JS_GetInstancePrivate(cx, obj2, &aztecVector3_class, argv);
		if (vec2 == NULL) {
			return JS_FALSE;
		}

		MVector3 av, bv;
		av = *vec;
		bv = *vec2;

		av.normalize();
		bv.normalize();

		JSObject *resVec = newVector3(cx);
		MVector3 *cv = (MVector3 *)JS_GetInstancePrivate(cx, resVec, &aztecVector3_class, NULL);
		if (cv == NULL) {
			return JS_FALSE;
		}

		*cv = av.crossProduct(bv);
		*rval = OBJECT_TO_JSVAL(resVec);

		return JS_TRUE;
	} else {
		return JS_FALSE;
	}
}

static JSBool
length(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	MVector3 *vec = (MVector3 *)JS_GetInstancePrivate(cx, obj, &aztecVector3_class, NULL);
	if (vec == NULL) {
		return JS_FALSE;
	} else if (argc != 0) {
		return JS_FALSE;
	}

	return JS_NewDoubleValue(cx, vec->length(), rval);
}

static JSBool
lengthSquared(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	MVector3 *vec = (MVector3 *)JS_GetInstancePrivate(cx, obj, &aztecVector3_class, NULL);
	if (vec == NULL) {
		return JS_FALSE;
	} else if (argc != 0) {
		return JS_FALSE;
	}

	jsdouble l2 = vec->x * vec->x + vec->y * vec->y + vec->z * vec->z;

	return JS_NewDoubleValue(cx, vec->length(), rval);
}

static JSBool
mul(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	MVector3 *vec = (MVector3 *)JS_GetInstancePrivate(cx, obj, &aztecVector3_class, NULL);
	if (vec == NULL) {
		return JS_FALSE;
	} else if (argc != 1) {
		return JS_FALSE;
	}

	// Two useful possibilities: Vector3 * Vector3 and Vector3 * vector4.

	return JS_NewDoubleValue(cx, 42.0, rval);
}

static JSBool
normalize(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	MVector3 *vec = (MVector3 *)JS_GetInstancePrivate(cx, obj, &aztecVector3_class, NULL);
	if (vec == NULL) {
		return JS_FALSE;
	}

	if (argc == 0) {
		vec->normalize();
		return JS_TRUE;
	} else if (argc == 1) {
		// Set this vector to the normalized value of the argument vector
		if (JSVAL_IS_OBJECT(*argv)) {
			JSObject *obj2 = JSVAL_TO_OBJECT(*argv);
			MVector3 *vec2 = (MVector3 *)JS_GetInstancePrivate(cx, obj2, &aztecVector3_class, argv);
			if (vec2 == NULL) {
				return JS_FALSE;
			}

			*vec = *vec2;
			vec->normalize();

			*rval = JSVAL_VOID;

			return JS_TRUE;
		} else {
			return JS_FALSE;
		}
		return JS_TRUE;
	} else {
		return JS_FALSE;
	}
}

static JSBool
set(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	MVector3 *vec = (MVector3 *)JS_GetInstancePrivate(cx, obj, &aztecVector3_class, NULL);
	if (vec == NULL) {
		return JS_FALSE;
	}

	if (argc == 1) {
		// Either an array of doubles or another Vector3
		if (JSVAL_IS_OBJECT(*argv)) {

			// First see if it is another Vector3
			JSObject *obj2 = JSVAL_TO_OBJECT(*argv);
			MVector3 *vec2 = (MVector3 *)JS_GetInstancePrivate(cx, obj2, &aztecVector3_class, NULL);
			if (vec2 != NULL) {
				*vec = *vec2;
				*rval = JSVAL_VOID;
				return JS_TRUE;
			}

			// See if the argument is an array of doubles
			jsuint lengthp;
			if (JS_IsArrayObject(cx, obj2) == JS_TRUE &&
				JS_GetArrayLength(cx, obj2, &lengthp) == JS_TRUE &&
				lengthp >= 3) {
				jsval v1, v2, v3;
				if (JS_GetElement(cx, obj2, 0, &v1) == JS_TRUE &&
					JS_GetElement(cx, obj2, 1, &v2) == JS_TRUE &&
					JS_GetElement(cx, obj2, 2, &v3) == JS_TRUE) {
					if (JSVAL_IS_NUMBER(v1) && JSVAL_IS_NUMBER(v2) &&
						JSVAL_IS_NUMBER(v3)) {
						double x, y, z;
						if (getDoubleVal(cx, v1, x) &&
							getDoubleVal(cx, v2, y) &&
							getDoubleVal(cx, v3, z)) {
							vec->x = (float)x;
							vec->y = (float)y;
							vec->z = (float)z;
							*rval = JSVAL_VOID;
							return JS_TRUE;
						}
					}

				}
			}
		}
	} else if (argc == 3) {
		// If there are three arguments and all three are doubles, then
		// assign them to the components of the vector.
		if (JSVAL_IS_NUMBER(*argv) && JSVAL_IS_NUMBER(*(argv+1)) &&
			JSVAL_IS_NUMBER(*(argv+2))) {
			double x, y, z;
			if (getDoubleVal(cx, *argv, x) &&
				getDoubleVal(cx, *(argv+1), y) &&
				getDoubleVal(cx, *(argv+2), z)) {
				vec->x = (float)x;
				vec->y = (float)y;
				vec->z = (float)z;
				*rval = JSVAL_VOID;
				return JS_TRUE;
			}
		}
	}

	// Don't know how to convert the arguments into the components of
	// a vector.
	return JS_FALSE;
}

static JSPropertySpec aztecVector3_props[] = {
	{"x",	0,	JSPROP_ENUMERATE},
	{"y",	1,	JSPROP_ENUMERATE},
	{"z",	2,	JSPROP_ENUMERATE},
	{0}
};

static JSFunctionSpec aztecVector3_methods[] = {
	{"add",				add,		1},
	{"angle",			angle,		1},
	{"cross",			cross,		1},
	{"dot",				dot,		1},
	{"length",			length,		0},
	{"lengthSquared",	length,		0},
	{"mul",				mul,		0},
	{"normalize",		normalize,	0},
	{"set",				set,		1},
	{0}
};

static JSBool
aztecVector3_getProperty(JSContext *cx, JSObject *obj, jsval id, jsval *vp)
{
	MVector3 *vec = (MVector3 *)JS_GetInstancePrivate(cx, obj, &aztecVector3_class, NULL);
	if (vec == NULL) {
		return JS_FALSE;
	}

	if (JSVAL_IS_STRING(id)) {
		char *propStr = JS_GetStringBytes(JS_ValueToString(cx, id));
		if (strcmp(propStr, "x") == 0) {
			return JS_NewDoubleValue(cx, vec->x, vp);
		} else if (strcmp(propStr, "y") == 0) {
			return JS_NewDoubleValue(cx, vec->y, vp);
		} else if (strcmp(propStr, "z") == 0) {
			return JS_NewDoubleValue(cx, vec->z, vp);
		}
		
		// May be a built-in method.
		for (int m=0;aztecVector3_methods[m].name != NULL;m++) {
			if (!strcmp(propStr, aztecVector3_methods[m].name)) {
				// Yup - is a built-in function.
				JSFunction *jfunc =
					JS_NewFunction(cx, aztecVector3_methods[m].call,
									aztecVector3_methods[m].nargs,
									0, obj, propStr);
				JSObject *jfobj = JS_GetFunctionObject(jfunc);
				*vp = OBJECT_TO_JSVAL(jfobj);
				return JS_TRUE;
			}
		}

		JS_ReportError(cx, "%s is not defined", propStr);
		return JS_FALSE;

	} else if (JSVAL_IS_INT(id)) {
		int index = JSVAL_TO_INT(id);
		if (index == 0) {
			return JS_NewDoubleValue(cx, vec->x, vp);
		} else if (index == 1) {
			return JS_NewDoubleValue(cx, vec->y, vp);
		} else if (index == 2) {
			return JS_NewDoubleValue(cx, vec->z, vp);
		} else {
			return JS_TRUE;
		}
	} else {
#if 0
		char *propStr = JS_GetStringBytes(JS_ValueToString(cx, id));
		OutputDebugString("Vec prop: ");
		OutputDebugString(propStr);
		OutputDebugString("\n");
#endif
		return JS_FALSE;
	}
}

static JSBool
aztecVector3_setProperty(JSContext *cx, JSObject *obj, jsval id, jsval *vp)
{
	MVector3 *vec = (MVector3 *)JS_GetInstancePrivate(cx, obj, &aztecVector3_class, NULL);
	if (vec == NULL) {
		return JS_FALSE;
	}

	double t;
	if (JSVAL_IS_INT(*vp)) {
		t = JSVAL_TO_INT(*vp);
	} else if (JSVAL_IS_DOUBLE(*vp)) {
		t = *JSVAL_TO_DOUBLE(*vp);
	} else if (JSVAL_IS_BOOLEAN(*vp)) {
		t = (JSVAL_TO_BOOLEAN(*vp) ? 1.0 : 0.0);
	} else if (JSVAL_IS_STRING(*vp)) {
		char *valStr = JS_GetStringBytes(JS_ValueToString(cx, *vp));
		t = atof(valStr);
	} else {
		return JS_FALSE;
	}

	if (JSVAL_IS_STRING(id)) {
		char *propStr = JS_GetStringBytes(JS_ValueToString(cx, id));
		if (strcmp(propStr, "x") == 0) {
			vec->x = (float)t;
			return JS_TRUE;
		} else if (strcmp(propStr, "y") == 0) {
			vec->y = (float)t;
			return JS_TRUE;
		} else if (strcmp(propStr, "z") == 0) {
			vec->z = (float)t;
			return JS_TRUE;
		} else {
			return JS_FALSE;
		}
	} else if (JSVAL_IS_INT(id)) {
		int index = JSVAL_TO_INT(id);
		if (index == 0) {
			vec->x = (float)t;
			return JS_TRUE;
		} else if (index == 1) {
			vec->y = (float)t;
			return JS_TRUE;
		} else if (index == 2) {
			vec->z = (float)t;
			return JS_TRUE;
		} else {
			return JS_FALSE;
		}
	} else {
#if 0
		char *propStr = JS_GetStringBytes(JS_ValueToString(cx, id));
		OutputDebugString("Vec prop: ");
		OutputDebugString(propStr);
		OutputDebugString("\n");
#endif
		return JS_FALSE;
	}
}

static JSBool
aztecVector3_convert(JSContext *cx, JSObject *obj, JSType typ, jsval *vp)
{
	MVector3 *vec = (MVector3 *)JS_GetInstancePrivate(cx, obj, &aztecVector3_class, NULL);
	if (vec == NULL) {
		return JS_FALSE;
	}
	
    switch (typ) {
    case JSTYPE_OBJECT:
        *vp = OBJECT_TO_JSVAL(obj);
        return JS_TRUE;
		
    case JSTYPE_VOID:
    case JSTYPE_STRING:
		{
			char buf[128];
			sprintf(buf, "%f %f %f", vec->x, vec->y, vec->z);
			JSString *str = JS_NewStringCopyZ(cx, buf);
			*vp = STRING_TO_JSVAL(str);
			return JS_TRUE;
		}
		
	case JSTYPE_FUNCTION:
	case JSTYPE_NUMBER:
    case JSTYPE_BOOLEAN:
    default:
        return JS_FALSE;
    }
}

static void
aztecVector3_finalize(JSContext *cx, JSObject *obj)
{
	MVector3 *vec = (MVector3 *)JS_GetInstancePrivate(cx, obj, &aztecVector3_class, NULL);
	if (vec != NULL) {
		delete vec;
	}
}

JSClass aztecVector3_class = { 
	"AztecVector3", JSCLASS_HAS_PRIVATE, 
		JS_PropertyStub, JS_PropertyStub,					// add/del property
		aztecVector3_getProperty, aztecVector3_setProperty,	// get/set property
		JS_EnumerateStub, JS_ResolveStub,
		aztecVector3_convert, aztecVector3_finalize
};

static JSBool aztecVector3_constructor(JSContext *cx, JSObject *obj, uintN argc,
                             jsval *argv, jsval *rval)
{
	MVector3 *vec = new MVector3();
	if (vec == NULL) {
		JS_ReportError(cx, "Memory allocation failure in Vector3 constructor");
		return JS_FALSE;
	}
	JS_SetPrivate(cx, obj, vec);

	// Possible constructors: Vector3(), Vector3([x, y, z]), Vector3(x, y, z), and
	// Vector3(Vector3).
	if ((argc == 0) || (set(cx, obj, argc, argv, rval) == JS_TRUE)) {
		return JS_TRUE;
	} else {
		delete vec;
		JS_SetPrivate(cx, obj, NULL);
		JS_ReportError(cx, "Invalid arguments to Vector3 constructor");
		return JS_FALSE;
	}

	// Bad args to constructor.
	delete vec;
	return JS_FALSE;
}

JSObject *newVector3(JSContext *cx)
{
	JSObject *jsvec = JS_NewObject(cx, &aztecVector3_class, NULL, JS_GetGlobalObject(cx));

	if (jsvec == NULL) {
		return NULL;
	}

	MVector3 *pvec = new MVector3();
	JS_SetPrivate(cx, jsvec, pvec);

	return jsvec;
}

void setVector3Val(JSContext *cx, JSObject *jvec, MVector3 *val)
{
	MVector3 *vec = (MVector3 *)JS_GetInstancePrivate(cx, jvec, &aztecVector3_class, NULL);
	if (vec == NULL || val == NULL) {
		return;
	}

	vec->set(*val);
}

void addVector3Class(JSContext *cx)
{
	// Add a Vector3 class (reflection of MVector33) to the global namespace
	JSObject *iobj =
    JS_InitClass(cx, JS_GetGlobalObject(cx), NULL,
		&aztecVector3_class, aztecVector3_constructor, 0,
		aztecVector3_props, aztecVector3_methods, 0, 0);

}

}
